/*
 * RADIUS Dynamic Authorization Server (DAS)
 * Copyright (c) 2012, Jouni Malinen <j@w1.fi>
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

#ifndef RADIUS_DAS_H
#define RADIUS_DAS_H
#include "utils/ip_addr.h"

#define MAX_DAS_CLIENT_COUNT 4

struct radius_das_attrs;

struct radius_das_client {
	u8 *shared_secret;
	size_t shared_secret_len;
	struct hostapd_ip_addr client_addr;
	char vrf[IFNAMSIZ + 1];
};

struct radius_das_data {
	int sock;
	struct radius_das_client radius_das_clients[MAX_DAS_CLIENT_COUNT];
	int radius_das_client_count;
	unsigned int time_window;
	int require_event_timestamp;
	int require_message_authenticator;
	char *nas_identifier;
	void *ctx;
	enum radius_das_res (*disconnect)(void *ctx,
					  struct radius_das_attrs *attr,
					  void *recv_if_addr);
	enum radius_das_res (*handle_coa)(void *ctx,
					  struct radius_das_attrs *attr,
					  void *recv_if_addr);
};

enum radius_das_res {
	RADIUS_DAS_SUCCESS,
	RADIUS_DAS_UNSUPPORTED_ATTRIBUTE,
	RADIUS_DAS_MISSING_ATTRIBUTE,
	RADIUS_DAS_NAS_MISMATCH,
	RADIUS_DAS_SESSION_NOT_FOUND,
	RADIUS_DAS_MULTI_SESSION_MATCH,
	RADIUS_DAS_COA_FAILED,
};

struct radius_das_attrs {
	/* NAS identification attributes */
	const u8 *nas_ip_addr;
	const u8 *nas_identifier;
	size_t nas_identifier_len;
	const u8 *nas_ipv6_addr;

	/* Session identification attributes */
	const u8 *sta_addr;
	const u8 *user_name;
	size_t user_name_len;
	const u8 *acct_session_id;
	size_t acct_session_id_len;
	const u8 *vendor_specific;
	size_t vendor_specific_len;
	const u8 *acct_multi_session_id;
	size_t acct_multi_session_id_len;
	const u8 *cui;
	size_t cui_len;
	const u8 *terminate_cause;
	size_t terminate_cause_len;

	/* Authorization changes */
	const u8 *hs20_t_c_filtering;
};

struct radius_das_conf {
	int port;
	struct radius_das_client radius_das_clients[MAX_DAS_CLIENT_COUNT];
	int radius_das_client_count;
	char vrf[IFNAMSIZ + 1];
	unsigned int time_window;
	int require_event_timestamp;
	int require_message_authenticator;
	char *nas_identifier;
	void *ctx;
	enum radius_das_res (*disconnect)(void *ctx,
					  struct radius_das_attrs *attr,
					  void *recv_if_addr);
	enum radius_das_res (*handle_coa)(void *ctx,
					  struct radius_das_attrs *attr,
					  void *recv_if_addr);
};

struct radius_das_data *
radius_das_init(struct radius_das_conf *conf);

void radius_das_deinit(struct radius_das_data *data);

#endif /* RADIUS_DAS_H */
