/*
 * RADIUS Dynamic Authorization Server (DAS) (RFC 5176)
 * Copyright (c) 2012-2013, Jouni Malinen <j@w1.fi>
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

#include "includes.h"
#include <net/if.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <netinet/in.h>

#include "utils/common.h"
#include "utils/eloop.h"
#include "radius.h"
#include "radius_das.h"


#define CMSG_HDR_LEN sizeof(struct cmsghdr)
#define CMSG_IN_PKT_INFO_LEN (CMSG_HDR_LEN + sizeof(struct in_pktinfo) + 4)

u8 msgbuf[1500];
struct iovec msgiov = {
    &(msgbuf[0]),
    sizeof(msgbuf)
};
u8 cmsgbuf[CMSG_IN_PKT_INFO_LEN];

struct sockaddr_in msgaddr;
struct msghdr msghdr = {
    (void *)&msgaddr,
    sizeof(msgaddr),
    &msgiov,
    1,
    (void *)&cmsgbuf,
    sizeof(cmsgbuf),
    0
};

static int radius_das_decode_msg(struct radius_das_data *das,
		struct radius_msg *msg,
		const char *abuf,
		int from_port,
		struct radius_das_attrs *attrs)
{
	int error = 405;
	u8 attr;
	u8 *buf;
	size_t len;
	char tmp[100];
	u8 sta_addr[ETH_ALEN];
	u8 allowed[] = {
		RADIUS_ATTR_USER_NAME,
		RADIUS_ATTR_NAS_IP_ADDRESS,
		RADIUS_ATTR_CALLING_STATION_ID,
		RADIUS_ATTR_NAS_IDENTIFIER,
		RADIUS_ATTR_ACCT_SESSION_ID,
		RADIUS_ATTR_ACCT_MULTI_SESSION_ID,
		RADIUS_ATTR_EVENT_TIMESTAMP,
		RADIUS_ATTR_VENDOR_SPECIFIC,
		RADIUS_ATTR_MESSAGE_AUTHENTICATOR,
		RADIUS_ATTR_CHARGEABLE_USER_IDENTITY,
		RADIUS_ATTR_VENDOR_SPECIFIC,
		RADIUS_ATTR_ACCT_TERMINATE_CAUSE,
#ifdef CONFIG_IPV6
		RADIUS_ATTR_NAS_IPV6_ADDRESS,
#endif /* CONFIG_IPV6 */
		0
	};

	attr = radius_msg_find_unlisted_attr(msg, allowed);
	if (attr) {
		wpa_printf(MSG_INFO, "DAS: Unsupported attribute %u in "
			   "Disconnect-Request from %s:%d", attr,
			   abuf, from_port);
		error = 401;
		return error;
	}

	os_memset(attrs, 0, sizeof(struct radius_das_attrs));

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_NAS_IP_ADDRESS,
				    &buf, &len, NULL) == 0) {
		if (len != 4) {
			wpa_printf(MSG_INFO, "DAS: Invalid NAS-IP-Address from %s:%d",
				   abuf, from_port);
			error = 407;
			return error;
		}
		attrs->nas_ip_addr = buf;
	}

#ifdef CONFIG_IPV6
	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_NAS_IPV6_ADDRESS,
				    &buf, &len, NULL) == 0) {
		if (len != 16) {
			wpa_printf(MSG_INFO, "DAS: Invalid NAS-IPv6-Address from %s:%d",
				   abuf, from_port);
			error = 407;
			return error;
		}
		attrs->nas_ipv6_addr = buf;
	}
#endif /* CONFIG_IPV6 */

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_NAS_IDENTIFIER,
				    &buf, &len, NULL) == 0) {
		attrs->nas_identifier = buf;
		attrs->nas_identifier_len = len;
	}

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_CALLING_STATION_ID,
				    &buf, &len, NULL) == 0) {
		if (len >= sizeof(tmp))
			len = sizeof(tmp) - 1;
		os_memcpy(tmp, buf, len);
		tmp[len] = '\0';
		if (hwaddr_aton2(tmp, sta_addr) < 0) {
			wpa_printf(MSG_INFO, "DAS: Invalid Calling-Station-Id "
				   "'%s' from %s:%d", tmp, abuf, from_port);
			error = 407;
			return error;
		}
		attrs->sta_addr = sta_addr;
	}

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_USER_NAME,
				    &buf, &len, NULL) == 0) {
		attrs->user_name = buf;
		attrs->user_name_len = len;
	}

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_ACCT_SESSION_ID,
				    &buf, &len, NULL) == 0) {
		attrs->acct_session_id = buf;
		attrs->acct_session_id_len = len;
	}

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_ACCT_MULTI_SESSION_ID,
				    &buf, &len, NULL) == 0) {
		attrs->acct_multi_session_id = buf;
		attrs->acct_multi_session_id_len = len;
	}

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_CHARGEABLE_USER_IDENTITY,
				    &buf, &len, NULL) == 0) {
		attrs->cui = buf;
		attrs->cui_len = len;
	}

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_VENDOR_SPECIFIC,
				    &buf, &len, NULL) == 0) {
		attrs->vendor_specific = buf;
		attrs->vendor_specific_len = len;
	}

	if (radius_msg_get_attr_ptr(msg, RADIUS_ATTR_ACCT_TERMINATE_CAUSE,
				    &buf, &len, NULL) == 0) {
		attrs->terminate_cause = buf;
		attrs->terminate_cause_len = len;
	}
	return error;
}

static int radius_das_convert_error(const char *abuf,
		int from_port,
		enum radius_das_res res)
{
	switch (res) {
	case RADIUS_DAS_UNSUPPORTED_ATTRIBUTE:
		wpa_printf(MSG_INFO, "DAS: Unsupported attribute from %s:%d",
			   abuf, from_port);
		return 401;
	case RADIUS_DAS_MISSING_ATTRIBUTE:
		wpa_printf(MSG_INFO, "DAS: Missing attribute from %s:%d",
			   abuf, from_port);
		return 402;
	case RADIUS_DAS_NAS_MISMATCH:
		wpa_printf(MSG_INFO, "DAS: NAS mismatch from %s:%d",
			   abuf, from_port);
		return 403;
	case RADIUS_DAS_SESSION_NOT_FOUND:
		wpa_printf(MSG_INFO, "DAS: Session not found for request from "
			   "%s:%d", abuf, from_port);
		return 503;
	case RADIUS_DAS_MULTI_SESSION_MATCH:
		wpa_printf(MSG_INFO,
			   "DAS: Multiple sessions match for request from %s:%d",
			   abuf, from_port);
		return 508;
	case RADIUS_DAS_SUCCESS:
		return 0;
	}

	return 405;
}

static struct radius_msg * radius_das_disconnect(struct radius_das_data *das,
						 struct radius_msg *msg,
						 const char *abuf,
						 int from_port,
						 struct hostapd_ip_addr recv_if_addr)
{
	struct radius_hdr *hdr;
	struct radius_msg *reply;
	int error;
	enum radius_das_res res;
	struct radius_das_attrs attrs;

	hdr = radius_msg_get_hdr(msg);

	error = radius_das_decode_msg(das, msg, abuf, from_port, &attrs);
	if (error != 405)
		goto fail;

	res = das->disconnect(das->ctx, &attrs, (void *)&recv_if_addr);

	error = radius_das_convert_error(abuf, from_port, res);

fail:
	reply = radius_msg_new(error ? RADIUS_CODE_DISCONNECT_NAK :
			       RADIUS_CODE_DISCONNECT_ACK, hdr->identifier);
	if (reply == NULL)
		return NULL;

	if (error) {
		if (!radius_msg_add_attr_int32(reply, RADIUS_ATTR_ERROR_CAUSE,
					       error)) {
			radius_msg_free(reply);
			return NULL;
		}
	}

	return reply;
}

static struct radius_msg * radius_das_coa(struct radius_das_data *das,
					  struct radius_msg *msg,
					  const char *abuf,
					  int from_port,
					  struct hostapd_ip_addr recv_if_addr)
{
	struct radius_hdr *hdr;
	struct radius_msg *reply;
	int error;
	enum radius_das_res res;
	struct radius_das_attrs attrs;

	hdr = radius_msg_get_hdr(msg);

	error = radius_das_decode_msg(das, msg, abuf, from_port, &attrs);
	if (error != 405)
		goto fail;

	res = das->handle_coa(das->ctx, &attrs, (void *)&recv_if_addr);

	error = radius_das_convert_error(abuf, from_port, res);

fail:
	reply = radius_msg_new(error ? RADIUS_CODE_COA_NAK :
			       RADIUS_CODE_COA_ACK, hdr->identifier);
	if (reply == NULL)
		return NULL;

	if (error) {
		if (!radius_msg_add_attr_int32(reply, RADIUS_ATTR_ERROR_CAUSE,
					       error)) {
			radius_msg_free(reply);
			return NULL;
		}
	}

	return reply;
}


static void radius_das_receive(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct radius_das_data *das = eloop_ctx;
	union {
		struct sockaddr_storage ss;
		struct sockaddr_in sin;
#ifdef CONFIG_IPV6
		struct sockaddr_in6 sin6;
#endif /* CONFIG_IPV6 */
	} from;
	char abuf[50];
	char tbuf[50];
	int from_port = 0;
	socklen_t fromlen;
	int len;
	struct radius_msg *msg, *reply = NULL;
	struct radius_hdr *hdr;
	struct wpabuf *rbuf;
	u32 val;
	int res;
	struct os_time now;
	struct in_pktinfo *pi = NULL;
	struct cmsghdr *cm;
	struct hostapd_ip_addr recv_if_addr;
	int i, client_index;
	int found_match;


	fromlen = sizeof(from);

	len = recvmsg(sock, &msghdr, MSG_DONTWAIT);
	if (len < 0) {
		wpa_printf(MSG_ERROR, "DAS: recvmsg: %s", strerror(errno));
		return;
	}

	/* Get source address */
	from.sin = *((struct sockaddr_in *)(msghdr.msg_name));

	/* Get IP pkt info */
	for (cm = CMSG_FIRSTHDR(&msghdr); cm != NULL;
			cm = CMSG_NXTHDR(&msghdr, cm)) {
		if (cm->cmsg_level == SOL_IP &&
					cm->cmsg_type == IP_PKTINFO) {
			pi = (struct in_pktinfo *)CMSG_DATA(cm);
			if (pi) {
				/* keep in network-byte order */
				recv_if_addr.af = AF_INET;
				recv_if_addr.u.v4.s_addr = pi->ipi_addr.s_addr;
			}
		}
	}

	os_strlcpy(abuf, inet_ntoa(from.sin.sin_addr), sizeof(abuf));
	os_strlcpy(tbuf, inet_ntoa(recv_if_addr.u.v4), sizeof(tbuf));
	from_port = ntohs(from.sin.sin_port);

	wpa_printf(MSG_DEBUG, "DAS: Received %d bytes from %s:%d to %s",
		   len, abuf, from_port, tbuf);

	found_match = 0;
	for (i = 0; i < das->radius_das_client_count; i++) {
		if (das->radius_das_clients[i].client_addr.u.v4.s_addr == from.sin.sin_addr.s_addr ||
				das->radius_das_clients[i].client_addr.u.v4.s_addr == 0) {
			/*
			 * we need to check the source address of the request against what is allowed
			 * if 0.0.0.0 src is configured, we allow it but also check the password.
			 */
			if (das->radius_das_clients[i].client_addr.u.v4.s_addr == 0)
				wpa_printf(MSG_DEBUG, "DAS: ANY (0) client source allowed configured");
			else
				wpa_printf(MSG_DEBUG, "DAS: found configured source client");
			found_match = 1;
			client_index = i;
			break;
		}
	}
	if (!found_match) {
		wpa_printf(MSG_DEBUG, "DAS: Drop message from unknown client");
		return;
	}

	msg = radius_msg_parse(msghdr.msg_iov->iov_base, len);
	if (msg == NULL) {
		wpa_printf(MSG_DEBUG, "DAS: Parsing incoming RADIUS packet "
			   "from %s:%d failed", abuf, from_port);
		return;
	}

	if (wpa_debug_level <= MSG_MSGDUMP)
		radius_msg_dump(msg);

	if (radius_msg_verify_das_req(msg, das->radius_das_clients[client_index].shared_secret,
				das->radius_das_clients[client_index].shared_secret_len,
				das->require_message_authenticator)) {
		wpa_printf(MSG_DEBUG,
			   "DAS: Invalid authenticator or Message-Authenticator in packet from %s:%d - drop",
			   abuf, from_port);
		goto fail;
	}

	os_get_time(&now);
	res = radius_msg_get_attr(msg, RADIUS_ATTR_EVENT_TIMESTAMP,
				  (u8 *) &val, 4);
	if (res == 4) {
		u32 timestamp = ntohl(val);
		if ((unsigned int) abs((int) (now.sec - timestamp)) >
		    das->time_window) {
			wpa_printf(MSG_DEBUG, "DAS: Unacceptable "
				   "Event-Timestamp (%u; local time %u) in "
				   "packet from %s:%d - drop",
				   timestamp, (unsigned int) now.sec,
				   abuf, from_port);
			goto fail;
		}
	} else if (das->require_event_timestamp) {
		wpa_printf(MSG_DEBUG, "DAS: Missing Event-Timestamp in packet "
			   "from %s:%d - drop", abuf, from_port);
		goto fail;
	}

	hdr = radius_msg_get_hdr(msg);

	switch (hdr->code) {
	case RADIUS_CODE_DISCONNECT_REQUEST:
		reply = radius_das_disconnect(das, msg, abuf, from_port, recv_if_addr);
		break;
	case RADIUS_CODE_COA_REQUEST:
		reply = radius_das_coa(das, msg, abuf, from_port, recv_if_addr);
		break;
	default:
		wpa_printf(MSG_DEBUG, "DAS: Unexpected RADIUS code %u in "
			   "packet from %s:%d",
			   hdr->code, abuf, from_port);
	}

	if (reply) {
		wpa_printf(MSG_DEBUG, "DAS: Reply to %s:%d", abuf, from_port);

		if (!radius_msg_add_attr_int32(reply,
					       RADIUS_ATTR_EVENT_TIMESTAMP,
					       now.sec)) {
			wpa_printf(MSG_DEBUG, "DAS: Failed to add "
				   "Event-Timestamp attribute");
		}

		if (radius_msg_finish_das_resp(reply,
				das->radius_das_clients[client_index].shared_secret,
				das->radius_das_clients[client_index].shared_secret_len, hdr) < 0) {
			wpa_printf(MSG_DEBUG, "DAS: Failed to add "
					"Message-Authenticator attribute");
		}

		if (wpa_debug_level <= MSG_MSGDUMP)
			radius_msg_dump(reply);

		rbuf = radius_msg_get_buf(reply);
		res = sendto(das->sock, wpabuf_head(rbuf),
			     wpabuf_len(rbuf), 0,
			     (struct sockaddr *) &from.ss, fromlen);
		if (res < 0) {
			wpa_printf(MSG_ERROR, "DAS: sendto(to %s:%d): %s",
				   abuf, from_port, strerror(errno));
		}
	}

fail:
	radius_msg_free(msg);
	radius_msg_free(reply);
}


static int radius_das_open_socket(int port)
{
	int s;
	struct sockaddr_in addr;
	int pktinfo = 1;

	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		wpa_printf(MSG_INFO, "RADIUS DAS: socket: %s", strerror(errno));
		return -1;
	}

	if (setsockopt(s, SOL_IP, IP_PKTINFO, &pktinfo, sizeof(pktinfo)) < 0) {
		wpa_printf(MSG_ERROR, "RADIUS DAS: Can't set socket option IP"
				" packet info: %s", strerror(errno));
		return(-1);
	}

	os_memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);
	if (bind(s, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		wpa_printf(MSG_INFO, "RADIUS DAS: bind: %s", strerror(errno));
		close(s);
		return -1;
	}

	return s;
}


struct radius_das_data *
radius_das_init(struct radius_das_conf *conf)
{
	struct radius_das_data *das;
	char iface[IFNAMSIZ + 1];
	socklen_t ifacelen = IFNAMSIZ + 1;
	int i;

	if (conf->port == 0) {
		wpa_printf(MSG_INFO, "RADIUS: no DAS port given. DAS inactive.");
		return NULL;
	}
	if (conf->radius_das_client_count == 0) {
		wpa_printf(MSG_ERROR, "RADIUS: no DAS Clients configured. DAS inactive.");
		return NULL;
	}

	das = os_zalloc(sizeof(*das));
	if (das == NULL)
		return NULL;

	das->time_window = conf->time_window;
	das->nas_identifier = conf->nas_identifier;
	das->require_event_timestamp = conf->require_event_timestamp;
	das->require_message_authenticator =
		conf->require_message_authenticator;
	das->ctx = conf->ctx;
	das->disconnect = conf->disconnect;
	das->handle_coa = conf->handle_coa;

	for (i = 0; i < conf->radius_das_client_count; i++) {
		os_memcpy(&das->radius_das_clients[i].client_addr,
				&conf->radius_das_clients[i].client_addr,
				sizeof(das->radius_das_clients[i].client_addr));
		das->radius_das_clients[i].shared_secret =
			os_memdup(conf->radius_das_clients[i].shared_secret,
					conf->radius_das_clients[i].shared_secret_len);
		if (das->radius_das_clients[i].shared_secret == NULL) {
			radius_das_deinit(das);
			return NULL;
		}
		das->radius_das_clients[i].shared_secret_len =
			conf->radius_das_clients[i].shared_secret_len;
	}

	das->radius_das_client_count = conf->radius_das_client_count;
	das->sock = radius_das_open_socket(conf->port);
	if (das->sock < 0) {
		wpa_printf(MSG_ERROR, "Failed to open UDP socket for RADIUS "
			   "DAS");
		radius_das_deinit(das);
		return NULL;
	}

	if (strlen(conf->vrf) && setsockopt(das->sock, SOL_SOCKET, SO_BINDTODEVICE,
				conf->vrf, strlen(conf->vrf)) == -1)
		wpa_printf(MSG_INFO, "RADIUS DAS: Could not bind VRF %s to socket", conf->vrf);
	else if (strlen(conf->vrf))
		wpa_printf(MSG_INFO, "RADIUS DAS: socket was bound to VRF %s", conf->vrf);

	if (strlen(conf->vrf)) {
		if (getsockopt(das->sock, SOL_SOCKET, SO_BINDTODEVICE, iface, &ifacelen) != 0)  {
			wpa_printf(MSG_INFO, "RADIUS DAS: radius_das_init: Could not get required VRF interface\n%s", strerror(errno));
		} else if (!ifacelen) {
			wpa_printf(MSG_INFO, "RADIUS: radius_das_inti:: SO_BINDTODEVICE not set ifacelen=%d\n", ifacelen);
		} else {
			wpa_printf(MSG_INFO, "RADIUS: radius_das_init: SO_BINDTODEVICE dev=%s\n", iface);
		}
	}

	if (eloop_register_read_sock(das->sock, radius_das_receive, das, NULL))
	{
		radius_das_deinit(das);
		return NULL;
	}

	return das;
}


void radius_das_deinit(struct radius_das_data *das)
{
	int i;
	if (das == NULL)
		return;

	if (das->sock >= 0) {
		eloop_unregister_read_sock(das->sock);
		close(das->sock);
	}

	for (i = 0; i < das->radius_das_client_count; i++) {
		os_free(das->radius_das_clients[i].shared_secret);
	}
	os_free(das);
}
