#include "includes.h"

#include "common.h"
#include "utils/list.h"
#include "driver_wired_if_info.h"

if_hash_table_t *if_hash_table_alloc()
{
	if_hash_table_t *ht = os_zalloc(sizeof(if_hash_table_t));

	return ht;
}

if_info_t *if_info_alloc(int type, int ifindex, char *ifname, u32 flags)
{
	if_info_t *if_node = os_zalloc(sizeof(if_info_t));

	if (!if_node)
		return NULL;

	if (type == IF_TYPE_BRIDGE)
		if_node->data = os_zalloc(sizeof(br_if_info_t));
	else if ((type == IF_TYPE_VLAN) || (type == IF_TYPE_OTHER))
		if_node->data = os_zalloc(sizeof(vlan_if_info_t));

	if (if_node->data) {
		((br_if_info_t *)if_node->data)->parent = if_node;
		if (type == IF_TYPE_BRIDGE) {
			dl_list_init(&((br_if_info_t *)if_node->data)->tag_mbr_interfaces);
			dl_list_init(&((br_if_info_t *)if_node->data)->untag_mbr_interfaces);
		}
	} else {
		os_free (if_node);
		return NULL;
	}

	if_node->type = type;
	if_node->ifindex = ifindex;
	if_node->flags = flags;
	strncpy(if_node->ifname, ifname, sizeof(if_node->ifname));

	return if_node;
}

if_info_t * if_hash_table_find(if_hash_table_t *ht, int ifindex)
{
	if_info_t *if_node;

	if (!ifindex)
		return NULL;

	if_node = ht->if_nodes[IF_HASH(ifindex)];
	while (if_node != NULL && (if_node->ifindex != ifindex))
		if_node = if_node->hnext;
	return if_node;
}

void if_hash_table_add(if_hash_table_t *ht, if_info_t *if_node)
{
	if_node->hnext = ht->if_nodes[IF_HASH(if_node->ifindex)];
	ht->if_nodes[IF_HASH(if_node->ifindex)] = if_node;

	ht->count++;
}

void if_hash_table_del(if_hash_table_t *ht, int ifindex)
{
	if_info_t *if_node;

	if_node = ht->if_nodes[IF_HASH(ifindex)];
	if (if_node == NULL)
		return;
	if (if_node->ifindex == ifindex) {
		ht->if_nodes[IF_HASH(ifindex)] = if_node->hnext;
		return;
	}

	while (if_node->hnext != NULL &&
		   (if_node->hnext->ifindex != ifindex))
		if_node = if_node->hnext;
	if (if_node->hnext != NULL)
		if_node->hnext = if_node->hnext->hnext;
	else
		wpa_printf(MSG_DEBUG, "could not remove %d from hash table",
					ifindex);
}

void if_hash_table_free(if_hash_table_t *ht)
{
	if_info_t *node;
	if_info_t *next;
	if_info_t **prev;
	int i;

	for (i = 0; i < IF_HASH_SIZE; i++) {
		node = ht->if_nodes[i];
		prev = &ht->if_nodes[i];
		while (node) {
			next = node->hnext;
			*prev = next;
			os_free(node->data);
			os_free(node);
			ht->count--;
			node = next;
		}
	}
}

void if_hash_table_foreach(if_hash_table_t *ht,
					int (*foreach_cb)(void *data, void *cbarg), void *cbarg)
{
	if_info_t *node;
	int i;

	for (i = 0; i < IF_HASH_SIZE; i++) {
		node = ht->if_nodes[i];
		while (node) {
			foreach_cb(node, cbarg);
			node = node->hnext;
		}
	}
}

if_info_t *if_hash_table_find_by_vlan(if_hash_table_t *ht, int vlan_id)
{
	if_info_t *node;
	int i;

	for (i = 0; i < IF_HASH_SIZE; i++) {
		node = ht->if_nodes[i];
		while (node) {
			if (node->vlan_id == vlan_id)
				return node;
			node = node->hnext;
		}
	}
	return NULL;
}

if_info_t *if_get_br(if_info_t *if_node)
{
	vlan_if_info_t *vif_info;

	if (if_node->data && if_node->type != IF_TYPE_BRIDGE) {
		vif_info = (vlan_if_info_t *)if_node->data;

		return vif_info->master;
	}

	return NULL;
}

void if_update_master(if_info_t *if_node, if_info_t *master)
{
	vlan_if_info_t *vlan_info = (vlan_if_info_t *)if_node->data;
	if_info_t *old_master = vlan_info->master;

	if (master) {
		if (old_master)
			dl_list_del(&vlan_info->mbr_if_list);

		if_mbr_list_add(master, if_node);
		if (!master->vlan_aware && (if_node->type == IF_TYPE_VLAN))
			master->vlan_id = if_node->vlan_id;
		vlan_info->master_id = master->ifindex;
		if_node->vlan_aware = master->vlan_aware;
	} else {
		if (old_master)
			dl_list_del(&vlan_info->mbr_if_list);
		vlan_info->master_id = 0;
	}

	if (old_master && (if_node->type == IF_TYPE_VLAN) &&
			!old_master->vlan_aware &&
			dl_list_empty(
				&((br_if_info_t *)(old_master->data))->tag_mbr_interfaces))
		old_master->vlan_id = 0;

	vlan_info->master = master;
}

void if_mbr_list_add (if_info_t *br_node, if_info_t *if_node)
{
	br_if_info_t *br_info = (br_if_info_t *)br_node->data;
	vlan_if_info_t *vlan_info = (vlan_if_info_t *)if_node->data;

	if (if_node->type  == IF_TYPE_VLAN) {
		dl_list_add(&br_info->tag_mbr_interfaces, &vlan_info->mbr_if_list);
	} else {
		dl_list_add(&br_info->untag_mbr_interfaces, &vlan_info->mbr_if_list);
	}
}
