/*
 * Wired Ethernet driver interface
 * Copyright (c) 2005-2009, Jouni Malinen <j@w1.fi>
 * Copyright (c) 2004, Gunter Burchardt <tira@isx.de>
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

#include <sys/stat.h>
#include <pthread.h>
#include "includes.h"

#include "common.h"
#include "eloop.h"
#include "driver.h"
#include "netlink.h"
#include "ap/sta_info.h"
#include "ap/hostapd.h"
#include "ap/ap_drv_ops.h"
#include "common/ieee802_11_defs.h"
#include "driver_wired_common.h"
#include "eapol_auth/eapol_auth_sm.h"
#include "eapol_auth/eapol_auth_sm_i.h"
#include "eap_server/eap.h"
#include "driver_wired_if_info.h"
#include "glob.h"

#include <sys/ioctl.h>
#include <linux/filter.h>
#include <linux/rtnetlink.h>
#include <netlink/netlink.h>
#include <netlink/list.h>
#include <netlink/cache.h>
#include <netlink/route/link/bridge.h>
#include <netlink/route/link/vlan.h>
#undef IFNAMSIZ
#include <linux/if.h>
#ifdef __linux__
#include <netpacket/packet.h>
#include <net/if_arp.h>
#endif /* __linux__ */
#if defined(__FreeBSD__) || defined(__DragonFly__) || defined(__FreeBSD_kernel__)
#include <net/if_dl.h>
#include <net/if_media.h>
#endif /* defined(__FreeBSD__) || defined(__DragonFly__) || defined(__FreeBSD_kernel__) */
#ifdef __sun__
#include <sys/sockio.h>
#endif /* __sun__ */

#ifdef _MSC_VER
#pragma pack(push, 1)
#endif /* _MSC_VER */

#ifndef IFF_LOWER_UP
#define IFF_LOWER_UP   0x10000         /* driver signals L1 up         */
#endif
#ifndef IFF_DORMANT
#define IFF_DORMANT    0x20000         /* driver signals dormant       */
#endif

#define MACBYTE(a) (a)[0], (a)[1], (a)[2], (a)[3], (a)[4], (a)[5]
#define COMPACT_MACBYTE "%02x%02x%02x%02x%02x%02x"
#define MAC4BYTE(a) (a)[0], (a)[1], (a)[2], (a)[3]
#define COMPACT_MAC4BYTE "%02x%02x%02x%02x"
#define MAC2BYTE(a) (a)[0], (a)[1]
#define COMPACT_MAC2BYTE "%02x%02x"
#define MACLAST4BYTE(a) (a)[2], (a)[3], (a)[4], (a)[5]
#define MACLAST2BYTE(a) (a)[4], (a)[5]
#define DROP_PRIORITY 50
#define MACLAST4HEX(a) (a)[5] | ((a)[4] << 8) | ((a)[3] << 16) | ((a)[2] << 24)
#define MAC2HEX(a) (a)[1] | ((a)[0] << 8)

#define ACL_TIMER 5
#define BUFF_LEN 256

const char aclcommand[] = "/usr/cumulus/bin/cl-acltool";
const char aclargs[] = "-i";
const char dot1x_aclargs[128] ;
const char tccommand[] = "/sbin/tc";
const char bridge_command[] = "/sbin/bridge";
const char brctl_command[] = "/sbin/brctl";
const char ip_command[] = "/sbin/ip";
const char lldp_command[] = "/usr/sbin/lldpcli";
const char acl_tmp_dir[] = "/etc/cumulus/acl/policy.d/tmp";
const char preauth_dacl_dir[] = "/etc/cumulus/acl/policy.d/dot1x_preauth_dacl";
const char preauth_dacl_file[] = "/etc/cumulus/acl/policy.d/dot1x_preauth_dacl/default_preauth_dacl.rules";

struct ieee8023_hdr {
	u8 dest[6];
	u8 src[6];
	u16 ethertype;
} STRUCT_PACKED;

struct ether_vlan_hdr {
	u8 dest[6];
	u8 src[6];
	u16 tpid;
	u16 tci;
} STRUCT_PACKED;

#ifdef _MSC_VER
#pragma pack(pop)
#endif /* _MSC_VER */


struct wired_global {
	struct hapd_interfaces *global_interfaces;
	struct dl_list interfaces;
	struct netlink_data *netlink;
	struct nl_sock *mngr_sock;
	struct nl_cache_mngr *mngr;
	struct nl_cache *cache;
	struct nl_sock *sync_sock;
	pthread_t tacl;
	pthread_cond_t acl_cv;
	pthread_mutex_t acl_lock;
	glob_t results;
	Boolean acl_reinstall;
	Boolean in_deinit;
};

struct wired_local {
	if_hash_table_t *br_hash;
	if_hash_table_t *if_hash;
	struct dl_list  no_vlan_interfaces;
};

struct wpa_driver_wired_if_info {
	int pvid;
	int dynamic_pvid;
	int dynamic_voice_vid;
	int bridge_id;
	int is_bridge;
	u32 flags;
	if_info_t *if_node;
	u8 addr[6];
};

#define WPA_WIRED_DATA_PVLAN_ENABLED BIT(0)
#define WPA_WIRED_DATA_DYNAMIC_VLAN_ENABLED BIT(1)
#define WPA_WIRED_DATA_VLAN_MOD BIT(2)
#define WPA_WIRED_DATA_NO_VLAN BIT(3)
#define WPA_WIRED_DATA_IGNORE_CHANGE BIT(4)
#define WPA_WIRED_DATA_DYN_VOICE BIT(5)
#define WPA_WIRED_DATA_STATIC_VOICE BIT(6)


struct wpa_driver_wired_data {
	struct dl_list  if_list;
	struct wired_global *global;
	struct driver_wired_common_data common;

	int dhcp_sock; /* socket for dhcp packets */
	int mab_sock; /* socket for learning source MACs */
	int use_pae_group_addr;
	int flags;
	struct wpa_driver_wired_if_info if_info;
	int pvid_ref;
	int vid_ref;
};

/* TODO: detecting new devices should eventually be changed from using DHCP
 * snooping to trigger on any packet from a new layer 2 MAC address, e.g.,
 * based on ebtables, etc. */

struct dhcp_message {
	u_int8_t op;
	u_int8_t htype;
	u_int8_t hlen;
	u_int8_t hops;
	u_int32_t xid;
	u_int16_t secs;
	u_int16_t flags;
	u_int32_t ciaddr;
	u_int32_t yiaddr;
	u_int32_t siaddr;
	u_int32_t giaddr;
	u_int8_t chaddr[16];
	u_int8_t sname[64];
	u_int8_t file[128];
	u_int32_t cookie;
	u_int8_t options[308]; /* 312 - cookie */
};

/* Berkeley Packet filter code to filter out BFD Echo packets.
 * tcpdump -dd -n inbound"
 */
static struct sock_filter mab_sock_filter[] = {
	{ 0x28, 0, 0, 0x0000000c },
	{ 0x15, 2, 0, 0x0000888e },
	{ 0x28, 0, 0, 0xfffff004 },
	{ 0x15, 0, 1, 0x00000004 },
	{ 0x6, 0, 0, 0x00000000 },
	{ 0x6, 0, 0, 0x00040000 },
};

static struct sock_filter mab_mac_sock_filter[32];

struct wired_local *local;
struct wired_global *global_ptr = NULL;

void wired_mab_deactivation(struct hostapd_data *hapd);
int br_print_cb (void *data, void *cbarg);
static int wired_del_dynamic_mac(struct wpa_driver_wired_data *drv);
struct rtnl_link *wpa_nl_get_link_by_family(struct nl_cache *cache,
		int family, int ifindex);

static int
wired_driver_hapd_is_l2_if(void *priv)
{
	struct wpa_driver_wired_data *drv = priv;
	return drv->if_info.is_bridge;
}

static int
wired_driver_is_vlan_aware_if(struct wpa_driver_wired_if_info *if_info)
{
	if (if_info->if_node)
		return if_info->if_node->vlan_aware;
	else
		return 1;
}

static int wired_is_last_parked_sta(struct hostapd_data *hapd)
{
	struct sta_info *sta;

	for (sta = hapd->sta_list; sta; sta = sta->next) {
		if (sta->flags & WLAN_STA_PARKED_VLAN)
			return 0;;
	}

	return 1;
}

static int globerr(const char *path, int eerrno)
{
	wpa_printf(MSG_DEBUG, "No ACL files found in %s: %s\n",
			path, strerror(eerrno));
	return 1;
}

static int
wired_remove_dacl_files(struct hostapd_data *hapd, const u8 *addr)
{
	char acl_filename[128];
	struct stat st = {0};

	snprintf(acl_filename, sizeof(acl_filename),
				"/etc/cumulus/acl/policy.d/150_dot1x_dacl_%s_"
				COMPACT_MACBYTE ".rules",
				hapd->conf->iface, MACBYTE(addr));

	if (stat(acl_filename, &st) == 0) {
		if (remove(acl_filename)) {
			wpa_printf(MSG_WARNING, "Could not remove dacl ACL file %s",
					acl_filename);
			return -1;
		} else {
			wpa_printf(MSG_DEBUG, "removed dacl ACL file %s",
					acl_filename);
			wired_start_acl_reinstall();
		}
	}

	return 0;
}

static int
wired_flush(void *priv)
{
	struct wpa_driver_wired_data *drv = priv;
	struct hostapd_data *hapd;
	struct sta_info *sta;

	wpa_printf(MSG_DEBUG, "WIRED: wired_flush called");
	hapd = drv->common.ctx;
	if (!hapd)
		return 0;

	for (sta = hapd->sta_list; sta; sta = sta->next) {
		hostapd_drv_sta_deauth(hapd, sta->addr,
					WLAN_REASON_DISASSOC_STA_HAS_LEFT);
		ap_sta_deauthenticate(hapd, sta, WLAN_REASON_DISASSOC_STA_HAS_LEFT);
	}

	hostapd_free_stas(hapd);
	return 0;
}

void *wired_install_acl()
{
	int i;
	char aclpath[] = "/etc/cumulus/acl/policy.d/tmp_install/*_dot1x*";
	char newaclpath[] = "/etc/cumulus/acl/policy.d";
	char acl_tmp_dir2[] = "/etc/cumulus/acl/policy.d/tmp_install";
	char *filename;
	char acl_filename[128];
	struct stat st = {0};
	int flags = 0, result;

	while (1) {
		pthread_mutex_lock(&(global_ptr->acl_lock));
		pthread_cond_wait(&(global_ptr->acl_cv), &(global_ptr->acl_lock));

		/* Move ACL files to correct directory */
		wpa_printf(MSG_DEBUG, "About to move %s", aclpath);

		if (global_ptr->in_deinit)
			return NULL;

		if (global_ptr->results.gl_pathc) {
			/*Move all the files to a temp dir tmp_install */
			if (stat(acl_tmp_dir2, &st) == -1) {
			    mkdir(acl_tmp_dir2, 0700);
			}
			for (i = 0; i < global_ptr->results.gl_pathc; i++) {
				filename = strrchr(global_ptr->results.gl_pathv[i], '/');
				if (!filename)
					continue;
				snprintf(acl_filename, sizeof(acl_filename), "%s%s",
				    acl_tmp_dir2, filename);
				if (rename(global_ptr->results.gl_pathv[i], acl_filename))
					wpa_printf(MSG_ERROR, "Could not rename ACL file %s to %s",
							global_ptr->results.gl_pathv[i], acl_filename);
				else
					wpa_printf(MSG_DEBUG, "renamed ACL file %s to %s",
							global_ptr->results.gl_pathv[i], acl_filename);
			}
			sprintf(dot1x_aclargs, "-p8021x -i -v -P %s", acl_tmp_dir2);
			if (os_exec(aclcommand, dot1x_aclargs, 1)) {
				wpa_printf(MSG_ERROR, "WIRED %s: could not install dot1x acl rules",
						__FUNCTION__);
			}
			globfree(&global_ptr->results);
			os_memset(&global_ptr->results, 0, sizeof(global_ptr->results));
			result = glob(aclpath, flags, globerr, &global_ptr->results);
			if (result) {
				wpa_printf(MSG_DEBUG, "No ACL files to install");
				globfree(&global_ptr->results);
			}
			for (i = 0; i < global_ptr->results.gl_pathc; i++) {
				filename = strrchr(global_ptr->results.gl_pathv[i], '/');
				if (!filename)
					continue;
				snprintf(acl_filename, sizeof(acl_filename), "%s%s",
				    newaclpath, filename);
				if (rename(global_ptr->results.gl_pathv[i], acl_filename))
					wpa_printf(MSG_ERROR, "Could not rename ACL file %s to %s",
							global_ptr->results.gl_pathv[i], acl_filename);
				else
					wpa_printf(MSG_DEBUG, "renamed ACL file %s to %s",
							global_ptr->results.gl_pathv[i], acl_filename);
				if ( stat(acl_filename, &st) == 0 && st.st_size == 0) {
					if (remove(acl_filename)) 
						wpa_printf(MSG_DEBUG, "Could not remove ACL file %s",
								acl_filename);
					 else 
						wpa_printf(MSG_DEBUG, "removed ACL file %s",
								acl_filename);
				 }  
			}

		} else if (global_ptr->acl_reinstall) {
			if (os_exec(aclcommand, aclargs, 1)) {
				wpa_printf(MSG_ERROR, "WIRED %s: could not run cl-acltool",
						__FUNCTION__);
			}
		}
		global_ptr->acl_reinstall = FALSE;
		globfree(&global_ptr->results);
		os_memset(&global_ptr->results, 0, sizeof(global_ptr->results));
		pthread_mutex_unlock(&(global_ptr->acl_lock));
	}
	return NULL;
}

void wired_acl_signal(void *eloop_ctx, void *timeout_ctx)
{
	char aclpath[] = "/etc/cumulus/acl/policy.d/tmp/*_dot1x*";
	int flags = 0;
	int result;

	wpa_printf(MSG_DEBUG, "%s: signaling ACL installation", __FUNCTION__);
	if (global_ptr) {
		if (global_ptr->global_interfaces->start_in_progress
			|| global_ptr->global_interfaces->stop_in_progress) {
			eloop_cancel_timeout(wired_acl_signal, global_ptr, NULL);
			eloop_register_timeout(ACL_TIMER, 0, wired_acl_signal,
						global_ptr, NULL);
			return;
		}

		eloop_cancel_timeout(wired_acl_signal, global_ptr, NULL);
		pthread_mutex_lock(&(global_ptr->acl_lock));
		result = glob(aclpath, flags, globerr, &global_ptr->results);
		if (result) {
			wpa_printf(MSG_DEBUG, "No ACL files to install");
			globfree(&global_ptr->results);
		}

		if (!result || global_ptr->acl_reinstall) {
			pthread_cond_signal(&(global_ptr->acl_cv));
		}

		pthread_mutex_unlock(&(global_ptr->acl_lock));
		eloop_register_timeout(ACL_TIMER, 0, wired_acl_signal, global_ptr, NULL);
	} else {
		if (os_exec(aclcommand, aclargs, 1))
			wpa_printf(MSG_ERROR, "%s: Error: could not run cl-acltool",
					__FUNCTION__);
	}
}

void wired_start_acl_reinstall()
{
	if (!global_ptr)
		return;
	global_ptr->acl_reinstall = TRUE;
}

void wired_process_acl_remove()
{
	if (!global_ptr)
		return;
	eloop_cancel_timeout(wired_acl_signal, global_ptr, NULL);
	pthread_mutex_lock(&(global_ptr->acl_lock));
	pthread_cond_signal(&(global_ptr->acl_cv));
	pthread_mutex_unlock(&(global_ptr->acl_lock));
	eloop_register_timeout(ACL_TIMER, 0, wired_acl_signal, global_ptr, NULL);

}

bool wired_if_link_down(struct wpa_driver_wired_data *drv)
{
	struct wired_global *global = drv->global;
	struct rtnl_link *link;
	bool proto_down = FALSE;

	link = wpa_nl_get_link_by_family(global->cache, 0,
				     drv->common.ifindex);
	if (link) {
		drv->if_info.flags = rtnl_link_get_flags(link);
		if (rtnl_link_get_operstate(link) != IF_OPER_UP) {
			wpa_printf(MSG_ERROR, "WIRED: link DOWN for ifname %s oper %d",
				drv->common.ifname, rtnl_link_get_operstate(link));
			rtnl_link_put(link);
			return TRUE;
		}
		wpa_printf(MSG_DEBUG, "WIRED: link UP for ifname %s oper %d",
		    drv->common.ifname, rtnl_link_get_operstate(link));
		rtnl_link_put(link);
	}
	return FALSE;
}

#ifdef __linux__
static void handle_data(void *ctx, unsigned char *buf, size_t len)
{
#ifdef HOSTAPD
	struct ieee8023_hdr *hdr;
	u8 *pos, *sa;
	struct hostapd_data *hapd = ctx;
	size_t left;
	union wpa_event_data event;
	struct sta_info *sta;
	int found = 0;

	wpa_printf(MSG_ERROR, "Handle_data:recv: packet on %s", hapd->conf->iface);

	/* must contain at least ieee8023_hdr 6 byte source, 6 byte dest,
	 * 2 byte ethertype */
	if (len < 14) {
		wpa_printf(MSG_MSGDUMP, "handle_data: too short (%lu)",
			   (unsigned long) len);
		return;
	}

	if (wired_if_link_down(hapd->drv_priv))
		return;

	hdr = (struct ieee8023_hdr *) buf;
	sa = hdr->src;
	if (os_memcmp(sa, hapd->own_addr, ETH_ALEN) == 0)
		return;

	os_memset(&event, 0, sizeof(event));
	event.new_sta.addr = sa;

	sta = ap_get_sta(hapd, sa);
	if (ntohs(hdr->ethertype) == ETH_P_PAE) {
		wpa_printf(MSG_MSGDUMP, "Received EAPOL packet");
		if (sta) {
			if (sta->flags & WLAN_STA_MAB) {
				if (!(sta->flags & WLAN_STA_AUTHORIZED))
					found = 1;
				else
					return;
			}
		}
		event.new_sta.pae_detected = TRUE;
	} else {
		wpa_printf(MSG_MSGDUMP, "Received NON-EAPOL packet");
		if (sta && (sta->flags & WLAN_STA_EAP))
			return;
		event.new_sta.pae_detected = FALSE;
	}

	if (found)
		ap_free_sta(hapd, sta);
	wpa_supplicant_event(ctx, EVENT_NEW_STA, &event);

	if (ntohs(hdr->ethertype) == ETH_P_PAE) {
		pos = (u8 *) (hdr + 1);
		left = len - sizeof(*hdr);
		drv_event_eapol_rx(ctx, sa, pos, left);
	}
#endif /* HOSTAPD */
}

static void handle_read(int sock, void *eloop_ctx, void *sock_ctx)
{
	int len;
	unsigned char buf[3000];

	len = recv(sock, buf, sizeof(buf), 0);
	if (len < 0) {
		wpa_printf(MSG_ERROR, "recv: %s", strerror(errno));
		return;
	}

	handle_data(eloop_ctx, buf, len);
}

struct rtnl_link *wpa_nl_get_link_by_family(struct nl_cache *cache,
											int family, int ifindex)
{
	struct rtnl_link *link, *link_filter;

	if (!(link_filter = rtnl_link_alloc()))
		return NULL;

	rtnl_link_set_family(link_filter, family);
	rtnl_link_set_ifindex(link_filter, ifindex);

	link = (struct rtnl_link *)nl_cache_search(cache, OBJ_CAST(link_filter));

	rtnl_link_put(link_filter);

	return link;
}

int wpa_nl_is_link_bridge(struct rtnl_link *link)
{
	char *type = rtnl_link_get_type(link);

	if (type)
		wpa_printf(MSG_DEBUG, "%s: type %s", __FUNCTION__, type);
	else
		wpa_printf(MSG_DEBUG, "%s: type not found", __FUNCTION__);
	return type && !strcmp(type, "bridge");
}

static void handle_dhcp(int sock, void *eloop_ctx, void *sock_ctx)
{
	int len;
	unsigned char buf[3000];
	struct dhcp_message *msg;
	u8 *mac_address;
	union wpa_event_data event;

	len = recv(sock, buf, sizeof(buf), 0);
	if (len < 0) {
		wpa_printf(MSG_ERROR, "recv: %s", strerror(errno));
		return;
	}

	/* must contain at least dhcp_message->chaddr */
	if (len < 44) {
		wpa_printf(MSG_MSGDUMP, "handle_dhcp: too short (%d)", len);
		return;
	}

	msg = (struct dhcp_message *) buf;
	mac_address = (u8 *) &(msg->chaddr);

	wpa_printf(MSG_MSGDUMP, "Got DHCP broadcast packet from " MACSTR,
			MAC2STR(mac_address));

	os_memset(&event, 0, sizeof(event));
	event.new_sta.addr = mac_address;
	wpa_supplicant_event(eloop_ctx, EVENT_NEW_STA, &event);
}
#endif /* __linux__ */


static int wired_init_sockets(struct wpa_driver_wired_data *drv, u8 *own_addr)
{
#ifdef __linux__
	struct ifreq ifr;
	struct sockaddr_ll addr;
	struct sockaddr_in addr2;
	int n = 1;

	/* XXXXX FIXME change to IPv4 for testing */
	drv->common.sock = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_PAE));
	if (drv->common.sock < 0) {
		wpa_printf(MSG_ERROR, "socket[PF_PACKET,SOCK_RAW]: %s",
			   strerror(errno));
		return -1;
	}

	if (eloop_register_read_sock(drv->common.sock, handle_read,
				     drv->common.ctx, NULL)) {
		wpa_printf(MSG_INFO, "Could not register read socket");
		return -1;
	}

	os_memset(&ifr, 0, sizeof(ifr));
	os_strlcpy(ifr.ifr_name, drv->common.ifname, sizeof(ifr.ifr_name));
	if (ioctl(drv->common.sock, SIOCGIFINDEX, &ifr) != 0) {
		wpa_printf(MSG_ERROR, "ioctl(SIOCGIFINDEX): %s",
			   strerror(errno));
		return -1;
	}

	os_memset(&addr, 0, sizeof(addr));
	addr.sll_family = AF_PACKET;
	addr.sll_ifindex = ifr.ifr_ifindex;
	drv->common.ifindex = ifr.ifr_ifindex;
	wpa_printf(MSG_DEBUG, "Opening raw packet socket for ifindex %d",
		   addr.sll_ifindex);

	if (bind(drv->common.sock, (struct sockaddr *) &addr, sizeof(addr)) < 0)
	{
		wpa_printf(MSG_ERROR, "bind: %s", strerror(errno));
		return -1;
	}

	/* filter multicast address */
	if (wired_multicast_membership(drv->common.sock, ifr.ifr_ifindex,
				       pae_group_addr, 1) < 0) {
		wpa_printf(MSG_ERROR, "wired: Failed to add multicast group "
			   "membership");
		return -1;
	}

	os_memset(&ifr, 0, sizeof(ifr));
	os_strlcpy(ifr.ifr_name, drv->common.ifname, sizeof(ifr.ifr_name));
	if (ioctl(drv->common.sock, SIOCGIFHWADDR, &ifr) != 0) {
		wpa_printf(MSG_ERROR, "ioctl(SIOCGIFHWADDR): %s",
			   strerror(errno));
		return -1;
	}

	if (ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER) {
		wpa_printf(MSG_INFO, "Invalid HW-addr family 0x%04x",
			   ifr.ifr_hwaddr.sa_family);
		return -1;
	}
	os_memcpy(own_addr, ifr.ifr_hwaddr.sa_data, ETH_ALEN);

	/* setup dhcp listen socket for sta detection */
	if ((drv->dhcp_sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
		wpa_printf(MSG_ERROR, "socket call failed for dhcp: %s",
			   strerror(errno));
		return -1;
	}

	if (eloop_register_read_sock(drv->dhcp_sock, handle_dhcp,
				     drv->common.ctx, NULL)) {
		wpa_printf(MSG_INFO, "Could not register read socket");
		return -1;
	}

	os_memset(&addr2, 0, sizeof(addr2));
	addr2.sin_family = AF_INET;
	addr2.sin_port = htons(67);
	addr2.sin_addr.s_addr = INADDR_ANY;

	if (setsockopt(drv->dhcp_sock, SOL_SOCKET, SO_REUSEADDR, (char *) &n,
		       sizeof(n)) == -1) {
		wpa_printf(MSG_ERROR, "setsockopt[SOL_SOCKET,SO_REUSEADDR]: %s",
			   strerror(errno));
		return -1;
	}
	if (setsockopt(drv->dhcp_sock, SOL_SOCKET, SO_BROADCAST, (char *) &n,
		       sizeof(n)) == -1) {
		wpa_printf(MSG_ERROR, "setsockopt[SOL_SOCKET,SO_BROADCAST]: %s",
			   strerror(errno));
		return -1;
	}

	os_memset(&ifr, 0, sizeof(ifr));
	os_strlcpy(ifr.ifr_ifrn.ifrn_name, drv->common.ifname, IFNAMSIZ);
	if (setsockopt(drv->dhcp_sock, SOL_SOCKET, SO_BINDTODEVICE,
		       (char *) &ifr, sizeof(ifr)) < 0) {
		wpa_printf(MSG_ERROR,
			   "setsockopt[SOL_SOCKET,SO_BINDTODEVICE]: %s",
			   strerror(errno));
		return -1;
	}

	if (bind(drv->dhcp_sock, (struct sockaddr *) &addr2,
		 sizeof(struct sockaddr)) == -1) {
		wpa_printf(MSG_ERROR, "bind: %s", strerror(errno));
		return -1;
	}

	return 0;
#else /* __linux__ */
	return -1;
#endif /* __linux__ */
}


static int wired_send_eapol(void *priv, const u8 *addr,
			    const u8 *data, size_t data_len, int encrypt,
			    const u8 *own_addr, u32 flags)
{
	struct wpa_driver_wired_data *drv = priv;
	struct ieee8023_hdr *hdr;
	size_t len;
	u8 *pos;
	int res;

	len = sizeof(*hdr) + data_len;
	hdr = os_zalloc(len);
	if (hdr == NULL) {
		wpa_printf(MSG_INFO,
			   "malloc() failed for wired_send_eapol(len=%lu)",
			   (unsigned long) len);
		return -1;
	}

	os_memcpy(hdr->dest, drv->use_pae_group_addr ? pae_group_addr : addr,
		  ETH_ALEN);
	os_memcpy(hdr->src, own_addr, ETH_ALEN);
	hdr->ethertype = htons(ETH_P_PAE);

	pos = (u8 *) (hdr + 1);
	os_memcpy(pos, data, data_len);

	res = send(drv->common.sock, (u8 *) hdr, len, 0);
	os_free(hdr);

	if (res < 0) {
		wpa_printf(MSG_ERROR,
			   "wired_send_eapol - packet len: %lu - failed: send: %s",
			   (unsigned long) len, strerror(errno));
	}

	return res;
}

void wired_delete_tc_qdisc (char *ifname)
{
	char tcargs[128];

	sprintf(tcargs, "qdisc del dev %s handle ffff: ingress", ifname);
	if (os_exec(tccommand, tcargs, 1)) {
		wpa_printf(MSG_DEBUG, "WIRED: could not delete ingress tc");
	}

	sprintf(tcargs, "qdisc del dev %s root handle 1: prio", ifname);
	if (os_exec(tccommand, tcargs, 1)) {
		wpa_printf(MSG_ERROR, "WIRED: could not delete egress tc");
	}
}

static int wired_tc_exec (FILE *tc_f, char *tcargs) {
	fprintf(tc_f, "%s\n", tcargs);
	if (os_exec(tccommand, tcargs, 1)) {
	    wpa_printf(MSG_ERROR, "WIRED: Error: could not run tc");
	    return -1;
	}

	return 0;
}

static void wired_update_mab_sock_filters(struct wpa_driver_wired_data *drv,
				int mac_whitelist,
				struct sta_info *changed_sta,
				Boolean auth)
{
	struct hostapd_data *hapd = drv->common.ctx;
	struct sta_info *sta;
	int jt_count = 0;
	int jf_count = 0;
	int i = 0, j;
	int len;
	struct sock_fprog bpf;

	if (mac_whitelist) {
		memset(mab_mac_sock_filter, 0, sizeof(mab_mac_sock_filter));
		jt_count = (mac_whitelist * 3) + 1;
		jf_count = (mac_whitelist - 1) * 3;
		len = sizeof(mab_sock_filter) / sizeof (mab_sock_filter[0]);

		mab_mac_sock_filter[i].code = 0x20;
		mab_mac_sock_filter[i].jt = 0;
		mab_mac_sock_filter[i].jf = 0;
		mab_mac_sock_filter[i++].k = 0x00000008;

		for (sta = hapd->sta_list; sta; sta = sta->next) {
			wpa_printf(MSG_DEBUG, "WIRED: sta addr " MACSTR " flags=%x",
					MAC2STR(sta->addr), sta->flags);
			if (!auth && sta == changed_sta)
				continue;

			/* we only allow traffic for authorized macs */
			if ((sta->flags & WLAN_STA_AUTHORIZED) ||
				(auth && (sta == changed_sta))) {
				mab_mac_sock_filter[i].code = 0x15;
				mab_mac_sock_filter[i].jt = 0;
				mab_mac_sock_filter[i].jf = 2;
				mab_mac_sock_filter[i++].k = MACLAST4HEX(sta->addr);


				mab_mac_sock_filter[i].code = 0x28;
				mab_mac_sock_filter[i].jt = 0;
				mab_mac_sock_filter[i].jf = 0;
				mab_mac_sock_filter[i++].k = 0x00000006;

				mab_mac_sock_filter[i].code = 0x15;
				mab_mac_sock_filter[i].jt = jt_count;
				mab_mac_sock_filter[i].jf = jf_count;
				mab_mac_sock_filter[i++].k = MAC2HEX(sta->addr);

				jt_count -= 3;
				jf_count -= 3;
			}
		}

		for (j = 0; j < len; j++) {
			mab_mac_sock_filter[i++] = mab_sock_filter[j];
		}

		wpa_printf(MSG_ERROR, "%s: entries %d", __FUNCTION__, i);
		for (j = 0; j < i; j++) {
			wpa_printf(MSG_ERROR, "{ 0x%x, %d, %d, 0x%x },",
				mab_mac_sock_filter[j].code,
				mab_mac_sock_filter[j].jt,
				mab_mac_sock_filter[j].jf,
				mab_mac_sock_filter[j].k);
		}

		bpf.len = i;
		bpf.filter = mab_mac_sock_filter;
	} else {
		bpf.len = sizeof(mab_sock_filter) / sizeof (mab_sock_filter[0]);
		bpf.filter = mab_sock_filter;
	}

	if (setsockopt(drv->mab_sock, SOL_SOCKET, SO_ATTACH_FILTER, &bpf,
			sizeof(bpf)) < 0) {
		wpa_printf(MSG_ERROR, "Setting mab socket filter failed %s", strerror(errno));
	}
}

static void wired_add_def_rules (struct wpa_driver_wired_data *drv,
								FILE *acl_f, FILE *def_acl_f)
{
	char *line = NULL;
	size_t len = 0;
	ssize_t read;
	char *token;
	char *extra;
	char *action;

	while ((read = getline(&line, &len, def_acl_f)) != -1) {
		token = os_strstr(line, "-j");
		if (token) {
			action = strtok(token + 3, " ");
			extra = token + (3 + strlen(action));
			token[0] = '\0';
			fprintf(acl_f, "%s -i %s -j mark --set-mark 2 %s\n",
						line, drv->common.ifname, extra);
			fprintf(acl_f, "%s -i %s -j %s %s\n",
						line, drv->common.ifname, action, extra);
		}
	}

	fclose(def_acl_f);
	if (line) {
		free(line);
	}
}

static int wired_handle_ebtable_rules(struct wpa_driver_wired_data *drv,
					struct sta_info *changed_sta,
					Boolean auth)
{
	struct hostapd_data *hapd;
	hapd = drv->common.ctx;
	struct sta_info *sta;
	char rulefile[128];
	char tcrulefile[128];
	char dacldeffile[128];
	char tcargs[512];
	int i = 0;
	int j = 0;
	int starting_prio = 2;
	struct stat st = {0};
	FILE *tc_f = NULL;
	FILE *f = NULL;
	FILE *dacl_def_f = NULL;
	int delete_tc = 0;
	int mac_whitelist = 0;

	hapd = drv->common.ctx;
	wpa_printf(MSG_DEBUG, "WIRED: wired_handle_ebtable_rules called");

	/* rewrite the rule file for this interface */
	sprintf(rulefile, "/etc/cumulus/acl/policy.d/tmp/200_dot1x_%s.rules",
			drv->common.ifname);
	sprintf(tcrulefile, "/var/lib/hostapd/acl/tc_%s.rules", drv->common.ifname);

	if (stat(tcrulefile, &st) != -1)
		delete_tc = 1;

	tc_f = fopen(tcrulefile, "w");
	if (!tc_f) {
		wpa_printf(MSG_ERROR, "WIRED: ERROR: could not write tc rule file %s",
				tcrulefile);
		return -1;
	}
	f = fopen(rulefile, "w");
	if (!f) {
		wpa_printf(MSG_ERROR, "WIRED: ERROR: could not write ebtables rule"
				" file %s", rulefile);
		return -1;
	}

	fprintf(f, "# Generated 802.1x EAPOL EBTABLES Rules File: Do not edit.\n");
	fprintf(f, "[ebtables]\n");

	fprintf(tc_f, "# Generated 802.1x EAPOL TC Rules File: Do not edit.\n");

	if (delete_tc) {
		wired_delete_tc_qdisc (drv->common.ifname);
	}

	sprintf(tcargs, "qdisc add dev %s root handle 1: prio", drv->common.ifname);
	if (wired_tc_exec(tc_f, tcargs) < 0)
		return -1;

	sprintf(tcargs, "filter add dev %s parent 1: protocol 0x888E prio %d"
			" flower flowid :1 action ok",
			drv->common.ifname, starting_prio + ++i);
	if (wired_tc_exec(tc_f, tcargs) < 0)
		return -1;

	sprintf(tcargs, "qdisc add dev %s handle ffff: ingress", drv->common.ifname);
	if (wired_tc_exec(tc_f, tcargs) < 0)
		return -1;

	if (hapd->conf->mab_enabled) {
		sprintf(tcargs, "filter add dev %s parent ffff: prio %d matchall"
				" flowid :1 action drop", drv->common.ifname, DROP_PRIORITY);
		if (wired_tc_exec(tc_f, tcargs) < 0)
			return -1;

		sprintf(tcargs, "filter add dev %s parent ffff: protocol 0x888E prio %d"
				" flower flowid :1 action ok",
				drv->common.ifname, starting_prio + ++j);
		if (wired_tc_exec(tc_f, tcargs) < 0)
			return -1;
	}

	for (sta = hapd->sta_list; sta; sta = sta->next) {
		wpa_printf(MSG_DEBUG, "WIRED: sta addr " MACSTR " flags=%x",
				MAC2STR(sta->addr), sta->flags);
		if (!auth && sta == changed_sta)
			continue;

		/* we only allow traffic for authorized macs */
		if ((sta->flags & WLAN_STA_AUTHORIZED) ||
			(auth && (sta == changed_sta))) {
			mac_whitelist++;

			if (sta->flags & WLAN_STA_TAGGED) {
				fprintf(f, "-A INPUT%s -i %s -p 8021q --vlan-id %d -s " MACSTR
						" -j mark --set-mark 2\n",
						(hapd->conf->dacl_enabled) ? "" : ",FORWARD",
						drv->common.ifname, sta->vlan_id, MAC2STR(sta->addr));
				fprintf(f, "-A INPUT%s -i %s -p 8021q --vlan-id %d -s " MACSTR
						" -j ACCEPT\n",
						(hapd->conf->dacl_enabled) ? "" : ",FORWARD",
						drv->common.ifname, sta->vlan_id, MAC2STR(sta->addr));
			}
			fprintf(f, "-A INPUT%s -i %s -p ! 8021q -s " MACSTR " -j mark --set-mark 2\n",
					(hapd->conf->dacl_enabled) ? "" : ",FORWARD",
					drv->common.ifname, MAC2STR(sta->addr));
			fprintf(f, "-A INPUT%s -i %s -p ! 8021q -s " MACSTR " -j ACCEPT\n",
					(hapd->conf->dacl_enabled) ? "" : ",FORWARD",
					drv->common.ifname, MAC2STR(sta->addr));

			sprintf(tcargs, "filter add dev %s parent 1: prio %d u32 match u32 0x"
					COMPACT_MAC4BYTE " 0xFFFFFFFF at -12 match u16 0x"
					COMPACT_MAC2BYTE " 0xFFFF at -14 flowid :1 action pass",
					drv->common.ifname,  ++i + starting_prio,
					MACLAST4BYTE(sta->addr), MAC2BYTE(sta->addr));
			if (wired_tc_exec(tc_f, tcargs) < 0)
				return -1;

			if (hapd->conf->mab_enabled) {
				sprintf(tcargs, "filter add dev %s parent ffff: prio %d"
						" u32 match u32 0x" COMPACT_MAC2BYTE
						"0000 0xFFFF0000 at -4 match u32 0x" COMPACT_MAC4BYTE
						" 0xFFFFFFFF at -8 flowid :1 action ok",
						drv->common.ifname,  ++j + starting_prio,
						MACLAST2BYTE(sta->addr), MAC4BYTE(sta->addr));
				if (wired_tc_exec(tc_f, tcargs) < 0)
					return -1;
			}
		}
	}

	if (mac_whitelist) {
		sprintf(tcargs, "filter add dev %s parent 1: prio %d u32 match u8"
				" 0x01 0x01 at -14 flowid :1 action pass",
				drv->common.ifname, ++i + starting_prio);
		if (wired_tc_exec(tc_f, tcargs) < 0)
			return -1;
	} else {
		if (hapd->conf->dacl_enabled) {
			sprintf(tcargs, "filter add dev %s parent 1: protocol arp prio %d"
					" flower flowid :1 action ok",
					drv->common.ifname, starting_prio + ++i);
			if (wired_tc_exec(tc_f, tcargs) < 0)
				return -1;

			sprintf(tcargs, "filter add dev %s parent 1: prio %d protocol ip"
				" u32 match ip dport 0x0044 0xffff flowid :1 action pass",
					drv->common.ifname, starting_prio + ++i);
			if (wired_tc_exec(tc_f, tcargs) < 0)
				return -1;

			sprintf(tcargs, "filter add dev %s parent 1: prio %d protocol ip"
				" u32 match ip dport 0x0043 0xffff flowid :1 action pass",
					drv->common.ifname, starting_prio + ++i);
			if (wired_tc_exec(tc_f, tcargs) < 0)
				return -1;
		}
	}

	if (hapd->conf->dacl_enabled) {
		if (hapd->conf->mab_enabled) {
			sprintf(tcargs, "filter add dev %s parent ffff: protocol arp prio %d"
					" flower flowid :1 action ok",
					drv->common.ifname, starting_prio + ++j);
			if (wired_tc_exec(tc_f, tcargs) < 0)
				return -1;

			sprintf(tcargs, "filter add dev %s parent ffff: prio %d protocol ip"
				" u32 match ip dport 0x0044 0xffff flowid :1 action pass",
					drv->common.ifname, starting_prio + ++j);
			if (wired_tc_exec(tc_f, tcargs) < 0)
				return -1;

			sprintf(tcargs, "filter add dev %s parent ffff: prio %d protocol ip"
				" u32 match ip dport 0x0043 0xffff flowid :1 action pass",
					drv->common.ifname, starting_prio + ++j);
			if (wired_tc_exec(tc_f, tcargs) < 0)
				return -1;
		}
		sprintf(dacldeffile, "/etc/cumulus/acl/policy.d/dot1x_preauth_dacl/%s",
			(strlen(hapd->conf->def_dacl) > 0) ?  hapd->conf->def_dacl :
				"default_preauth_dacl.rules");
		dacl_def_f = fopen(dacldeffile, "r");
		if (!dacl_def_f) {
			wpa_printf(MSG_ERROR, "WIRED: ERROR: could not read default dacl rule"
					" file %s", dacldeffile);
		} else {
			wired_add_def_rules(drv, f, dacl_def_f);
		}
	}

	if (hapd->conf->mab_enabled && drv->mab_sock) {
		wired_update_mab_sock_filters(drv, mac_whitelist, changed_sta,
										auth);
	}

	if (mac_whitelist || !hapd->conf->dacl_enabled) {
		sprintf(tcargs, "filter add dev %s parent 1: prio %d matchall flowid :1 action drop",
			drv->common.ifname, DROP_PRIORITY);
		if (wired_tc_exec(tc_f, tcargs) < 0)
			return -1;
	}

	fprintf(f, "-A INPUT -i %s -p 0x888E -j mark --set-mark 2\n", drv->common.ifname);
	fprintf(f, "-A INPUT -i %s -p 0x888E -j police --set-mode pkt --set-rate 100 --set-burst 100\n", drv->common.ifname);

	if (hapd->conf->mab_enabled) {
		fprintf(f, "-A FORWARD -i %s -j mark --set-mark 3\n", drv->common.ifname);
		fprintf(f, "-A FORWARD -i %s -j DROP\n", drv->common.ifname);
	} else {
		fprintf(f, "-A INPUT,FORWARD -i %s -j mark --set-mark 2\n", drv->common.ifname);
		fprintf(f, "-A INPUT,FORWARD -i %s -j DROP\n", drv->common.ifname);
	}

	fclose(f);
	fclose(tc_f);

	return 0;
}

static int wired_start_mab(struct wpa_driver_wired_data *drv)
{
	struct sockaddr_ll addr;
	struct sock_fprog bpf =
	{
		.len = sizeof(mab_sock_filter) / sizeof (mab_sock_filter[0]),
		.filter = mab_sock_filter
	};

	wpa_printf(MSG_DEBUG, "wired_start_mab called:");
	if (drv->mab_sock)
		return 0;

	drv->mab_sock = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
	if (drv->mab_sock < 0) {
		wpa_printf(MSG_ERROR, "socket[PF_PACKET,SOCK_RAW,ETH_P_ALL]: %s",
					strerror(errno));
		return -1;
	}
	wpa_printf(MSG_DEBUG, "wired_start_mab: created new socket");

	if (eloop_register_read_sock(drv->mab_sock, handle_read,
				     drv->common.ctx, NULL)) {
		wpa_printf(MSG_INFO, "Could not register read socket");
		return -1;
	}

	os_memset(&addr, 0, sizeof(addr));
	addr.sll_family = AF_PACKET;
	addr.sll_ifindex = drv->common.ifindex;
	wpa_printf(MSG_DEBUG, "Opening raw packet socket for ifindex %d",
				addr.sll_ifindex);

	if (bind(drv->mab_sock, (struct sockaddr *) &addr, sizeof(addr)) < 0)
	{
		wpa_printf(MSG_ERROR, "bind: %s", strerror(errno));
		return -1;
	}

	if (setsockopt(drv->mab_sock, SOL_SOCKET, SO_ATTACH_FILTER, &bpf,
			sizeof(bpf)) < 0) {
		wpa_printf(MSG_ERROR, "Setting mab socket filter for %d failed",
				addr.sll_ifindex);
		return-1;
	}

	wired_handle_ebtable_rules(drv, NULL, 0);

	return 0;
}

static void wired_stop_mab(struct wpa_driver_wired_data *drv)
{
	if (drv->mab_sock > 0) {
		eloop_unregister_read_sock(drv->mab_sock);
		close(drv->mab_sock);
		drv->mab_sock = 0;
		wpa_printf(MSG_DEBUG, "wired_stop_mab: %s closing socket"
			" and stopped reading", drv->common.ifname);
	}
}

void wired_mab_deactivation(struct hostapd_data *hapd)
{
	struct wpa_driver_wired_data *drv = hapd->drv_priv;

	if (drv->mab_sock) {
		wpa_printf(MSG_DEBUG, "wired_mab_deactivation: stopping mab");
		wired_stop_mab(drv);
	}
}

void wired_init_mab(struct hostapd_data *hapd)
{
	struct wpa_driver_wired_data *drv = hapd->drv_priv;

	/*
	 * for slow hardware, we do not want to start another
	 * timer to create a mab socket since this will reset ACL rules
	 * so we need to check for an existing mab socket
	 */
	if (drv->mab_sock) {
		wpa_printf(MSG_ERROR, "wired_init_mab: MAB socket exists!");
		return;
	}

	wired_del_dynamic_mac(drv);

	if (hapd->conf->mab_enabled)
		wired_start_mab(drv);
}

void
wired_apply_dacl_change (struct hostapd_data *hapd)
{
	struct wpa_driver_wired_data *drv = hapd->drv_priv;
	int auth_stas = hapd->num_sta;
	struct sta_info *sta;

	wired_mab_deactivation(hapd);
	for (sta = hapd->sta_list; sta; sta = sta->next) {
		if (sta->flags & WLAN_STA_AUTHORIZED) {
			hostapd_drv_sta_deauth(hapd, sta->addr,
					WLAN_REASON_DISASSOC_STA_HAS_LEFT);
			ap_sta_deauthenticate(hapd, sta, WLAN_REASON_DISASSOC_STA_HAS_LEFT);
			auth_stas++;
		}
	}
	wired_init_mab(hapd);
	if (!auth_stas && !hapd->conf->mab_enabled)
		wired_handle_ebtable_rules(drv, NULL, 0);
}

static int wired_init_interface(struct wpa_driver_wired_data *drv)
{
	struct wired_global *global = drv->global;
	struct rtnl_link *link;
	struct rtnl_link_bridge_vlan *bv;
	char *name;

	link = wpa_nl_get_link_by_family(global->cache, AF_BRIDGE,
						drv->common.ifindex);

	if (link) {
		drv->if_info.is_bridge = rtnl_link_is_bridge(link);

		drv->if_info.flags = rtnl_link_get_flags(link);
		name = rtnl_link_get_name(link);
		wpa_printf(MSG_DEBUG, "WIRED: link found in the cache for %s %d",
					name?name:"-", drv->if_info.is_bridge);

		bv = rtnl_link_bridge_get_port_vlan(link);
		if (bv)
			drv->if_info.pvid = bv->pvid;
		rtnl_link_put(link);
	} else {
		link = wpa_nl_get_link_by_family(global->cache, 0,
						 drv->common.ifindex);
		if (link) {
			drv->if_info.flags = rtnl_link_get_flags(link);
			rtnl_link_put(link);
		} else {
			wpa_printf(MSG_ERROR, "WIRED: Error: link not found in the cache for %s",
							drv->common.ifname);
		}
	}

	drv->if_info.if_node = if_hash_table_find(local->if_hash,
								drv->common.ifindex);
	return 0;
}

static void * wired_driver_hapd_init(struct hostapd_data *hapd,
				     struct wpa_init_params *params)
{
	struct wpa_driver_wired_data *drv;

	drv = os_zalloc(sizeof(struct wpa_driver_wired_data));
	if (drv == NULL) {
		wpa_printf(MSG_INFO,
			   "Could not allocate memory for wired driver data");
		return NULL;
	}

	drv->global = params->global_priv;
	drv->common.ctx = hapd;

	os_strlcpy(drv->common.ifname, params->ifname,
		   sizeof(drv->common.ifname));
	drv->use_pae_group_addr = params->use_pae_group_addr;

	wpa_printf(MSG_DEBUG, "wired_driver_hapd_init called iface=%s",
			drv->common.ifname);

	if (wired_init_sockets(drv, params->own_addr)) {
		os_free(drv);
		return NULL;
	}

	wired_init_interface(drv);

	/* initialize the ebtable rules for interface */
	/* this call will be made later in hostapd_setup_interface()
	   so no need to call it now.
	*/
	if (wired_handle_ebtable_rules(drv, NULL, 0))
		return NULL;

	hapd->drv_priv = drv;
	wired_init_mab(hapd);

	dl_list_add(&drv->global->interfaces, &drv->if_list);
	return drv;
}

static void wired_driver_hapd_deinit(void *priv)
{
	struct wpa_driver_wired_data *drv = priv;
	struct hostapd_data *hapd;

	hapd = drv->common.ctx;

	wpa_printf(MSG_DEBUG, "wired_driver_hapd_deinit called iface=%s",
			drv->common.ifname);

	if (drv->common.sock >= 0) {
		eloop_unregister_read_sock(drv->common.sock);
		close(drv->common.sock);
	}

	if (drv->dhcp_sock >= 0) {
		eloop_unregister_read_sock(drv->dhcp_sock);
		close(drv->dhcp_sock);
	}

	if (drv->mab_sock >= 0) {
		eloop_unregister_read_sock(drv->mab_sock);
		close(drv->mab_sock);
	}

	/* we should remove the interface filters and any dynamic vlans if the driver is going away */
	hostapd_handle_acl_pvid_removal(hapd);

	dl_list_del(&drv->if_list);
	os_free(drv);
}

static void wired_driver_ifup(void *priv)
{
	struct wpa_driver_wired_data *drv = priv;
	int flags;

	wpa_printf(MSG_INFO, "wired_driver_ifup called");
	if (driver_wired_get_ifflags(drv->common.ifname, &flags) == 0 &&
			driver_wired_set_ifflags(drv->common.ifname, flags | IFF_UP) < 0)
		wpa_printf(MSG_WARNING, "%s: Failed to set the interface %s up",
				__func__, drv->common.ifname);
	else
		wpa_printf(MSG_DEBUG, "%s: Brought up interface %s",
				__func__, drv->common.ifname);
}

static void wired_driver_ifdown(void *priv)
{
	struct wpa_driver_wired_data *drv = priv;
	int flags;

	wpa_printf(MSG_INFO, "wired_driver_ifdown called");
	if (driver_wired_get_ifflags(drv->common.ifname, &flags) == 0 &&
			driver_wired_set_ifflags(drv->common.ifname, flags & ~IFF_UP) < 0)
		wpa_printf(MSG_WARNING, "%s: Failed to set the interface %s down",
				__func__, drv->common.ifname);
	else
		wpa_printf(MSG_DEBUG, "%s: Brought down interface %s",
				__func__, drv->common.ifname);
}

static void * wpa_driver_wired_init(void *ctx, const char *ifname)
{
	struct wpa_driver_wired_data *drv;

	drv = os_zalloc(sizeof(*drv));
	if (drv == NULL)
		return NULL;

	if (driver_wired_init_common(&drv->common, ifname, ctx) < 0) {
		os_free(drv);
		return NULL;
	}

	return drv;
}

static void wpa_driver_wired_deinit(void *priv)
{
	struct wpa_driver_wired_data *drv = priv;
	wpa_printf(MSG_INFO, "wpa_driver_wired_deinit called");

	driver_wired_deinit_common(&drv->common);
	os_free(drv);
}

static int
wired_del_dynamic_mac(struct wpa_driver_wired_data *drv)
{

	struct wired_global *global = drv->global;
	struct rtnl_link *link;
	struct rtnl_link_bridge_vlan *bv;
	char name[BUFF_LEN];
	char flushcmd[BUFF_LEN];
	int br_index;
	FILE *fp = NULL;

	link = wpa_nl_get_link_by_family(global->cache, AF_BRIDGE,
			drv->common.ifindex);

	if (link) {
		br_index = rtnl_link_get_master(link);
		if (br_index == 0 ) {
			rtnl_link_put(link);
			wpa_printf(MSG_ERROR,"%s: %s Not part of any bridge",
					__FUNCTION__,drv->common.ifname);
			return 0;
		}
		rtnl_link_i2name(global->cache,br_index,name,BUFF_LEN);
		snprintf(flushcmd,BUFF_LEN-1,"echo 1 > /sys/class/net/%s/brif/%s/flush",
				name,drv->common.ifname);
		flushcmd[BUFF_LEN-1] = '\0';
		wpa_printf(MSG_DEBUG, "%s: command: %s", __FUNCTION__,flushcmd);
		fp = popen(flushcmd,"r");
		if(!fp)
			wpa_printf(MSG_ERROR,"%s: Failed to delete dynamic MACs",__FUNCTION__);
		else
			pclose(fp);
		rtnl_link_put(link);
	}
	return 0;
}

#define BUFSIZE 256
#define CMDSIZE 128

static int
wired_del_dyn_fdb_entry(const u8 *addr)
{
	char buf[BUFSIZE];
	FILE *fp;
	char cmd[CMDSIZE];
	char *tok, *dev, *vlan;
	const char dev_s[6] = "dev ";
	const char vlan_s[6] = "vlan ";
	char *cmd_prefix = "bridge fdb show | grep ";
	char bridge_args[254];
	
	sprintf(cmd,"%s" MACSTR "", cmd_prefix,MAC2STR(addr));

	if ((fp = popen(cmd, "r")) == NULL) {
		wpa_printf(MSG_ERROR,"Error opening pipe!\n");
		return -1;
	}

	while (fgets(buf, BUFSIZE, fp) != NULL) {
		wpa_printf(MSG_INFO,"OUTPUT: %s", buf);
		tok = strtok(buf, dev_s);
		dev = strtok(0, dev_s);
		vlan = strtok(0, vlan_s);
		if (dev != NULL && vlan != NULL) 
			sprintf(bridge_args, "fdb del " MACSTR " dev %s vlan %s master",
							MAC2STR(addr), dev, vlan);
		else if (dev != NULL)
			sprintf(bridge_args, "fdb del " MACSTR " dev %s master",
					    MAC2STR(addr), dev);
		if (os_exec(bridge_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not del bridge fdb entry");
			return -1;
		}
		wpa_printf(MSG_INFO, "%s: command: %s %s", __FUNCTION__,
			    bridge_command, bridge_args);
	}

	if(pclose(fp))  {
		wpa_printf(MSG_ERROR, "Command not found or exited with error status\n");
		return -1;
	}
	return 0;
}

static int
wired_add_fdb_entry (struct wpa_driver_wired_data *drv, const u8 *addr,
		int vlan, Boolean tagged)
{
	char bridge_args[254];
	struct hostapd_data *hapd = drv->common.ctx;

        wired_del_dyn_fdb_entry(addr);
	if (wired_driver_hapd_is_l2_if(drv)) {
		if (!wired_driver_is_vlan_aware_if(&drv->if_info))
			if (tagged)
				sprintf (bridge_args, "fdb add " MACSTR " dev %s.%d master static",
						MAC2STR(addr), drv->common.ifname, vlan);
			else
				sprintf (bridge_args, "fdb add " MACSTR " dev %s master static",
						MAC2STR(addr), drv->common.ifname);
		else if (drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED)
			sprintf (bridge_args, "fdb add " MACSTR " dev %s vlan %d master static",
						MAC2STR(addr), drv->common.ifname, hapd->conf->pvlan_id);
		else if (vlan)
			sprintf (bridge_args, "fdb add " MACSTR " dev %s vlan %d master static",
						MAC2STR(addr), drv->common.ifname, vlan);
		else
			sprintf (bridge_args, "fdb add " MACSTR " dev %s master static",
						MAC2STR(addr), drv->common.ifname);
		if (os_exec(bridge_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not add bridge fdb entry");
			return -1;
		}
		wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
				bridge_command, bridge_args);
	} else {
		/* we should not be doing anything for non-L2 interfaces */
		wpa_printf(MSG_ERROR, "WIRED: Cannot add fdb entries to non-L2 Interface for %s",
				drv->common.ifname);
	}

	return 0;
}

static int
wired_del_fdb_entry (struct wpa_driver_wired_data *drv, const u8 *addr,
						int vlan, Boolean tagged)
{
	char bridge_args[254];
	struct hostapd_data *hapd = drv->common.ctx;

	if (wired_driver_hapd_is_l2_if(drv)) {
		if (!wired_driver_is_vlan_aware_if(&drv->if_info))
			if (tagged)
				sprintf (bridge_args, "fdb del " MACSTR " dev %s.%d master static",
						MAC2STR(addr), drv->common.ifname, vlan);
			else
				sprintf (bridge_args, "fdb del " MACSTR " dev %s master static",
						MAC2STR(addr), drv->common.ifname);
		else if (drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED)
			sprintf (bridge_args, "fdb del " MACSTR " dev %s vlan %d master static",
						MAC2STR(addr), drv->common.ifname, hapd->conf->pvlan_id);
		else if (vlan)
			sprintf (bridge_args, "fdb del " MACSTR " dev %s vlan %d master static",
						MAC2STR(addr), drv->common.ifname, vlan);
		else
			sprintf (bridge_args, "fdb del " MACSTR " dev %s master static",
						MAC2STR(addr), drv->common.ifname);

		if (os_exec(bridge_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not del bridge fdb entry");
			return -1;
		}
		wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
				bridge_command, bridge_args);
	} else {
		/* we should not be doing anything for non-L2 interfaces */
		wpa_printf(MSG_ERROR, "WIRED: Cannot del fdb entries to non-L2 Interface for %s",
				drv->common.ifname);
	}

	return 0;
}

static int
wired_modify_traditional_bridge_ports(struct wpa_driver_wired_data *drv,
		if_info_t *old_br_info, if_info_t *new_br_info)
{
	char old_bridge_filename[128];
	char new_bridge_filename[128];
	FILE *new_bridge_file = NULL;

	/* add a new config file if needed */
	if (new_br_info && new_br_info->ifname) {
		snprintf(new_bridge_filename, sizeof(new_bridge_filename),
				"/etc/network/interfaces.d/trad_bridge_dot1x_%s_%s.intf", new_br_info->ifname,
				drv->common.ifname);
		/* Handle ifupdown bridge access PVID assignment file for dynamic VLANs */
		new_bridge_file = fopen(new_bridge_filename, "w");
		if (new_bridge_file) {
			fprintf(new_bridge_file, "# hostapd generated bridge config\nauto %s\niface %s\n    bridge-ports %s\n",
					new_br_info->ifname, new_br_info->ifname, drv->common.ifname);
			fclose(new_bridge_file);
			wpa_printf(MSG_DEBUG, "Wrote new bridge bridge-ports config %s", new_bridge_filename);
		} else {
			wpa_printf(MSG_ERROR, "Could not write %s", new_bridge_filename);
			return -1;
		}
	}

	/* remove the old config if it exists */
	if (old_br_info && old_br_info->ifname) {
		snprintf(old_bridge_filename, sizeof(old_bridge_filename),
				"/etc/network/interfaces.d/trad_bridge_dot1x_%s_%s.intf", old_br_info->ifname,
				drv->common.ifname);
		if (remove(old_bridge_filename)) {
			/* the file may or may not be there so we tried anyway */
			wpa_printf(MSG_INFO, "Did not remove old traditional bridge config file %s",
					old_bridge_filename);
			return 0;
		} else {
			wpa_printf(MSG_DEBUG, "removed traditional bridge file %s", old_bridge_filename);
		}

	}

	return 0;
}

static int
wired_modify_traditional_bridge_tagged_ports(struct wpa_driver_wired_data *drv,
		if_info_t *br_info, int vid, Boolean remove_only)
{
	char filename[128];
	FILE *file = NULL;

	/* add a new config file if needed */
	snprintf(filename, sizeof(filename),
			"/etc/network/interfaces.d/trad_bridge_dot1x_%s_%d.intf",
			drv->common.ifname, vid);
	if (remove_only) {
		if (remove(filename)) {
			/* the file may or may not be there so we tried anyway */
			wpa_printf(MSG_INFO, "Did not remove trad bridge port config file %s",
					filename);
			return -1;
		} else {
			wpa_printf(MSG_DEBUG, "removed trad bridge-port config file %s",
					filename);
		}
		return 0;
	}

	/* Handle ifupdown bridge access PVID assignment file for dynamic VLANs */
	file = fopen(filename, "w");
	if (file) {
		fprintf(file, "# hostapd generated bridge config\nauto %s.%d\niface %s.%d\n    bridge-learning off\n",
				drv->common.ifname, vid, drv->common.ifname, vid);
		if (br_info) {
			fprintf(file, "auto %s\niface %s\n    bridge-ports %s.%d\n",
				br_info->ifname, br_info->ifname, drv->common.ifname, vid);
		}
		fclose(file);
		wpa_printf(MSG_DEBUG, "Wrote new trad bridge port config %s", filename);
	} else {
		wpa_printf(MSG_ERROR, "Could not write %s", filename);
		return -1;
	}

	return 0;
}

static int
wired_modify_ifupdown_bridge_access_vlan(struct wpa_driver_wired_data *drv,
		int pvid,
		Boolean remove_only)
{
	char single_filename[64];
	FILE *iface_file = NULL;

	snprintf(single_filename, sizeof(single_filename),
			"/etc/network/interfaces.d/dot1x_%s_pvid.intf", drv->common.ifname);

	/* we may be called to clean up only */
	if (remove_only) {
		if (remove(single_filename)) {
			/* the file may or may not be there so we tried anyway */
			wpa_printf(MSG_INFO, "Did not remove bridge-pvid config file %s",
					single_filename);
			return -1;
		} else {
			wpa_printf(MSG_DEBUG, "removed bridge-pvid config file %s",
					single_filename);
		}
		return 0;

	}
	/* For this interface, we will overwrite any existing interface dynamic vlan config file*/
	if ((pvid > 0) && drv && drv->common.ifname) {
		/* Handle ifupdown bridge access PVID assignment file for dynamic VLANs */
		iface_file = fopen(single_filename, "w");
		if (iface_file) {
			fprintf(iface_file, "# IEEE 802.1x Dynamically Generated Dynamic VLAN or Parking VLAN Configuration File\niface %s\n    bridge-pvid %d\n",
				drv->common.ifname, pvid);
			fclose(iface_file);
			wpa_printf(MSG_DEBUG, "Wrote bridge-pvid config %s", single_filename);
		} else {
			wpa_printf(MSG_ERROR, "Could not write %s", single_filename);
			return -1;
		}
	} else {
		return -1;
	}

	return 0;
}

static int
wired_modify_ifupdown_bridge_vid(struct wpa_driver_wired_data *drv, int vid,
		Boolean remove_only)
{
	char single_filename[64];
	FILE *iface_file = NULL;

	snprintf(single_filename, sizeof(single_filename),
			"/etc/network/interfaces.d/dot1x_%s_vid.intf", drv->common.ifname);

	/* we may be called to clean up only */
	if (remove_only) {
		if (remove(single_filename)) {
			/* the file may or may not be there so we tried anyway */
			wpa_printf(MSG_INFO, "Did not remove bridge-vids config file %s",
					single_filename);
			return -1;
		} else {
			wpa_printf(MSG_DEBUG, "removed bridge-vids config file %s",
					single_filename);
		}
		return 0;

	}
	/* For this interface, we will overwrite any existing interface dynamic
	 * vlan config file*/
	if ((vid > 0) && drv && drv->common.ifname) {
		/* Handle ifupdown bridge VID assignment file for dynamic VLANs */
		iface_file = fopen(single_filename, "w");
		if (iface_file) {
			fprintf(iface_file, "# IEEE 802.1x Dynamically Generated Dynamic VLAN Configuration File\niface %s\n    bridge-vids %d\n",
					drv->common.ifname, vid);
			fclose(iface_file);
			wpa_printf(MSG_DEBUG, "Wrote bridge-vids config %s", single_filename);
		} else {
			wpa_printf(MSG_ERROR, "Could not write %s", single_filename);
			return -1;
		}
	} else {
		return -1;
	}

	return 0;
}

static int
wired_change_bridge (struct wpa_driver_wired_data *drv, int new_vid,
		int to_orig, struct sta_info *sta)
{
	char ip_args[254];
	if_info_t *new_br_info;
	if_info_t *old_br_info;
	if_info_t *old_if_info = drv->if_info.if_node;
	vlan_if_info_t *vlan_info;

	wpa_printf(MSG_DEBUG, "%s: %s %s vid %d",
		__FUNCTION__, drv->common.ifname,
		to_orig? "move to original bridge":"move to new bridge",
		new_vid);

	if (!wired_driver_hapd_is_l2_if(drv) || (!to_orig && !new_vid))
		return 0;

	vlan_info = (vlan_if_info_t *)old_if_info->data;

	if (to_orig) {
		new_br_info = if_hash_table_find(local->br_hash,
						vlan_info->orig_master_id);
		old_br_info = if_hash_table_find(local->br_hash,
						vlan_info->master_id);
	} else {
		new_br_info = if_hash_table_find_by_vlan(local->br_hash, new_vid);
		old_br_info = if_hash_table_find(local->br_hash,
						vlan_info->orig_master_id);
	}

	if (vlan_info->master == new_br_info) {
		wpa_printf(MSG_DEBUG, "%s Already %s in right bridge %s",
				__FUNCTION__, drv->common.ifname,
				new_br_info ? new_br_info->ifname : "na");
		return 0;
	}

	if (!new_br_info) {
		wpa_printf(MSG_DEBUG, "%s bridge not found for %s and vid %d",
				__FUNCTION__, drv->common.ifname, new_vid);
		if (to_orig)
			return -1;

		drv->flags |= WPA_WIRED_DATA_NO_VLAN;
		if (sta)
			sta->flags |= WLAN_STA_UNKNOWN_BR;
		return 0;
	}

	sprintf (ip_args, "link set dev %s master %s",
			drv->common.ifname, new_br_info->ifname);
	if (os_exec(ip_command, ip_args, 1)) {
		wpa_printf(MSG_ERROR, "%s: could not add %s to %s",
				__FUNCTION__, drv->common.ifname, new_br_info->ifname);
		return -1;
	}
	wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
			ip_command, ip_args);

	drv->flags |= WPA_WIRED_DATA_IGNORE_CHANGE;
	if (to_orig)
		drv->flags &= ~WPA_WIRED_DATA_VLAN_MOD;
	else
		drv->flags |= WPA_WIRED_DATA_VLAN_MOD;
	if_update_master(old_if_info, new_br_info);

	wired_modify_traditional_bridge_ports(drv, old_br_info, new_br_info);
	return 0;
}

static int
wired_change_pvid (struct wpa_driver_wired_data *drv, int old_pvid, int new_pvid,
		int to_orig)
{
	char bridge_args[254];

	wpa_printf(MSG_DEBUG, "%s: for %s old %d new %d",
			__FUNCTION__, drv->common.ifname, old_pvid, new_pvid);
	if (wired_driver_hapd_is_l2_if(drv)) {
		if (old_pvid != new_pvid) {
			if (new_pvid) {
				sprintf (bridge_args, "vlan add vid %d untagged pvid dev %s",
						new_pvid, drv->common.ifname);
				if (os_exec(bridge_command, bridge_args, 1)) {
					wpa_printf(MSG_ERROR, "WIRED: Error: could not add pvid for %s",
								drv->common.ifname);
					return -1;
				}
				wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
						bridge_command, bridge_args);
			}

			if (old_pvid) {
				sprintf (bridge_args, "vlan del vid %d untagged dev %s",
						old_pvid, drv->common.ifname);
				if (os_exec(bridge_command, bridge_args, 1)) {
					wpa_printf(MSG_ERROR, "WIRED: Error: could not del pvid for %s",
								drv->common.ifname);
					return -1;
				}
				wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
						bridge_command, bridge_args);
			}
		}
		/* if to_orig is true, we only remove so the pvid is ignored */
		wired_modify_ifupdown_bridge_access_vlan(drv, new_pvid, to_orig);

	} else {
		/* we should not be doing anything for non-L2 interfaces */
		wpa_printf(MSG_ERROR, "WIRED: Cannot change pvid for non-L2 Interface for %s",
				drv->common.ifname);
	}

	return 0;
}

static int
wired_update_pvid(struct wpa_driver_wired_data *drv, int old_vid, int new_vid,
                    int to_orig, struct sta_info *sta)
{
	if (wired_driver_is_vlan_aware_if(&drv->if_info)) {
		if (wired_change_pvid(drv, old_vid, new_vid, to_orig)) {
			wpa_printf(MSG_ERROR, "%s: error setting %s PVID %d",
					__FUNCTION__, drv->common.ifname, new_vid);
			return -1;
		}
	} else {
		if (wired_change_bridge(drv, new_vid, to_orig, sta)) {
			wpa_printf(MSG_ERROR, "%s: error changing %s bridge VLAN %d",
				__FUNCTION__, drv->common.ifname, new_vid);
			return -1;
		}
	}

	return 0;
}

static int
wired_add_vid (struct wpa_driver_wired_data *drv, int vid, struct sta_info *sta)
{
	char bridge_args[254];
	if_info_t *br_node = NULL;
	int send_lldp = TRUE;

	wpa_printf(MSG_ERROR, "%s: for %s add vid %d",
			__FUNCTION__, drv->common.ifname, vid);
	if (!wired_driver_hapd_is_l2_if(drv)) {
		/* we should not be doing anything for non-L2 interfaces */
		wpa_printf(MSG_ERROR, "WIRED: Cannot add vids to non-L2 Interface %s",
				drv->common.ifname);
		return 0;
	}

	if (wired_driver_is_vlan_aware_if(&drv->if_info)) {
		wpa_printf(MSG_ERROR, "%s: for vlan aware interface %s add vid %d",
		__FUNCTION__, drv->common.ifname, vid);

		sprintf (bridge_args, "vlan add vid %d dev %s",
				vid, drv->common.ifname);
		if (os_exec(bridge_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not add vid for %s",
						drv->common.ifname);
			return -1;
		}
		wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
				bridge_command, bridge_args);

		wired_modify_ifupdown_bridge_vid(drv, vid, FALSE);
	} else {
		wpa_printf(MSG_ERROR, "%s: for trad interface %s add vid %d",
		__FUNCTION__, drv->common.ifname, vid);

		sprintf (bridge_args, "link add link %s name %s.%d type vlan "
			" id %d protocol 802.1q", drv->common.ifname,
			drv->common.ifname, vid, vid);
		if (os_exec(ip_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not add interface"
				" %s.%d", drv->common.ifname, vid);
			return -1;
		}

		br_node = if_hash_table_find_by_vlan(local->br_hash, vid);
		if (br_node) {
			sprintf (bridge_args, "addif %s %s.%d", br_node->ifname,
				drv->common.ifname, vid);
			if (os_exec(brctl_command, bridge_args, 1)) {
				wpa_printf(MSG_ERROR, "WIRED: Error: could not add"
					" interface %s.%d to bridge %s", drv->common.ifname,
					vid, br_node->ifname);
				return -1;
			}
			sprintf (bridge_args, "link set dev %s.%d learning off",
				drv->common.ifname, vid);
			if (os_exec(bridge_command, bridge_args, 1)) {
				wpa_printf(MSG_ERROR, "WIRED: Error: could not set learning"
					" off for interface %s.%d", drv->common.ifname, vid);
				return -1;
			}

			sprintf (bridge_args, "link set dev %s.%d up",
				drv->common.ifname, vid);
			if (os_exec(ip_command, bridge_args, 1)) {
				wpa_printf(MSG_ERROR, "WIRED: Error: could not bring up"
					" interface %s.%d", drv->common.ifname, vid);
				return -1;
			}
		} else {
			sta->flags |= WLAN_STA_UNKNOWN_BR;
			send_lldp = FALSE;
		}

		wired_modify_traditional_bridge_tagged_ports(drv, br_node, vid,
										FALSE);
	}

	if (send_lldp) {
		sprintf (bridge_args, "configure ports %s med policy"
			" application voice tagged vlan %d priority voice dscp 46",
			drv->common.ifname, vid);

		if (os_exec(lldp_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not set"
				" LLDP MED for interface %s", drv->common.ifname);
			return -1;
		}
		wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
				lldp_command, bridge_args);
	}
	return 0;
}

static int
wired_remove_vid (struct wpa_driver_wired_data *drv, int vid)
{
	char bridge_args[254];

	wpa_printf(MSG_ERROR, "%s: for %s remove vid %d",
			__FUNCTION__, drv->common.ifname, vid);
	if (!wired_driver_hapd_is_l2_if(drv)) {
		/* we should not be doing anything for non-L2 interfaces */
		wpa_printf(MSG_ERROR, "WIRED: Cannot remove vids from non-L2"
			" Interface %s", drv->common.ifname);
		return 0;
	}

	if (wired_driver_is_vlan_aware_if(&drv->if_info)) {
		sprintf (bridge_args, "vlan del vid %d dev %s",
				vid, drv->common.ifname);
		if (os_exec(bridge_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not remove vid for %s",
						drv->common.ifname);
			return -1;
		}
		wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
				bridge_command, bridge_args);

		wired_modify_ifupdown_bridge_vid(drv, vid, TRUE);
	} else {
		sprintf (bridge_args, "link del %s.%d", drv->common.ifname, vid);
		if (os_exec(ip_command, bridge_args, 1)) {
			wpa_printf(MSG_ERROR, "WIRED: Error: could not delete interface"
				" %s.%d", drv->common.ifname, vid);
			return -1;
		}

		wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
				ip_command, bridge_args);

		wired_modify_traditional_bridge_tagged_ports(drv, NULL, vid, TRUE);
	}

	return 0;
}

static void
wired_check_vlan_list(void)
{
	struct wpa_driver_wired_data *drv;
	struct hostapd_data *hapd;
	struct sta_info *sta, *tmp_sta;
	struct wired_global *global = global_ptr;
	if_info_t *br_node = NULL;
	int new_vid = 0;
	int tagged;

	if (!global_ptr)
		return;

	dl_list_for_each(drv, &global->interfaces,
				struct wpa_driver_wired_data, if_list) {
		if (!wired_driver_hapd_is_l2_if(drv) ||
			wired_driver_is_vlan_aware_if(&drv->if_info))
			continue;

		hapd = drv->common.ctx;
		if (!hapd)
			continue;

		sta = hapd->sta_list;
		while (sta) {
			new_vid = sta->vlan_id;
			tagged = sta->flags & WLAN_STA_TAGGED;
			br_node = if_hash_table_find_by_vlan(local->br_hash, new_vid);

			if (br_node) {
				if (!(sta->flags & WLAN_STA_UNKNOWN_BR))
					goto next;
				if (tagged)
					wired_add_vid (drv, new_vid, sta);
				else
					wired_update_pvid (drv, 0, new_vid, FALSE, sta);

				if (!(sta->flags & WLAN_STA_UNKNOWN_BR))
					goto next;
				if (tagged) {
					if (wired_handle_ebtable_rules(drv, sta, 1)) {
						wpa_printf(MSG_DEBUG, "WIRED: could not write ebtable rule");
						goto next;
					}
					wired_add_fdb_entry(drv, sta->addr, sta->vlan_id, TRUE);
				} else if (drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED) {
					if (wired_handle_ebtable_rules(drv, sta, 1)) {
						wpa_printf(MSG_DEBUG, "WIRED: could not write ebtable rule");
						goto next;
					}
					wired_add_fdb_entry(drv, sta->addr, -1, FALSE);
				} else if (drv->flags & WPA_WIRED_DATA_DYNAMIC_VLAN_ENABLED) {
					if (wired_handle_ebtable_rules(drv, NULL, 1)) {
						wpa_printf(MSG_DEBUG, "WIRED: could not write ebtable rule");
						goto next;
					}
					wired_add_fdb_entry(drv, sta->addr, new_vid, FALSE);
				}

				sta->flags &= ~WLAN_STA_UNKNOWN_BR;
				if (!tagged)
					drv->flags &= ~WPA_WIRED_DATA_NO_VLAN;
next:
				sta = sta->next;
			} else {
				tmp_sta = sta;
				sta = sta->next;
				if ((!tagged && (drv->flags & WPA_WIRED_DATA_VLAN_MOD)) ||
					(tagged && !(tmp_sta->flags & WLAN_STA_UNKNOWN_BR))) {
					hostapd_drv_sta_deauth(hapd, tmp_sta->addr,
							WLAN_REASON_DISASSOC_STA_HAS_LEFT);
					ap_sta_deauthenticate(hapd, tmp_sta,
							WLAN_REASON_DISASSOC_STA_HAS_LEFT);
					ap_free_sta(hapd, tmp_sta);
					if (!tagged && (drv->flags & WPA_WIRED_DATA_VLAN_MOD))
						drv->flags &= ~WPA_WIRED_DATA_VLAN_MOD;
				}
			}
		}
	}
}

int wired_data_pvlan_enabled(struct hostapd_data *hapd)
{
	struct wpa_driver_wired_data *drv = hapd->drv_priv;

	if (drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED)
		return 1;

	return 0;
}

void wired_trigger_reauth(struct hostapd_data *hapd)
{
	struct wpa_driver_wired_data *drv = hapd->drv_priv;
	struct sta_info *sta;

	for (sta = hapd->sta_list; sta; sta = sta->next) {
		if (sta->eapol_sm)
			sta->eapol_sm->reAuthWhen = 5;
	}

	drv->flags &= ~WPA_WIRED_DATA_PVLAN_ENABLED;
}

void
wired_global_pvlan_change(void *priv, int old_pvlan_id, int new_pvlan_id)
{
	struct wired_global *global = priv;
	struct wpa_driver_wired_data *drv;
	struct hostapd_data *hapd;

	dl_list_for_each(drv, &global->interfaces,
				struct wpa_driver_wired_data, if_list) {
		hapd = drv->common.ctx;
		hapd->conf->pvlan_id = new_pvlan_id;
		if (drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED) {
			wired_trigger_reauth(hapd);
		}
	}

}

static int
wired_set_sta_authorized(void *priv, const u8 *addr,
			 unsigned int total_flags, unsigned int flags_or,
			 unsigned int flags_and)
{
	struct wpa_driver_wired_data *drv = priv;
	struct hostapd_data *hapd = drv->common.ctx;
	struct sta_info *sta;
	int tagged;

	wpa_printf(MSG_DEBUG, "WIRED: set STA entry ifname=%s ifindex=%d %s for " MACSTR,
			drv->common.ifname,
			drv->common.ifindex,
			(flags_or & WPA_STA_AUTHORIZED) ? "AUTHORIZED" : "NOT AUTHORIZED",
			MAC2STR(addr));

	if (flags_or & WPA_STA_AUTHORIZED) {

		sta = ap_get_sta(hapd, addr);
		if (!sta)
			return 0;

		tagged = sta->flags & WLAN_STA_TAGGED;
		if (hapd && hapd->conf->pvlan_enabled &&
				(sta->flags & WLAN_STA_PARKED_VLAN)) {
			wired_del_fdb_entry(drv, addr, hapd->conf->pvlan_id, FALSE);
			if (tagged) {
				if (!(drv->flags & WPA_WIRED_DATA_DYN_VOICE)) {
					/* TODO - check if the flag is set even in the parking VLAN case */
					wired_add_vid(drv, sta->vlan_id, sta);
					drv->flags |= WPA_WIRED_DATA_DYN_VOICE;
				}
				drv->vid_ref++;
			} else if ((drv->flags & WPA_WIRED_DATA_DYN_VOICE) ||
				!hapd->conf->voice_enabled || !hapd->conf->voice_vlan_id) {
				wired_update_pvid(drv, hapd->conf->pvlan_id, drv->if_info.pvid,
						TRUE, sta);
				drv->pvid_ref++;
			}
			sta->flags &= ~WLAN_STA_PARKED_VLAN;
			if (wired_is_last_parked_sta(hapd))
				drv->flags &= ~WPA_WIRED_DATA_PVLAN_ENABLED;

			/* TODO - check if it need to be reset at this point */
			hapd->auth_fail_no_sta_del = 0;
		}

		if (!tagged && !(drv->flags & WPA_WIRED_DATA_DYN_VOICE) &&
				!sta->vlan_id && hapd->conf->voice_enabled &&
				hapd->conf->voice_vlan_id && (sta->flags & WLAN_STA_MAB)) {
			sta->vlan_id = hapd->conf->voice_vlan_id;
			sta->flags |= WLAN_STA_TAGGED;
			drv->vid_ref++;
			if (!(drv->flags & WPA_WIRED_DATA_STATIC_VOICE)) {
				wired_add_vid(drv, sta->vlan_id, sta);
				drv->flags |= WPA_WIRED_DATA_STATIC_VOICE;
			}
		}

		if ((hapd->conf->eap_send_identity) && (sta->flags & WLAN_STA_MAB)) {
			sta->eapol_sm->eap_if->portEnabled = FALSE;
			sta->eapol_sm->reAuthenticate = FALSE;
			wpa_printf(MSG_DEBUG, "%s: sta address=" MACSTR " EAP port disabled",
				__FUNCTION__, MAC2STR(addr));
		}

		if (drv->flags & WPA_WIRED_DATA_NO_VLAN)
			return 0;

		/* loop through existing stations and allow traffic from source mac*/
		if (wired_handle_ebtable_rules(drv, NULL, 1)) {
			wpa_printf(MSG_DEBUG, "WIRED: could not write ebtable rule");
			return -1;
		}
		if (wired_driver_hapd_is_l2_if(drv) && 
			!wired_driver_is_vlan_aware_if(&drv->if_info) &&
			(sta->flags & WLAN_STA_TAGGED)) {
			wired_add_fdb_entry(drv, addr, sta->vlan_id, TRUE);

		} else {
			wired_add_fdb_entry(drv, addr, sta->vlan_id, FALSE);
		}
	} else {
		sta = ap_get_sta(hapd, addr);

		if (sta && sta->eapol_sm && sta->eapol_sm->eap_if &&
				sta->eapol_sm->eap_if->aaaFail ) {
			wpa_printf(MSG_DEBUG, "%s: sta address=" MACSTR " auth fail state %d",
					__FUNCTION__, MAC2STR(addr), sta->eapol_sm->eap_if->aaaFail);

			if (drv->flags & WPA_WIRED_DATA_NO_VLAN)
				return 0;

			if ((hapd->conf->eap_send_identity) &&
				(sta->flags & WLAN_STA_MAB)) {
				sta->eapol_sm->eap_if->portEnabled = FALSE;
				sta->eapol_sm->reAuthenticate = FALSE;
				wpa_printf(MSG_DEBUG, "%s: sta address=" MACSTR " EAP port disabled",
					__FUNCTION__, MAC2STR(addr));
			}
			if ((sta->eapol_sm->authAuthFailWhileAuthenticating > 1) &&
				(drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED))
				return 0;

			if (hapd && hapd->conf->pvlan_enabled && hapd->conf->pvlan_id) {
				wired_update_pvid(drv, drv->if_info.pvid, hapd->conf->pvlan_id,
						FALSE, sta);
				/* write a temporary config file for the new VLAN ifor ifreload */
				drv->flags |= WPA_WIRED_DATA_PVLAN_ENABLED;
				sta->flags |= WLAN_STA_PARKED_VLAN;
				sta->vlan_id = hapd->conf->pvlan_id;
				hapd->auth_fail_no_sta_del = 1;
				if (drv->flags & WPA_WIRED_DATA_NO_VLAN)
					return 0;

				if (wired_handle_ebtable_rules(drv, sta, 1)) {
					wpa_printf(MSG_DEBUG, "WIRED: could not write ebtable rule");
					return -1;
				}
				wired_add_fdb_entry(drv, addr, sta->vlan_id, FALSE);
			}
		} else {
			wpa_printf(MSG_DEBUG, "%s: sta address=" MACSTR " missing sm",
							__FUNCTION__, MAC2STR(addr) );
		}
	}

	return 0;
}

static int wired_sta_remove(void *priv, const u8 *addr)
{
	struct wpa_driver_wired_data *drv = priv;
	struct hostapd_data *hapd = drv->common.ctx;
	struct sta_info *sta;

	wpa_printf(MSG_DEBUG, "WIRED: wired_sta_remove called ifname=%s address=" MACSTR,
			drv->common.ifname, MAC2STR(addr));

	sta = ap_get_sta(hapd, addr);
	if (!sta)
		return 0;
	wired_del_fdb_entry(drv, addr, sta->vlan_id, sta->flags & WLAN_STA_TAGGED);
	if (sta && hapd && (sta->flags & WLAN_STA_MAB))
		sta->flags &= ~ WLAN_STA_MAB;
	if (drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED) {
		if (sta && (sta->flags & WLAN_STA_PARKED_VLAN))
			sta->flags &= ~ WLAN_STA_PARKED_VLAN;
		if (wired_is_last_parked_sta(hapd)) {
			wired_update_pvid(drv, hapd->conf->pvlan_id, drv->if_info.pvid,
						TRUE, sta);
			drv->flags &= ~WPA_WIRED_DATA_PVLAN_ENABLED;
		}
	}
	if (drv->flags & WPA_WIRED_DATA_DYNAMIC_VLAN_ENABLED) {
		if (sta && (sta->flags & WLAN_STA_DYNAMIC_VLAN))
			sta->flags &= ~WLAN_STA_DYNAMIC_VLAN;
	}
	if ((drv->flags & WPA_WIRED_DATA_STATIC_VOICE) &&
		(sta->flags & WLAN_STA_TAGGED) &&
		(sta->vlan_id == hapd->conf->voice_vlan_id)) {
		wired_remove_vid(drv, sta->vlan_id);
		drv->vid_ref--;
		if (!drv->vid_ref)
			drv->flags &= ~WPA_WIRED_DATA_STATIC_VOICE;
	}
	return 0;
}


static int wired_sta_deauth(void *priv, const u8 *own_addr, const u8 *addr,
				int reason)
{
	struct wpa_driver_wired_data *drv = priv;
	struct hostapd_data *hapd = drv->common.ctx;
	struct sta_info *sta;
	char acl_filename[128];
	char acl_filename_tmp[128];
	FILE *f;
	struct stat st = {0};

	sta = ap_get_sta(hapd, addr);

	wpa_printf(MSG_DEBUG, "WIRED: wired_sta_deauth called ifname=%s address=" MACSTR,
			drv->common.ifname, MAC2STR(addr));

	if (sta && (sta->flags & WLAN_STA_PARKED_VLAN))
		sta->flags &= ~ WLAN_STA_PARKED_VLAN;
	if (sta && hapd && (sta->flags & WLAN_STA_MAB))
		sta->flags &= ~ WLAN_STA_MAB;

	if (sta && (sta->flags & WLAN_STA_AUTHORIZED ||
			hapd->auth_fail_no_sta_del)) {
		if (wired_handle_ebtable_rules(drv, sta, 0)) {
			wpa_printf(MSG_DEBUG, "WIRED: could not write ebtable rule");
			return -1;
		}
		wired_del_fdb_entry(drv, addr, sta->vlan_id,
					sta->flags & WLAN_STA_TAGGED);
		if ((drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED) && 
				wired_is_last_parked_sta(hapd)) {
			wired_update_pvid(drv, hapd->conf->pvlan_id, drv->if_info.pvid, TRUE,
					sta);
			drv->flags &= ~WPA_WIRED_DATA_PVLAN_ENABLED;
		}
	}

	//wired_remove_dacl_files(hapd, addr);
	/* Instead of removing the dacl acl files, an empty dacl_file will be created in 
	/etc/cumulus/acl/policy.d/tmp/ dir so that rules in dacl_file by same name in 
	/etc/cumulus/acl/policy.d/ dir get deleted when cl-acltool gets invoked next. 
	Once the rules are deleted, empty dacl_file is deleted as well.*/
	snprintf(acl_filename, sizeof(acl_filename),
				"/etc/cumulus/acl/policy.d/150_dot1x_dacl_%s_"
				COMPACT_MACBYTE ".rules",
				hapd->conf->iface, MACBYTE(addr));
	if (stat(acl_filename, &st) == 0) {
	    snprintf(acl_filename_tmp, sizeof(acl_filename_tmp),
				    "/etc/cumulus/acl/policy.d/tmp/150_dot1x_dacl_%s_"
				    COMPACT_MACBYTE ".rules",
				    hapd->conf->iface, MACBYTE(addr));
	    f = fopen(acl_filename_tmp,"w");
	    if (!f) {
		    wpa_printf(MSG_ERROR, "WIRED: ERROR: could not create"
				    " file %s", acl_filename);
		    return -1;
	    }
	    fclose(f);
	}

	return 0;
}

static void
wired_interface_type_change (struct wpa_driver_wired_data *drv,
				int new_type_bridge, int *flush)
{
	struct hostapd_data *hapd = drv->common.ctx;
	int old_type_bridge = drv->if_info.is_bridge;

	if (old_type_bridge && !new_type_bridge) {
		wpa_printf(MSG_DEBUG, "%s %s changed to L3 interface %d",
					__FUNCTION__, drv->common.ifname,
					hapd->conf->mab_enabled);
		if (drv->flags & WPA_WIRED_DATA_IGNORE_CHANGE)
            return;
		wired_stop_mab(drv);
		if (!*flush)
			wired_flush(drv);
		drv->if_info.is_bridge = new_type_bridge;
	} else if (!old_type_bridge && new_type_bridge) {
		wpa_printf(MSG_DEBUG, "%s %s changed to bridge member",
					__FUNCTION__, drv->common.ifname);
		if (!*flush)
			wired_flush(drv);
		drv->flags &= ~WPA_WIRED_DATA_IGNORE_CHANGE;
		drv->if_info.is_bridge = new_type_bridge;
		wired_init_mab(hapd);
	}
}

void wired_update_fdb_entry(struct wpa_driver_wired_data *drv)
{
	struct hostapd_data *hapd = drv->common.ctx;
	struct sta_info *sta;

	if (drv->flags & WPA_WIRED_DATA_NO_VLAN)
		return;

	for (sta = hapd->sta_list; sta; sta = sta->next) {
		if (sta->flags & WLAN_STA_AUTHORIZED) {
			wired_del_fdb_entry(drv, sta->addr, 0, sta->flags & WLAN_STA_TAGGED);
			wired_add_fdb_entry(drv, sta->addr, sta->vlan_id,
				sta->flags & WLAN_STA_TAGGED);
		}
	}
}

void wired_del_fdb_per_vlan(struct wpa_driver_wired_data *drv, int vlan)
{
	struct hostapd_data *hapd = drv->common.ctx;
	struct sta_info *sta;

	wpa_printf(MSG_DEBUG, "%s delete for vlan %d", __FUNCTION__, vlan);
	for (sta = hapd->sta_list; sta; sta = sta->next) {
		if (sta->flags & WLAN_STA_AUTHORIZED) {
			wired_del_fdb_entry(drv, sta->addr, vlan,
					sta->flags & WLAN_STA_TAGGED);
		}
	}
}

static void wired_del_or_update_vlan(struct wpa_driver_wired_data *drv,
					u32 *old_bitmap, u32 *new_bitmap)
{
	Boolean update = FALSE;
	int word, bit;
	u32 mask;

	for (word = 0; word < RTNL_LINK_BRIDGE_VLAN_BITMAP_LEN; word++) {
		if (!old_bitmap[word] && !new_bitmap[word])
			continue;
		for (bit = 0; bit < 32; bit++) {
			mask = 1 << bit;
			if ((mask & new_bitmap[word]) && !(mask & old_bitmap[word])) {
				wpa_printf(MSG_DEBUG, "%s update needed for vlan %d", __FUNCTION__,
					(word * 32) + bit);
				update = TRUE;
			} else if (!(mask & new_bitmap[word]) && (mask & old_bitmap[word])) {
				wired_del_fdb_per_vlan(drv, ((word * 32) + bit));
			}
		}
	}

	if (update)
		wired_update_fdb_entry(drv);
}

static int wired_driver_is_br_untagged (if_info_t *br_node)
{
	br_if_info_t *br_info = (br_if_info_t *) br_node->data;
	vlan_if_info_t *vlan_info;
	if_info_t *parent_if_info;
	struct wpa_driver_wired_data *drv = NULL;
	char *tag_ifname;
	char *tmp;
	int tot_if_count = 0;
	int dot1x_if_count = 0;

	wpa_printf(MSG_DEBUG, "%s", __FUNCTION__);

	if (dl_list_empty(&br_info->tag_mbr_interfaces)) {
		wpa_printf(MSG_ERROR, "%s tag member list is empty", __FUNCTION__);
		return 1;
	}

	dl_list_for_each(vlan_info, &br_info->tag_mbr_interfaces,
				vlan_if_info_t, mbr_if_list) {
		parent_if_info = vlan_info->parent;

		tot_if_count++;
		tag_ifname = strtok_r(parent_if_info->ifname, ".", &tmp);
		if (!tag_ifname)
			continue;
		wpa_printf(MSG_ERROR, "%s tag member %s/%s", __FUNCTION__,
				parent_if_info->ifname, tag_ifname);
		dl_list_for_each(drv, &global_ptr->interfaces,
					struct wpa_driver_wired_data, if_list) {
			if (os_strcmp(tag_ifname, drv->common.ifname) == 0) {
				dot1x_if_count++;
				break;
			}
		}
	}

	wpa_printf(MSG_ERROR, "%s count %d/%d", __FUNCTION__, tot_if_count,
			dot1x_if_count);
	if (tot_if_count > dot1x_if_count)
		return 0;
	else
		return 1;
}

static void wired_driver_if_create(struct rtnl_link *link)
{
	int ifindex;
	int master;
	char *inp;
	char *type;
	if_info_t *if_node;
	if_info_t *br_node;
	br_if_info_t *br_info;
	vlan_if_info_t *vlan_node;
	int is_vlan;
	u32 flags;
	int was_empty;

	ifindex = rtnl_link_get_ifindex(link);
	inp = rtnl_link_get_name(link);
	flags = rtnl_link_get_flags(link);
	master = rtnl_link_get_master(link);
	type = rtnl_link_get_type(link);

	if (wpa_nl_is_link_bridge(link)) {
		if_node = if_hash_table_find(local->br_hash, ifindex);
		if (!if_node) {
			if_node = if_info_alloc(IF_TYPE_BRIDGE, ifindex,
								inp, flags);
			if_hash_table_add(local->br_hash, if_node);
		}

		if (if_node) {
			br_info = (br_if_info_t *)if_node->data;
			if (rtnl_link_bridge_get_vlan_filtering(link) ||
				!os_strcmp(inp, "bridge"))
				if_node->vlan_aware = 1;
			br_info->master_id = master;
			if (if_node->vlan_aware)
				wpa_printf(MSG_DEBUG, "%s: Vlan Aware Bridge %s/%d type %s"
					" master %d created", __FUNCTION__,
					inp, ifindex, type ? type : "na", master);
			else
				wpa_printf(MSG_DEBUG, "%s: Trad Bridge %s/%d type %s"
					" master %d created", __FUNCTION__,
					inp, ifindex, type ? type : "na", master);
		}
	} else {
		is_vlan = rtnl_link_is_vlan(link);

		if_node = if_hash_table_find(local->if_hash, ifindex);
		if (!if_node) {
			if_node = if_info_alloc(is_vlan?IF_TYPE_VLAN:IF_TYPE_OTHER,
									ifindex, inp, flags);
			if_hash_table_add(local->if_hash, if_node);
		}
		if (if_node) {
			vlan_node = (vlan_if_info_t *)if_node->data;
			if (is_vlan)
				if_node->vlan_id = rtnl_link_vlan_get_id(link);

			if (master) {
				br_node = if_hash_table_find(local->br_hash, master);

				if (!br_node || !br_node->data)
					return;
				br_info = (br_if_info_t *)br_node->data;

				was_empty = dl_list_empty(&br_info->tag_mbr_interfaces);
				if_update_master(if_node, br_node);
				vlan_node->orig_master_id = master;

				if (!br_node->vlan_aware && is_vlan && was_empty &&
						!dl_list_empty(&br_info->tag_mbr_interfaces))
					wired_check_vlan_list();
			}
			wpa_printf(MSG_DEBUG, "%s: Vlan interface %s/%d type %s"
					" master %d vlan %d created", __FUNCTION__,
					inp, ifindex, type ? type : "na", master,
					is_vlan ? rtnl_link_vlan_get_id(link): 0);
		}
	}
}

static void wired_driver_if_delete(struct rtnl_link *link)
{
	int ifindex;
	int master;
	char *inp;
	char *type;
	if_info_t *if_node;
	if_info_t *br_node;
	br_if_info_t *br_info;
	vlan_if_info_t *vlan_info;
	vlan_if_info_t *tmp;

	ifindex = rtnl_link_get_ifindex(link);
	inp = rtnl_link_get_name(link);
	master = rtnl_link_get_master(link);
	type = rtnl_link_get_type(link);

	if (wpa_nl_is_link_bridge(link)) {
		br_node = if_hash_table_find(local->br_hash, ifindex);
		if (!br_node)
			return;

		if (br_node->vlan_aware)
			wpa_printf(MSG_DEBUG, "%s: DELETE Vlan Aware Bridge %s/%d"
				" type %s master %d", __FUNCTION__, inp, ifindex,
				type ? type : "na", master);
		else
			wpa_printf(MSG_DEBUG, "%s: DELETE trad Bridge %s/%d type %s"
				" master %d", __FUNCTION__, inp,
				ifindex, type ? type : "na", master);

		br_info = (br_if_info_t *)br_node->data;

		dl_list_for_each_safe(vlan_info, tmp, &br_info->tag_mbr_interfaces,
				vlan_if_info_t, mbr_if_list) {
			vlan_info->master = NULL;
			dl_list_del(&(vlan_info->mbr_if_list));
		}

		dl_list_for_each_safe(vlan_info, tmp, &br_info->untag_mbr_interfaces,
				vlan_if_info_t, mbr_if_list) {
			vlan_info->master = NULL;
			dl_list_del(&(vlan_info->mbr_if_list));
		}

		if_hash_table_del(local->br_hash, ifindex);
		os_free(br_node);
	} else {
		if_node = if_hash_table_find(local->if_hash, ifindex);
		if (!if_node)
			return;

		wpa_printf(MSG_DEBUG, "%s: DELETE Vlan interface %s/%d type %s"
				" master %d", __FUNCTION__, inp, ifindex,
				type ? type : "na", master);

		vlan_info = (vlan_if_info_t *)if_node->data;
        if (vlan_info->mbr_if_list.next && vlan_info->mbr_if_list.prev)
		    dl_list_del(&vlan_info->mbr_if_list);

		if_hash_table_del(local->if_hash, ifindex);

		if(if_node->type == IF_TYPE_VLAN) {
			br_node = vlan_info->master;

			if (br_node && !br_node->vlan_aware &&
				wired_driver_is_br_untagged(br_node)) {
				br_node->vlan_id = 0;
				wired_check_vlan_list();
			}
		}

		os_free(if_node);
	}
}

static void wired_interface_master_change(struct rtnl_link *link, 
				struct wpa_driver_wired_data *drv, int *flush)
{
	int ifindex;
	int master;
	char *inp;
	if_info_t *if_node;
	if_info_t *br_if_node;
	vlan_if_info_t *vlan_info;
	char ip_args[254];

	ifindex = rtnl_link_get_ifindex(link);
	inp = rtnl_link_get_name(link);
	master = rtnl_link_get_master(link);

	if (wpa_nl_is_link_bridge(link))
		return;

	if_node = if_hash_table_find(local->if_hash, ifindex);
	if (!if_node)
		return;

	vlan_info = (vlan_if_info_t *)if_node->data;
	if (vlan_info->master_id != master)	{
		if (drv && (drv->flags & WPA_WIRED_DATA_IGNORE_CHANGE)) {
			/* Check if port was moved to its original bridge even though
			 * hostapd had moved the port to another bridge */
			if ((vlan_info->orig_master_id != vlan_info->master_id) &&
				(vlan_info->orig_master_id == master)) {
				br_if_node = vlan_info->master;
				if (!br_if_node)
					return;

				wpa_printf(MSG_ERROR, "%s being put back into correct bridge,"
					" since %d -> %d attempt had failed", inp,
					master, vlan_info->master_id);

				sprintf (ip_args, "link set dev %s master %s",
						drv->common.ifname, br_if_node->ifname);
				if (os_exec(ip_command, ip_args, 1)) {
					wpa_printf(MSG_ERROR, "%s: could not add %s to %s",
							__FUNCTION__, drv->common.ifname, br_if_node->ifname);
					return;
				}
				wpa_printf(MSG_DEBUG, "%s: command: %s %s", __FUNCTION__,
						ip_command, ip_args);

			} else {
				wpa_printf(MSG_ERROR, "Master change %d -> %d ignored for %s",
					vlan_info->master_id, master, inp);
			}
			return;
		}

		wpa_printf(MSG_DEBUG, "Master changed %d -> %d for %s",
			vlan_info->master_id, master, inp);

		if (drv) {
			wired_stop_mab(drv);
			*flush = 1;
			wired_flush(drv);
			if (wired_driver_hapd_is_l2_if(drv))
				wired_init_mab((struct hostapd_data *)drv->common.ctx);
		}

		if (master == 0) {
			if_update_master(if_node, NULL);
			wired_check_vlan_list();
		} else {
			br_if_node = if_hash_table_find(local->br_hash, master);
			if (br_if_node && br_if_node->data) {
				if_update_master(if_node, br_if_node);
				wired_check_vlan_list();
			}
		}
	}
}

static void wired_link_change_cb(struct nl_cache *cache, struct nl_object *obj,
				int action, void *data)
{
	struct rtnl_link *link = (struct rtnl_link *) obj;
	struct wired_global *global = (struct wired_global *)data;
	int ifindex;
	char *inp, *type;
	int family;
	unsigned int flags;
	struct wpa_driver_wired_data *drv = NULL;
	struct wpa_driver_wired_data *tmp_drv;
	struct hostapd_data *hapd = NULL;
	int master;

	if (!link)
		return;

	ifindex = rtnl_link_get_ifindex(link);
	family = rtnl_link_get_family(link);
	inp = rtnl_link_get_name(link);
	flags = rtnl_link_get_flags(link);
	master = rtnl_link_get_master(link);

	dl_list_for_each(tmp_drv, &global->interfaces,
				struct wpa_driver_wired_data, if_list) {
		if (tmp_drv->common.ifindex == ifindex) {
			drv = tmp_drv;
			break;
		}
	}

	if (drv)
	    hapd = drv->common.ctx;

	if (action == NL_ACT_NEW) {
        if (family == AF_UNSPEC) {
            type = rtnl_link_get_type(link);

			wpa_printf(MSG_DEBUG, "%s: %s/%d type %s family %d master %d"
					" created", __FUNCTION__, inp, ifindex,
					type ? type : "na", family, master);

			wired_driver_if_create(link);
			if (wpa_debug_level <= MSG_DEBUG)
				if_hash_table_foreach (local->br_hash, br_print_cb, NULL);
        }
		if (drv)
			drv->if_info.flags = flags;
	} else if (action == NL_ACT_DEL) {
        if (family == AF_UNSPEC) {
			type = rtnl_link_get_type(link);

			wpa_printf(MSG_DEBUG, "%s: %s/%d type %s family %d master %d"
					" deleted", __FUNCTION__, inp, ifindex,
					type ? type : "na", family, master);

			wired_driver_if_delete(link);
			if (wpa_debug_level <= MSG_DEBUG)
				if_hash_table_foreach (local->br_hash, br_print_cb, NULL);
		}
	} else if (action == NL_ACT_CHANGE) {
		struct nl_object *old_obj;
		struct rtnl_link *old_link;
		unsigned int old_flags = 0;

		if (drv) {
			wpa_printf(MSG_ERROR, "family %d, ifname %s, flags old/new %d/%d,"
				" admin %s/%s oper %s/%s master %d",
				family, inp, drv->if_info.flags, flags,
				(drv->if_info.flags & IFF_UP) ? "up" : "down",
				(flags & IFF_UP) ? "up" : "down",
				(drv->if_info.flags & IFF_RUNNING) ? "up" : "down",
				(flags & IFF_RUNNING) ? "up" : "down", master);
		} else {
			wpa_printf(MSG_DEBUG, "family %d, ifname %s, flags %d,"
				" admin %s oper %s master %d",
				family, inp, flags,
				(flags & IFF_UP) ? "up" : "down",
				(flags & IFF_RUNNING) ? "up" : "down", master);
		}

		if (drv) {
			old_flags = drv->if_info.flags;
			drv->if_info.flags = flags;

			if ((flags & IFF_RUNNING) && !(old_flags & IFF_RUNNING)) {
				wpa_printf(MSG_ERROR, "family %d, ifname %s starting mab",
					family, inp);
				wired_init_mab(hapd);
			} else if (!(flags & IFF_RUNNING) && (old_flags & IFF_RUNNING)) {
				wpa_printf(MSG_ERROR, "family %d, ifname %s stopping mab",
					family, inp);
				wired_mab_deactivation(hapd);
				wired_flush((void *)drv);
			}
		}
		if ((family == AF_BRIDGE)) {
			struct rtnl_link_bridge_vlan *bv;
			struct rtnl_link_bridge_vlan *old_bv;
			int is_bridge;
			int flush = 0;

			if (!drv)
				return;

			is_bridge = rtnl_link_is_bridge(link);
			wired_interface_master_change(link, drv, &flush);
			wired_interface_type_change(drv, is_bridge, &flush);

			bv = rtnl_link_bridge_get_port_vlan(link);
			if (bv) {
				old_obj = nl_cache_get_last_change_orig_obj(cache);
				if (old_obj) {
					old_link = (struct rtnl_link *)old_obj;
					old_bv = rtnl_link_bridge_get_port_vlan(old_link);

					wpa_printf(MSG_ERROR, "Bridge vlan: pvid %d:%d, %x %x\n", bv->pvid, drv->if_info.pvid,
						bv->vlan_bitmap[0], bv->untagged_bitmap[0]);
					if (drv->flags & WPA_WIRED_DATA_PVLAN_ENABLED) {
						wpa_printf(MSG_DEBUG, "Parking VLAN Enabled on interface");
						if (bv->pvid && hapd && (hapd->conf->pvlan_id != bv->pvid)) {
							drv->if_info.pvid = bv->pvid;
							wired_change_pvid(drv, bv->pvid, hapd->conf->pvlan_id, false);
						}
					} else if (drv->flags & WPA_WIRED_DATA_DYNAMIC_VLAN_ENABLED) {
						wpa_printf(MSG_DEBUG, "Dynamic VLAN Enabled on interface");
						if (bv->pvid && hapd && drv->if_info.dynamic_pvid &&
							(bv->pvid != drv->if_info.dynamic_pvid)) {
							wpa_printf(MSG_DEBUG, "Changing from vlan %d"
								" to dynamic VLAN %d", bv->pvid,
								drv->if_info.dynamic_pvid);
							drv->if_info.pvid = bv->pvid;
							wired_change_pvid(drv, bv->pvid,
								drv->if_info.dynamic_pvid, false);
						}
					} else {
						if ((bv->pvid != drv->if_info.pvid) ||
							(old_bv &&
							 os_memcmp(bv->vlan_bitmap, old_bv->vlan_bitmap,
							  RTNL_LINK_BRIDGE_VLAN_BITMAP_MAX / 8))) {
							drv->if_info.pvid = bv->pvid;
							if (old_bv)
								wired_del_or_update_vlan(drv,
									old_bv->vlan_bitmap, bv->vlan_bitmap);
						}
					}

				} else {
					drv->if_info.pvid = bv->pvid;
				}

			}
		} else if ((family == AF_UNSPEC)) {
			struct rtnl_link *tmp_link;
			int flush = 0;

			if (!wpa_nl_is_link_bridge(link)) {
				wired_interface_master_change(link, drv, &flush);

				if (drv && drv->if_info.is_bridge) {
					tmp_link = wpa_nl_get_link_by_family(global->cache,
							AF_BRIDGE, drv->common.ifindex);

					if (!tmp_link ||
						(tmp_link && !rtnl_link_is_bridge(tmp_link))) {
						wired_interface_type_change(drv, 0, &flush);
					}

					if (tmp_link)
						rtnl_link_put(tmp_link);
				}
			}
		}
	}
}

static void handle_nl(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct wired_global *global = eloop_ctx;
	int err;

	if ((err = nl_cache_mngr_data_ready(global->mngr)) < 0) {
		if (err == -NLE_NOMEM || err == -NLE_MSG_OVERFLOW) {
			wpa_printf(MSG_ERROR, "netlink socket buffer overflow: %s",
					nl_geterror(err));
		} else {
			wpa_printf(MSG_ERROR, "nl_cache_mngr_data_ready failed: %s\n",
					nl_geterror(err));
		}
	}
}

int br_print_cb (void *data, void *cbarg)
{
	if_info_t *if_info = (if_info_t *)data;
	br_if_info_t *br_info = (br_if_info_t *) if_info->data;
	vlan_if_info_t *vlan_info;
	if_info_t *parent_if_info;

	if (br_info)
		wpa_printf(MSG_DEBUG, "%s Bridge %s/%d type %d"
			" vlan/master %d/%d flags 0x%x",
			if_info->vlan_aware ? "Vlan Aware" : "Trad Bridge",
			if_info->ifname, if_info->ifindex, if_info->type,
			if_info->vlan_id, br_info->master_id, if_info->flags);
	else
		wpa_printf(MSG_DEBUG, "Trad Bridge %s/%d type %d"
			" flags 0x%x",
			if_info->ifname, if_info->ifindex, if_info->type,
			if_info->flags);

	dl_list_for_each(vlan_info, &br_info->tag_mbr_interfaces,
				vlan_if_info_t, mbr_if_list) {

		parent_if_info = vlan_info->parent;

		wpa_printf(MSG_DEBUG, "Tagged Interface %s/%d type %d"
			" vlan/master %d/%d flags 0x%x",
			parent_if_info->ifname, parent_if_info->ifindex,
			parent_if_info->type, parent_if_info->vlan_id,
			vlan_info->master_id, parent_if_info->flags);
	}

	dl_list_for_each(vlan_info, &br_info->untag_mbr_interfaces,
				vlan_if_info_t, mbr_if_list) {

		parent_if_info = vlan_info->parent;

		wpa_printf(MSG_DEBUG, "Untagged Interface %s/%d type %d"
			" vlan/master %d/%d flags 0x%x",
			parent_if_info->ifname, parent_if_info->ifindex,
			parent_if_info->type, parent_if_info->vlan_id,
			vlan_info->master_id, parent_if_info->flags);
	}

	return 0;
}

static int wired_driver_global_if_init(struct wired_global *global)
{
	struct nl_object *obj;
	struct rtnl_link *link;
	int family;

	local->br_hash = if_hash_table_alloc();
	if (!local->br_hash)
		return -1;

	local->if_hash = if_hash_table_alloc();
	if (!local->if_hash) {
		os_free(local->br_hash);
		return -1;
	}

	obj = nl_cache_get_first(global->cache);
	while(obj) {
		link = (struct rtnl_link *) obj;
		family = rtnl_link_get_family(link);

		if (family == AF_UNSPEC) {
			if (wpa_nl_is_link_bridge(link))
				wired_driver_if_create(link);
		}

		obj = nl_cache_get_next(obj);
	}

	obj = nl_cache_get_first(global->cache);
	while(obj) {
		link = (struct rtnl_link *) obj;
		family = rtnl_link_get_family(link);

		if (family == AF_UNSPEC) {
			if (!wpa_nl_is_link_bridge(link))
				wired_driver_if_create(link);
		}

		obj = nl_cache_get_next(obj);
	}

	if (wpa_debug_level <= MSG_DEBUG)
		if_hash_table_foreach (local->br_hash, br_print_cb, NULL);
	return 0;
}

static int wired_driver_global_nl_init(struct wired_global *global)
{
	int err;
	int buf_size = (9 * 1024 * 1024);

	if ((global->mngr_sock = nl_socket_alloc()) == NULL) {
		wpa_printf(MSG_ERROR, "%s: Unable to allocate cache manager"
			" netlink socket\n", __FUNCTION__);
		return -1;
	}

	/* set msg buffer size: used by nl_recv during receive on this socket */
	nl_socket_set_msg_buf_size(global->mngr_sock, (32 * 1024));

	err = nl_cache_mngr_alloc(global->mngr_sock, NETLINK_ROUTE,
				NL_AUTO_PROVIDE, &global->mngr);
	if (err < 0) {
		wpa_printf(MSG_ERROR, "wired: Unable to allocate cache manager: %s",
				nl_geterror(err));
		return -1;
	}

	if ((err = setsockopt(nl_socket_get_fd(global->mngr_sock),
			SOL_SOCKET, SO_RCVBUFFORCE, &buf_size,
			sizeof (buf_size))) < 0) {
		wpa_printf(MSG_ERROR, "%s: Unable to increase netlink rbuf"
			" size: %s\n", __FUNCTION__, strerror(errno));
		return -1;
	}

	if ((err = setsockopt(nl_socket_get_fd(global->mngr_sock), SOL_SOCKET,
			SO_SNDBUFFORCE, &buf_size, sizeof (buf_size))) < 0) {
		wpa_printf(MSG_ERROR, "%s: Unable to increase netlink wbuf"
			" size: %s\n", __FUNCTION__, strerror(errno));
		return -1;
	}

	if ((err = nl_cache_alloc_name("route/link", &global->cache)) < 0) {
		wpa_printf(MSG_ERROR, "wired: Couldn't add link cache: %s\n",
				nl_geterror(err));
		return -1;
	}

	/* set cache manager sync sock to 32k */
	global->sync_sock = nl_cache_mngr_get_sync_sock(global->mngr);
	nl_socket_set_msg_buf_size(global->sync_sock, (32 * 1024));

	/* set sync socket buffer size to 2MB */
	if ((err = nl_socket_set_buffer_size(global->sync_sock,
			(2 * 1024 *1024), (2 * 1024 * 1024))) < 0) {
		wpa_printf(MSG_ERROR, "%s: Unable to set socket buffer size: %s\n",
			__FUNCTION__, nl_geterror(err));
		return -1;
	}

	/*
	 * Make link cache iter over all supported families. We need this for
	 * AF_BRIDGE
	 */
	nl_cache_set_flags(global->cache, NL_CACHE_AF_ITER);
	if ((err = nl_cache_mngr_add_cache(global->mngr, global->cache,
					&wired_link_change_cb, global)) < 0) {
		wpa_printf(MSG_ERROR, "Unable to add cache route/link: %s",
			     nl_geterror(err));
		return -1;
	}

	if (eloop_register_read_sock(nl_cache_mngr_get_fd(global->mngr),
				handle_nl, global, NULL)) {
		wpa_printf(MSG_INFO, "Could not register nl read socket");
		return -1;
	}

	if (wired_driver_global_if_init(global) < 0)
		return -1;

	return 0;
}

static void wired_driver_global_deinit(void *priv)
{
	struct wired_global *global = priv;

	if (global == NULL)
		return;
	if (!dl_list_empty(&global->interfaces)) {
		wpa_printf(MSG_ERROR, "wired: %u interface(s) remain at "
				"wired_driver_global_deinit",
				dl_list_len(&global->interfaces));
	}

	if (global->netlink)
		netlink_deinit(global->netlink);

	if (global->mngr) {
		eloop_unregister_read_sock(nl_cache_mngr_get_fd(global->mngr));
		nl_cache_mngr_free(global->mngr);
	}

	if (global->mngr_sock)
		nl_socket_free(global->mngr_sock);

	global_ptr->in_deinit = TRUE;
	pthread_cond_broadcast(&global->acl_cv);
	pthread_cond_destroy(&global->acl_cv);
	pthread_mutex_destroy(&global->acl_lock);

	os_free(global);

	if (local) {
		if_hash_table_free (local->br_hash);
		if_hash_table_free (local->if_hash);
		os_free(local);
	}

	rmdir(acl_tmp_dir);

}

static void *wired_driver_global_init(void *ctx)
{
	struct wired_global *global;
	struct netlink_config *cfg;
	struct stat st = {0};
	FILE *f = NULL;

	wpa_printf(MSG_ERROR, "wired: wired_driver_global_init");
	global = os_zalloc(sizeof(struct wired_global));
	if (global == NULL)
		return NULL;
	global->global_interfaces = (struct hapd_interfaces *)ctx;
	dl_list_init(&global->interfaces);

	cfg = os_zalloc(sizeof(*cfg));
	if (cfg == NULL)
		goto err;

	cfg->ctx = global;
	global->netlink = netlink_init(cfg);
	if (global->netlink == NULL) {
		os_free(cfg);
		goto err;
	}

	local = os_zalloc(sizeof(struct wired_local));
	if (local == NULL)
		goto err;

	dl_list_init(&local->no_vlan_interfaces);

	if (wired_driver_global_nl_init(global) < 0)
		goto err;

	if (stat(acl_tmp_dir, &st) == -1) {
		mkdir(acl_tmp_dir, 0700);
	}

	if (stat(preauth_dacl_dir, &st) == -1) {
		mkdir(preauth_dacl_dir, 0770);
	}

	if (stat(preauth_dacl_file, &st) == -1) {
            f = fopen(preauth_dacl_file, "w");
            if (!f) {
                    wpa_printf(MSG_ERROR, "WIRED: ERROR: could not write preauth dacl rule"
                                    " file %s", preauth_dacl_file);
                    return NULL;
            }
            fprintf(f, "-A FORWARD -p IPV4 --ip-protocol UDP --ip-dport 53 -j ACCEPT\n");
            fprintf(f, "-A INPUT -p IPV4 --ip-protocol UDP --ip-dport 67 --ip-sport 68 -j ACCEPT\n");
            fclose(f);
        }

	global_ptr = global;
	pthread_cond_init(&global->acl_cv, NULL);
	pthread_mutex_init(&global->acl_lock, NULL);
	pthread_create(&(global->tacl), NULL, wired_install_acl, NULL);
	eloop_cancel_timeout(wired_acl_signal, global, NULL);
	eloop_register_timeout(ACL_TIMER, 0, wired_acl_signal, global, NULL);
	wpa_printf(MSG_DEBUG, "%s: registered timeout for acl signaling",
			__FUNCTION__);
	return global;

err:
	wired_driver_global_deinit(global);
	return NULL;
}

static int wired_if_add(void *priv, enum wpa_driver_if_type type,
		const char *ifname, const u8 *addr,
		void *bss_ctx, void **drv_priv,
		char *force_ifname, u8 *if_addr,
		const char *bridge, int use_existing,
		int setup_ap)
{
	struct wpa_driver_wired_data *drv = priv;
	struct hostapd_data *hapd = drv->common.ctx;
	struct hostapd_vlan *vlan;
	struct sta_info *sta;

	wpa_printf(MSG_ERROR, "wired_if_add: Received VLAN add %s"
			" for interface %s",
			ifname, drv->common.ifname);

	for (vlan = hapd->conf->vlan; vlan; vlan = vlan->next) {
		if (!os_strncmp(ifname, hapd->conf->vlan->ifname, strlen(ifname)))
			break;
	}

	if (!vlan) {
		wpa_printf(MSG_ERROR, "wired_if_add: Vlan info for interface %s not"
			" found", drv->common.ifname);
		return 0;
	}

	wpa_printf(MSG_ERROR, "wired_if_add: Received VLAN %d:tag %d add"
		" for interface %s", vlan->vlan_id, vlan->vlan_desc.tagged[0], ifname);

	if (!wired_driver_hapd_is_l2_if(drv)) {
		/* we should not do anything for non-L2 interfaces but allow them to authenticate */
		wpa_printf(MSG_ERROR, "wired_if_add: can not handle VLAN/FDB %d"
				" for non-L2 interface %s",
				vlan->vlan_id, drv->common.ifname);
		return 0;
	}

	for (sta = hapd->sta_list; sta; sta = sta->next) {
		if (sta->flags & WLAN_STA_ADDING_VLAN) {
			sta->flags &= ~WLAN_STA_ADDING_VLAN;
			break;
		}
	}
	if (!sta) {
		wpa_printf(MSG_ERROR, "wired_if_add: No station found for %s: " MACSTR,
				drv->common.ifname, MAC2STR(addr));
		return -1;
	}

	drv->flags |= WPA_WIRED_DATA_DYNAMIC_VLAN_ENABLED;

	if (vlan->vlan_desc.tagged[0]) {
		vlan->vlan_id = vlan->vlan_desc.tagged[0];

		if (drv->flags & WPA_WIRED_DATA_STATIC_VOICE) {
			wpa_printf(MSG_ERROR, "wired_if_add: %s static vid is in use, can't"
				" set Dynamic voice VLAN %d",
					drv->common.ifname, vlan->vlan_id);
			return -1;
		} else if ((drv->flags & WPA_WIRED_DATA_DYN_VOICE) &&
			(vlan->vlan_id != drv->if_info.dynamic_voice_vid)) {
			wpa_printf(MSG_ERROR, "wired_if_add: %s different dynamic voice vid"
				" is in use, can't set tagged Dynamic VLAN %d",
					drv->common.ifname, vlan->vlan_id);
			return -1;
		}

		if (wired_add_vid(drv, vlan->vlan_id, sta)) {
			wpa_printf(MSG_ERROR, "wired_if_add: error setting %s"
					" tagged Dynamic VLAN %d",
					drv->common.ifname, vlan->vlan_id);
			return -1;
		}
		drv->flags |= WPA_WIRED_DATA_DYN_VOICE;
		drv->if_info.dynamic_voice_vid = vlan->vlan_id;
		drv->vid_ref++;
	} else {
		if (drv->if_info.dynamic_pvid) {
			if (drv->if_info.dynamic_pvid != vlan->vlan_id) {
				wpa_printf(MSG_ERROR, "wired_if_add: %s dynamic pvid %d already"
					" set, can't set another pvid %d",
					drv->common.ifname, drv->if_info.dynamic_pvid,
					vlan->vlan_id);
				return -1;
			} else {
				wpa_printf(MSG_ERROR, "wired_if_add: %s dynamic pvid %d already"
					" set, nothing to do",
					drv->common.ifname, drv->if_info.dynamic_pvid);
				return 0;
			}
		}

		if (wired_update_pvid(drv, drv->if_info.pvid, vlan->vlan_id,
							FALSE, sta)) {
			wpa_printf(MSG_ERROR, "wired_if_add: error setting %s"
					" Dynamic VLAN %d",
					drv->common.ifname, vlan->vlan_id);
			return -1;
		}
		drv->if_info.dynamic_pvid = vlan->vlan_id;
		drv->pvid_ref++;
	}

	wpa_printf(MSG_DEBUG, "wired_if_add: setting %s Dynamic VLAN %d",
			drv->common.ifname, vlan->vlan_id);
	return 0;
}

static int wired_is_last_dynamic_vlan(struct hostapd_data *hapd,
			int vlan_id)
{
	struct hostapd_vlan *vlan;

	for (vlan = hapd->conf->vlan; vlan; vlan = vlan->next) {
		if (vlan->vlan_id != vlan_id && vlan->dynamic_vlan)
			return 0;;
	}

	return 1;
}

static int wired_if_remove(void *priv, enum wpa_driver_if_type type,
				    const char *ifname)
{
	struct wpa_driver_wired_data *drv = priv;
	struct hostapd_data *hapd = drv->common.ctx;
	struct hostapd_vlan *vlan;

	wpa_printf(MSG_ERROR, "wired_if_remove: Received VLAN remove %s"
			" for interface %s",
			ifname, drv->common.ifname);

	for (vlan = hapd->conf->vlan; vlan; vlan = vlan->next) {
		if (!os_strncmp(ifname, hapd->conf->vlan->ifname, strlen(ifname)))
			break;
	}

	if (!vlan) {
		wpa_printf(MSG_ERROR, "wired_if_remove: Vlan info for interface %s not"
			" found", drv->common.ifname);
		return 0;
	}

	wpa_printf(MSG_ERROR, "wired_if_remove: Received VLAN %d:tag %d remove"
		" for interface %s", vlan->vlan_id, vlan->vlan_desc.tagged[0], ifname);

	if (!wired_driver_hapd_is_l2_if(drv)) {
		/* we should not do anything for non-L2 interfaces but allow them to authenticate */
		wpa_printf(MSG_ERROR, "wired_if_remove: can not handle VLAN/FDB %d"
				" for non-L2 interface %s",
				vlan->vlan_id, drv->common.ifname);
		return 0;
	}


	if (vlan->vlan_desc.tagged[0]) {
		vlan->vlan_id = vlan->vlan_desc.tagged[0];
		if (wired_remove_vid(drv, vlan->vlan_id)) {
			wpa_printf(MSG_ERROR, "wired_if_add: error setting %s"
					" tagged Dynamic VLAN %d",
					drv->common.ifname, vlan->vlan_id);
			return -1;
		}
		drv->vid_ref--;
		if (!drv->vid_ref)
			drv->flags &= ~WPA_WIRED_DATA_DYN_VOICE;
	} else {
		if (wired_update_pvid(drv, vlan->vlan_id, drv->if_info.pvid,
							TRUE, NULL)) {
			wpa_printf(MSG_ERROR, "wired_if_remove: error removing %s"
					" Dynamic VLAN %d",
					drv->common.ifname, vlan->vlan_id);
			return -1;
		}
		drv->pvid_ref--;
		if (!drv->pvid_ref)
			drv->if_info.dynamic_pvid = 0;
	}

	if (wired_is_last_dynamic_vlan(hapd, vlan->vlan_id))
		drv->flags &= ~WPA_WIRED_DATA_DYNAMIC_VLAN_ENABLED;

	wpa_printf(MSG_DEBUG, "wired_if_remove: removing %s Dynamic VLAN %d",
			drv->common.ifname, vlan->vlan_id);
	return 0;
}

static unsigned int driver_wired_get_ifindex(void *priv)
{
	struct wpa_driver_wired_data *drv = priv;
	return drv->common.ifindex;
}


const struct wpa_driver_ops wpa_driver_wired_ops = {
	.name = "wired",
	.desc = "Wired Ethernet driver",
	.global_init = wired_driver_global_init,
	.global_deinit = wired_driver_global_deinit,
	.hapd_init = wired_driver_hapd_init,
	.hapd_deinit = wired_driver_hapd_deinit,
	.hapd_is_l2_if = wired_driver_hapd_is_l2_if,
	.ifdown = wired_driver_ifdown,
	.ifup = wired_driver_ifup,
	.hapd_send_eapol = wired_send_eapol,
	.sta_set_flags = wired_set_sta_authorized,
	.sta_deauth = wired_sta_deauth,
	.sta_remove = wired_sta_remove,
	.flush = wired_flush,
	.remove_dacl_files = wired_remove_dacl_files,
	.if_add = wired_if_add,
	.if_remove = wired_if_remove,
	.get_ssid = driver_wired_get_ssid,
	.get_bssid = driver_wired_get_bssid,
	.get_capa = driver_wired_get_capa,
	.get_ifindex = driver_wired_get_ifindex,
	.init = wpa_driver_wired_init,
	.deinit = wpa_driver_wired_deinit,
};
