#include <stdarg.h>
#include <config.h>
#include <errno.h>
#include <inttypes.h>
#include <stdlib.h>
#include <unistd.h>
#include "coverage.h"
#include "daemon.h"
#include "include/openvswitch/dynamic-string.h"
#include "hash.h"
#include "include/openvswitch/hmap.h"
#include "hmapx.h"
#include "jsonrpc.h"
#include "include/openvswitch/list.h"
#include "include/openvswitch/shash.h"
#include "socket-util.h"
#include "sset.h"
#include "timeval.h"
#include "util.h"
#include "unixctl.h"
#include "vtep/vtep-idl.h"
#include "vtep/vtep.h"
#include "include/openvswitch/vlog.h"
#include "include/openvswitch/poll-loop.h"
#include "ptm_lib.h"
#include <dirent.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>
#include <ifaddrs.h>
#include <sys/un.h>
#include <poll.h>
#include <bitmap.h>
#include <vlan-bitmap.h>
#include "clag.h"

/* TBD: remember to replace local phy loc when local ip changes */

VLOG_DEFINE_THIS_MODULE(vtep);

#define HWVTEP_FLAGS_ADD       1
#define HWVTEP_FLAGS_DELETE    2
#define HWVTEP_FLAGS_TOUCHED   4

#define HWVTEP_MAX_STR_LEN     256
#define HWVTEP_MAX_SVC_NODE    16

#define PTM_SOCK_NAME "\0/var/run/ptmd.socket"

static struct ovsdb_idl *vtep_idl;
static unsigned int idl_seqno;
static const char vxlan_encap_type[] = "vxlan_over_ipv4";
static const char unknown_dst_str[] = "unknown-dst";
static const char nvp_internal_str[] = "_nvp_internal";

static const char bfd_dst_mac[] = "bfd_dst_mac";
static const char bfd_dst_ip[] = "bfd_dst_ip";
static const char bfd_enable[] = "enable";
static const char bfd_min_rx[] = "min_rx";
static const char bfd_min_tx[] = "min_tx";
static const char bfd_decay_min_rx[] = "decay_min_rx";
static const char bfd_fwd_if_rx[] = "forwarding_if_rx";
static char bfd_cpath_down[] = "cpath_down";
static const char bfd_check_tnl_key[] = "check_tnl_key";
static const char bfd_enabled[] = "enabled";
static const char bfd_state[] = "state";
static const char bfd_forwarding[] = "forwarding";
static const char bfd_diag[] = "diagnostic";
static const char bfd_remote_state[] = "remote_state";
static const char bfd_remote_diag[] = "remote_diagnostic";
static const char bfd_info[] = "info";

static char default_bfd_mac[] = "00:23:20:00:00:01";
static char default_bfd_local_dst_ip[] = "169.254.1.0";
static char default_bfd_remote_dst_ip[] = "169.254.1.1";
static char default_bfd_enable[] = "false";
static char default_min_rx[] = "1000";
static char default_min_tx[] = "100";
static char default_decay_min_rx[] = "0";
static char default_forwarding_if_rx[] = "true";
static char default_cpath_down[] = "false";
static char default_check_tnl_key[] = "false";
static char default_enabled[] = "false";
static const char default_state[] = "none";
static const char default_forwarding[] = "false";
static const char default_diagnostic[] = "";
static const char default_remote_state[] = "none";
static const char default_remote_diagnostic[] = "";
static const char default_info[] = "";

static char ptm_peer[] = "peer";
static char ptm_local[] = "local";
static char ptm_state[] = "state";
static char ptm_diag[] = "diag";
static char ptm_cmd[] = "cmd";
static char ptm_status[] = "cmd_status";

static int ptm_sock = -1;
static bool ptm_poll = true;
static ptm_lib_handle_t *ptm_hdl;
static bool service_node_update = false;
static char ptm_msg_buf[PTMLIB_MSG_SZ];

extern bool vtep_mac_learn_en;
extern bool vtep_vlan_aware;
extern bool user_bridge_name_provided;
extern char *user_bridge_name;

static FILE *batch_fp;

struct phy_switch {
    char *name;
    char *description;
    struct shash local_ips;
    struct shash mgmt_ips;
    struct shash ports;
    uint32_t flags;
    const struct vteprec_physical_switch *ps_cfg;
};

struct phy_port {
    char *name;
    char *description;
    uint32_t flags;
    bool is_bond;
    bool is_clag;
    struct shash vlan_bindings;
};

struct binding_stats {
    int64_t bytes_from_local;
    int64_t packets_from_local;
    int64_t bytes_to_local;
    int64_t packets_to_local;
};

struct vlan_binding {
    char *name; /* port-x-vlan-y */
    uint32_t flags;
    int64_t vlan;
    char *ls_name;
    struct binding_stats stats;
};

struct log_switch {
    char *name;
    char *dev_name; /* vxlanX */
    char *description;
    char *local_ip;
    int64_t vlan;
    int64_t tunnel_key;
    struct shash log_ports;
    uint32_t flags;
    const struct vteprec_logical_switch *ls_cfg;
};

struct phy_locator {
    char *key;  /* type + dst_ip */
    char *type; /* "vxlan_over_ipv4" */
    char *dst_ip;
    bool local;
    uint32_t flags;
    const struct vteprec_physical_locator *pl_cfg;
};

struct unicast_mac_addr {
    char *mac;
    char *ls_name;
    char *key;
    char *ip_addr;
    uint32_t flags;
    struct phy_locator *loc;
    void *mac_cfg;
};

struct multicast_mac_addr {
    char *mac;
    char *ls_name;
    char *key;
    char *ip_addr;
    uint32_t flags;
    struct shash *locs;
    void *mac_cfg;
};

struct switch_prefix_binding {
    char *prefix;
    char *ls_name;
    uint32_t flags;
};

struct static_route {
    char *prefix;
    char *nh;
    uint32_t flags;
};

struct logical_router {
    char *name;
    char *description;
    struct shash switch_bindings;
    struct shash static_routes;
    uint32_t flags;
};

struct tunnel {
    char *key; /* local: type + dst_ip; remote: type + dst_ip */
    char *local_type;
    char *local_dst_ip;
    char *remote_type;
    char *remote_dst_ip;
    struct shash bfd_config_local;
    struct shash bfd_config_remote;
    struct shash bfds;
    struct shash bfd_status;
    uint32_t flags;
    const struct vteprec_tunnel *tnl_cfg;
};

static struct shash *db_tunnels;
static struct shash *db_phy_locs;
static struct shash *db_ucast_local_macs;
static struct shash *db_mcast_local_macs;
static struct shash *db_ucast_remote_macs;
static struct shash *db_mcast_remote_macs;
static struct shash *db_ucast_bfd_local_macs;
static struct shash *db_phy_switches;
static struct shash *db_log_switches;
static struct shash *db_log_routers;

static struct shash *kern_tunnels;
static struct shash *kern_phy_locs;
static struct shash *kern_ucast_local_macs;
static struct shash *kern_mcast_local_macs;
static struct shash *kern_ucast_remote_macs;
static struct shash *kern_mcast_remote_macs;
static struct shash *kern_ucast_bfd_local_macs;
static struct shash *kern_phy_switches;
static struct shash *kern_log_switches;
static struct shash *kern_log_routers;
static struct shash *kern_clag_b2c;
static struct shash *kern_clag_c2b;

static struct shash *bfd_tunnels;

static struct shash *ls_pl_pair;
static struct shash *source_node_ls_list;
static struct shash *ls_dev_to_db_name_ht;
static struct shash *port_vlans_ht;
static struct shash *links_ht;
static struct shash *del_links_ht;
static struct shash *vlan_to_ls_dev_ht;
static struct shash *vb_rn_pair;
static struct shash *vb_ls_pair;

/* TBD: this can be less frequent once we start using netlink to get
 *      mac events, since we will only be polling for stats which
 *      does not need to be very frequent
 */
#define LOCAL_MAC_INTERVAL (2 * 1000) /* In milliseconds. */
#define STATS_POLL_INTERVAL (10 * 1000) /* In milliseconds. */
#define SWOVER_HOLD_TIME (25 * 1000) /* In milliseconds. */
#define PTM_RETRY_INTERVAL (2 * 1000) /* In milliseconds. */
#define PTM_MAX_RETRY_INTERVAL (60 * 1000) /* In milliseconds. */
#define PTM_RETRY_MULTIPLIER 2
static long long int local_mac_timer = LLONG_MIN;
static long long int stats_timer = LLONG_MIN;
static long long int ptm_retry_timer = LLONG_MIN;
static long long int ptm_retry_interval = PTM_RETRY_INTERVAL;
static long long int switchover_timer = LLONG_MIN;
static bool startup_done = false;

static struct multicast_mac_addr *
mcast_mac_lookup(struct shash *mac_tbl, const char *mac, char *ls_dev);

static struct phy_locator *
phy_loc_lookup(struct shash *loc_tbl, const char *encap_type, char *dst_ip);

static struct phy_locator *
phy_loc_create(const char *encap_type, char *dst_ip);

static struct multicast_mac_addr *
mcast_mac_create(struct log_switch *ls, const char *mac, char *ipaddr,
                 struct shash *locator_set);

static struct unicast_mac_addr*
add_del_bfd_local_mac(char* bfd_mac, struct log_switch* ls);

static void phy_loc_destroy(struct phy_locator *pl);

static bool clag_enabled = false;
static bool peerisalive = false;
static bool active = true;
static bool bfd_redirect_added = false;
static bool aborted = false;
unsigned long int *vlan_bitmap = 0;
static bool bfd_sessions_updated = false;
static bool vxln0_protodown_off = false;
static bool db_active = true;
static bool is_db_active(void);
static bool db_active_checked = false;
static bool backup_to_active = false;

typedef struct {
    int pvid;
    unsigned long *vlan_bmp;
} port_vlan_info_t;

static void time_elapsed(struct timeval tv, struct timeval tv1, struct timeval *tv2)
{
    tv2->tv_sec = tv1.tv_sec - tv.tv_sec;
    if (tv1.tv_usec < tv.tv_usec) {
        tv2->tv_usec = 1000000 - tv.tv_usec + tv1.tv_usec;
        tv2->tv_sec -= 1;
    } else
        tv2->tv_usec = tv1.tv_usec - tv.tv_usec;
}

static void get_kernel_port_vlans(void)
{
    size_t lsize = HWVTEP_MAX_STR_LEN - 1;
    char *line = NULL;
    char *save_ptr;
    FILE *fp;

    if (!vtep_vlan_aware)
        return;

    fp = fopen("/tmp/bridge_vlan.tmp", "r");
    if (!fp)
        return;

    while (getline(&line, &lsize, fp) != -1) {
        static char dev_name[HWVTEP_MAX_STR_LEN];
        static port_vlan_info_t *pvinfo;
        char *dn = NULL, *vn = NULL;
        bool pvid = false;

        if (strstr(line, "vlan ids") || strstr(line, "br-vxlan")) {
            free(line);
            line = NULL;
            continue;
        }
        line[strlen(line) - 1] = '\0';
        if (strstr(line, "vxln")) {
            dn = strtok_r(line, "\t", &save_ptr);
            vn = strtok_r(NULL, " ", &save_ptr);
            while (vn && *vn == ' ')
                vn++;
            char *dn_str = xstrdup(dn);
            if (!shash_add_once(vlan_to_ls_dev_ht, vn, dn_str)){
                 free(dn_str);
            }
            free(line);
            line = NULL;
            continue;
        }

        if (line[0] != '\t') {
            dn = strtok_r(line, "\t", &save_ptr);
            vn = strtok_r(NULL, "\t", &save_ptr);
        } else {
            vn = strtok_r(line, "\t", &save_ptr);
        }

        if (dn) {
            strcpy(dev_name, dn);
            VLOG_DBG("The physical port being checked is %s\n", dn);
            pvinfo = xzalloc(sizeof(port_vlan_info_t));
            pvinfo->vlan_bmp = bitmap_allocate(4096);
            if (!shash_add_once(port_vlans_ht, dev_name, pvinfo)) {
                 bitmap_free(pvinfo->vlan_bmp);
                 free(pvinfo);
            }
        }
        if (vn) {
            int vlan1 = 0, vlan2= 0;
            char *v1 = NULL, *v2 = NULL;

            if (strstr(vn, "PVID"))
                pvid = true;

            v1 = strtok_r(vn, "-", &save_ptr);
            if (v1) {
                sscanf(v1, "%d", &vlan1);
                vlan2 = vlan1;
                v2 = strtok_r(NULL, "-", &save_ptr);
                if (v2)
                    sscanf(v2, "%d", &vlan2);
                bitmap_set_multiple(pvinfo->vlan_bmp, vlan1, (vlan2 - vlan1 + 1), 1);
                if (pvid)
                    pvinfo->pvid = vlan1;
            }
        }
        free(line);
        line = NULL;
    }
    if (line) free(line);
    line = NULL;
    fclose(fp);
}

static void get_kernel_links(void)
{
    size_t lsize = HWVTEP_MAX_STR_LEN - 1;
    char *line = NULL;
    char *save_ptr;
    FILE *fp;

    exec_cmd("ip -d -o link show > /tmp/link.tmp");
    fp = fopen("/tmp/link.tmp", "r");
    if (!fp)
        return;

    while (getline(&line, &lsize, fp) != -1) {
        char *dev, *mp, *master = NULL;

        mp = strstr(line, "master");
        strtok_r(line, ":", &save_ptr);        /* ifindex */
        dev = strtok_r(NULL, ":", &save_ptr);  /* name */
        while (*dev == ' ')
            dev++;

        if (mp) {
            strtok_r(mp, " ", &save_ptr);    /* 'master' */
            master = strtok_r(NULL, " ", &save_ptr);
        }

        if (master && vtep_vlan_aware) {
            if (!strncmp(master, "br-vxln", 7) &&
                strcmp(master, "br-vxln0") &&
                (strchr(dev, '.') || !strncmp(dev, "dummy", 5))) {
                char *mname = xstrdup(master);
                char *dname = xstrdup(dev);

                if (!shash_add_once(del_links_ht, dev, dname))
                    free(dname);
                if (!shash_add_once(del_links_ht, master, mname))
                    free(mname);
            }
        }

        if (!vtep_vlan_aware) {
            if (!strncmp(dev, "br-vxlan", 8) ||
                !strncmp(dev, "dummy-vx", 8)) {
                shash_add_once(del_links_ht, dev, xstrdup(dev));
            }
        }
        char *master_str = xstrdup(master?master:"nomaster");
        if (!shash_add_once(links_ht, dev, master_str)) {
            free(master_str);
        }
        free(line);
        line = NULL;
    }
    if (line) free(line);
    line = NULL;
    fclose(fp);
}

static char *find_kernel_link(char *dev)
{
    return shash_find_data(links_ht, dev);
}

static char *
phy_switch_get_local_ip(struct phy_switch *ps)
{
    if (ps) {
        struct shash_node *node = shash_first(&ps->local_ips);
        if (node)
            return node->data;
    }
    return NULL;
}

static struct phy_switch *
phy_switch_lookup(struct shash *ht)
{
    struct shash_node *node;

    if (!ht)
        return NULL;

    node = shash_first(ht);
    if (node)
        return node->data;

    return NULL;
}

static struct phy_switch *
db_phy_switch_lookup(void)
{
    return phy_switch_lookup(db_phy_switches);
}

static struct phy_switch *
kern_phy_switch_lookup(void)
{
    return phy_switch_lookup(kern_phy_switches);
}

static char *
kern_local_ip_get(void)
{
    char *ip = get_clag_vxlan_anycast_ip();
    if (ip)
        return ip;

    return phy_switch_get_local_ip(kern_phy_switch_lookup());
}

static bool
kern_local_ip_valid(char *ip)
{
    struct phy_switch *ps;
    char *anycast = get_clag_vxlan_anycast_ip();

    if (!ip)
        return false;

    if (anycast)
        return strcmp(anycast, ip) == 0;

    ps = kern_phy_switch_lookup();
    if (ps) {
        return (shash_find_data(&ps->local_ips, ip) != NULL);
    }
    return false;
}

static char *
log_switch_dev_name_get(int64_t tunnel_key, int len, char *name)
{
    snprintf(name, len, "vxln%"PRId64"", tunnel_key);
    return name;
}

static struct log_switch *
log_switch_create(char *name, char *description, int64_t tunnel_key)
{
    struct log_switch *ls;
    char dev_name[HWVTEP_MAX_STR_LEN];

    ls = xzalloc(sizeof(*ls));
    ls->name = xstrdup(name);
    ls->description = xstrdup(description);
    ls->tunnel_key = tunnel_key;
    ls->dev_name = xstrdup(log_switch_dev_name_get(tunnel_key,
                               HWVTEP_MAX_STR_LEN, dev_name));
    shash_init(&ls->log_ports);

    VLOG_DBG("Creating logical switch instance %s, %"PRId64" %s\n",
             ls->name, ls->tunnel_key, ls->dev_name);

    return ls;
}

static void
log_switch_destroy(struct log_switch *ls)
{
    if (ls) {
        if (ls->name) free(ls->name);
        if (ls->dev_name) free(ls->dev_name);
        if (ls->description) free(ls->description);
        if (ls->local_ip) free(ls->local_ip);
        shash_destroy(&ls->log_ports);
        free(ls);
    }
}

static struct log_switch *
log_switch_lookup(struct shash *ht, const char *name)
{
    if (!ht || !name) return NULL;
    return shash_find_data(ht, name);
}

static bool
log_switch_sync(struct log_switch *kls, struct log_switch *dbls)
{
    struct shash_node *vnode, *vnode_next;

    VLOG_DBG("    sync log switch: %s\n", kls->name);
    if (kls->tunnel_key != dbls->tunnel_key ||
        strcmp(kls->local_ip, dbls->local_ip)) {
        VLOG_DBG("log switch %s [%s, %"PRId64"] [%s, %"PRId64"]\n", kls->name,
                  kls->local_ip, kls->tunnel_key, dbls->local_ip, dbls->tunnel_key);
        kls->flags |= HWVTEP_FLAGS_DELETE;
        dbls->flags |= HWVTEP_FLAGS_ADD;
        return false;
    }

    /* check the vlan bindings */
    SHASH_FOR_EACH_SAFE (vnode, vnode_next, &dbls->log_ports) {
        struct vlan_binding *kvb, *dbvb = vnode->data;

        kvb = shash_find_data(&kls->log_ports, dbvb->name);
        if (!kvb) {
            dbvb->flags |= HWVTEP_FLAGS_ADD;
            VLOG_DBG("    to add binding %s to kernel\n", dbvb->name);
            continue;
        }

        VLOG_DBG("    %s binding [%"PRId64" %s] [%"PRId64" %s]\n",
                 dbvb->name, kvb->vlan, kvb->ls_name, dbvb->vlan,
                 dbls->dev_name);
        if (kvb->vlan == dbvb->vlan &&
            !strcmp(kvb->ls_name, dbls->dev_name)) {
            kvb->flags |= HWVTEP_FLAGS_TOUCHED;
            dbvb->flags |= HWVTEP_FLAGS_TOUCHED;
        } else {
            kvb->flags |= HWVTEP_FLAGS_DELETE;
            dbvb->flags |= HWVTEP_FLAGS_ADD;
        }
        memcpy(&dbvb->stats, &kvb->stats, sizeof(dbvb->stats));
    }

    kls->flags |= HWVTEP_FLAGS_TOUCHED;
    dbls->flags |= HWVTEP_FLAGS_TOUCHED;
    return true;
}

static void
add_del_log_switches(void)
{
    const struct vteprec_logical_switch *logical_switch;
    struct phy_switch *ps = NULL;
    struct shash_node *node, *next;
    /* TODO check the tip and dbls->local_ip logic*/
    char *tip = kern_local_ip_get();

    VLOG_DBG("sync: log switches\n");
    logical_switch = vteprec_logical_switch_first(vtep_idl);
    while (logical_switch) {
        struct log_switch *dbls = NULL;

        if (!strcmp(nvp_internal_str, logical_switch->name)) {
            logical_switch = vteprec_logical_switch_next(logical_switch);
            continue;
        }

        if (logical_switch->n_tunnel_key == 0) {
            VLOG_ERR("     log switch %s missing tunnel key",
                     logical_switch->name);
        } else if (!tip) {
            VLOG_ERR("    logical switch %s missing local tunnel ip",
                     logical_switch->name);
        } else if (shash_find_data(db_log_switches, logical_switch->name)) {
            VLOG_ERR("    logical switch %s already exists!",
                     logical_switch->name);
        } else {
            dbls = log_switch_create(logical_switch->name,
                                     logical_switch->description,
                                     logical_switch->tunnel_key[0]);
            dbls->ls_cfg = (struct vteprec_logical_switch *)logical_switch;
            shash_add(db_log_switches, dbls->name, dbls);
            if (!shash_find_data(ls_dev_to_db_name_ht, dbls->dev_name)) {
                shash_add(ls_dev_to_db_name_ht, dbls->dev_name,
                          xstrdup(dbls->name));
            }
        }
        logical_switch = vteprec_logical_switch_next(logical_switch);
    }
    /* walk the physical ports and insert the port bindings */
    ps = db_phy_switch_lookup();
    if (!ps)
        return;

    SHASH_FOR_EACH_SAFE (node, next, &ps->ports) {
        struct phy_port *pp = node->data;
        struct shash_node *vnode, *vnode_next;

        SHASH_FOR_EACH_SAFE (vnode, vnode_next, &pp->vlan_bindings) {
            struct vlan_binding *vb = vnode->data;
            struct log_switch *ls;

            /* record the binding in the logical switch also */
            ls = log_switch_lookup(db_log_switches, vb->ls_name);
            if (ls) {
                shash_add_once(&ls->log_ports, vb->name, vb);
                if (vb->vlan != 0)
                    ls->vlan = vb->vlan;
                VLOG_DBG("    => got db log port %s vlan %"PRId64" : log switch %s (%s)\n",
                          vb->name, vb->vlan, vb->ls_name, ls->dev_name);
            }
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, db_log_switches) {
        struct log_switch *dbls = node->data;
        struct log_switch *kls = NULL;
        struct multicast_mac_addr *mm;

        dbls->local_ip = xstrdup(tip);
        mm = mcast_mac_lookup(kern_mcast_local_macs, unknown_dst_str,
                              dbls->dev_name);
        if (!mm) {
            struct phy_locator *pl;
            struct shash *locs;

            locs = xzalloc(sizeof(*locs));
            shash_init(locs);

            pl = phy_loc_lookup(db_phy_locs, vxlan_encap_type, tip);
            if (!pl) {
                pl = phy_loc_create(vxlan_encap_type, tip);
                pl->local = true;
                if(!shash_add_once(db_phy_locs, pl->key, pl)){
                    phy_loc_destroy(pl);
                }
            }
            if (pl)
                shash_add_once(locs, pl->key, pl);
            mm = mcast_mac_create(dbls, unknown_dst_str, "", locs);
            /* TBD: hack */
            free(mm->ls_name);
            mm->ls_name = xstrdup(dbls->dev_name);
            VLOG_DBG("    => got local mcast unknown-dst, %d %s\n",
                      (int)shash_count(locs), dbls->name);
            if(!shash_add_once(kern_mcast_local_macs, mm->key, mm)) {
                free(mm->ls_name);
            }
        }

        if (shash_count(&dbls->log_ports) != 0) {
            /* already created the local unknown-dst mcast mac */
            VLOG_DBG("non empty logical ports\n");
        } else {
            /* mark the unknown-dst mcast mac for delete */
            VLOG_DBG("LS no more bindings, deleting.  %d\n",
                      (int)shash_count(kern_mcast_local_macs));
            if (mm)
                mm->flags |= HWVTEP_FLAGS_DELETE;

            /* delete the log switch also.  There should be at most one
             * unique active logical switch with a particular tunnel key
             */
            /* TBD: we should be deleting here, otherwise, we may have
             *      multiple db log switch instances with the same tunnel key
             *      as it is allowed by the controller.  It is a controller
             *      cleanup issue which will be fixed, but here we do allow
             *      the chance to get incorrect reference due to aliasing.
             *      If we delete here, objects that reference the log switch
             *      may not be able to build, and the case that breaks
             *      currently is that unknown-dst local in the db cannot be
             *      removed because the kernel object cannot get back the db
             *      handle.
            shash_delete(db_log_switches, node);
            log_switch_destroy(dbls);
             */
            continue;
        }

        kls = log_switch_lookup(kern_log_switches, dbls->dev_name);
        if (!kls) {
            dbls->flags |= HWVTEP_FLAGS_ADD;
            VLOG_DBG("    to add log switch %s to kernel\n", dbls->name);
            continue;
        }
        if (!kern_local_ip_valid(kls->local_ip)) {
            VLOG_DBG("    local ip changed %s -> %s\n",
                      kls->local_ip?kls->local_ip:"-", dbls->local_ip);
            //kls->flags |= HWVTEP_FLAGS_DELETE;
            //dbls->flags |= HWVTEP_FLAGS_ADD;
            kls->flags |= HWVTEP_FLAGS_TOUCHED;
            dbls->flags = 0;
            continue;
        }

        /* sync between the old and new, and reuse same local ip */
        if (kls->local_ip)
            free(kls->local_ip);

        kls->local_ip = xstrdup(dbls->local_ip);
        log_switch_sync(kls, dbls);
    }
}

/*static struct logical_router *
log_router_create(const struct vteprec_logical_router *log_rtr)
{
    struct logical_router *lr;
    int i;

    lr = xzalloc(sizeof(*lr));
    lr->name = xstrdup(log_rtr->name);
    lr->description = xstrdup(log_rtr->description);
    shash_init(&lr->switch_bindings);
    shash_init(&lr->static_routes);
    for (i = 0; i < log_rtr->n_switch_binding; i++) {
        struct switch_prefix_binding *spb = xzalloc(sizeof(*spb));
        spb->prefix = xstrdup(log_rtr->key_switch_binding[i]);
        spb->ls_name = xstrdup(log_rtr->value_switch_binding[i]->name);
        if (!shash_add_once(&lr->switch_bindings, spb->prefix, spb)) {
            free(spb->prefix);
            free(spb->ls_name);
            free(spb);
            continue;
        }
    }
    for (i = 0; i < log_rtr->n_static_routes; i ++) {
        struct static_route *srt = xzalloc(sizeof(*srt));
        srt->prefix = xstrdup(log_rtr->key_static_routes[i]);
        srt->nh = xstrdup(log_rtr->value_static_routes[i]);
        if (!shash_add_once(&lr->static_routes, srt->prefix, srt)) {
            free(srt->prefix);
            free(srt->nh);
            free(srt);
            continue;
        }
    }
    return lr;
}

static void
log_router_destroy(struct logical_router *lr)
{
    struct shash_node *node, *next;

    free(lr->name);
    free(lr->description);

    SHASH_FOR_EACH_SAFE (node, next, &lr->switch_bindings) {
        struct switch_prefix_binding *spb = node->data;

        shash_delete(&lr->switch_bindings, node);
        free(spb->prefix);
        free(spb->ls_name);
        free(spb);
    }
    shash_destroy(&lr->switch_bindings);

    SHASH_FOR_EACH_SAFE (node, next, &lr->static_routes) {
        struct static_route *srt = node->data;

        shash_delete(&lr->static_routes, node);
        free(srt->prefix);
        free(srt->nh);
        free(srt);
    }
    shash_destroy(&lr->static_routes);
    free(lr);
}

static struct logical_router *
log_router_lookup(struct shash *lr_tbl, char *lr_name)
{
    if (!lr_tbl) return NULL;

    return shash_find_data(lr_tbl, lr_name);
}

static void
log_router_sync(struct logical_router *olr, struct logical_router *nlr)
{
    struct shash_node *node, *next;

    olr->flags |= HWVTEP_FLAGS_TOUCHED;
    nlr->flags |= HWVTEP_FLAGS_TOUCHED;

    SHASH_FOR_EACH_SAFE (node, next, &nlr->switch_bindings) {
        struct switch_prefix_binding *ospb, *nspb;

        nspb = node->data;
        ospb = shash_find_data(&olr->switch_bindings, nspb->prefix);
        if (!ospb) {
            nspb->flags |= HWVTEP_FLAGS_ADD;
            continue;
        }
        if (strcmp(ospb->ls_name, nspb->ls_name)) {
            ospb->flags |= HWVTEP_FLAGS_DELETE;
            nspb->flags |= HWVTEP_FLAGS_ADD;
        } else {
            ospb->flags |= HWVTEP_FLAGS_TOUCHED;
            nspb->flags |= HWVTEP_FLAGS_TOUCHED;
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, &nlr->static_routes) {
        struct static_route *osrt, *nsrt;

        nsrt = node->data;
        osrt = shash_find_data(&olr->static_routes, nsrt->prefix);
        if (!osrt) {
            nsrt->flags |= HWVTEP_FLAGS_ADD;
            continue;
        }
        if (strcmp(osrt->nh, nsrt->nh)) {
            osrt->flags |= HWVTEP_FLAGS_DELETE;
            nsrt->flags |= HWVTEP_FLAGS_ADD;
        } else {
            osrt->flags |= HWVTEP_FLAGS_TOUCHED;
            nsrt->flags |= HWVTEP_FLAGS_TOUCHED;
        }
    }
}

static void
add_del_log_routers(void)
{
    const struct vteprec_logical_router *log_rtr;

    log_rtr = vteprec_logical_router_first(vtep_idl);
    while(log_rtr) {
        struct logical_router *nlr;
        struct logical_router *olr;

        nlr = log_router_create(log_rtr);
        olr = log_router_lookup(db_log_routers, log_rtr->name);
        log_rtr = vteprec_logical_router_next(log_rtr);
        if (!shash_add_once(db_log_routers, nlr->name, nlr)) {
            log_router_destroy(nlr);
            continue;
        }
        if (!olr) {
            nlr->flags |= HWVTEP_FLAGS_ADD;
        } else {
            log_router_sync(olr, nlr);
        }
    }
}*/

static struct log_switch*
add_del_bfd_ctrl_ls(int64_t tunnel_key)
{
    struct log_switch *dbls, *kls;
    char *tip = kern_local_ip_get();
    char dev_name[HWVTEP_MAX_STR_LEN];
    char ls_name[HWVTEP_MAX_STR_LEN];

    log_switch_dev_name_get(tunnel_key, HWVTEP_MAX_STR_LEN, dev_name);
    dbls = log_switch_lookup(db_log_switches,
                             shash_find_data(ls_dev_to_db_name_ht, dev_name));
    if (!dbls) {
        snprintf(ls_name, HWVTEP_MAX_STR_LEN, "_BFD_LS%"PRId64"", tunnel_key);
        dbls = log_switch_create(dev_name, "", tunnel_key);
        dbls->ls_cfg = NULL; // implicitly created log switch for bfd ctrl
        dbls->local_ip = xstrdup(tip);
        shash_add_once(db_log_switches, dbls->name, dbls);
        shash_add_once(ls_dev_to_db_name_ht, dbls->dev_name, xstrdup(dbls->name));
    }

    kls = log_switch_lookup(kern_log_switches, dbls->dev_name);
    if (!kls) {
        dbls->flags |= HWVTEP_FLAGS_ADD;
        VLOG_DBG("    to add log switch %s to kernel\n", dbls->name);
        return dbls;
    }
    if (!kern_local_ip_valid(kls->local_ip)) {
        VLOG_DBG("    local ip changed %s -> %s\n",
                  kls->local_ip?kls->local_ip:"-", dbls->local_ip);
        kls->flags |= HWVTEP_FLAGS_DELETE;
        dbls->flags |= HWVTEP_FLAGS_ADD;
        return dbls;
    }
    /* sync between the old and new, and reuse same local ip */
    //if (kls->local_ip)
    //    free(kls->local_ip);

    //kls->local_ip = xstrdup(dbls->local_ip);
    log_switch_sync(kls, dbls);
    return dbls;
}

static struct phy_locator *
phy_loc_create(const char *encap_type, char *dst_ip)
{
    struct phy_locator *pl;

    VLOG_DBG("create phy loc: dst %s\n", dst_ip);
    pl = xzalloc(sizeof(*pl));
    pl->type = xstrdup(encap_type);
    pl->dst_ip = xstrdup(dst_ip);
    pl->key = xzalloc(strlen(pl->type) + strlen(pl->dst_ip) + 1);
    strcpy(pl->key, pl->type);
    strcat(pl->key, pl->dst_ip);
    return pl;
}

static struct phy_locator *
phy_loc_lookup(struct shash *loc_tbl, const char *encap_type, char *dst_ip)
{
    char *key;
    struct phy_locator *pl;

    if (!loc_tbl) return NULL;

    key = xzalloc(strlen(encap_type) + strlen(dst_ip) + 1);
    strcpy(key, encap_type);
    strcat(key, dst_ip);
    pl = shash_find_data(loc_tbl, key);
    free(key);
    return pl;
}

static void
phy_loc_destroy(struct phy_locator *pl)
{
    if (!pl) return;
    if (pl->key) free(pl->key);
    if (pl->type) free(pl->type);
    if (pl->dst_ip) free(pl->dst_ip);
    free(pl);
    return;
}

static void
add_del_phy_locators(void)
{
    const struct vteprec_physical_locator *phy_loc;
    char *tip = kern_local_ip_get();

    VLOG_DBG("sync: phy locators\n");
    phy_loc = vteprec_physical_locator_first(vtep_idl);
    while (phy_loc) {
        bool is_local = false;
        struct phy_locator *dbpl;
        struct phy_locator *kpl;
        char *dst_ip = phy_loc->dst_ip;

        kpl = phy_loc_lookup(kern_phy_locs, phy_loc->encapsulation_type,
                             phy_loc->dst_ip);
        if (kpl && kpl->local) {
            /* this is a local locator, check for local ip change */
            if (!tip) {
                kpl->flags = HWVTEP_FLAGS_DELETE;
                phy_loc = vteprec_physical_locator_next(phy_loc);
                continue;
            }
            if (strcmp(dst_ip, tip) != 0)
                kpl->flags = HWVTEP_FLAGS_DELETE;

            dst_ip = tip;
            is_local = kpl->local;
        }

        dbpl = phy_loc_lookup(db_phy_locs, phy_loc->encapsulation_type,
                              phy_loc->dst_ip);
        if (!dbpl) {
            dbpl = phy_loc_create(phy_loc->encapsulation_type,
                                  dst_ip);
            if(!shash_add_once(db_phy_locs, dbpl->key, dbpl)) {
                phy_loc_destroy(dbpl);
                phy_loc = vteprec_physical_locator_next(phy_loc);
                continue;
            }
        }
        dbpl->pl_cfg = phy_loc;
        dbpl->local = is_local;

        phy_loc = vteprec_physical_locator_next(phy_loc);

        if (!kpl) {
            dbpl->flags |= HWVTEP_FLAGS_ADD;
        } else {
            /* sync the locators */
            /* TBD: how do we deal with bfds ? */
            kpl->flags |= HWVTEP_FLAGS_TOUCHED;
            dbpl->flags |= HWVTEP_FLAGS_TOUCHED;
        }
    }
}

static struct shash *
phy_loc_set_create(struct vteprec_physical_locator_set *phy_loc_set)
{
    struct shash *pls = NULL;
    int i;

    if (!phy_loc_set)
        return NULL;

    pls = xzalloc(sizeof(*pls));
    shash_init(pls);
    for (i = 0; i < phy_loc_set->n_locators; i++) {
        struct vteprec_physical_locator *phy_loc;
        struct phy_locator *pl;

        phy_loc = phy_loc_set->locators[i];
        pl = phy_loc_lookup(db_phy_locs,
                            phy_loc->encapsulation_type,
                            phy_loc->dst_ip);
        if (pl) {
            shash_add_once(pls, pl->key, pl);
        }
        /* this assumes db_phy_locs already synced */
    }

    return pls;
}

static void
phy_loc_set_destroy(struct shash *phy_loc_set)
{
    /* no need to destroy each locator, as this is just a reference */
    shash_destroy(phy_loc_set);
    free(phy_loc_set);
}

static struct tunnel*
tunnel_create(const char *local_encap_type, char *local_ip,
              const char *remote_encap_type, char *remote_ip,
              const struct vteprec_tunnel *tunnel)
{
    struct tunnel *tnl;

    VLOG_DBG("create tunnel local %s - remote %s\n",
             local_ip, remote_ip);

    tnl = xzalloc(sizeof(*tnl));
    tnl->key = xzalloc(strlen(local_encap_type) + strlen(local_ip)
                       + strlen(remote_encap_type) + strlen(remote_ip) + 1);
    strcpy(tnl->key, local_encap_type);
    strcat(tnl->key, local_ip);
    strcat(tnl->key, remote_encap_type);
    strcat(tnl->key, remote_ip);
    tnl->local_type = xstrdup(local_encap_type);
    tnl->local_dst_ip = xstrdup(local_ip);
    tnl->remote_type = xstrdup(remote_encap_type);
    tnl->remote_dst_ip = xstrdup(remote_ip);

    shash_init(&tnl->bfd_config_local);
    shash_init(&tnl->bfd_config_remote);
    shash_init(&tnl->bfds);
    shash_init(&tnl->bfd_status);

    if (!tunnel) {
        return tnl;
    }

    tnl->tnl_cfg = tunnel;

    struct smap_node *node;
    size_t i;
    struct ovsdb_datum *datum = xmalloc(sizeof(struct ovsdb_datum));

    datum->n = smap_count(&tunnel->bfd_config_local);
    datum->keys = xmalloc(datum->n * sizeof *datum->keys);
    datum->values = xmalloc(datum->n * sizeof *datum->values);
    i = 0;
    SMAP_FOR_EACH (node, &tunnel->bfd_config_local) {
                   datum->keys[i].string = node->key;
                   datum->values[i].string = node->value;
        i++;
    }
    for (i = 0; i < datum->n; i++) {
        shash_add_once(&tnl->bfd_config_local,
                       datum->keys[i].string,
                       xstrdup(datum->values[i].string));
    }
    free(datum->keys);
    free(datum->values);

    datum->n = smap_count(&tunnel->bfd_config_remote);
    datum->keys = xmalloc(datum->n * sizeof *datum->keys);
    datum->values = xmalloc(datum->n * sizeof *datum->values);
    i = 0;
    SMAP_FOR_EACH (node, &tunnel->bfd_config_remote) {
                   datum->keys[i].string = node->key;
                   datum->values[i].string = node->value;
        i++;
    }
    for (i = 0; i < datum->n; i++) {
        shash_add_once(&tnl->bfd_config_remote,
                       datum->keys[i].string,
                       xstrdup(datum->values[i].string));
    }
    free(datum->keys);
    free(datum->values);

    datum->n = smap_count(&tunnel->bfd_params);
    datum->keys = xmalloc(datum->n * sizeof *datum->keys);
    datum->values = xmalloc(datum->n * sizeof *datum->values);
    i = 0;
    SMAP_FOR_EACH (node, &tunnel->bfd_params) {
                   datum->keys[i].string = node->key;
                   datum->values[i].string = node->value;
        i++;
    }
    for (i = 0; i < datum->n; i++) {
        shash_add_once(&tnl->bfds,
                       datum->keys[i].string,
                       xstrdup(datum->values[i].string));
    }
    free(datum->keys);
    free(datum->values);

    datum->n = smap_count(&tunnel->bfd_status);
    datum->keys = xmalloc(datum->n * sizeof *datum->keys);
    datum->values = xmalloc(datum->n * sizeof *datum->values);
    i = 0;
    SMAP_FOR_EACH (node, &tunnel->bfd_status) {
                   datum->keys[i].string = node->key;
                   datum->values[i].string = node->value;
        i++;
    }
    for (i = 0; i < datum->n; i++) {
        shash_add_once(&tnl->bfd_status,
                       datum->keys[i].string,
                       xstrdup(datum->values[i].string));
    }
    free(datum->keys);
    free(datum->values);
    free(datum);
    return tnl;
}

static struct tunnel*
tunnel_lookup(struct shash* tunnels, const char* local_encap_type, char* local_dst_ip,
              const char* remote_encap_type, char* remote_dst_ip)
{
    char *key;
    struct tunnel *tnl;

    if (!tunnels) return NULL;

    key = xzalloc(strlen(local_encap_type) + strlen(local_dst_ip)
                  + strlen(remote_encap_type) + strlen(remote_dst_ip) + 1);
    strcpy(key, local_encap_type);
    strcat(key, local_dst_ip);
    strcat(key, remote_encap_type);
    strcat(key, remote_dst_ip);
    tnl = shash_find_data(tunnels, key);
    free(key);
    return tnl;
}

static bool
tunnel_compare_bfd_config_local(struct tunnel* db_tnl, struct tunnel* kern_tnl)
{
    const char *db_bfd_dst_mac, *db_bfd_dst_ip;
    const char *kern_bfd_dst_mac, *kern_bfd_dst_ip;

    db_bfd_dst_mac = shash_find_data(&db_tnl->bfd_config_local, bfd_dst_mac);
    if (!db_bfd_dst_mac) {
         db_bfd_dst_mac = default_bfd_mac;
    }
    kern_bfd_dst_mac = shash_find_data(&kern_tnl->bfd_config_local, bfd_dst_mac);
    if (!kern_bfd_dst_mac) {
        kern_bfd_dst_mac = default_bfd_mac;
    }
    if (strcmp(db_bfd_dst_mac, kern_bfd_dst_mac) != 0) {
        return false;
    }

    db_bfd_dst_ip = shash_find_data(&db_tnl->bfd_config_local, bfd_dst_ip);
    if (!db_bfd_dst_ip) {
        db_bfd_dst_ip = default_bfd_local_dst_ip;
    }
    kern_bfd_dst_ip = shash_find_data(&kern_tnl->bfd_config_local, bfd_dst_ip);
    if (!kern_bfd_dst_ip) {
        kern_bfd_dst_ip = default_bfd_local_dst_ip;
    }
    if (strcmp(db_bfd_dst_ip, kern_bfd_dst_ip) != 0) {
        return false;
    }
    return true;
}

static bool
tunnel_compare_bfd_config_remote(struct tunnel* db_tnl, struct tunnel* kern_tnl)
{
    const char *db_bfd_dst_mac, *db_bfd_dst_ip;
    const char *kern_bfd_dst_mac, *kern_bfd_dst_ip;

    db_bfd_dst_mac = shash_find_data(&db_tnl->bfd_config_remote, bfd_dst_mac);
    if (!db_bfd_dst_mac) {
         db_bfd_dst_mac = default_bfd_mac;
    }
    kern_bfd_dst_mac = shash_find_data(&kern_tnl->bfd_config_remote, bfd_dst_mac);
    if (!kern_bfd_dst_mac) {
        kern_bfd_dst_mac = default_bfd_mac;
    }
    if (strcmp(db_bfd_dst_mac, kern_bfd_dst_mac) != 0) {
        return false;
    }

    db_bfd_dst_ip = shash_find_data(&db_tnl->bfd_config_remote, bfd_dst_ip);
    if (!db_bfd_dst_ip) {
        db_bfd_dst_ip = default_bfd_remote_dst_ip;
    }
    kern_bfd_dst_ip = shash_find_data(&kern_tnl->bfd_config_remote, bfd_dst_ip);
    if (!kern_bfd_dst_ip) {
        kern_bfd_dst_ip = default_bfd_remote_dst_ip;
    }
    if (strcmp(db_bfd_dst_ip, kern_bfd_dst_ip) != 0) {
        return false;
    }
    return true;
}

static bool
tunnel_compare_bfd_params(struct tunnel* db_tnl, struct tunnel* kern_tnl)
{
    const char *db_enable, *kern_enable;
    const char *db_min_rx, *kern_min_rx;
    const char *db_min_tx, *kern_min_tx;
    const char *db_decay_min_rx, *kern_decay_min_rx;
    const char *db_forwarding_if_rx, *kern_forwarding_if_rx;
    const char *db_cpath_down, *kern_cpath_down;
    const char *db_check_tnl_key, *kern_check_tnl_key;

    // bfd_params: enable
    db_enable = shash_find_data(&db_tnl->bfds, bfd_enable);
    if(!db_enable)
       db_enable = default_bfd_enable;
    kern_enable = shash_find_data(&kern_tnl->bfds, bfd_enable);
    if (!kern_enable)
        kern_enable = default_bfd_enable;

    if (strcmp(db_enable, kern_enable) != 0)
        return false;

    // bfd_params: min_rx
    db_min_rx = shash_find_data(&db_tnl->bfds, bfd_min_rx);
    if (!db_min_rx)
        db_min_rx = default_min_rx;
    kern_min_rx = shash_find_data(&kern_tnl->bfds, bfd_min_rx);
    if (!kern_min_rx)
        kern_min_rx = default_min_rx;

    if (strcmp(db_min_rx, kern_min_rx) != 0)
        return false;

    // bfd_params: min_tx
    db_min_tx = shash_find_data(&db_tnl->bfds, bfd_min_tx);
    if (!db_min_tx)
        db_min_tx = default_min_tx;
    kern_min_tx = shash_find_data(&kern_tnl->bfds, bfd_min_tx);
    if (!kern_min_tx)
        kern_min_tx = default_min_tx;

    if (strcmp(db_min_tx, kern_min_tx) != 0)
        return false;

    // bfd_params: decay_min_rx
    db_decay_min_rx = shash_find_data(&db_tnl->bfds, bfd_decay_min_rx);
    if (!db_decay_min_rx)
        db_decay_min_rx = default_decay_min_rx;
    kern_decay_min_rx = shash_find_data(&kern_tnl->bfds, bfd_decay_min_rx);
    if (!kern_decay_min_rx)
        kern_decay_min_rx = default_decay_min_rx;

    if (strcmp(db_decay_min_rx, kern_decay_min_rx) != 0)
        return false;

    // bfd_params: forwarding_if_rx
    db_forwarding_if_rx = shash_find_data(&db_tnl->bfds, bfd_fwd_if_rx);
    if (!db_forwarding_if_rx)
        db_forwarding_if_rx = default_forwarding_if_rx;
    kern_forwarding_if_rx = shash_find_data(&kern_tnl->bfds, bfd_fwd_if_rx);
    if (!kern_forwarding_if_rx)
        kern_forwarding_if_rx = default_forwarding_if_rx;

    if (strcmp(db_forwarding_if_rx, kern_forwarding_if_rx) != 0)
        return false;

    // bfd_params: cpath_down
    db_cpath_down = shash_find_data(&db_tnl->bfds, bfd_cpath_down);
    if (!db_cpath_down)
        db_cpath_down = default_cpath_down;
    kern_cpath_down = shash_find_data(&kern_tnl->bfds, bfd_cpath_down);
    if (!kern_cpath_down)
        kern_cpath_down = default_cpath_down;

    if (strcmp(db_cpath_down, kern_cpath_down) != 0)
        return false;

    // bfd_params: check_tnl_key
    db_check_tnl_key = shash_find_data(&db_tnl->bfds, bfd_check_tnl_key);
    if (!db_check_tnl_key)
        db_check_tnl_key = default_check_tnl_key;
    kern_check_tnl_key = shash_find_data(&kern_tnl->bfds, bfd_check_tnl_key);
    if (!kern_check_tnl_key)
        kern_check_tnl_key = default_check_tnl_key;

    if (strcmp(db_check_tnl_key, kern_check_tnl_key) != 0)
        return false;

    return true;
}

static int
tunnel_compare_bfd_status(struct tunnel* db_tnl, struct tunnel* kern_tnl)
{
    const char *db_enabled, *kern_enabled;
    const char *db_state, *kern_state;
    const char *db_forwarding, *kern_forwarding;
    const char *db_diagnostic, *kern_diagnostic;
    const char *db_remote_state, *kern_remote_state;
    const char *db_rem_diagnostic, *kern_rem_diagnostic;
    const char *db_info, *kern_info;

    // bfd_status: enabled
    db_enabled = shash_find_data(&db_tnl->bfd_status, bfd_enabled);
    if (!db_enabled)
        db_enabled = default_enabled;
    kern_enabled = shash_find_data(&kern_tnl->bfd_status, bfd_enabled);
    if (!kern_enabled)
        kern_enabled = default_enabled;

    if (strcmp(db_enabled, kern_enabled) != 0)
        return false;

    //bfd_status: state
    db_state = shash_find_data(&db_tnl->bfd_status, bfd_state);
    if (!db_state)
        db_state = default_state;
    kern_state = shash_find_data(&kern_tnl->bfd_status, bfd_state);
    if (!kern_state)
        kern_state = default_state;

    if (strcmp(db_state, kern_state) != 0)
        return false;

    //bfd_status: forwarding
    db_forwarding = shash_find_data(&db_tnl->bfd_status, bfd_forwarding);
    if (!db_forwarding)
        db_forwarding = default_forwarding;
    kern_forwarding = shash_find_data(&kern_tnl->bfd_status, bfd_forwarding);
    if (!kern_forwarding)
        kern_forwarding = default_forwarding;

    if (strcmp(db_forwarding, kern_forwarding) != 0)
        return false;

    //bfd_status: diagnostic
    db_diagnostic = shash_find_data(&db_tnl->bfd_status, bfd_diag);
    if (!db_diagnostic)
        db_diagnostic = "";
    kern_diagnostic = shash_find_data(&kern_tnl->bfd_status, bfd_diag);
    if (!kern_diagnostic)
        kern_diagnostic = "";

    if (strcmp(db_diagnostic, kern_diagnostic) != 0)
        return false;

    //bfd_status: remote_state
    db_remote_state = shash_find_data(&db_tnl->bfd_status, bfd_remote_state);
    if (!db_remote_state)
        db_remote_state = default_remote_state;
    kern_remote_state = shash_find_data(&kern_tnl->bfd_status, bfd_remote_state);
    if (!kern_remote_state)
        kern_remote_state = default_remote_state;
    if (strcmp(db_remote_state, kern_remote_state) != 0) {
        return false;
    }

    //bfd_status: remote_diagnostic
    db_rem_diagnostic = shash_find_data(&db_tnl->bfd_status,
                                        bfd_remote_diag);
    if (!db_rem_diagnostic)
        db_rem_diagnostic = "";
    kern_rem_diagnostic = shash_find_data(&kern_tnl->bfd_status,
                                          bfd_remote_diag);
    if (!kern_rem_diagnostic)
        kern_rem_diagnostic = "";

    if (strcmp(db_rem_diagnostic, kern_rem_diagnostic) != 0)
        return false;

    // bfd_status: info
    db_info = shash_find_data(&db_tnl->bfd_status, bfd_info);
    if (!db_info)
        db_info = "";
    kern_info = shash_find_data(&kern_tnl->bfd_status, bfd_info);
    if (!kern_info)
        kern_info = "";

    if (strcmp(db_info, kern_info) != 0)
        return false;

    return true;
}

static void
tunnel_destroy(struct tunnel *tnl)
{
    if (!tnl) return;
    if (tnl->key) free(tnl->key);
    if (tnl->local_type) free(tnl->local_type);
    if (tnl->local_dst_ip) free(tnl->local_dst_ip);
    if (tnl->remote_type) free(tnl->remote_type);
    if (tnl->remote_dst_ip) free(tnl->remote_dst_ip);

    shash_destroy_free_data(&tnl->bfd_config_local);
    shash_destroy_free_data(&tnl->bfd_config_remote);
    shash_destroy_free_data(&tnl->bfds);
    shash_destroy_free_data(&tnl->bfd_status);

    free(tnl);
    return;
}

static void
tunnel_copy_bfd_info(struct tunnel* tnl1, struct tunnel* tnl2)
{
    char *dst_mac, *dst_ip;
    char *enable, *min_rx, *min_tx, *decay_min_rx, *forwarding_if_rx;
    char *check_tnl_key, *cpath_down;
    char *enabled, *state, *forwarding, *diagnostic;

    /* config local */
    dst_mac = shash_find_data(&tnl1->bfd_config_local, bfd_dst_mac);
    if (dst_mac) {
        shash_add_once(&tnl2->bfd_config_local, bfd_dst_mac, xstrdup(dst_mac));
    }
    dst_ip = shash_find_data(&tnl1->bfd_config_local, bfd_dst_ip);
    if (dst_ip) {
        shash_add_once(&tnl2->bfd_config_local, bfd_dst_ip, xstrdup(dst_ip));
    }

    /* config remote */
    dst_mac = shash_find_data(&tnl1->bfd_config_remote, bfd_dst_mac);
    if (dst_mac) {
        shash_add_once(&tnl2->bfd_config_remote, bfd_dst_mac, xstrdup(dst_mac));
    }
    dst_ip = shash_find_data(&tnl1->bfd_config_remote, bfd_dst_ip);
    if (dst_ip) {
        shash_add_once(&tnl2->bfd_config_remote, bfd_dst_ip, xstrdup(dst_ip));
    }

    /* bfd params */
    enable = shash_find_data(&tnl1->bfds, bfd_enable);
    if (enable) {
        shash_add_once(&tnl2->bfds, bfd_enable, xstrdup(enable));
    }
    min_rx = shash_find_data(&tnl1->bfds, bfd_min_rx);
    if (min_rx) {
        shash_add_once(&tnl2->bfds, bfd_min_rx, xstrdup(min_rx));
    }
    min_tx = shash_find_data(&tnl1->bfds, bfd_min_tx);
    if (min_tx) {
        shash_add_once(&tnl2->bfds, bfd_min_tx, xstrdup(min_tx));
    }
    decay_min_rx = shash_find_data(&tnl1->bfds, bfd_decay_min_rx);
    if (decay_min_rx) {
        shash_add_once(&tnl2->bfds, bfd_decay_min_rx,
                       xstrdup(decay_min_rx));
    }
    forwarding_if_rx = shash_find_data(&tnl1->bfds, bfd_fwd_if_rx);
    if (forwarding_if_rx) {
        shash_add_once(&tnl2->bfds, bfd_fwd_if_rx,
                       xstrdup(forwarding_if_rx));
    }
    cpath_down = shash_find_data(&tnl1->bfds, bfd_cpath_down);
    if (cpath_down) {
        shash_add_once(&tnl2->bfds, bfd_cpath_down,
                       xstrdup(cpath_down));
    }
    check_tnl_key = shash_find_data(&tnl1->bfds, bfd_check_tnl_key);
    if (check_tnl_key) {
        shash_add_once(&tnl2->bfds, bfd_check_tnl_key,
                       xstrdup(check_tnl_key));
    }

    /* bfd status */
    enabled = shash_find_data(&tnl1->bfd_status, bfd_enabled);
    if (enabled) {
        shash_add_once(&tnl2->bfd_status, bfd_enabled, xstrdup(enabled));
    }

    state = shash_find_data(&tnl1->bfd_status, bfd_state);
    if (state) {
        shash_add_once(&tnl2->bfd_status, bfd_state, xstrdup(state));
    }
    forwarding = shash_find_data(&tnl1->bfd_status, bfd_forwarding);
    if (forwarding) {
        shash_add_once(&tnl2->bfd_status, bfd_forwarding, xstrdup(forwarding));
    }
    diagnostic = shash_find_data(&tnl1->bfd_status, bfd_diag);
    if (diagnostic) {
        shash_add_once(&tnl2->bfd_status, bfd_diag, xstrdup(diagnostic));
    }
}

static void
add_bfd_tunnel(struct tunnel *tnl)
{
    struct tunnel *bfd_tnl;
    char *enabled = "true";
    char *old_data;

    bfd_tnl = tunnel_lookup(bfd_tunnels, tnl->local_type, tnl->local_dst_ip,
                            tnl->remote_type, tnl->remote_dst_ip);
    if (!bfd_tnl) {
        bfd_tnl = tunnel_create(tnl->local_type, tnl->local_dst_ip,
                                tnl->remote_type, tnl->remote_dst_ip, NULL);
        shash_add_once(bfd_tunnels, bfd_tnl->key, bfd_tnl);
        VLOG_DBG("Create bfd tunnel %s - %s\n", bfd_tnl->local_dst_ip,
                  bfd_tnl->remote_dst_ip);
    }
    tunnel_copy_bfd_info(tnl, bfd_tnl);
    VLOG_DBG("Add bfd tunnel %s - %s\n", bfd_tnl->local_dst_ip,
              bfd_tnl->remote_dst_ip);
    /* assume bfd session got enabled successfully.
     * if 'start-bfd-sess' returns failure - then take appropriate action
     */
    old_data = shash_replace(&bfd_tnl->bfd_status, bfd_enabled,
                             xstrdup(enabled));
    if (old_data) {
        free(old_data);
    }
}

static void
del_bfd_tunnel(struct tunnel *tnl)
{
    struct tunnel *bfd_tnl;

    bfd_tnl = shash_find_and_delete(bfd_tunnels, tnl->key);
    if (bfd_tnl) {
        VLOG_DBG("Delete bfd tunnel %s - %s\n", bfd_tnl->local_dst_ip,
                  bfd_tnl->remote_dst_ip);
        tunnel_destroy(bfd_tnl);
    }
}

static void
add_del_tunnels(void)
{
    const struct vteprec_tunnel *tunnel;
    struct tunnel *db_tnl;
    struct tunnel *kern_tnl;
    char *bfd;
    char *check_tnl_key;
    char *bfd_mac;
    struct log_switch *dbls;
    struct shash_node *node, *next;
    bool status;

    VLOG_DBG("sync: tunnels\n");

    tunnel = vteprec_tunnel_first(vtep_idl);
    while (tunnel) {
        db_tnl = tunnel_create(tunnel->local->encapsulation_type,
                               tunnel->local->dst_ip,
                               tunnel->remote->encapsulation_type,
                               tunnel->remote->dst_ip,
                               tunnel);
        shash_add_once(db_tunnels, db_tnl->key, db_tnl);

        kern_tnl = tunnel_lookup(kern_tunnels,
                                 db_tnl->local_type,
                                 db_tnl->local_dst_ip,
                                 db_tnl->remote_type,
                                 db_tnl->remote_dst_ip);
        if (kern_tnl) {
            if ((tunnel_compare_bfd_params(db_tnl, kern_tnl) == false)
                || (tunnel_compare_bfd_config_remote(db_tnl, kern_tnl)
                    == false)) {
                VLOG_INFO("mismatch between db and local tunnel %s-%s\n",
                           db_tnl->local_dst_ip, db_tnl->remote_dst_ip);
                db_tnl->flags |= HWVTEP_FLAGS_ADD;
            }
            else {
                db_tnl->flags |= HWVTEP_FLAGS_TOUCHED;
            }

            if (((status = tunnel_compare_bfd_status(db_tnl, kern_tnl)) == false)
                || (tunnel_compare_bfd_config_local(db_tnl, kern_tnl)
                    == false)) {
                kern_tnl->flags |= HWVTEP_FLAGS_ADD;
                if (status == false)
                    service_node_update = true;
            }
            else {
                kern_tnl->flags |= HWVTEP_FLAGS_TOUCHED;
            }
            kern_tnl->tnl_cfg = db_tnl->tnl_cfg;

            /* Absence of kern_tnl means no remote phy locator and so
             *  ctrl ls and bfd mac have to be removed if they are
             *  not shared across multiple tunnels
             */
            bfd = shash_find_data(&db_tnl->bfds, bfd_enable);
            if (bfd && !strcmp(bfd, "true")) {
                VLOG_DBG("bfd requested for tnl ip %s\n", db_tnl->remote_dst_ip);
                bfd_mac = shash_find_data(&db_tnl->bfd_config_local, bfd_dst_mac);
                if (!bfd_mac) {
                    bfd_mac = (char *)default_bfd_mac;
                }
                dbls = add_del_bfd_ctrl_ls(0);

                check_tnl_key = shash_find_data(&db_tnl->bfds, bfd_check_tnl_key);
                if (check_tnl_key && !strcmp(check_tnl_key, "true")) {
                    /* Add bfd mac only in log switch 0 */
                    VLOG_DBG("check_tnl_key: enabled\n");
                    if (dbls) {
                        add_del_bfd_local_mac(bfd_mac, dbls);
                    }
                }
                else {
                    /* bfd can be on any configured log switch, so
                     *  install bfd mac on all db log switch
                     */
                    VLOG_DBG("bfd dst mac: %s \n", bfd_mac);
                    SHASH_FOR_EACH_SAFE (node, next, db_log_switches) {
                        struct log_switch *dbls = node->data;

                        add_del_bfd_local_mac(bfd_mac, dbls);
                    }
                }
            }
        }
        tunnel = vteprec_tunnel_next(tunnel);
    }
}

static struct unicast_mac_addr *
ucast_mac_create(struct log_switch *ls, char *mac, char *ipaddr,
                 struct phy_locator *pl)
{
    struct unicast_mac_addr *new_mac;

    new_mac = xzalloc(sizeof(*new_mac));
    new_mac->mac = xstrdup(mac);
    new_mac->ls_name = xstrdup(ls->name);
    new_mac->key = xzalloc(strlen(new_mac->mac) + strlen(ls->dev_name) + 1);
    strcpy(new_mac->key, new_mac->mac);
    strcat(new_mac->key, ls->dev_name);
    new_mac->ip_addr = xstrdup(ipaddr);
    new_mac->loc = pl;
    return new_mac;
}

static struct unicast_mac_addr *
ucast_remote_mac_create_from_cfg(const struct vteprec_ucast_macs_remote *ucm)
{
    struct log_switch *ls;
    struct phy_locator *pl;
    struct unicast_mac_addr *mac;

    ls = log_switch_lookup(db_log_switches, ucm->logical_switch->name);
    if (!ls)
        return NULL;
    pl = phy_loc_lookup(db_phy_locs, ucm->locator->encapsulation_type,
                        ucm->locator->dst_ip);
    if (!pl)
        return NULL;

    mac = ucast_mac_create(ls, ucm->MAC, ucm->ipaddr, pl);
    mac->mac_cfg = (void *)ucm;
    return mac;
}

static struct unicast_mac_addr *
ucast_local_mac_create_from_cfg(const struct vteprec_ucast_macs_local *ucm)
{
    struct log_switch *ls;
    struct phy_locator *pl;
    struct unicast_mac_addr *mac;

    ls = log_switch_lookup(db_log_switches, ucm->logical_switch->name);
    if (!ls)
        return NULL;
    pl = phy_loc_lookup(db_phy_locs, ucm->locator->encapsulation_type,
                        ucm->locator->dst_ip);
    if (!pl)
        return NULL;

    mac = ucast_mac_create(ls, ucm->MAC, ucm->ipaddr, pl);
    mac->mac_cfg = (void *)ucm;
    return mac;
}

static struct unicast_mac_addr *
ucast_mac_lookup(struct shash *mac_tbl, char *mac, char *ls_dev)
{
    struct unicast_mac_addr *ucm = NULL;
    char *key;

    if (!mac_tbl || !mac || !ls_dev)
        return NULL;

    key = xzalloc(strlen(mac) + strlen(ls_dev) + 1);
    strcpy(key, mac);
    strcat(key, ls_dev);

    ucm = shash_find_data(mac_tbl, key);
    free(key);
    return ucm;
}

static void
ucast_mac_destroy(struct unicast_mac_addr *mac)
{
    if (!mac)
        return;
    if (mac->mac) free(mac->mac);
    if (mac->ls_name) free(mac->ls_name);
    if (mac->ip_addr) free(mac->ip_addr);
    if (mac->key) free(mac->key);
    free(mac);
}

static void
ucast_mac_sync(struct unicast_mac_addr *kmac, struct unicast_mac_addr *dbmac,
               struct log_switch *dbls)
{
    bool same_ls = true;

    if (strcmp(kmac->ls_name, dbls->dev_name))
        same_ls = false;

    /* Do not compare ipaddr field as it is ignored and not programmed in
     * the kernel
     */
    if (!same_ls ||
        strcmp(kmac->loc->key, dbmac->loc->key)) {

        VLOG_DBG("ucast mac sync: kern [%s %s %s] DB [%s %s %s]\n",
                  kmac->ls_name, kmac->ip_addr, kmac->loc->key,
                  dbmac->ls_name, dbmac->ip_addr, dbmac->loc->key);
        kmac->flags |= HWVTEP_FLAGS_DELETE;
        dbmac->flags |= HWVTEP_FLAGS_ADD;
        return;
    }

    kmac->flags |= HWVTEP_FLAGS_TOUCHED;
    dbmac->flags |= HWVTEP_FLAGS_TOUCHED;

    /* TBD: what is this? */
    if (kmac->flags & HWVTEP_FLAGS_ADD)
        dbmac->flags = HWVTEP_FLAGS_ADD;
}

static struct multicast_mac_addr *
mcast_mac_create(struct log_switch *ls, const char *mac, char *ipaddr,
                 struct shash *locator_set)
{
    struct multicast_mac_addr *new_mac;

    new_mac = xzalloc(sizeof(*new_mac));
    new_mac->mac = xstrdup(mac);
    new_mac->ls_name = xstrdup(ls->name);
    new_mac->key = xzalloc(strlen(new_mac->mac) + strlen(ls->dev_name) + 1);
    strcpy(new_mac->key, new_mac->mac);
    strcat(new_mac->key, ls->dev_name);
    new_mac->ip_addr = xstrdup(ipaddr);
    new_mac->locs = locator_set;
    return new_mac;
}

static struct multicast_mac_addr *
mcast_remote_mac_create_from_cfg(const struct vteprec_mcast_macs_remote *mcm)
{
    struct log_switch *ls;
    struct shash *locators;
    struct multicast_mac_addr *mac;

    locators = phy_loc_set_create(mcm->locator_set);
    if (!locators)
        return NULL;
    ls = log_switch_lookup(db_log_switches, mcm->logical_switch->name);
    if (!ls) {
        phy_loc_set_destroy(locators);
        return NULL;
    }
    mac = mcast_mac_create(ls, mcm->MAC, mcm->ipaddr, locators);
    mac->mac_cfg = (void *)mcm;
    return mac;
}

static struct multicast_mac_addr *
mcast_local_mac_create_from_cfg(const struct vteprec_mcast_macs_local *mcm)
{
    struct log_switch *ls;
    struct shash *locators;
    struct multicast_mac_addr *mac;

    locators = phy_loc_set_create(mcm->locator_set);
    if (!locators)
        return NULL;
    ls = log_switch_lookup(db_log_switches, mcm->logical_switch->name);
    if (!ls) {
        phy_loc_set_destroy(locators);
        return NULL;
    }
    mac = mcast_mac_create(ls, mcm->MAC, mcm->ipaddr, locators);
    mac->mac_cfg = (void *)mcm;
    return mac;
}

static struct multicast_mac_addr *
mcast_mac_lookup(struct shash *mac_tbl, const char *mac, char *ls_dev)
{
    struct multicast_mac_addr *mcm = NULL;
    char *key;

    if (!mac_tbl || !mac || !ls_dev)
        return NULL;

    key = xzalloc(strlen(mac) + strlen(ls_dev) + 1);
    strcpy(key, mac);
    strcat(key, ls_dev);
    mcm = shash_find_data(mac_tbl, key);
    free(key);

    return mcm;
}

static void
mcast_mac_destroy(struct multicast_mac_addr *mac)
{
    if (!mac)
        return;
    if (mac->mac) free(mac->mac);
    if (mac->ls_name) free(mac->ls_name);
    if (mac->ip_addr) free(mac->ip_addr);
    if (mac->key) free(mac->key);
    phy_loc_set_destroy(mac->locs);
    free(mac);
}

static bool
mcast_mac_sync(struct multicast_mac_addr *kmac, struct multicast_mac_addr *dbmac,
               struct log_switch *dbls, bool mcast_mac_remote)
{
    struct shash_node *node, *next;
    int olocs_count = 0, nlocs_count = 0;
    bool same_ls = true;
    bool found = false;

    if (strcmp(kmac->ls_name, dbls->dev_name))
        same_ls = false;

    /* Do not compare ipaddr field as it is ignored and not programmed in
     * the kernel
     */
    if (!same_ls ||
        strcmp(kmac->key, dbmac->key)) {
        kmac->flags |= HWVTEP_FLAGS_DELETE;
        dbmac->flags |= HWVTEP_FLAGS_ADD;
        return true;
    }

    kmac->flags |= HWVTEP_FLAGS_TOUCHED;
    dbmac->flags |= HWVTEP_FLAGS_TOUCHED;

    olocs_count = shash_count(kmac->locs);
    nlocs_count = shash_count(dbmac->locs);

    if (nlocs_count == 0) {
        dbmac->flags = HWVTEP_FLAGS_DELETE;
        kmac->flags = HWVTEP_FLAGS_DELETE;
        return true;
    }
    else if (olocs_count > nlocs_count) {
        dbmac->flags = HWVTEP_FLAGS_ADD;
        kmac->flags = HWVTEP_FLAGS_DELETE;
        return true;
    }

    /* check the phy locator set */
    SHASH_FOR_EACH_SAFE (node, next, dbmac->locs) {
        struct phy_locator *dbpl = node->data;
        struct phy_locator *kpl;
        char *tip = kern_local_ip_get();
        struct tunnel *tnl;

        if (mcast_mac_remote && (strcmp(unknown_dst_str, dbmac->mac) == 0)) {
            /* only one service node is configured per vxlan interface in
             * kernel. So create tunnel and phy locators for all other service
             * nodes based on ovsdb so that bfd sesssions can be established.
             * Also let sync to go through if the service node configured in
             * vxlan interface matches one of those in ovsdb.
             */
            kpl = phy_loc_lookup(kern_phy_locs, dbpl->type, dbpl->dst_ip);
            if (!kpl) {
                kpl = phy_loc_create(dbpl->type, dbpl->dst_ip);
                shash_add_once(kern_phy_locs, kpl->key, kpl);
            }
            if (!kpl->local) {
                if (shash_find_data(kmac->locs, dbpl->key))
                    found = true;
                else
                    shash_add_once(kmac->locs, kpl->key, kpl);

                if (tip) {
                    tnl = tunnel_lookup(kern_tunnels, dbpl->type, tip,
                                        dbpl->type, dbpl->dst_ip);
                    if (!tnl) {
                        tnl = tunnel_create(dbpl->type, tip, dbpl->type,
                                            dbpl->dst_ip, NULL);
                        shash_add_once(kern_tunnels, tnl->key, tnl);
                    }
                }
            }
        }

        if (!shash_find_data(kmac->locs, dbpl->key)) {
            dbmac->flags = HWVTEP_FLAGS_ADD;
            kmac->flags = HWVTEP_FLAGS_DELETE;
            return true;
        }
    }
    if (mcast_mac_remote && (strcmp(unknown_dst_str, dbmac->mac) == 0)
        && !found) {
        dbmac->flags = HWVTEP_FLAGS_ADD;
        kmac->flags = HWVTEP_FLAGS_DELETE;
        return true;
    }
    return false;
}

static void
add_del_remote_macs(void)
{
    const struct vteprec_ucast_macs_remote *uc_remote_mac;
    const struct vteprec_mcast_macs_remote *mc_remote_mac;
    struct shash_node *node, *next;

    VLOG_DBG("sync: remote macs\n");
    uc_remote_mac = vteprec_ucast_macs_remote_first(vtep_idl);
    while (uc_remote_mac) {
        struct unicast_mac_addr *nucm;
        struct unicast_mac_addr *oucm;
        struct log_switch *ls;

        if (!strcmp(uc_remote_mac->logical_switch->name, nvp_internal_str)){
            uc_remote_mac = vteprec_ucast_macs_remote_next(uc_remote_mac);
            continue;
        }
        nucm = ucast_remote_mac_create_from_cfg(uc_remote_mac);
        if (!nucm) {
            uc_remote_mac = vteprec_ucast_macs_remote_next(uc_remote_mac);
            continue;
        }
        if (!shash_add_once(db_ucast_remote_macs, nucm->key, nucm)) {
            ucast_mac_destroy(nucm);
            uc_remote_mac = vteprec_ucast_macs_remote_next(uc_remote_mac);
            continue;
        }
        ls = log_switch_lookup(db_log_switches,
                               uc_remote_mac->logical_switch->name);
        oucm = ucast_mac_lookup(kern_ucast_remote_macs,
                                uc_remote_mac->MAC, ls->dev_name);
        uc_remote_mac = vteprec_ucast_macs_remote_next(uc_remote_mac);

        if (!oucm)
            nucm->flags |= HWVTEP_FLAGS_ADD;
        else
            ucast_mac_sync(oucm, nucm, ls);
    }

    mc_remote_mac = vteprec_mcast_macs_remote_first(vtep_idl);
    while (mc_remote_mac) {
        bool is_unknown = false;
        bool update_ls = true;
        struct multicast_mac_addr *nmcm;
        struct multicast_mac_addr *omcm;
        struct log_switch *ls;

        if (!strcmp(mc_remote_mac->logical_switch->name, nvp_internal_str)){
            mc_remote_mac = vteprec_mcast_macs_remote_next(mc_remote_mac);
            continue;
        }
        nmcm = mcast_remote_mac_create_from_cfg(mc_remote_mac);
        if (!nmcm) {
            mc_remote_mac = vteprec_mcast_macs_remote_next(mc_remote_mac);
            continue;
        }
        if (!shash_add_once(db_mcast_remote_macs, nmcm->key, nmcm)) {
            mcast_mac_destroy(nmcm);
            mc_remote_mac = vteprec_mcast_macs_remote_next(mc_remote_mac);
            continue;
        }
        ls = log_switch_lookup(db_log_switches,
                               mc_remote_mac->logical_switch->name);
        omcm = mcast_mac_lookup(kern_mcast_remote_macs,
                                mc_remote_mac->MAC, ls->dev_name);
        mc_remote_mac = vteprec_mcast_macs_remote_next(mc_remote_mac);

        if (nmcm && !strcmp(unknown_dst_str, nmcm->mac))
            is_unknown = true;

        if (!omcm)
            nmcm->flags |= HWVTEP_FLAGS_ADD;
        else
            update_ls = mcast_mac_sync(omcm, nmcm, ls, true);

        /* when the unknown-dst entry is configured, we need to update the
         * kernel vxlan instance definition
         */
        if (is_unknown && update_ls) {
            struct log_switch *kls;
            ls->flags = HWVTEP_FLAGS_ADD;
            kls = log_switch_lookup(kern_log_switches, ls->dev_name);
            if (kls && !(kls->flags & HWVTEP_FLAGS_DELETE)) {
                //VLOG_DBG("remote svcnode update for %s\n", ls->name);
                if (shash_count(&ls->log_ports) != 0)
                    kls->flags = HWVTEP_FLAGS_TOUCHED;
                else
                    kls->flags = HWVTEP_FLAGS_DELETE;
            }
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_mcast_remote_macs) {
        struct multicast_mac_addr *mcm = node->data;

        if (strcmp(unknown_dst_str, mcm->mac) != 0)
            continue;

        /* deleting the unknown-dst remote mac, need to reprogram the vxlan */
        /* TBD: why is it not just deleting the vxlan ? */
        if (!(mcm->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED))) {
            struct log_switch *kls, *dbls;
            kls = log_switch_lookup(kern_log_switches, mcm->ls_name);
            dbls = log_switch_lookup(db_log_switches,
                                     shash_find_data(ls_dev_to_db_name_ht,
                                                     mcm->ls_name));
            if (dbls) {
                dbls->flags = HWVTEP_FLAGS_ADD;
                if (kls && !(kls->flags & HWVTEP_FLAGS_DELETE)) {
                    if (shash_count(&dbls->log_ports) != 0)
                        kls->flags = HWVTEP_FLAGS_TOUCHED;
                    else
                        kls->flags = HWVTEP_FLAGS_DELETE;
                }
            }
        }
    }
}

static void
vlan_binding_destroy(struct vlan_binding *vb)
{
    if (!vb) return;
    if (vb->name) free(vb->name);
    if (vb->ls_name) free(vb->ls_name);
    free(vb);
}

static void
phy_port_destroy(struct phy_port *pp)
{
    if (pp) {
        struct shash_node *vnode, *vnode_next;

        SHASH_FOR_EACH_SAFE (vnode, vnode_next, &pp->vlan_bindings) {
            struct vlan_binding *vb = vnode->data;
            shash_delete(&pp->vlan_bindings, vnode);
            vlan_binding_destroy(vb);
        }
        shash_destroy(&pp->vlan_bindings);
        //VLOG_DBG("Destroyed port: %s", pp->name);
        if (pp->name) free(pp->name);
        if (pp->description) free(pp->description);
        free(pp);
    }
}

static void
phy_switch_destroy(struct phy_switch *ps)
{
    if (ps) {
        struct shash_node *node, *next;

        shash_destroy_free_data(&ps->local_ips);
        shash_destroy_free_data(&ps->mgmt_ips);
        SHASH_FOR_EACH_SAFE (node, next, &ps->ports) {
            struct phy_port *pp = node->data;
            shash_delete(&ps->ports, node);
            phy_port_destroy(pp);
        }
        shash_destroy(&ps->ports);
        VLOG_DBG("Destroyed physical switch: %s", ps->name);
        if (ps->name) free(ps->name);
        if (ps->description) free(ps->description);
        free(ps);
    }
}

static char *
vb_name_get(char *name, int len, char *pp_name, int64_t vlan)
{
    snprintf(name, len, "port-%s-vlan-%"PRId64"", pp_name, vlan);
    return name;
}

static struct phy_switch *
phy_switch_create(const struct vteprec_physical_switch *ps_cfg)
{
    struct phy_switch *ps;
    int i, j;

    ps = xzalloc(sizeof *ps);

    ps->name = xstrdup(ps_cfg->name);
    ps->description = xstrdup(ps_cfg->description);
    ps->ps_cfg = ps_cfg;

    VLOG_DBG("Creating physical switch instance %s, [%s]\n",
              ps->name, ps->description);

    shash_init(&ps->local_ips);
    shash_init(&ps->mgmt_ips);
    shash_init(&ps->ports);

    for (i = 0; i < ps_cfg->n_management_ips; i++) {
        char *mip;
        mip = xstrdup(ps_cfg->management_ips[i]);
        if (!shash_add_once(&ps->mgmt_ips, mip, mip)) {
            VLOG_DBG("    mgmt ip: %s\n", mip);
            free(mip);
        }
    }
    for (i = 0; i < ps_cfg->n_tunnel_ips; i++) {
        char *tip;
        tip = xstrdup(ps_cfg->tunnel_ips[i]);
        if (!shash_add_once(&ps->local_ips, tip, tip)) {
            VLOG_DBG("    tunnel ip: %s\n", tip);
            free(tip);
        }
    }
    for (i = 0; i < ps_cfg->n_ports; i++) {
        struct vteprec_physical_port *p_cfg = ps_cfg->ports[i];
        struct phy_port *pp;

        if (shash_find_data(&ps->ports, p_cfg->name))
            continue;

        pp = xzalloc(sizeof *pp);
        pp->name = xstrdup(p_cfg->name);
        pp->description = xstrdup(p_cfg->description);
        if(!shash_add_once(&ps->ports, pp->name, pp)) {
            free(pp->name);
            free(pp->description);
            free(pp);
            continue;
        }
        VLOG_DBG("    port: %s\n", pp->name);

        shash_init(&pp->vlan_bindings);
        for (j = 0; j < p_cfg->n_vlan_bindings; j++) {
            char vb_name[HWVTEP_MAX_STR_LEN];
            struct vlan_binding *vb;

            vb_name_get(vb_name, HWVTEP_MAX_STR_LEN, pp->name,
                        p_cfg->key_vlan_bindings[j]);
            if (shash_find_data(&pp->vlan_bindings, vb_name))
                continue;

            vb = xzalloc(sizeof *vb);
            vb->vlan = p_cfg->key_vlan_bindings[j];
            vb->name = xstrdup(vb_name);
            vb->ls_name = xstrdup(p_cfg->value_vlan_bindings[j]->name);
            VLOG_DBG("Adding binding to physical port %s with %s\n",
                      pp->name, vb->name);
            if(!shash_add_once(&pp->vlan_bindings, vb->name, vb)){
                 free(vb->name);
                 free(vb->ls_name);
                 free(vb);
                 continue;
            }
            VLOG_DBG("        vlan: %"PRId64" - logical switch: %s\n",
                      vb->vlan, vb->ls_name);
        }
    }

    return ps;
}

static bool
phy_switch_sync(struct phy_switch *kps, struct phy_switch *dbps)
{
    struct shash_node *node, *next;

    VLOG_DBG("sync: phy switches\n");
    SHASH_FOR_EACH_SAFE (node, next, &dbps->ports) {
        struct phy_port *kpp, *dbpp = node->data;

        kpp = shash_find_data(&kps->ports, dbpp->name);
        if (!kpp) {
            dbpp->flags |= HWVTEP_FLAGS_DELETE;
            VLOG_DBG("no kernel phy switch\n");
            continue;
        }

        dbpp->is_bond = kpp->is_bond;
        dbpp->is_clag = kpp->is_clag;
        dbpp->flags |= HWVTEP_FLAGS_TOUCHED;
        kpp->flags |= HWVTEP_FLAGS_TOUCHED;
    }

    kps->flags |= HWVTEP_FLAGS_TOUCHED;
    dbps->flags |= HWVTEP_FLAGS_TOUCHED;

    shash_destroy_free_data(&kps->local_ips);
    shash_init(&kps->local_ips);
    SHASH_FOR_EACH_SAFE (node, next, &dbps->local_ips) {
        char *local_ip = node->data;
        shash_add_once(&kps->local_ips, local_ip, xstrdup(local_ip));
    }
    return true;
}

static void
add_del_phy_switches(void)
{
    const struct vteprec_global *vtep_global;
    int i;

    vtep_global = vteprec_global_first(vtep_idl);
    if (!vtep_global) {
        /* there is bigger problem with the db server when
         * this happens, do nothing here
         */
        return;
    }

    for (i = 0; i < vtep_global->n_switches; i++) {
        const struct vteprec_physical_switch *ps_cfg;
        struct phy_switch *dbps;
        struct phy_switch *kps;

        ps_cfg = vtep_global->switches[i];
        if (shash_find_data(db_phy_switches, ps_cfg->name))
            continue;

        kps = phy_switch_lookup(kern_phy_switches);
        if (!kps) {
            /* TBD: should not happen */
            VLOG_ERR("kernel phy switch not found!\n");
            continue;
        }
        dbps = phy_switch_create(ps_cfg);
        if(!shash_add_once(db_phy_switches, dbps->name, dbps)) {
            phy_switch_destroy(dbps);
            continue;
        }
        phy_switch_sync(kps, dbps);
    }
}

static void
add_del_local_macs(void)
{
    const struct vteprec_ucast_macs_local *um;
    const struct vteprec_mcast_macs_local *mm;

    VLOG_DBG("sync: local macs\n");
    um = vteprec_ucast_macs_local_first(vtep_idl);
    while (um) {
        struct unicast_mac_addr *db_mac;
        struct unicast_mac_addr *kern_mac;
        struct log_switch *ls;

        db_mac = ucast_local_mac_create_from_cfg(um);
        if (!db_mac) {
            um = vteprec_ucast_macs_local_next(um);
            continue;
        }

        if (!shash_add_once(db_ucast_local_macs, db_mac->key, db_mac)) {
            ucast_mac_destroy(db_mac);
            um = vteprec_ucast_macs_local_next(um);
            continue;
        }

        ls = log_switch_lookup(db_log_switches, um->logical_switch->name);
        kern_mac = ucast_mac_lookup(kern_ucast_local_macs, um->MAC, ls->dev_name);
        um = vteprec_ucast_macs_local_next(um);
        if (kern_mac) {
            kern_mac->mac_cfg = db_mac->mac_cfg;
        }
        if (!kern_mac || (kern_mac->flags & HWVTEP_FLAGS_DELETE)) {
            db_mac->flags |= HWVTEP_FLAGS_DELETE;
        }
        else {
            VLOG_DBG("mac sync");
            ucast_mac_sync(kern_mac, db_mac, ls);
        }
    }

    mm = vteprec_mcast_macs_local_first(vtep_idl);
    while (mm) {
        struct multicast_mac_addr *db_mac;
        struct multicast_mac_addr *kern_mac;
        struct log_switch *ls;

        db_mac = mcast_local_mac_create_from_cfg(mm);
        if (!db_mac) {
            mm = vteprec_mcast_macs_local_next(mm);
            continue;
        }
        if (!shash_add_once(db_mcast_local_macs, db_mac->key, db_mac)) {
            mcast_mac_destroy(db_mac);
            mm = vteprec_mcast_macs_local_next(mm);
            continue;
        }

        ls = log_switch_lookup(db_log_switches, mm->logical_switch->name);
        kern_mac = mcast_mac_lookup(kern_mcast_local_macs, mm->MAC, ls->dev_name);
        mm = vteprec_mcast_macs_local_next(mm);
        if (kern_mac)
            kern_mac->mac_cfg = db_mac->mac_cfg;

        if (!kern_mac || (kern_mac->flags & HWVTEP_FLAGS_DELETE))
            db_mac->flags |= HWVTEP_FLAGS_DELETE;
        else
            mcast_mac_sync(kern_mac, db_mac, ls, false);
    }
}

static struct unicast_mac_addr*
add_del_bfd_local_mac(char* bfd_mac, struct log_switch* ls)
{
    struct unicast_mac_addr *db_mac;
    struct unicast_mac_addr *kern_mac;
    struct phy_locator* pl;
    char *tip;

    db_mac = ucast_mac_lookup(db_ucast_bfd_local_macs, bfd_mac, ls->dev_name);
    if (!db_mac) {
        if (!(tip = kern_local_ip_get())) {
            VLOG_INFO("local ip missing, bfd_mac %s, LS %s create failed\n",
                      bfd_mac, ls->name);
            return NULL;
        }
        /* physical locator for local vtep */
        pl = phy_loc_lookup(db_phy_locs, vxlan_encap_type, tip);
        if (!pl) {
            VLOG_INFO("Cannot find phy loc, bfd_mac %s, LS %s create failed\n",
                      bfd_mac, ls->name);
            return NULL;
        }
        db_mac = ucast_mac_create(ls, bfd_mac, "", pl);
        if (!db_mac) {
            VLOG_ERR("bfd_mac %s, LS %s create failed!\n", bfd_mac, ls->name);
            return NULL;
        }
        if(!shash_add_once(db_ucast_bfd_local_macs, db_mac->key, db_mac)){
             ucast_mac_destroy(db_mac);
        }
    }
    kern_mac = ucast_mac_lookup(kern_ucast_bfd_local_macs, bfd_mac, ls->dev_name);
    if (!kern_mac) {
        db_mac->flags |= HWVTEP_FLAGS_ADD;
    }
    else {
        ucast_mac_sync(kern_mac, db_mac, ls);
    }
    return db_mac;
}

static void
db_ucast_local_mac_create(struct ovsdb_idl_txn *txn, char *mac,
                          char *ls_name, char *ipaddr, char *dst_ip)
{
    struct vteprec_ucast_macs_local *cfg;
    struct phy_locator *pl;
    struct log_switch *dbls;

    dbls = log_switch_lookup(db_log_switches,
                             shash_find_data(ls_dev_to_db_name_ht, ls_name));
    if (!dbls) {
        VLOG_INFO("Cannot find ls: %s\n", ls_name);
        return;
    }

    pl = phy_loc_lookup(db_phy_locs, vxlan_encap_type, dst_ip);
    if (!pl) {
        VLOG_INFO("Cannot find phy loc\n");
        return;
    }

    VLOG_DBG("db ADD ucast local mac: %s ls: %s dst: %s",
              mac, ls_name, dst_ip);

    if ((pl->flags & HWVTEP_FLAGS_ADD) || !pl->pl_cfg) {
        struct vteprec_physical_locator *pl_cfg;

        VLOG_DBG("db ADD phy locator: %s", dst_ip);
        pl_cfg = vteprec_physical_locator_insert(txn);
        vteprec_physical_locator_set_encapsulation_type(pl_cfg,
                                                              vxlan_encap_type);
        vteprec_physical_locator_set_dst_ip(pl_cfg, dst_ip);
        pl->pl_cfg = pl_cfg;
        pl->flags = 0;
    }

    cfg = vteprec_ucast_macs_local_insert(txn);
    vteprec_ucast_macs_local_set_MAC(cfg, mac);
    vteprec_ucast_macs_local_set_ipaddr(cfg, ipaddr);
    vteprec_ucast_macs_local_set_locator(cfg, pl->pl_cfg);
    vteprec_ucast_macs_local_set_logical_switch(cfg, dbls->ls_cfg);

    return;
}

static void
db_mcast_local_mac_create(struct ovsdb_idl_txn *txn, const char *mac,
                          char *ls_name, char *ipaddr,
                          int num_dst, char **dst_ips)
{
    struct vteprec_physical_locator_set *pls_cfg;
    struct vteprec_physical_locator **locs;
    struct vteprec_mcast_macs_local *mm_cfg;
    struct phy_locator *pl;
    struct log_switch *dbls;
    int i;

    dbls = log_switch_lookup(db_log_switches,
                             shash_find_data(ls_dev_to_db_name_ht, ls_name));
    if (!dbls)
        return;

    /* add the unknown-dst */
    VLOG_DBG("db ADD mcast local mac: %s ls %s", mac, ls_name);

    locs = xzalloc(sizeof(struct vteprec_physical_locator *) * num_dst);
    pls_cfg = vteprec_physical_locator_set_insert(txn);
    for (i = 0; i < num_dst; i++) {
        struct vteprec_physical_locator *pl_cfg;
        pl = phy_loc_lookup(db_phy_locs, vxlan_encap_type, dst_ips[i]);
        if (!pl) {
            VLOG_ERR("Cannot find phy locator: %s\n", dst_ips[i]);
            continue;
        }

        if (!pl->pl_cfg) {
            VLOG_DBG("db ADD phy locator: %s", dst_ips[i]);
            pl_cfg = vteprec_physical_locator_insert(txn);

            vteprec_physical_locator_set_encapsulation_type(pl_cfg,
                                                             vxlan_encap_type);
            vteprec_physical_locator_set_dst_ip(pl_cfg, dst_ips[i]);
            pl->pl_cfg = pl_cfg;
        }
        locs[i] = (struct vteprec_physical_locator *)pl->pl_cfg;
    }
    vteprec_physical_locator_set_set_locators(pls_cfg, locs, num_dst);
    mm_cfg = vteprec_mcast_macs_local_insert(txn);
    vteprec_mcast_macs_local_set_MAC(mm_cfg, mac);
    vteprec_mcast_macs_local_set_ipaddr(mm_cfg, ipaddr);
    vteprec_mcast_macs_local_set_locator_set(mm_cfg, pls_cfg);
    vteprec_mcast_macs_local_set_logical_switch(mm_cfg, dbls->ls_cfg);
    free(locs);

    return;
}

static void
update_db_local_macs(void)
{
    struct ovsdb_idl_txn *txn;
    enum ovsdb_idl_txn_status status;
    struct shash_node *node, *next;
    txn = ovsdb_idl_txn_create(vtep_idl);
    VLOG_DBG("db update: local macs - ucast %d mcast %d\n",
              (int)shash_count(kern_ucast_local_macs),
              (int)shash_count(kern_mcast_local_macs));
    SHASH_FOR_EACH_SAFE (node, next, kern_ucast_local_macs) {
        struct unicast_mac_addr *mac = node->data;

        VLOG_DBG("mac %s flags %x\n", mac->mac, mac->flags);
        if (mac->flags & HWVTEP_FLAGS_DELETE) {
            /* delete from the db and the hash */
            if (mac->mac_cfg) {
                struct unicast_mac_addr *dbmac;
                VLOG_DBG("db DEL local mac %s %s", mac->mac, mac->ls_name);
                vteprec_ucast_macs_local_delete(mac->mac_cfg);

                dbmac = shash_find_and_delete(db_ucast_local_macs, mac->key);
                if (dbmac)
                    ucast_mac_destroy(dbmac);

                shash_delete(kern_ucast_local_macs, node);
                ucast_mac_destroy(mac);
            }
        } else if (!(mac->flags & HWVTEP_FLAGS_TOUCHED)) {
            db_ucast_local_mac_create(txn, mac->mac, mac->ls_name, mac->ip_addr,
                                      mac->loc->dst_ip);
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, db_ucast_local_macs) {
        struct unicast_mac_addr *mac = node->data;

        if (mac->flags & HWVTEP_FLAGS_DELETE) {
            /* delete from the db */
            if (mac->mac_cfg) {
                VLOG_DBG("db DEL local mac %s %s", mac->mac, mac->ls_name);
                vteprec_ucast_macs_local_delete(mac->mac_cfg);
                shash_delete(db_ucast_local_macs, node);
                ucast_mac_destroy(mac);
            }
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_mcast_local_macs) {
        struct multicast_mac_addr *mac = node->data;

        if (mac->flags & HWVTEP_FLAGS_DELETE) {
            /* delete from the db and the hash */
            if (mac->mac_cfg) {
                struct multicast_mac_addr *dbmac;
                VLOG_DBG("db DEL local mac %s %s", mac->mac, mac->ls_name);
                vteprec_mcast_macs_local_delete(mac->mac_cfg);
                dbmac = shash_find_and_delete(db_mcast_local_macs, mac->key);
                if (dbmac)
                    mcast_mac_destroy(dbmac);

                shash_delete(kern_mcast_local_macs, node);
                mcast_mac_destroy(mac);
            }
        } else if (!(mac->flags & HWVTEP_FLAGS_TOUCHED)) {
            struct shash_node *lnode, *lnext;
            char **dst_ips;
            int i = 0;

            dst_ips = xzalloc(shash_count(mac->locs) * sizeof(char *));
            SHASH_FOR_EACH_SAFE (lnode, lnext, mac->locs) {
                struct phy_locator *pl = lnode->data;
                dst_ips[i++] = pl->dst_ip;
            }
            db_mcast_local_mac_create(txn, mac->mac, mac->ls_name, mac->ip_addr,
                                      shash_count(mac->locs), dst_ips);
            free(dst_ips);
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, db_mcast_local_macs) {
        struct multicast_mac_addr *mac = node->data;

        if (mac->flags & HWVTEP_FLAGS_DELETE) {
            /* delete from the db */
            if (mac->mac_cfg) {
                VLOG_DBG("db dEL local mac %s %s", mac->mac, mac->ls_name);
                vteprec_mcast_macs_local_delete(mac->mac_cfg);
                shash_delete(db_mcast_local_macs, node);
                mcast_mac_destroy(mac);
            }
        }
    }

    status = ovsdb_idl_txn_commit_block(txn);
    if ((status != TXN_SUCCESS) && (status != TXN_UNCHANGED)) {
        VLOG_ERR("local mcast macs commit status %d\n", status);
    }
    ovsdb_idl_txn_destroy(txn);
}

static void
update_db_phy_switch(void)
{
    struct ovsdb_idl_txn *txn;
    enum ovsdb_idl_txn_status status;
    const struct vteprec_global *global = NULL;
    struct vteprec_physical_switch *ps_cfg = NULL;
    struct phy_switch *kps, *dbps;
    struct shash_node *node;
    const char **ips;
    struct vteprec_physical_port **ports;
    int count, i;
    struct shash *port_ht;

    VLOG_DBG("db update: phy switch\n");
    kps = kern_phy_switch_lookup();
    if (!kps) {
        VLOG_ERR("Cannot find kernel phy switch\n");
        return;
    }
    dbps = db_phy_switch_lookup();

    txn = ovsdb_idl_txn_create(vtep_idl);
    global = vteprec_global_first(vtep_idl);
    if (!global)
        global = vteprec_global_insert(txn);

    if (global->n_switches == 0) {
        ps_cfg = vteprec_physical_switch_insert(txn);
    } else {
        ps_cfg = global->switches[0];
    }

    if (dbps && dbps->name)
        vteprec_physical_switch_set_name(ps_cfg, dbps->name);
    else
        vteprec_physical_switch_set_name(ps_cfg, kps->name);

    if (dbps && dbps->description)
        vteprec_physical_switch_set_description(ps_cfg, dbps->description);
    else
        vteprec_physical_switch_set_description(ps_cfg, kps->name);


    /* set the tunnel ips */
    count = shash_count(&kps->local_ips);
    if (count) {
        ips = xzalloc(sizeof(char *) * count);
        i = 0;
        SHASH_FOR_EACH (node, &kps->local_ips) {
            char *tip = node->data;
            ips[i++] = tip;
        }
        vteprec_physical_switch_set_tunnel_ips(ps_cfg, ips, count);
        free(ips);
    }

    /* set the ports */
    port_ht = xzalloc(sizeof(*port_ht));
    shash_init(port_ht);
    for (i = 0; i < ps_cfg->n_ports; i++) {
        struct vteprec_physical_port *pport = ps_cfg->ports[i];
        shash_add_once(port_ht, pport->name, pport);
    }

    count = shash_count(&kps->ports) + ps_cfg->n_ports;
    ports = xzalloc(sizeof(struct vteprec_physical_port *) * count);

    i = 0;
    SHASH_FOR_EACH (node, &kps->ports) {
        struct phy_port *kpp = node->data;
        struct vteprec_physical_port *pport;

        pport = shash_find_data(port_ht, kpp->name);
        if (!pport) {
            pport = vteprec_physical_port_insert(txn);
            vteprec_physical_port_set_name(pport, kpp->name);
            vteprec_physical_port_set_description(pport, kpp->name);
        }
        ports[i++] = pport;

        /* bindings are preserved if already exists in db, otherwise,
         * there is no bindings anyways. For binding delete, the stats
         * will be deleted during stats update.
         */
    }
    if (clag_enabled) {
        /* need one more pass through the db ports to make sure we don't
         * throw away ports from the clag peer
         */
        SHASH_FOR_EACH (node, port_ht) {
            struct vteprec_physical_port *dbpp = node->data;
            struct phy_port *kpp;

            kpp = shash_find_data(&kps->ports, dbpp->name);
            if (!kpp) {
                if (!strstr(dbpp->name, kps->name)) {
                    char n2[HWVTEP_MAX_STR_LEN];

                    sprintf(n2, "%s-%s", kps->name, dbpp->name);
                    if (!shash_find_data(&kps->ports, n2)) {
                        if (!shash_find_data(kern_clag_b2c, dbpp->name))
                            ports[i++] = dbpp;
                    }
                }
            }
        }
    }
    vteprec_physical_switch_set_ports(ps_cfg, ports, i);
    vteprec_global_set_switches(global, &ps_cfg, 1);
    status = ovsdb_idl_txn_commit_block(txn);

    VLOG_DBG("phy switch commit status: %d\n", status);
    ovsdb_idl_txn_destroy(txn);
    shash_destroy(port_ht);
    free(port_ht);
    free(ports);
}

static void
update_db_binding_stats(void)
{
    struct ovsdb_idl_txn *txn;
    enum ovsdb_idl_txn_status status;
    struct phy_switch *dbps;
    const struct vteprec_global *global = NULL;
    struct vteprec_physical_switch *ps_cfg = NULL;
    int i, j;

    VLOG_DBG("db update: binding stats\n");
    dbps = db_phy_switch_lookup();
    if (!dbps)
        return;

    global = vteprec_global_first(vtep_idl);
    for (i = 0; i < global->n_switches; i++) {
        ps_cfg = global->switches[i];
        if (!strcmp(ps_cfg->name, dbps->name)) {
            break;
        }
        ps_cfg = NULL;
    }
    if (!ps_cfg) {
        VLOG_ERR("Cannot find ovsdb cfg for physical switch");
        return;
    }

    txn = ovsdb_idl_txn_create(vtep_idl);
    for (i = 0; i < ps_cfg->n_ports; i++) {
        struct vteprec_physical_port *p_cfg;
        struct vteprec_logical_binding_stats **stats_cfg;
        struct phy_port *pp;
        struct shash stats_ht;

        p_cfg = ps_cfg->ports[i];
        pp = shash_find_data(&dbps->ports, p_cfg->name);
        if (!pp) {
            VLOG_ERR("Cannot find port %s", p_cfg->name);
            continue;
        }

        shash_init(&stats_ht);
        for (j = 0; j < p_cfg->n_vlan_stats; j++) {
            char vb_name[HWVTEP_MAX_STR_LEN];

            vb_name_get(vb_name, HWVTEP_MAX_STR_LEN, pp->name,
                        p_cfg->key_vlan_stats[j]);
            shash_add(&stats_ht, vb_name, p_cfg->value_vlan_stats[j]);
        }

        stats_cfg = xzalloc(sizeof(struct vteprec_logical_binding_stats *) * p_cfg->n_vlan_bindings);
        for (j = 0; j < p_cfg->n_vlan_bindings; j++) {
            char vb_name[HWVTEP_MAX_STR_LEN];
            struct vlan_binding *vb;

            vb_name_get(vb_name, HWVTEP_MAX_STR_LEN, pp->name,
                        p_cfg->key_vlan_bindings[j]);
            stats_cfg[j] = shash_find_data(&stats_ht, vb_name);
            if (!stats_cfg[j])
                stats_cfg[j] = vteprec_logical_binding_stats_insert(txn);

            if (!(vb = shash_find_data(&pp->vlan_bindings, vb_name)))
                continue;

            vteprec_logical_binding_stats_set_bytes_from_local(
                stats_cfg[j], vb->stats.bytes_from_local);
            vteprec_logical_binding_stats_set_bytes_to_local(
                stats_cfg[j], vb->stats.bytes_to_local);
            vteprec_logical_binding_stats_set_packets_from_local(
                stats_cfg[j], vb->stats.packets_from_local);
            vteprec_logical_binding_stats_set_packets_to_local(
                stats_cfg[j], vb->stats.packets_to_local);
        }
        vteprec_physical_port_set_vlan_stats(p_cfg,
                                                   p_cfg->key_vlan_bindings,
                                                   stats_cfg,
                                                   p_cfg->n_vlan_bindings);
        free(stats_cfg);
        shash_destroy(&stats_ht);
    }
    status = ovsdb_idl_txn_commit_block(txn);
    if ((status != TXN_SUCCESS) && (status != TXN_UNCHANGED)) {
        VLOG_ERR("binding stats commit status %d\n", status);
    }
    ovsdb_idl_txn_destroy(txn);
}

static void
tunnel_bfd_config_local_to_vteprec(struct tunnel* tnl,
                                   struct smap *bfd_config_local)
{
    const char *dst_mac;
    const char *dst_ip;

    dst_mac = shash_find_data(&tnl->bfd_config_local, bfd_dst_mac);
    if (!dst_mac)
        dst_mac = default_bfd_mac;
    smap_add(bfd_config_local,
                 (char *)bfd_dst_mac,
                 (char *)dst_mac);
    dst_ip = shash_find_data(&tnl->bfd_config_local, bfd_dst_ip);
    if (!dst_ip)
        dst_ip = default_bfd_local_dst_ip;
    smap_add(bfd_config_local,
                 (char *)bfd_dst_ip,
                 (char *)dst_ip);

    /* controller requires default values of confg local to be
     *  written in the db explicitly to enable bfd in the service
     *  node.
    dst_mac = shash_find_data(&tnl->bfd_config_local, bfd_dst_mac);
    if (!dst_mac)
        dst_mac = default_bfd_mac;
    key_bfd_config_local[*n_bfd_config_local] = (char *)bfd_dst_mac;
    value_bfd_config_local[*n_bfd_config_local] = (char *)dst_mac;
    (*n_bfd_config_local)++;

    dst_ip = shash_find_data(&tnl->bfd_config_local, bfd_dst_ip);
    if (!dst_ip)
        dst_ip = default_bfd_local_dst_ip;
    key_bfd_config_local[*n_bfd_config_local] = (char *)bfd_dst_ip;
    value_bfd_config_local[*n_bfd_config_local] = (char *)dst_ip;
    (*n_bfd_config_local)++;*/
}

static void
tunnel_bfd_status_to_vteprec(struct tunnel* tnl,
                             struct smap *bfd_status)
{
    char *enabled;
    char *state;
    char *forwarding;
    char *diagnostic;
    char *remote_state;
    char *rem_diagnostic;
    char *info;

    // bfd_status: enabled
    enabled = shash_find_data(&tnl->bfd_status, bfd_enabled);
    if (enabled) {
        smap_add(bfd_status,
                 (char *)bfd_enabled,
                 enabled);
    }

    //bfd_status: state
    state = shash_find_data(&tnl->bfd_status, bfd_state);
    if (state) {
        smap_add(bfd_status,
                 (char *)bfd_state,
                 state);
    }

    //bfd_status: forwarding
    forwarding = shash_find_data(&tnl->bfd_status, bfd_forwarding);
    if (forwarding) {
        smap_add(bfd_status,
                 (char *)bfd_forwarding,
                 forwarding);
    }

    //bfd_status: diagnostic
    diagnostic = shash_find_data(&tnl->bfd_status, bfd_diag);
    if (diagnostic) {
        smap_add(bfd_status,
                 (char *)bfd_diag,
                 diagnostic);
    }

    //bfd_status: remote_state
    remote_state = shash_find_data(&tnl->bfd_status, bfd_remote_state);
    if (remote_state) {
        smap_add(bfd_status,
                 (char *)bfd_remote_state,
                 remote_state);
    }

    //bfd_status: remote_diagnostic
    rem_diagnostic = shash_find_data(&tnl->bfd_status,
                                          bfd_remote_diag);
    if (rem_diagnostic) {
        smap_add(bfd_status,
                 (char *)bfd_remote_diag,
                 remote_state);
    }

    // bfd_status: info
    info = shash_find_data(&tnl->bfd_status, bfd_info);
    if (info) {
        smap_add(bfd_status,
                 (char *)bfd_info,
                 info);
    }
}

static const struct vteprec_tunnel*
db_tunnel_create_or_update(struct tunnel* tnl, struct ovsdb_idl_txn *txn)
{
    const struct vteprec_tunnel *tnl_cfg;
    struct phy_locator *local, *remote;
    struct smap bfd_config_local;
    struct smap bfd_status;
    smap_init(&bfd_config_local);
    smap_init(&bfd_status);

    tnl_cfg = tnl->tnl_cfg;
    if (!tnl_cfg) {
        local = phy_loc_lookup(db_phy_locs, tnl->local_type, tnl->local_dst_ip);
        remote = phy_loc_lookup(db_phy_locs, tnl->remote_type,
                                tnl->remote_dst_ip);
        if (!local || !remote) {
            VLOG_DBG("missing local/remote db phy locator\n");
            return NULL;
        }
        if (!local->pl_cfg) {
            VLOG_DBG("local ip not updated in db phy locator\n");
            return NULL;
        }
        if (!remote->pl_cfg) {
            VLOG_ERR("missing db phy locator entry for remote %s\n",
                     remote->dst_ip);
            return NULL;
        }
        tnl_cfg = vteprec_tunnel_insert(txn);
        if (!tnl_cfg) {
            VLOG_ERR("DB Tunnel insert failed %s - %s\n",
                     tnl->local_dst_ip, tnl->remote_dst_ip);
            return NULL;
        }
        tnl->tnl_cfg = tnl_cfg;
        vteprec_tunnel_set_local(tnl_cfg, local->pl_cfg);
        vteprec_tunnel_set_remote(tnl_cfg, remote->pl_cfg);
    }

    tunnel_bfd_config_local_to_vteprec(tnl, &bfd_config_local);

    tunnel_bfd_status_to_vteprec(tnl,
                                 &bfd_status);

    vteprec_tunnel_set_bfd_config_local(tnl_cfg,
                                        &bfd_config_local);

    vteprec_tunnel_set_bfd_status(tnl_cfg,
                                  &bfd_status);
    return (tnl_cfg);
}

static void
update_db_tunnels(void)
{
    struct ovsdb_idl_txn *txn;
    struct shash_node *node, *next;
    enum ovsdb_idl_txn_status status;
    const struct vteprec_tunnel **tunnels = NULL;
    int n_tunnels = shash_count(kern_tunnels);
    struct phy_switch *ps;
    int i = 0;
    bool update = false;

    if (n_tunnels) {
        tunnels = (const struct vteprec_tunnel **)xzalloc
                      (n_tunnels * sizeof(const struct vteprec_tunnel **));
    }

    txn = ovsdb_idl_txn_create(vtep_idl);
    VLOG_DBG("db update: tunnels -  %d \n",
              (int)shash_count(kern_tunnels));

    // Delete
    SHASH_FOR_EACH_SAFE (node, next, db_tunnels) {
        struct tunnel *db_tnl = node->data;

        if (!(db_tnl->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED))
            && db_tnl->tnl_cfg) {
            VLOG_DBG("del tnl %s - %s from db\n", db_tnl->local_dst_ip,
                     db_tnl->remote_dst_ip);
            vteprec_tunnel_delete(db_tnl->tnl_cfg);
            shash_delete(db_tunnels, node);
            tunnel_destroy(db_tnl);
            update = true;
        }
    }
    // Update or Add
    SHASH_FOR_EACH_SAFE (node, next, kern_tunnels) {
        struct tunnel *kern_tnl = node->data;

        if (((kern_tnl->flags & HWVTEP_FLAGS_ADD) && (kern_tnl->tnl_cfg))
            || (!(kern_tnl->flags
                  & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_DELETE)))) {

           VLOG_DBG("insert/update tnl %s - %s in db\n", kern_tnl->local_dst_ip,
                    kern_tnl->remote_dst_ip);
           db_tunnel_create_or_update(kern_tnl, txn);
           update = true;
        }
        if (tunnels && kern_tnl->tnl_cfg && !(kern_tnl->flags & HWVTEP_FLAGS_DELETE))
            tunnels[i++] = kern_tnl->tnl_cfg;
    }
    ps = db_phy_switch_lookup();
    if (ps && ps->ps_cfg && update) {
        VLOG_DBG("update phy switch with %d tunnels\n", i);
        vteprec_physical_switch_set_tunnels(ps->ps_cfg,
                            (struct vteprec_tunnel **)tunnels, i);
    }
    if (tunnels) {
        free(tunnels);
    }
    status = ovsdb_idl_txn_commit_block(txn);

    if ((status != TXN_SUCCESS) && (status != TXN_UNCHANGED)) {
        VLOG_ERR("tunnel commit status %d\n", status);
    }
    ovsdb_idl_txn_destroy(txn);
}

static void
update_db_config(bool sync_local_macs, bool sync_remote_macs,
                 bool sync_stats, bool has_logical_config)
{
    struct timeval tv1, tv2, tv;
    gettimeofday(&tv, 0);
    update_db_phy_switch();
    if (has_logical_config) {
        if (sync_local_macs)
            update_db_local_macs();
        if (sync_stats)
            update_db_binding_stats();
        if (sync_remote_macs && db_active)
            update_db_tunnels();
    }
    gettimeofday(&tv1, 0);
    time_elapsed(tv, tv1, &tv2);
}

static char*
get_service_node_addr(char *ls_dev_name)
{
    struct multicast_mac_addr *mc;
    struct shash_node *node, *next;
    int num = 0;
    uint32_t hash;
    char *tip = kern_local_ip_get();
    char *addrs[HWVTEP_MAX_SVC_NODE];

    mc = mcast_mac_lookup(db_mcast_remote_macs, unknown_dst_str, ls_dev_name);
    if (!mc)
        return NULL;

    if (!tip) {
        return NULL;
    }

    SHASH_FOR_EACH_SAFE (node, next, mc->locs) {
        struct phy_locator *pl = node->data;
        struct tunnel *tnl;
        char *enabled, *state;

        tnl = tunnel_lookup(kern_tunnels, pl->type, tip,
                            pl->type, pl->dst_ip);
        if (tnl) {
            /* if bfd is enabled and state is not down, then select
               the service node
             */
            enabled = shash_find_data(&tnl->bfd_status, bfd_enabled);
            state = shash_find_data(&tnl->bfd_status, bfd_state);
            if (enabled && (strcmp(enabled, "true") == 0)) {
                if (state && (strcasecmp(state, "down") == 0)) {
                    continue;
                }
            }
        }
        addrs[num++] = pl->dst_ip;
    }
    if (num) {
        hash = hash_string(ls_dev_name, 0);
        return (addrs[hash % num]);
    }
    return NULL;
}

static void
cleanup_after_sync(void)
{
    struct shash_node *node, *next;

    VLOG_DBG("Sync done, cleanup\n");
    SHASH_FOR_EACH_SAFE (node, next, kern_ucast_remote_macs) {
        struct unicast_mac_addr *ucm = node->data;
        shash_delete(kern_ucast_remote_macs, node);
        ucast_mac_destroy(ucm);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_mcast_remote_macs) {
        struct multicast_mac_addr *mcm = node->data;
        shash_delete(kern_mcast_remote_macs, node);
        mcast_mac_destroy(mcm);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_ucast_local_macs) {
        struct unicast_mac_addr *ucm = node->data;
        shash_delete(kern_ucast_local_macs, node);
        ucast_mac_destroy(ucm);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_mcast_local_macs) {
        struct multicast_mac_addr *mcm = node->data;
        shash_delete(kern_mcast_local_macs, node);
        mcast_mac_destroy(mcm);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_tunnels) {
        struct tunnel *tnl = node->data;
        shash_delete(kern_tunnels, node);
        tunnel_destroy(tnl);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_ucast_bfd_local_macs) {
        struct unicast_mac_addr *ucm = node->data;
        shash_delete(kern_ucast_bfd_local_macs, node);
        ucast_mac_destroy(ucm);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_phy_locs) {
        struct phy_locator *pl = node->data;
        shash_delete(kern_phy_locs, node);
        phy_loc_destroy(pl);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_log_switches) {
        struct log_switch *ls = node->data;
        shash_delete(kern_log_switches, node);
        log_switch_destroy(ls);
    }

    /*SHASH_FOR_EACH_SAFE (node, next, kern_log_routers) {
        struct logical_router *lr = node->data;
        shash_delete(kern_log_routers, node);
        log_router_destroy(lr);
    }*/

    SHASH_FOR_EACH_SAFE (node, next, kern_phy_switches) {
        struct phy_switch *ps = node->data;
        shash_delete(kern_phy_switches, node);
        phy_switch_destroy(ps);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_clag_c2b) {
        shash_delete(kern_clag_c2b, node);
    }

    SHASH_FOR_EACH_SAFE (node, next, kern_clag_b2c) {
        shash_delete(kern_clag_b2c, node);
    }

    /* DB */
    SHASH_FOR_EACH_SAFE (node, next, db_ucast_remote_macs) {
        struct unicast_mac_addr *ucm = node->data;
        shash_delete(db_ucast_remote_macs, node);
        ucast_mac_destroy(ucm);
    }

    SHASH_FOR_EACH_SAFE (node, next, db_mcast_remote_macs) {
        struct multicast_mac_addr *mcm = node->data;
        shash_delete(db_mcast_remote_macs, node);
        mcast_mac_destroy(mcm);
    }

    SHASH_FOR_EACH_SAFE (node, next, db_ucast_local_macs) {
        struct unicast_mac_addr *ucm = node->data;
        shash_delete(db_ucast_local_macs, node);
        ucast_mac_destroy(ucm);
    }

    SHASH_FOR_EACH_SAFE (node, next, db_mcast_local_macs) {
        struct multicast_mac_addr *mcm = node->data;
        shash_delete(db_mcast_local_macs, node);
        mcast_mac_destroy(mcm);
    }

    SHASH_FOR_EACH_SAFE (node, next, db_tunnels) {
        struct tunnel *tnl = node->data;
        shash_delete(db_tunnels, node);
        tunnel_destroy(tnl);
    }

    SHASH_FOR_EACH_SAFE (node, next, db_ucast_bfd_local_macs) {
        struct unicast_mac_addr *ucm = node->data;
        shash_delete(db_ucast_bfd_local_macs, node);
        ucast_mac_destroy(ucm);
    }

    SHASH_FOR_EACH_SAFE (node, next, db_phy_locs) {
        struct phy_locator *pl = node->data;
        shash_delete(db_phy_locs, node);
        phy_loc_destroy(pl);
    }

    SHASH_FOR_EACH_SAFE (node, next, db_log_switches) {
        struct log_switch *ls = node->data;
        shash_delete(db_log_switches, node);
        log_switch_destroy(ls);
    }

    /*SHASH_FOR_EACH_SAFE (node, next, db_log_routers) {
        struct logical_router *lr = node->data;
        shash_delete(db_log_routers, node);
        log_router_destroy(lr);
    }*/

    SHASH_FOR_EACH_SAFE (node, next, db_phy_switches) {
        struct phy_switch *ps = node->data;
        shash_delete(db_phy_switches, node);
        phy_switch_destroy(ps);
    }

    SHASH_FOR_EACH_SAFE (node, next, port_vlans_ht) {
        port_vlan_info_t *pvinfo = node->data;
        if (pvinfo) {
            bitmap_free(pvinfo->vlan_bmp);
            free(pvinfo);
        }
    }

    shash_init(port_vlans_ht);

    shash_destroy_free_data(links_ht);
    shash_init(links_ht);
    shash_destroy_free_data(del_links_ht);
    shash_init(del_links_ht);
    unlink("/tmp/link.tmp");

    shash_destroy_free_data(vlan_to_ls_dev_ht);
    shash_init(vlan_to_ls_dev_ht);

    shash_destroy_free_data(ls_dev_to_db_name_ht);
    shash_init(ls_dev_to_db_name_ht);
    service_node_update = false;
}

/*static void
add_source_node_config(struct log_switch *ls) {

    struct multicast_mac_addr *mc;
    struct shash_node *node, *next;
    char *key;
    char *mac = "00:00:00:00:00:00";

    // adding the source node info
    exec_cmd("ip link add %s type vxlan id %"PRId64" local"
             " %s %s dstport 4789",
             ls->dev_name, ls->tunnel_key, ls->local_ip,
             vtep_mac_learn_en?"learning":"nolearning");

    mc = mcast_mac_lookup(db_mcast_remote_macs, unknown_dst_str,
                          ls->dev_name);
    if (!mc)
        VLOG_DBG("no unknown_dst mac defined yet for %s", ls->name);
    else {
        SHASH_FOR_EACH_SAFE (node, next, mc->locs) {
            struct phy_locator *pl = node->data;
            key = xzalloc(strlen(ls->dev_name) + strlen(pl->dst_ip));
            strcpy(key, ls->dev_name);
            strcat(key, pl->dst_ip);
            if (!shash_find(ls_pl_pair, key)) {
                exec_cmd("bridge fdb append %s dev %s "
                         "dst %s self", mac,
                         ls->dev_name, pl->dst_ip);
                shash_add(ls_pl_pair, key, xstrdup(key));
            }
            free(key);
        }
    }
}

static void
remove_source_node_config(struct log_switch *ls) {

    struct multicast_mac_addr *mc;
    struct shash_node *node, *next;
    char *key;
    char *mac = "00:00:00:00:00:00";

    mc = mcast_mac_lookup(db_mcast_remote_macs, unknown_dst_str,
                          ls->dev_name);
    if (!mc)
        VLOG_DBG("no unknown_dst mac defined yet for %s", ls->name);
    else {
        VLOG_DBG("Removing source nod info mac found");
        SHASH_FOR_EACH_SAFE (node, next, mc->locs) {
            struct phy_locator *pl = node->data;

            key = xzalloc(strlen(ls->dev_name) + strlen(pl->dst_ip));
            strcpy(key, ls->dev_name);
            strcat(key, pl->dst_ip);
            if (shash_find(ls_pl_pair, key)) {
                exec_cmd("bridge fdb  del %s dev %s "
                         "dst %s self static", mac,
                         ls->dev_name, pl->dst_ip);
                VLOG_INFO("Removing source nod info mac found br fdb del");
                shash_find_and_delete(ls_pl_pair, key);
            }
            free(key);
        }
    }
}
*/

static struct log_switch *get_kernel_log_switch_by_vlan(int vlan)
{
    char vstr[HWVTEP_MAX_STR_LEN];
    char *name;

    snprintf(vstr, HWVTEP_MAX_STR_LEN, "%d", vlan);
    name = shash_find_data(vlan_to_ls_dev_ht, vstr);
    if (name)
        return log_switch_lookup(kern_log_switches, name);
    return NULL;
}

static void
kernel_del_remote_mac(struct unicast_mac_addr *mac, struct log_switch *ls)
{
    char vlan_str[HWVTEP_MAX_STR_LEN] = "";

/*
    if (vtep_vlan_aware)
        snprintf(vlan_str, HWVTEP_MAX_STR_LEN, "vlan %"PRId64"", ls->vlan);
*/

    exec_cmd("bridge fdb del %s dev %s %s self master static",
             mac->mac, ls->dev_name, vlan_str);
}

static void
kernel_del_bfd_local_mac(struct unicast_mac_addr *mac, struct log_switch *ls)
{
    char name[HWVTEP_MAX_STR_LEN];

    if (vtep_vlan_aware && strcmp(ls->dev_name, "vxln0")) {
        exec_cmd("bridge fdb del dummy-vx vlan %"PRId64" master", ls->vlan);
        exec_cmd("bridge vlan del vid %"PRId64" dev dummy-vx", ls->vlan);
    } else {
        char *m;
        snprintf(name, HWVTEP_MAX_STR_LEN, "dummy%"PRId64"", ls->tunnel_key);
        exec_cmd("ip link del %s address %s", name, mac->mac);
        m = shash_find_and_delete(links_ht, name);
        if (m)
            free(m);
    }
}

static void
kernel_del_log_switch(struct log_switch *ls)
{
    char *m, name[HWVTEP_MAX_STR_LEN];
    char key_vb_ls[20];
    char *peerlink = get_clag_vxlan_peerlink_name();
    FILE *fp = NULL;
    char *line = NULL;
    char *save_ptr;
    size_t lsize = HWVTEP_MAX_STR_LEN - 1;
    char *dn = NULL, *vn = NULL;
    int64_t vlan = 0;

    fp = fopen("/tmp/bridge_vlan.tmp", "r");
    if (!fp) {
        VLOG_DBG("Failed to open bridge_vlan file\n");
    }
    else {
		while (getline(&line, &lsize, fp) != -1) {
			if (strstr(line, "vlan ids") || !strstr(line, "vxln")) {
				free(line);
				line = NULL;
				continue;
			}
			if (strstr(line, ls->dev_name)) {
				dn = strtok_r(line, "\t", &save_ptr);
				vn = strtok_r(NULL, " ", &save_ptr);
				if (vn)
					sscanf(vn, "%"PRId64"", &vlan);
			}
            free(line);
            line = NULL;
        }
        if (line) free(line);
        line = NULL;
        fclose(fp);
    }

    exec_cmd("ip link del %s", ls->dev_name);
    m = shash_find_and_delete(links_ht, ls->dev_name);
    if (m)
        free(m);

    if (vtep_vlan_aware) {
        if (clag_enabled && peerlink && bitmap_is_set(vlan_bitmap, ls->vlan)) {
            if (vlan)
                 ls->vlan = vlan;
            exec_cmd("bridge vlan del vid %"PRId64" dev %s",
                     ls->vlan, peerlink);
        }
        if (vlan)
            ls->vlan = vlan;
        if (bitmap_is_set(vlan_bitmap, ls->vlan)) {
            if (vlan)
                 ls->vlan = vlan;
            exec_cmd("bridge vlan del vid %"PRId64" dev %s", 
                     ls->vlan, ls->dev_name);
            exec_cmd("bridge vlan del vid %"PRId64" dev dummy-vx", ls->vlan);
            bitmap_set(vlan_bitmap, ls->vlan, 0);
            sprintf(key_vb_ls, "%s.%"PRId64"", ls->dev_name, ls->vlan);
            m = shash_find_and_delete(vb_ls_pair, key_vb_ls);
            if (m)
                free(m);
        }
        return;
    }

    snprintf(name, HWVTEP_MAX_STR_LEN, "br-%s", ls->dev_name);
    exec_cmd("ip link set br-%s down", ls->dev_name);
    exec_cmd("ip link del %s", name);
    m = shash_find_and_delete(links_ht, name);
    if (m)
        free(m);
}

static void
kernel_del_vlan_binding(struct phy_port *pp,
                        struct phy_switch *ps, struct vlan_binding *vb)
{
    char iface_str[HWVTEP_MAX_STR_LEN];
    char *real_name = pp->name;
    char key_vb_rn[20];

    if (pp->is_clag) {
        real_name = shash_find_data(kern_clag_c2b, pp->name);
        if (!real_name) {
            VLOG_ERR("Cannot find real name for clag %s\n", pp->name);
            return;
        }
    } else if (clag_enabled) {
        if (strstr(pp->name, ps->name))
            real_name = real_name + strlen(ps->name) + 1;
        else {
            VLOG_DBG("%s not my port\n", pp->name);
            return;
        }
    }

    if (vtep_vlan_aware) {
        char *m;
        exec_cmd("bridge vlan del vid %"PRId64" dev %s", vb->vlan, real_name);
        sprintf(key_vb_rn, "%s.%"PRId64"", real_name, vb->vlan);
        m = shash_find_and_delete(vb_rn_pair, key_vb_rn);
        if (m)
            free(m);
    } else {
        char *m;

        if (vb->vlan != 0) {
            sprintf(iface_str, "%s.%"PRId64"", real_name, vb->vlan);
        } else {
            sprintf(iface_str, "%s", real_name);
        }

        exec_cmd("ip link set dev %s nomaster", iface_str);
        m = shash_replace(links_ht, real_name, xstrdup("nomaster"));
        if (m)
            free(m);

        if (vb->vlan != 0) {
            char *m;
            exec_cmd("ip link del link %s dev %s type vlan",
                     real_name, iface_str);
            m = shash_find_and_delete(links_ht, iface_str);
            if (m)
                free(m);
        }
    }
}

static void
kernel_del_prefix_binding(struct switch_prefix_binding *spb,
                          struct log_switch *ls)
{
    if (strchr(spb->prefix, '.')) {
        exec_cmd("ip addr del %s dev br-%s", spb->prefix, ls->dev_name);
    } else {
        exec_cmd("ip -6 addr del %s dev br-%s", spb->prefix, ls->dev_name);
    }
}

static void
kernel_del_static_route(struct static_route *srt)
{
    if (strchr(srt->prefix, '.')) {
        exec_cmd("ip route del %s nexthop %s", srt->prefix, srt->nh);
    } else {
        exec_cmd("ip -6 route del %s nexthop %s", srt->prefix, srt->nh);
    }
}

static void
kernel_update_log_switch(struct log_switch *ls)
{
    char cmd[HWVTEP_MAX_STR_LEN];
    char *service_node_addr;
    char sn_str[HWVTEP_MAX_STR_LEN];
    char bridge_name[HWVTEP_MAX_STR_LEN];

    memset(sn_str, 0, HWVTEP_MAX_STR_LEN);
    service_node_addr = get_service_node_addr(ls->dev_name);
    if (!service_node_addr) {
        VLOG_DBG("no service node defined yet for %s", ls->name);
        snprintf(sn_str, HWVTEP_MAX_STR_LEN, "noremote");
    }
    else {
        snprintf(cmd, HWVTEP_MAX_STR_LEN, "remote %s ", service_node_addr);
        strcat(sn_str, cmd);
    }

    snprintf(cmd, HWVTEP_MAX_STR_LEN,
             "link set %s type vxlan local %s %s %s\n",
             ls->dev_name, ls->local_ip, sn_str,
             vtep_mac_learn_en?"learning":"nolearning");
    fprintf(batch_fp, "%s", cmd);
    VLOG_DBG(">> batch: %s", cmd);

    if (!vtep_vlan_aware || !strcmp(ls->dev_name, "vxln0")) {
        snprintf(bridge_name, HWVTEP_MAX_STR_LEN, "br-%s", ls->dev_name);
    }
    else if(vtep_vlan_aware && user_bridge_name_provided) {
        strncpy(bridge_name, user_bridge_name, strlen(user_bridge_name));
    }
    else if (vtep_vlan_aware && !user_bridge_name_provided) {
        strcpy(bridge_name, "br-vxlan");
    }

    fprintf(batch_fp, "link set dev %s master %s up\n",
            ls->dev_name, bridge_name);

}

static void
kernel_add_log_switch(struct log_switch *ls)
{
    char *service_node_addr;
    char sn_str[HWVTEP_MAX_STR_LEN];
    char bridge_name[HWVTEP_MAX_STR_LEN];
    bool vlan_aware;
    char *key;

    memset(sn_str, 0, HWVTEP_MAX_STR_LEN);
    service_node_addr = get_service_node_addr(ls->dev_name);
    if (!service_node_addr) {
        VLOG_DBG("no service node defined yet for %s", ls->name);
    } else {
        snprintf(sn_str, HWVTEP_MAX_STR_LEN, "remote %s ", service_node_addr);
    }

    vlan_aware = vtep_vlan_aware && strcmp(ls->dev_name, "vxln0");

    if (!vlan_aware) {
         snprintf(bridge_name, HWVTEP_MAX_STR_LEN, "br-%s", ls->dev_name);
    }
    else if(vtep_vlan_aware && user_bridge_name_provided) {
        strncpy(bridge_name, user_bridge_name, strlen(user_bridge_name));
    }
    else if (vtep_vlan_aware && !user_bridge_name_provided) {
        strcpy(bridge_name, "br-vxlan");
    }

    if (!find_kernel_link(bridge_name)) {
        exec_cmd("ip link add %s type bridge", bridge_name);
        shash_add_once(links_ht, bridge_name, xstrdup("nomaster"));
    }

    if (service_node_addr) {
        key = xzalloc(strlen(ls->dev_name) + strlen(service_node_addr));
        strcpy(key, ls->dev_name);
        strcat(key, service_node_addr);
        if (!shash_find(ls_pl_pair, key)) {
            exec_cmd("ip link add %s type vxlan id %"PRId64" local"
                     " %s %s %s dstport 4789",
                     ls->dev_name, ls->tunnel_key, ls->local_ip, sn_str,
                     vtep_mac_learn_en?"learning":"nolearning");

            shash_add(ls_pl_pair, key, xstrdup(key));
            exec_cmd("ip link set dev %s master %s %s up",
                     ls->dev_name, bridge_name, clag_enabled?"protodown on":"");

            shash_add_once(links_ht, ls->dev_name, xstrdup(bridge_name));

            exec_cmd("ip link set dev %s up type bridge %s", bridge_name,
                                             vlan_aware?"vlan_filtering 1":"");
            exec_cmd("bridge link set dev %s learning %s",
                     ls->dev_name, vtep_mac_learn_en?"on":"off");
        }
        /* case where the ls was deleted and readded */
        else if(!shash_find(links_ht, ls->dev_name)) {
            exec_cmd("ip link add %s type vxlan id %"PRId64" local"
                     " %s %s %s dstport 4789",
                     ls->dev_name, ls->tunnel_key, ls->local_ip, sn_str,
                     vtep_mac_learn_en?"learning":"nolearning");
 
            exec_cmd("ip link set dev %s master %s %s up",
                     ls->dev_name, bridge_name, clag_enabled?"protodown on":"");

            shash_add_once(links_ht, ls->dev_name, xstrdup(bridge_name));

            exec_cmd("ip link set dev %s up type bridge %s", bridge_name,
                                             vlan_aware?"vlan_filtering 1":"");
            exec_cmd("bridge link set dev %s learning %s",
                     ls->dev_name, vtep_mac_learn_en?"on":"off");
        }
        free(key);
    }
    else {
        exec_cmd("ip link add %s type vxlan id %"PRId64" local"
                 " %s %s %s dstport 4789",
                 ls->dev_name, ls->tunnel_key, ls->local_ip, sn_str,
                 vtep_mac_learn_en?"learning":"nolearning");

        exec_cmd("ip link set dev %s master %s %s up",
                 ls->dev_name, bridge_name, clag_enabled?"protodown on":"");

        shash_add_once(links_ht, ls->dev_name, xstrdup(bridge_name));

        exec_cmd("ip link set dev %s up type bridge %s", bridge_name,
                                         vlan_aware?"vlan_filtering 1":"");
        exec_cmd("bridge link set dev %s learning %s",
                 ls->dev_name, vtep_mac_learn_en?"on":"off");
    }
}

static void
kernel_add_remote_ucast_mac(struct unicast_mac_addr *ucm, struct log_switch *ls)
{
    char vlan_str[HWVTEP_MAX_STR_LEN] = "";

    if (vtep_vlan_aware) {
        if (ls->vlan == 0) {
            ls->vlan = 1;
        }
        snprintf(vlan_str, HWVTEP_MAX_STR_LEN, "vlan %"PRId64"", ls->vlan);
    }

    /* TBD: why do we expect failure here? */
    if (exec_cmd("bridge fdb replace %s dev %s %s dst %s self master static sticky",
                 ucm->mac, ls->dev_name, vlan_str, ucm->loc->dst_ip) != 0)
        ucm->flags |= HWVTEP_FLAGS_ADD;
}

/* we do not have the support for kernel and iproute
 * TODO: need to enable this when we have the support
*/
/*static void
kernel_add_remote_mcast_mac(struct multicast_mac_addr *mcm,
                            struct log_switch *ls)
{
    if (!(mcm->flags & HWVTEP_FLAGS_TOUCHED)) {
        exec_cmd("bridge fdb replace %s dev %s self", mcm->mac, mcm->ls_name);
    }

}*/

static void
kernel_add_bfd_local_mac(struct unicast_mac_addr *ucm, struct log_switch *ls)
{
    if (vtep_vlan_aware && strcmp(ls->dev_name, "vxln0")) {
        port_vlan_info_t *pvinfo;

        if (!find_kernel_link("dummy-vx")) {
            exec_cmd("ip link add dummy-vx address %s type dummy", ucm->mac);
            exec_cmd("ip link set dev dummy-vx master br-vxlan up");
            exec_cmd("bridge vlan del vid 1 dev dummy-vx");
            shash_add_once(links_ht, "dummy-vx", xstrdup("br-vxlan"));
        }
        pvinfo = shash_find_data(port_vlans_ht, "dummy-vx");
        if (pvinfo && pvinfo->vlan_bmp) {
            if (ls->vlan && !bitmap_is_set(pvinfo->vlan_bmp, ls->vlan)) {
                bitmap_set(pvinfo->vlan_bmp, ls->vlan, 1);
                exec_cmd("bridge vlan add vid %d dev dummy-vx", ls->vlan);
                exec_cmd("bridge fdb append %s dev dummy-vx vlan %"PRId64" master static sticky",
                         ucm->mac, ls->vlan);
            }
        }
    } else {
        char dev_name[HWVTEP_MAX_STR_LEN];

        snprintf(dev_name, HWVTEP_MAX_STR_LEN,"dummy%"PRId64"", ls->tunnel_key);
        exec_cmd("ip link add %s address %s type dummy", dev_name, ucm->mac);
        exec_cmd("ip link set dev %s master br-%s up", dev_name, ls->dev_name);
        shash_add_once(links_ht, dev_name, xstrdup(ls->dev_name));
    }
}

static void
kernel_add_prefix_binding(struct switch_prefix_binding *spb,
                          struct log_switch *ls)
{
    if (strchr(spb->prefix, '.')) {
        exec_cmd("ip addr add %s dev br-%s", spb->prefix, ls->dev_name);
    } else {
        exec_cmd("ip -6 addr add %s dev br-%s", spb->prefix, ls->dev_name);
    }
}

static void
kernel_add_static_route(struct static_route *srt)
{
    if (strchr(srt->prefix, '.')) {
        exec_cmd("ip route add %s nexthop %s", srt->prefix, srt->nh);
    } else {
        exec_cmd("ip -6 route add %s nexthop %s", srt->prefix, srt->nh);
    }
}

static void
kernel_add_vlan_binding(struct vlan_binding *vb, struct phy_port *pp,
                        struct phy_switch *ps, struct log_switch *ls)
{
    char iface_str[20];
    char *real_name = pp->name;
    char *peerlink = get_clag_vxlan_peerlink_name();
    char iface_str_peerlink[20];
    char key_vb_rn[20];
    char key_vb_ls[20];

    if (pp->is_clag) {
        real_name = shash_find_data(kern_clag_c2b, pp->name);
        if (!real_name) {
            VLOG_ERR("Cannot find real name for clag %s\n", pp->name);
            return;
        }
    } else if (clag_enabled) {
        if (strstr(pp->name, ps->name))
            real_name = real_name + strlen(ps->name) + 1;
        else {
            VLOG_DBG("%s not my port\n", pp->name);
            return;
        }
    }

    if (vtep_vlan_aware) {
        char *m = find_kernel_link(real_name);

        if (m && strcmp(m, "nomaster") == 0) {
            exec_cmd("ip link set dev %s master br-vxlan up", real_name);
            exec_cmd("bridge vlan del vid 1 dev %s", real_name);
            m = shash_replace(links_ht, real_name, xstrdup("br-vxlan"));
            if (m)
                free(m);
        }

        if (vb->vlan != 0) {
            sprintf(key_vb_rn, "%s.%"PRId64"", real_name, vb->vlan);
            if (!shash_find(vb_rn_pair, key_vb_rn)) {
                exec_cmd("bridge vlan add vid %"PRId64" dev %s",
                         vb->vlan, real_name);
                shash_add(vb_rn_pair, key_vb_rn, xstrdup(key_vb_rn));
            }
            if (clag_enabled && peerlink && !bitmap_is_set(vlan_bitmap, vb->vlan)) {
                bitmap_set(vlan_bitmap, vb->vlan, 1);
                exec_cmd("bridge vlan add vid %"PRId64" dev %s",
                         vb->vlan, peerlink);
            }
            /* add the vtep as well */
            sprintf(key_vb_ls, "%s.%"PRId64"", ls->dev_name, vb->vlan);
            if (!shash_find(vb_ls_pair, key_vb_ls)) {
                exec_cmd("bridge vlan add vid %"PRId64" pvid untagged dev %s",
                         vb->vlan, ls->dev_name);
                if (vb->vlan != 1)
                    exec_cmd("bridge vlan del vid 1 dev %s", ls->dev_name);
                bitmap_set(vlan_bitmap, vb->vlan, 1);
                shash_add(vb_ls_pair, key_vb_ls, xstrdup(key_vb_ls));
            }
        } else {
            if (ls->vlan != 0) {
                exec_cmd("bridge vlan add vid %"PRId64" pvid untagged dev %s",
                         ls->vlan, real_name);
            } else {
                exec_cmd("bridge vlan add vid %"PRId64" pvid untagged dev %s",
                         1, real_name);
            }
        }
    } else {
        if (vb->vlan != 0) {
            sprintf(iface_str, "%s.%"PRId64"", real_name, vb->vlan);
            if(peerlink) {
                sprintf(iface_str_peerlink, "%s.%"PRId64"", peerlink, vb->vlan);
            }
        } else {
            sprintf(iface_str, "%s", real_name);
            if(peerlink) {
                sprintf(iface_str_peerlink, "%s", peerlink);
            }
        }

        if (vb->vlan != 0) {
            exec_cmd("ip link add link %s dev %s type vlan id %"PRId64"",
                     real_name, iface_str, vb->vlan);

            exec_cmd("ip link set dev %s up", iface_str);

            if(peerlink) {
                 exec_cmd("ip link add link %s dev %s type vlan id %"PRId64"",
                          peerlink, iface_str_peerlink, vb->vlan);

                 exec_cmd("ip link set dev %s up", iface_str_peerlink);
            }
        }

        exec_cmd("ip link set dev %s master br-%s up", iface_str, ls->dev_name);
        shash_add_once(links_ht, iface_str, xstrdup(ls->dev_name));

    }
}

static void
compose_stop_bfd_session_msg(struct tunnel *tnl, char *ptm_msg, int *len)
{
    int64_t tunnel_key;
    char dev_name[HWVTEP_MAX_STR_LEN];
    void *ctxt;
    char vnid_buf[32];

    /* if is sent as empty string as session was started with
     * random vnid
     */
    tunnel_key = 0;
    log_switch_dev_name_get(tunnel_key, HWVTEP_MAX_STR_LEN, dev_name);
    sprintf(vnid_buf, "%"PRId64"", tunnel_key);
    /* create a stop-bfd-sess cmd */
    ptm_lib_init_msg(ptm_hdl, 0, PTMLIB_MSG_TYPE_CMD, NULL, &ctxt);
    ptm_lib_append_msg(ptm_hdl, ctxt, "cmd", "stop-bfd-sess");
    ptm_lib_append_msg(ptm_hdl, ctxt, "client", "vtepd");
    ptm_lib_append_msg(ptm_hdl, ctxt, "srcIPaddr", tnl->local_dst_ip);
    ptm_lib_append_msg(ptm_hdl, ctxt, "dstIPaddr", tnl->remote_dst_ip);
    ptm_lib_append_msg(ptm_hdl, ctxt, "ifName", dev_name);
    ptm_lib_append_msg(ptm_hdl, ctxt, "vnid", vnid_buf);

    ptm_lib_complete_msg(ptm_hdl, ctxt, ptm_msg, len);
}

static void
compose_start_bfd_session_msg(struct tunnel *tnl, char *ptm_msg, int *len)
{
    char *min_rx, *min_tx, *decay_min_rx;
    char *forwarding_if_rx, *cpath_down, *check_tnl_key;
    char *local_bfd_dst_mac, *local_bfd_dst_ip;
    char *remote_bfd_dst_mac, *remote_bfd_dst_ip;
    int64_t tunnel_key;
    char dev_name[HWVTEP_MAX_STR_LEN];
    int f_if_rx = 0, c_tnl_key = 0, c_dn = 0;
    char seqid_buf[32], vnid_buf[32];
    char fwd_buf[32], cdn_buf[32], ctnl_key_buf[32];
    void *ctxt;

    min_rx = shash_find_data(&tnl->bfds, bfd_min_rx);
    if (!min_rx)
        min_rx = default_min_rx;

    min_tx = shash_find_data(&tnl->bfds, bfd_min_tx);
    if (!min_tx)
        min_tx = default_min_tx;

    decay_min_rx = shash_find_data(&tnl->bfds, bfd_decay_min_rx);
    if (!decay_min_rx)
        decay_min_rx = default_decay_min_rx;

    forwarding_if_rx = shash_find_data(&tnl->bfds, bfd_fwd_if_rx);
    if (!forwarding_if_rx)
        forwarding_if_rx = default_forwarding_if_rx;
    if (strcmp(forwarding_if_rx, "true") == 0) {
        f_if_rx = 1;
    }

    cpath_down = shash_find_data(&tnl->bfds, bfd_cpath_down);
    if (!cpath_down)
        cpath_down = default_cpath_down;
    if (strcmp(cpath_down, "true") == 0) {
        c_dn = 1;
    }

    check_tnl_key = shash_find_data(&tnl->bfds, bfd_check_tnl_key);
    if (!check_tnl_key)
        check_tnl_key = default_check_tnl_key;
    if (strcmp(check_tnl_key, "true") == 0) {
        c_tnl_key = 1;
    }

    local_bfd_dst_mac = shash_find_data(&tnl->bfd_config_local, bfd_dst_mac);
    if (!local_bfd_dst_mac)
        local_bfd_dst_mac = default_bfd_mac;

    local_bfd_dst_ip = shash_find_data(&tnl->bfd_config_local, bfd_dst_ip);
    if (!local_bfd_dst_ip)
        local_bfd_dst_ip = default_bfd_local_dst_ip;


    remote_bfd_dst_mac = shash_find_data(&tnl->bfd_config_local, bfd_dst_mac);
    if (!remote_bfd_dst_mac)
        remote_bfd_dst_mac = default_bfd_mac;

    remote_bfd_dst_ip = shash_find_data(&tnl->bfd_config_remote, bfd_dst_ip);
    if (!remote_bfd_dst_ip)
        remote_bfd_dst_ip = default_bfd_remote_dst_ip;

    /* set tunnel_key to zero for transmit */
    tunnel_key = 0;
    log_switch_dev_name_get(tunnel_key, HWVTEP_MAX_STR_LEN, dev_name);

    sprintf(seqid_buf, "%d", getpid());
    sprintf(vnid_buf, "%"PRId64"", tunnel_key);
    sprintf(fwd_buf, "%d", f_if_rx);
    sprintf(cdn_buf, "%d", c_dn);
    sprintf(ctnl_key_buf, "%d", c_tnl_key);

    /* create a start-bfd-sess cmd */
    ptm_lib_init_msg(ptm_hdl, 0, PTMLIB_MSG_TYPE_CMD, NULL, &ctxt);
    ptm_lib_append_msg(ptm_hdl, ctxt, "cmd", "start-bfd-sess");
    ptm_lib_append_msg(ptm_hdl, ctxt, "client", "vtepd");
    ptm_lib_append_msg(ptm_hdl, ctxt, "srcIPaddr", tnl->local_dst_ip);
    ptm_lib_append_msg(ptm_hdl, ctxt, "dstIPaddr", tnl->remote_dst_ip);
    ptm_lib_append_msg(ptm_hdl, ctxt, "ifName", dev_name);
    ptm_lib_append_msg(ptm_hdl, ctxt, "vnid", vnid_buf);
    ptm_lib_append_msg(ptm_hdl, ctxt, "seqid", seqid_buf);
    ptm_lib_append_msg(ptm_hdl, ctxt, "local_dst_mac", local_bfd_dst_mac);
    ptm_lib_append_msg(ptm_hdl, ctxt, "local_dst_ip", local_bfd_dst_ip);
    ptm_lib_append_msg(ptm_hdl, ctxt, "remote_dst_mac", remote_bfd_dst_mac);
    ptm_lib_append_msg(ptm_hdl, ctxt, "remote_dst_ip", remote_bfd_dst_ip);
    ptm_lib_append_msg(ptm_hdl, ctxt, "min_rx", min_rx);
    ptm_lib_append_msg(ptm_hdl, ctxt, "min_tx", min_tx);
    ptm_lib_append_msg(ptm_hdl, ctxt, "decay_min_rx", decay_min_rx);
    ptm_lib_append_msg(ptm_hdl, ctxt, "forwarding_if_rx", fwd_buf);
    ptm_lib_append_msg(ptm_hdl, ctxt, "cpath_down", cdn_buf);
    ptm_lib_append_msg(ptm_hdl, ctxt, "check_tnl_key", ctnl_key_buf);

    ptm_lib_complete_msg(ptm_hdl, ctxt, ptm_msg, len);

}

static bool
send_ptm_msg(char *ptm_msg, int len)
{
    int nbytes;

    if (ptm_sock == -1) {
        VLOG_INFO("Unable to send ptm msg: \n%s\n", ptm_msg);
        return false;
    }

    if (len >= PTMLIB_MSG_SZ) {
        VLOG_INFO("ptm msg exceeds max size: \n%s\n", ptm_msg);
        return false;
    }

    nbytes = send(ptm_sock, ptm_msg, len, MSG_DONTWAIT);
    if (nbytes <= 0) {
        if ((nbytes < 0) && (errno != EWOULDBLOCK) && (errno != EAGAIN)) {
            VLOG_WARN("PTM socket failure : %s\n", strerror(errno));
        }
        VLOG_INFO("Failed to send msg: \n%s\n", ptm_msg);
        close(ptm_sock);
        ptm_sock = -1;
        return false;
    }
    VLOG_INFO("Sent PTM msg: \n%s\n", ptm_msg);
    return true;
}

static void update_all_bfd_sessions(bool db_active)
{
    struct shash_node *node, *next;
    struct shash remote_tbl;

    VLOG_DBG("Update all bfd sessions: %d\n", db_active);
    shash_init(&remote_tbl);
    SHASH_FOR_EACH_SAFE (node, next, db_tunnels) {
        struct tunnel *tnl = node->data;
        char *enabled;
        int len;
        enabled = shash_find_data(&tnl->bfd_status, bfd_enabled);
        if (enabled && !strcmp(enabled, "true")) {
            len = PTMLIB_MSG_SZ;
            if (db_active) {
                VLOG_DBG("Starting bfd session\n");
                compose_start_bfd_session_msg(tnl, ptm_msg_buf, &len);
            } else {
                VLOG_DBG("Stopping bfd session\n");
                compose_stop_bfd_session_msg(tnl, ptm_msg_buf, &len);
            }
            VLOG_DBG("ADD %s for bfd\n", tnl->remote_dst_ip);
            shash_add_once(&remote_tbl, tnl->remote_dst_ip, tnl);
            send_ptm_msg(ptm_msg_buf, len);
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, &remote_tbl) {
        struct tunnel *tnl = node->data;
        update_bfd_redirect(tnl->remote_dst_ip, db_active);
    }
    shash_destroy(&remote_tbl);
}

/*static void
change_from_source_to_service_node_config(struct log_switch *ls) {
    char *service_node_addr;
    char *mac = "00:00:00:00:00:00";
    char *key;

    VLOG_DBG("Converting source node config to service node %s", ls->dev_name);
    service_node_addr = get_service_node_addr(ls->dev_name);
    key = xzalloc(strlen(ls->dev_name) + strlen(service_node_addr));
    strcpy(key, ls->dev_name);
    strcat(key, service_node_addr);
    exec_cmd("bridge fdb append %s dev %s "
              "dst %s self", mac,
              ls->dev_name, service_node_addr);
    VLOG_DBG("Ran bridge fdb append on %s", service_node_addr);
    shash_add(ls_pl_pair, key, xstrdup(key));
    free(key);
}
*/

static void
update_kernel_config(bool sync_remote_macs)
{
    struct phy_switch *kps, *dbps;
    struct shash_node *node, *next, *pnode, *pnode_next;
    struct timeval tv1, tv2, tv3, tv4, tv5, tv6;

    kps = phy_switch_lookup(kern_phy_switches);
    dbps = phy_switch_lookup(db_phy_switches);
    char *peerlink = get_clag_vxlan_peerlink_name();

    VLOG_DBG("===== Update Kernel\n");
    gettimeofday(&tv1, 0);

    /* DELETE */

    if (sync_remote_macs) {
        VLOG_DBG("sync_remote_macs");
        /*  - walk the kernel ucast mac table and do deletions - */
        SHASH_FOR_EACH_SAFE (node, next, kern_ucast_remote_macs) {
            bool delete = false;
            struct unicast_mac_addr *ucm = node->data;
            struct log_switch *ls;

            ls = log_switch_lookup(kern_log_switches, ucm->ls_name);
            delete = !(ucm->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED));
            delete = delete || (ls && (ls->flags & HWVTEP_FLAGS_DELETE));
            if (delete)
                kernel_del_remote_mac(ucm, ls);
        }


        /* - walk the kernel mcast mac table and do deletions - */
        SHASH_FOR_EACH_SAFE (node, next, kern_mcast_remote_macs) {
            bool delete = false;
            struct multicast_mac_addr *mcm = node->data;
            struct log_switch *ls;

            ls = log_switch_lookup(kern_log_switches, mcm->ls_name);
            delete = !(mcm->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED));
            delete = delete || (ls && (ls->flags & HWVTEP_FLAGS_DELETE));
            if (delete) {
                 /* TBD: there is actually no action currently. If this is the
                  *      unknown-dst, it would have triggered an LS reprogram
                  *      to update the svcnodes
                  */
                if (strcmp(unknown_dst_str, mcm->mac) != 0) {
                    mcm->flags |= HWVTEP_FLAGS_DELETE;
                }
            }
        }

        /* walk the old kernel ucast bfd mac table and do deletions - */
        SHASH_FOR_EACH_SAFE (node, next, kern_ucast_bfd_local_macs) {
            bool delete = false;
            struct unicast_mac_addr *ucm = node->data;
            struct log_switch *ls;

            ls = log_switch_lookup(kern_log_switches, ucm->ls_name);
            delete = !(ucm->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED));
            delete = delete || (ls && (ls->flags & HWVTEP_FLAGS_DELETE));
            if (delete)
                kernel_del_bfd_local_mac(ucm, ls);
        }

        /* walk the old tunnels and do delete */
        SHASH_FOR_EACH_SAFE (node, next, kern_tunnels) {
            struct tunnel *tnl = node->data;
            struct phy_locator *db_pl;
            char *enabled;
            int len;

            db_pl = phy_loc_lookup(db_phy_locs, tnl->remote_type,
                                 tnl->remote_dst_ip);
            if (!db_pl
                || !(tnl->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED))) {

                enabled = shash_find_data(&tnl->bfd_status, bfd_enabled);
                if (enabled && !strcmp(enabled, "true")) {
                    len = PTMLIB_MSG_SZ;
                    compose_stop_bfd_session_msg(tnl, ptm_msg_buf, &len);
                    if (send_ptm_msg(ptm_msg_buf, len))
                        del_bfd_tunnel(tnl);
                }
                shash_find_and_delete(kern_tunnels, tnl->key);
                tunnel_destroy(tnl);
            }
        }
    }

    /* - walk the ports of the old physical switch and do deletion */
    VLOG_DBG("kernel update: remove bindings\n");
    SHASH_FOR_EACH_SAFE (pnode, pnode_next, &kps->ports) {
        struct phy_port *pp = pnode->data;
        struct shash_node *vnode, *vnode_next;

        SHASH_FOR_EACH_SAFE (vnode, vnode_next, &pp->vlan_bindings) {
            struct vlan_binding *vb = vnode->data;
            struct log_switch *ls;

            ls = log_switch_lookup(kern_log_switches, vb->ls_name);

            if (ls &&
                (!(kps->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_ADD)) ||
                 !(ls->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_ADD)) ||
                 !(pp->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_ADD)) ||
                 !(vb->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_ADD)))) {
                VLOG_DBG("Deleting vlan binding for %s\n", pp->name);
                kernel_del_vlan_binding(pp, kps, vb);
            }
        }
    }

    /* - walk the old logical switch table and do deletions - */
    SHASH_FOR_EACH_SAFE (node, next, kern_log_switches) {
        struct log_switch *ls = node->data;

        if (!(ls->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED))) {
            VLOG_DBG("log switch %s flags %x\n", ls->name, ls->flags);
            /* source node replication enabled log switches 
            if (ls->ls_cfg) {
                if (ls->ls_cfg->replication_mode &&
                    !strcmp(ls->ls_cfg->replication_mode, "source_node")) {
                    remove_source_node_config(ls);
                }
            }*/
            kernel_del_log_switch(ls);
        }
    }

    /* - walk the old logical router and do deletions - */
    SHASH_FOR_EACH_SAFE (node, next, kern_log_routers) {
        struct logical_router *lr = node->data;
        struct shash_node *spb_node, *spb_node_next;
        struct shash_node *srt_node, *srt_node_next;

        SHASH_FOR_EACH_SAFE (spb_node, spb_node_next, &lr->switch_bindings) {
            struct switch_prefix_binding *spb = spb_node->data;
            if (!(lr->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED)) ||
                !(spb->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED))) {
                /* delete */
                struct log_switch *ls;
                ls = log_switch_lookup(kern_log_switches, spb->ls_name);
                if (ls)
                    kernel_del_prefix_binding(spb, ls);
            }
        }

        SHASH_FOR_EACH_SAFE (srt_node, srt_node_next, &lr->static_routes) {
            struct static_route *srt = srt_node->data;
            if (!(lr->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED)) ||
                !(srt->flags & (HWVTEP_FLAGS_ADD | HWVTEP_FLAGS_TOUCHED))) {
                /* delete */
                kernel_del_static_route(srt);
            }
        }
    }


    /* ADD */
    batch_fp = fopen("/tmp/ip_batch.tmp", "w");
    gettimeofday(&tv4, 0);
    VLOG_DBG("logical switch update start\n");

    /* walk the new logical switch table and do addition/modification */
    SHASH_FOR_EACH_SAFE (node, next, db_log_switches) {
        struct log_switch *ls = node->data;
        struct log_switch *kls;
        char bridge_name[HWVTEP_MAX_STR_LEN];

        if (vtep_vlan_aware && clag_enabled) {
            if(peerlink) {
                exec_cmd("ip link set dev %s master br-vxlan up", peerlink);
            }
        }

        if (!vtep_vlan_aware) {
            snprintf(bridge_name, HWVTEP_MAX_STR_LEN, "br-%s", ls->dev_name);
        }
        else if(vtep_vlan_aware && user_bridge_name_provided) {
            strncpy(bridge_name, user_bridge_name, strlen(user_bridge_name));
        }
        else if (vtep_vlan_aware && !user_bridge_name_provided) {
            strcpy(bridge_name, "br-vxlan");
        }

        if (!find_kernel_link(bridge_name)) {
            exec_cmd("ip link add %s type bridge", bridge_name);
            exec_cmd("ip link set dev %s", bridge_name);
            exec_cmd("ip link set dev %s up type bridge %s", bridge_name,
                         vtep_vlan_aware ? "vlan_filtering 1" : "");
            shash_add_once(links_ht, bridge_name, xstrdup("nomaster"));
        }

        //VLOG_DBG("logical switch %s, vlan %"PRId64"\n", ls->dev_name, ls->vlan);
        if (!(ls->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_DELETE))) {
            /* bfd ctrl vni does not have any ports associated with it.
             * if locally created log switch, allow it even if ports doesnt
             * exist.
             */
            if ((shash_count(&ls->log_ports) != 0) || (!ls->ls_cfg)) {
                kls = log_switch_lookup(kern_log_switches, ls->dev_name);
                if (kls && (kls->flags & HWVTEP_FLAGS_TOUCHED)) {
                    kernel_update_log_switch(ls);
                }
                else {
                    kernel_add_log_switch(ls);
                }
            }
        }
        else if ((ls->flags & HWVTEP_FLAGS_TOUCHED) && service_node_update) {
            VLOG_DBG("need service node update\n");
            kernel_update_log_switch(ls);
        }

        /* if the logical switch replication mode config was changed
         * from source_node to service_node or vice versa, make changes
         * Will be fixed soon.. commenting out source node config
        if (ls->ls_cfg) {
            if (!ls->ls_cfg->replication_mode ||
                !strcmp(ls->ls_cfg->replication_mode, "service_node")) {
                if (shash_find(source_node_ls_list, ls->dev_name)) {
                    remove_source_node_config(ls);
                    shash_find_and_delete(source_node_ls_list, ls->dev_name);
                    change_from_source_to_service_node_config(ls);
                }
            }
            else if (ls->ls_cfg->replication_mode &&
                !strcmp(ls->ls_cfg->replication_mode, "source_node")) {
                VLOG_DBG("Found source_node switch %s", ls->dev_name);
                if (!shash_find(source_node_ls_list, ls->dev_name)) {
                    add_source_node_config(ls);
                    shash_add(source_node_ls_list, ls->dev_name,
                              xstrdup(ls->dev_name));
                }
            }
        }*/
        /* TBD: the vlan bindings may need to be reprogrammed depending on the
         * logical switch, so clear the flag after the sync
         */
    }
    fflush(batch_fp);
    exec_cmd("ip -force -batch /tmp/ip_batch.tmp");
    gettimeofday(&tv5, 0);
    time_elapsed(tv4, tv5, &tv6);
    VLOG_DBG("logical switch update end (took %u.%06u)\n", (unsigned int)tv6.tv_sec, (unsigned int)tv6.tv_usec);
    fclose(batch_fp);
    unlink("/tmp/ip_batch.tmp");

    if (sync_remote_macs) {
        /* walk the new mac tables and do addition */
        SHASH_FOR_EACH_SAFE (node, next, db_ucast_remote_macs) {
            struct unicast_mac_addr *ucm = node->data;
            struct log_switch *ls = log_switch_lookup(db_log_switches, ucm->ls_name);
            if (!ls || !ls->dev_name)
                continue;
            if (!(ucm->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_DELETE))) {
                if (shash_count(&ls->log_ports) !=0)
                    kernel_add_remote_ucast_mac(ucm, ls);
            }
        }
        // We do not have support for kernel and iproute.. commenting out for now.
        /*SHASH_FOR_EACH_SAFE (node, next, db_mcast_remote_macs) {
            struct multicast_mac_addr *mcm = node->data;
            struct log_switch *ls = log_switch_lookup(db_log_switches, mcm->ls_name);
            if ((strcmp(unknown_dst_str, mcm->mac) != 0) &&
                (shash_count(&ls->log_ports) !=0))
                kernel_add_remote_mcast_mac(mcm, ls);
        }*/

        /* walk the new bfd mac tables and do addition */
        SHASH_FOR_EACH_SAFE (node, next, db_ucast_bfd_local_macs) {
            struct unicast_mac_addr *ucm = node->data;
            struct log_switch *ls = log_switch_lookup(db_log_switches, ucm->ls_name);
            if (!ls || !ls->dev_name)
                continue;
            if (!(ucm->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_DELETE))) {
                kernel_add_bfd_local_mac(ucm, ls);
            }
        }

        /*  walk existing db_tunnels and update BFD params, config remote (only on active db)*/
        if(db_active)
        {
            SHASH_FOR_EACH_SAFE (node, next, db_tunnels) {
                struct tunnel *tnl = node->data;
                char *db_enable , *db_enabled;
                int len;

                if (!(tnl->flags & (HWVTEP_FLAGS_TOUCHED | HWVTEP_FLAGS_DELETE))) {
                    db_enable = shash_find_data(&tnl->bfds, bfd_enable);
                    if (db_enable && (strcmp(db_enable, "true") == 0)) {
                        len = PTMLIB_MSG_SZ;
                        compose_start_bfd_session_msg(tnl, ptm_msg_buf, &len);
                        if (send_ptm_msg(ptm_msg_buf, len)) {
                            add_bfd_tunnel(tnl);
                        }
                    }
                    else {
                        db_enabled = shash_find_data(&tnl->bfd_status, bfd_enabled);
                        if (db_enabled && (strcmp(db_enabled, "true") == 0)) {
                            len = PTMLIB_MSG_SZ;
                            compose_stop_bfd_session_msg(tnl, ptm_msg_buf, &len);
                            if (send_ptm_msg(ptm_msg_buf, len)) {
                                del_bfd_tunnel(tnl);
                            }
                        }
                    }
                update_bfd_redirect(tnl->remote_dst_ip, db_active);
                }
            }
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, db_log_routers) {
        struct logical_router *lr = node->data;
        struct shash_node *spb_node, *spb_node_next;
        struct shash_node *srt_node, *srt_node_next;

        SHASH_FOR_EACH_SAFE (spb_node, spb_node_next, &lr->switch_bindings) {
            struct switch_prefix_binding *spb = spb_node->data;
            struct log_switch *ls;

            if ((lr->flags & HWVTEP_FLAGS_TOUCHED) ||
                (spb->flags & HWVTEP_FLAGS_TOUCHED))
                continue;

            ls = log_switch_lookup(db_log_switches, spb->ls_name);
            kernel_add_prefix_binding(spb, ls);
        }

        SHASH_FOR_EACH_SAFE (srt_node, srt_node_next, &lr->static_routes) {
            struct static_route *srt = srt_node->data;
            if ((lr->flags & HWVTEP_FLAGS_TOUCHED) ||
                (srt->flags & HWVTEP_FLAGS_TOUCHED))
                continue;
            kernel_add_static_route(srt);
        }
    }

    /* walk the ports of the new physical switch and do addition/modification */
    VLOG_DBG("kernel update: add binding\n");
    if (!dbps)
        return;

    SHASH_FOR_EACH_SAFE (pnode, pnode_next, &dbps->ports) {
        struct phy_port *pp = pnode->data;
        struct shash_node *vnode, *vnode_next;

        SHASH_FOR_EACH_SAFE (vnode, vnode_next, &pp->vlan_bindings) {
            struct vlan_binding *vb = vnode->data;
            struct log_switch *ls;

            //VLOG_DBG("port:%s vb flags: %x, vlan: %s",
            //         pp->name, vb->flags, vb->name);

            ls = log_switch_lookup(db_log_switches, vb->ls_name);
            if (ls) {
                if ((vb->flags & HWVTEP_FLAGS_ADD) || (vb->flags == 0)) {
                    kernel_add_vlan_binding(vb, pp, kps, ls);
                }
            }
        }
    }

    /* delete unused kernel config */
    SHASH_FOR_EACH_SAFE (node, next, del_links_ht) {
        char *dev = (char *)node->data;

        exec_cmd("ip link del %s\n", dev);
    }

    gettimeofday(&tv2, 0);
    time_elapsed(tv1, tv2, &tv3);
    VLOG_DBG("===== Done updating kernel (took %u.%06u)\n",
             (unsigned int)tv3.tv_sec, (unsigned int)tv3.tv_usec);

    /* handle the tunnel_ips */
}

struct stat_ {
    char iface[HWVTEP_MAX_STR_LEN];
    int64_t rx_p;
    int64_t rx_b;
    int64_t tx_p;
    int64_t tx_b;
};

static void
get_kernel_binding_stats(void)
{
    char iface_str[HWVTEP_MAX_STR_LEN];
    FILE *fp = NULL;
    struct phy_switch *kps;
    struct shash_node *node, *next;
    struct shash stat_tbl;

    kps = kern_phy_switch_lookup();
    if (!kps)
        return;

    VLOG_DBG("=> get kernel binding stats\n");
    shash_init(&stat_tbl);
    exec_cmd("cp /proc/net/dev /tmp/stat.tmp");
    fp = fopen("/tmp/stat.tmp", "r");
    if (fp) {
        size_t lsize = HWVTEP_MAX_STR_LEN - 1;
        char *line = NULL;
        char *save_ptr;
        char *name, *rxb, *rxp, *txb, *txp;
        char *t = NULL;
        struct stat_ *st;
        int i;

        while (getline(&line, &lsize, fp) != -1) {
            if (strstr(line, "Receive") || strstr(line, "packets")) {
                if (line) free(line);
                line = NULL;
                continue;
            }
            name = strtok_r(line, ":", &save_ptr);
            while (*name == ' ')
                name++;
            rxb = strtok_r(NULL, " ", &save_ptr);
            rxp = strtok_r(NULL, " ", &save_ptr);
            for (i = 0; i < 6; i++)
                t = strtok_r(NULL, " ", &save_ptr);
            txb = strtok_r(NULL, " ", &save_ptr);
            txp = strtok_r(NULL, " ", &save_ptr);

            st = xzalloc(sizeof(struct stat_));
            strcpy(st->iface, name);
            sscanf(rxb, "%"PRId64"", &st->rx_b);
            sscanf(rxp, "%"PRId64"", &st->rx_p);
            sscanf(txb, "%"PRId64"", &st->tx_b);
            sscanf(txp, "%"PRId64"", &st->tx_p);
            if(!shash_add_once(&stat_tbl, st->iface, st))
                free(st);
            if (line) 
                free(line);
            line = NULL;
        }
        if (line) free(line);
        line = NULL;
        fclose(fp);
        unlink("/tmp/stat.tmp");
    }
    SHASH_FOR_EACH_SAFE (node, next, &kps->ports) {
        struct phy_port *pp = node->data;
        struct shash_node *vnode, *vnode_next;

        SHASH_FOR_EACH_SAFE (vnode, vnode_next, &pp->vlan_bindings) {
            struct vlan_binding *vb = vnode->data;
            struct stat_ *st;

            if (vb->vlan == 0)
                sprintf(iface_str, "%s", pp->name);
            else
                sprintf(iface_str, "%s.%"PRId64"", pp->name, vb->vlan);

            st = shash_find_data(&stat_tbl, iface_str);
            if (st) {
                vb->stats.bytes_from_local = st->rx_b;
                vb->stats.packets_from_local = st->rx_p;
                vb->stats.bytes_to_local = st->tx_b;
                vb->stats.packets_to_local = st->tx_p;
            }
        }
    }
    shash_destroy_free_data(&stat_tbl);
}

/* - create phy switch
 * - get mgmt ip
 * - get local ip (if vxlan link exists in kernel)
 * - get phy ports and vlan bindings on each
 * - add vlan bindings to corresponding LS
 * - binding stats update separated, such that it can be done periodically
 *   and independently
 */
static void
get_kernel_phy_switch(void)
{
    struct ifaddrs *ifaddr, *ifa;
    struct phy_switch *ps;
    FILE *fp = NULL;
    size_t lsize = HWVTEP_MAX_STR_LEN - 1;
    char *line = NULL;
    char *save_ptr;
    char cur_hostname[HWVTEP_MAX_STR_LEN];
    struct shash bonds;
    DIR *dir;
    char *tmp;

    /* create physical switch */
    gethostname(cur_hostname, HWVTEP_MAX_STR_LEN);
    tmp = strtok(cur_hostname, ".");
    ps = xzalloc(sizeof(*ps));
    ps->name = xstrdup(cur_hostname);
    shash_init(&ps->local_ips);
    shash_init(&ps->mgmt_ips);
    shash_init(&ps->ports);
    shash_init(&bonds);
    shash_add_once(kern_phy_switches, ps->name, ps);
    VLOG_DBG("=== Get kernel physical switch: %s\n", ps->name);

    /* get mgmt address */
    if (getifaddrs(&ifaddr) == 0) {
        int family;
        char host[HWVTEP_MAX_STR_LEN];
        char *np;

        for (ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next) {
            if (ifa->ifa_addr == NULL)
                continue;

            family = ifa->ifa_addr->sa_family;
            if (strncmp(ifa->ifa_name, "eth", 3) || (family != AF_INET))
                continue;

            if (getnameinfo(ifa->ifa_addr, sizeof(struct sockaddr_in),
                            host, HWVTEP_MAX_STR_LEN, NULL,
                            0, NI_NUMERICHOST) != 0)
                continue;

            np = xstrdup(host);

            if (!shash_add_once(&ps->mgmt_ips, np, np))
                free(np);
        }
        freeifaddrs(ifaddr);
    }

    if (clag_enabled) {
        char *anycast_ip = get_clag_vxlan_anycast_ip();
        get_clag_bonds(kern_clag_b2c, kern_clag_c2b);

        if (anycast_ip) {
            if (!shash_find_data(&ps->local_ips, anycast_ip)) {
                struct phy_locator *loc;

                VLOG_DBG("=> local tunnel ip: %s", anycast_ip);
                loc = phy_loc_create(vxlan_encap_type, anycast_ip);
                loc->local = true;
                shash_add_once(kern_phy_locs, loc->key, loc);
                shash_add_once(&ps->local_ips, anycast_ip, xstrdup(anycast_ip));
            }
        }
    }

    fp = fopen("/tmp/link.tmp", "r");
    if (!fp)
        return;

    dir = opendir("/proc/net/bonding");
    if (dir) {
        struct dirent *ent;
        while ((ent = readdir(dir))) {
            char *np;

            if (!strcmp(ent->d_name, ".") ||
                !strcmp(ent->d_name, ".."))
                continue;

            np = xstrdup(ent->d_name);
            if(!shash_add_once(&bonds, np, np)) {
                free(np);
            }
        }
        closedir(dir);
    }

    while (getline(&line, &lsize, fp) != -1) {
        char local_ip[HWVTEP_MAX_STR_LEN];
        char pp_name[HWVTEP_MAX_STR_LEN];
        char orig_pp_name[HWVTEP_MAX_STR_LEN];
        char *tip = NULL;
        char *np = NULL, *si = NULL;
        bool is_bond = false;
        bool is_clag = false;

        if (strstr(line, "vxlan id")) {
            tip = strstr(line, "local");
        }
        /* get local tunnel ip */
        if (tip) {
            sscanf(tip, "local %s port", local_ip);
            if (!shash_find_data(&ps->local_ips, local_ip)) {
                struct phy_locator *loc;

                VLOG_DBG("=> local tunnel ip: %s", local_ip);
                loc = phy_loc_create(vxlan_encap_type, local_ip);
                loc->local = true;
                shash_add_once(kern_phy_locs, loc->key, loc);
                shash_add_once(&ps->local_ips, local_ip, xstrdup(local_ip));
            }
            free(line);
            line = NULL;
            continue;
        }

        memset(pp_name, 0, HWVTEP_MAX_STR_LEN);
        strtok_r(line, ":", &save_ptr);
        np = strtok_r(NULL, ":", &save_ptr);
        if (np) {
            while (*np == ' ')
                np++;
            if (strstr(np, "@")) {
                si = strtok_r(np, "@", &save_ptr);
	        while (*si == ' ')
                    si++;
            }
        }

        if (si)
            strncpy(pp_name, si, HWVTEP_MAX_STR_LEN);
        else if (np)
            strncpy(pp_name, np, HWVTEP_MAX_STR_LEN);
        else {
            free(line);
            line = NULL;
            continue;
        }

        if (!strstr(pp_name, "swp")) {
            struct shash_node *node, *next;
            SHASH_FOR_EACH_SAFE(node, next, &bonds) {
                if (!strcmp(pp_name, node->data)) {
                    is_bond = true;
                    free(line);
                    line = NULL;
                    break;
                }
            }
            if (!is_bond) {
                free(line);
                line = NULL;
                continue;
            }
        }

        if (np || si) {
            struct phy_port *pp;
            char *vlanp;
            int vlan = 0;
            port_vlan_info_t *pvinfo = NULL;

            vlanp = strchr(pp_name, '.');
            if (vlanp) {
                vlanp[0] = '\0';
                vlanp++;
                vlan = atoi(vlanp);
            }
#if 0
            char slave_path[HWVTEP_MAX_STR_LEN];
            /* if this is a bond slave, skip */
            sprintf(slave_path, "/sys/class/net/%s/bonding_slave", pp_name);
            if (access(slave_path, F_OK) != -1) {
                VLOG_DBG("skipping bond slave %s\n", pp_name);
                continue;
            }
#endif

            if (clag_enabled) {
                char *clag_id = shash_find_data(kern_clag_b2c, pp_name);
                /* prepend the switch name if it is not a clag bond */
                if (clag_id) {
                    strcpy(orig_pp_name, pp_name);
                    is_clag = true;
                    strcpy(pp_name, clag_id);
                } else {
                    char full_name[HWVTEP_MAX_STR_LEN];
                    sprintf(full_name, "%s-%s", ps->name, pp_name);
                    strcpy(pp_name, full_name);
                }
            }

            /* populate the local swp ports, skip the sub-interfaces */
            pp = shash_find_data(&ps->ports, pp_name);
            if (!pp) {
                pp = xzalloc(sizeof *pp);
                pp->name = xstrdup(pp_name);
                shash_init(&pp->vlan_bindings);
                shash_add_once(&ps->ports, pp->name, pp);
                VLOG_DBG("  got port %s on phy switch %s\n", pp_name, ps->name);
            }
            pp->is_bond = is_bond;
            pp->is_clag = is_clag;

            if (vtep_vlan_aware) {
                if (is_clag)
                    pvinfo = shash_find_data(port_vlans_ht, orig_pp_name);
                else
                    pvinfo = shash_find_data(port_vlans_ht, pp_name);
            } else {
                pvinfo = xzalloc(sizeof(port_vlan_info_t));
                pvinfo->vlan_bmp = bitmap_allocate(4096);
                /* allow vlan=0 too to make sure we populate every binding */
                bitmap_set1(pvinfo->vlan_bmp, vlan);
            }

            /* populate vlan bindings */
            if (pvinfo && pvinfo->vlan_bmp) {
                int vlan = 0;
                while (vlan < 4096) {
                    vlan = bitmap_scan(pvinfo->vlan_bmp, 1, vlan, 4096);
                    VLOG_DBG("Vlan found in bitmap for binding is %d\n", vlan);
                    if (vlan == 4096)
                        break;
                    char vb_name[HWVTEP_MAX_STR_LEN];
                    struct vlan_binding *vb = xzalloc(sizeof *vb);
                    //vlan vtep aware only char *ep;

                    struct log_switch *ls;

                    /* vlan vtep aware only change
                    master = strstr(master, "br-vxlan");
                    ep = strchr(master, ' ');
                    if (ep)
                        *ep = '\0';
                    */
                    if (vlan == pvinfo->pvid)
                        vb->vlan = 0;
                    else
                        vb->vlan = vlan;
                    vb_name_get(vb_name, HWVTEP_MAX_STR_LEN, pp->name, vb->vlan);
                    vb->name = xstrdup(vb_name);
                    if (vtep_vlan_aware) {
                        ls = get_kernel_log_switch_by_vlan(vlan);
                        if (ls)
                            vb->ls_name = xstrdup(ls->dev_name);
                    } /* vlan vtep aware only supported
                    else {
                        vb->ls_name = xstrdup(master + 3);
                        ls = log_switch_lookup(kern_log_switches, vb->ls_name);
                    }*/
                    VLOG_DBG("Adding vlan binding for %s with %s\n", pp->name, vb->name);
                    shash_add_once(&pp->vlan_bindings, vb->name, vb);

                    if (ls) {
                        shash_add_once(&ls->log_ports, vb->name, vb);
                        VLOG_DBG("  got kernel %s binding for port %s vlan %d to %s\n",
                                  vb->name, pp->name, vlan, vb->ls_name);
                    }
                    vlan = vlan + 1;
                }
            }

            if (!vtep_vlan_aware && pvinfo) {
                free(pvinfo->vlan_bmp);
                free(pvinfo);
            }
        }
        free(line);
        line = NULL;
    }
    if (line) free(line);
    line = NULL;
    fclose(fp);
    shash_destroy_free_data(&bonds);
}

/* - create log switch if vxlan link exists
 * - set local ip
 * - create remote mcast unknown-dst mac for each LS based on svcnode (if exist)
 * - not creating local mcast unknown-dst, that is only needed for the db
 */
static void
get_kernel_log_switches(void)
{
    FILE *fp = NULL;
    char *line = NULL;
    char *save_ptr;
    size_t lsize = HWVTEP_MAX_STR_LEN - 1;

    fp = fopen("/tmp/link.tmp", "r");
    if (!fp)
        return;

    VLOG_DBG("=== Get kernel logical switches\n");

    while (getline(&line, &lsize, fp) != -1) {
        char dev_name[HWVTEP_MAX_STR_LEN];
        char *dst, *np, *local_ip, *tl;
        struct log_switch *ls;
        struct shash *locs;
        struct multicast_mac_addr *mcm;
        int64_t tunnel_key;

        np = strstr(line, "vxlan id");
        if (!np) {
            free(line);
            line = NULL;
            continue;
        }
        sscanf(np+9, "%"PRId64"", &tunnel_key);
        log_switch_dev_name_get(tunnel_key, HWVTEP_MAX_STR_LEN, dev_name);
        ls = log_switch_create(dev_name, "", tunnel_key);

        local_ip = strstr(line, "local");
        tl = xstrdup(local_ip);
        local_ip = strtok_r(tl, " ", &save_ptr);
        local_ip = strtok_r(NULL, " ", &save_ptr);
        ls->local_ip = xstrdup(local_ip);
        shash_add_once(kern_log_switches, ls->name, ls);
        VLOG_DBG("  got kernel logical switch: %s, local ip %s\n",
                  ls->name, local_ip);
        free(tl);

        if (ls->tunnel_key == 0) {
            free(line);
            line = NULL;
            /* No service node for vnid 0 log switch */
            continue;
        }

        /* create remote mcast unknown-dst mac if exists */
        locs = xzalloc(sizeof(*locs));
        shash_init(locs);

        dst = strstr(line, "remote");
        if (dst) {
            struct phy_locator *loc;

            dst = strtok_r(dst, " ", &save_ptr);
            dst = strtok_r(NULL, " ", &save_ptr);

            loc = phy_loc_lookup(kern_phy_locs, vxlan_encap_type, dst);
            if (!loc) {
                loc = phy_loc_create(vxlan_encap_type, dst);
                shash_add_once(kern_phy_locs, loc->key, loc);
            }
            if (loc)
                shash_add_once(locs, loc->key, loc);
        }
        /* create mcast mac object even if no service node is
         * configured in the kernel. Because in the case
         * of all nodes being bfd down, it should not be assumed
         * as missing kernel mcast object and update the kernel
         */
        VLOG_DBG("  got kernel remote unknown-dst mac\n");
        mcm = mcast_mac_create(ls, unknown_dst_str, "", locs);
        shash_add_once(kern_mcast_remote_macs, mcm->key, mcm);
        free(line);
        line = NULL;
    }
    if (line) free(line);
    line = NULL;
    fclose(fp);

    if (!vtep_vlan_aware)
        return;

    fp = fopen("/tmp/bridge_vlan.tmp", "r");
    if (!fp)
        return;

    while (getline(&line, &lsize, fp) != -1) {
        struct log_switch *ls;
        char *vn, *dn;
        int64_t vlan = 0;

        if (strstr(line, "vlan ids") || !strstr(line, "vxln")) {
            free(line);
            line = NULL;
            continue;
        }

        dn = strtok_r(line, "\t", &save_ptr);
        vn = strtok_r(NULL, " ", &save_ptr);
        if (vn)
            sscanf(vn, "%"PRId64"", &vlan);

        ls = log_switch_lookup(db_log_switches, dn);
        if (ls) {
            ls->vlan = vlan;
            VLOG_DBG("log switch %s vlan %"PRId64"\n", ls->dev_name, vlan);
        }
        free(line);
        line = NULL;
    }
    if (line) free(line);
    line = NULL;
    fclose(fp);
}

/* - only collect remote ucast macs if exist
 */
static void
get_kernel_remote_macs(void)
{
    FILE *fp = NULL;
    char *line = NULL;
    char *save_ptr;
    size_t lsize = HWVTEP_MAX_STR_LEN - 1;

    exec_cmd("grep \"dev vxln\" /tmp/fdb.tmp | grep -w dst"
           " | grep -v \"00:00:00:00:00:00\" > /tmp/fdb_remote.tmp");
    fp = fopen("/tmp/fdb_remote.tmp", "r");
    if (!fp)
        return;

    VLOG_DBG("=== Get kernel remote macs\n");
    while (getline(&line, &lsize, fp) != -1) {
        char *ls_name;
        char *mac;
        char *dst;
        struct log_switch *ls;
        struct phy_locator *loc;
        struct unicast_mac_addr *ucm;

        mac = strtok_r(line, " ", &save_ptr);
        strtok_r(NULL, " ", &save_ptr);
        ls_name = strtok_r(NULL, " ", &save_ptr);
        strtok_r(NULL, " ", &save_ptr);
        dst = strtok_r(NULL, " ", &save_ptr);
        VLOG_DBG("  got remote mac:%s dev:%s dst:%s\n", mac, ls_name, dst);
        ls = log_switch_lookup(kern_log_switches, ls_name);
        if (!ls) {
            VLOG_DBG("cannot find ls\n");
            free(line);
            line = NULL;
            continue;
        }

        loc = phy_loc_lookup(kern_phy_locs, vxlan_encap_type, dst);
        if (!loc) {
            loc = phy_loc_create(vxlan_encap_type, dst);
            shash_add_once(kern_phy_locs, loc->key, loc);
        }
        ucm = ucast_mac_create(ls, mac, "", loc);

        shash_add_once(kern_ucast_remote_macs, ucm->key, ucm);
        free(line);
        line = NULL;
    }
    if (line) free(line);
    fclose(fp);
    unlink("/tmp/fdb_remote.tmp");

    /* not doing any remote mcast macs here. The remote unknown-dst
     * mac is created during get_kernel_log_switch
     */
}

/* - collect local ucast macs only if exist
 */
struct mac_ {
    char if_name[HWVTEP_MAX_STR_LEN];
    char mac_str[HWVTEP_MAX_STR_LEN];
};

struct br_mac_ {
    char br_name[HWVTEP_MAX_STR_LEN];
    struct shash tbl;
};

static void
get_kernel_local_macs(bool bfd_mac)
{
    struct shash_node *node, *next;
    struct shash mac_tbl;
    size_t lsize = HWVTEP_MAX_STR_LEN - 1;
    char *line = NULL;
    char *save_ptr;
    FILE *fp;
    struct shash* kern_local_mac_ht;
    char *bfd_def_mac = "00:23:20:00:00:01";

    if (bfd_mac) {
        exec_cmd("grep dummy /tmp/fdb.tmp | grep br-vx > /tmp/fdb_local.tmp");
        kern_local_mac_ht = kern_ucast_bfd_local_macs;
    }
    else {
        if (vtep_vlan_aware) {
            exec_cmd("grep \"br-vxlan\" /tmp/fdb.tmp | grep -vw permanent | grep -v \"dev vxln\" > /tmp/fdb_local.tmp");
            exec_cmd("grep \"br-vxln0\" /tmp/fdb.tmp | grep -vw permanent | grep -v \"dev vxln\" >> /tmp/fdb_local.tmp");
        } else {
            exec_cmd("grep \"br-vxln\" /tmp/fdb.tmp | grep -vw permanent | grep -v \"dev vxln\" > /tmp/fdb_local.tmp");
        }
        kern_local_mac_ht = kern_ucast_local_macs;
    }

    fp = fopen("/tmp/fdb_local.tmp", "r");
    if (!fp)
        return;

    shash_init(&mac_tbl);
    while (getline(&line, &lsize, fp) != -1) {
        char *ifp, *bp, *mp, *t, *v;
        struct mac_ *m;
        struct br_mac_ *br;
        int vlan = 0;
        struct log_switch *ls;

        mp = strtok_r(line, " ", &save_ptr);       /* mac */
        t = strtok_r(NULL, " ", &save_ptr);        /* 'dev' */
        ifp = strtok_r(NULL, " ", &save_ptr);      /* dev name */
        t = strtok_r(NULL, " ", &save_ptr);        /* 'vlan' or 'master' */
        if (!strcmp(t, "vlan")) {
            v = strtok_r(NULL, " ", &save_ptr);    /* vlan number */
            t = strtok_r(NULL, " ", &save_ptr);    /* 'master' */
            sscanf(v, "%d", &vlan);
        }
        bp = strtok_r(NULL, " ", &save_ptr);       /* master name */
        if (!vtep_vlan_aware || !strcmp(bp, "br-vxln0")) {
            if (strstr(bp, "br-"))
                bp = bp + 3;          /* vxlan name */
        } else {
            ls = get_kernel_log_switch_by_vlan(vlan);
            if (ls)
                bp = ls->dev_name;
        }

        br = shash_find_data(&mac_tbl, bp);
        if (!br) {
            br = xzalloc(sizeof(struct br_mac_));
            strcpy(br->br_name, bp);
            shash_init(&br->tbl);
            shash_add_once(&mac_tbl, br->br_name, br);
        }
        m = xzalloc(sizeof(struct mac_));
        strcpy(m->if_name, ifp);
        strcpy(m->mac_str, mp);
        if (!shash_add_once(&br->tbl, m->mac_str, m)) {
           /* bfd mac can come as a duplicate entry */
           if (strncmp(bfd_def_mac, m->mac_str, sizeof(bfd_def_mac))) {
                VLOG_ERR("duplicate mac %s in %s\n", m->mac_str, br->br_name);
           }
           free(m);
        }
        if (line) free(line);
        line = NULL;
    }
    if (line) free(line);
    line = NULL;
    fclose(fp);
    unlink("/tmp/fdb_local.tmp");

    VLOG_DBG("=== Get kernel local macs\n");
    SHASH_FOR_EACH_SAFE (node, next, kern_log_switches) {
        struct log_switch *ls = node->data;
        struct shash_node *bn, *bnnext;
        struct br_mac_ *br;

        br = shash_find_data(&mac_tbl, ls->dev_name);
        if (!br)
            continue;

        SHASH_FOR_EACH_SAFE (bn, bnnext, &br->tbl) {
            struct mac_ *mp = bn->data;
            struct unicast_mac_addr *new_mac;
            struct phy_locator *pl;

            pl = phy_loc_lookup(kern_phy_locs, vxlan_encap_type, ls->local_ip);
            if (!pl) {
                VLOG_DBG("cannot find phy loc: %s\n", ls->local_ip);
                continue;
            }

            VLOG_DBG("  got MAC[%s %s %s]", ls->name, mp->if_name, mp->mac_str);
            new_mac = ucast_mac_lookup(kern_local_mac_ht, mp->mac_str,
                                       ls->dev_name);
            if (new_mac) {
                new_mac->flags |= HWVTEP_FLAGS_TOUCHED;
                continue;
            }
            if (!new_mac) {
                VLOG_DBG("  got new local mac %s\n", mp->mac_str);
                new_mac = ucast_mac_create(ls, mp->mac_str, "", pl);
                shash_add_once(kern_local_mac_ht, new_mac->key, new_mac);
            }
        }
    }

    SHASH_FOR_EACH_SAFE (node, next, &mac_tbl) {
        struct br_mac_ *br = node->data;
        shash_destroy_free_data(&br->tbl);
        shash_delete(&mac_tbl, node);
        free(br);
    }
    shash_destroy(&mac_tbl);
    /* No need to get local mcast mac here.  The local mcast unknown-dst mac
     * will be created during sync based on existence of db's logical ports
     */
}

static void
get_kernel_tunnels(void)
{
    struct shash_node *node, *next;
    struct tunnel *tnl, *bfd_tnl;
    struct phy_locator *pl;
    char *tip = kern_local_ip_get();

    VLOG_DBG("=== Get kernel tunnels\n");

    if (!tip) {
        VLOG_DBG("local ip not configured - no kernel tunnels\n");
        return;
    }
    SHASH_FOR_EACH_SAFE (node, next, kern_phy_locs) {
        pl = node->data;

        if (pl->local) {
            continue;
        }
        tnl = tunnel_create(pl->type, tip, pl->type, pl->dst_ip, NULL);
        shash_add_once(kern_tunnels, tnl->key, tnl);
    }

    /* any remote ip (such as service node) for which bfd is down, is not programmed
       in the kernel. As a result, kern_phy_loc does not exist. Form kern tunnels
       for these ips using bfd_tunnels
     */
    SHASH_FOR_EACH_SAFE(node, next, bfd_tunnels) {
        bfd_tnl = node->data;

        tnl = tunnel_lookup(kern_tunnels,
                            bfd_tnl->local_type, bfd_tnl->local_dst_ip,
                            bfd_tnl->remote_type, bfd_tnl->remote_dst_ip);
        if (!tnl) {
            pl = phy_loc_lookup(kern_phy_locs, bfd_tnl->remote_type,
                                bfd_tnl->remote_dst_ip);
            if (!pl) {
                pl = phy_loc_create(bfd_tnl->remote_type,
                                    bfd_tnl->remote_dst_ip);
                shash_add_once(kern_phy_locs, pl->key, pl);
            }
            tnl = tunnel_create(pl->type, tip, pl->type, pl->dst_ip, NULL);
            shash_add_once(kern_tunnels, tnl->key, tnl);
        }
        if (ptm_sock != -1) {
            tunnel_copy_bfd_info(bfd_tnl, tnl);
        }
        else {
            /* if connectivity to ptm is lost destroy bfd tunnels so that
             * bfd sesssions could be re-started when connection is restored
             */
            shash_delete(bfd_tunnels, node);
            tunnel_destroy(bfd_tnl);
        }
    }
}

static void
get_kernel_config(bool sync_local_macs, bool sync_remote_macs, bool sync_stats)
{
    struct timeval tv, tv1, tv2;

    gettimeofday(&tv, 0);
    VLOG_DBG("===== Get kernel config\n");
    if (vtep_vlan_aware) {
        exec_cmd("bridge -c vlan show > /tmp/bridge_vlan.tmp");
        get_kernel_port_vlans();
    }
    get_kernel_links();

    get_kernel_log_switches();
    get_kernel_phy_switch();

    /* collect all fdb entries one time for local, remote mac processing */
    exec_cmd("bridge fdb show > /tmp/fdb.tmp");

    if (sync_remote_macs)
        get_kernel_remote_macs();

    if (sync_local_macs)
        get_kernel_local_macs(false);

    if (sync_stats)
        get_kernel_binding_stats();

    get_kernel_tunnels();
    get_kernel_local_macs(true);

    unlink("/tmp/fdb.tmp");
    gettimeofday(&tv1, 0);
    time_elapsed(tv, tv1, &tv2);
    VLOG_DBG("===== Done with get kernel config (took %u.%06u)\n",
             (unsigned int)tv2.tv_sec, (unsigned int)tv2.tv_usec);
}

static int
open_ptm_socket(void)
{
    struct sockaddr_un addr;
    int sock;
    int ret;

    ptm_sock = -1;

    sock = socket(PF_UNIX, SOCK_STREAM, 0);
    if (sock < 0) {
        VLOG_ERR("socket failed : %s", strerror(errno));
        return -1;
    }
    memset(&addr, 0, sizeof(struct sockaddr_un));
    addr.sun_family = AF_UNIX;
    memcpy(&addr.sun_path, PTM_SOCK_NAME, sizeof(PTM_SOCK_NAME));
    ret = connect(sock, (struct sockaddr *)&addr,
                  (sizeof(addr.sun_family)+sizeof(PTM_SOCK_NAME) - 1));
    if (ret < 0) {
        VLOG_ERR("Unable to connect socket %s\n", strerror(errno));
        close(sock);
        return -1;
    }
    VLOG_DBG("Connected to PTM socket %s\n", PTM_SOCK_NAME);
    ptm_sock = sock;
    return sock;
}

static int
ptm_process_notify(void *arg, void *in_ctxt)
{
    struct tunnel *tnl;
    char remote_ip[32];
    char local_ip[32];
    char state[32];
    char diagnostic[32];
    char *enabled = "true";
    char *old_data;
    bool *update = arg;

    *update = false;

    ptm_lib_find_key_in_msg(in_ctxt, ptm_peer, remote_ip);
    if (!strlen(remote_ip)) {
        VLOG_ERR("ptm: Cannot find %s in notification\n", ptm_peer);
        return 0;
    }

    ptm_lib_find_key_in_msg(in_ctxt, ptm_local, local_ip);
    if (!strlen(local_ip)) {
        VLOG_ERR("ptm: Cannot find %s in notification\n", ptm_local);
        return 0;
    }
    tnl =  tunnel_lookup(bfd_tunnels,  vxlan_encap_type, local_ip,
            vxlan_encap_type, remote_ip);
    if (!tnl) {
        VLOG_INFO("ptm: tunnel %s - %s deleted already, ignore notification\n",
                local_ip, remote_ip);
        return 0;
    }

    ptm_lib_find_key_in_msg(in_ctxt, ptm_state, state);
    if (!strlen(state)) {
        VLOG_ERR("ptm: Cannot find %s in notification\n", ptm_state);
        return 0;
    }

    ptm_lib_find_key_in_msg(in_ctxt, ptm_diag, diagnostic);
    if (!strlen(diagnostic)) {
        VLOG_ERR("ptm: Cannot find %s in notification\n", ptm_diag);
        return 0;
    }
    VLOG_INFO("ptm: Received notification Rem[%s] Lcl[%s] St[%s] Diag[%s]\n",
            remote_ip, local_ip, state, diagnostic);

    old_data = shash_replace(&tnl->bfd_status, bfd_enabled, xstrdup(enabled));
    if (old_data) {
        free(old_data);
    }
    old_data = shash_replace(&tnl->bfd_status, bfd_state, xstrdup(state));
    if (old_data) {
        free(old_data);
    }
    if (strcasecmp(state, "up") == 0) {
        old_data = shash_replace(&tnl->bfd_status, bfd_forwarding,
                xstrdup("true"));
    }
    else {
        old_data = shash_replace(&tnl->bfd_status, bfd_forwarding,
                xstrdup("false"));
    }
    if (old_data) {
        free(old_data);
    }
    old_data = shash_replace(&tnl->bfd_status, bfd_diag, xstrdup(diagnostic));
    if (old_data) {
        free(old_data);
    }
    *update = true;

    return 0;
}

static int
ptm_process_response(void *arg, void *in_ctxt)
{
    struct tunnel *tnl;
    char remote_ip[32];
    char local_ip[32];
    char *disabled = "false";
    char *old_data;
    bool *update = arg;
    char cmd[32], status[32];

    ptm_lib_find_key_in_msg(in_ctxt, ptm_cmd, cmd);
    if (!strlen(cmd)) {
        VLOG_ERR("ptm: Cannot find %s in response\n", ptm_cmd);
        return 0;
    }
    ptm_lib_find_key_in_msg(in_ctxt, ptm_status, status);
    if (!strlen(status)) {
        VLOG_ERR("ptm: Cannot find %s in response\n", ptm_status);
        return 0;
    }
    *update = false;

    VLOG_INFO("ptm: cmd [%s] st [%s]\n", cmd, status);

    if (!strcmp(cmd, "start-bfd-sess")) {
        if (!strcmp(status, "pass"))
            return 0;
    } else if (!strcmp(cmd, "stop-bfd-sess")) {
        /* do nothing */
        return 0;
    }

    /* if we are here - start-bfd-sess failed. mark tunnel for deletion */
    ptm_lib_find_key_in_msg(in_ctxt, ptm_peer, remote_ip);
    if (!strlen(remote_ip)) {
        VLOG_ERR("ptm: Cannot find %s in response\n", ptm_peer);
        return 0;
    }
    ptm_lib_find_key_in_msg(in_ctxt, ptm_local, local_ip);
    if (!strlen(local_ip)) {
        VLOG_ERR("ptm: Cannot find %s in response\n", ptm_local);
        return 0;
    }
    tnl =  tunnel_lookup(bfd_tunnels,  vxlan_encap_type, local_ip,
            vxlan_encap_type, remote_ip);
    if (!tnl) {
        VLOG_INFO("ptm: tunnel %s - %s deleted already, ignore msg\n",
                  local_ip, remote_ip);
        return 0;
    }

    VLOG_INFO("ptm: set tunnel bfd disabled %s - %s \n", local_ip, remote_ip);
    old_data = shash_replace(&tnl->bfd_status, bfd_enabled, xstrdup(disabled));
    if (old_data) {
        free(old_data);
    }
    *update = true;

    return 0;
}

static bool
bfd_status_change(void)
{
    char *ptm_msg = xzalloc(PTMLIB_MSG_SZ);
    int msglen = PTMLIB_MSG_SZ;
    bool update = false;
    int rc;

    rc = ptm_lib_process_msg(ptm_hdl, ptm_sock, ptm_msg, msglen, &update);
    if ((rc < 0) && errno && (errno != EWOULDBLOCK)) {
        /* since we are polling its possible to return -1 from lib */
        VLOG_WARN("PTM socket error: %s\n", strerror(errno));
        close(ptm_sock);
        ptm_sock = -1;
    } else {
        ptm_poll = true;
    }
    free(ptm_msg);
    return update;
}

static bool is_db_active(void) {

    FILE *fp;
    char data[HWVTEP_MAX_STR_LEN];
    char *active_str = "active";
    char *backup_str = "backup";
    char *t2;
    char *cmd = "ovs-appctl -t /var/run/openvswitch/ovsdb-server."
                "`pidof ovsdb-server`.ctl ovsdb-server/sync-status";

    /* Open the command for reading. */
    fp = popen(cmd, "r");
    if (fp == NULL) {
        VLOG_ERR("Failed to run command to check if db is active\n" );
        return false;
    }

    /* Read the output a line at a time */
    while (fgets(data, sizeof(data)-1, fp) != NULL) {
        if (strstr(data, "state")) {
            char *t1 = strchr(data, ':');
            t1 += 2;
            t2 = strtok(t1, " ");
            if(!strncmp(t2, active_str, strlen(active_str))) {
                pclose(fp);
                return true;
            }
            else if(!strncmp(t2, backup_str, strlen(backup_str))) {
                pclose(fp);
                return false;
            }
        }
    }

    /* close */
    pclose(fp);
    return true;
}


static bool
vxln0_protodown_on_status() {
    char data[HWVTEP_MAX_STR_LEN];
    FILE *fp;
    char *cmd = "ip link show vxln0 | grep protodown";
    char *protodown_status = "protodown on";

    fp = popen(cmd, "r");
    if (!fp) {
        VLOG_ERR("Failed to check vxln0 link status");
        return false;
    }
    while (fgets(data, sizeof(data)-1, fp) != NULL) {
        if (strstr(data, protodown_status)) {
            pclose(fp);
            return true;
        }
    }
    pclose(fp);
    return false;
}

static bool
bfd_sessions_established() {
    char data[HWVTEP_MAX_STR_LEN];
    FILE *fp;
    char *cmd = "ptmctl -b";
    char *err_str = "ERR: No BFD sessions";

    VLOG_DBG("Running command to check bfd established %s", cmd);

    fp = popen(cmd, "r");
    if (!fp) {
        VLOG_DBG("Failed to check bfd status for service nodes");
        return false;
    }
    while (fgets(data, sizeof(data)-1, fp) != NULL) {
        if (strstr(data, err_str)) {
            pclose(fp);
            VLOG_DBG("No BFD sessions found, check service nodes");
            return false;
        }
    }
    pclose(fp);
    return true;
}

static bool
iptables_entry_installed() {
    char data[HWVTEP_MAX_STR_LEN];
    FILE *fp;
    char *peer_clag_ip = get_clag_vxlan_peer_clag_ip();
    char cmd[HWVTEP_MAX_STR_LEN];

    snprintf(cmd, HWVTEP_MAX_STR_LEN, "iptables -t mangle -L | grep %s",
                              peer_clag_ip);
    VLOG_DBG("Running command to check the entries installed %s", cmd);

    fp = popen(cmd, "r");
    if (!fp) {
        VLOG_DBG("Failed to open the file for iptables entry");
        return false;
    }
    while (fgets(data, sizeof(data)-1, fp) != NULL) {
        VLOG_DBG("Data is %s", data);
        if (strstr(data, peer_clag_ip)) {
            pclose(fp);
            return true;
        }
    }
    pclose(fp);
    return false;
}

void
hwvtep_run(void)
{
    bool database_changed = false;
    bool sync_local_macs = false, sync_remote_macs = false, sync_stats = false;
    bool sync_tunnels = false;

    if(!db_active_checked) {
        db_active = is_db_active();
        db_active_checked = true;
    }

    /* (Re)configure if necessary. */
    ovsdb_idl_run(vtep_idl);
    if(ovsdb_idl_get_seqno(vtep_idl) != idl_seqno) {
        idl_seqno = ovsdb_idl_get_seqno(vtep_idl);
        database_changed = true;
    }

    active = clag_status(&clag_enabled, &peerisalive);

    /* we only check once if the db is active so when secondary becomes 
     * active, peer is not alive so check this condition */
    if (!db_active && !peerisalive) {
        db_active = is_db_active();
        backup_to_active = true;
    }

    if (backup_to_active && db_active && peerisalive) {
       db_active = is_db_active();
       backup_to_active = false;
    }

    if (db_active && ptm_sock == -1) {
        if (shash_count(bfd_tunnels) > 0) {
            /* socket connection to ptm is lost but bfd tunnels still
             * exist. Trigger sync to clean up tunnel bfd status
             */
            sync_tunnels = true;
        }
        if ((time_msec() > ptm_retry_timer) && (open_ptm_socket() != -1)) {
            /* reset the retry interval given that socket is connected */
            ptm_retry_interval = PTM_RETRY_INTERVAL;
            VLOG_DBG("ptm_retry_interval reset to %d\n", PTM_RETRY_INTERVAL);
            sync_tunnels = true;
        }
    }

    if (!startup_done) {
        VLOG_DBG("Vtep startup in progess");
        sync_local_macs = sync_stats = sync_remote_macs = true;
    }
    else if (database_changed) {
        sync_local_macs = sync_stats = sync_remote_macs = true;
    }
    else {
        /* update the local macs and stats once timer is over */
	    sync_local_macs = (time_msec() > local_mac_timer);
	    sync_stats = (time_msec() > stats_timer);
    }

    if ((ptm_sock != -1) && (sync_tunnels == false)) {
         sync_tunnels = bfd_status_change();
    }
    sync_remote_macs = (database_changed | sync_tunnels);

    if (!startup_done)
        daemonize_complete();

    /* parse the tables and do some stuff */
    if (database_changed || !startup_done ||
        sync_local_macs || sync_remote_macs || sync_stats) {

        bool has_logical_config = false;
        struct timeval tv, tv1, tv2;

        aborted = false;

        VLOG_INFO("<<SYNC START>>\n");

        VLOG_DBG("database_changed = %d, sync_local_macs = %d,"
                  "sync_remote_macs = %d, sync_stats = %d\n",
                  database_changed, sync_local_macs, sync_remote_macs,
                  sync_stats);

        get_kernel_config(sync_local_macs, sync_remote_macs, sync_stats);
        VLOG_DBG("----- Get DB config -----\n");
        gettimeofday(&tv, 0);
        add_del_phy_switches();
        add_del_log_switches();
        add_del_phy_locators();

        has_logical_config = shash_count(db_log_switches) ||
                             shash_count(kern_log_switches);

        if (has_logical_config) {
            if (sync_remote_macs)
                add_del_remote_macs();
            if (sync_local_macs)
                add_del_local_macs();
            add_del_tunnels();
            //add_del_log_routers();
        }

        gettimeofday(&tv1, 0);
        time_elapsed(tv, tv1, &tv2);
        VLOG_DBG("----- Done with get DB config (took %u.%06u) -----\n",
                 (unsigned int)tv2.tv_sec, (unsigned int)tv2.tv_usec);

        /* add the bfd redirect on backup*/
        if (!db_active && !bfd_redirect_added) {
            VLOG_DBG("Standby ovsdb, add iptables entry and close ptm sock");
            update_all_bfd_sessions(false);
            close(ptm_sock);
            if(iptables_entry_installed()) {
                bfd_redirect_added = true;
            }
        }

        /* if backup becomes active and bfd redirect is added
         * then remove the rule */
        if (db_active && bfd_redirect_added) {
            VLOG_DBG("failover: delete iptables entry and open ptm sock");
            if(ptm_sock == -1) {
                open_ptm_socket();
            }
            update_all_bfd_sessions(true);
            bfd_redirect_added = false;
        }

        /* db is active but bfd sessions are not started */
        if(db_active && !bfd_sessions_updated) {
            VLOG_DBG("db is active and bfd sessions not started");
            if(ptm_sock == -1) {
                open_ptm_socket();
            }
            update_all_bfd_sessions(true);
            if(bfd_sessions_established()) {
                 bfd_sessions_updated = true;
            }
        }

        /* after reboot, vxln0 is set to protodown on by clag, setting it off */
        if(db_active && !vxln0_protodown_off && clag_enabled) {
            if(vxln0_protodown_on_status()) {
                exec_cmd("ip link set dev vxln0 master br-vxln0 protodown off");
                if(!bfd_sessions_established()) {
                    VLOG_DBG("sessions have error close and reopen ptm sock");
                    close(ptm_sock);
                    ptm_sock = -1;
                    open_ptm_socket();
                    update_all_bfd_sessions(true);
                }
                vxln0_protodown_off = true;
            }
        }

        update_db_config(sync_local_macs, sync_remote_macs,
                         sync_stats, has_logical_config);
        if (has_logical_config || shash_count(del_links_ht)) {
            update_kernel_config(sync_remote_macs);
            VLOG_DBG("Done with update_kernel_config");
        }
        cleanup_after_sync();

        VLOG_INFO("<<SYNC DONE>>\n");

        if (!startup_done) {
            //daemonize_complete();
            startup_done = true;
            VLOG_INFO("VTEP startup done\n");
        }
    }
}

void
hwvtep_wait(void)
{
    ovsdb_idl_wait(vtep_idl);
    if (!aborted) {
        if (local_mac_timer <= time_msec()) {
            local_mac_timer = time_msec() + LOCAL_MAC_INTERVAL;
            poll_timer_wait_until(local_mac_timer);
        }

        if (stats_timer <= time_msec()) {
            stats_timer = time_msec() + STATS_POLL_INTERVAL;
            poll_timer_wait_until(stats_timer);
        }
    }

    if (switchover_timer != LLONG_MIN)
        poll_timer_wait_until(switchover_timer);

    if (ptm_sock == -1) {
        if (!aborted) {
            if (ptm_retry_timer <= time_msec()) {
                ptm_retry_timer = time_msec() + ptm_retry_interval;
                poll_timer_wait_until(ptm_retry_timer);
                if (ptm_retry_interval < PTM_MAX_RETRY_INTERVAL)
                    ptm_retry_interval = ptm_retry_interval * PTM_RETRY_MULTIPLIER;
            }
            VLOG_DBG("ptm_retry_interval %lld\n", ptm_retry_interval);
        }
    }
    else if (ptm_poll) {
        poll_fd_wait(ptm_sock, POLLIN);
        ptm_poll = false;
    }
}

void
hwvtep_init(const char *remote)
{
    db_phy_locs = xzalloc(sizeof(*db_phy_locs));
    db_ucast_local_macs = xzalloc(sizeof(*db_ucast_local_macs));
    db_ucast_remote_macs = xzalloc(sizeof(*db_ucast_remote_macs));
    db_mcast_local_macs = xzalloc(sizeof(*db_mcast_local_macs));
    db_mcast_remote_macs = xzalloc(sizeof(*db_mcast_remote_macs));
    db_phy_switches = xzalloc(sizeof(*db_phy_switches));
    db_log_switches = xzalloc(sizeof(*db_log_switches));
    db_log_routers = xzalloc(sizeof(*db_log_routers));
    db_tunnels = xzalloc(sizeof(*db_tunnels));
    db_ucast_bfd_local_macs = xzalloc(sizeof(*db_ucast_bfd_local_macs));
    shash_init(db_phy_locs);
    shash_init(db_ucast_local_macs);
    shash_init(db_ucast_remote_macs);
    shash_init(db_mcast_local_macs);
    shash_init(db_mcast_remote_macs);
    shash_init(db_phy_switches);
    shash_init(db_log_switches);
    shash_init(db_log_routers);
    shash_init(db_tunnels);
    shash_init(db_ucast_bfd_local_macs);

    kern_phy_locs = xzalloc(sizeof(*kern_phy_locs));
    kern_ucast_local_macs = xzalloc(sizeof(*kern_ucast_local_macs));
    kern_ucast_remote_macs = xzalloc(sizeof(*kern_ucast_remote_macs));
    kern_mcast_local_macs = xzalloc(sizeof(*kern_mcast_local_macs));
    kern_mcast_remote_macs = xzalloc(sizeof(*kern_mcast_remote_macs));
    kern_phy_switches = xzalloc(sizeof(*kern_phy_switches));
    kern_clag_b2c = xzalloc(sizeof(*kern_clag_b2c));
    kern_clag_c2b = xzalloc(sizeof(*kern_clag_c2b));
    kern_log_switches = xzalloc(sizeof(*kern_log_switches));
    kern_log_routers = xzalloc(sizeof(*kern_log_routers));
    kern_tunnels = xzalloc(sizeof(*kern_tunnels));
    bfd_tunnels = xzalloc(sizeof(*bfd_tunnels));
    kern_ucast_bfd_local_macs = xzalloc(sizeof(*kern_ucast_bfd_local_macs));
    shash_init(kern_phy_locs);
    shash_init(kern_ucast_local_macs);
    shash_init(kern_ucast_remote_macs);
    shash_init(kern_mcast_local_macs);
    shash_init(kern_mcast_remote_macs);
    shash_init(kern_phy_switches);
    shash_init(kern_clag_b2c);
    shash_init(kern_clag_c2b);
    shash_init(kern_log_switches);
    shash_init(kern_log_routers);
    shash_init(kern_tunnels);
    shash_init(bfd_tunnels);
    shash_init(kern_ucast_bfd_local_macs);

    ls_pl_pair = xzalloc(sizeof(*ls_pl_pair));
    shash_init(ls_pl_pair);
    vb_rn_pair = xzalloc(sizeof(*vb_rn_pair));
    shash_init(vb_rn_pair);
    vb_ls_pair = xzalloc(sizeof(*vb_ls_pair));
    shash_init(vb_ls_pair);
    source_node_ls_list = xzalloc(sizeof(*source_node_ls_list));
    shash_init(source_node_ls_list);
    ls_dev_to_db_name_ht = xzalloc(sizeof(*ls_dev_to_db_name_ht));
    shash_init(ls_dev_to_db_name_ht);
    port_vlans_ht = xzalloc(sizeof(*port_vlans_ht));
    shash_init(port_vlans_ht);
    links_ht = xzalloc(sizeof(*links_ht));
    shash_init(links_ht);
    del_links_ht = xzalloc(sizeof(*del_links_ht));
    shash_init(del_links_ht);
    vlan_to_ls_dev_ht = xzalloc(sizeof(*vlan_to_ls_dev_ht));
    shash_init(vlan_to_ls_dev_ht);
    vtep_idl = ovsdb_idl_create(remote, &vteprec_idl_class, true, true);
    idl_seqno = ovsdb_idl_get_seqno(vtep_idl);
    char cur_hostname[HWVTEP_MAX_STR_LEN];
    gethostname(cur_hostname, HWVTEP_MAX_STR_LEN);
    int counter = 0;
    /*ovsdb doesnt like any other char than '_' for hostname so replace */
    while(cur_hostname[counter] != '\0')
    {
        if((cur_hostname[counter] == '-') || (cur_hostname[counter] == '.'))
        {
            cur_hostname[counter] = '_';
        }
        counter++;
    }
    strcat(cur_hostname, "_ops_vtepd");
    ovsdb_idl_set_lock(vtep_idl, cur_hostname);
    ptm_hdl = ptm_lib_register("vtepd", NULL, ptm_process_notify,
                               ptm_process_response);
    vlan_bitmap = bitmap_allocate(4096);
}

void
hwvtep_exit(void)
{
    /* destroy all tables */
    ovsdb_idl_destroy(vtep_idl);
    ptm_lib_deregister(ptm_hdl);
    free(vlan_bitmap);
    shash_destroy_free_data(ls_pl_pair);
    shash_destroy_free_data(source_node_ls_list);
    shash_destroy_free_data(vb_rn_pair);
    shash_destroy_free_data(vb_ls_pair);
}
