/* Copyright (c) 2008, 2009, 2010, 2011 Nicira Networks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <config.h>

#include <errno.h>
#include <getopt.h>
#include <limits.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_MLOCKALL
#include <sys/mman.h>
#endif

#include "vtep.h"
#include "command-line.h"
#include "compiler.h"
#include "daemon.h"
#include "dirs.h"
#include "dpif.h"
#include "dummy.h"
#include "netdev.h"
#include "openflow/openflow.h"
#include "ovsdb-idl.h"
#include "include/openvswitch/poll-loop.h"
#include "process.h"
#include "signals.h"
#include "stream.h"
#include "svec.h"
#include "timeval.h"
#include "unixctl.h"
#include "util.h"
#include "include/openvswitch/vconn.h"
#include "include/openvswitch/vlog.h"
#include "vswitch-idl.h"
#include "stream-ssl.h"

VLOG_DEFINE_THIS_MODULE(vswitchd);

bool vtep_mac_learn_en;
bool vtep_vlan_aware;
bool user_bridge_name_provided;
char *user_bridge_name;
static unixctl_cb_func ovs_vswitchd_exit;

static char *parse_options(int argc, char *argv[], char **unixctl_path);        
OVS_NO_RETURN static void usage(void);                                          
                                                                                
struct ovs_vswitchd_exit_args {                                                 
    bool *exiting;                                                              
    bool *cleanup;                                                              
};

#define OFP_VERSION   0x01

int
main(int argc, char *argv[])
{
    struct unixctl_server *unixctl;
    char *remote;
    bool exiting, cleanup;
    int retval;
    char *unixctl_path = NULL;
    struct ovs_vswitchd_exit_args exit_args = {&exiting, &cleanup};

    set_program_name(argv[0]);
    ovs_cmdl_proctitle_init(argc, argv);
    service_start(&argc, &argv);
    remote = parse_options(argc, argv, &unixctl_path);
    process_init();
    //vteprec_init();

    daemonize_start(true);

    retval = unixctl_server_create(unixctl_path, &unixctl);
    if (retval) {
        exit(EXIT_FAILURE);
    }
    unixctl_command_register("exit", "[--cleanup]", 0, 1,
                             ovs_vswitchd_exit, &exit_args);

    hwvtep_init(remote);
    free(remote);

    exiting = false;
    cleanup = false;
    while (!exiting) {
        hwvtep_run();
        unixctl_server_run(unixctl);

        hwvtep_wait();
        unixctl_server_wait(unixctl);
        if (exiting) {
            poll_immediate_wake();
        }
        poll_block();
    }
    unixctl_server_destroy(unixctl);
    service_stop();

    return 0;
}

static char *
parse_options(int argc, char *argv[], char **unixctl_path)
{
    enum {
        OPT_PEER_CA_CERT = UCHAR_MAX + 1,
        OPT_MLOCKALL,
        OPT_UNIXCTL,
        VLOG_OPTION_ENUMS,
        SSL_OPTION_ENUMS,
        OPT_BOOTSTRAP_CA_CERT,
        OPT_ENABLE_DUMMY,
        OPT_DISABLE_SYSTEM,
        OPT_ENABLE_MAC_LEARNING,
        OPT_ENABLE_VLAN_AWARE_MODE,
        OPT_BRIDGE_NAME,
        DAEMON_OPTION_ENUMS
    };
    static struct option long_options[] = {
        {"help",        no_argument, NULL, 'h'},
        {"version",     no_argument, NULL, 'V'},
        {"mlockall",    no_argument, NULL, OPT_MLOCKALL},
        {"unixctl",     required_argument, NULL, OPT_UNIXCTL},
        DAEMON_LONG_OPTIONS,
        VLOG_LONG_OPTIONS,
        STREAM_SSL_LONG_OPTIONS,
        {"peer-ca-cert", required_argument, NULL, OPT_PEER_CA_CERT},
        {"bootstrap-ca-cert", required_argument, NULL, OPT_BOOTSTRAP_CA_CERT},
        {"enable-dummy", no_argument, NULL, OPT_ENABLE_DUMMY},
        {"disable-system", no_argument, NULL, OPT_DISABLE_SYSTEM},
        {"enable-mac-learning", no_argument, NULL, OPT_ENABLE_MAC_LEARNING},
        {"enable-vlan-aware-mode", no_argument, NULL, OPT_ENABLE_VLAN_AWARE_MODE},
        {"bridge", required_argument, NULL, OPT_BRIDGE_NAME},
        {NULL, 0, NULL, 0},
    };
    char *short_options = ovs_cmdl_long_options_to_short_options(long_options);

    vtep_mac_learn_en = false;
    vtep_vlan_aware = false;
    for (;;) {
        int c;

        c = getopt_long(argc, argv, short_options, long_options, NULL);
        if (c == -1) {
            break;
        }

        switch (c) {
        case 'h':
            usage();

        case 'V':
            ovs_print_version(OFP_VERSION, OFP_VERSION);
            exit(EXIT_SUCCESS);

        case OPT_MLOCKALL:
#ifdef HAVE_MLOCKALL
            if (mlockall(MCL_CURRENT | MCL_FUTURE)) {
                VLOG_ERR("mlockall failed: %s", strerror(errno));
            }
#else
            VLOG_ERR("mlockall not supported on this system");
#endif
            break;

       case OPT_UNIXCTL:                                           
            *unixctl_path = optarg;
            break;

        VLOG_OPTION_HANDLERS
        DAEMON_OPTION_HANDLERS
        STREAM_SSL_OPTION_HANDLERS

        case OPT_PEER_CA_CERT:
            stream_ssl_set_peer_ca_cert_file(optarg);
            break;
        
        case OPT_BOOTSTRAP_CA_CERT:
            stream_ssl_set_ca_cert_file(optarg, true);
            break;

        case OPT_ENABLE_DUMMY:
            dummy_enable(optarg);
            break;

        case OPT_DISABLE_SYSTEM:
            dp_blacklist_provider("system");
            break;

        case OPT_ENABLE_MAC_LEARNING:
            vtep_mac_learn_en = true;
            break;

        case OPT_ENABLE_VLAN_AWARE_MODE:
            vtep_vlan_aware = true;
            break;

        case OPT_BRIDGE_NAME:
            user_bridge_name_provided = true;
            user_bridge_name = optarg;
            break;

        case '?':
            exit(EXIT_FAILURE);

        default:
            abort();
        }
    }
    free(short_options);

    argc -= optind;
    argv += optind;

    switch (argc) {
    case 0:
        return xasprintf("unix:%s/db.sock", ovs_rundir());

    case 1:
        return xstrdup(argv[0]);

    default:
        VLOG_FATAL("at most one non-option argument accepted; "
                   "use --help for usage");
    }
}

static void
usage(void)
{
    printf("%s: Open vSwitch daemon\n"
           "usage: %s [OPTIONS] [DATABASE]\n"
           "where DATABASE is a socket on which ovsdb-server is listening\n"
           "      (default: \"unix:%s/db.sock\").\n",
           program_name, program_name, ovs_rundir());
    stream_usage("DATABASE", true, false, true);
    daemon_usage();
    vlog_usage();
    printf("\nOther options:\n"
           "  -h, --help              display this help message\n"
           "  -V, --version           display version information\n");
    exit(EXIT_SUCCESS);
}

static void                                                                     
ovs_vswitchd_exit(struct unixctl_conn *conn, int argc,                          
                  const char *argv[], void *exit_args_)                         
{                                                                               
    struct ovs_vswitchd_exit_args *exit_args = exit_args_;                      
    *exit_args->exiting = true;                                                 
    *exit_args->cleanup = argc == 2 && !strcmp(argv[1], "--cleanup");           
    unixctl_command_reply(conn, NULL);                                          
}
