/*********************************************************************
 * Copyright 2015 Cumulus Networks, LLC.  All rights reserved.
 *
 * library file used by clients for sending commands and parsing response
 *
 */
#include <config.h>
#include <errno.h>
#include <inttypes.h>
#include <stdlib.h>
#include <unistd.h>
#include "util.h"
#include "openvswitch/vlog.h"
#include "clag.h"

VLOG_DEFINE_THIS_MODULE(ovsdb_clag);

struct remote_info {
    char *proto;
    char *ip;
    char *port;
    char *normalized;
};

char peerlink[20] = {0};
char peerlink_name[20] = {0};
char peer_clag_ip[20] = {0};
char *anycast_ip = NULL;

bool clag_status(bool *clag_enabled, bool *peeralive)
{
    FILE *clagfp = NULL;
    size_t lsize = 256;
    char *line = NULL;
    char cmd[80], rmcmd[80], fname[80];
    *peeralive = false;
    bool roleisprimary = false;
    bool backupactive = true;

    *clag_enabled = false;

    /* see if clag is supposed to run */
    clagfp = fopen("/etc/default/clagd", "r");
    if (!clagfp)
        return true;

    while ((getline(&line, &lsize, clagfp) != -1)) {
        if (strstr(line, "CLAGD_ENABLE=")) {
            char *ans = strchr(line, '=') + 1;
            while (*ans == ' ' || *ans == '"')
                ans++;

            if (ans[0] == 'y' || ans[0] == 'Y')
                *clag_enabled = true;

            free(line);
            line = NULL;
            break;
        }
        free(line);
        line = NULL;
    }
    if (line) {
        free(line);
        line = NULL;
    }
    fclose(clagfp);

    if (!*clag_enabled) {
        VLOG_DBG("clag not enabled\n");
        return true;
    }

    /* see if clagd thinks we should run */
    snprintf(fname, 80, "/tmp/clagstatus.%d", getpid());
    snprintf(cmd, 80, "/usr/bin/clagctl > %s", fname);
    snprintf(rmcmd, 80, "/bin/rm %s", fname);
    if (system(cmd) != 0) {
        VLOG_DBG("cannot get clagd status\n");
        system(rmcmd);
        return false;
    }

    clagfp = fopen(fname, "r");
    if (!clagfp) {
        VLOG_ERR("cannot get clagd status\n");
        system(rmcmd);
        return true;
    }

    while ((getline(&line, &lsize, clagfp) != -1)) {
        if (strstr(line, "The peer is")) {
            if (strstr(line, "is alive"))
                *peeralive = true;
        }
        if (strstr(line, "Our Priority")) {
            if (strstr(line, "primary"))
                roleisprimary = true;
        }
        if (strstr(line, "Peer Interface and IP")) {
            char *t1 = strchr(line, ':');
            peerlink[0] = '\0';
            peerlink_name[0] = '\0';
            if (t1) {
                char *t2;
                t1 += 2;
                t2 = strtok(t1, " ");
                if (t2) {
                    char *t3 = strtok(NULL, " ");
                    strncpy(peerlink, t2, 16);
                    char *t4 = strtok(t2, ".");
                    strncpy(peerlink_name, t4, 16);
                    if (t3) {
                        strncpy(peer_clag_ip, t3, 16);
                        VLOG_DBG("peer clag ip: %s\n", peer_clag_ip);
                    }
                }
            }
        }
        if (strstr(line, "VxLAN Anycast IP:")) {
            char *t2 = NULL;
            char *t1 = strchr(line, ':');
            t1 += 2;

            t2 = strtok(t1, "\n");
            if (t2) {
                if (anycast_ip)
                    free(anycast_ip);
                anycast_ip = xstrdup(t2);
                VLOG_DBG("vxlan anycast ip: %s\n", anycast_ip);
            }
        }
        if (strstr(line, "Backup IP:")) {
            if (strstr(line, "inactive"))
                backupactive = false;
        }
        free(line);
        line = NULL;
    }
    if (line) {
        free(line);
        line = NULL;
    }
    fclose(clagfp);
    system(rmcmd);

    if (roleisprimary) {
        VLOG_DBG("we're primary\n");
        return true;
    }
    else {
        VLOG_DBG("we're secondary\n");
        return false;
    }

    VLOG_DBG("peer likely dead\n");
    return true;
}


void get_clag_bonds(struct shash *b2c, struct shash *c2b)
{
    FILE *fp;
    size_t lsize = 256;
    char *line = NULL;
    char *cmd = "/usr/bin/clagctl showclagid > /tmp/clagid.vtep";

    system(cmd);
    fp = fopen("/tmp/clagid.vtep", "r");
    if (!fp) {
        VLOG_ERR("cannot get clagid info\n");
        return;
    }

    while ((getline(&line, &lsize, fp) != -1)) {
        char *bond, *id, *l;
        char clag_id[256];
        if (strstr(line, "Our Interface") || strstr(line, "--------") ||
            strstr(line, "CLAG Interfaces")) {
            free(line);
            line = NULL;
            continue;
        }

        l = strtok(line, "\n");
        bond = strtok(l, " ");
        id = strtok(NULL, " ");

        if (bond && id) {
            sprintf(clag_id, "clag-%s", id);
            shash_add(b2c, bond, xstrdup(clag_id));
            shash_add(c2b, clag_id, xstrdup(bond));
            VLOG_DBG("Found clag bond %s, id %s\n", bond, clag_id);
        }

        free(line);
        line = NULL;
    }
    if (line) {
        free(line);
        line = NULL;
    }
    fclose(fp);
    system("rm /tmp/clagid.vtep");
}

static struct remote_info *parse_remote_ip(char *remote)
{
    char d[256], e[256], *tok1, *tok2, *tok3, *ip, *port;
    struct remote_info *ri;

    if (!remote)
        return NULL;

    strcpy(d, remote);
    tok1 = strtok(d, ":");
    tok2 = strtok(NULL, ":");
    tok3 = strtok(NULL, ":");

    if (!tok1 || !tok2 || !tok3)
        return NULL;

    if (!strcmp(tok1, "ptcp")) {
        tok1 = "tcp";
        ip   = tok3;
        port = tok2;
    } else if (!strcmp(tok1, "pssl")) {
        tok1 = "ssl";
        ip   = tok3;
        port = tok2;
    } else {
        ip   = tok2;
        port = tok3;
    }
    sprintf(e, "%s:%s:%s", tok1, ip, port);
    ri = xzalloc(sizeof(*ri));
    ri->proto = xstrdup(tok1);
    ri->ip    = xstrdup(ip);
    ri->port  = xstrdup(port);
    ri->normalized = xstrdup(e);

    return ri;
}

static void free_remote_info(struct remote_info *ri)
{
    if (ri) {
        if (ri->proto)
            free(ri->proto);
        if (ri->ip)
            free(ri->ip);
        if (ri->port)
            free(ri->port);
        if (ri->normalized)
            free(ri->normalized);
        free(ri);
    }
}

void update_mgmt_internal_addr(char *remote, bool active, bool clag_enabled)
{
    char cmd[256];
    struct remote_info *ri;

    if (remote) {
        ri = parse_remote_ip(remote);
        if (ri) {
            snprintf(cmd, 256, "ip addr %s %s dev %s",
                     (active && clag_enabled)?"add":"del", ri->ip, peerlink);
            system(cmd);
            VLOG_DBG(">> %s\n", cmd);
            free_remote_info(ri);
        }
    }
}

void backup_db_from_remote(char *remote, char *path)
{
    char cmd[256];
    struct remote_info *ri;

    if (!remote)
        return;

    ri = parse_remote_ip(remote);
    if (ri) {
        snprintf(cmd, 256, "/usr/bin/ovsdb-backup --target %s --db %s",
                 path, ri->normalized);
        VLOG_DBG("%s\n", cmd);
        system(cmd);
        free_remote_info(ri);
    }
}

void update_mgmt_external_redirect(char *remote, bool active)
{
    struct remote_info *ri;

    if (!remote)
        return;

    ri = parse_remote_ip(remote);
    if (ri) {
        char cmd[256], *action = NULL;
        int absent = 0;

        snprintf(cmd, 256,
            "iptables -t mangle -C INPUT -d %s -p tcp -j TEE --gateway %s",
            ri->ip, peer_clag_ip);
        absent = system(cmd);

        if (active && !absent)
            action = "-D";
        if (!active && absent)
            action = "-A";

        if (action) {
            snprintf(cmd, 256,
                "iptables -t mangle %s INPUT -d %s -p tcp -j TEE --gateway %s",
                action, ri->ip, peer_clag_ip);
            VLOG_DBG(">> %s\n", cmd);
            system(cmd);

            snprintf(cmd, 256,
                "iptables -t mangle %s INPUT -d %s -p tcp -j DROP",
                action, ri->ip);
            VLOG_DBG(">> %s\n", cmd);
            system(cmd);
        }
        free_remote_info(ri);
    }
}

void update_bfd_redirect(char *remote, bool db_active)
{
    if (!remote)
        return;

    if (remote) {
        char cmd[256], *action = NULL;
        int absent = 0;

        snprintf(cmd, 256,
            "iptables -t mangle -C INPUT -s %s -p udp --dport 4789 -j TEE --gateway %s", remote, peer_clag_ip);
        absent = system(cmd);

        if (db_active && !absent) {
            action = "-D";
        }
        if (!db_active && absent) {
            action = "-A";
        }
        if (action) {
            snprintf(cmd, 256,
                "iptables -t mangle %s INPUT -s %s -p udp --dport 4789 -j TEE --gateway %s", action, remote, peer_clag_ip);
            VLOG_DBG(">> %s\n", cmd);
            system(cmd);

            /* If we install the drop, BUM packets don't get forwarded locally
             * on a soft node.  The redirected copy to the peer will also be
             * blocked for dually connected hosts by egress_mask.
             *
             * If we don't install the drop, singly connected hosts on the
             * peer side may get two copies of a BUM packet.
             *
            snprintf(cmd, 256,
                "iptables -t mangle %s INPUT -s %s -p udp --dport 4789 -j DROP", action, remote);
            VLOG_DBG(">> %s\n", cmd);
            system(cmd);
             */
        }
    }
}

char *get_clag_vxlan_anycast_ip(void)
{
    return anycast_ip;
}

char *get_clag_vxlan_peerlink_name(void)
{
    return peerlink_name;
}

char *get_clag_vxlan_peerlink(void)
{
    return peerlink;
}

char *get_clag_vxlan_peer_clag_ip(void)
{
    return peer_clag_ip;
}
