#include <errno.h>
#include <stdarg.h>
#include <stdbool.h>

#include <netlink/route/route.h>
#include <netlink/object.h>

uint32_t sock_portid = 0;
static void _vpferrno(int errnum, bool do_errnum, const char* format, va_list ap)
{
	fprintf(stderr, "Error: ");
	vfprintf(stderr, format, ap);
	if (do_errnum) {
		char buf[256];
		fprintf(stderr, ": %s", strerror_r(errnum, buf, sizeof(buf)));
	}
	fprintf(stderr, "\n");

	exit(1);
}


static void pferrno(int errnum, const char* format, ...)
{
	va_list ap;

	va_start(ap, format);
	_vpferrno(errnum, true, format, ap);
	va_end(ap);
}


static void pferror(const char* format, ...)
{
	va_list ap;

	va_start(ap, format);
	_vpferrno(0, false, format, ap);
	va_end(ap);
}


static int event_input(struct nl_msg *msg, void *arg)
{
	struct nl_sock *sock = (struct nl_sock *)arg;
	struct nlmsghdr *nlh = nlmsg_hdr(msg);
	int protocol = nlmsg_get_proto(msg);
	struct nl_dump_params dp = {
		.dp_type = NL_DUMP_LINE,
		.dp_fd = stdout,
	};
	struct rtnl_route *route;
	uint32_t flags;
	int err;

	if (protocol != NETLINK_ROUTE || nlh->nlmsg_type != RTM_NEWROUTE) {
		return NL_SKIP;
	}

	if ((err = rtnl_route_parse(nlh, &route)) < 0) {
		pferror("Unable to parse route message: %s",
			nl_geterror(err));
	}

	nl_object_dump(OBJ_CAST(route), &dp);

	if (nl_object_get_msgpid(OBJ_CAST(route)) == sock_portid)
		printf("notification due to our own msg to kernel\n");


	flags = rtnl_route_get_flags(route);
	if (flags & RTM_F_OFFLOAD) {
		rtnl_route_unset_flags(route, RTM_F_OFFLOAD);
		rtnl_route_set_flags(route, RTM_F_OFFLOAD_FAILED);
		if ((err = rtnl_route_set_hw_flags(sock, route)) < 0) {
			pferror("Unable to set flags: %s",
				nl_geterror(err));
		}
	}

	rtnl_route_put(route);
	return NL_OK;
}


int main(int argc, char *argv[])
{
	struct nl_sock *mon_sock, *cmd_sock;
	int err;

	if (argc != 1) {
		fprintf(stderr, "Usage: %s\n", argv[0]);
		exit(2);
	}

	if (!(cmd_sock = nl_socket_alloc())) {
		pferrno(ENOBUFS, "Unable to allocate netlink command socket");
	}
	if ((err = nl_connect(cmd_sock, NETLINK_ROUTE)) < 0) {
		pferror("Unable to connect netlink command socket: %s",
			nl_geterror(err));
	}

	if (!(mon_sock = nl_socket_alloc())) {
		pferrno(ENOBUFS, "Unable to allocate netlink monitor socket");
	}
	nl_socket_disable_seq_check(mon_sock);
	nl_socket_modify_cb(mon_sock, NL_CB_VALID, NL_CB_CUSTOM, &event_input,
			    cmd_sock);

	if ((err = nl_connect(mon_sock, NETLINK_ROUTE)) < 0) {
		pferror("Unable to connect netlink monitor socket: %s",
			nl_geterror(err));
	}
	if ((err = nl_socket_add_membership(mon_sock, RTNLGRP_IPV4_ROUTE)) < 0) {
		pferror("Unable to join ipv4-route group: %s",
			nl_geterror(err));
	}
	if ((err = nl_socket_add_membership(mon_sock, RTNLGRP_IPV6_ROUTE)) < 0) {
		pferror("Unable to join ipv6-route group: %s",
			nl_geterror(err));
	}
	sock_portid = nl_socket_get_local_port(cmd_sock);

	while (1) {
		nl_recvmsgs_default(mon_sock);
	}

	nl_socket_free(mon_sock);
	nl_socket_free(cmd_sock);

	return 0;
}
