#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <netlink/netlink.h>
#include <netlink/route/link.h>
#include <netlink/route/link/bridge.h>
#include <netlink/route/link/vxlan.h>


static void pferror(const int errsv, const char* format, ...)
{
	va_list ap;
	char buf[256];

	fprintf(stderr, "Error: ");

	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
	fprintf(stderr, ": %s\n", strerror_r(errsv, buf, sizeof(buf)));

	exit(1);
}


void dump_bridge_link(struct rtnl_link *link)
{
	if (rtnl_link_bridge_get_flags(link) & RTNL_BRIDGE_VLAN_TUNNEL) {
		struct rtnl_link_bridge_vlan_tunnel *tunnel;
		struct nl_list_head *vlan_list;

		printf("\tvlan_tunnel on\n");

		if (!(vlan_list = rtnl_link_bridge_get_port_vlan_tunnels(link)))
			pferror(EINVAL, "Unable to get vlan info");

		nl_list_for_each_entry(tunnel, vlan_list, rbvt_list)
			printf("\tvlan id %u tunnel id %u\n", tunnel->rbvt_vid,
			       tunnel->rbvt_tunid);
	} else {
		printf("\tvlan_tunnel off\n");
	}
}


void dump_link(struct nl_object *link_obj, void *arg)
{
	struct rtnl_link *link = (struct rtnl_link *)link_obj;

	if (rtnl_link_is_vxlan(link)) {
		struct rtnl_link *match, *needle;
		struct nl_cache *cache = arg;
		uint32_t vni;
		int err;

		printf("link %s\n", rtnl_link_get_name(link));

		if ((err = rtnl_link_vxlan_get_id(link, &vni)) < 0) {
			pferror(err, "Unable to get vni: %s",
				nl_geterror(err));
		}
		printf("\tvxlan id %u\n", vni);

		if (!(needle = rtnl_link_alloc()))
			pferror(ENOMEM, "Unable to allocate needle link");
		rtnl_link_set_family(needle, AF_BRIDGE);
		rtnl_link_set_ifindex(needle, rtnl_link_get_ifindex(link));

		if ((match = (struct rtnl_link *)nl_cache_find(
					cache, OBJ_CAST(needle)))) {
			dump_bridge_link(match);
			rtnl_link_put(match);
		}
		rtnl_link_put(needle);
	}
}


int main(int argc, char *argv[])
{
	struct nl_cache_mngr *mngr;
	struct rtnl_link *filter;
	struct nl_cache *cache;
	struct nl_sock *sock;
	int err;

	if (argc != 1) {
		fprintf(stderr, "Usage: %s\n", argv[0]);
		exit(2);
	}

	/* mngr initialization is modelled after switchd/netlink.c:nl_init() */
	if (!(sock = nl_socket_alloc()))
		pferror(ENOBUFS, "Unable to allocate netlink socket");

	if ((err = nl_cache_alloc_name("route/link", &cache)) < 0) {
		pferror(err, "Unable to allocate link cache: %s",
			nl_geterror(err));
	}
	nl_cache_set_flags(cache, NL_CACHE_AF_ITER);

	if ((err = nl_cache_mngr_alloc(sock, NETLINK_ROUTE, NL_AUTO_PROVIDE,
				       &mngr)) < 0) {
		pferror(err, "Unable to allocate cache manager: %s",
			nl_geterror(err));
	}
	if ((err = nl_cache_mngr_add_cache(mngr, cache, NULL, NULL)) < 0) {
		pferror(err, "Unable to add link cache: %s",
			nl_geterror(err));
	}

	if (!(filter = rtnl_link_alloc()))
		pferror(ENOMEM, "Unable to allocate needle link");
	rtnl_link_set_family(filter, AF_UNSPEC);
	nl_cache_foreach_filter(cache, OBJ_CAST(filter), dump_link, cache);
	rtnl_link_put(filter);

	nl_cache_mngr_free(mngr);
	nl_socket_free(sock);

	return 0;
}
