/*
 * lib/route/nexthop.c	Routing Nexthop
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation version 2.1
 *	of the License.
 *
 * Copyright (c) 2003-2008 Thomas Graf <tgraf@suug.ch>
 *
 * nexthop object support by:
 * Copyright (c) 2019 David Ahern <dsahern@gmail.com>
 */

/**
 * @ingroup route_obj
 * @defgroup nexthop Nexthop
 * @{
 */

#include <linux/nexthop.h>
#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/utils.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/route.h>
#include <netlink/route/lwt.h>
#include <netlink/route/lwt/api.h>
#include <netlink/route/nexthop.h>

/** @cond SKIP */
#define NH_ATTR_FLAGS   0x000001
#define NH_ATTR_WEIGHT  0x000002
#define NH_ATTR_IFINDEX 0x000004
#define NH_ATTR_GATEWAY 0x000008
#define NH_ATTR_REALMS  0x000010
#define NH_ATTR_ENCAP   0x000020
#define NH_ATTR_NEWDST  0x000040
#define NH_ATTR_FAMILY  0x000080
#define NH_ATTR_PROTO   0x000100
#define NH_ATTR_ID      0x000200
#define NH_ATTR_BLKHOLE 0x000400
#define NH_ATTR_GROUP   0x000800
#define NH_ATTR_FDB     0x100000

static struct nl_object_ops nexthop_obj_ops;
static struct nl_cache_ops rtnl_nexthop_ops;
/** @endcond */

/**
 * @name Allocation/Freeing
 * @{
 */

struct rtnl_nexthop *rtnl_route_nh_alloc(void)
{
	return (struct rtnl_nexthop *) nl_object_alloc(&nexthop_obj_ops);
}

static int rtnl_route_nh_copy(struct rtnl_nexthop *src,
			      struct rtnl_nexthop *nh)
{
	nh->rtnh_flags = src->rtnh_flags;
	nh->rtnh_flag_mask = src->rtnh_flag_mask;
	nh->rtnh_weight = src->rtnh_weight;
	nh->rtnh_ifindex = src->rtnh_ifindex;
	nh->ce_mask = src->ce_mask;

	if (src->rtnh_gateway) {
		nh->rtnh_gateway = nl_addr_clone(src->rtnh_gateway);
		if (!nh->rtnh_gateway)
			return -1;
	}

	if (src->rtnh_lwt_encap) {
		nh->rtnh_lwt_encap = rtnl_lwt_clone_encap(src->rtnh_lwt_encap);
		if (!nh->rtnh_lwt_encap)
			return -1;
	}

	/* nexthop as a separate object */
	nh->rtnh_family = src->rtnh_family;
	nh->rtnh_protocol = src->rtnh_protocol;
	nh->rtnh_id = src->rtnh_id;
	nh->rtnh_blackhole = src->rtnh_blackhole;

	nh->rtnh_num_grp = src->rtnh_num_grp;
	if (src->rtnh_num_grp) {
		struct rtnl_nh_grp *p;
		int len = src->rtnh_num_grp * sizeof(*p);

		p = malloc(len);
		if (!p)
			return -1;

		memcpy(p, src->rtnh_grp, len);
		nh->rtnh_grp = p;
	}

	return 0;
}

struct rtnl_nexthop *rtnl_route_nh_clone(struct rtnl_nexthop *src)
{
	struct rtnl_nexthop *nh;

	nh = rtnl_route_nh_alloc();
	if (nh) {
		if (rtnl_route_nh_copy(src, nh)) {
			rtnl_route_nh_free(nh);
			nh = NULL;
		}
	}

	return nh;
}

void rtnl_route_nh_data_free(struct rtnl_nexthop *nh)
{
	nl_addr_put(nh->rtnh_gateway);
	rtnl_lwt_release_encap(nh->rtnh_lwt_encap);
	free(nh->rtnh_label);
	free(nh->rtnh_grp);
}

void rtnl_route_nh_free(struct rtnl_nexthop *nh)
{
	rtnl_route_nh_data_free(nh);
	free(nh);
}

static void nexthop_constructor(struct nl_object *c)
{
	struct rtnl_nexthop *nh = (struct rtnl_nexthop *) c;

	nl_init_list_head(&nh->rtnh_list);
}

static void nexthop_free_data(struct nl_object *c)
{
	struct rtnl_nexthop *nh = nl_object_priv(c);

	if (!nh)
		return;

	rtnl_route_nh_data_free(nh);
}

static int nexthop_clone(struct nl_object *_dst, struct nl_object *_src)
{
        struct rtnl_nexthop *dst = nl_object_priv(_dst);
        struct rtnl_nexthop *src = nl_object_priv(_src);

	return rtnl_route_nh_copy(src, dst);
}

void rtnl_nexthop_get(struct rtnl_nexthop *nh)
{
	nl_object_get((struct nl_object *) nh);
}

void rtnl_nexthop_put(struct rtnl_nexthop *nh)
{
	nl_object_put((struct nl_object *) nh);
}

/** @} */

uint64_t rtnl_route_nh_id_attrs()
{
	return (NH_ATTR_IFINDEX | NH_ATTR_GATEWAY);
}

int rtnl_route_nh_compare(struct rtnl_nexthop *a, struct rtnl_nexthop *b,
			  uint32_t attrs, int loose)
{
	int diff = 0;

#define NH_DIFF(ATTR, EXPR) ATTR_DIFF(attrs, NH_ATTR_##ATTR, a, b, EXPR)

	diff |= NH_DIFF(IFINDEX,	a->rtnh_ifindex != b->rtnh_ifindex);
	diff |= NH_DIFF(WEIGHT,		a->rtnh_weight != b->rtnh_weight);
	diff |= NH_DIFF(REALMS,		a->rtnh_realms != b->rtnh_realms);
	diff |= NH_DIFF(GATEWAY,	nl_addr_cmp(a->rtnh_gateway,
						    b->rtnh_gateway));
	diff |= NH_DIFF(ENCAP,		rtnl_lwt_compare_encap(
							a->rtnh_lwt_encap,
							b->rtnh_lwt_encap));

	if (loose)
		diff |= NH_DIFF(FLAGS,
			  (a->rtnh_flags ^ b->rtnh_flags) & b->rtnh_flag_mask);
	else
		diff |= NH_DIFF(FLAGS, a->rtnh_flags != b->rtnh_flags);

	diff |= NH_DIFF(NEWDST,		a->rtnh_num_labels != b->rtnh_num_labels);

	diff |= NH_DIFF(FAMILY,	a->rtnh_family != b->rtnh_family);
	diff |= NH_DIFF(PROTO,	a->rtnh_protocol != b->rtnh_protocol);
	diff |= NH_DIFF(ID,	a->rtnh_id != b->rtnh_id);
	diff |= NH_DIFF(BLKHOLE, a->rtnh_blackhole != b->rtnh_blackhole);
	diff |= NH_DIFF(GROUP,	a->rtnh_num_grp != b->rtnh_num_grp);
#undef NH_DIFF

	if (attrs & NH_ATTR_GROUP && !(diff & NH_ATTR_GROUP)) {
		int i;

		/* The previous "NH_DIFF(GROUP, ..." statement implies that
		 * a->rtnh_num_grp == b->rtnh_num_grp
		 */
		for (i = 0; i < a->rtnh_num_grp; ++i) {
			if (a->rtnh_grp[i].id != b->rtnh_grp[i].id ||
			    a->rtnh_grp[i].weight != b->rtnh_grp[i].weight) {
				diff |= NH_ATTR_GROUP;
				break;
			}
		}
	}

	if (attrs & NH_ATTR_GROUP && !(diff & NH_ATTR_NEWDST)) {
		int i;

		/* The previous "NH_DIFF(NEWDST, ..." statement implies that
		 * a->rtnh_num_labels == b->rtnh_num_labels
		 */
		for (i = 0; i < a->rtnh_num_labels; i++) {
			if (a->rtnh_label[i] != b->rtnh_label[i]) {
				diff |= NH_ATTR_NEWDST;
				break;
			}
		}

	}

	return diff;
}

static uint64_t nexthop_compare(struct nl_object *_a, struct nl_object *_b,
				uint64_t attrs, int flags)
{
	struct rtnl_nexthop *a = (struct rtnl_nexthop *) _a;
	struct rtnl_nexthop *b = (struct rtnl_nexthop *) _b;

	return rtnl_route_nh_compare(a, b, attrs, 0);
}

static void nexthop_keygen(struct nl_object *obj, uint32_t *hashkey,
			   uint32_t table_sz)
{
	struct rtnl_nexthop *nh = (struct rtnl_nexthop *) obj;

	*hashkey = nh->rtnh_id % table_sz;

	NL_DBG(5, "nexthop %p key (id %u), hash 0x%x\n", nh, nh->rtnh_id,
	       *hashkey);

	return;
}

static void nh_dump_line(struct rtnl_nexthop *nh, struct nl_dump_params *dp)
{
	struct nl_cache *link_cache;
	char buf[128];

	link_cache = nl_cache_mngt_require_safe("route/link");

	if (nh->ce_mask & NH_ATTR_ENCAP)
		rtnl_lwt_dump_encap(nh->rtnh_lwt_encap, dp);

	nl_dump(dp, "via");

	if (nh->ce_mask & NH_ATTR_ID)
		nl_dump(dp, " nhid %u", nh->rtnh_id);

	if (nh->ce_mask & NH_ATTR_BLKHOLE)
		nl_dump(dp, " blackhole");

	if (nh->ce_mask & NH_ATTR_GATEWAY)
		nl_dump(dp, " %s", nl_addr2str(nh->rtnh_gateway,
						   buf, sizeof(buf)));

	if (nh->ce_mask & NH_ATTR_FAMILY)
		nl_dump(dp, " family %d", nh->rtnh_family);

	if(nh->ce_mask & NH_ATTR_IFINDEX) {
		if (link_cache) {
			nl_dump(dp, " dev %s",
				rtnl_link_i2name(link_cache,
						 nh->rtnh_ifindex,
						 buf, sizeof(buf)));
		} else
			nl_dump(dp, " dev %d", nh->rtnh_ifindex);
	}

	nl_dump(dp, " ");

	if (link_cache)
		nl_cache_put(link_cache);
}

static void nh_dump_details(struct rtnl_nexthop *nh, struct nl_dump_params *dp)
{
	struct nl_cache *link_cache;
	char buf[128];
	int i;

	link_cache = nl_cache_mngt_require_safe("route/link");

	nl_dump(dp, "nexthop");

	if (nh->ce_mask & NH_ATTR_ENCAP)
		rtnl_lwt_dump_encap(nh->rtnh_lwt_encap, dp);

	if (nh->ce_mask & NH_ATTR_PROTO)
		nl_dump(dp, " protocol %s",
			rtnl_route_proto2str(nh->rtnh_protocol,
					     buf, sizeof(buf)));

	if (nh->ce_mask & NH_ATTR_GROUP) {
		int i;

		for (i = 0; i < nh->rtnh_num_grp; ++i) {
			if (i == 0)
				nl_dump(dp, " ");
			else
				nl_dump(dp, "/");
			nl_dump(dp, "%u", nh->rtnh_grp[i].id);
			if (nh->rtnh_grp[i].weight)
				nl_dump(dp, ",%u", nh->rtnh_grp[i].weight);
		}
	}

	if (nh->ce_mask & NH_ATTR_GATEWAY)
		nl_dump(dp, " via %s", nl_addr2str(nh->rtnh_gateway,
						   buf, sizeof(buf)));

	if(nh->ce_mask & NH_ATTR_IFINDEX) {
		if (link_cache) {
			nl_dump(dp, " dev %s",
				rtnl_link_i2name(link_cache,
						 nh->rtnh_ifindex,
						 buf, sizeof(buf)));
		} else
			nl_dump(dp, " dev %d", nh->rtnh_ifindex);
	}

	if (nh->ce_mask & NH_ATTR_WEIGHT)
		nl_dump(dp, " weight %u", nh->rtnh_weight);

	if (nh->ce_mask & NH_ATTR_REALMS)
		nl_dump(dp, " realm %04x:%04x",
			RTNL_REALM_FROM(nh->rtnh_realms),
			RTNL_REALM_TO(nh->rtnh_realms));

	if (nh->ce_mask & NH_ATTR_FLAGS)
		nl_dump(dp, " <%s>", rtnl_route_nh_flags2str(nh->rtnh_flags,
							buf, sizeof(buf)));

	if (nh->rtnh_num_labels) {
		nl_dump(dp, " label");
		for (i = 0; i < nh->rtnh_num_labels; i++)
			nl_dump(dp, " %d", nh->rtnh_label[i]);
	}

	if (link_cache)
		nl_cache_put(link_cache);
}

void rtnl_route_nh_dump(struct rtnl_nexthop *nh, struct nl_dump_params *dp)
{
	switch (dp->dp_type) {
	case NL_DUMP_LINE:
		nh_dump_line(nh, dp);
		break;

	case NL_DUMP_DETAILS:
	case NL_DUMP_STATS:
		if (dp->dp_ivar == NH_DUMP_FROM_DETAILS)
			nh_dump_details(nh, dp);
		break;

	default:
		break;
	}
}

static void nexthop_dump_line(struct nl_object *o, struct nl_dump_params *dp)
{
	struct rtnl_nexthop *nh = nl_object_priv(o);

	nh_dump_line(nh, dp);

	if (dp->dp_type == NL_DUMP_DETAILS ||
	    dp->dp_type == NL_DUMP_STATS)
		nh_dump_details(nh, dp);

	nl_dump(dp, "\n");
}

/**
 * @name Attributes
 * @{
 */

void rtnl_route_nh_set_weight(struct rtnl_nexthop *nh, uint8_t weight)
{
	nh->rtnh_weight = weight;
	nh->ce_mask |= NH_ATTR_WEIGHT;
}

uint8_t rtnl_route_nh_get_weight(struct rtnl_nexthop *nh)
{
	return nh->rtnh_weight;
}

void rtnl_route_nh_set_ifindex(struct rtnl_nexthop *nh, int ifindex)
{
	nh->rtnh_ifindex = ifindex;
	nh->ce_mask |= NH_ATTR_IFINDEX;
}

int rtnl_route_nh_get_ifindex(struct rtnl_nexthop *nh)
{
	return nh->rtnh_ifindex;
}	

/* FIXME: Convert to return an int */
void rtnl_route_nh_set_gateway(struct rtnl_nexthop *nh, struct nl_addr *addr)
{
	struct nl_addr *old = nh->rtnh_gateway;

	if (addr) {
		nh->rtnh_gateway = nl_addr_get(addr);
		nh->ce_mask |= NH_ATTR_GATEWAY;
	} else {
		nh->ce_mask &= ~NH_ATTR_GATEWAY;
		nh->rtnh_gateway = NULL;
	}

	if (old)
		nl_addr_put(old);
}

struct nl_addr *rtnl_route_nh_get_gateway(struct rtnl_nexthop *nh)
{
	return nh->rtnh_gateway;
}

void rtnl_route_nh_set_flags(struct rtnl_nexthop *nh, unsigned int flags)
{
	nh->rtnh_flag_mask |= flags;
	nh->rtnh_flags |= flags;
	nh->ce_mask |= NH_ATTR_FLAGS;
}

void rtnl_route_nh_unset_flags(struct rtnl_nexthop *nh, unsigned int flags)
{
	nh->rtnh_flag_mask |= flags;
	nh->rtnh_flags &= ~flags;
	nh->ce_mask |= NH_ATTR_FLAGS;
}

unsigned int rtnl_route_nh_get_flags(struct rtnl_nexthop *nh)
{
	return nh->rtnh_flags;
}

void rtnl_route_nh_set_realms(struct rtnl_nexthop *nh, uint32_t realms)
{
	nh->rtnh_realms = realms;
	nh->ce_mask |= NH_ATTR_REALMS;
}

uint32_t rtnl_route_nh_get_realms(struct rtnl_nexthop *nh)
{
	return nh->rtnh_realms;
}

int rtnl_route_nh_get_labels(struct rtnl_nexthop *nh,
			      uint32_t *num_labels,
			      uint32_t **labels)
{
	uint32_t *tmp;
	int i;

	if (!nh->rtnh_num_labels) {
		*num_labels = 0;
		*labels = NULL;
		return 0;
	}

	tmp = malloc(sizeof(uint32_t) * nh->rtnh_num_labels);
	if (!tmp)
		return -ENOMEM;

	for (i = 0; i < nh->rtnh_num_labels; i++)
		tmp[i] = nh->rtnh_label[i];

	*num_labels = nh->rtnh_num_labels;
	*labels = tmp;

	return 0;
}

int rtnl_route_nh_set_labels(struct rtnl_nexthop *nh, uint32_t num_labels,
			      uint32_t *labels)
{
	int i;

	nh->rtnh_num_labels = 0;
	free(nh->rtnh_label);

	nh->rtnh_label = malloc(sizeof(uint32_t) * num_labels);
	if (!nh->rtnh_label)
		return -ENOMEM;

	nh->rtnh_num_labels = num_labels;
	for (i = 0; i < num_labels; i++)
		nh->rtnh_label[i] = labels[i];

	nh->ce_mask |= NH_ATTR_NEWDST;

	return 0;
}

void rtnl_route_nh_set_lwt_encap(struct rtnl_nexthop *nh,
				 struct rtnl_lwt_encap *en)
{
	nh->rtnh_lwt_encap = en;
	nh->ce_mask |= NH_ATTR_ENCAP;
}

struct rtnl_lwt_encap *rtnl_route_nh_get_lwt_encap(struct rtnl_nexthop *nh)
{
	return nh->rtnh_lwt_encap;
}

void rtnl_route_nh_set_family(struct rtnl_nexthop *nh, uint8_t family)
{
	nh->rtnh_family = family;
	if (family)
		nh->ce_mask |= NH_ATTR_FAMILY;
	else
		nh->ce_mask &= ~NH_ATTR_FAMILY;
}

uint8_t rtnl_route_nh_get_family(struct rtnl_nexthop *nh)
{
	return nh->rtnh_family;
}

void rtnl_route_nh_set_proto(struct rtnl_nexthop *nh, uint8_t proto)
{
	nh->rtnh_protocol = proto;
	if (proto)
		nh->ce_mask |= NH_ATTR_PROTO;
	else
		nh->ce_mask &= ~NH_ATTR_PROTO;
}

uint8_t rtnl_route_nh_get_proto(struct rtnl_nexthop *nh)
{
	return nh->rtnh_protocol;
}

void rtnl_route_nh_set_id(struct rtnl_nexthop *nh, uint32_t id)
{
	nh->rtnh_id = id;
	if (id)
		nh->ce_mask |= NH_ATTR_ID;
	else
		nh->ce_mask &= ~NH_ATTR_ID;
}

uint32_t rtnl_route_nh_get_id(struct rtnl_nexthop *nh)
{
	return nh->rtnh_id;
}

void rtnl_route_nh_set_blackhole(struct rtnl_nexthop *nh, uint8_t arg)
{
	if (arg) {
		nh->ce_mask |= NH_ATTR_BLKHOLE;
		nh->rtnh_blackhole = 1;
	} else {
		nh->ce_mask &= ~NH_ATTR_BLKHOLE;
		nh->rtnh_blackhole = 0;
	}
}

int rtnl_route_nh_get_blackhole(struct rtnl_nexthop *nh)
{
	return nh->rtnh_blackhole;
}

int rtnl_route_nh_get_fdb(struct rtnl_nexthop *nh)
{
	return nh->rtnh_fdb;
}

int rtnl_route_nh_set_group(struct rtnl_nexthop *nh, uint32_t num,
			    struct rtnl_nh_grp *entries)
{
	struct rtnl_nh_grp *p = NULL;

	if ((num && !entries) || (!num && entries))
		return -NLE_INVAL;

	/* TO-DO: attempts to set these if group is set should fail */
	if (nh->ce_mask & (NH_ATTR_IFINDEX | NH_ATTR_GATEWAY |
			   NH_ATTR_REALMS | NH_ATTR_ENCAP))
		return -NLE_INVAL;

	if (num) {
		int len = num * sizeof(*p);

		p = malloc(len);
		if (!p)
			return -NLE_NOMEM;
		memcpy(p, entries, len);
	}

	if (nh->rtnh_grp)
		free(nh->rtnh_grp);

	if (num)
		nh->ce_mask |= NH_ATTR_GROUP;
	else
		nh->ce_mask &= ~NH_ATTR_GROUP;

	nh->rtnh_num_grp = num;
	nh->rtnh_grp = p;

	return 0;
}

int rtnl_route_nh_get_group(struct rtnl_nexthop *nh, uint32_t *num,
			    struct rtnl_nh_grp **entries)
{
	struct rtnl_nh_grp *p;
	int len;

	if (!(nh->ce_mask & NH_ATTR_GROUP))
		return -NLE_INVAL;

	len = nh->rtnh_num_grp * sizeof(*p);
	p = malloc(len);
	if (!p)
		return -NLE_NOMEM;

	memcpy(p, nh->rtnh_grp, len);

	*num = nh->rtnh_num_grp;
	*entries = p;

	return 0;
}

int rtnl_route_nh_put_group(struct rtnl_nexthop *nh,
         struct rtnl_nh_grp *entries)
{
    if (!(nh->ce_mask & NH_ATTR_GROUP))
        return -NLE_INVAL;

    if (!entries)
        return -NLE_INVAL;

    free(entries);

    return 0;
}

void rtnl_route_nh_set_fdb(struct rtnl_nexthop *nh, uint8_t arg)
{
	if (arg) {
		nh->ce_mask |= NH_ATTR_FDB;
		nh->rtnh_fdb = 1;
	} else {
		nh->ce_mask &= ~NH_ATTR_FDB;
		nh->rtnh_fdb = 0;
	}
}

/** @} */

/**
 * @name Nexthop Flags Translations
 * @{
 */

static const struct trans_tbl nh_flags[] = {
	__ADD(RTNH_F_DEAD, dead),
	__ADD(RTNH_F_PERVASIVE, pervasive),
	__ADD(RTNH_F_ONLINK, onlink),
};

char *rtnl_route_nh_flags2str(int flags, char *buf, size_t len)
{
	return __flags2str(flags, buf, len, nh_flags, ARRAY_SIZE(nh_flags));
}

int rtnl_route_nh_str2flags(const char *name)
{
	return __str2flags(name, nh_flags, ARRAY_SIZE(nh_flags));
}

/** @} */

/**
 * @name Cache Management
 * @{
 */

/**
 * Build a nexthop cache including all nexthops currently configured in the kernel.
 * @arg sock            Netlink socket.
 * @arg family          Address family or AF_UNSPEC.
 * @arg result          Pointer to store resulting cache.
 *
 * Allocates a new nexhtop cache, initializes it properly and updates it
 * to include all nexthops currently configured in the kernel.
 *
 * @return 0 on success or a negative error code.
 */
int rtnl_nexthop_alloc_cache(struct nl_sock *sock, int family,
			     struct nl_cache **result)
{
	struct nl_cache * cache;
	int err;

	if (!(cache = nl_cache_alloc(&rtnl_nexthop_ops)))
		return -NLE_NOMEM;

	cache->c_iarg1 = family;

	if (sock && (err = nl_cache_refill(sock, cache)) < 0) {
		free(cache);
		return err;
	}

	*result = cache;
	return 0;
}

/** @} */

static int nexthop_request_update(struct nl_cache *c, struct nl_sock *h)
{
	return nl_nhgen_request(h, RTM_GETNEXTHOP, AF_UNSPEC, NLM_F_DUMP);
}

static struct nla_policy nexthop_policy[NHA_MAX+1] = {
	[NHA_ID]		= { .type = NLA_U32 },
	[NHA_GROUP_TYPE]	= { .type = NLA_U8  },
	[NHA_BLACKHOLE]		= { .type = NLA_FLAG },
	[NHA_OIF]		= { .type = NLA_U32 },
	[NHA_ENCAP_TYPE]	= { .type = NLA_U16 },
	[NHA_FDB]               = { .type = NLA_FLAG }
};

int rtnl_nexthop_parse(struct nlmsghdr *nlh, struct rtnl_nexthop **result)
{
	struct nlattr *tb[NHA_MAX+1];
	struct rtnl_lwt_encap *encap;
	struct rtnl_nh_grp *rtnl_grp;
	int err = -NLE_INVAL, i, j;
	struct rtnl_nexthop *nh;
	struct nexthop_grp *nhg;
	struct nl_addr *addr;
	struct nhmsg *nhm;
	unsigned int len;

	nh = rtnl_route_nh_alloc();
	if (!nh)
		return -NLE_NOMEM;

	nh->ce_msgtype = nlh->nlmsg_type;
	nh->ce_msgflags = nlh->nlmsg_flags;

	err = nlmsg_parse(nlh, sizeof(*nhm), tb, NHA_MAX, nexthop_policy);
	if (err < 0)
		goto errout;

	nhm = nlmsg_data(nlh);
	rtnl_route_nh_set_family(nh, nhm->nh_family);
	rtnl_route_nh_set_proto(nh, nhm->nh_protocol);
	rtnl_route_nh_set_flags(nh, nhm->nh_flags);

	for (i = 1; i < NHA_MAX + 1; ++i) {
		if (!tb[i])
			continue;

		switch(i) {
		case NHA_ID:
			rtnl_route_nh_set_id(nh, nla_get_u32(tb[i]));
			break;
		case NHA_GROUP:
			nhg = nla_data(tb[i]);
			len = nla_len(tb[i]);
			if (len & (sizeof(struct nexthop_grp) - 1)) {
				err = -NLE_INVAL;
				goto errout;
			}
			len /= sizeof(*nhg);
			if (len == 0)
				break;
			rtnl_grp = calloc(len, sizeof(*rtnl_grp));
			if (!rtnl_grp) {
				err = -ENOMEM;
				goto errout;
			}
			for (j = 0; j < len; ++j) {
				rtnl_grp[j].id = nhg[j].id;
				rtnl_grp[j].weight = nhg[j].weight;
			}
			err = rtnl_route_nh_set_group(nh, len, rtnl_grp);
			free(rtnl_grp);
			if (err)
				goto errout;
			break;
		case NHA_GROUP_TYPE:
			if (nla_get_u16(tb[i]) != NEXTHOP_GRP_TYPE_MPATH) {
				err = -NLE_INVAL;
				goto errout;
			}
			break;
		case NHA_BLACKHOLE:
			rtnl_route_nh_set_blackhole(nh, 1);
			break;
		case NHA_FDB:
			rtnl_route_nh_set_fdb(nh, 1);
			break;
		case NHA_OIF:
			rtnl_route_nh_set_ifindex(nh, nla_get_u32(tb[i]));
			break;
		case NHA_GATEWAY:
			addr = nl_addr_alloc_attr(tb[i], nhm->nh_family);

			if (!addr) {
			    err = -NLE_NOMEM;
			    goto errout;
			}

			rtnl_route_nh_set_gateway(nh, addr);
			nl_addr_put(addr);
			break;
		case NHA_ENCAP:
			if (!tb[NHA_ENCAP_TYPE]) {
				err = -NLE_INVAL;
				goto errout;
			}
			if (rtnl_lwt_build_encap(tb[i],
						nla_get_u16(tb[NHA_ENCAP_TYPE]),
						&encap) < 0) {
				err = -NLE_INVAL;
				goto errout;
			}
			rtnl_route_nh_set_lwt_encap(nh, encap);
			break;
		}
	}

	err = 0;
	*result = nh;
errout:
	if (err)
		rtnl_nexthop_put(nh);
	return err;
}

static int nexthop_msg_parser(struct nl_cache_ops *ops, struct sockaddr_nl *who,
			      struct nlmsghdr *n, struct nl_parser_param *pp)
{
	struct rtnl_nexthop *nh;
	int err;

	if ((err = rtnl_nexthop_parse(n, &nh)) < 0)
		return err;

	err = pp->pp_cb((struct nl_object *) nh, pp);

	rtnl_nexthop_put(nh);
	return err;
}

static struct nl_object_ops nexthop_obj_ops = {
	.oo_name		= "route/nexthop",
	.oo_size		= sizeof(struct rtnl_nexthop),
	.oo_constructor		= nexthop_constructor,
	.oo_free_data		= nexthop_free_data,
	.oo_clone		= nexthop_clone,
	.oo_dump = {
		[NL_DUMP_LINE]		= nexthop_dump_line,
		[NL_DUMP_DETAILS]	= nexthop_dump_line,
		[NL_DUMP_STATS]		= nexthop_dump_line,
	},
	.oo_compare		= nexthop_compare,
	.oo_keygen		= nexthop_keygen,
	.oo_id_attrs		= (NH_ATTR_ID),
};

static struct nl_af_group nexthop_groups[] = {
	{ AF_UNSPEC,    RTNLGRP_NEXTHOP },
	{ AF_INET,      RTNLGRP_NEXTHOP },
	{ AF_INET6,     RTNLGRP_NEXTHOP },
	{ END_OF_GROUP_LIST },
};

static struct nl_cache_ops rtnl_nexthop_ops = {
	.co_name		= "route/nexthop",
	.co_hdrsize		= sizeof(struct nhmsg),
	.co_msgtypes		= {
					{ RTM_NEWNEXTHOP, NL_ACT_NEW, "new" },
					{ RTM_DELNEXTHOP, NL_ACT_DEL, "del" },
					{ RTM_GETNEXTHOP, NL_ACT_GET, "get" },
					END_OF_MSGTYPES_LIST,
				},
	.co_protocol		= NETLINK_ROUTE,
	.co_request_update	= nexthop_request_update,
	.co_msg_parser		= nexthop_msg_parser,
	.co_obj_ops		= &nexthop_obj_ops,
	.co_groups		= nexthop_groups,
	.co_hash_size           = 16384,
};

static void __init nexthop_init(void)
{
	nl_cache_mngt_register(&rtnl_nexthop_ops);
}

static void __exit nexthop_exit(void)
{
	nl_cache_mngt_unregister(&rtnl_nexthop_ops);
}

/** @} */
