/*
 * lib/route/mroute_obj.c	Multicast Route Object
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation version 2.1
 *	of the License.
 *
 * Copyright (c) 2016-2017 Roopa Prabhu <roopa@cumulusnetworks.com>
 */

/**
 * @ingroup route
 * @defgroup mroute_obj Route Object
 *
 * @par Attributes
 * @code
 * Name                                           Default
 * -------------------------------------------------------------
 * routing table                                  RT_TABLE_MAIN
 * scope                                          RT_SCOPE_NOWHERE
 * tos                                            0
 * protocol                                       RTPROT_STATIC
 * prio                                           0
 * family                                         AF_UNSPEC
 * type                                           RTN_UNICAST
 * iif                                            NULL
 * @endcode
 *
 * @{
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/cache.h>
#include <netlink/utils.h>
#include <netlink/data.h>
#include <netlink/hashtable.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/route.h>
#include <netlink/route/mroute.h>
#include <netlink/route/link.h>
#include <netlink/route/nexthop.h>

/** @cond SKIP */
#define ROUTE_ATTR_FAMILY    0x000001
#define ROUTE_ATTR_TOS       0x000002
#define ROUTE_ATTR_TABLE     0x000004
#define ROUTE_ATTR_PROTOCOL  0x000008
#define ROUTE_ATTR_SCOPE     0x000010
#define ROUTE_ATTR_TYPE      0x000020
#define ROUTE_ATTR_FLAGS     0x000040
#define ROUTE_ATTR_DST       0x000080
#define ROUTE_ATTR_SRC       0x000100
#define ROUTE_ATTR_IIF       0x000200
#define ROUTE_ATTR_OIF       0x000400
#define ROUTE_ATTR_GATEWAY   0x000800
#define ROUTE_ATTR_PRIO      0x001000
#define ROUTE_ATTR_PREF_SRC  0x002000
#define ROUTE_ATTR_METRICS   0x004000
#define ROUTE_ATTR_MULTIPATH 0x008000
#define ROUTE_ATTR_REALMS    0x010000
#define ROUTE_ATTR_CACHEINFO 0x020000
/** @endcond */

static void mroute_constructor(struct nl_object *c)
{
	struct rtnl_route *r = (struct rtnl_route *) c;

	r->rt_family = RTNL_FAMILY_IPMR;
	r->rt_scope = RT_SCOPE_UNIVERSE;
	r->rt_table = RT_TABLE_DEFAULT;
	r->rt_protocol = RTPROT_STATIC;
	r->rt_type = RTN_MULTICAST;
	r->rt_prio = 0;

	nl_init_list_head(&r->rt_nexthops);
}

static void mroute_free_data(struct nl_object *c)
{
	struct rtnl_route *r = (struct rtnl_route *) c;
	struct rtnl_nexthop *nh, *tmp;

	if (r == NULL)
		return;

	nl_addr_put(r->rt_dst);
	nl_addr_put(r->rt_src);
	nl_addr_put(r->rt_pref_src);

	nl_list_for_each_entry_safe(nh, tmp, &r->rt_nexthops, rtnh_list) {
		rtnl_route_remove_nexthop(r, nh);
		rtnl_route_nh_free(nh);
	}
}

static int mroute_clone(struct nl_object *_dst, struct nl_object *_src)
{
	struct rtnl_route *dst = (struct rtnl_route *) _dst;
	struct rtnl_route *src = (struct rtnl_route *) _src;
	struct rtnl_nexthop *nh, *new;

	if (src->rt_dst)
		if (!(dst->rt_dst = nl_addr_clone(src->rt_dst)))
			return -NLE_NOMEM;

	if (src->rt_src)
		if (!(dst->rt_src = nl_addr_clone(src->rt_src)))
			return -NLE_NOMEM;

	if (src->rt_pref_src)
		if (!(dst->rt_pref_src = nl_addr_clone(src->rt_pref_src)))
			return -NLE_NOMEM;

	/* Will be inc'ed again while adding the nexthops of the source */
	dst->rt_nr_nh = 0;

	nl_init_list_head(&dst->rt_nexthops);
	nl_list_for_each_entry(nh, &src->rt_nexthops, rtnh_list) {
		new = rtnl_route_nh_clone(nh);
		if (!new)
			return -NLE_NOMEM;

		rtnl_route_add_nexthop(dst, new);
	}

	return 0;
}

static void mroute_dump_line(struct nl_object *a, struct nl_dump_params *p)
{
	struct rtnl_route *r = (struct rtnl_route *) a;
	int cache = 0, flags;
	char buf[64];

	if (r->rt_flags & RTM_F_CLONED)
		cache = 1;

	nl_dump_line(p, "%s ", nl_af2str(r->rt_family, buf, sizeof(buf)));

	if (cache)
		nl_dump(p, "cache ");

	if (!(r->ce_mask & ROUTE_ATTR_DST) ||
	    nl_addr_get_len(r->rt_dst) == 0)
		nl_dump(p, "default ");
	else
		nl_dump(p, "%s ", nl_addr2str(r->rt_dst, buf, sizeof(buf)));

	if (r->ce_mask & ROUTE_ATTR_TABLE && !cache)
		nl_dump(p, "table %s ",
			rtnl_route_table2str(r->rt_table, buf, sizeof(buf)));

	if (r->ce_mask & ROUTE_ATTR_TYPE)
		nl_dump(p, "type %s ",
			nl_rtntype2str(r->rt_type, buf, sizeof(buf)));

	if (r->ce_mask & ROUTE_ATTR_TOS && r->rt_tos != 0)
		nl_dump(p, "tos %#x ", r->rt_tos);

	if (r->ce_mask & ROUTE_ATTR_MULTIPATH) {
		struct rtnl_nexthop *nh;

		nl_list_for_each_entry(nh, &r->rt_nexthops, rtnh_list) {
			p->dp_ivar = NH_DUMP_FROM_ONELINE;
			rtnl_route_nh_dump(nh, p);
		}
	}

	flags = r->rt_flags & ~(RTM_F_CLONED);
	if (r->ce_mask & ROUTE_ATTR_FLAGS && flags) {

		nl_dump(p, "<");

#define PRINT_FLAG(f) if (flags & RTNH_F_##f) { \
		flags &= ~RTNH_F_##f; nl_dump(p, #f "%s", flags ? "," : ""); }
		PRINT_FLAG(DEAD);
		PRINT_FLAG(ONLINK);
		PRINT_FLAG(PERVASIVE);
#undef PRINT_FLAG

#define PRINT_FLAG(f) if (flags & RTM_F_##f) { \
		flags &= ~RTM_F_##f; nl_dump(p, #f "%s", flags ? "," : ""); }
		PRINT_FLAG(NOTIFY);
		PRINT_FLAG(EQUALIZE);
		PRINT_FLAG(PREFIX);
		PRINT_FLAG(OFFLOAD);
		PRINT_FLAG(TRAP);
#undef PRINT_FLAG

#define PRINT_FLAG(f) if (flags & RTCF_##f) { \
		flags &= ~RTCF_##f; nl_dump(p, #f "%s", flags ? "," : ""); }
		PRINT_FLAG(NOTIFY);
		PRINT_FLAG(REDIRECTED);
		PRINT_FLAG(DOREDIRECT);
		PRINT_FLAG(DIRECTSRC);
		PRINT_FLAG(DNAT);
		PRINT_FLAG(BROADCAST);
		PRINT_FLAG(MULTICAST);
		PRINT_FLAG(LOCAL);
#undef PRINT_FLAG

		nl_dump(p, ">");
	}

	nl_dump(p, "\n");
}

static void mroute_dump_details(struct nl_object *a, struct nl_dump_params *p)
{
	struct rtnl_route *r = (struct rtnl_route *) a;
	struct nl_cache *link_cache;
	char buf[256];
	int i;

	link_cache = nl_cache_mngt_require_safe("route/link");

	mroute_dump_line(a, p);
	nl_dump_line(p, "    ");

	if (r->ce_mask & ROUTE_ATTR_PREF_SRC)
		nl_dump(p, "preferred-src %s ",
			nl_addr2str(r->rt_pref_src, buf, sizeof(buf)));

	if (r->ce_mask & ROUTE_ATTR_SCOPE && r->rt_scope != RT_SCOPE_NOWHERE)
		nl_dump(p, "scope %s ",
			rtnl_scope2str(r->rt_scope, buf, sizeof(buf)));

	if (r->ce_mask & ROUTE_ATTR_PRIO)
		nl_dump(p, "priority %#x ", r->rt_prio);

	if (r->ce_mask & ROUTE_ATTR_PROTOCOL)
		nl_dump(p, "protocol %s ",
			rtnl_route_proto2str(r->rt_protocol, buf, sizeof(buf)));

	if (r->ce_mask & ROUTE_ATTR_IIF) {
		if (link_cache) {
			nl_dump(p, "iif %s ",
				rtnl_link_i2name(link_cache, r->rt_iif,
						 buf, sizeof(buf)));
		} else
			nl_dump(p, "iif %d ", r->rt_iif);
	}

	if (r->ce_mask & ROUTE_ATTR_SRC)
		nl_dump(p, "src %s ", nl_addr2str(r->rt_src, buf, sizeof(buf)));

	nl_dump(p, "\n");

	if (r->ce_mask & ROUTE_ATTR_MULTIPATH) {
		struct rtnl_nexthop *nh;

		nl_list_for_each_entry(nh, &r->rt_nexthops, rtnh_list) {
			nl_dump_line(p, "    ");
			p->dp_ivar = NH_DUMP_FROM_DETAILS;
			rtnl_route_nh_dump(nh, p);
			nl_dump(p, "\n");
		}
	}

	if ((r->ce_mask & ROUTE_ATTR_CACHEINFO) && r->rt_cacheinfo.rtci_error) {
		nl_dump_line(p, "    cacheinfo error %d (%s)\n",
			r->rt_cacheinfo.rtci_error,
			strerror_r(-r->rt_cacheinfo.rtci_error, buf, sizeof(buf)));
	}

	if (r->ce_mask & ROUTE_ATTR_METRICS) {
		nl_dump_line(p, "    metrics [");
		for (i = 0; i < RTAX_MAX; i++)
			if (r->rt_metrics_mask & (1 << i))
				nl_dump(p, "%s %u ",
					rtnl_route_metric2str(i+1,
							      buf, sizeof(buf)),
					r->rt_metrics[i]);
		nl_dump(p, "]\n");
	}

	if (link_cache)
		nl_cache_put(link_cache);
}

static void mroute_dump_stats(struct nl_object *obj, struct nl_dump_params *p)
{
	struct rtnl_route *route = (struct rtnl_route *) obj;

	mroute_dump_details(obj, p);

#if 0
	if (route->ce_mask & ROUTE_ATTR_MFC_STATS) {
		/* XXX */
		struct rta_mfc_stats *mfcs = &route->rt_mfc_stats;

		nl_dump_line(p, "    used %u refcnt %u last-use %us "
				"expires %us\n",
			     ci->rtci_used, ci->rtci_clntref,
			     ci->rtci_last_use / nl_get_user_hz(),
			     ci->rtci_expires / nl_get_user_hz());
	}
#endif
}

static uint32_t mroute_id_attrs_get(struct nl_object *obj)
{
	struct nl_object_ops *ops = obj->ce_ops;

	if (!ops)
		BUG();

	return ops->oo_id_attrs;
}

static void mroute_keygen(struct nl_object *obj, uint32_t *hashkey,
			  uint32_t table_sz)
{
	struct rtnl_route *route = (struct rtnl_route *) obj;
	unsigned int rkey_sz;
	struct nl_addr *dst = NULL;
	struct nl_addr *src = NULL;
	char *addr_ptr = NULL;
	struct route_hash_key {
		uint8_t		rt_family;
		uint8_t		rt_tos;
		uint32_t	rt_table;
		uint32_t	rt_prio;
		char 		rt_addrs[0];
	} __attribute__((packed)) *rkey;
#ifdef NL_DEBUG
	char buf[INET6_ADDRSTRLEN+5];
#endif

	if (route->rt_dst)
		dst = route->rt_dst;
	if (route->rt_src)
		src = route->rt_src;

	rkey_sz = sizeof(*rkey);
	if (dst)
		rkey_sz += nl_addr_get_len(dst);
	if (src)
		rkey_sz += nl_addr_get_len(src);

	rkey = calloc(1, rkey_sz);
	if (!rkey) {
		NL_DBG(2, "Warning: calloc failed for %d bytes...\n", rkey_sz);
		*hashkey = 0;
		return;
	}
	rkey->rt_family = route->rt_family;
	rkey->rt_tos = route->rt_tos;
	rkey->rt_table = route->rt_table;
	rkey->rt_prio = route->rt_prio;

	addr_ptr = rkey->rt_addrs;
	if (dst) {
		memcpy(addr_ptr, nl_addr_get_binary_addr(dst),
			nl_addr_get_len(dst));
		addr_ptr += nl_addr_get_len(dst);
	}
	if (src)
		memcpy(addr_ptr, nl_addr_get_binary_addr(src),
			nl_addr_get_len(src));

	*hashkey = nl_hash(rkey, rkey_sz, 0) % table_sz;

	NL_DBG(5, "route %p key (fam %d tos %d table %d dst %s src %s)"
	        " keysz %d hash 0x%x\n", route, rkey->rt_family, rkey->rt_tos,
		rkey->rt_table, nl_addr2str(dst, buf, sizeof(buf)),
		nl_addr2str(src, buf, sizeof(buf)),
		rkey_sz, *hashkey);

	free(rkey);

	return;
}

static uint64_t mroute_compare(struct nl_object *_a, struct nl_object *_b,
			      uint64_t attrs, int flags)
{
	struct rtnl_route *a = (struct rtnl_route *) _a;
	struct rtnl_route *b = (struct rtnl_route *) _b;
	struct rtnl_nexthop *nh_a, *nh_b;
	int i, found;
	uint64_t diff = 0;
	uint64_t nh_attrs;
	struct nl_list_head *nh_a_head;
	struct nl_list_head *nh_b_head;

#define ROUTE_DIFF(ATTR, EXPR) ATTR_DIFF(attrs, ROUTE_ATTR_##ATTR, a, b, EXPR)

	diff |= ROUTE_DIFF(FAMILY,	a->rt_family != b->rt_family);
	diff |= ROUTE_DIFF(TOS,		a->rt_tos != b->rt_tos);
	diff |= ROUTE_DIFF(TABLE,	a->rt_table != b->rt_table);
	diff |= ROUTE_DIFF(PROTOCOL,	a->rt_protocol != b->rt_protocol);
	diff |= ROUTE_DIFF(SCOPE,	a->rt_scope != b->rt_scope);
	diff |= ROUTE_DIFF(TYPE,	a->rt_type != b->rt_type);
	diff |= ROUTE_DIFF(PRIO,	a->rt_prio != b->rt_prio);
	diff |= ROUTE_DIFF(DST,		nl_addr_cmp(a->rt_dst, b->rt_dst));
	diff |= ROUTE_DIFF(SRC,		nl_addr_cmp(a->rt_src, b->rt_src));
	diff |= ROUTE_DIFF(IIF,		a->rt_iif != b->rt_iif);
	diff |= ROUTE_DIFF(PREF_SRC,	nl_addr_cmp(a->rt_pref_src,
						    b->rt_pref_src));

	if (flags & LOOSE_COMPARISON) {
		nl_list_for_each_entry(nh_b, &b->rt_nexthops, rtnh_list) {
			found = 0;
			nl_list_for_each_entry(nh_a, &a->rt_nexthops,
					       rtnh_list) {
				if (!rtnl_route_nh_compare(nh_a, nh_b,
							nh_b->ce_mask, 1)) {
					found = 1;
					break;
				}
			}

			if (!found)
				goto nh_mismatch;
		}

		for (i = 0; i < RTAX_MAX - 1; i++) {
			if (a->rt_metrics_mask & (1 << i) &&
			    (!(b->rt_metrics_mask & (1 << i)) ||
			     a->rt_metrics[i] != b->rt_metrics[i]))
				diff |= ROUTE_DIFF(METRICS, 1);
		}

		diff |= ROUTE_DIFF(FLAGS,
			  (a->rt_flags ^ b->rt_flags) & b->rt_flag_mask);
	} else {
		if (a->rt_nr_nh != b->rt_nr_nh)
			goto nh_mismatch;

		if (attrs == ~0)
			nh_attrs = ~0;
		else
			nh_attrs = rtnl_route_nh_id_attrs();

		found = 1;
		nh_a_head = &a->rt_nexthops;
		nh_b_head = &b->rt_nexthops;
		nh_a = nl_list_entry(nh_a_head->next, struct rtnl_nexthop,
				     rtnh_list);
		nh_b = nl_list_entry(nh_b_head->next, struct rtnl_nexthop,
				     rtnh_list);
		while((&nh_a->rtnh_list != nh_a_head) &&
		      (&nh_b->rtnh_list != nh_b_head)) {
			if (rtnl_route_nh_compare(nh_a, nh_b, nh_attrs, 0)) {
				found = 0;
				break;
			}
			nh_a = nl_list_entry(nh_a->rtnh_list.next,
					     struct rtnl_nexthop, rtnh_list);
			nh_b = nl_list_entry(nh_b->rtnh_list.next,
					     struct rtnl_nexthop, rtnh_list);
		}
		if (!found)
			goto nh_mismatch;

		for (i = 0; i < RTAX_MAX - 1; i++) {
			if ((a->rt_metrics_mask & (1 << i)) ^
			    (b->rt_metrics_mask & (1 << i)))
				diff |= ROUTE_DIFF(METRICS, 1);
			else
				diff |= ROUTE_DIFF(METRICS,
					a->rt_metrics[i] != b->rt_metrics[i]);
		}

		diff |= ROUTE_DIFF(FLAGS, a->rt_flags != b->rt_flags);
	}

out:
	return diff;

nh_mismatch:
	diff |= ROUTE_DIFF(MULTIPATH, 1);
	goto out;

#undef ROUTE_DIFF
}

static const struct trans_tbl mroute_attrs[] = {
	__ADD(ROUTE_ATTR_FAMILY, family),
	__ADD(ROUTE_ATTR_TOS, tos),
	__ADD(ROUTE_ATTR_TABLE, table),
	__ADD(ROUTE_ATTR_PROTOCOL, protocol),
	__ADD(ROUTE_ATTR_SCOPE, scope),
	__ADD(ROUTE_ATTR_TYPE, type),
	__ADD(ROUTE_ATTR_FLAGS, flags),
	__ADD(ROUTE_ATTR_DST, dst),
	__ADD(ROUTE_ATTR_SRC, src),
	__ADD(ROUTE_ATTR_IIF, iif),
	__ADD(ROUTE_ATTR_OIF, oif),
	__ADD(ROUTE_ATTR_GATEWAY, gateway),
	__ADD(ROUTE_ATTR_PRIO, prio),
	__ADD(ROUTE_ATTR_PREF_SRC, pref_src),
	__ADD(ROUTE_ATTR_METRICS, metrics),
	__ADD(ROUTE_ATTR_MULTIPATH, multipath),
	__ADD(ROUTE_ATTR_REALMS, realms),
	__ADD(ROUTE_ATTR_CACHEINFO, cacheinfo),
};

static char *mroute_attrs2str(int attrs, char *buf, size_t len)
{
	return __flags2str(attrs, buf, len, mroute_attrs,
			   ARRAY_SIZE(mroute_attrs));
}

/** @} */

/**
 * @name Allocation/Freeing
 * @{
 */

struct rtnl_route *rtnl_mroute_alloc(void)
{
	return (struct rtnl_route *) nl_object_alloc(&mroute_obj_ops);
}

void rtnl_mroute_get(struct rtnl_route *route)
{
	nl_object_get((struct nl_object *) route);
}

void rtnl_mroute_put(struct rtnl_route *route)
{
	nl_object_put((struct nl_object *) route);
}

/**
 * @name Attributes
 * @{
 */

struct nl_object_ops mroute_obj_ops = {
	.oo_name		= "route/mroute",
	.oo_size		= sizeof(struct rtnl_route),
	.oo_constructor		= mroute_constructor,
	.oo_free_data		= mroute_free_data,
	.oo_clone		= mroute_clone,
	.oo_dump = {
	    [NL_DUMP_LINE]	= mroute_dump_line,
	    [NL_DUMP_DETAILS]	= mroute_dump_details,
	    [NL_DUMP_STATS]	= mroute_dump_stats,
	},
	.oo_compare		= mroute_compare,
	.oo_keygen		= mroute_keygen,
	.oo_attrs2str		= mroute_attrs2str,
	.oo_id_attrs		= (ROUTE_ATTR_FAMILY | ROUTE_ATTR_TOS |
				   ROUTE_ATTR_TABLE | ROUTE_ATTR_DST |
				   ROUTE_ATTR_PRIO | ROUTE_ATTR_SRC),
	.oo_id_attrs_get        = mroute_id_attrs_get,
};
/** @endcond */

/** @} */
