/*
 * lib/route/mpls.c	mpls route obj functions
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation version 2.1
 *	of the License.
 *
 * Copyright (c) 2015-2016 Roopa Prabhu <roopa@cumulusnetworks.com>
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/cache.h>
#include <netlink/utils.h>
#include <netlink/data.h>
#include <netlink/hashtable.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/route.h>
#include <linux/mpls.h>

#define MAX_MPLS_LABELS  16

int nla_put_labels(struct nl_msg *msg, int attrtype,
		   uint32_t num_labels, uint32_t *label)
{
	struct nlattr *nla;
	int datalen;

	datalen = num_labels * 4;
	nla = nla_reserve(msg, attrtype, datalen);
	if (!nla)
		return -NLE_NOMEM;

	memcpy(nla_data(nla), label, datalen);

	NL_DBG(2, "msg %p: attr <%p> %d: Wrote %d bytes at offset +%td\n",
	       msg, nla, nla->nla_type, datalen,
	       (void *) nla - nlmsg_data(msg->nm_nlh));

	return 0;
}

int nla_get_labels(int len, const void *data,
		   uint32_t *num_labels, uint32_t **labels)
{
	struct mpls_label *nla_label;
	uint32_t nla_labels;
	uint32_t *tmp;
	int i;

	/* len needs to be an even multiple of 4 (the label size) */
	if (len & 3)
		return -EINVAL;

	/* Limit the number of new labels allowed */
	nla_labels = len/4;
	if (nla_labels > MAX_MPLS_LABELS)
		return -EINVAL;

	tmp = malloc(sizeof(uint32_t) * nla_labels);
	if (!tmp)
		return -ENOMEM;

	/* We only care about the label part. */
	nla_label = (struct mpls_label *)data;
	for (i = 0; i < nla_labels; i++) {
		uint32_t entry;

		entry = ntohl(nla_label[i].entry);
		tmp[i] = (entry & MPLS_LS_LABEL_MASK) >> MPLS_LS_LABEL_SHIFT;
	}

	*num_labels = nla_labels;
	*labels = tmp;

	return 0;
}
