/*
 * lib/route/lwt/mpls.c	mpls lwt operations
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation version 2.1
 *	of the License.
 *
 * Copyright (c) 2015 Roopa Prabhu <roopa@cumulusnetworks.com>
 */

/**
 * @ingroup link
 * @defgroup bridge Bridge
 * Bridge link operations
 *
 * @route_doc{lwt, lwt tunnel Documentation}
 *
 * @{
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/attr.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/mpls.h>
#include <netlink/route/lwt/mpls.h>
#include <netlink/route/lwt/api.h>
#include <linux/mpls_iptunnel.h>
#include <netlink/route/lwt.h>

static struct nla_policy mpls_iptunnel_policy[MPLS_IPTUNNEL_MAX+1] = {
	[MPLS_IPTUNNEL_DST]	= { .type = NLA_U32 },
};

static struct mpls_iptunnel_encap *mpls_encap_hdr(struct rtnl_lwt_encap *encap)
{
	return (struct mpls_iptunnel_encap *)encap->l_hdr;
}

static int mpls_build_encap(struct nlattr *nla, struct rtnl_lwt_encap **encap)
{
	struct rtnl_lwt_encap *newencap;
	struct mpls_iptunnel_encap *mpls_encap;
	struct nlattr *tb[MPLS_IPTUNNEL_MAX+1];
	int ret;

	ret = nla_parse_nested(tb, MPLS_IPTUNNEL_MAX, nla,
			       mpls_iptunnel_policy);
	if (ret < 0)
		return ret;

	if (!tb[MPLS_IPTUNNEL_DST])
		return -EINVAL;

	newencap = calloc(1, sizeof(struct rtnl_lwt_encap)
			  + sizeof(*mpls_encap));
	if (!newencap)
		return -ENOMEM;

	newencap->l_len = sizeof(*mpls_encap);
	mpls_encap = mpls_encap_hdr(newencap);
	ret = nla_get_labels(nla_len(tb[MPLS_IPTUNNEL_DST]),
			     nla_data(tb[MPLS_IPTUNNEL_DST]),
			     &mpls_encap->num_labels, &mpls_encap->labels);
	if (ret)
		goto errout;

	newencap->l_type = LWTUNNEL_ENCAP_MPLS;

	*encap = newencap;

	return 0;

errout:
	free(newencap);
	*encap = NULL;

	return ret;
}

static struct rtnl_lwt_encap *mpls_clone_encap(struct rtnl_lwt_encap *src)
{
	struct mpls_iptunnel_encap *src_hdr = mpls_encap_hdr(src);
	struct mpls_iptunnel_encap *new_hdr;
	struct rtnl_lwt_encap *newencap;
	int l;

	newencap = calloc(1, sizeof(struct rtnl_lwt_encap) + sizeof(*new_hdr));
	if (!newencap)
		return NULL;

	newencap->l_type = LWTUNNEL_ENCAP_MPLS;
	newencap->l_len = sizeof(*new_hdr);

	new_hdr = mpls_encap_hdr(newencap);
	new_hdr->num_labels = src_hdr->num_labels;
	if (!new_hdr->num_labels)
		return newencap;

	new_hdr->labels = malloc(sizeof(uint32_t) * new_hdr->num_labels);
	if (!new_hdr->labels) {
		free(newencap);
		return -ENOMEM;
	}

	for (l = 0; l < src_hdr->num_labels; l++)
		new_hdr->labels[l] = src_hdr->labels[l];

	return newencap;
}


static void mpls_release_encap(struct rtnl_lwt_encap *encap)
{
	struct mpls_iptunnel_encap *e_hdr;

	if (!encap)
		return;

	e_hdr = mpls_encap_hdr(encap);
	free(e_hdr->labels);

	free(encap);
}

static int mpls_compare_encap(struct rtnl_lwt_encap *a,
			      struct rtnl_lwt_encap *b, uint16_t type,
			      uint32_t attrs, int flags)
{
	struct mpls_iptunnel_encap *a_hdr = mpls_encap_hdr(a);
	struct mpls_iptunnel_encap *b_hdr = mpls_encap_hdr(b);
	int diff = 0;
	int l;

	if (a->l_type != b->l_type || a->l_len != b->l_len)
		return 1;

	if (a_hdr->num_labels != b_hdr->num_labels)
		return 1;

	for (l = 0; l < a_hdr->num_labels; l++)
		if (a_hdr->labels[l] != b_hdr->labels[l])
			return 1;
	return diff;
}

static void mpls_dump_encap(struct rtnl_lwt_encap *e,
			    struct nl_dump_params *p)
{
	struct mpls_iptunnel_encap *e_hdr = mpls_encap_hdr(e);
	uint32_t l;

	nl_dump_line(p, " encap mpls ");
	for (l = 0; l < e_hdr->num_labels; l++) {
		if (l == 0)
			nl_dump(p, "%d", e_hdr->labels[l]);
		else
			nl_dump(p, "/%d", e_hdr->labels[l]);
	}
}

int rtnl_lwt_is_mpls(struct rtnl_lwt_encap *e)
{
	return (e->l_type == LWTUNNEL_ENCAP_MPLS);
}

void rtnl_lwt_mpls_get_labels(struct rtnl_lwt_encap *e, uint32_t max_labels,
			      uint32_t *num_labels, uint32_t *labels)
{
	struct mpls_iptunnel_encap *e_hdr;
	uint32_t i;

	*num_labels = 0;
	if (e->l_type != LWTUNNEL_ENCAP_MPLS)
		return;

	e_hdr = mpls_encap_hdr(e);
	if (e_hdr->num_labels > max_labels)
		return;

	for (i = 0; i < e_hdr->num_labels; i++)
		labels[i] = e_hdr->labels[i];

	*num_labels = e_hdr->num_labels;
}

static const struct nla_policy protinfo_policy = {
	.type			= NLA_U8,
};

static struct rtnl_lwt_ops mpls_lwt_ops = {
	.lwt_type		= LWTUNNEL_ENCAP_MPLS,
	.lwt_build_encap	= &mpls_build_encap,
	.lwt_clone_encap	= &mpls_clone_encap,
	.lwt_release_encap	= &mpls_release_encap,
	.lwt_compare_encap	= &mpls_compare_encap,
	.lwt_dump_encap		= &mpls_dump_encap,
};

static void __init mpls_lwt_init(void)
{
	rtnl_lwt_ops_register(&mpls_lwt_ops);
}

static void __exit mpls_lwt_exit(void)
{
	rtnl_lwt_ops_unregister(&mpls_lwt_ops);
}

/** @} */
