/*
 * lib/route/lwt/api.c		lwt ops API
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation version 2.1
 *	of the License.
 *
 * Copyright (c) 2015-2016 Roopa Prabhu <roopa@cumulusnetworks.com>
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/utils.h>
#include <netlink/route/link.h>
#include <netlink/route/lwt.h>
#include <netlink/route/lwt/api.h>

#define LWTUNNEL_OPS_MAX 2
static struct rtnl_lwt_ops *lwt_ops[LWTUNNEL_OPS_MAX];

static struct rtnl_lwt_ops *__rtnl_lwt_ops_lookup(const uint16_t type)
{
	if (type == LWTUNNEL_ENCAP_NONE ||
	    type >= LWTUNNEL_OPS_MAX)
		return NULL;

	return lwt_ops[type];
}

/**
 * @name lwt Modules
 * @{
 */

/**
 * Return operations of a specific lwt type
 * @arg type		lwt type
 *
 * @note The returned pointer must be given back using rtnl_lwt_ops_put()
 *
 * @return Pointer to operations or NULL if unavailable.
 */
struct rtnl_lwt_ops *rtnl_lwt_ops_lookup(const uint16_t type)
{
	struct rtnl_lwt_ops *ops;

	if ((ops = __rtnl_lwt_ops_lookup(type)))
		ops->lwt_refcnt++;

	return ops;
}

/**
 * Give back reference to a set of operations.
 * @arg ops		lwt operations.
 */
void rtnl_lwt_ops_put(struct rtnl_lwt_ops *ops)
{
	if (ops)
		ops->lwt_refcnt--;
}

/**
 * Register operations for a lwt
 * @arg ops		lwt operations
 *
 * This function must be called by modules implementing a specific lwt
 * type. It will make the operations implemented by the module
 * available for everyone else.
 *
 * @return 0 on success or a negative error code.
 * @return -NLE_INVAL type is out of range (0..AF_MAX)
 * @return -NLE_EXIST Operations for lwt type already registered.
 */
int rtnl_lwt_ops_register(struct rtnl_lwt_ops *ops)
{
	if (ops->lwt_type == LWTUNNEL_ENCAP_NONE ||
	    ops->lwt_type >= LWTUNNEL_OPS_MAX)
		return -NLE_INVAL;

	if (lwt_ops[ops->lwt_type])
		return -NLE_EXIST;

	ops->lwt_refcnt = 0;
	lwt_ops[ops->lwt_type] = ops;

	NL_DBG(1, "Registered lwt ops for type %u\n", ops->lwt_type);

	return 0;
}

/**
 * Unregister operations for a lwt type
 * @arg ops		lwt operations
 *
 * This function must be called if a module implementing a specific lwt
 * type is unloaded or becomes unavailable. It must provide a
 * set of operations which have previously been registered using
 * rtnl_lwt_ops_register().
 *
 * @return 0 on success or a negative error code
 * @return -NLE_INVAL ops is NULL
 * @return -NLE_OBJ_NOTFOUND type operations not registered.
 * @return -NLE_BUSY type operations still in use.
 */
int rtnl_lwt_ops_unregister(struct rtnl_lwt_ops *ops)
{
	if (!ops)
		return -NLE_INVAL;

	if (!lwt_ops[ops->lwt_type])
		return -NLE_OBJ_NOTFOUND;

	if (ops->lwt_refcnt > 0)
		return -NLE_BUSY;

	lwt_ops[ops->lwt_type] = NULL;

	NL_DBG(1, "Unregistered lwt operations for type %u\n", ops->lwt_type);

	return 0;
}

/**
 * Compare lwt encap data for a lwt type
 * @arg a		lwt object a
 * @arg b		lwt object b
 * @arg type		lwt data type
 *
 * This function will compare lwt encap data between two lwt encaps
 * a and b of type given by arg type
 *
 * @return 0 if address family specific data matches or is not present
 * or != 0 if it mismatches.
 */
int rtnl_lwt_compare_encap(struct rtnl_lwt_encap *a, struct rtnl_lwt_encap *b)
{
	struct rtnl_lwt_ops *ops;
	int ret = 0;

	if (!a && !b)
		return 0;

	if (!a || !b)
		return ~0;

	if (!a->l_type && !b->l_type)
		return 0;

	if (!a->l_len || !b->l_len)
		return ~0;

	ops = rtnl_lwt_ops_lookup(a->l_type);
	if (!ops)
		return ~0;

	if (ops->lwt_compare_encap == NULL) {
		ret = ~0;
		goto out;
	}

	ret = ops->lwt_compare_encap(a, b, a->l_type, ~0, 0);

out:
	rtnl_lwt_ops_put(ops);

	return ret;
}

/**
 * Dump lwt encap data
 * @arg a		lwt object a
 *
 * This function will dump lwt encap data
 * a and b of type given by arg type
 *
 */
void rtnl_lwt_dump_encap(struct rtnl_lwt_encap *a, struct nl_dump_params *p)

{
	struct rtnl_lwt_ops *ops;

	if (!a)
		return;

	ops = rtnl_lwt_ops_lookup(a->l_type);
	if (!ops)
		return;

	if (ops->lwt_dump_encap)
		ops->lwt_dump_encap(a, p);

	rtnl_lwt_ops_put(ops);

	return;
}

/**
 * Build lwt encap data
 * @arg  nlattr		lwt encap nlattr
 * @arg  encap_type	lwt encap type
 * @arg  encap		return built encap data
 *
 * This function will build lwt encap data
 *
 */
int rtnl_lwt_build_encap(struct nlattr *nlattr, uint16_t encap_type,
			  struct rtnl_lwt_encap **encap)

{
	struct rtnl_lwt_ops *ops;
	int ret = 0;

	ops = rtnl_lwt_ops_lookup(encap_type);
	if (!ops)
		return -NLE_INVAL;

	if (ops->lwt_build_encap)
		ret = ops->lwt_build_encap(nlattr, encap);

	rtnl_lwt_ops_put(ops);

	return ret;
}

struct rtnl_lwt_encap *rtnl_lwt_clone_encap(struct rtnl_lwt_encap *encap)
{
	struct rtnl_lwt_ops *ops;

	if (!encap)
		return NULL;

	ops = rtnl_lwt_ops_lookup(encap->l_type);
	if (ops)
		return ops->lwt_clone_encap(encap);

	return NULL;
}


void rtnl_lwt_release_encap(struct rtnl_lwt_encap *encap)
{
	struct rtnl_lwt_ops *ops;

	if (!encap)
		return;

	ops = rtnl_lwt_ops_lookup(encap->l_type);
	if (ops)
		ops->lwt_release_encap(encap);
}

/** @} */

/** @} */
