/*
 * lib/route/link/mlx_sx_netdev.c      mlx_sx_netdev Link Info
 *
 *     This library is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU Lesser General Public
 *     License as published by the Free Software Foundation version 2.1
 *     of the License.
 *
 * Copyright (c) 2016 Cumulus Networks. All rights reserved.
 * Copyright (c) 2016 Andy Gospodarek <gospo@cumulusnetworks.com>
 * Copyright (c) 2016 Roopa Prabhu <roopa@cumulusnetworks.com>
 * Copyright (c) 2017 Andy Roulin <roopa@cumulusnetworks.com>
 */

/**
 * @ingroup link
 * @defgroup mlx_sx_netdev MLX_SX_NETDEV
 * Mellanox SX_NETDEV
 *
 * @details
 * \b Link Type Name: "mlx_sx_netdev"
 *
 * @route_doc{link_mlx_sx_netdev, MLX_SX_NETDEV Documentation}
 *
 * @{
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/attr.h>
#include <netlink/utils.h>
#include <netlink/object.h>
#include <netlink/route/rtnl.h>
#include <netlink-private/route/link/api.h>
#include <netlink/route/link/mlx_sx_netdev.h>

#include <linux/if_link.h>
#include <linux-private/linux/rtnetlink.h>
#include <linux-private/linux/mlx_sx_netdev.h>

static struct rtnl_link_info_ops sx_netdev_info_ops;

/** @cond SKIP */
#define SX_NETDEV_HAS_UNIT    (1<<0)
#define SX_NETDEV_HAS_PORT    (1<<1)

struct sx_netdev_info {
	uint32_t        unit;
	uint32_t	port;
	uint32_t        mask;
};

/** @endcond */

static struct nla_policy sx_netdev_policy[IFLA_SX_NETDEV_MAX + 1] = {
	[IFLA_SX_NETDEV_UNIT]         = { .type = NLA_U32 },
	[IFLA_SX_NETDEV_PORT]         = { .type = NLA_U32 },

};

/** @cond SKIP */
#define IS_SX_NETDEV_LINK_ASSERT(link) \
	if ((link)->l_info_ops != &sx_netdev_info_ops) { \
		APPBUG("Link is not a SX_NETDEV link. set type \"mlx_sx_netdev\" first."); \
		return -NLE_OPNOTSUPP; \
	}
/** @endcond */

static int sx_netdev_alloc(struct rtnl_link *link)
{
	struct sx_netdev_info *ki;

	if (link->l_info) {
		memset(link->l_info, 0, sizeof (*ki));
		return 0;
	}

	if ((ki = calloc(1, sizeof(*ki))) == NULL)
		return -NLE_NOMEM;

	link->l_info = ki;

	return 0;
}

static int sx_netdev_parse(struct rtnl_link *link, struct nlattr *data,
						   struct nlattr *xstats)
{
	struct nlattr *tb[IFLA_SX_NETDEV_MAX+1];
	struct sx_netdev_info *ki;
	int err;

	NL_DBG(3, "Parsing SX_NETDEV link info");

	if ((err = nla_parse_nested(tb, IFLA_SX_NETDEV_MAX, data, sx_netdev_policy)) < 0)
		goto errout;

	if ((err = sx_netdev_alloc(link)) < 0)
		goto errout;

	ki = link->l_info;

	if (tb[IFLA_SX_NETDEV_UNIT]) {
		ki->unit = nla_get_u32(tb[IFLA_SX_NETDEV_UNIT]);
		ki->mask |= SX_NETDEV_HAS_UNIT;
	}

	if (tb[IFLA_SX_NETDEV_PORT]) {
		ki->port = nla_get_u32(tb[IFLA_SX_NETDEV_PORT]);
		ki->mask |= SX_NETDEV_HAS_PORT;
	}

	err = 0;

errout:
	return err;
}

static void sx_netdev_free(struct rtnl_link *link)
{
	free(link->l_info);
	link->l_info = NULL;
}

static int sx_netdev_clone(struct rtnl_link *dst, struct rtnl_link *src)
{
	struct sx_netdev_info *vdst, *vsrc = src->l_info;
	int err;

	if ((err = rtnl_link_set_type(dst, "mlx_sx_netdev")) < 0)
		return err;
	vdst = dst->l_info;

	BUG_ON(!vdst || !vsrc);

	memcpy(vdst, vsrc, sizeof(struct sx_netdev_info));

	return 0;
}

static int sx_netdev_put_attrs(struct nl_msg *msg, struct rtnl_link *link)
{
	struct sx_netdev_info *kn = link->l_info;
	struct nlattr *data;

	if (!(data = nla_nest_start(msg, IFLA_INFO_DATA)))
		return -NLE_NOMEM;

	if (kn->mask & SX_NETDEV_HAS_UNIT) {
		NLA_PUT_U32(msg, IFLA_SX_NETDEV_UNIT, kn->unit);
	}

	if (kn->mask & SX_NETDEV_HAS_PORT) {
		NLA_PUT_U32(msg, IFLA_SX_NETDEV_PORT, kn->port);
	}

	nla_nest_end(msg, data);

nla_put_failure:
	return 0;
}

static void sx_netdev_dump(struct rtnl_link *link, struct nl_dump_params *p)
{
	struct sx_netdev_info *kn = link->l_info;

	if (kn->mask & SX_NETDEV_HAS_UNIT) {
		nl_dump(p, "t %u", kn->unit);
	}

	if (kn->mask & SX_NETDEV_HAS_PORT) {
		nl_dump(p, "p %u", kn->port);
	}
}

struct rtnl_link *rtnl_link_mlx_sx_netdev_alloc(void)
{
	struct rtnl_link *link;
	int err;

	if (!(link = rtnl_link_alloc()))
		return NULL;

	if ((err = rtnl_link_set_type(link, "mlx_sx_netdev")) < 0) {
		rtnl_link_put(link);
		return NULL;
	}

	return link;
}

/**
 * Get SX_NETDEV unit
 * @arg link           Link object
 * @arg id             Pointer to store unit identifier
 *
 * @return 0 on success or a negative error code
 */
int rtnl_link_mlx_sx_netdev_get_unit(struct rtnl_link *link, uint32_t *unit)
{
	struct sx_netdev_info *ki = link->l_info;

	IS_SX_NETDEV_LINK_ASSERT(link);
	if(!unit)
		return -NLE_INVAL;

	if (ki->mask & SX_NETDEV_HAS_UNIT)
		*unit = ki->unit;
	else
		return -NLE_AGAIN;

	return 0;
}

/**
 * Set SX_NETDEV unit
 * @arg link           Link object
 * @arg unit           Unit associated with SX_NETDEV link
 *
 * @return 0 on success or a negative error code
 */
int rtnl_link_mlx_sx_netdev_set_unit(struct rtnl_link *link, uint32_t unit)
{
	struct sx_netdev_info *ki = link->l_info;

	IS_SX_NETDEV_LINK_ASSERT(link);
	ki->unit = unit;
	ki->mask |= SX_NETDEV_HAS_UNIT;

	return 0;
}

/**
 * Get SX_NETDEV port
 * @arg link           Link object
 * @arg id             Pointer to store port identifier
 *
 * @return 0 on success or a negative error code
 */
int rtnl_link_mlx_sx_netdev_get_port(struct rtnl_link *link, uint32_t *port)
{
	struct sx_netdev_info *ki = link->l_info;

	IS_SX_NETDEV_LINK_ASSERT(link);
	if(!port)
		return -NLE_INVAL;

	if (ki->mask & SX_NETDEV_HAS_PORT)
		*port = ki->port;
	else
		return -NLE_AGAIN;

	return 0;
}

/**
 * Set SX_NETDEV port
 * @arg link           Link object
 * @arg unit           Port associated with SX_NETDEV link
 *
 * @return 0 on success or a negative error code
 */
int rtnl_link_mlx_sx_netdev_set_port(struct rtnl_link *link, uint32_t port)
{
	struct sx_netdev_info *ki = link->l_info;

	IS_SX_NETDEV_LINK_ASSERT(link);
	ki->port = port;
	ki->mask |= SX_NETDEV_HAS_PORT;

	return 0;
}

static struct rtnl_link_info_ops sx_netdev_info_ops = {
	.io_name                = "mlx_sx_netdev",
	.io_alloc               = sx_netdev_alloc,
	.io_parse               = sx_netdev_parse,
	.io_dump = {
		[NL_DUMP_LINE]     = sx_netdev_dump,
		[NL_DUMP_DETAILS]  = sx_netdev_dump,
	},
	.io_clone               = sx_netdev_clone,
	.io_put_attrs           = sx_netdev_put_attrs,
	.io_free                = sx_netdev_free,
};


/**
 * @name SX_NETDEV Object
 * @{
 */

/**
 * Check if link is a SX_NETDEV link
 * @arg link           Link object
 *
 * @return True if link is a SX_NETDEV link, otherwise false is returned.
 */
int rtnl_link_is_mlx_sx_netdev(struct rtnl_link *link)
{
	return link->l_info_ops && !strcmp(link->l_info_ops->io_name, "mlx_sx_netdev");
}

/** @} */

static void __init sx_netdev_init(void)
{
	rtnl_link_register_info(&sx_netdev_info_ops);
}

static void __exit sx_netdev_exit(void)
{
	rtnl_link_unregister_info(&sx_netdev_info_ops);
}

/** @} */
