/*
 * lib/route/link/bridge.c	AF_BRIDGE link support
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation version 2.1
 *	of the License.
 *
 * Copyright (c) 2010-2013 Thomas Graf <tgraf@suug.ch>
 */

/**
 * @ingroup link
 * @defgroup bridge Bridging
 * Bridge link operations
 *
 * @route_doc{link_bridge, Bridge Documentation}
 *
 * @details
 * @{
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/attr.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/link/bridge.h>
#include <netlink-private/route/link/api.h>
#include <linux/if_bridge.h>

#define VLAN_VID_MASK           0x0fff /* VLAN Identifier */
#define VLAN_MAX RTNL_LINK_BRIDGE_VLAN_BITMAP_MAX

/** @cond SKIP */
#define BRIDGE_ATTR_PORT_STATE		(1 << 0)
#define BRIDGE_ATTR_PRIORITY		(1 << 1)
#define BRIDGE_ATTR_COST		(1 << 2)
#define BRIDGE_ATTR_FLAGS		(1 << 3)
#define BRIDGE_ATTR_PORT_VLAN           (1 << 4)
#define BRIDGE_ATTR_BACKUP_PORT         (1 << 5)
#define BRIDGE_ATTR_SPH_FILTER		(1 << 6)
#define BRIDGE_ATTR_BACKUP_NHID		(1 << 7)

#define PRIV_FLAG_NEW_ATTRS		(1 << 0)

/* Bridge info attributes */
#define BRIDGE_INFO_ATTR_FORWARD_DELAY	(1 << 0)
#define BRIDGE_INFO_ATTR_HELLO_TIME	(1 << 1)
#define BRIDGE_INFO_ATTR_MAX_AGE	(1 << 2)
#define BRIDGE_INFO_ATTR_AGEING_TIME	(1 << 3)
#define BRIDGE_INFO_ATTR_STP_STATE	(1 << 4)
#define BRIDGE_INFO_ATTR_PRIORITY	(1 << 5)
#define BRIDGE_INFO_ATTR_VLAN_FILTERING	(1 << 6)
#define BRIDGE_INFO_ATTR_VLAN_PROTOCOL	(1 << 7)
#define BRIDGE_INFO_ATTR_MCAST_SNOOPING (1 << 8)

struct bridge_data
{
	uint8_t			b_port_state;
	uint8_t			b_priv_flags; /* internal flags */
	uint16_t		b_priority;
	uint32_t		b_cost;
	uint32_t		b_flags;
	uint32_t		b_backup_port;
	uint32_t		b_flags_mask;
	uint32_t		b_sph_filter_addrs[BR_SPH_LIST_SIZE];
	uint32_t		b_port_backup_nhid;
	uint32_t                ce_mask; /* HACK to support attr macros */
	struct rtnl_link_bridge_vlan vlan_info;
	struct nl_list_head     b_vlan_list;
};
/** @endcond */

#define BITS_PER_LONG (sizeof(long) * 8)

/*
 * index starts at 0
 */
static void set_bit(unsigned long index, unsigned long *addr)
{
	ldiv_t r = ldiv(index, BITS_PER_LONG);

	addr += r.quot;
	*addr |= 1UL << r.rem;
}

static int get_bit(unsigned long index, const unsigned long *addr)
{
	ldiv_t r = ldiv(index, BITS_PER_LONG);

	addr += r.quot;
	return !!(*addr & 1UL << r.rem);
}

static unsigned long __find_next_bit(const unsigned long *addr,
				     unsigned long size, unsigned long start,
				     int invert)
{
	unsigned long tmp;
	ldiv_t r;

	if (start >= size)
		return size;

	r = ldiv(start, BITS_PER_LONG);

	tmp = addr[r.quot];
	if (invert)
		tmp = ~tmp;
	tmp &= ~0UL << r.rem;
	start -= r.rem;

	while (!tmp) {
		start += BITS_PER_LONG;
		if (start >= size)
			return size;

		r.quot++;
		tmp = addr[r.quot];
		if (invert)
			tmp = ~tmp;
	}

	return min(start + ffsl(tmp) - 1, size);
}

/*
 * Return the 0-based index of the first bit that is set within size bits.
 * If no bits are set, return size.
 */
static unsigned long find_first_bit(const unsigned long *addr,
				    unsigned long size)
{
	return __find_next_bit(addr, size, 0, 0);
}

static unsigned long find_next_bit(const unsigned long *addr,
				   unsigned long size, unsigned long start)
{
	return __find_next_bit(addr, size, start, 0);
}

static struct rtnl_link_af_ops bridge_ops;

#define IS_BRIDGE_LINK_ASSERT(link) \
	if (!rtnl_link_is_bridge(link)) { \
		APPBUG("A function was expecting a link object of type bridge."); \
		return -NLE_OPNOTSUPP; \
	}

static inline struct bridge_data *bridge_data(struct rtnl_link *link)
{
	return rtnl_link_af_data(link, &bridge_ops);
}

static void *bridge_alloc(struct rtnl_link *link)
{
	struct bridge_data *bd;

	if (!(bd = calloc(1, sizeof(struct bridge_data))))
		return NULL;

	nl_init_list_head(&bd->b_vlan_list);

	return bd;
}

static void bridge_free(struct rtnl_link *link, void *data)
{
	struct rtnl_link_bridge_vlan_tunnel *tunnel, *tmp;
	struct bridge_data *bd = data;

	nl_list_for_each_entry_safe(tunnel, tmp, &bd->b_vlan_list, rbvt_list) {
		nl_list_del(&tunnel->rbvt_list);
		free(tunnel);
	}

	free(bd);
}

static void *bridge_clone(struct rtnl_link *link, void *data)
{
	struct rtnl_link_bridge_vlan_tunnel *tunnel;
	struct bridge_data *src = data, *dst;

	if (!(dst = bridge_alloc(link)))
		return NULL;

	memcpy(dst, src, sizeof(*dst));

	nl_list_for_each_entry(tunnel, &src->b_vlan_list, rbvt_list) {
		struct rtnl_link_bridge_vlan_tunnel *new;

		if (!(new = calloc(1, sizeof(*new)))) {
			bridge_free(link, dst);
			return NULL;
		}
		memcpy(new, tunnel, sizeof(*new));
		nl_list_add_tail(&new->rbvt_list, &dst->b_vlan_list);
	}

	return dst;
}

static struct nla_policy br_attrs_policy[IFLA_BRPORT_MAX+1] = {
	[IFLA_BRPORT_STATE]		= { .type = NLA_U8 },
	[IFLA_BRPORT_PRIORITY]		= { .type = NLA_U16 },
	[IFLA_BRPORT_COST]		= { .type = NLA_U32 },
	[IFLA_BRPORT_MODE]		= { .type = NLA_U8 },
	[IFLA_BRPORT_GUARD]		= { .type = NLA_U8 },
	[IFLA_BRPORT_PROTECT]		= { .type = NLA_U8 },
	[IFLA_BRPORT_FAST_LEAVE]	= { .type = NLA_U8 },
	[IFLA_BRPORT_LEARNING]		= { .type = NLA_U8 },
	[IFLA_BRPORT_UNICAST_FLOOD]	= { .type = NLA_U8 },
	[IFLA_BRPORT_VLAN_TUNNEL]	= { .type = NLA_U8 },
	[IFLA_BRPORT_PEER_LINK]		= { .type = NLA_U8 },
	[IFLA_BRPORT_DUAL_LINK]		= { .type = NLA_U8 },
	[IFLA_BRPORT_NEIGH_SUPPRESS]	= { .type = NLA_U8 },
	[IFLA_BRPORT_BACKUP_PORT] = { .type = NLA_U32 },
	[IFLA_BRPORT_DUMMY_SPH_FILTER]	= { .type = NLA_UNSPEC,
					    .minlen = sizeof(uint32_t) * BR_SPH_LIST_SIZE },
	[IFLA_BRPORT_DUMMY_BLOCK_BUM]	= { .type = NLA_U8 },
	[IFLA_BRPORT_DUMMY_BACKUP_NHID]	= { .type = NLA_U32 },
};

static void check_flag(struct rtnl_link *link, struct nlattr *attrs[],
		       int type, int flag)
{
	if (attrs[type] && nla_get_u8(attrs[type]))
		rtnl_link_bridge_set_flags(link, flag);
}

static int bridge_parse_protinfo(struct rtnl_link *link, struct nlattr *attr,
				 void *data)
{
	struct bridge_data *bd = data;
	struct nlattr *br_attrs[IFLA_BRPORT_MAX+1];
	int err;

	/* Backwards compatibility */
	if (!nla_is_nested(attr)) {
		if (nla_len(attr) < 1)
			return -NLE_RANGE;

		bd->b_port_state = nla_get_u8(attr);
		bd->ce_mask |= BRIDGE_ATTR_PORT_STATE;

		return 0;
	}

	if ((err = nla_parse_nested(br_attrs, IFLA_BRPORT_MAX, attr,
	     br_attrs_policy)) < 0)
		return err;

	bd->b_priv_flags |= PRIV_FLAG_NEW_ATTRS;

	if (br_attrs[IFLA_BRPORT_STATE]) {
		bd->b_port_state = nla_get_u8(br_attrs[IFLA_BRPORT_STATE]);
		bd->ce_mask |= BRIDGE_ATTR_PORT_STATE;
	}

	if (br_attrs[IFLA_BRPORT_PRIORITY]) {
		bd->b_priority = nla_get_u16(br_attrs[IFLA_BRPORT_PRIORITY]);
		bd->ce_mask |= BRIDGE_ATTR_PRIORITY;
	}

	if (br_attrs[IFLA_BRPORT_COST]) {
		bd->b_cost = nla_get_u32(br_attrs[IFLA_BRPORT_COST]);
		bd->ce_mask |= BRIDGE_ATTR_COST;
	}

	if (br_attrs[IFLA_BRPORT_BACKUP_PORT]) {
		bd->b_backup_port = nla_get_u32(br_attrs[IFLA_BRPORT_BACKUP_PORT]);
		bd->ce_mask |= BRIDGE_ATTR_BACKUP_PORT;
	}

	if (br_attrs[IFLA_BRPORT_DUMMY_SPH_FILTER]) {
		memcpy(bd->b_sph_filter_addrs,
		       nla_data(br_attrs[IFLA_BRPORT_DUMMY_SPH_FILTER]),
		       sizeof(bd->b_sph_filter_addrs));
		       bd->ce_mask |= BRIDGE_ATTR_SPH_FILTER;
	}

	if (br_attrs[IFLA_BRPORT_DUMMY_BACKUP_NHID]) {
		struct nlattr *attr = br_attrs[IFLA_BRPORT_DUMMY_BACKUP_NHID];

		bd->b_port_backup_nhid = nla_get_u32(attr);
		bd->ce_mask |= BRIDGE_ATTR_BACKUP_NHID;
	}

	check_flag(link, br_attrs, IFLA_BRPORT_MODE, RTNL_BRIDGE_HAIRPIN_MODE);
	check_flag(link, br_attrs, IFLA_BRPORT_GUARD, RTNL_BRIDGE_BPDU_GUARD);
	check_flag(link, br_attrs, IFLA_BRPORT_PROTECT, RTNL_BRIDGE_ROOT_BLOCK);
	check_flag(link, br_attrs, IFLA_BRPORT_FAST_LEAVE, RTNL_BRIDGE_FAST_LEAVE);
	check_flag(link, br_attrs, IFLA_BRPORT_LEARNING, RTNL_BRIDGE_LEARNING);
	check_flag(link, br_attrs, IFLA_BRPORT_UNICAST_FLOOD, RTNL_BRIDGE_UNICAST_FLOOD);
	check_flag(link, br_attrs, IFLA_BRPORT_VLAN_TUNNEL,
		   RTNL_BRIDGE_VLAN_TUNNEL);
	check_flag(link, br_attrs, IFLA_BRPORT_PEER_LINK, RTNL_BRIDGE_PEER_LINK);
	check_flag(link, br_attrs, IFLA_BRPORT_DUAL_LINK, RTNL_BRIDGE_DUAL_LINK);
	check_flag(link, br_attrs, IFLA_BRPORT_NEIGH_SUPPRESS, RTNL_BRIDGE_NEIGH_SUPPRESS);
	check_flag(link, br_attrs, IFLA_BRPORT_DUMMY_BLOCK_BUM, RTNL_BRIDGE_BLOCK_BUM);
	return 0;
}

static void bridge_vlan_info(struct bridge_data *bd,
			     struct bridge_vlan_info *vinfo)
{
	if (vinfo->flags & BRIDGE_VLAN_INFO_PVID)
		bd->vlan_info.pvid = vinfo->vid;

	if (vinfo->flags & BRIDGE_VLAN_INFO_UNTAGGED)
		set_bit(vinfo->vid, bd->vlan_info.untagged_bitmap);

	set_bit(vinfo->vid, bd->vlan_info.vlan_bitmap);
	bd->ce_mask |= BRIDGE_ATTR_PORT_VLAN;
}

static int bridge_process_vlan_info(struct bridge_data *bd,
				    struct bridge_vlan_info *vinfo_curr,
				    struct bridge_vlan_info **vinfo_last)
{
	if (!vinfo_curr->vid || vinfo_curr->vid >= VLAN_VID_MASK) {
		NL_DBG(1, "invalid VLAN id\n");
		return -ERANGE;
	}

	if (vinfo_curr->flags & BRIDGE_VLAN_INFO_RANGE_BEGIN) {
		/* check if we are already processing a range */
		if (*vinfo_last) {
			NL_DBG(1, "VLAN range begin not followed by range end\n");
			return -EINVAL;
		}
		*vinfo_last = vinfo_curr;
		/* don't allow range of pvids */
		if ((*vinfo_last)->flags & BRIDGE_VLAN_INFO_PVID) {
			NL_DBG(1, "VLAN range of pvids\n");
			return -EINVAL;
		}
		return 0;
	}

	if (*vinfo_last) {
		struct bridge_vlan_info tmp_vinfo;
		uint16_t v;

		if (!(vinfo_curr->flags & BRIDGE_VLAN_INFO_RANGE_END)) {
			NL_DBG(1, "VLAN range begin not followed by range end\n");
			return -EINVAL;
		}

		if (vinfo_curr->vid <= (*vinfo_last)->vid) {
			NL_DBG(1, "VLAN range vid does not increase\n");
			return -EINVAL;
		}

		/* sanity check the range flags */
		if ((vinfo_curr->flags ^ BRIDGE_VLAN_INFO_RANGE_END) !=
		    ((*vinfo_last)->flags ^ BRIDGE_VLAN_INFO_RANGE_BEGIN)) {
			NL_DBG(1, "VLAN range flags differ; cannot handle it.\n");
			return -EINVAL;
		}

		memcpy(&tmp_vinfo, *vinfo_last,
		       sizeof(struct bridge_vlan_info));
		for (v = (*vinfo_last)->vid; v <= vinfo_curr->vid; v++) {
			tmp_vinfo.vid = v;
			bridge_vlan_info(bd, &tmp_vinfo);
		}
		*vinfo_last = NULL;

		return 0;
	}

	bridge_vlan_info(bd, vinfo_curr);
	return 0;
}

struct vlan_tunnel_info {
	uint32_t vti_tunid;
	uint16_t vti_vid;
	uint16_t vti_flags;
};

static const struct nla_policy vlan_tunnel_policy[IFLA_BRIDGE_VLAN_TUNNEL_MAX + 1] = {
	[IFLA_BRIDGE_VLAN_TUNNEL_ID] = { .type = NLA_U32 },
	[IFLA_BRIDGE_VLAN_TUNNEL_VID] = { .type = NLA_U16 },
	[IFLA_BRIDGE_VLAN_TUNNEL_FLAGS] = { .type = NLA_U16 },
};

static int bridge_parse_vlan_tunnel_info(struct nlattr *attr,
					 struct vlan_tunnel_info *info)
{
	struct nlattr *tb[IFLA_BRIDGE_VLAN_TUNNEL_MAX + 1];
	uint32_t tunid;
	uint16_t vid;
	int err;

	memset(info, 0, sizeof(*info));

	if ((err = nla_parse_nested(tb, IFLA_BRIDGE_VLAN_TUNNEL_MAX, attr,
				    (struct nla_policy *)vlan_tunnel_policy))
	    < 0)
		return err;

	if (!tb[IFLA_BRIDGE_VLAN_TUNNEL_ID] ||
	    !tb[IFLA_BRIDGE_VLAN_TUNNEL_VID]) {
		NL_DBG(1, "VLAN tunnel info missing attributes\n");
		return -EINVAL;
	}

	tunid = nla_get_u32(tb[IFLA_BRIDGE_VLAN_TUNNEL_ID]);
	vid = nla_get_u16(tb[IFLA_BRIDGE_VLAN_TUNNEL_VID]);
	if (vid >= VLAN_VID_MASK) {
		NL_DBG(1, "invalid VLAN id\n");
		return -ERANGE;
	}

	info->vti_tunid = tunid;
	info->vti_vid = vid;

	if (tb[IFLA_BRIDGE_VLAN_TUNNEL_FLAGS])
		info->vti_flags = nla_get_u16(tb[IFLA_BRIDGE_VLAN_TUNNEL_FLAGS]);

	return 0;
}

static int bridge_vlan_tunnel_info(struct bridge_data *bd, uint16_t vid,
				   uint32_t tunid)
{
	struct rtnl_link_bridge_vlan_tunnel *new;

	if (!(new = calloc(1, sizeof(*new))))
		return NLE_NOMEM;
	new->rbvt_vid = vid;
	new->rbvt_tunid = tunid;
	nl_list_add_tail(&new->rbvt_list, &bd->b_vlan_list);

	return 0;
}

static int bridge_process_vlan_tunnel_info(struct bridge_data *bd,
					   struct vlan_tunnel_info *info_curr,
					   struct vlan_tunnel_info *info_last)
{
	int err;

	if (info_curr->vti_flags & BRIDGE_VLAN_INFO_RANGE_BEGIN) {
		if (info_last->vti_flags & BRIDGE_VLAN_INFO_RANGE_BEGIN) {
			NL_DBG(1, "VLAN tunnel range begin not followed by range end\n");
			return -EINVAL;
		}
		memcpy(info_last, info_curr, sizeof(*info_last));
	} else if (info_curr->vti_flags & BRIDGE_VLAN_INFO_RANGE_END) {
		uint16_t v;
		uint32_t t;

		if (!(info_last->vti_flags & BRIDGE_VLAN_INFO_RANGE_BEGIN)) {
			NL_DBG(1, "VLAN tunnel range end not preceded by range begin\n");
			return -EINVAL;
		} else if (info_curr->vti_vid - info_last->vti_vid !=
			   info_curr->vti_tunid - info_last->vti_tunid) {
			NL_DBG(1, "VLAN tunnel id ranges of different lengths\n");
			return -EINVAL;
		}
		for (v = info_last->vti_vid, t = info_last->vti_tunid;
		     v <= info_curr->vti_vid; v++, t++) {
			if ((err = bridge_vlan_tunnel_info(bd, v, t)))
				return err;
		}
		memset(info_last, 0, sizeof(*info_last));
	} else {
		if (info_last->vti_flags) {
			NL_DBG(1, "VLAN tunnel range begin not followed by range end\n");
			return -EINVAL;
		}
		if ((err = bridge_vlan_tunnel_info(bd, info_curr->vti_vid,
						   info_curr->vti_tunid)))
			return err;
	}

	return 0;
}


static int bridge_parse_af_full(struct rtnl_link *link, struct nlattr *attr_full,
                                void *data)
{
	struct vlan_tunnel_info tinfo_curr, tinfo_last = {};
	struct bridge_vlan_info *vinfo_curr = NULL;
	struct bridge_vlan_info *vinfo_last = NULL;
	struct bridge_data *bd = data;
	struct nlattr *attr;
	int remaining;

	nla_for_each_nested(attr, attr_full, remaining) {
		int err;

		switch (nla_type(attr)) {
		case IFLA_BRIDGE_VLAN_INFO:
			if (nla_len(attr) != sizeof(struct bridge_vlan_info))
				return -EINVAL;
			vinfo_curr = nla_data(attr);
			if ((err = bridge_process_vlan_info(bd, vinfo_curr,
							    &vinfo_last)))
				return err;
			break;
		case IFLA_BRIDGE_VLAN_TUNNEL_INFO:
			if ((err = bridge_parse_vlan_tunnel_info(attr,
								 &tinfo_curr)))
				return err;
			if ((err = bridge_process_vlan_tunnel_info(bd,
								   &tinfo_curr,
								   &tinfo_last)))
				return err;
			break;
		}
	}

	return 0;
}

static int bridge_get_af(struct nl_msg *msg)
{
	__u32 ext_filter_mask = RTEXT_FILTER_BRVLAN | RTEXT_FILTER_BRVLAN_COMPRESSED;

	return nla_put(msg, IFLA_EXT_MASK, sizeof(ext_filter_mask), &ext_filter_mask);
}

struct vlan_attrs {
	uint16_t id;
	uint8_t tagged;
	uint8_t tunnel_info;
	uint32_t tunnel_id;
};

static void fill_vlan_attrs(struct vlan_attrs *attrs,
			    const struct bridge_data *bd,
			    struct nl_list_head **pos)
{
	struct rtnl_link_bridge_vlan_tunnel *entry;

	attrs->tagged = !get_bit(attrs->id, bd->vlan_info.untagged_bitmap);

	while (*pos != &bd->b_vlan_list) {
		entry = nl_list_entry(*pos,
				      struct rtnl_link_bridge_vlan_tunnel,
				      rbvt_list);
		if (entry->rbvt_vid >= attrs->id)
			break;
		*pos = (*pos)->next;
	}

	if (*pos != &bd->b_vlan_list && entry->rbvt_vid == attrs->id) {
		attrs->tunnel_info = 1;
		attrs->tunnel_id = entry->rbvt_tunid;
	} else
		attrs->tunnel_info = 0;
}

static int vlan_range_continuous(const struct vlan_attrs *a,
				 const struct vlan_attrs *b)
{
	return b->id == a->id + 1 && b->tagged == a->tagged &&
		b->tunnel_info == a->tunnel_info &&
		(!a->tunnel_info || b->tunnel_id == a->tunnel_id + 1);
}

static void format_id_range(char *str, size_t size, unsigned int begin,
			    unsigned int end)
{
	int retval;

	if (!size)
		return;
	
	retval = snprintf(str, size, "%u", begin);
	if (retval < 0) {
		str[0] = '\0';
		return;
	} else if (retval >= size) {
		str[size - 1] = '\0';
		return;
	}
	str += retval;
	size -= retval;

	if (end != begin) {
		retval = snprintf(str, size, "-%u", end);
		if (retval < 0) {
			str[0] = '\0';
			return;
		} else if (retval >= size)
			str[size - 1] = '\0';
	}
}

static void rtnl_link_bridge_dump_vlans(struct nl_dump_params *p,
					const struct bridge_data *bd)
{
	struct vlan_attrs begin, last, cur;
	struct nl_list_head *pos;

	begin.id = find_first_bit(bd->vlan_info.vlan_bitmap, VLAN_MAX);
	if (begin.id == VLAN_MAX)
		return;

	nl_dump_line(p, "      vlan id    tagged  tunnel info\n");
	nl_dump_line(p, "                         ↳ id\n");

	pos = bd->b_vlan_list.next;
	fill_vlan_attrs(&begin, bd, &pos);

	memcpy(&last, &begin, sizeof(last));
	do {
		cur.id = find_next_bit(bd->vlan_info.vlan_bitmap, VLAN_MAX,
				       last.id + 1);
		if (cur.id != VLAN_MAX)
			fill_vlan_attrs(&cur, bd, &pos);
		if (cur.id == VLAN_MAX || !vlan_range_continuous(&last, &cur)) {
			char vlan_range[10], tunid_range[22] = "";

			format_id_range(vlan_range, ARRAY_SIZE(vlan_range),
					begin.id, last.id);
			if (begin.tunnel_info)
				format_id_range(tunid_range,
						ARRAY_SIZE(tunid_range),
						begin.tunnel_id, last.tunnel_id);

			nl_dump_line(p, "      %-9s  %s         %s\n",
				     vlan_range, begin.tagged ? "✔" : " ",
				     tunid_range);

			memcpy(&begin, &cur, sizeof(begin));
		}
		memcpy(&last, &cur, sizeof(last));
	} while (cur.id != VLAN_MAX);
}

static void rtnl_link_bridge_dump_sph_list(struct nl_dump_params *p,
					   struct bridge_data *bd)
{
	char sph_str[256], addr[INET_ADDRSTRLEN];
	int len, i, found = 0;

	if (!bd || !(bd->ce_mask & BRIDGE_ATTR_SPH_FILTER))
		return;

	len = snprintf(sph_str, sizeof(sph_str), "[ ");
	for (i = 0; i < BR_SPH_LIST_SIZE; i++) {
		struct in_addr in = { .s_addr = bd->b_sph_filter_addrs[i] };

		if (bd->b_sph_filter_addrs[i] &&
		    inet_ntop(AF_INET, &in, addr, sizeof(addr))) {
			len += snprintf(sph_str + len, sizeof(sph_str) - len,
					"%d: %s; ", i, addr);
			found = 1;
		}

		if (len >= sizeof(sph_str) - 24)
			break;
	}
	len += snprintf(sph_str + len, sizeof(sph_str) - len, "]");

	if (found)
		nl_dump(p, "   sph_filter: %s ", sph_str);
}

static void bridge_dump_details(struct rtnl_link *link,
				struct nl_dump_params *p, void *data)
{
	struct bridge_data *bd = data;

	nl_dump_line(p, "    bridge:");

	if (bd->ce_mask & BRIDGE_ATTR_PORT_STATE)
		nl_dump(p, " port-state %u", bd->b_port_state);

	if (bd->ce_mask & BRIDGE_ATTR_PRIORITY)
		nl_dump(p, " prio %u", bd->b_priority);

	if (bd->ce_mask & BRIDGE_ATTR_COST)
		nl_dump(p, " cost %u", bd->b_cost);

	if (bd->ce_mask & BRIDGE_ATTR_BACKUP_NHID)
		nl_dump(p, "backup_nhid %u ", bd->b_port_backup_nhid);

	if (bd->ce_mask & BRIDGE_ATTR_PORT_VLAN)
		nl_dump(p, " pvid %u", bd->vlan_info.pvid);

	rtnl_link_bridge_dump_sph_list(p, bd);

	if (bd->ce_mask & BRIDGE_ATTR_FLAGS) {
		char buf[256];

		rtnl_link_bridge_flags2str(bd->b_flags & bd->b_flags_mask,
					   buf, sizeof(buf));
		nl_dump(p, " flags: %s", buf);
	}
	nl_dump(p, "\n");

	if (bd->ce_mask & BRIDGE_ATTR_PORT_VLAN)
		rtnl_link_bridge_dump_vlans(p, bd);
}

static int bridge_compare(struct rtnl_link *_a, struct rtnl_link *_b,
			  int family, uint32_t attrs, int flags)
{
	struct bridge_data *a = bridge_data(_a);
	struct bridge_data *b = bridge_data(_b);
	int diff = 0;

#define BRIDGE_DIFF(ATTR, EXPR) ATTR_DIFF(attrs, BRIDGE_ATTR_##ATTR, a, b, EXPR)
	diff |= BRIDGE_DIFF(PORT_STATE,	a->b_port_state != b->b_port_state);
	diff |= BRIDGE_DIFF(PRIORITY, a->b_priority != b->b_priority);
	diff |= BRIDGE_DIFF(COST, a->b_cost != b->b_cost);
	diff |= BRIDGE_DIFF(PORT_VLAN, memcmp(&a->vlan_info, &b->vlan_info,
					      sizeof(struct rtnl_link_bridge_vlan)));
	diff |= BRIDGE_DIFF(BACKUP_NHID,
				a->b_port_backup_nhid != b->b_port_backup_nhid);
	diff |= BRIDGE_DIFF(SPH_FILTER,
			memcmp(a->b_sph_filter_addrs, b->b_sph_filter_addrs,
				(sizeof(uint32_t) * BR_SPH_LIST_SIZE)));
	if (flags & LOOSE_COMPARISON)
		diff |= BRIDGE_DIFF(FLAGS,
				  (a->b_flags ^ b->b_flags) & b->b_flags_mask);
	else
		diff |= BRIDGE_DIFF(FLAGS, a->b_flags != b->b_flags);
#undef BRIDGE_DIFF

	return diff;
}
/** @endcond */

/**
 * Allocate link object of type bridge
 *
 * @return Allocated link object or NULL.
 */
struct rtnl_link *rtnl_link_bridge_alloc(void)
{
	struct rtnl_link *link;
	int err;

	if (!(link = rtnl_link_alloc()))
		return NULL;

	if ((err = rtnl_link_set_type(link, "bridge")) < 0) {
		rtnl_link_put(link);
		return NULL;
	}

	return link;
}
		
/** 
 * Create a new kernel bridge device
 * @arg sk              netlink socket
 * @arg name            name of the bridge device or NULL
 *
 * Creates a new bridge device in the kernel. If no name is
 * provided, the kernel will automatically pick a name of the
 * form "type%d" (e.g. bridge0, vlan1, etc.)
 *
 * @return 0 on success or a negative error code
*/
int rtnl_link_bridge_add(struct nl_sock *sk, const char *name)
{
	int err;
	struct rtnl_link *link;

	if (!(link = rtnl_link_bridge_alloc()))
		return -NLE_NOMEM;

	if(name)
		rtnl_link_set_name(link, name);

	err = rtnl_link_add(sk, link, NLM_F_CREATE);
	rtnl_link_put(link);

	return err;
}

/**
 * Check if a link is a bridge
 * @arg link		Link object
 *
 * @return 1 if the link is a bridge, 0 otherwise.
 */
int rtnl_link_is_bridge(struct rtnl_link *link)
{
	return link->l_family == AF_BRIDGE &&
	       link->l_af_ops == &bridge_ops;
}

/**
 * Check if bridge has extended information
 * @arg link		Link object of type bridge
 *
 * Checks if the bridge object has been constructed based on
 * information that is only available in newer kernels. This
 * affectes the following functions:
 *  - rtnl_link_bridge_get_cost()
 *  - rtnl_link_bridge_get_priority()
 *  - rtnl_link_bridge_get_flags()
 *
 * @return 1 if extended information is available, otherwise 0 is returned.
 */
int rtnl_link_bridge_has_ext_info(struct rtnl_link *link)
{
	struct bridge_data *bd;

	if (!rtnl_link_is_bridge(link))
		return 0;

	bd = bridge_data(link);
	return !!(bd->b_priv_flags & PRIV_FLAG_NEW_ATTRS);
}

/**
 * Set Spanning Tree Protocol (STP) port state
 * @arg link		Link object of type bridge
 * @arg state		New STP port state
 *
 * The value of state must be one of the following:
 *   - BR_STATE_DISABLED
 *   - BR_STATE_LISTENING
 *   - BR_STATE_LEARNING
 *   - BR_STATE_FORWARDING
 *   - BR_STATE_BLOCKING
 *
 * @see rtnl_link_bridge_get_port_state()
 *
 * @return 0 on success or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 * @retval -NLE_INVAL Invalid state value (0..BR_STATE_BLOCKING)
 */
int rtnl_link_bridge_set_port_state(struct rtnl_link *link, uint8_t state)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	if (state > BR_STATE_BLOCKING)
		return -NLE_INVAL;

	bd->b_port_state = state;
	bd->ce_mask |= BRIDGE_ATTR_PORT_STATE;

	return 0;
}

/**
 * Get Spanning Tree Protocol (STP) port state
 * @arg link		Link object of type bridge
 *
 * @see rtnl_link_bridge_set_port_state()
 *
 * @return The STP port state or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 */
int rtnl_link_bridge_get_port_state(struct rtnl_link *link)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	return bd->b_port_state;
}

/**
 * Get port backup nexthop id
 * @arg link		Link object of type bridge
 *
 * @return The backup nexthop id or 0 if it is not present.
 */
uint32_t rtnl_link_bridge_get_port_backup_nhid(struct rtnl_link *link)
{
	struct bridge_data *bd = bridge_data(link);

	if (bd && (bd->ce_mask & BRIDGE_ATTR_BACKUP_NHID))
		return bd->b_port_backup_nhid;

	return 0;
}

/**
 * Set priority
 * @arg link		Link object of type bridge
 * @arg prio		Bridge priority
 *
 * @see rtnl_link_bridge_get_priority()
 *
 * @return 0 on success or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 */
int rtnl_link_bridge_set_priority(struct rtnl_link *link, uint16_t prio)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	bd->b_priority = prio;
	bd->ce_mask |= BRIDGE_ATTR_PRIORITY;

	return 0;
}

/**
 * Get priority
 * @arg link		Link object of type bridge
 *
 * @see rtnl_link_bridge_set_priority()
 *
 * @return 0 on success or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 */
int rtnl_link_bridge_get_priority(struct rtnl_link *link)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	return bd->b_priority;
}

/**
 * Set Spanning Tree Protocol (STP) path cost
 * @arg link		Link object of type bridge
 * @arg cost		New STP path cost value
 *
 * @see rtnl_link_bridge_get_cost()
 *
 * @return The bridge priority or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 */
int rtnl_link_bridge_set_cost(struct rtnl_link *link, uint32_t cost)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	bd->b_cost = cost;
	bd->ce_mask |= BRIDGE_ATTR_COST;

	return 0;
}

/**
 * Get Spanning Tree Protocol (STP) path cost
 * @arg link		Link object of type bridge
 * @arg cost		Pointer to store STP cost value
 *
 * @see rtnl_link_bridge_set_cost()
 *
 * @return 0 on success or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 * @retval -NLE_INVAL `cost` is not a valid pointer
 */
int rtnl_link_bridge_get_cost(struct rtnl_link *link, uint32_t *cost)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	if (!cost)
		return -NLE_INVAL;

	*cost = bd->b_cost;

	return 0;
}

/**
 * Unset flags
 * @arg link		Link object of type bridge
 * @arg flags		Bridging flags to unset
 *
 * @see rtnl_link_bridge_set_flags()
 * @see rtnl_link_bridge_get_flags()
 *
 * @return 0 on success or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 */
int rtnl_link_bridge_unset_flags(struct rtnl_link *link, unsigned int flags)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	bd->b_flags_mask |= flags;
	bd->b_flags &= ~flags;
	bd->ce_mask |= BRIDGE_ATTR_FLAGS;

	return 0;
}

/**
 * Set flags
 * @arg link		Link object of type bridge
 * @arg flags		Bridging flags to set
 *
 * Valid flags are:
 *   - RTNL_BRIDGE_HAIRPIN_MODE
 *   - RTNL_BRIDGE_BPDU_GUARD
 *   - RTNL_BRIDGE_ROOT_BLOCK
 *   - RTNL_BRIDGE_FAST_LEAVE
 *   - RTNL_BRIDGE_VLAN_TUNNEL
 *
 * @see rtnl_link_bridge_unset_flags()
 * @see rtnl_link_bridge_get_flags()
 *
 * @return 0 on success or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 */
int rtnl_link_bridge_set_flags(struct rtnl_link *link, unsigned int flags)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	bd->b_flags_mask |= flags;
	bd->b_flags |= flags;
	bd->ce_mask |= BRIDGE_ATTR_FLAGS;

	return 0;
}

/**
 * Get flags
 * @arg link		Link object of type bridge
 *
 * @see rtnl_link_bridge_set_flags()
 * @see rtnl_link_bridge_unset_flags()
 *
 * @return Flags or a negative error code.
 * @retval -NLE_OPNOTSUPP Link is not a bridge
 */
int rtnl_link_bridge_get_flags(struct rtnl_link *link)
{
	struct bridge_data *bd = bridge_data(link);

	IS_BRIDGE_LINK_ASSERT(link);

	return bd->b_flags;
}

static const struct trans_tbl bridge_flags[] = {
	__ADD(RTNL_BRIDGE_HAIRPIN_MODE, hairpin_mode),
	__ADD(RTNL_BRIDGE_BPDU_GUARD, 	bpdu_guard),
	__ADD(RTNL_BRIDGE_ROOT_BLOCK,	root_block),
	__ADD(RTNL_BRIDGE_FAST_LEAVE,	fast_leave),
	__ADD(RTNL_BRIDGE_LEARNING,	learning),
	__ADD(RTNL_BRIDGE_UNICAST_FLOOD, flood),
	__ADD(RTNL_BRIDGE_VLAN_TUNNEL,	vlan_tunnel),
	__ADD(RTNL_BRIDGE_PEER_LINK,	peer_link),
	__ADD(RTNL_BRIDGE_DUAL_LINK,	dual_link),
	__ADD(RTNL_BRIDGE_NEIGH_SUPPRESS, neigh_suppress),
	__ADD(RTNL_BRIDGE_BLOCK_BUM,	block_bum),
};

/**
 * @name Flag Translation
 * @{
 */

char *rtnl_link_bridge_flags2str(int flags, char *buf, size_t len)
{
	return __flags2str(flags, buf, len, bridge_flags, ARRAY_SIZE(bridge_flags));
}

int rtnl_link_bridge_str2flags(const char *name)
{
	return __str2flags(name, bridge_flags, ARRAY_SIZE(bridge_flags));
}

/** @} */

int rtnl_link_bridge_pvid(struct rtnl_link *link)
{
	struct bridge_data *bd;

	IS_BRIDGE_LINK_ASSERT(link);

	bd = link->l_af_data[AF_BRIDGE];
	if (bd->ce_mask & BRIDGE_ATTR_PORT_VLAN)
		return (int) bd->vlan_info.pvid;

	return -EINVAL;
}

int rtnl_link_bridge_has_vlan(struct rtnl_link *link)
{
	struct bridge_data *bd;

	IS_BRIDGE_LINK_ASSERT(link);

	bd = link->l_af_data[AF_BRIDGE];
	return !!((bd->ce_mask & BRIDGE_ATTR_PORT_VLAN) && (
			bd->vlan_info.pvid ||
			find_first_bit(bd->vlan_info.vlan_bitmap, VLAN_MAX) !=
			VLAN_MAX));
}

struct rtnl_link_bridge_vlan *rtnl_link_bridge_get_port_vlan(struct rtnl_link *link)
{
	struct bridge_data *data;

	if (!rtnl_link_is_bridge(link))
		return NULL;

	data = link->l_af_data[AF_BRIDGE];
	if (data && (data->ce_mask & BRIDGE_ATTR_PORT_VLAN))
		return &data->vlan_info;

	return NULL;
}

struct nl_list_head *rtnl_link_bridge_get_port_vlan_tunnels(
	struct rtnl_link *link)
{
	struct bridge_data *bd;

	if (!rtnl_link_is_bridge(link))
		return NULL;

	if ((bd = link->l_af_data[AF_BRIDGE]))
		return &bd->b_vlan_list;

	return NULL;
}

static struct rtnl_link_af_ops bridge_ops = {
	.ao_family			= AF_BRIDGE,
	.ao_alloc			= &bridge_alloc,
	.ao_clone			= &bridge_clone,
	.ao_free			= &bridge_free,
	.ao_parse_protinfo		= &bridge_parse_protinfo,
	.ao_dump[NL_DUMP_DETAILS]	= &bridge_dump_details,
	.ao_compare			= &bridge_compare,
	.ao_parse_af_full		= &bridge_parse_af_full,
	.ao_get_af			= &bridge_get_af,
};

/**
 * @name Bridge Object
 * @{
 */

/**
 * Get bridge port state
 * @arg link            Link object
 *
 * @return port state if link has bridge port af data, otherwise -1 is returned.
 */
int rtnl_bridge_get_port_state(struct rtnl_link *link)
{
	struct bridge_data *data = link->l_af_data[AF_BRIDGE];

	if (data && (data->ce_mask & BRIDGE_ATTR_PORT_STATE))
		return data->b_port_state;

	return -1;
}

/**
 * Get bridge port backup port
 * @arg link            Link object
 *
 * @return backup port if link has bridge port af data, otherwise -1 is returned.
 */
int rtnl_bridge_get_backup_port(struct rtnl_link *link)
{
	struct bridge_data *data = link->l_af_data[AF_BRIDGE];

	if (data && (data->ce_mask & BRIDGE_ATTR_BACKUP_PORT))
		return data->b_backup_port;

	return -1;
}

/**
 * Get bridge port sph filter list
 * @arg link            Link object
 * @arg sph_arr         Target storage array for the sph list
 *
 * @return if there's at least one non-zero entry in the sph list then it is
 *         copied to sph_arr and 0 is returned, otherwise -1 is returned.
 */
int rtnl_bridge_get_sph_filter_list(struct rtnl_link *link, uint32_t *sph_arr)
{
	struct bridge_data *data = link->l_af_data[AF_BRIDGE];
	int ret = -1, i;

	if (!data || !(data->ce_mask & BRIDGE_ATTR_SPH_FILTER))
		return ret;

	for (i = 0; i < BR_SPH_LIST_SIZE; i++) {
		sph_arr[i] = data->b_sph_filter_addrs[i];
		if (ret == -1 && sph_arr[i])
			ret = 0;
	}

	return ret;
}

/** @} */

static struct nla_policy bridge_info_policy[IFLA_BR_MAX + 1] = {
	[IFLA_BR_FORWARD_DELAY]	= { .type = NLA_U32 },
	[IFLA_BR_HELLO_TIME]	= { .type = NLA_U32 },
	[IFLA_BR_MAX_AGE]	= { .type = NLA_U32 },
	[IFLA_BR_AGEING_TIME] = { .type = NLA_U32 },
	[IFLA_BR_STP_STATE] = { .type = NLA_U32 },
	[IFLA_BR_PRIORITY] = { .type = NLA_U16 },
	[IFLA_BR_VLAN_FILTERING] = { .type = NLA_U8 },
	[IFLA_BR_VLAN_PROTOCOL] = { .type = NLA_U16 },
	[IFLA_BR_GROUP_FWD_MASK] = { .type = NLA_U16 },
	[IFLA_BR_GROUP_ADDR] = { .type = NLA_BINARY},
	[IFLA_BR_MCAST_ROUTER] = { .type = NLA_U8 },
	[IFLA_BR_MCAST_SNOOPING] = { .type = NLA_U8 },
	[IFLA_BR_MCAST_QUERY_USE_IFADDR] = { .type = NLA_U8 },
	[IFLA_BR_MCAST_QUERIER] = { .type = NLA_U8 },
	[IFLA_BR_MCAST_HASH_ELASTICITY] = { .type = NLA_U32 },
	[IFLA_BR_MCAST_HASH_MAX] = { .type = NLA_U32 },
	[IFLA_BR_MCAST_LAST_MEMBER_CNT] = { .type = NLA_U32 },
	[IFLA_BR_MCAST_STARTUP_QUERY_CNT] = { .type = NLA_U32 },
	[IFLA_BR_MCAST_LAST_MEMBER_INTVL] = { .type = NLA_U64 },
	[IFLA_BR_MCAST_MEMBERSHIP_INTVL] = { .type = NLA_U64 },
	[IFLA_BR_MCAST_QUERIER_INTVL] = { .type = NLA_U64 },
	[IFLA_BR_MCAST_QUERY_INTVL] = { .type = NLA_U64 },
	[IFLA_BR_MCAST_QUERY_RESPONSE_INTVL] = { .type = NLA_U64 },
	[IFLA_BR_MCAST_STARTUP_QUERY_INTVL] = { .type = NLA_U64 },
	[IFLA_BR_NF_CALL_IPTABLES] = { .type = NLA_U8 },
	[IFLA_BR_NF_CALL_IP6TABLES] = { .type = NLA_U8 },
	[IFLA_BR_NF_CALL_ARPTABLES] = { .type = NLA_U8 },
	[IFLA_BR_VLAN_DEFAULT_PVID] = { .type = NLA_U16 },
};

struct bridge_info
{
	uint32_t	forward_delay;
	uint32_t	hello_time;
	uint32_t	max_age;
	uint32_t	ageing_time;
	uint32_t	stp_state;
	uint16_t	priority;
	uint16_t	vlan_protocol;
	uint8_t		vlan_filtering;
	uint8_t		mcast_snooping;
	uint32_t	ce_mask;
};

static int bridge_info_alloc(struct rtnl_link *link)
{
	struct bridge_info *brinfo;

	if (link->l_info)
		memset(link->l_info, 0, sizeof(*brinfo));
	else {
		if ((brinfo = calloc(1, sizeof(*brinfo))) == NULL)
			return -NLE_NOMEM;

		link->l_info = brinfo;
	}

	return 0;
}

static int bridge_info_parse(struct rtnl_link *link, struct nlattr *data,
			     struct nlattr *xstats)
{
	struct nlattr *tb[IFLA_BR_MAX+1];
	struct bridge_info *brinfo;
	int err;

	NL_DBG(3, "Parsing Bridge link info\n");

	if ((err = nla_parse_nested(tb, IFLA_BR_MAX, data,
				    bridge_info_policy)) < 0)
		goto errout;

	if ((err = bridge_info_alloc(link)) < 0)
		goto errout;

	brinfo = link->l_info;

	if (tb[IFLA_BR_FORWARD_DELAY]) {
		brinfo->forward_delay = nla_get_u32(tb[IFLA_BR_FORWARD_DELAY]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_FORWARD_DELAY;
	}

	if (tb[IFLA_BR_HELLO_TIME]) {
		brinfo->hello_time = nla_get_u32(tb[IFLA_BR_HELLO_TIME]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_HELLO_TIME;
	}

	if (tb[IFLA_BR_MAX_AGE]) {
		brinfo->max_age = nla_get_u32(tb[IFLA_BR_MAX_AGE]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_MAX_AGE;
	}

	if (tb[IFLA_BR_AGEING_TIME]) {
		brinfo->ageing_time = nla_get_u32(tb[IFLA_BR_AGEING_TIME]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_AGEING_TIME;
	}

	if (tb[IFLA_BR_STP_STATE]) {
		brinfo->stp_state = nla_get_u32(tb[IFLA_BR_STP_STATE]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_STP_STATE;
	}

	if (tb[IFLA_BR_PRIORITY]) {
		brinfo->priority = nla_get_u16(tb[IFLA_BR_PRIORITY]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_PRIORITY;
	}

	if (tb[IFLA_BR_VLAN_FILTERING]) {
		brinfo->vlan_filtering = nla_get_u8(tb[IFLA_BR_VLAN_FILTERING]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_VLAN_FILTERING;
	}

	if (tb[IFLA_BR_VLAN_PROTOCOL]) {
		brinfo->vlan_protocol = nla_get_u16(tb[IFLA_BR_VLAN_PROTOCOL]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_VLAN_PROTOCOL;
	}

	if (tb[IFLA_BR_MCAST_SNOOPING]) {
		brinfo->mcast_snooping = nla_get_u8(tb[IFLA_BR_MCAST_SNOOPING]);
		brinfo->ce_mask |= BRIDGE_INFO_ATTR_MCAST_SNOOPING;
	}

	err = 0;
errout:
	return err;
}

static void bridge_info_free(struct rtnl_link *link)
{
	free(link->l_info);
	link->l_info = NULL;
}

static void bridge_info_dump_line(struct rtnl_link *link,
				  struct nl_dump_params *p)
{
	struct bridge_info *brinfo = link->l_info;

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_VLAN_FILTERING)
		nl_dump(p, "     bridge-vlan-filtering %d ",
			brinfo->vlan_filtering);
}

static void bridge_info_dump_details(struct rtnl_link *link,
				     struct nl_dump_params *p)
{
	struct bridge_info *brinfo = link->l_info;

	nl_dump(p, "     ");

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_FORWARD_DELAY)
		nl_dump(p, "bridge-forward-delay %d ", brinfo->forward_delay);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_HELLO_TIME)
		nl_dump(p, "bridge-hello-time %d ", brinfo->hello_time);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_MAX_AGE)
		nl_dump(p, "bridge-max-age %d ", brinfo->max_age);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_AGEING_TIME)
		nl_dump(p, "bridge-ageing-time %d ", brinfo->ageing_time);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_STP_STATE)
		nl_dump(p, "bridge-stp-state %d ", brinfo->stp_state);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_PRIORITY)
		nl_dump(p, "bridge-priority %d ", brinfo->priority);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_VLAN_FILTERING)
		nl_dump(p, "bridge-vlan-filtering %d ", brinfo->vlan_filtering);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_VLAN_PROTOCOL)
		nl_dump(p, "bridge-vlan-protocol %d ", brinfo->vlan_protocol);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_MCAST_SNOOPING)
		nl_dump(p, "bridge-mcast-snooping %d ", brinfo->mcast_snooping);

	nl_dump(p, "\n");
}

static int bridge_info_clone(struct rtnl_link *dst, struct rtnl_link *src)
{
	struct bridge_info *bdst, *bsrc = src->l_info;
	int err;

	dst->l_info = NULL;
	if ((err = rtnl_link_set_type(dst, "bridge")) < 0)
		return err;
	bdst = dst->l_info;

	if (!bdst || !bsrc)
		return -NLE_NOMEM;

	memcpy(bdst, bsrc, sizeof(struct bridge_info));

	return 0;
}

static int bridge_info_put_attrs(struct nl_msg *msg, struct rtnl_link *link)
{
	struct bridge_info *brinfo = link->l_info;
	struct nlattr *data;

	if (!(data = nla_nest_start(msg, IFLA_INFO_DATA)))
		return -NLE_MSGSIZE;

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_FORWARD_DELAY)
		NLA_PUT_U32(msg, IFLA_BR_FORWARD_DELAY,
			    brinfo->forward_delay);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_HELLO_TIME)
		NLA_PUT_U32(msg, IFLA_BR_HELLO_TIME,
			    brinfo->hello_time);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_MAX_AGE)
		NLA_PUT_U32(msg, IFLA_BR_MAX_AGE,
			    brinfo->max_age);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_AGEING_TIME)
		NLA_PUT_U32(msg, IFLA_BR_AGEING_TIME,
			    brinfo->ageing_time);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_STP_STATE)
		NLA_PUT_U32(msg, IFLA_BR_STP_STATE,
			    brinfo->stp_state);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_PRIORITY)
		NLA_PUT_U16(msg, IFLA_BR_PRIORITY,
			    brinfo->priority);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_VLAN_FILTERING)
		NLA_PUT_U8(msg, IFLA_BR_VLAN_FILTERING,
			    brinfo->vlan_filtering);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_VLAN_PROTOCOL)
		NLA_PUT_U16(msg, IFLA_BR_VLAN_PROTOCOL,
			    brinfo->vlan_protocol);

	if (brinfo->ce_mask & BRIDGE_INFO_ATTR_MCAST_SNOOPING)
		NLA_PUT_U8(msg, IFLA_BR_MCAST_SNOOPING,
				brinfo->mcast_snooping);

	nla_nest_end(msg, data);

nla_put_failure:

	return 0;
}

static int bridge_info_compare(struct rtnl_link *link_a,
			       struct rtnl_link *link_b, uint32_t attrs,
			       int flags)
{
        struct bridge_info *a = link_a->l_info;
        struct bridge_info *b = link_b->l_info;
        int diff = 0;

#define BRIDGE_INFO_DIFF(ATTR, EXPR) ATTR_DIFF(attrs, BRIDGE_INFO_ATTR_##ATTR, a, b, EXPR)

	diff |= BRIDGE_INFO_DIFF(FORWARD_DELAY, a->forward_delay !=
				 b->forward_delay);
	diff |= BRIDGE_INFO_DIFF(HELLO_TIME, a->hello_time != b->hello_time);
	diff |= BRIDGE_INFO_DIFF(MAX_AGE, a->max_age != b->max_age);
	diff |= BRIDGE_INFO_DIFF(AGEING_TIME, a->ageing_time != b->ageing_time);
	diff |= BRIDGE_INFO_DIFF(STP_STATE, a->stp_state != b->stp_state);
	diff |= BRIDGE_INFO_DIFF(PRIORITY, a->priority != b->priority);
	diff |= BRIDGE_INFO_DIFF(VLAN_FILTERING, a->vlan_filtering != b->vlan_filtering);
	diff |= BRIDGE_INFO_DIFF(VLAN_PROTOCOL, a->vlan_protocol !=
				 b->vlan_protocol);
	diff |= BRIDGE_INFO_DIFF(MCAST_SNOOPING, a->mcast_snooping != b->mcast_snooping);
#undef VXLAN_DIFF

	return diff;
}

/**
 * Get bridge port state
 * @arg link            Link object
 *
 * @return port state if link has bridge port af data, otherwise -1 is returned.
 */
int rtnl_link_bridge_get_vlan_filtering(struct rtnl_link *link)
{
	struct bridge_info *data = link->l_info;

	if (data && (data->ce_mask & BRIDGE_INFO_ATTR_VLAN_FILTERING))
		return (data->vlan_filtering ? 1: 0);

	return 0;
}

int rtnl_link_bridge_get_vlan_protocol(struct rtnl_link *link)
{
	struct bridge_info *data = link->l_info;

	if (data && (data->ce_mask & BRIDGE_INFO_ATTR_VLAN_PROTOCOL))
		return data->vlan_protocol;

	return 0;
}

int rtnl_link_bridge_get_mcast_snooping(struct rtnl_link *link)
{
	struct bridge_info *data = link->l_info;

	if (data && (data->ce_mask & BRIDGE_INFO_ATTR_MCAST_SNOOPING))
		return (data->mcast_snooping ? 1 : 0);

	return 0;
}

int rtnl_link_bridge_get_ageing_time(struct rtnl_link *link)
{
	struct bridge_info *data = link->l_info;

	if (data && (data->ce_mask & BRIDGE_INFO_ATTR_AGEING_TIME))
		return data->ageing_time;

	return 0;
}

static struct rtnl_link_info_ops bridge_info_ops = {
	.io_name		= "bridge",
	.io_alloc		= bridge_info_alloc,
	.io_parse		= bridge_info_parse,
	.io_dump = {
	    [NL_DUMP_LINE]	= bridge_info_dump_line,
	    [NL_DUMP_DETAILS]	= bridge_info_dump_details,
	},
	.io_clone		= bridge_info_clone,
	.io_put_attrs		= bridge_info_put_attrs,
	.io_free		= bridge_info_free,
	.io_compare		= bridge_info_compare,
};

static void __init bridge_init(void)
{
	rtnl_link_af_register(&bridge_ops);
	rtnl_link_register_info(&bridge_info_ops);
}

static void __exit bridge_exit(void)
{
	rtnl_link_af_unregister(&bridge_ops);
	rtnl_link_unregister_info(&bridge_info_ops);
}

/** @} */
