/*
 * lib/route/link/bonding.c	Bonding Link Module
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation version 2.1
 *	of the License.
 *
 * Copyright (c) 2011-2013 Thomas Graf <tgraf@suug.ch>
 */

/**
 * @ingroup link
 * @defgroup bonding Bonding
 *
 * @details
 * \b Link Type Name: "bond"
 *
 * @route_doc{link_bonding, Bonding Documentation}
 * @{
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/attr.h>
#include <netlink/route/link/bonding.h>
#include <netlink-private/route/link/api.h>

/**
 * Allocate link object of type bond
 *
 * @return Allocated link object or NULL.
 */
struct rtnl_link *rtnl_link_bond_alloc(void)
{
	struct rtnl_link *link;
	int err;

	if (!(link = rtnl_link_alloc()))
		return NULL;

	if ((err = rtnl_link_set_type(link, "bond")) < 0) {
		rtnl_link_put(link);
		return NULL;
	}

	return link;
}

/**
 * Create a new kernel bonding device
 * @arg sock		netlink socket
 * @arg name		name of bonding device or NULL
 * @arg opts		bonding options (currently unused)
 *
 * Creates a new bonding device in the kernel. If no name is
 * provided, the kernel will automatically pick a name of the
 * form "type%d" (e.g. bond0, vlan1, etc.)
 *
 * The \a opts argument is currently unused. In the future, it
 * may be used to carry additional bonding options to be set
 * when creating the bonding device.
 *
 * @note When letting the kernel assign a name, it will become
 *       difficult to retrieve the interface afterwards because
 *       you have to guess the name the kernel has chosen. It is
 *       therefore not recommended to not provide a device name.
 *
 * @see rtnl_link_bond_enslave()
 * @see rtnl_link_bond_release()
 *
 * @return 0 on success or a negative error code
 */
int rtnl_link_bond_add(struct nl_sock *sock, const char *name,
		       struct rtnl_link *opts)
{
	struct rtnl_link *link;
	int err;

	if (!(link = rtnl_link_bond_alloc()))
		return -NLE_NOMEM;

	if (!name && opts)
		name = rtnl_link_get_name(opts);

	if (name)
		rtnl_link_set_name(link, name);

	err = rtnl_link_add(sock, link, NLM_F_CREATE);

	rtnl_link_put(link);

	return err;
}

/**
 * Add a link to a bond (enslave)
 * @arg sock		netlink socket
 * @arg master		ifindex of bonding master
 * @arg slave		ifindex of slave link to add to bond
 *
 * This function is identical to rtnl_link_bond_enslave() except that
 * it takes interface indices instead of rtnl_link objcets.
 *
 * @see rtnl_link_bond_enslave()
 *
 * @return 0 on success or a negative error code.
 */
int rtnl_link_bond_enslave_ifindex(struct nl_sock *sock, int master,
				   int slave)
{
	struct rtnl_link *link;
	int err;

	if (!(link = rtnl_link_bond_alloc()))
		return -NLE_NOMEM;

	rtnl_link_set_ifindex(link, slave);
	rtnl_link_set_master(link, master);

	if ((err = rtnl_link_change(sock, link, link, 0)) < 0)
		goto errout;

	rtnl_link_put(link);

	/*
	 * Due to the kernel not signaling whether this opertion is
	 * supported or not, we will retrieve the attribute to see  if the
	 * request was successful. If the master assigned remains unchanged
	 * we will return NLE_OPNOTSUPP to allow performing backwards
	 * compatibility of some sort.
	 */
	if ((err = rtnl_link_get_kernel(sock, slave, NULL, &link)) < 0)
		return err;

	if (rtnl_link_get_master(link) != master)
		err = -NLE_OPNOTSUPP;

errout:
	rtnl_link_put(link);

	return err;
}

/**
 * Add a link to a bond (enslave)
 * @arg sock		netlink socket
 * @arg master		bonding master
 * @arg slave		slave link to add to bond
 *
 * Constructs a RTM_NEWLINK or RTM_SETLINK message adding the slave to
 * the master and sends the request via the specified netlink socket.
 *
 * @note The feature of enslaving/releasing via netlink has only been added
 *       recently to the kernel (Feb 2011). Also, the kernel does not signal
 *       if the operation is not supported. Therefore this function will
 *       verify if the master assignment has changed and will return
 *       -NLE_OPNOTSUPP if it did not.
 *
 * @see rtnl_link_bond_enslave_ifindex()
 * @see rtnl_link_bond_release()
 *
 * @return 0 on success or a negative error code.
 */
int rtnl_link_bond_enslave(struct nl_sock *sock, struct rtnl_link *master,
			   struct rtnl_link *slave)
{
	return rtnl_link_bond_enslave_ifindex(sock,
				rtnl_link_get_ifindex(master),
				rtnl_link_get_ifindex(slave));
}

/**
 * Release a link from a bond
 * @arg sock		netlink socket
 * @arg slave		slave link to be released
 *
 * This function is identical to rtnl_link_bond_release() except that
 * it takes an interface index instead of a rtnl_link object.
 *
 * @see rtnl_link_bond_release()
 *
 * @return 0 on success or a negative error code.
 */
int rtnl_link_bond_release_ifindex(struct nl_sock *sock, int slave)
{
	return rtnl_link_bond_enslave_ifindex(sock, 0, slave);
}

/**
 * Release a link from a bond
 * @arg sock		netlink socket
 * @arg slave		slave link to be released
 *
 * Constructs a RTM_NEWLINK or RTM_SETLINK message releasing the slave from
 * its master and sends the request via the specified netlink socket.
 *
 * @note The feature of enslaving/releasing via netlink has only been added
 *       recently to the kernel (Feb 2011). Also, the kernel does not signal
 *       if the operation is not supported. Therefore this function will
 *       verify if the master assignment has changed and will return
 *       -NLE_OPNOTSUPP if it did not.
 *
 * @see rtnl_link_bond_release_ifindex()
 * @see rtnl_link_bond_enslave()
 *
 * @return 0 on success or a negative error code.
 */
int rtnl_link_bond_release(struct nl_sock *sock, struct rtnl_link *slave)
{
	return rtnl_link_bond_release_ifindex(sock,
				rtnl_link_get_ifindex(slave));
}

/* Bond info attributes */
#define BOND_INFO_ATTR_MODE	(1 << 0)
#define BOND_INFO_ATTR_ACTIVE_SLAVE	(1 << 1)
#define BOND_INFO_ATTR_MIIMON	(1 << 2)
#define BOND_INFO_ATTR_UPDELAY	(1 << 3)
#define BOND_INFO_ATTR_DOWNDELAY	(1 << 4)
#define BOND_INFO_ATTR_USE_CARRIER	(1 << 5)
#define BOND_INFO_ATTR_ARP_INTERVAL	(1 << 6)
#define BOND_INFO_ATTR_ARP_IP_TARGET	(1 << 7)
#define BOND_INFO_ATTR_ARP_VALIDATE	(1 << 8)
#define BOND_INFO_ATTR_ARP_ALL_TARGETS	(1 << 9)
#define BOND_INFO_ATTR_PRIMARY		(1 << 10)
#define BOND_INFO_ATTR_PRIMARY_RESELECT	(1 << 11)
#define BOND_INFO_ATTR_FAIL_OVER_MAC	(1 << 12)
#define BOND_INFO_ATTR_XMIT_HASH_POLICY	(1 << 13)
#define BOND_INFO_ATTR_RESEND_IGMP	(1 << 14)
#define BOND_INFO_ATTR_NUM_PEER_NOTIF	(1 << 15)
#define BOND_INFO_ATTR_ALL_SLAVES_ACTIVE	(1 << 16)
#define BOND_INFO_ATTR_MIN_LINKS	(1 << 17)
#define BOND_INFO_ATTR_LP_INTERVAL	(1 << 18)
#define BOND_INFO_ATTR_PACKETS_PER_SLAVE	(1 << 19)
#define BOND_INFO_ATTR_AD_LACP_RATE	(1 << 20)
#define BOND_INFO_ATTR_AD_SELECT	(1 << 21)
#define BOND_INFO_ATTR_AD_INFO		(1 << 22)
#define BOND_INFO_ATTR_AD_LACP_BYPASS	(1 << 23)

static struct nla_policy bond_policy[IFLA_BOND_MAX + 1] = {
	[IFLA_BOND_MODE]		= { .type = NLA_U8 },
	[IFLA_BOND_ACTIVE_SLAVE]	= { .type = NLA_U32 },
	[IFLA_BOND_MIIMON]		= { .type = NLA_U32 },
	[IFLA_BOND_UPDELAY]		= { .type = NLA_U32 },
	[IFLA_BOND_DOWNDELAY]		= { .type = NLA_U32 },
	[IFLA_BOND_USE_CARRIER]		= { .type = NLA_U8 },
	[IFLA_BOND_ARP_INTERVAL]	= { .type = NLA_U32 },
	[IFLA_BOND_ARP_IP_TARGET]	= { .type = NLA_NESTED },
	[IFLA_BOND_ARP_VALIDATE]	= { .type = NLA_U32 },
	[IFLA_BOND_ARP_ALL_TARGETS]	= { .type = NLA_U32 },
	[IFLA_BOND_PRIMARY]		= { .type = NLA_U32 },
	[IFLA_BOND_PRIMARY_RESELECT]	= { .type = NLA_U8 },
	[IFLA_BOND_FAIL_OVER_MAC]	= { .type = NLA_U8 },
	[IFLA_BOND_XMIT_HASH_POLICY]	= { .type = NLA_U8 },
	[IFLA_BOND_RESEND_IGMP]		= { .type = NLA_U32 },
	[IFLA_BOND_NUM_PEER_NOTIF]	= { .type = NLA_U8 },
	[IFLA_BOND_ALL_SLAVES_ACTIVE]	= { .type = NLA_U8 },
	[IFLA_BOND_MIN_LINKS]		= { .type = NLA_U32 },
	[IFLA_BOND_LP_INTERVAL]		= { .type = NLA_U32 },
	[IFLA_BOND_PACKETS_PER_SLAVE]	= { .type = NLA_U32 },
	[IFLA_BOND_AD_LACP_RATE]	= { .type = NLA_U8 },
	[IFLA_BOND_AD_SELECT]		= { .type = NLA_U8 },
	[IFLA_BOND_AD_INFO]		= { .type = NLA_NESTED },
	[IFLA_BOND_AD_LACP_BYPASS]	= { .type = NLA_U8 },
};

static struct nla_policy bond_ad_info_policy[IFLA_BOND_AD_INFO_MAX + 1] = {
	[IFLA_BOND_AD_INFO_AGGREGATOR]	= { .type = NLA_U16 },
	[IFLA_BOND_AD_INFO_NUM_PORTS]	= { .type = NLA_U16 },
	[IFLA_BOND_AD_INFO_ACTOR_KEY]	= { .type = NLA_U16 },
	[IFLA_BOND_AD_INFO_PARTNER_KEY]	= { .type = NLA_U16 },
	[IFLA_BOND_AD_INFO_PARTNER_MAC]	= { .type = NLA_BINARY },
};

struct bond_ad_info
{
	uint16_t	aggregator;
	uint16_t	num_ports;
	uint16_t	actor_key;
	uint16_t	partner_key;
	struct nl_addr *partner_mac;
};

struct bond_info
{
	uint8_t		mode;
	uint32_t	active_slave;
	uint32_t	miimon;
	uint32_t	updelay;
	uint32_t	downdelay;
	uint8_t		use_carrier;
	uint32_t	arp_interval;
	uint32_t	arp_validate;
	uint32_t	arp_all_targets;
	uint32_t	primary;
	uint8_t		primary_reselect;
	uint8_t		fail_over_mac;
	uint8_t		xmit_hash_policy;
	uint32_t	resend_igmp;
	uint8_t		num_peer_notif;
	uint8_t		all_slaves_active;
	uint32_t	min_links;
	uint32_t	lp_interval;
	uint32_t	packets_per_slave;
	uint8_t		ad_lacp_rate;
	uint8_t		ad_select;
	struct bond_ad_info *ad_info;
	uint8_t		ad_lacp_bypass;
	uint32_t	ce_mask;
};

struct bond_slave_info
{
	uint8_t		state;
	uint8_t		mii_status;
	uint32_t	link_fail_count;
	struct nl_addr *perm_hwaddr;
	uint16_t	queue_id;
	uint16_t	ad_agg_id;
	uint16_t	ad_partner_oper_port_state;
	uint8_t		ad_actor_oper_port_state;
	uint32_t	ad_rx_bypass;
	uint32_t	ce_mask;
};

static int bond_info_alloc(struct rtnl_link *link)
{
	struct bond_info *bondinfo;

	if (link->l_info)
		memset(link->l_info, 0, sizeof(*bondinfo));
	else {
		if ((bondinfo = calloc(1, sizeof(*bondinfo))) == NULL)
			return -NLE_NOMEM;

		link->l_info = bondinfo;
	}

	return 0;
}

static int bond_ad_info_alloc(struct bond_info *bondinfo)
{
	struct bond_ad_info *bondadinfo;

	if (bondinfo->ad_info)
		memset(bondinfo->ad_info, 0, sizeof(*bondadinfo));
	else {
		if ((bondadinfo = calloc(1, sizeof(*bondadinfo))) == NULL)
			return -NLE_NOMEM;

		bondinfo->ad_info = bondadinfo;
	}

	return 0;
}

static int bond_info_parse_ad_info(struct nlattr *adinfo_attr,
				   struct bond_info *bondinfo)
{
	struct nlattr *tb[IFLA_BOND_AD_INFO_MAX + 1];
	struct bond_ad_info *bondadinfo;
	int err;

	if ((err = bond_ad_info_alloc(bondinfo)) < 0)
		goto errout;

	bondadinfo = bondinfo->ad_info;

	err = nla_parse_nested(tb, IFLA_BOND_AD_INFO_MAX, adinfo_attr,
			       bond_ad_info_policy);
	if (err < 0)
		goto errout;

	if (tb[IFLA_BOND_AD_INFO_AGGREGATOR])
		bondadinfo->aggregator = nla_get_u32(tb[IFLA_BOND_AD_INFO_AGGREGATOR]);

	if (tb[IFLA_BOND_AD_INFO_NUM_PORTS])
		bondadinfo->num_ports =
			nla_get_u32(tb[IFLA_BOND_AD_INFO_NUM_PORTS]);

	if (tb[IFLA_BOND_AD_INFO_ACTOR_KEY])
		bondadinfo->actor_key =
			nla_get_u32(tb[IFLA_BOND_AD_INFO_ACTOR_KEY]);

	if (tb[IFLA_BOND_AD_INFO_PARTNER_KEY])
		bondadinfo->partner_key =
			nla_get_u32(tb[IFLA_BOND_AD_INFO_PARTNER_KEY]);

	if (tb[IFLA_BOND_AD_INFO_PARTNER_MAC]) {
		bondadinfo->partner_mac =
			nl_addr_alloc_attr(tb[IFLA_BOND_AD_INFO_PARTNER_MAC],
					   AF_UNSPEC);
		if (!bondadinfo->partner_mac) {
			err = -NLE_NOMEM;
			goto errout;
		}
	}

	return 0;
errout:
	if (bondinfo->ad_info) {
		free(bondinfo->ad_info);
		bondinfo->ad_info = NULL;
	}

	return err;
}

static int bond_info_parse(struct rtnl_link *link, struct nlattr *data,
			   struct nlattr *xstats)
{
	struct nlattr *tb[IFLA_BOND_MAX+1];
	struct bond_info *bondinfo;
	int err;

	NL_DBG(3, "Parsing Bond link info\n");

	if ((err = nla_parse_nested(tb, IFLA_BOND_MAX, data,
				    bond_policy)) < 0)
		goto errout;

	if ((err = bond_info_alloc(link)) < 0)
		goto errout;

	bondinfo = link->l_info;

	if (tb[IFLA_BOND_MODE]) {
		bondinfo->mode = nla_get_u32(tb[IFLA_BOND_MODE]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_MODE;
	}

	if (tb[IFLA_BOND_ACTIVE_SLAVE]) {
		bondinfo->active_slave = nla_get_u32(tb[IFLA_BOND_ACTIVE_SLAVE]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_ACTIVE_SLAVE;
	}

	if (tb[IFLA_BOND_MIIMON]) {
		bondinfo->miimon = nla_get_u32(tb[IFLA_BOND_MIIMON]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_MIIMON;
	}

	if (tb[IFLA_BOND_UPDELAY]) {
		bondinfo->updelay = nla_get_u32(tb[IFLA_BOND_UPDELAY]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_UPDELAY;
	}

	if (tb[IFLA_BOND_DOWNDELAY]) {
		bondinfo->downdelay = nla_get_u32(tb[IFLA_BOND_DOWNDELAY]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_DOWNDELAY;
	}

	if (tb[IFLA_BOND_USE_CARRIER]) {
		bondinfo->use_carrier = nla_get_u32(tb[IFLA_BOND_USE_CARRIER]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_USE_CARRIER;
	}

	if (tb[IFLA_BOND_ARP_INTERVAL]) {
		bondinfo->arp_interval = nla_get_u32(tb[IFLA_BOND_ARP_INTERVAL]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_ARP_INTERVAL;
	}

	if (tb[IFLA_BOND_ARP_VALIDATE]) {
		bondinfo->arp_validate = nla_get_u32(tb[IFLA_BOND_ARP_VALIDATE]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_ARP_VALIDATE;
	}

	if (tb[IFLA_BOND_ARP_ALL_TARGETS]) {
		bondinfo->arp_all_targets = nla_get_u32(tb[IFLA_BOND_ARP_ALL_TARGETS]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_ARP_ALL_TARGETS;
	}

	if (tb[IFLA_BOND_PRIMARY]) {
		bondinfo->primary = nla_get_u32(tb[IFLA_BOND_PRIMARY]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_PRIMARY;
	}

	if (tb[IFLA_BOND_PRIMARY_RESELECT]) {
		bondinfo->primary_reselect = nla_get_u32(tb[IFLA_BOND_PRIMARY_RESELECT]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_PRIMARY_RESELECT;
	}

	if (tb[IFLA_BOND_FAIL_OVER_MAC]) {
		bondinfo->fail_over_mac = nla_get_u32(tb[IFLA_BOND_FAIL_OVER_MAC]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_FAIL_OVER_MAC;
	}

	if (tb[IFLA_BOND_XMIT_HASH_POLICY]) {
		bondinfo->xmit_hash_policy =
			nla_get_u32(tb[IFLA_BOND_XMIT_HASH_POLICY]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_XMIT_HASH_POLICY;
	}

	if (tb[IFLA_BOND_RESEND_IGMP]) {
		bondinfo->resend_igmp = nla_get_u32(tb[IFLA_BOND_RESEND_IGMP]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_RESEND_IGMP;
	}

	if (tb[IFLA_BOND_NUM_PEER_NOTIF]) {
		bondinfo->num_peer_notif = nla_get_u32(tb[IFLA_BOND_NUM_PEER_NOTIF]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_NUM_PEER_NOTIF;
	}

	if (tb[IFLA_BOND_ALL_SLAVES_ACTIVE]) {
		bondinfo->all_slaves_active =
			nla_get_u32(tb[IFLA_BOND_ALL_SLAVES_ACTIVE]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_ALL_SLAVES_ACTIVE;
	}

	if (tb[IFLA_BOND_MIN_LINKS]) {
		bondinfo->min_links = nla_get_u32(tb[IFLA_BOND_MIN_LINKS]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_MIN_LINKS;
	}

	if (tb[IFLA_BOND_LP_INTERVAL]) {
		bondinfo->lp_interval = nla_get_u32(tb[IFLA_BOND_LP_INTERVAL]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_LP_INTERVAL;
	}

	if (tb[IFLA_BOND_PACKETS_PER_SLAVE]) {
		bondinfo->packets_per_slave =
			nla_get_u32(tb[IFLA_BOND_PACKETS_PER_SLAVE]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_PACKETS_PER_SLAVE;
	}

	if (tb[IFLA_BOND_AD_LACP_RATE]) {
		bondinfo->ad_lacp_rate = nla_get_u32(tb[IFLA_BOND_AD_LACP_RATE]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_AD_LACP_RATE;
	}

	if (tb[IFLA_BOND_AD_SELECT]) {
		bondinfo->ad_select = nla_get_u32(tb[IFLA_BOND_AD_SELECT]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_AD_SELECT;
	}

	if (tb[IFLA_BOND_AD_INFO]) {
		err = bond_info_parse_ad_info(tb[IFLA_BOND_AD_INFO], bondinfo);
		if (err < 0)
			goto errout;
		bondinfo->ce_mask |= BOND_INFO_ATTR_AD_INFO;
	}

	if (tb[IFLA_BOND_AD_LACP_BYPASS]) {
		bondinfo->ad_lacp_bypass =
			nla_get_u8(tb[IFLA_BOND_AD_LACP_BYPASS]);
		bondinfo->ce_mask |= BOND_INFO_ATTR_AD_LACP_BYPASS;
	}

	err = 0;
errout:
	return err;
}

#define BOND_SLAVE_INFO_ATTR_STATE		(1 << 0)
#define BOND_SLAVE_INFO_ATTR_MII_STATUS		(1 << 1)
#define BOND_SLAVE_INFO_ATTR_FAIL_COUNT		(1 << 2)
#define BOND_SLAVE_INFO_ATTR_PERM_HWADDR	(1 << 3)
#define BOND_SLAVE_INFO_ATTR_QUEUE_ID		(1 << 4)
#define BOND_SLAVE_INFO_ATTR_AD_AGG_ID		(1 << 5)
#define BOND_SLAVE_INFO_ATTR_AD_ACTOR_OPER_PORT_STATE	(1 << 6)
#define BOND_SLAVE_INFO_ATTR_AD_PARTNER_OPER_PORT_STATE	(1 << 7)
#define BOND_SLAVE_INFO_ATTR_AD_RX_BYPASS	(1 << 8)

static int bond_slave_info_alloc(struct rtnl_link *link)
{
	struct bond_slave_info *slaveinfo;

	if (link->l_info_slave)
		memset(link->l_info_slave, 0, sizeof(*slaveinfo));
	else {
		if ((slaveinfo = calloc(1, sizeof(*slaveinfo))) == NULL)
			return -NLE_NOMEM;

		link->l_info_slave = slaveinfo;
	}

	return 0;
}

static void bond_slave_info_free(struct rtnl_link *link)
{
	struct bond_slave_info *slave_info = link->l_info_slave;

	free(slave_info->perm_hwaddr);
	free(slave_info);

	link->l_info_slave = NULL;
}

static struct nla_policy bond_slave_policy[IFLA_BOND_SLAVE_MAX + 1] = {
	[IFLA_BOND_SLAVE_STATE]			     = { .type = NLA_U8 },
	[IFLA_BOND_SLAVE_MII_STATUS]		     = { .type = NLA_U8 },
	[IFLA_BOND_SLAVE_LINK_FAILURE_COUNT]	     = { .type = NLA_U32 },
	[IFLA_BOND_SLAVE_PERM_HWADDR]		     = { .type = NLA_BINARY,
						         .maxlen = MAX_ADDR_LEN },
	[IFLA_BOND_SLAVE_QUEUE_ID]		     = { .type = NLA_U16 },
	[IFLA_BOND_SLAVE_AD_AGGREGATOR_ID]	     = { .type = NLA_U16 },
	[IFLA_BOND_SLAVE_AD_ACTOR_OPER_PORT_STATE]   = { .type = NLA_U8 },
	[IFLA_BOND_SLAVE_AD_PARTNER_OPER_PORT_STATE] = { .type = NLA_U16 },
	[IFLA_BOND_SLAVE_AD_RX_BYPASS]	     = { .type = NLA_U8 },
};

static int bond_slave_info_parse(struct rtnl_link *link, struct nlattr *data)
{
	struct nlattr *tb[IFLA_BOND_SLAVE_MAX + 1], *tmp;
	struct bond_slave_info *slave_info;
	int err;

	NL_DBG(3, "Parsing Bond slave link info\n");

	if ((err = nla_parse_nested(tb, IFLA_BOND_SLAVE_MAX, data,
				    bond_slave_policy)) < 0)
		goto errout;

	if ((err = bond_slave_info_alloc(link)) < 0)
		goto errout;

	slave_info = link->l_info_slave;

	tmp = tb[IFLA_BOND_SLAVE_STATE];
	if (tmp) {
		slave_info->state = nla_get_u8(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_STATE;
	}

	tmp = tb[IFLA_BOND_SLAVE_MII_STATUS];
	if (tmp) {
		slave_info->mii_status = nla_get_u8(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_MII_STATUS;
	}

	tmp = tb[IFLA_BOND_SLAVE_LINK_FAILURE_COUNT];
	if (tmp) {
		slave_info->link_fail_count = nla_get_u32(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_FAIL_COUNT;
	}

	tmp = tb[IFLA_BOND_SLAVE_PERM_HWADDR];
	if (tmp) {
		slave_info->perm_hwaddr = nl_addr_alloc_attr(tmp, AF_UNSPEC);
		if (!slave_info->perm_hwaddr) {
			err = -NLE_NOMEM;
			goto errout;
		}
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_PERM_HWADDR;
	}

	tmp = tb[IFLA_BOND_SLAVE_QUEUE_ID];
	if (tmp) {
		slave_info->queue_id = nla_get_u16(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_QUEUE_ID;
	}

	tmp = tb[IFLA_BOND_SLAVE_AD_AGGREGATOR_ID];
	if (tmp) {
		slave_info->ad_agg_id = nla_get_u16(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_AD_AGG_ID;
	}

	tmp = tb[IFLA_BOND_SLAVE_AD_ACTOR_OPER_PORT_STATE];
	if (tmp) {
		slave_info->ad_actor_oper_port_state = nla_get_u8(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_AD_ACTOR_OPER_PORT_STATE;
	}

	tmp = tb[IFLA_BOND_SLAVE_AD_PARTNER_OPER_PORT_STATE];
	if (tmp) {
		slave_info->ad_partner_oper_port_state = nla_get_u16(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_AD_PARTNER_OPER_PORT_STATE;
	}

	tmp = tb[IFLA_BOND_SLAVE_AD_RX_BYPASS];
	if (tmp) {
		slave_info->ad_rx_bypass = nla_get_u8(tmp);
		slave_info->ce_mask |= BOND_SLAVE_INFO_ATTR_AD_RX_BYPASS;
	}

	err = 0;
errout:
	return err;
}

static int bond_slave_info_compare(struct rtnl_link *link_a,
				   struct rtnl_link *link_b,
				   uint32_t attrs, int flags)
{
        struct bond_slave_info *a = link_a->l_info_slave;
        struct bond_slave_info *b = link_b->l_info_slave;
        int diff = 0;

#define BOND_SINFO_DIFF(ATTR, EXPR)  \
		ATTR_DIFF(attrs, BOND_SLAVE_INFO_ATTR_##ATTR, a, b, EXPR)

	diff |= BOND_SINFO_DIFF(STATE, a->state != b->state);
	diff |= BOND_SINFO_DIFF(MII_STATUS, a->mii_status != b->mii_status);
	diff |= BOND_SINFO_DIFF(FAIL_COUNT,
				a->link_fail_count != b->link_fail_count);
	diff |= BOND_SINFO_DIFF(PERM_HWADDR,
				nl_addr_cmp(a->perm_hwaddr, b->perm_hwaddr));
	diff |= BOND_SINFO_DIFF(QUEUE_ID, a->queue_id != b->queue_id);
	diff |= BOND_SINFO_DIFF(AD_AGG_ID, a->ad_agg_id != b->ad_agg_id);
	diff |= BOND_SINFO_DIFF(AD_ACTOR_OPER_PORT_STATE,
				a->ad_actor_oper_port_state != b->ad_actor_oper_port_state);
	diff |= BOND_SINFO_DIFF(AD_PARTNER_OPER_PORT_STATE,
				a->ad_partner_oper_port_state != b->ad_partner_oper_port_state);
	diff |= BOND_SINFO_DIFF(AD_RX_BYPASS, a->ad_rx_bypass != b->ad_rx_bypass);
#undef BOND_SINFO_DIFF

	return diff;
}

static void bond_slave_info_dump_line(struct rtnl_link *link,
				      struct nl_dump_params *p)
{
	struct bond_slave_info *slave_info = link->l_info_slave;

	if (slave_info->ce_mask)
		nl_dump(p, "     ");

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_STATE)
		nl_dump(p, "  bond-slave-state %d ", slave_info->state);

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_MII_STATUS)
		nl_dump(p, "  bond-slave-mii-status %d ", slave_info->mii_status);

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_FAIL_COUNT)
		nl_dump(p, "  bond-slave-failure-count %u ", slave_info->link_fail_count);

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_PERM_HWADDR)
		nl_dump(p, "  bond-slave-perm-hwaddr TBD ");

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_QUEUE_ID)
		nl_dump(p, "  bond-slave-queue-id %u", slave_info->queue_id);

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_AD_AGG_ID)
		nl_dump(p, "  bond-slave-ad-aggregator %u", slave_info->ad_agg_id);

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_AD_ACTOR_OPER_PORT_STATE)
		nl_dump(p, "  bond-slave-ad-actor-oper-port-state %u", slave_info->ad_actor_oper_port_state);

	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_AD_PARTNER_OPER_PORT_STATE)
		nl_dump(p, "  bond-slave-ad-partner-oper-port-state %u", slave_info->ad_partner_oper_port_state);
	if (slave_info->ce_mask & BOND_SLAVE_INFO_ATTR_AD_RX_BYPASS)
		nl_dump(p, "  bond-slave-ad-rx-bypass %u",
			slave_info->ad_rx_bypass);
}

static void bond_ad_info_free(struct bond_info *bondinfo)
{
	struct bond_ad_info *adinfo = bondinfo->ad_info;

	if (!adinfo)
		return;

	if (adinfo->partner_mac)
		free(adinfo->partner_mac);
	free(adinfo);
	bondinfo->ad_info = NULL;
}

static void bond_info_free(struct rtnl_link *link)
{
	bond_ad_info_free(link->l_info);
	free(link->l_info);
	link->l_info = NULL;
}

static void bond_info_dump_line(struct rtnl_link *link,
				struct nl_dump_params *p)
{
	struct bond_info *bondinfo = link->l_info;

	if (bondinfo->ce_mask)
		nl_dump(p, "     ");

	if (bondinfo->ce_mask & BOND_INFO_ATTR_MODE)
		nl_dump(p, "bond-mode %d ", bondinfo->mode);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_ACTIVE_SLAVE)
		nl_dump(p, "bond-active-slave %d ", bondinfo->active_slave);
}

static void bond_ad_info_dump(struct bond_info *bondinfo,
			      struct nl_dump_params *p)
{
	struct bond_ad_info *bondadinfo = bondinfo->ad_info;
	char buf[64];

	if (!bondadinfo)
		return;

	nl_dump(p, "bond-ad-aggregator %d ", bondadinfo->aggregator);
	nl_dump(p, "bond-ad-num-ports %d ", bondadinfo->num_ports);
	nl_dump(p, "bond-ad-actor-key %d ", bondadinfo->actor_key);
	nl_dump(p, "bond-ad-partner-key %d ", bondadinfo->partner_key);
	nl_dump(p, "bond-ad-partner-mac %s ",
		nl_addr2str(bondadinfo->partner_mac, buf, sizeof(buf)));
}

static void bond_info_dump_details(struct rtnl_link *link,
				   struct nl_dump_params *p)
{
	struct bond_info *bondinfo = link->l_info;

	if (bondinfo->ce_mask)
		nl_dump(p, "     ");

	if (bondinfo->ce_mask & BOND_INFO_ATTR_MODE)
		nl_dump(p, "bond-mode %d ", bondinfo->mode);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_ACTIVE_SLAVE)
		nl_dump(p, "bond-active-slave %d ", bondinfo->active_slave);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_MIIMON)
		nl_dump(p, "bond-miimon %d ", bondinfo->miimon);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_UPDELAY)
		nl_dump(p, "bond-updelay %d ", bondinfo->updelay);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_DOWNDELAY)
		nl_dump(p, "bond-downdelay %d ", bondinfo->downdelay);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_USE_CARRIER)
		nl_dump(p, "bond-use-carrier %d ", bondinfo->use_carrier);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_ARP_INTERVAL)
		nl_dump(p, "bond-arp-interval %d ", bondinfo->arp_interval);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_ARP_VALIDATE)
		nl_dump(p, "bond-arp-validate %d ", bondinfo->arp_validate);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_ARP_ALL_TARGETS)
		nl_dump(p, "bond-arp-all-targets %d ",
			bondinfo->arp_all_targets);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_PRIMARY)
		nl_dump(p, "bond-primary %d ", bondinfo->primary);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_PRIMARY_RESELECT)
		nl_dump(p, "bond-primary-reselect %d ",
			bondinfo->primary_reselect);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_FAIL_OVER_MAC)
		nl_dump(p, "bond-failover-mac %d ",
			bondinfo->fail_over_mac);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_XMIT_HASH_POLICY)
		nl_dump(p, "bond-xmit-hash-policy %d ",
			bondinfo->xmit_hash_policy);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_RESEND_IGMP)
		nl_dump(p, "bond-resend-igmp %d ", bondinfo->resend_igmp);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_NUM_PEER_NOTIF)
		nl_dump(p, "bond-peer-notif %d ", bondinfo->num_peer_notif);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_ALL_SLAVES_ACTIVE)
		nl_dump(p, "bond-all-slaves-active %d ",
			bondinfo->all_slaves_active);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_MIN_LINKS)
		nl_dump(p, "bond-min-links %d ", bondinfo->min_links);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_LP_INTERVAL)
		nl_dump(p, "bond-lp-interval %d ", bondinfo->lp_interval);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_PACKETS_PER_SLAVE)
		nl_dump(p, "bond-packets-per-slave %d ",
			bondinfo->packets_per_slave);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_AD_LACP_RATE)
		nl_dump(p, "bond-ad-lacp-rate %d ", bondinfo->ad_lacp_rate);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_AD_SELECT)
		nl_dump(p, "bond-ad-select %d ", bondinfo->ad_select);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_AD_INFO)
		bond_ad_info_dump(bondinfo, p);

	if (bondinfo->ce_mask & BOND_INFO_ATTR_AD_LACP_BYPASS)
		nl_dump(p, "bond-ad-lacp-bypass %d ", bondinfo->ad_lacp_bypass);

	nl_dump(p, "\n");
}

static int bond_info_clone(struct rtnl_link *dst, struct rtnl_link *src)
{
	struct bond_info *bdst, *bsrc = src->l_info;
	int err;

	dst->l_info = NULL;
	if ((err = rtnl_link_set_type(dst, "bond")) < 0)
		return err;
	bdst = dst->l_info;

	if (!bdst || !bsrc)
		return -NLE_NOMEM;

	memcpy(bdst, bsrc, sizeof(struct bond_info));

	return 0;
}

static int bond_ad_info_compare(struct bond_info *a, struct bond_info *b)
{
	struct bond_ad_info *a_ad_info = a->ad_info;
	struct bond_ad_info *b_ad_info = b->ad_info;

	if (!a_ad_info && !b_ad_info)
		return 0;

	if ((a_ad_info && !b_ad_info) || (!a_ad_info && b_ad_info))
		return 1;

	if (a_ad_info->aggregator != b_ad_info->aggregator)
		return 1;

	if (a_ad_info->num_ports != b_ad_info->num_ports)
		return 1;

	if (a_ad_info->actor_key != b_ad_info->actor_key)
		return 1;

	if (a_ad_info->partner_key != b_ad_info->partner_key)
		return 1;

	if (a_ad_info->partner_mac && b_ad_info->partner_mac) {
		if (nl_addr_cmp(a_ad_info->partner_mac, b_ad_info->partner_mac))
			return 1;
		else
			return 0;
	} else if (!a_ad_info->partner_mac && !b_ad_info->partner_mac) {
		return 0;
	} else {
		return 1;
	}

	return 0;
}

static int bond_info_compare(struct rtnl_link *link_a,
			     struct rtnl_link *link_b, uint32_t attrs,
			     int flags)
{
        struct bond_info *a = link_a->l_info;
        struct bond_info *b = link_b->l_info;
        int diff = 0;

#define BOND_INFO_DIFF(ATTR, EXPR) ATTR_DIFF(attrs, BOND_INFO_ATTR_##ATTR, a, b, EXPR)

	diff |= BOND_INFO_DIFF(MODE, a->mode != b->mode);
	diff |= BOND_INFO_DIFF(ACTIVE_SLAVE, a->active_slave != b->active_slave);
	diff |= BOND_INFO_DIFF(MIIMON, a->miimon != b->miimon);
	diff |= BOND_INFO_DIFF(UPDELAY, a->updelay != b->updelay);
	diff |= BOND_INFO_DIFF(DOWNDELAY, a->downdelay != b->downdelay);
	diff |= BOND_INFO_DIFF(USE_CARRIER, a->use_carrier != b->use_carrier);
	diff |= BOND_INFO_DIFF(ARP_INTERVAL, a->arp_interval != b->arp_interval);
	diff |= BOND_INFO_DIFF(ARP_VALIDATE, a->arp_validate != b->arp_validate);
	diff |= BOND_INFO_DIFF(ARP_ALL_TARGETS, a->arp_all_targets !=
			       b->arp_all_targets);
	diff |= BOND_INFO_DIFF(PRIMARY, a->primary != b->primary);
	diff |= BOND_INFO_DIFF(PRIMARY_RESELECT, a->primary_reselect !=
			       b->primary_reselect);
	diff |= BOND_INFO_DIFF(FAIL_OVER_MAC, a->fail_over_mac !=
			       b->fail_over_mac);
	diff |= BOND_INFO_DIFF(XMIT_HASH_POLICY, a->xmit_hash_policy !=
			       b->xmit_hash_policy);
	diff |= BOND_INFO_DIFF(RESEND_IGMP, a->resend_igmp != b->resend_igmp);
	diff |= BOND_INFO_DIFF(NUM_PEER_NOTIF, a->num_peer_notif !=
			       b->num_peer_notif);
	diff |= BOND_INFO_DIFF(ALL_SLAVES_ACTIVE, a->all_slaves_active !=
			       b->all_slaves_active);
	diff |= BOND_INFO_DIFF(MIN_LINKS, a->min_links != b->min_links);
	diff |= BOND_INFO_DIFF(LP_INTERVAL, a->lp_interval != b->lp_interval);
	diff |= BOND_INFO_DIFF(PACKETS_PER_SLAVE, a->packets_per_slave !=
			       b->packets_per_slave);
	diff |= BOND_INFO_DIFF(AD_LACP_RATE, a->ad_lacp_rate !=
			       b->ad_lacp_rate);
	diff |= BOND_INFO_DIFF(AD_SELECT, a->ad_select != b->ad_select);
	diff |= BOND_INFO_DIFF(AD_INFO, bond_ad_info_compare(a, b) != 0);
	diff |= BOND_INFO_DIFF(AD_LACP_BYPASS, a->ad_lacp_bypass !=
			       b->ad_lacp_bypass);

#undef BOND_INFO_DIFF

	return diff;
}

static struct rtnl_link_info_ops bond_info_ops = {
	.io_name		= "bond",
	.io_alloc		= bond_info_alloc,
	.io_parse		= bond_info_parse,
	.io_dump = {
	    [NL_DUMP_LINE]	= bond_info_dump_line,
	    [NL_DUMP_DETAILS]	= bond_info_dump_details,
	},
	.io_clone		= bond_info_clone,
	.io_free		= bond_info_free,
	.io_compare		= bond_info_compare,

	.io_slave_parse		= bond_slave_info_parse,
	.io_slave_alloc		= bond_slave_info_alloc,
	.io_slave_free		= bond_slave_info_free,
	.io_slave_compare	= bond_slave_info_compare,
	.io_slave_dump = {
	    [NL_DUMP_LINE]	= bond_slave_info_dump_line,
	},
};

/** @cond SKIP */
#define IS_BOND_LINK_ASSERT(link) \
	if ((link)->l_info_ops != &bond_info_ops) { \
		APPBUG("Link is not a bond link. set type \"bond\" first."); \
		return -NLE_OPNOTSUPP; \
	}
/** @endcond */

/**
 * Get bond mode
 * @arg link            Link object
 *
 * @return bond mode if link is bond, otherwise -EINVAL is returned.
 */
int rtnl_link_bond_get_mode(struct rtnl_link *link)
{
	struct bond_info *data = link->l_info;

	IS_BOND_LINK_ASSERT(link);

	if (data && (data->ce_mask & BOND_INFO_ATTR_MODE))
		return data->mode;

	return -EINVAL;
}

/**
 * Get active slave
 * @arg link            Link object
 *
 * @return active slave if link is bond, otherwise -EINVAL is returned.
 */
int rtnl_link_bond_get_active_slave(struct rtnl_link *link)
{
	struct bond_info *data = link->l_info;

	IS_BOND_LINK_ASSERT(link);

	if (data && (data->ce_mask & BOND_INFO_ATTR_ACTIVE_SLAVE))
		return data->active_slave;

	return -EINVAL;
}

/**
 * Get primary
 * @arg link            Link object
 *
 * @return bond primary if link is bond, otherwise -EINVAL is returned.
 */
int rtnl_link_bond_get_primary(struct rtnl_link *link)
{
	struct bond_info *data = link->l_info;

	IS_BOND_LINK_ASSERT(link);

	if (data && (data->ce_mask & BOND_INFO_ATTR_PRIMARY))
		return data->primary;

	return -EINVAL;
}

/**
 * Get bond all slaves active flag
 * @arg link            Link object
 *
 * @return bond all slaves active flag, if link is bond
 * otherwise -EINVAL is returned.
 */
int rtnl_link_bond_get_all_slaves_active(struct rtnl_link *link)
{
	struct bond_info *data = link->l_info;

	IS_BOND_LINK_ASSERT(link);

	if (data && (data->ce_mask & BOND_INFO_ATTR_ALL_SLAVES_ACTIVE))
		return data->all_slaves_active;

	return -EINVAL;
}

/**
 * Get lacp bypass
 * @arg link            Link object
 *
 * @return lacp bypass attr if link is bond
 */
int rtnl_link_bond_get_lacp_bypass(struct rtnl_link *link)
{
	struct bond_info *data = link->l_info;

	IS_BOND_LINK_ASSERT(link);

	if (data && (data->ce_mask & BOND_INFO_ATTR_AD_LACP_BYPASS))
		return data->ad_lacp_bypass;

	return 0;
}

/**
 * Is link a slave to a bond
 * @arg link            Link object
 *
 * @return 1 if link is a slave to a bond, 0 otherwise
 */
int rtnl_link_is_bond_slave(struct rtnl_link *link)
{
	return link->l_info_slave_ops == &bond_info_ops;
}

/**
 * Return slave state of link if link is enslaved to a bond device
 * @arg link            Link object
 *
 * @return slave state if link is a slave to a bond, otherwise -EINVAL
 */
int rtnl_link_bond_slave_state(struct rtnl_link *link)
{
	struct bond_slave_info *slave_info;

	if (link->l_info_slave_ops != &bond_info_ops)
		return -EINVAL;

	slave_info = link->l_info_slave;

	return slave_info->state;
}

/**
 * Return slave ad rx state of link if link is enslaved to a bond device
 * @arg link            Link object
 *
 * @return slave ad rx state if link is a slave to a bond, otherwise -EINVAL
 */
int rtnl_link_bond_slave_get_ad_rx_bypass(struct rtnl_link *link)
{
	struct bond_slave_info *slave_info;

	if (link->l_info_slave_ops != &bond_info_ops)
		return -EINVAL;

	slave_info = link->l_info_slave;
	if (!slave_info)
		return -EINVAL;

	return slave_info->ad_rx_bypass;
}


static void __init bonding_init(void)
{
	rtnl_link_register_info(&bond_info_ops);
}

static void __exit bonding_exit(void)
{
	rtnl_link_unregister_info(&bond_info_ops);
}

/** @} */
