/*
 * lib/route/link/bcm_knet.c      bcm_knet Link Info
 *
 *     This library is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU Lesser General Public
 *     License as published by the Free Software Foundation version 2.1
 *     of the License.
 *
 * Copyright (c) 2016 Cumulus Networks. All rights reserved.
 * Copyright (c) 2016 Roopa Prabhu <roopa@cumulusnetworks.com>
 */

/**
 * @ingroup link
 * @defgroup knet KNET
 * KNET bcm
 *
 * @details
 * \b Link Type Name: "knet"
 *
 * @route_doc{link_knet, KNET Documentation}
 *
 * @{
 */

#include <netlink-private/netlink.h>
#include <netlink/netlink.h>
#include <netlink/attr.h>
#include <netlink/utils.h>
#include <netlink/object.h>
#include <netlink/route/rtnl.h>
#include <netlink-private/route/link/api.h>
#include <netlink/route/link/bcm_knet.h>

#include <linux/if_link.h>
#include <linux-private/linux/rtnetlink.h>

static struct rtnl_link_info_ops knet_info_ops;

/** @cond SKIP */
#define KNET_HAS_UNIT    (1<<0)

struct knet_info {
	uint32_t        unit;
	uint32_t        mask;
};

/** @endcond */

static struct nla_policy knet_policy[IFLA_KNET_MAX + 1] = {
	[IFLA_KNET_UNIT]         = { .type = NLA_U32 },
};

/** @cond SKIP */
#define IS_KNET_LINK_ASSERT(link) \
	if ((link)->l_info_ops != &knet_info_ops) { \
		APPBUG("Link is not a KNET link. set type \"knet\" first."); \
		return -NLE_OPNOTSUPP; \
	}
/** @endcond */

static int knet_alloc(struct rtnl_link *link)
{
	struct knet_info *ki;

	if (link->l_info) {
		memset(link->l_info, 0, sizeof (*ki));
		return 0;
	}

	if ((ki = calloc(1, sizeof(*ki))) == NULL)
		return -NLE_NOMEM;

	link->l_info = ki;

	return 0;
}

static int knet_parse(struct rtnl_link *link, struct nlattr *data,
		     struct nlattr *xstats)
{
	struct nlattr *tb[IFLA_KNET_MAX+1];
	struct knet_info *ki;
	int err;

	NL_DBG(3, "Parsing KNET link info");

	if ((err = nla_parse_nested(tb, IFLA_KNET_MAX, data, knet_policy)) < 0)
		goto errout;

	if ((err = knet_alloc(link)) < 0)
		goto errout;

	ki = link->l_info;

	if (tb[IFLA_KNET_UNIT]) {
		ki->unit = nla_get_u32(tb[IFLA_KNET_UNIT]);
		ki->mask |= KNET_HAS_UNIT;
	}

	err = 0;

errout:
	return err;
}

static void knet_free(struct rtnl_link *link)
{
	free(link->l_info);
	link->l_info = NULL;
}

static int knet_clone(struct rtnl_link *dst, struct rtnl_link *src)
{
	struct knet_info *vdst, *vsrc = src->l_info;
	int err;

	if ((err = rtnl_link_set_type(dst, "bcm_knet")) < 0)
		return err;
	vdst = dst->l_info;

	BUG_ON(!vdst || !vsrc);

	memcpy(vdst, vsrc, sizeof(struct knet_info));

	return 0;
}

static int knet_put_attrs(struct nl_msg *msg, struct rtnl_link *link)
{
	struct knet_info *kn = link->l_info;
	struct nlattr *data;

	if (!(data = nla_nest_start(msg, IFLA_INFO_DATA)))
		return -NLE_NOMEM;

	if (kn->mask & KNET_HAS_UNIT) {
		NLA_PUT_U32(msg, IFLA_KNET_UNIT, kn->unit);
	}

	nla_nest_end(msg, data);

nla_put_failure:

	return 0;
}

static void knet_dump(struct rtnl_link *link, struct nl_dump_params *p)
{
	struct knet_info *kn = link->l_info;

	if (kn->mask & KNET_HAS_UNIT) {
		nl_dump(p, "t %u", kn->unit);
	}
}

struct rtnl_link *rtnl_link_bcm_knet_alloc(void)
{
	struct rtnl_link *link;
	int err;

	if (!(link = rtnl_link_alloc()))
		return NULL;

	if ((err = rtnl_link_set_type(link, "bcm_knet")) < 0) {
		rtnl_link_put(link);
		return NULL;
	}

	return link;
}

/**
 * Get KNET unit
 * @arg link           Link object
 * @arg id             Pointer to store unit identifier
 *
 * @return 0 on success or a negative error code
 */
int rtnl_link_bcm_knet_get_unit(struct rtnl_link *link, uint32_t *unit)
{
	struct knet_info *ki = link->l_info;

	IS_KNET_LINK_ASSERT(link);
	if(!unit)
		return -NLE_INVAL;

	if (ki->mask & KNET_HAS_UNIT)
		*unit = ki->unit;
	else
		return -NLE_AGAIN;

	return 0;
}

/**
 * Set KNET unit
 * @arg link           Link object
 * @arg unit           Unit associated with KNET link
 *
 * @return 0 on success or a negative error code
 */
int rtnl_link_bcm_knet_set_unit(struct rtnl_link *link, uint32_t unit)
{
	struct knet_info *ki = link->l_info;

	IS_KNET_LINK_ASSERT(link);
	ki->unit = unit;
	ki->mask |= KNET_HAS_UNIT;

	return 0;
}

static struct rtnl_link_info_ops knet_info_ops = {
	.io_name                = "bcm_knet",
	.io_alloc               = knet_alloc,
	.io_parse               = knet_parse,
	.io_dump = {
		[NL_DUMP_LINE]     = knet_dump,
		[NL_DUMP_DETAILS]  = knet_dump,
	},
	.io_clone               = knet_clone,
	.io_put_attrs           = knet_put_attrs,
	.io_free                = knet_free,
};


/**
 * @name KNET Object
 * @{
 */

/**
 * Check if link is a KNET link
 * @arg link           Link object
 *
 * @return True if link is a KNET link, otherwise false is returned.
 */
int rtnl_link_is_bcm_knet(struct rtnl_link *link)
{
	return link->l_info_ops && !strcmp(link->l_info_ops->io_name, "bcm_knet");
}

/** @} */

static void __init knet_init(void)
{
	rtnl_link_register_info(&knet_info_ops);
}

static void __exit knet_exit(void)
{
	rtnl_link_unregister_info(&knet_info_ops);
}

/** @} */
