/* -*- mode: c; c-file-style: "openbsd" -*- */
/*
 * Copyright (c) 2009 Vincent Bernat <bernat@luffy.cx>
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "lldpd.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>

#include <unistd.h>
#include <ctype.h>

#ifdef ENABLE_LLDPMED
	/* Fill in inventory stuff:
	    - hardware version: /sys/class/dmi/id/product_version
	    - firmware version: /sys/class/dmi/id/bios_version
	    - software version: `uname -r`
	    - serial number: /sys/class/dmi/id/product_serial
	    - manufacturer: /sys/class/dmi/id/sys_vendor
	    - model: /sys/class/dmi/id/product_name
	    - asset: /sys/class/dmi/id/chassis_asset_tag
	*/

struct dmi {
	char bios_version[40];
	char sys_manufacturer[40];
	char sys_model[40];
	char sys_serialnum[40];
	char sys_version[40];
	char sys_asset[40];
} dmiinfo;

char *
get_value(char *key, char *buf)
{
	char *p;
	int i;
	char *unknown_str = "UNKNOWN";

	if (!key || !buf)
		return NULL;

	if (!strstr(buf, key))
		return NULL;

	p = strstr(buf, ":");
	if (!p)
		return NULL;
	p++;

	while (p && *p != '\0' && isspace(*p))
			p++;

	/* if non-ascii is found - replace with 'unknown'*/
	i = 0;
	while (p+i && *(p+i) != '\0') {
		if (!isascii(*(p+i)))
			return strdup(unknown_str);
		i++;
	}
	if (p && *p != '\0' && *p != '\n' && strlen(p) > 0)
		return p;

	return NULL;
}

int
dmi_get_info()
{
	char *const command[] = { "/usr/sbin/dmidecode", NULL };
	int pid, status, devnull, count;
	int pipefd[2];
	int buflen = 1024 * 4;
	char *line = NULL, *buffer = NULL, *tmpbuf = NULL, *val;
	unsigned int infomask = 0;
#define SYSTEM_INFO 0x1
#define BIOS_INFO 0x2

	log_debug("dmi", "get dmi info");

	memset(&dmiinfo, 0, sizeof(dmiinfo));
	if (pipe(pipefd)) {
		log_warn("dmi_get_info", "unable to get a pair of pipes");
		return -1;
	}

	if ((pid = fork()) < 0) {
		log_warn("dmi_get_info", "unable to fork");
		return -1;
	}
	switch (pid) {
	case 0:
		/* Child, exec dmidecode */
		close(pipefd[0]);
		if ((devnull = open("/dev/null", O_RDWR, 0)) != -1) {
			dup2(devnull, STDIN_FILENO);
			dup2(devnull, STDERR_FILENO);
			dup2(pipefd[1], STDOUT_FILENO);
			if (devnull > 2) close(devnull);
			if (pipefd[1] > 2) close(pipefd[1]);
			execvp("/usr/sbin/dmidecode", command);
		}
		exit(127);
		break;
	default:
		/* Father, read the output from the children */
		close(pipefd[1]);
		count = 0;
		buffer = calloc(1, buflen);
		if (!buffer)
			return -1;
		memset(buffer, 0, buflen);
		do {
			status = read(pipefd[0], buffer + count, buflen - count - 1);
			if ((status == -1) && (errno == EINTR)) continue;
			if (status > 0)
				count += status;
		} while (status > 0 && count < buflen - 1);

		if (status < 0) {
			log_info("dmi_info", "unable to read from dmidecode");
			close(pipefd[0]);
			free(buffer);
			waitpid(pid, &status, 0);
			return -1;
		}
		close(pipefd[0]);
		if (count >= buflen) {
			log_info("dmi", "output of dmidecode is too large");
			waitpid(pid, &status, 0);
			free(buffer);
			return -1;
		}
		status = -1;
		if (waitpid(pid, &status, 0) != pid) {
			log_info("dmi", "waitpid failed");
			free(buffer);
			return -1;
		}
		if (!WIFEXITED(status) || (WEXITSTATUS(status) != 0)) {
			log_info("dmi", "dmidecode information not available");
			free(buffer);
			return -1;
		}
		if (!count) {
			log_info("dmi", "dmidecode returned an empty string");
			free(buffer);
			return -1;
		}

		/* Parse the buffer */
		tmpbuf = strdup(buffer);
		if (!tmpbuf) {
			log_info("dmi", "strdup failed (%s)", strerror(errno));
			free(buffer);
			return -1;
		}
		infomask = 0;
		for(line = strtok(tmpbuf, "\n"); line != NULL;
			line = strtok(NULL, "\n")) {
			if (strstr(line, "BIOS Information") != NULL) {
				infomask = BIOS_INFO;
				continue;
			} else if (strstr(line, "System Information") != NULL) {
				infomask = SYSTEM_INFO;
				continue;
			} else if (strstr(line, "DMI type") != NULL) {
				infomask = 0;
				continue;
			}

			if (infomask & BIOS_INFO) {
				val = get_value("Version:", line);
				if (val) {
					strncpy(dmiinfo.bios_version, val,
						sizeof(dmiinfo.bios_version) - 1);
					continue;
				}
			} else if (infomask & SYSTEM_INFO) {
				val = get_value("Manufacturer:", line);
				if (val) {
					strncpy(dmiinfo.sys_manufacturer, val,
						sizeof(dmiinfo.sys_manufacturer) - 1);
					continue;
				}

				val = get_value("Product Name:", line);
				if (val) {
					strncpy(dmiinfo.sys_model, val,
						sizeof(dmiinfo.sys_model) - 1);
					continue;
				}

				val = get_value("Serial Number:", line);
				if (val) {
					strncpy(dmiinfo.sys_serialnum, val,
						sizeof(dmiinfo.sys_serialnum) - 1);
					continue;
				}
			}
		}
		free(buffer);
		free(tmpbuf);

		log_info("dmi", "dmiinfo = {bios_version:%s,"
					   "manufacturer:%s,"
					   "model:%s,"
					   "sysserialnum:%s,"
					   "sysversion:%s,"
					   "sysasset:%s}\n",
					   dmiinfo.bios_version,
					   dmiinfo.sys_manufacturer,
					   dmiinfo.sys_model,
					   dmiinfo.sys_serialnum,
					   dmiinfo.sys_version,
					   dmiinfo.sys_asset);
		return 0;
	}
	/* Should not be here */
	return NULL;
}

static char*
dmi_get(char *file)
{
	int dmi, s;
	char buffer[100] = {};

	log_debug("localchassis", "DMI request for file %s", file);
	if ((dmi = priv_open(file)) < 0) {
		log_debug("localchassis", "cannot get DMI file %s", file);
		return NULL;
	}
	if ((s = read(dmi, buffer, sizeof(buffer))) == -1) {
		log_debug("localchassis", "cannot read DMI file %s", file);
		close(dmi);
		return NULL;
	}
	close(dmi);
	buffer[sizeof(buffer) - 1] = '\0';
	if ((s > 0) && (buffer[s-1] == '\n'))
		buffer[s-1] = '\0';
	if (strlen(buffer))
		return strdup(buffer);
	return NULL;
}

char*
dmi_hw()
{
	return (strlen(dmiinfo.sys_version) ?
			strdup(dmiinfo.sys_version) : NULL);
}

char*
dmi_fw()
{
	return (strlen(dmiinfo.bios_version) ?
			strdup(dmiinfo.bios_version) : NULL);
}

char*
dmi_sn()
{
	return (strlen(dmiinfo.sys_serialnum) ?
			strdup(dmiinfo.sys_serialnum) : NULL);
}

char*
dmi_manuf()
{
	return (strlen(dmiinfo.sys_manufacturer) ?
			strdup(dmiinfo.sys_manufacturer) : NULL);
}

char*
dmi_model()
{
	return (strlen(dmiinfo.sys_model) ?
			strdup(dmiinfo.sys_model) : NULL);
}

char*
dmi_asset()
{
	return (strlen(dmiinfo.sys_asset) ?
			strdup(dmiinfo.sys_asset) : NULL);
}
#endif
