/*
 * "SETCLASS" target extension for Xtables
 * Copyright 2013 Cumulus Networks, LLC.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/ip.h>
#include <net/checksum.h>

#include <linux/netfilter_ipv4.h>
#include <linux/netfilter_ipv6.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter/xt_SETCLASS.h>
#include <linux/netfilter_arp.h>

MODULE_AUTHOR("Wilson Kok <wkok@cumulusnetworks.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Xtables: Ingress classification");
MODULE_ALIAS("ipt_SETCLASS");
MODULE_ALIAS("ip6t_SETCLASS");
MODULE_ALIAS("arpt_SETCLASS");

static unsigned int
setclass_tg(struct sk_buff *skb, const struct xt_action_param *par)
{
	const struct xt_setclass_target_info *clinfo = par->targinfo;

	skb->priority = clinfo->priority;

	return NF_ACCEPT;
}

static struct xt_target setclass_tg_reg[] __read_mostly = {
	{
		.name       = "SETCLASS",
		.revision   = 0,
		.family     = NFPROTO_UNSPEC,
		.hooks      = (1 << NF_INET_LOCAL_IN) | (1 << NF_INET_FORWARD) |
			      (1 << NF_INET_PRE_ROUTING),
		.target     = setclass_tg,
		.targetsize = sizeof(struct xt_setclass_target_info),
		.me         = THIS_MODULE,
	},
	{
		.name       = "SETCLASS",
		.revision   = 0,
		.family     = NFPROTO_ARP,
		.hooks      = (1 << NF_ARP_IN) | (1 << NF_ARP_FORWARD),
		.target     = setclass_tg,
		.targetsize = sizeof(struct xt_setclass_target_info),
		.me         = THIS_MODULE,
	},
};

static int __init setclass_tg_init(void)
{
	return xt_register_targets(setclass_tg_reg,
				   ARRAY_SIZE(setclass_tg_reg));
}

static void __exit setclass_tg_exit(void)
{
	xt_unregister_targets(setclass_tg_reg, ARRAY_SIZE(setclass_tg_reg));
}

module_init(setclass_tg_init);
module_exit(setclass_tg_exit);
