/*
 *	"ERSPAN" target extension for Xtables
 *	Copyright 2013 Cumulus Networks, LLC.  All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	version 2 or later, as published by the Free Software Foundation.
 */
#include <linux/ip.h>
#include <linux/module.h>
#include <linux/percpu.h>
#include <linux/route.h>
#include <linux/skbuff.h>
#include <linux/notifier.h>
#include <net/checksum.h>
#include <net/icmp.h>
#include <net/ip.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/route.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter/xt_ERSPAN.h>

#if defined(CONFIG_NF_CONNTRACK) || defined(CONFIG_NF_CONNTRACK_MODULE)
#	define WITH_CONNTRACK 1
#	include <net/netfilter/nf_conntrack.h>
#endif
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
#	define WITH_IPV6 1
#endif

struct xt_erspan_priv {
	struct notifier_block	notifier;
	struct xt_erspan_tginfo	*tginfo;
	int			oif;
};

static const union nf_inet_addr erspan_zero_address;

static unsigned int
erspan_tg4(struct sk_buff *skb, const struct xt_action_param *par)
{
	return XT_CONTINUE;
}

#ifdef WITH_IPV6
static unsigned int
erspan_tg6(struct sk_buff *skb, const struct xt_action_param *par)
{
	return XT_CONTINUE;
}
#endif /* WITH_IPV6 */

static int erspan_tg_check(const struct xt_tgchk_param *par)
{
	struct xt_erspan_tginfo *info = par->targinfo;

	/* 0.0.0.0 and :: not allowed */
	if (memcmp(&info->src_ip, &erspan_zero_address,
		   sizeof(erspan_zero_address)) == 0)
		return -EINVAL;

	if (memcmp(&info->dst_ip, &erspan_zero_address,
		   sizeof(erspan_zero_address)) == 0)
		return -EINVAL;

	return 0;
}

static void erspan_tg_destroy(const struct xt_tgdtor_param *par)
{
	struct xt_erspan_tginfo *info = par->targinfo;

	if (info->priv) {
		unregister_netdevice_notifier(&info->priv->notifier);
		kfree(info->priv);
	}
}

static struct xt_target erspan_tg_reg[] __read_mostly = {
	{
		.name       = "ERSPAN",
		.revision   = 1,
		.family     = NFPROTO_IPV4,
		.target     = erspan_tg4,
		.targetsize = sizeof(struct xt_erspan_tginfo),
		.checkentry = erspan_tg_check,
		.destroy    = erspan_tg_destroy,
		.me         = THIS_MODULE,
	},
#ifdef WITH_IPV6
	{
		.name       = "ERSPAN",
		.revision   = 1,
		.family     = NFPROTO_IPV6,
		.target     = erspan_tg6,
		.targetsize = sizeof(struct xt_erspan_tginfo),
		.checkentry = erspan_tg_check,
		.destroy    = erspan_tg_destroy,
		.me         = THIS_MODULE,
	},
#endif
};

static int __init erspan_tg_init(void)
{
	return xt_register_targets(erspan_tg_reg, ARRAY_SIZE(erspan_tg_reg));
}

static void __exit erspan_tg_exit(void)
{
	xt_unregister_targets(erspan_tg_reg, ARRAY_SIZE(erspan_tg_reg));
}

module_init(erspan_tg_init);
module_exit(erspan_tg_exit);
MODULE_AUTHOR("Wilson Kok <wkok@cumulusnetworks.com>");
MODULE_DESCRIPTION("Xtables: Encapsulated Remote Switch Port Analyzer");
MODULE_LICENSE("GPL");
MODULE_ALIAS("ipt_ERSPAN");
MODULE_ALIAS("ip6t_ERSPAN");
