/*
 * $Copyright Open Broadcom Corporation$
 *
 */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/suspend.h>
#include <linux/version.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/clk.h>
#include <linux/usb/phy.h>

#define ICFG_USB_CTRL_ADDR(base)					(base + 0x00)
#define  ICFG_USB_CTRL__DRD_FORCE_HOST_MODE			6
#define  ICFG_USB_CTRL__DRD_FORCE_DEVICE_MODE		5
#define  ICFG_USB_CTRL__XHC_CSR_RESET				4
#define  ICFG_USB_CTRL__BDC_CSR_RESET				3
#define  ICFG_USB_CTRL__DRD_SOFT_RESET				2
#define  ICFG_USB_CTRL__XHC_SOFT_RESET				1
#define  ICFG_USB_CTRL__BDC_SOFT_RESET				0

#define IPROC_WRAP_USBPHY_CTRL_0_ADDR(base)			(base + 0x00)
#define  IPROC_WRAP_USBPHY_CTRL_0__PHY_ISO			18
#define  IPROC_WRAP_USBPHY_CTRL_0__PLL_CTRL_45		17
#define  IPROC_WRAP_USBPHY_CTRL_0__PLL_SUSPEND_EN	16
#define  IPROC_WRAP_USBPHY_CTRL_0__PLL_RESETB		15
#define  IPROC_WRAP_USBPHY_CTRL_0__RESETB			14
#define IPROC_WRAP_USBPHY_CTRL_2_ADDR(base)			(base + 0x08)
#define  IPROC_WRAP_USBPHY_CTRL_2__AFE_LDO_PWRDWNB	2
#define  IPROC_WRAP_USBPHY_CTRL_2__AFE_PLL_PWRDWNB	1
#define  IPROC_WRAP_USBPHY_CTRL_2__AFE_BG_PWRDWNB	0
#define IPROC_WRAP_MISC_STATUS_0_ADDR(base)			(base + 0x1c)
#define  IPROC_WRAP_MISC_STATUS_0__USBPHY_PLL_LOCK	0
#define IPROC_WRAP_MISC_STATUS_1_ADDR(base)			(base + 0x20)

struct iproc_usb_priv {
	struct usb_phy phy;
	struct device *dev;
	struct device_node *dn;
	void __iomem *wrap_base;
    void __iomem *icfg_usb_base;
};

extern void __iomem *get_iproc_wrap_ctrl_base(void);

/***************************************************************************
****************************************************************************
***************************************************************************/
static int iproc_usb_phy_init(struct usb_phy *phy)
{
    struct iproc_usb_priv *iproc_usb_data = container_of(phy, struct iproc_usb_priv, phy);
    void __iomem *wrap_base = iproc_usb_data->wrap_base;
    struct device *dev = iproc_usb_data->dev;
    uint val;
    ulong mask, count = 0;

    if (!wrap_base) {
        return -EINVAL;
    }

    /* FIXME. PHY initial sequence, need to get the sequence from DE */
	val = readl_relaxed(IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));
	val |= (1 << IPROC_WRAP_USBPHY_CTRL_0__PHY_ISO);
	val |= (1 << IPROC_WRAP_USBPHY_CTRL_0__PLL_SUSPEND_EN);
	writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));

	val &= ~(1 << IPROC_WRAP_USBPHY_CTRL_0__PLL_RESETB);
	val &= ~(1 << IPROC_WRAP_USBPHY_CTRL_0__RESETB);
    writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));

	val = readl_relaxed(IPROC_WRAP_USBPHY_CTRL_2_ADDR(wrap_base));
	val &= ~(1 << IPROC_WRAP_USBPHY_CTRL_2__AFE_BG_PWRDWNB);
	val &= ~(1 << IPROC_WRAP_USBPHY_CTRL_2__AFE_LDO_PWRDWNB);
    writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_2_ADDR(wrap_base));

	udelay(10);

	val |= (1 << IPROC_WRAP_USBPHY_CTRL_2__AFE_BG_PWRDWNB);
    writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_2_ADDR(wrap_base));

	udelay(150);

	val |= (1 << IPROC_WRAP_USBPHY_CTRL_2__AFE_LDO_PWRDWNB);
	writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_2_ADDR(wrap_base));

	udelay(160);

	val = readl_relaxed(IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));
	val &= ~(1 << IPROC_WRAP_USBPHY_CTRL_0__PHY_ISO);
    writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));

	udelay(20);

	val |= (1 << IPROC_WRAP_USBPHY_CTRL_0__PLL_RESETB);
    writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));

    mdelay(20);


    /* check pll_lock */
    mask = (1 << IPROC_WRAP_MISC_STATUS_0__USBPHY_PLL_LOCK);
    do {
        val = readl_relaxed(IPROC_WRAP_MISC_STATUS_0_ADDR(wrap_base));
        if ((val & mask) == mask) {
            break;
        } else {
            udelay(10);
            count ++;
        }
    } while(count <= 10);
    if (count > 10) {
        dev_err(dev, "%s : PLL not lock! IPROC_WRAP_MISC_STATUS_0 = 0x%.8x\n",
                    __FUNCTION__, val);
    }

	val = readl_relaxed(IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));
	val |= (1 << IPROC_WRAP_USBPHY_CTRL_0__RESETB);
    writel_relaxed(val, IPROC_WRAP_USBPHY_CTRL_0_ADDR(wrap_base));
    udelay(2);

    return 0;
}

static int iproc_usb_reset(struct iproc_usb_priv *iproc_usb_data)
{
    void __iomem *icfg_usb_base = iproc_usb_data->icfg_usb_base;
    uint val;

    if (!icfg_usb_base) {
        return -EINVAL;
    }

    /* Put DRD into reset state */
    val = readl_relaxed(ICFG_USB_CTRL_ADDR(icfg_usb_base));
    val |= (1 << ICFG_USB_CTRL__DRD_SOFT_RESET);
    writel_relaxed(val, ICFG_USB_CTRL_ADDR(icfg_usb_base));

    /* Put BDC and XHC into reset state */
    val = readl_relaxed(ICFG_USB_CTRL_ADDR(icfg_usb_base));
    val |= (1 << ICFG_USB_CTRL__BDC_SOFT_RESET);
    val |= (1 << ICFG_USB_CTRL__XHC_SOFT_RESET);
    writel_relaxed(val, ICFG_USB_CTRL_ADDR(icfg_usb_base));

    mdelay(10);

    /* Get the BDC and XHC out of reset  */
    val = readl_relaxed(ICFG_USB_CTRL_ADDR(icfg_usb_base));
    val &= ~(1 << ICFG_USB_CTRL__BDC_SOFT_RESET);
    val &= ~(1 << ICFG_USB_CTRL__XHC_SOFT_RESET);
    writel_relaxed(val, ICFG_USB_CTRL_ADDR(icfg_usb_base));

    /* Get the DRD out of reset  */
    val = readl_relaxed(ICFG_USB_CTRL_ADDR(icfg_usb_base));
    val &= ~(1 << ICFG_USB_CTRL__DRD_SOFT_RESET);
    writel_relaxed(val, ICFG_USB_CTRL_ADDR(icfg_usb_base));

    return 0;
}

static int xgs_iproc_drd_probe(struct platform_device *pdev)
{
    struct device *dev = &pdev->dev;
    struct device_node *dn = pdev->dev.of_node;
    struct iproc_usb_priv *iproc_usb_data;
    int ret;

    if (!of_device_is_available(dn)) {
        return -ENODEV;
    }

    iproc_usb_data = devm_kzalloc(dev, sizeof(*iproc_usb_data), GFP_KERNEL);
    if (!iproc_usb_data) {
        dev_err(dev, "devm_kzalloc() failed\n" );
        return -ENOMEM;
    }
    memset(iproc_usb_data, 0, sizeof(*iproc_usb_data));
    platform_set_drvdata(pdev, iproc_usb_data);

    iproc_usb_data->dev = dev;

    iproc_usb_data->wrap_base = get_iproc_wrap_ctrl_base();
    if (!iproc_usb_data->wrap_base) {
        dev_err(&pdev->dev, "can't iomap usb phy base address\n");
        ret = -ENOMEM;
        goto err;
    }

    iproc_usb_data->icfg_usb_base = (void *)of_iomap(dn, 1);
    if (!iproc_usb_data->icfg_usb_base) {
        dev_err(&pdev->dev, "can't iomap icfg usb base address\n");
        ret = -ENOMEM;
        goto err;
    }

    iproc_usb_data->phy.dev = dev;
    iproc_usb_data->phy.type = USB_PHY_TYPE_USB2;
    iproc_usb_data->phy.init = iproc_usb_phy_init;

    iproc_usb_reset(iproc_usb_data);
    iproc_usb_phy_init(&iproc_usb_data->phy);

    ret = usb_add_phy_dev(&iproc_usb_data->phy);
    if (ret) {
        dev_err(&pdev->dev, "failed to add the phy device\n");
        goto err;
    }

    return 0;

err:
    if (iproc_usb_data->icfg_usb_base) {
        iounmap(iproc_usb_data->icfg_usb_base);
    }
    if (iproc_usb_data) {
        iounmap(iproc_usb_data);
    }

    return ret;
}

static int xgs_iproc_drd_remove(struct platform_device *pdev)
{
    struct iproc_usb_priv *iproc_usb_data = platform_get_drvdata(pdev);

    usb_remove_phy(&iproc_usb_data->phy);

    platform_set_drvdata(pdev, NULL);
    if (iproc_usb_data->icfg_usb_base) {
        iounmap(iproc_usb_data->icfg_usb_base);
    }

    if (iproc_usb_data) {
        iounmap(iproc_usb_data);
    }

    return 0;
}

static const struct of_device_id xgs_iproc_drd_dt_ids[] = {
	{ .compatible = "brcm,usb-phy,hx5", },
	{ }
};
MODULE_DEVICE_TABLE(of, xgs_iproc_drd_dt_ids);

static struct platform_driver xgs_iproc_drd_driver =
{
	.driver = {
		.name = "usb-phy",
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(xgs_iproc_drd_dt_ids),
	},
	.probe = xgs_iproc_drd_probe,
	.remove = xgs_iproc_drd_remove,
};

module_platform_driver(xgs_iproc_drd_driver);

MODULE_AUTHOR("Broadcom");
MODULE_DESCRIPTION("Broadcom USB DRD controller driver");
MODULE_LICENSE("GPL");
