/*
 * knet-netlink.h
 *
 * Copyright (c) 2015 Cumulus Networks. All rights reserved.
 *
 * Andy Gospodarek <gospo@cumulusnetworks.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ip.h>
#include <linux/init.h>
#include <linux/moduleparam.h>
#include <linux/netfilter.h>
#include <linux/rtnetlink.h>
#include <net/rtnetlink.h>

/* KNET section */
enum {
	IFLA_KNET_UNSPEC,
	IFLA_KNET_UNIT,
	__IFLA_KNET_MAX
};

#define IFLA_KNET_MAX (__IFLA_KNET_MAX - 1)

static void knet_setup(struct net_device *dev)
{
	ether_setup(dev);

	dev->netdev_ops = &bkn_netdev_ops;
	dev->ethtool_ops = &bkn_ethtool_ops;
	dev->switchdev_ops = &bkn_switchdev_ops;
	dev->vlan_features = dev->features;
	dev->features |= NETIF_F_HW_VLAN_CTAG_TX | NETIF_F_HW_VLAN_STAG_TX |
			 NETIF_F_HW_TC;
	dev->hw_features = dev->features;
	dev->priv_flags |= IFF_LIVE_ADDR_CHANGE;

	eth_hw_addr_random(dev);
}

static int knet_validate(struct nlattr *tb[], struct nlattr *data[],
			 struct netlink_ext_ack *extack)
{
	if (tb[IFLA_ADDRESS]) {
		if (nla_len(tb[IFLA_ADDRESS]) != ETH_ALEN)
			return -EINVAL;
		if (!is_valid_ether_addr(nla_data(tb[IFLA_ADDRESS])))
			return -EADDRNOTAVAIL;
	}

	/* if unit is not set, this is an invalid config */
	if (!data || !data[IFLA_KNET_UNIT])
		return -EINVAL;

	return 0;
}

static void knet_dellink(struct net_device *dev, struct list_head *head)
{
	bkn_priv_t *priv = netdev_priv(dev);
	bkn_switch_info_t *sinfo;
	unsigned long flags;

	if (!priv) {
		DBG_WARN(("%s failed to find netdev_priv(%s)\n",
			  __func__, dev->name));
		return;
	}

	sinfo = priv->sinfo;
	spin_lock_irqsave(&sinfo->lock, flags);
	list_del(&priv->list);
	kfree(priv->sample_ingress);
	kfree(priv->sample_egress);

	while (!list_empty(&priv->vlan_xlate_list)) {
		bkn_vlan_xlate_t *vlan_xlate = list_entry(priv->vlan_xlate_list.next,
							  bkn_vlan_xlate_t, list);
		list_del(&vlan_xlate->list);
		DBG_VERB(("Cleared vlan_xlate int vlan %d ext vlan %d.\n",
			  vlan_xlate->int_vlan, vlan_xlate->ext_vlan));
		kfree(vlan_xlate);
	}
	spin_unlock_irqrestore(&sinfo->lock, flags);

	DBG_VERB(("Removing virtual Ethernet device %s.\n", dev->name));
	port_uninit_ethtool_stats(dev);
	unregister_netdevice_queue(dev, head);
	return;
}

static int knet_newlink(struct net *src_net, struct net_device *dev,
			struct nlattr *tb[], struct nlattr *data[],
			struct netlink_ext_ack *extack)
{
	int found, id, unit;
	bkn_priv_t *priv, *lpriv = NULL;
	bkn_switch_info_t *sinfo;
	struct list_head *list;
	unsigned long flags;

	unit = nla_get_u32(data[IFLA_KNET_UNIT]);
	sinfo = bkn_sinfo_from_unit(unit);
	if (!sinfo) {
		gprintk("%s failed to find proper unit for %s\n",
			__func__, dev->name);
		goto create_fail;
	}

	priv = netdev_priv(dev);
	if (!priv) {
		gprintk("%s failed to find netdev_priv(%s)\n",
			__func__, dev->name);
		goto create_fail;
	}

	priv->dev = dev;
	priv->sinfo = sinfo;
	priv->type = KCOM_NETIF_T_PORT;
	priv->flags = 0;
	priv->sample_ingress = kzalloc(sizeof(struct bkn_sample), GFP_KERNEL);
	priv->sample_egress = kzalloc(sizeof(struct bkn_sample), GFP_KERNEL);

	dev->min_mtu = 68;
	dev->max_mtu = MAX_DEVICE_MTU;

	if (register_netdevice(dev))
		goto create_fail;

	port_init_ethtool_stats(dev);

	spin_lock_irqsave(&sinfo->lock, flags);
	INIT_LIST_HEAD(&priv->vlan_xlate_list);
	found = 0;
	id = 1;
	list_for_each(list, &sinfo->ndev_list) {
		lpriv = (bkn_priv_t *)list;
		if (id < lpriv->id) {
			found = 1;
			break;
		}
		id = lpriv->id + 1;
	}
	/* id possibly not getting set properly */
	priv->id = id;
	if (found) {
		/* Replace previously removed interface */
		list_add_tail(&priv->list, &lpriv->list);
	} else {
		/* No holes - add to end of list */
		list_add_tail(&priv->list, &sinfo->ndev_list);
	}

	netif_carrier_off(dev);
	spin_unlock_irqrestore(&sinfo->lock, flags);

	DBG_VERB(("knet_netlink: assigned ID %d to Ethernet device %s\n",
		  priv->id, dev->name));
	return 0;

create_fail:
	return -EINVAL;
}

static size_t knet_nl_getsize(const struct net_device *dev)
{
	return nla_total_size(sizeof(__u32));  /* IFLA_KNET_UNIT */
}

static int knet_fillinfo(struct sk_buff *skb,
			 const struct net_device *dev)
{
	bkn_priv_t *priv = netdev_priv(dev);

	if (priv && priv->sinfo)
		return nla_put_u32(skb, IFLA_KNET_UNIT, priv->sinfo->dev_no);
	return 0;
}

static const struct nla_policy knet_nl_policy[IFLA_KNET_MAX + 1] = {
	[IFLA_KNET_UNIT] = { .type = NLA_U32 },
};

static struct rtnl_link_ops knet_link_ops __read_mostly = {
	.kind		= "bcm_knet",
	.priv_size	= sizeof(bkn_priv_t),

	.get_size	= knet_nl_getsize,
	.policy		= knet_nl_policy,
	.validate	= knet_validate,
	.fill_info	= knet_fillinfo,

	.newlink	= knet_newlink,
	.dellink	= knet_dellink,
	.setup		= knet_setup,
	.maxtype	= IFLA_KNET_MAX,
};

static int knet_init_netlink(void)
{
	int rc;

	rc = rtnl_link_register(&knet_link_ops);
	if (rc < 0) {
		gprintk("%s Failed to regsiter knet_link_ops\n", __func__);
		return rc;
	}
	return 0;
}

static void knet_uninit_netlink(void)
{
	rtnl_link_unregister(&knet_link_ops);
}
