/*
 *  knet addons - code that makes knet more useful on a general purpose
 *  Linux distribution where each switchport is a kernel netdevice
 *  Copyright 2015-2016, Cumulus Networks, Inc
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms and conditions of the GNU General Public
 *  License, version 2, as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 */

#ifndef __LINUX_BCM_KNET_ADDONS_H__
#define __LINUX_BCM_KNET_ADDONS_H__

#include <linux/if_vlan.h>
#include <linux/port.h>
#include <linux/kernel.h>
#include <gmodule.h>
#include <bcm-knet.h>
#include <net/pkt_cls.h>
#include <net/tc_act/tc_sample.h>
#include <net/switchdev.h>

#if defined(__LITTLE_ENDIAN_BITFIELD)
#define LE_HOST
#endif

#define KNET_DRIVER_VERSION	"6.5.14-cl"
#define KNET_CPU_QUEUE(__q__)	(__q__ + 3)
#define DESC_PKT_START(d, s)	(d->skb_data + s)

typedef enum {
	BKN_TAG_STATUS_UNTAGGED = 0,
	BKN_TAG_STATUS_SIT      = 1, /* single inner tagged */
	BKN_TAG_STATUS_SOT      = 2, /* single outer tagged */
	BKN_TAG_STATUS_DT       = 3  /* double tagged */
} bkn_tag_status_e;

typedef struct bkn_vlan_xlate_s {
	struct list_head list;
	int int_vlan;
	int int_ivid;
	int ext_vlan;
	int ext_ivid;
} bkn_vlan_xlate_t;

typedef struct bkn_vlan_pair_s {
	u16 ovid;
	u16 ivid;
} bkn_vlan_pair_t;

typedef struct bkn_ilpbk__entry_s {
	struct hlist_node hlist;

	int loopback_id;
	int rx_ifindex;
	u32 num_ports;
	u32 loopback_ports[KCOM_ILPBK_MAX_PORTS];
} bkn_ilpbk_entry_t;

/* Taken from type28.h of bcm-sdk */
#if defined(__LITTLE_ENDIAN_BITFIELD)
typedef struct {
	uint32  :11,
		ingress_untagged:1,	/* Pkt came in untagged */
		outer_vid:12,		/* VID */
		outer_cfi:1,		/* CFI */
		outer_pri:3,		/* Priority */
		dscp_lo:4;		/* New DSCP */
} dcb28_word9_t;
#else
/* NOTE: DNX is not supported on BE hosts. */
typedef struct {
	uint32  dscp_lo:4,              /* New DSCP */
		outer_pri:3,            /* Priority (D)*/
		outer_cfi:1,            /* CFI (D)*/
		outer_vid:12,           /* VID (D)*/
		ingress_untagged:1,     /* Pkt came in untagged (D)*/
		:11;
} dcb28_word9_t;
#endif

static void bkn_dump_pkt(uint8_t *data, int size, int txrx);

static bkn_switch_info_t *bkn_sinfo_from_unit(int unit);

static bkn_priv_t *bkn_netif_lookup_port(bkn_switch_info_t *sinfo, int port)
{
	struct list_head *list;
	bkn_priv_t *priv;
	int found;

	/* Slow path - should normally not get here */
	found = 0;
	priv = NULL;
	list_for_each(list, &sinfo->ndev_list) {
		priv = (bkn_priv_t *)list;
		if (priv->port == port) {
			found = 1;
			break;
		}
	}
	if (found && priv)
		return priv;
	return NULL;
}

static bkn_vlan_pair_t bkn_vlan_tci_translate(bkn_priv_t *priv, u16 vlan_tci,
					      u16 inner_vlan_tci)
{
	struct list_head *list;
	bkn_vlan_xlate_t *vlan_xlate;
	int int_vlan = vlan_tci & VLAN_VID_MASK;
	int int_ivid = inner_vlan_tci & VLAN_VID_MASK;
	bkn_vlan_pair_t vlan_pair = {int_vlan, int_ivid};

	list_for_each(list, &priv->vlan_xlate_list) {
		vlan_xlate = (bkn_vlan_xlate_t *)list;
		if (vlan_xlate->int_vlan == int_vlan &&
		    vlan_xlate->int_ivid == int_ivid) {
			vlan_pair.ovid = vlan_xlate->ext_vlan;
                        vlan_pair.ivid = vlan_xlate->ext_ivid;
			return vlan_pair;
		}
	}
	return vlan_pair;
}

static bool hw_translated_vlan(bkn_vlan_pair_t int_vlan_pair,
			     u16 ext_vlan, u16 ext_inner_vlan)
{
	return int_vlan_pair.ovid != (ext_vlan & VLAN_VID_MASK) ||
	       int_vlan_pair.ivid != (ext_inner_vlan & VLAN_VID_MASK);
}

static bool bkn_vlan_tag_process(bkn_switch_info_t *sinfo, int chan,
				 uint32_t *dcb, struct sk_buff *skb, int *pktlen)
{
	int inner_ethertype = 0;
	u16 vlan_tci;
	u16 inner_vlan_tci;
	u16 xlate_vlan_tci;
	u16 xlate_inner_vlan_tci;
	u16 outer_ethertype = 0;
	bkn_priv_t *priv = netdev_priv(skb->dev);
	bkn_tag_status_e tag_status = BKN_TAG_STATUS_UNTAGGED;
	bkn_vlan_pair_t vlan_pair;

	switch (sinfo->dcb_type) {
	case 19:
	case 20:
	case 30:
		/* bits 11-12 of word 12 are for vlan itag status */
		/* used by hurricane2 */
		tag_status = (dcb[12] >> 10) & 0x3;
		break;
    case 36:
        /* In TD3 based devices, tag status is obtained from match ID.
        * bits 09-10 of word 12 are for vlan itag status
        * itag -> bit  9 of match id.
        * otag -> bit 10 of match id.
        * dcb36 - used by Trident3
        */
        tag_status = (dcb[13] >> 9) & 0x3;
        break;
    case 38:
        /*
         * In TH3, even though the number of bits for the
         * tag status is 2, we support only 2 states
         * 0 = untagged
         * 1 = single outer-tag
         * and rest of the code assumes single outer-tag if the tag_status
         * is 2. So we need to remap/reassign tag_status as BKN_TAG_STATUS_SOT
         * as 2 if it's single outer-tag.
         */
        tag_status = (dcb[9] >> 13) & 0x3;
        if (tag_status) {
            tag_status = BKN_TAG_STATUS_SOT;
        }
        break;

	case 28: /* Qumran-MX */
	{
		/*
		 * Word-9 in DCB encodes VLAN info of the received packet.
		 */
		dcb28_word9_t *dcb28_word9 = (dcb28_word9_t *)(&dcb[9]);

		/* On DNX VLAN translation is not used to map packets from
		 * external VLAN to internal VLAN. Packets punted to CPU will
		 * retain packet-VLAN. Hence knet does not need to do xlate.
		 * DNX uses Virtual Switch Instance to achieve the same.
		 */
		return dcb28_word9->ingress_untagged;
	}

	default:
		/* lower 2 bits of word 13 are for vlan tag status,
		 * dcb23 - used by helix4
		 * dcb26 - used by trident2
		 * dcb32 - used by tomahawk
		 * dcb33 - used by trident2+
		 */
		tag_status = dcb[13] & 0x3;
		break;
	}

	if (tag_status == BKN_TAG_STATUS_SIT) {
		inner_ethertype = (skb->data[12] << 8) | skb->data[13];
	} else {
		outer_ethertype = (skb->data[12] << 8) | skb->data[13];
		if (tag_status == BKN_TAG_STATUS_DT)
			inner_ethertype = (skb->data[16] << 8) | skb->data[17];
	}

	if (outer_ethertype == ETH_P_8021Q || outer_ethertype == ETH_P_8021AD) {
		/* Save VLAN tag */
		vlan_tci = (skb->data[14] << 8) | skb->data[15];
		if (inner_ethertype == ETH_P_8021Q) {
			inner_vlan_tci = (skb->data[18] << 8) | skb->data[19];
		} else
			inner_vlan_tci = 0;
		vlan_pair = bkn_vlan_tci_translate(priv, vlan_tci, inner_vlan_tci);
		xlate_vlan_tci = vlan_pair.ovid | (vlan_tci & ~VLAN_VID_MASK);
		xlate_inner_vlan_tci = vlan_pair.ivid | (inner_vlan_tci & ~VLAN_VID_MASK);

		DBG_PKT(("vlan_tci=%d.%d xlate_vlan_tci=%d.%d vlan_proto=0x%x ietype=0x%x oetype=0x%x tag_status=%d\n",
			 vlan_tci, inner_vlan_tci, xlate_vlan_tci,
			 xlate_inner_vlan_tci, skb->vlan_proto, inner_ethertype,
			 outer_ethertype, tag_status));

		/* tagged frame, check to see if reserved and strip */
		if (tag_status != BKN_TAG_STATUS_UNTAGGED) {
			/* add the translated tag to the skb */
			skb->data[14] = (xlate_vlan_tci & 0xff00) >> 8;
			skb->data[15] = xlate_vlan_tci & 0xff;
			if (xlate_inner_vlan_tci) {
				if (hw_translated_vlan(vlan_pair, vlan_tci, inner_vlan_tci)) {
					/* insert inner vlan tag */
					skb_push(skb, 4);
					((u32*)skb->data)[0] = ((u32*)skb->data)[1];
					((u32*)skb->data)[1] = ((u32*)skb->data)[2];
					((u32*)skb->data)[2] = ((u32*)skb->data)[3];
					((u32*)skb->data)[3] = ((u32*)skb->data)[4];
					*pktlen += 4;
				}
				skb->data[16] = 0x81;
				skb->data[17] = 0x00;
				skb->data[18] = (xlate_inner_vlan_tci & 0xff00) >> 8;
				skb->data[19] = xlate_inner_vlan_tci & 0xff;
			}
		}
	} else if (!outer_ethertype && inner_ethertype == ETH_P_8021AD &&
			tag_status == BKN_TAG_STATUS_SIT) {
		/* Strip outer VLAN tag, port is 802.1ad access port */
		((u32*)skb->data)[3] = ((u32*)skb->data)[2];
		((u32*)skb->data)[2] = ((u32*)skb->data)[1];
		((u32*)skb->data)[1] = ((u32*)skb->data)[0];
		skb_pull(skb, 4);
		*pktlen -= 4;
	} else {
		DBG_PKT((
		    "%s: Rx'd packet with no vlan, likely ASIC misprogramming, "
		    "ethertype: 0x%x, tag_status: %d, dcb_type: %d, dcb[13]: %u, "
		    "dcb[12]: %u\n", __func__, outer_ethertype, tag_status,
		    sinfo->dcb_type, dcb[13], dcb[12]));
		bkn_dump_pkt(skb->data, *pktlen, 1);
		sinfo->rx[chan].pkts_no_vlan++;
		return false; /* do not strip packet */
	}

	return (tag_status == BKN_TAG_STATUS_UNTAGGED);
}

static void bkn_get_drvinfo(struct net_device *bond_dev,
			    struct ethtool_drvinfo *drvinfo)
{
	strlcpy(drvinfo->driver, "knet", sizeof(drvinfo->driver));
	strlcpy(drvinfo->version, KNET_DRIVER_VERSION,
		sizeof(drvinfo->version));
}

static const struct ethtool_ops bkn_ethtool_ops = {
	.get_link_ksettings = port_get_link_settings,
	.set_link_ksettings = port_set_link_settings,
	.get_link = ethtool_op_get_link,
	.get_strings = port_get_strings,
	.get_ethtool_stats = port_get_ethtool_stats,
	.get_ethtool_stats_clear = port_get_ethtool_stats_clear,
	.get_sset_count = port_get_sset_count,
	.set_phys_id = port_set_phys_id,
	.get_drvinfo = bkn_get_drvinfo,
	.get_module_info = port_get_module_info,
	.get_module_eeprom = port_get_module_eeprom,
	.get_fecparam = port_get_fecparam,
	.set_fecparam = port_set_fecparam,
};

/* Device QNum Control Proc Read Entry */
static int bkn_proc_qnum_show(struct seq_file *m, void *v)
{
	struct list_head *slist, *dlist;
	struct net_device *dev;
	bkn_priv_t *priv;
	bkn_switch_info_t *sinfo;

	seq_puts(m, "Software qnum status:\n");
	list_for_each(slist, &_sinfo_list) {
		sinfo = (bkn_switch_info_t *)slist;
		list_for_each(dlist, &sinfo->ndev_list) {
			priv = (bkn_priv_t *)dlist;
			dev = priv->dev;
			if (dev && dev->name)
				seq_printf(m, "  %-14s %d\n", dev->name,
					   priv->qnum);
		}
	}
	return 0;
}

static int bkn_proc_qnum_open(struct inode *inode, struct file *file)
{
	return single_open(file, bkn_proc_qnum_show, NULL);
}

/* Device QNum Control Proc Write Entry */
static ssize_t bkn_proc_qnum_write(struct file *file, const char *buf,
				   size_t count, loff_t *loff)
{
	struct list_head *slist, *dlist;
	struct net_device *dev;
	bkn_priv_t *priv;
	bkn_switch_info_t *sinfo;
	char qnum_str[IFNAMSIZ + 6];
	char *ptr;
	int len;
	long qnum = 0;

	if (count >= sizeof(qnum_str))
		count = sizeof(qnum_str) - 1;

	if (copy_from_user(qnum_str, buf, count))
		return -EFAULT;

	qnum_str[count] = 0;

	ptr = strchr(qnum_str, '=');
	if (!ptr) {
		ptr = strchr(qnum_str, ':');
		if (!ptr) {
			gprintk("Error: qnum syntax not recognized\n");
			return count;
		}
	}
	*ptr++ = 0;
	len = strlen(qnum_str);

	dev = NULL;
	list_for_each(slist, &_sinfo_list) {
		sinfo = (bkn_switch_info_t *)slist;
		list_for_each(dlist, &sinfo->ndev_list) {
			priv = (bkn_priv_t *)dlist;
			dev = priv->dev;
			if (dev && dev->name)
				if (memcmp(dev->name, qnum_str, len) == 0)
					break;
		}
		if (dev) {
			if (!kstrtol(ptr, 0, &qnum))
				priv->qnum = qnum;
			else
				gprintk("Warning: unknown qnum setting\n");
			return count;
		}
	}

	gprintk("Warning: unknown network interface\n");

	return count;
}

const struct file_operations bkn_proc_qnum_file_ops = {
	.owner = THIS_MODULE,
	.open = bkn_proc_qnum_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.write = bkn_proc_qnum_write,
	.release = single_release,
};

/* Device VLAN Translate Proc Read Entry */
static int bkn_proc_vlan_xlate_show(struct seq_file *m, void *v)
{
	struct list_head *slist, *dlist, *vlist;
	struct net_device *dev;
	bkn_priv_t *priv;
	bkn_switch_info_t *sinfo;

	seq_puts(m, "VLAN Translation Table:\n");
	list_for_each(slist, &_sinfo_list) {
		unsigned long flags;
		sinfo = (bkn_switch_info_t *)slist;
		spin_lock_irqsave(&sinfo->lock, flags);
		list_for_each(dlist, &sinfo->ndev_list) {
			priv = (bkn_priv_t *)dlist;
			dev = priv->dev;
			if (dev && dev->name &&
			    !list_empty(&priv->vlan_xlate_list)) {
				seq_printf(m, "%-16s Internal  External\n",
					   dev->name);
				list_for_each(vlist, &priv->vlan_xlate_list) {
					bkn_vlan_xlate_t *vlan_xlate =
					    (bkn_vlan_xlate_t *)vlist;
					seq_printf(m, "%25d.%d%10d.%d\n",
						   vlan_xlate->int_vlan,
						   vlan_xlate->int_ivid,
						   vlan_xlate->ext_vlan,
						   vlan_xlate->ext_ivid);
				}
			}
		}
		spin_unlock_irqrestore(&sinfo->lock, flags);
	}
	return 0;
}

static int bkn_proc_vlan_xlate_open(struct inode *inode, struct file *file)
{
	return single_open(file, bkn_proc_vlan_xlate_show, NULL);
}

const struct file_operations bkn_proc_vlan_xlate_file_ops = {
	.owner = THIS_MODULE,
	.open = bkn_proc_vlan_xlate_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

/* Internal Loopback Proc Read Entry */
static int bkn_proc_ilpbk_show(struct seq_file *m, void *v)
{
	bkn_ilpbk_entry_t *ilpbk;
	bkn_switch_info_t *sinfo;
	struct list_head *slist;
	int i;

	seq_puts(m, "Internal loopback table:\n");
	list_for_each(slist, &_sinfo_list) {
		unsigned long flags;

		sinfo = (bkn_switch_info_t *)slist;
		spin_lock_irqsave(&sinfo->lock, flags);
		seq_printf(m, "%5s  %8s  %16s\n", "ID", "NUM-PORTS", "RX-DEVICE");
		for (i = 0; i < ILPBK_ENTRY_MAX; i++) {
			hlist_for_each_entry(ilpbk, &sinfo->ilpbk_entry[i],
					     hlist) {
				int j;
				seq_printf(m, "%5u  %3u:%5u         %u\n",
					   ilpbk->loopback_id, ilpbk->num_ports,
					   ilpbk->loopback_ports[0],
					   ilpbk->rx_ifindex);
				if (ilpbk->num_ports <= 1)
					continue;

				for (j = 1; j < ilpbk->num_ports; j++) {
					seq_printf(m, "           %5u\n",
						ilpbk->loopback_ports[j]);
				}
			}
		}
		spin_unlock_irqrestore(&sinfo->lock, flags);
	}
	return 0;
}

static int bkn_proc_ilpbk_open(struct inode *inode, struct file *file)
{
	return single_open(file, bkn_proc_ilpbk_show, NULL);
}

const struct file_operations bkn_proc_ilpbk_file_ops = {
	.owner = THIS_MODULE,
	.open = bkn_proc_ilpbk_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

/*
 * NOTE:
 *
 * This is a replacement of bkn_knet_netif_create from Broadcom. Please
 * upgrade this function whenever BCM-SDK is upgraded.
 *
 * TODO: Merge bkn_knet_netif_modify and bkn_knet_netif_create for easy
 * upgrade in future.
 */
static int bkn_knet_netif_modify(kcom_msg_netif_modify_t *kmsg, int len)
{
	bkn_switch_info_t *sinfo;
	struct net_device *dev;
	unsigned long flags;
	bkn_priv_t *priv;
	uint8 *ma;

	kmsg->hdr.type = KCOM_MSG_TYPE_RSP;

	switch (kmsg->netif.type) {
	case KCOM_NETIF_T_VLAN:
	case KCOM_NETIF_T_PORT:
	case KCOM_NETIF_T_META:
		break;
	default:
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}
	sinfo = bkn_sinfo_from_unit(kmsg->hdr.unit);
	if (!sinfo) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}
	ma = kmsg->netif.macaddr;
	if ((ma[0] | ma[1] | ma[2] | ma[3] | ma[4] | ma[5]) == 0) {
		bkn_dev_mac[5]++;
		ma = bkn_dev_mac;
	}
	dev = dev_get_by_name(&init_net, kmsg->netif.name);
	if (!dev) {
		kmsg->hdr.status = KCOM_E_RESOURCE;
		return sizeof(kcom_msg_hdr_t);
	}

	priv = netdev_priv(dev);
	priv->dev = dev;

	/* Check to make sure that devices created via netlink are the
	 * same and warn if needed */
	if (priv->sinfo != sinfo) {
		gprintk("%s: Initial sinfo does not match new sinfo\n",
			dev->name);
		priv->sinfo = sinfo;
	}

	if (priv->type != kmsg->netif.type) {
		gprintk("%s: Initial type %d does not match new type %d\n",
			dev->name, priv->type, kmsg->netif.type);
		priv->type = kmsg->netif.type;
	}

	priv->vlan = kmsg->netif.vlan;
	if (priv->type == KCOM_NETIF_T_PORT) {
		priv->port = kmsg->netif.port;
		memcpy(priv->itmh, kmsg->netif.itmh, 4);
		priv->qnum = kmsg->netif.qnum;
	} else {
		if ((device_is_dune(sinfo)) &&
		    (priv->type == KCOM_NETIF_T_VLAN)) {
			/* set port as SSP to PTCH */
			priv->port = kmsg->netif.port;
			priv->qnum = kmsg->netif.qnum;
		} else {
			priv->port = -1;
		}
	}
	priv->flags = kmsg->netif.flags;
	priv->cb_user_data = kmsg->netif.cb_user_data;

	spin_lock_irqsave(&sinfo->lock, flags);
	while (!list_empty(&priv->vlan_xlate_list)) {
		bkn_vlan_xlate_t *vlan_xlate = list_entry(priv->vlan_xlate_list.next,
							  bkn_vlan_xlate_t, list);
		list_del(&vlan_xlate->list);
		DBG_VERB(("Removed stale vlan_xlate int vlan %d.%d ext vlan %d.%d\n",
			  vlan_xlate->int_vlan, vlan_xlate->int_ivid,
			  vlan_xlate->ext_vlan, vlan_xlate->ext_ivid));
		kfree(vlan_xlate);
	}
	spin_unlock_irqrestore(&sinfo->lock, flags);

	/* Force RCPU encapsulation if rcpu_mode */
	if (rcpu_mode) {
		priv->flags |= KCOM_NETIF_F_RCPU_ENCAP;
		DBG_RCPU(("RCPU auto-enabled\n"));
	}

	DBG_VERB(("Modified Ethernet device %s\n", dev->name));

	kmsg->netif.id = priv->id;
	memcpy(kmsg->netif.macaddr, dev->dev_addr, 6);
	memcpy(kmsg->netif.name, dev->name, KCOM_NETIF_NAME_MAX - 1);

	dev_put(dev);
	return sizeof(*kmsg);
}

/* vlan_xlate functions */

static int
bkn_knet_vlan_xlate_create2(kcom_msg_vlan_xlate_create2_t *kmsg, int len)
{
	bkn_switch_info_t *sinfo;
	struct list_head *list;
	bkn_priv_t *priv;
	bkn_vlan_xlate_t *vlan_xlate;
	unsigned long flags;

	sinfo = bkn_sinfo_from_unit(kmsg->hdr.unit);
	if (!sinfo) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}

	spin_lock_irqsave(&sinfo->lock, flags);

	priv = bkn_netif_lookup_port(sinfo, kmsg->vlan_info.port);

	if (!priv) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_RESOURCE;
		return sizeof(kcom_msg_hdr_t);
	}

	list_for_each(list, &priv->vlan_xlate_list) {
		vlan_xlate = (bkn_vlan_xlate_t *)list;
		if (!vlan_xlate ||
		    (vlan_xlate->int_vlan == kmsg->vlan_info.int_vlan &&
		     vlan_xlate->int_ivid == kmsg->vlan_info.int_ivid &&
		     vlan_xlate->ext_vlan == kmsg->vlan_info.ext_vlan &&
		     vlan_xlate->ext_ivid == kmsg->vlan_info.ext_ivid)) {
			/* entry already exists, fail */
			spin_unlock_irqrestore(&sinfo->lock, flags);
			kmsg->hdr.status = KCOM_E_RESOURCE;
			return sizeof(kcom_msg_hdr_t);
		}
	}
	vlan_xlate = kzalloc(sizeof(*vlan_xlate), GFP_ATOMIC);
	if (!vlan_xlate) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}
	vlan_xlate->int_vlan = kmsg->vlan_info.int_vlan;
	vlan_xlate->int_ivid = kmsg->vlan_info.int_ivid;
	vlan_xlate->ext_vlan = kmsg->vlan_info.ext_vlan;
	vlan_xlate->ext_ivid = kmsg->vlan_info.ext_ivid;

	list_add_tail(&vlan_xlate->list, &priv->vlan_xlate_list);

	spin_unlock_irqrestore(&sinfo->lock, flags);

	DBG_VERB(("Created vlan_xlate int vlan %d.%d ext vlan %d.%d\n",
		  vlan_xlate->int_vlan, vlan_xlate->int_ivid,
		  vlan_xlate->ext_vlan, vlan_xlate->ext_ivid));

	return len;
}

static int
bkn_knet_vlan_xlate_create(kcom_msg_vlan_xlate_create_t *kmsg, int len)
{
	kcom_msg_vlan_xlate_create2_t kmsg2;

	kmsg2.hdr = kmsg->hdr;
	kmsg2.vlan_info.port = kmsg->vlan_info.port;
	kmsg2.vlan_info.int_vlan = kmsg->vlan_info.int_vlan;
	kmsg2.vlan_info.int_ivid = 0;
	kmsg2.vlan_info.ext_vlan = kmsg->vlan_info.ext_vlan;
	kmsg2.vlan_info.ext_ivid = 0;

	return bkn_knet_vlan_xlate_create2(&kmsg2, len);
}

static int
bkn_knet_vlan_xlate_destroy2(kcom_msg_vlan_xlate_destroy2_t *kmsg, int len)
{
	bkn_switch_info_t *sinfo;
	struct list_head *list;
	bkn_priv_t *priv;
	bkn_vlan_xlate_t *vlan_xlate;
	unsigned long flags;

	sinfo = bkn_sinfo_from_unit(kmsg->hdr.unit);
	if (!sinfo) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}

	spin_lock_irqsave(&sinfo->lock, flags);

	priv = bkn_netif_lookup_port(sinfo, kmsg->vlan_info.port);

	if (!priv) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_RESOURCE;
		return sizeof(kcom_msg_hdr_t);
	}

	list_for_each(list, &priv->vlan_xlate_list) {
		vlan_xlate = (bkn_vlan_xlate_t *)list;
		if (vlan_xlate &&
		    (vlan_xlate->int_vlan == kmsg->vlan_info.int_vlan &&
		     vlan_xlate->int_ivid == kmsg->vlan_info.int_ivid &&
		     vlan_xlate->ext_vlan == kmsg->vlan_info.ext_vlan &&
		     vlan_xlate->ext_ivid == kmsg->vlan_info.ext_ivid)) {
			/* entry found, remove it */
			DBG_VERB(("Removed vlan_xlate int vlan %d.%d ext vlan %d.%d\n",
				  vlan_xlate->int_vlan, vlan_xlate->int_ivid,
				  vlan_xlate->ext_vlan, vlan_xlate->ext_ivid));
			list_del(&vlan_xlate->list);
			kfree(vlan_xlate);
			spin_unlock_irqrestore(&sinfo->lock, flags);
			return len;
		}
	}

	/* failure case when entry was not found */
	spin_unlock_irqrestore(&sinfo->lock, flags);
	kmsg->hdr.status = KCOM_E_NOT_FOUND;
	return sizeof(kcom_msg_hdr_t);
}

static int
bkn_knet_vlan_xlate_destroy(kcom_msg_vlan_xlate_destroy_t *kmsg, int len)
{
	kcom_msg_vlan_xlate_destroy2_t kmsg2;

	kmsg2.hdr = kmsg->hdr;
	kmsg2.vlan_info.port = kmsg->vlan_info.port;
	kmsg2.vlan_info.int_vlan = kmsg->vlan_info.int_vlan;
	kmsg2.vlan_info.int_ivid = 0;
	kmsg2.vlan_info.ext_vlan = kmsg->vlan_info.ext_vlan;
	kmsg2.vlan_info.ext_ivid = 0;

	return bkn_knet_vlan_xlate_destroy2(&kmsg2, len);
}

static int bkn_knet_vlan_xlate_get2(kcom_msg_vlan_xlate_get2_t *kmsg, int len)
{
	bkn_switch_info_t *sinfo;
	bkn_vlan_xlate_t *vlan_xlate;
	struct list_head *list;
	unsigned long flags;
	bkn_priv_t *priv;

	kmsg->hdr.type = KCOM_MSG_TYPE_RSP;

	sinfo = bkn_sinfo_from_unit(kmsg->hdr.unit);
	if (!sinfo) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}

	/* either int_vlan or ext_vlan should be zero */
	if (!kmsg->vlan_info.int_vlan && !kmsg->vlan_info.ext_vlan) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}

	spin_lock_irqsave(&sinfo->lock, flags);

	priv = bkn_netif_lookup_port(sinfo, kmsg->vlan_info.port);

	if (!priv) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_RESOURCE;
		return sizeof(kcom_msg_hdr_t);
	}

	list_for_each(list, &priv->vlan_xlate_list) {
		vlan_xlate = (bkn_vlan_xlate_t *)list;
		if (vlan_xlate &&
		    vlan_xlate->ext_vlan == kmsg->vlan_info.ext_vlan &&
		    vlan_xlate->ext_ivid == kmsg->vlan_info.ext_ivid) {
			/* entry found, remove it */
			DBG_VERB(("Resolved vlan_xlate int vlan %d.%d ext vlan %d.%d\n",
				  vlan_xlate->int_vlan, vlan_xlate->int_ivid,
				  vlan_xlate->ext_vlan, vlan_xlate->ext_ivid));
			kmsg->vlan_info.int_vlan = kmsg->vlan_info.int_vlan;
			spin_unlock_irqrestore(&sinfo->lock, flags);
			return len;
		}
	}

	/* failure case when entry was not found */
	spin_unlock_irqrestore(&sinfo->lock, flags);
	kmsg->hdr.status = KCOM_E_NOT_FOUND;
	return sizeof(kcom_msg_hdr_t);
}

static int bkn_knet_vlan_xlate_get(kcom_msg_vlan_xlate_get_t *kmsg, int len)
{
	kcom_msg_vlan_xlate_get2_t kmsg2;

	kmsg2.hdr = kmsg->hdr;
	kmsg2.vlan_info.port = kmsg->vlan_info.port;
	kmsg2.vlan_info.int_vlan = kmsg->vlan_info.int_vlan;
	kmsg2.vlan_info.int_ivid = 0;
	kmsg2.vlan_info.ext_vlan = kmsg->vlan_info.ext_vlan;
	kmsg2.vlan_info.ext_ivid = 0;

	return bkn_knet_vlan_xlate_get2(&kmsg2, len);
}

static bkn_ilpbk_entry_t *bkn_knet_ilpbk_entry_find(bkn_switch_info_t *sinfo,
					            int loopback_id)
{
	bkn_ilpbk_entry_t *ilpbk;
	struct hlist_head *h;

	h = &sinfo->ilpbk_entry[loopback_id & (ILPBK_ENTRY_MAX - 1)];
	hlist_for_each_entry(ilpbk, h, hlist)
		if (ilpbk->loopback_id == loopback_id)
			return ilpbk;

	return NULL;
}

static bkn_ilpbk_entry_t *bkn_knet_ilpbk_entry_find_by_port(
					bkn_switch_info_t *sinfo,
					int port)
{
	bkn_ilpbk_entry_t *ilpbk;
	struct hlist_head *h;
	int i, j;

	for (i = 0; i < ILPBK_ENTRY_MAX; i++) {
		h = &sinfo->ilpbk_entry[i];
		hlist_for_each_entry(ilpbk, h, hlist) {
			for (j = 0; j < ilpbk->num_ports; j++) {
				if (ilpbk->loopback_ports[j] == port)
					return ilpbk;
			}
		}
	}
	return NULL;
}

static void bkn_knet_ilpbk_entry_add(bkn_switch_info_t *sinfo,
				     bkn_ilpbk_entry_t *ilpbk)
{
	struct hlist_head *h;
	u32 hash;

	hash = ilpbk->loopback_id & (ILPBK_ENTRY_MAX - 1);
	h = &sinfo->ilpbk_entry[hash];
	hlist_add_head(&ilpbk->hlist, h);
}

static void bkn_knet_ilpbk_entry_del(bkn_ilpbk_entry_t *ilpbk)
{
	hlist_del_init(&ilpbk->hlist);
}

static void bkn_knet_ilpbk_entry_clean(bkn_switch_info_t *sinfo)
{
	bkn_ilpbk_entry_t *ilpbk;
	struct hlist_node *tmp;
	int i;

	for (i = 0; i < ILPBK_ENTRY_MAX; i++) {
		hlist_for_each_entry_safe(ilpbk, tmp, &sinfo->ilpbk_entry[i],
					  hlist) {
			bkn_knet_ilpbk_entry_del(ilpbk);
			kfree(ilpbk);
		}
	}
}

/* Interl loopback functions */
static int
bkn_knet_ilpbk_entry_create(kcom_msg_ilpbk_entry_create_t *kmsg, int len)
{
	bkn_ilpbk_entry_t *ilpbk;
	bkn_switch_info_t *sinfo;
	unsigned long flags;
	int i;

	sinfo = bkn_sinfo_from_unit(kmsg->hdr.unit);
	if (!sinfo) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}

	spin_lock_irqsave(&sinfo->lock, flags);
	ilpbk = bkn_knet_ilpbk_entry_find(sinfo, kmsg->ilpbk_info.loopback_id);
	if (ilpbk) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_RESOURCE;
		return sizeof(kcom_msg_hdr_t);
	}

	ilpbk = kzalloc(sizeof(*ilpbk), GFP_ATOMIC);
	if (!ilpbk) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}
	ilpbk->loopback_id = kmsg->ilpbk_info.loopback_id;
	ilpbk->rx_ifindex  = kmsg->ilpbk_info.rx_ifindex;
	ilpbk->num_ports   = kmsg->ilpbk_info.num_ports;
	for (i = 0; i < ilpbk->num_ports; i++)
		ilpbk->loopback_ports[i] = kmsg->ilpbk_info.loopback_ports[i];

	bkn_knet_ilpbk_entry_add(sinfo, ilpbk);

	pr_info("Added loopback entry %u\n", ilpbk->loopback_id);

	spin_unlock_irqrestore(&sinfo->lock, flags);

	return len;
}

static int
bkn_knet_ilpbk_entry_destroy(kcom_msg_ilpbk_entry_destroy_t *kmsg, int len)
{
	bkn_ilpbk_entry_t *ilpbk;
	bkn_switch_info_t *sinfo;
	unsigned long flags;

	sinfo = bkn_sinfo_from_unit(kmsg->hdr.unit);
	if (!sinfo) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}

	spin_lock_irqsave(&sinfo->lock, flags);
	ilpbk = bkn_knet_ilpbk_entry_find(sinfo, kmsg->ilpbk_info.loopback_id);
	if (!ilpbk) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_RESOURCE;
		return sizeof(kcom_msg_hdr_t);
	}

	pr_info("Deleted loopback entry %u\n", ilpbk->loopback_id);

	bkn_knet_ilpbk_entry_del(ilpbk);
	kfree(ilpbk);

	spin_unlock_irqrestore(&sinfo->lock, flags);

	return len;
}

static int
bkn_knet_ilpbk_entry_get(kcom_msg_ilpbk_entry_get_t *kmsg, int len)
{
	bkn_ilpbk_entry_t *ilpbk;
	bkn_switch_info_t *sinfo;
	unsigned long flags;
        int i;

	sinfo = bkn_sinfo_from_unit(kmsg->hdr.unit);
	if (!sinfo) {
		kmsg->hdr.status = KCOM_E_PARAM;
		return sizeof(kcom_msg_hdr_t);
	}

	spin_lock_irqsave(&sinfo->lock, flags);
	ilpbk = bkn_knet_ilpbk_entry_find(sinfo, kmsg->ilpbk_info.loopback_id);
	if (!ilpbk) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		kmsg->hdr.status = KCOM_E_RESOURCE;
		return sizeof(kcom_msg_hdr_t);
	}

	kmsg->ilpbk_info.loopback_id = ilpbk->loopback_id;
	kmsg->ilpbk_info.rx_ifindex  = ilpbk->rx_ifindex;
        kmsg->ilpbk_info.num_ports   = ilpbk->num_ports;
        for (i = 0; i < ilpbk->num_ports; i++)
		kmsg->ilpbk_info.loopback_ports[i] = ilpbk->loopback_ports[i];

	spin_unlock_irqrestore(&sinfo->lock, flags);

	return len;
}

static void bkn_netif_wake_queue(bkn_switch_info_t *sinfo,
				 struct net_device *dev)
{
	if (netif_queue_stopped(dev) && (sinfo->tx.free > 1) &&
	    (!dev->proto_down))
		netif_wake_queue(dev);
}

static int bkn_change_proto_down(struct net_device *dev, bool proto_down)
{
	bkn_priv_t *priv = netdev_priv(dev);
	bkn_switch_info_t *sinfo = priv->sinfo;
	unsigned long flags;

	spin_lock_irqsave(&sinfo->lock, flags);
	if (dev->proto_down == proto_down) {
		spin_unlock_irqrestore(&sinfo->lock, flags);
		return 0;
	}

	/* On protodown-on switchd disables the front-panel port and sends a
	 * carrier down notification to knet.
	 * knet must stop transmitting packets before the disable happens in
	 * switchd otherwise the packets will stay in the port's egress queue
	 * and if the number is large enough backpressure the ingress/CPU
	 * port resulting in HOLB
	 */
	dev->proto_down = proto_down;
	if (dev->proto_down) {
		++sinfo->tx.protodown_stops;
		netif_stop_queue(dev);
	} else {
		bkn_netif_wake_queue(sinfo, dev);
	}

	spin_unlock_irqrestore(&sinfo->lock, flags);
	return 0;
}

struct psample_packet_container {
	struct net_device *in;
	struct net_device *out;
	bool ingress;
};

static struct sk_buff_head psample_packet_queue;
static struct tasklet_struct psample_tasklet;

static void psample_packet_handler(unsigned long data);

static void psample_queue_init(void)
{
	skb_queue_head_init(&psample_packet_queue);
	tasklet_init(&psample_tasklet, psample_packet_handler, 0);
}

static void psample_drop_refs(struct sk_buff *skb)
{
	struct psample_packet_container *ppc;

	if (!skb)
		return;

	ppc = (struct psample_packet_container *)skb_tail_pointer(skb);

	/* refs were held until packet was logged, drop them now */
	if (ppc->in)
		dev_put(ppc->in);
	if (ppc->out)
		dev_put(ppc->out);

	/* free skb as this was a clone and no one else was using it */
	kfree_skb(skb);
}

static void psample_skb_queue_purge(struct sk_buff_head *list)
{
	struct sk_buff *skb;

	while ((skb = __skb_dequeue(list)) != NULL)
		psample_drop_refs(skb);
}

static void psample_queue_cleanup(void)
{
	tasklet_kill(&psample_tasklet);
	psample_skb_queue_purge(&psample_packet_queue);
}

static void psample_packet_handler(unsigned long data)
{
	struct psample_packet_container *ppc;
	struct psample_group *psample_group;
	struct bkn_sample *sample;
	bkn_priv_t *bkn_port;
	struct sk_buff *skb;
	u32 size;

	while ((skb = skb_dequeue(&psample_packet_queue)) != NULL) {
		ppc = (struct psample_packet_container *)skb_tail_pointer(skb);
		bkn_port = ppc->ingress ?
			   netdev_priv(ppc->in) : netdev_priv(ppc->out);

		if (unlikely(!bkn_port)) {
			netdev_warn(bkn_port->dev,
				    "Port %s: sample skb received for non-existent port\n",
				    bkn_port->dev->name);
			goto out;
		}

		sample = ppc->ingress ?
			 bkn_port->sample_ingress : bkn_port->sample_egress;

		if (unlikely(!sample)) {
			netdev_warn(bkn_port->dev,
				    "Port %s: sample skb received on unsupported port\n",
				    bkn_port->dev->name);
			goto out;
		}

		size = sample->truncate ? sample->trunc_size : skb->len;

		rcu_read_lock();

		psample_group = rcu_dereference(sample->psample_group);
		if (!psample_group)
			goto out;

		psample_sample_packet(psample_group, skb, size,
				      ppc->in->ifindex,
				      ppc->out ? ppc->out->ifindex : 0,
				      sample->rate);
out:
		rcu_read_unlock();
		psample_drop_refs(skb);
	}
}

void bkn_sample_psample(struct sk_buff *skb,
			struct net_device *srcdev,
			struct net_device *destdev,
			bool ingress)

{
	struct psample_packet_container *ppc;
	struct sk_buff *nskb;

	skb_get(skb);
	nskb = skb_copy_expand(skb, skb_headroom(skb),
			       sizeof(struct psample_packet_container),
			       GFP_ATOMIC);
	kfree_skb(skb);

	if (!nskb) {
		net_warn_ratelimited("Unable to alloc sk_buff in %s\n",
				     __func__);
		return;
	}

	ppc = (struct psample_packet_container *)skb_tail_pointer(nskb);
	ppc->in = srcdev;
	ppc->out = destdev;

	if (ppc->in)
		dev_hold(ppc->in);
	if (ppc->out)
		dev_hold(ppc->out);

	ppc->ingress = ingress;

	skb_queue_tail(&psample_packet_queue, nskb);
	tasklet_schedule(&psample_tasklet);
}

static struct sk_buff *bkn_rx_skb_sample(struct sk_buff *skb,
					 int dev_no, int chan, void *meta)
{
	u32 *dcb = (u32 *)meta;
	bool ingress_sample;
	bool egress_sample;
	bkn_priv_t *priv = netdev_priv(skb->dev);
	bkn_switch_info_t *sinfo = bkn_sinfo_from_unit(dev_no);
	unsigned char dest_port;
	bkn_priv_t *dest_priv = NULL;
	struct net_device *dest_dev = NULL;
	bool is_mcast;
	unsigned char dest_mod;
    u32 *mh0;

    /* Extract the dest-port info from the Hg2 header. For mcast packets the
     * destport=mc_grp_idx-LSBs and destmod=mg_grp_idx-MSBs
     */
    if(sinfo->cmic_type == 'x')
        mh0 = dcb;
    else
        mh0 = &(dcb[6]);

    is_mcast = !!(mh0[0] & (1 << 20));
    dest_port = (mh0[0] & 0xff);
    dest_mod = (mh0[0] & 0xff00) >> 8;

	/* if mcast the dest_port field is overloaded so skip xlate to dev;
	 * otherwise we will have unexpected results
	 */
	if (!is_mcast) {
		dest_priv = bkn_netif_lookup_port(sinfo, dest_port);
		dest_dev = dest_priv ? dest_priv->dev : NULL;
	}

	/* only sample if DCB says sampled and psample is enabled */
	/* refer reason map for bit position (src/soc/common/dcb.c)
	 * e.g. dcb26_rx_reason_map for T2
	 */
    if(sinfo->dcb_type == 36) {          //Trident3
        ingress_sample = !!(dcb[5] & 0x8) &&
            priv->sample_ingress->psample_group;
        egress_sample = !!(dcb[5] & 0x10) && dest_priv &&
            dest_priv->sample_egress->psample_group;
    } else if (sinfo->dcb_type == 38) {      //TH3
        ingress_sample = !!(dcb[5] & 0x8) &&
            priv->sample_ingress->psample_group;
        egress_sample = !!(dcb[5] & 0x4) && dest_priv &&
            dest_priv->sample_egress->psample_group;
    } else {
        ingress_sample = !!(dcb[3] & 0x20) &&
            priv->sample_ingress->psample_group;
        egress_sample = !!(dcb[3] & 0x40) && dest_priv &&
            dest_priv->sample_egress->psample_group;
    }

	if (!ingress_sample && !egress_sample)
		return skb;

	if (ingress_sample) {
		DBG_PKT(("Sampled Rx (%d %s %d) mc %s dport %d/%d dest %s\n",
			 priv->sample_ingress->psample_group->group_num,
			 skb->dev->name, skb->len,
			 is_mcast ? "y" : "n", dest_mod, dest_port,
			 dest_dev ? dest_dev->name : "-"));

		bkn_sample_psample(skb, skb->dev, dest_dev, true);
		sinfo->rx[chan].num_ingress_samples++;
	}
	if (egress_sample) {
		DBG_PKT(("Sampled Tx SKB (%d %s %d bytes) src %s\n",
			 dest_priv->sample_egress->psample_group->group_num,
			 dest_dev ? dest_dev->name : "-", skb->len,
			 skb->dev->name));

		bkn_sample_psample(skb, skb->dev, dest_dev, false);
		sinfo->rx[chan].num_egress_samples++;

		/* traffic that is only egress sampled must be dropped */
		if (!ingress_sample)
			goto drop;
	}

	/* There are three categories here -
	 * 1. Packet directed to the CPU port will have dest_port=0 so dest_dev
	 *    lookup will fail. In this case we need to send the packet to the
	 *    kernel stack for further processing.
	 * 2. Multicast or broadcast packet will have a dest_modport=mgi and
	 *    dest_dev will also be NULL in this case. Broadcast, link-local
	 *    multicast and ipmc-misses need to be sent to the kernel stack.
	 *    IP multicast forwarded data must be dropped; that decision can be
	 *    made based on repl_nhi field which is NOT parsed currently. So for
	 *    now everything in this category is sent up the stack. End result
	 *    is that duplicate IPMC packets may be seen by the mcast rxers.
	 * 3. Unicast (l2/l3) forwarded packet will have valid dest_port number
	 *    that maps to a kernel netdev i.e. dest_dev will be non-NULL. This
	 *    packet can be dropped.
	 */
	if (dest_dev)
		goto drop;

	return skb;
drop:
	DBG_PKT(("Dropped sampled SKB (%s %d bytes)\n",
		 skb->dev->name, skb->len));
	kfree_skb(skb);
	return NULL;
}

#define SPAN2CPU_IFNAME "mirror"
struct net_device *span2cpu_dev;

#define DCB_TYPE_TRIDENT2	26
#define DCB_TYPE_TOMAHAWK2	32
#define DCB_TYPE_TRIDENT2P	33
#define DCB_TYPE_MAVERICK	35
#define DCB_TYPE_TRIDENT3	36
#define DCB_TYPE_TOMAHAWK3	38

static void knet_setup(struct net_device *dev);

static int bkn_span2cpu_create_mirror_netdev(void)
{
	span2cpu_dev = alloc_netdev(0, SPAN2CPU_IFNAME, NET_NAME_UNKNOWN,
				    knet_setup);

	if (!span2cpu_dev)
		return -ENOMEM;

	span2cpu_dev->switchdev_ops = NULL;

	rtnl_lock();
	if (register_netdevice(span2cpu_dev)) {
		rtnl_unlock();
		pr_err("knet: Error when registering SPAN-to-CPU netdevice");
		free_netdev(span2cpu_dev);
		return -ENXIO;
	}

	dev_open(span2cpu_dev);
	netif_carrier_off(span2cpu_dev);
	rtnl_unlock();

	return 0;
}

static void bkn_span2cpu_free_mirror_netdev(void)
{
	rtnl_lock();
	dev_close(span2cpu_dev);
	unregister_netdevice(span2cpu_dev);
	rtnl_unlock();
	free_netdev(span2cpu_dev);
}

static bool bkn_span2cpu_is_span2cpu_pkt(bkn_switch_info_t *sinfo, u32 *dcb)
{

#ifdef LE_HOST
#define SPAN2CPU_DCB_BIT(x) (x)
#else
#define SPAN2CPU_DCB_BIT(x) (31-(x))
#endif

	switch (sinfo->dcb_type) {
	case DCB_TYPE_TRIDENT2:
	case DCB_TYPE_TRIDENT2P:
	case DCB_TYPE_TOMAHAWK2:
	case DCB_TYPE_MAVERICK:
		return !(dcb[13] & (1 << SPAN2CPU_DCB_BIT(7)));
	case DCB_TYPE_TRIDENT3:
		return !(dcb[12] & (1 << SPAN2CPU_DCB_BIT(23)));
	case DCB_TYPE_TOMAHAWK3:
		return !(dcb[6] & (1 << SPAN2CPU_DCB_BIT(31)));
	default:
		return false;
	}
}

static void bkn_span2cpu_deliver_to_taps(struct sk_buff *skb,
					 struct net_device *netdev)
{
	skb->dev = span2cpu_dev;
	skb->protocol = eth_type_trans(skb, skb->dev);

	skb_reset_network_header(skb);
	skb_reset_mac_len(skb);
	skb_push(skb, skb->mac_len);
	dev_queue_xmit_nit(skb, skb->dev);
	skb_pull(skb, skb->mac_len);

}

static struct sk_buff *bkn_rx_cb(struct sk_buff *skb,
				 int dev_no, int chan, void *meta)
{
	bkn_switch_info_t *sinfo = bkn_sinfo_from_unit(dev_no);
	u32 *dcb = (u32 *)meta;

	/* Handle SPAN-to-CPU packets */
	if (mirror_device_en && bkn_span2cpu_is_span2cpu_pkt(sinfo, dcb)) {
		bkn_span2cpu_deliver_to_taps(skb, span2cpu_dev);
		dev_kfree_skb(skb);
		return NULL;
	}

	/* Handle sampled packets */
	return bkn_rx_skb_sample(skb, dev_no, chan, meta);
}

static int
bkn_add_cls_matchall_sample(bkn_priv_t *bkn_port,
			    struct tc_cls_matchall_offload *cls,
			    const struct tc_action *a,
			    bool ingress)
{
	struct bkn_sample *sample = ingress ?
				    bkn_port->sample_ingress :
				    bkn_port->sample_egress;

	if (!sample)
		return -EOPNOTSUPP;
	if (rtnl_dereference(sample->psample_group)) {
		netdev_err(bkn_port->dev, "sample already active\n");
		return -EEXIST;
	}

	DBG_VERB(("sampling: %s called with group %u\n", __func__,
		 tcf_sample_psample_group(a)->group_num));

	rcu_assign_pointer(sample->psample_group, tcf_sample_psample_group(a));
	sample->truncate = tcf_sample_truncate(a);
	sample->trunc_size = tcf_sample_trunc_size(a);
	sample->rate = tcf_sample_rate(a);

	return 0;
}

static void
bkn_del_cls_matchall_sample(bkn_priv_t *bkn_port, bool ingress)
{
	struct bkn_sample *sample = ingress ?
				    bkn_port->sample_ingress :
				    bkn_port->sample_egress;

	if (sample)
		RCU_INIT_POINTER(sample->psample_group, NULL);

}

static int bkn_add_cls_matchall(bkn_priv_t *bkn_port,
				struct tc_cls_matchall_offload *f,
				bool ingress)
{
	__be16 protocol = f->common.protocol;
	const struct tc_action *a;
	int err;

	if (!tcf_exts_has_one_action(f->exts)) {
		netdev_err(bkn_port->dev,
			   "only singular actions are supported\n");
		return -EOPNOTSUPP;
	}

	a = tcf_exts_first_action(f->exts);

	if (is_tcf_sample(a) && protocol == htons(ETH_P_ALL))
		err = bkn_add_cls_matchall_sample(bkn_port, f, a, ingress);
	else
		err = -EOPNOTSUPP;

	return err;
}

static void bkn_del_cls_matchall(bkn_priv_t *bkn_port,
				 struct tc_cls_matchall_offload *f,
				 bool ingress)
{
	bkn_del_cls_matchall_sample(bkn_port, ingress);
}

static int bkn_setup_tc_cls_matchall(bkn_priv_t *bkn_port,
				     struct tc_cls_matchall_offload *f,
				     bool ingress)
{
	switch (f->command) {
	case TC_CLSMATCHALL_REPLACE:
		return bkn_add_cls_matchall(bkn_port, f, ingress);
	case TC_CLSMATCHALL_DESTROY:
		bkn_del_cls_matchall(bkn_port, f, ingress);
		return 0;
	default:
		return -EOPNOTSUPP;
	}
}

static int bkn_setup_tc_block_cb(enum tc_setup_type type,
				 void *type_data, void *cb_priv,
				 bool ingress)
{
	bkn_priv_t *bkn_port = cb_priv;

	switch (type) {
	case TC_SETUP_CLSMATCHALL:
		if (!tc_cls_can_offload_and_chain0(bkn_port->dev, type_data))
			return -EOPNOTSUPP;

		return bkn_setup_tc_cls_matchall(bkn_port, type_data, ingress);
	default:
		return -EOPNOTSUPP;
	}
}

static int bkn_setup_tc_block_cb_ingress(enum tc_setup_type type,
					 void *type_data, void *cb_priv)
{
	return bkn_setup_tc_block_cb(type, type_data, cb_priv, true);
}

static int bkn_setup_tc_block_cb_egress(enum tc_setup_type type,
					void *type_data, void *cb_priv)
{
	return bkn_setup_tc_block_cb(type, type_data, cb_priv, false);
}

static int bkn_setup_tc_block(bkn_priv_t *bkn_port,
			      struct tc_block_offload *f)
{
	tc_setup_cb_t *cb;

	if (f->binder_type == TCF_BLOCK_BINDER_TYPE_CLSACT_INGRESS)
		cb = bkn_setup_tc_block_cb_ingress;
	else if (f->binder_type == TCF_BLOCK_BINDER_TYPE_CLSACT_EGRESS)
		cb = bkn_setup_tc_block_cb_egress;
	else
		return -EOPNOTSUPP;

	switch (f->command) {
	case TC_BLOCK_BIND:
		return tcf_block_cb_register(f->block, cb, bkn_port, bkn_port,
					     f->extack);
	case TC_BLOCK_UNBIND:
		tcf_block_cb_unregister(f->block, cb, bkn_port);
		return 0;
	default:
		return -EOPNOTSUPP;
	}
}

static int bkn_setup_tc(struct net_device *dev, enum tc_setup_type type,
			void *type_data)
{
	bkn_priv_t *bkn_port = netdev_priv(dev);

	if (type != TC_SETUP_BLOCK)
		return -EOPNOTSUPP;

	return bkn_setup_tc_block(bkn_port, type_data);
}

static inline void bkn_stat64_add(bkn_priv_t *priv, u64 *stat, int incr)
{
	u64_stats_update_begin(&priv->syncp);
	(*stat) += incr;
	u64_stats_update_end(&priv->syncp);
}

void bkn_get_stats64(struct net_device *dev, struct rtnl_link_stats64 *stats)
{
	bkn_priv_t *priv = netdev_priv(dev);
	struct rtnl_link_stats64 bkn_stats;
	struct ethtool_stats estats = { 0, };
	unsigned int start;
	int count;
	u64 *hw_stats;

	do {
		start = u64_stats_fetch_begin(&priv->syncp);
		memcpy(&bkn_stats, &priv->stats64,
		       sizeof(struct rtnl_link_stats64));
	} while (u64_stats_fetch_retry(&priv->syncp, start));

	count = port_get_sset_count(dev, ETH_SS_STATS);
	if (count <= 0)
		return;

	hw_stats = kcalloc(count, sizeof(u64), GFP_ATOMIC);
	if (!hw_stats)
		return;

	estats.n_stats = count;
	port_get_ethtool_stats(dev, &estats, hw_stats);

	stats->rx_packets = hw_stats[1] + hw_stats[2] + hw_stats[3];
	stats->rx_bytes = hw_stats[0];
	stats->multicast = hw_stats[3];
	stats->rx_errors = hw_stats[12] + hw_stats[13] + hw_stats[25] + bkn_stats.rx_errors;
	stats->rx_dropped = hw_stats[8] + hw_stats[9] + hw_stats[10] + bkn_stats.rx_dropped;
	stats->rx_frame_errors = bkn_stats.rx_frame_errors;

	stats->tx_packets = hw_stats[5] + hw_stats[6] + hw_stats[7];
	stats->tx_bytes = hw_stats[4];
	stats->tx_errors = hw_stats[18] + bkn_stats.tx_fifo_errors;
	stats->tx_dropped = hw_stats[17] + bkn_stats.tx_dropped;

	kfree(hw_stats);
}

static int bkn_port_attr_get(struct net_device *dev,
			     struct switchdev_attr *attr)
{
	bkn_priv_t *priv = netdev_priv(dev);
	bkn_switch_info_t *sinfo = priv->sinfo;

	switch (attr->id) {
	case SWITCHDEV_ATTR_ID_PORT_PARENT_ID:
		attr->u.ppid.id_len = sizeof(sinfo->dev_no);
		memcpy(&attr->u.ppid.id, &sinfo->dev_no,
		       attr->u.ppid.id_len);
		break;
	default:
		return -EOPNOTSUPP;
	}

	return 0;
}

static const struct switchdev_ops bkn_switchdev_ops = {
	.switchdev_port_attr_get = bkn_port_attr_get,
};

#define KNET_OFFLOAD_FWD_MARK -5

static int bkn_netdev_notifier_event(struct notifier_block *this,
				     unsigned long event, void *ptr)
{
	struct net_device *dev = netdev_notifier_info_to_dev(ptr);
	struct netdev_notifier_changeupper_info *info = ptr;
	bool is_switch_port;
	bool is_vxlan;

	struct switchdev_attr attr = {
		.orig_dev = dev,
		.id = SWITCHDEV_ATTR_ID_PORT_PARENT_ID,
	};

	is_switch_port = !switchdev_port_attr_get(dev, &attr);
	is_vxlan = dev->rtnl_link_ops ?
		   !strcmp(dev->rtnl_link_ops->kind, "vxlan") :
		   false;

	switch (event) {
	case NETDEV_CHANGEUPPER:
		if (info->linking && (is_switch_port || is_vxlan))
			br_port_set_offload_fwd_mark(info->upper_dev,
						     dev,
						     KNET_OFFLOAD_FWD_MARK);
		break;
	default:
		break;
	}

	return NOTIFY_DONE;
}

static struct notifier_block bkn_netdev_notifier = {
	.notifier_call = bkn_netdev_notifier_event,
};

static int knet_addons_init(void)
{
	int ret;

	if (register_netdevice_notifier(&bkn_netdev_notifier)) {
		printk(KERN_INFO "%s: failed to register netdev notifier\n",
		       __func__);
	}

	psample_queue_init();

	if (mirror_device_en) {
		ret = bkn_span2cpu_create_mirror_netdev();
		if (ret) {
			pr_err("knet: Error when creating SPAN-to-CPU netdevice");
			return ret;
		}
	}

	return bkn_rx_skb_cb_register(bkn_rx_cb);
}

static int knet_addons_cleanup(void)
{
	if (mirror_device_en)
		bkn_span2cpu_free_mirror_netdev();
	psample_queue_cleanup();
	return unregister_netdevice_notifier(&bkn_netdev_notifier);
}
#endif /* __LINUX_BCM_KNET_ADDONS_H__ */
