/*
 * I2C multiplexer driver for PCA9541/PCA9641 bus master selectors
 *
 * Copyright (c) 2010 Ericsson AB.
 *
 * Author: Guenter Roeck <linux@roeck-us.net>
 *
 * Derived from:
 *  pca954x.c
 *
 *  Copyright (c) 2008-2009 Rodolfo Giometti <giometti@linux.it>
 *  Copyright (c) 2008-2009 Eurotech S.p.A. <info@eurotech.it>
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2. This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/bitops.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/i2c-mux.h>
#include <linux/jiffies.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/platform_data/pca954x.h>
#include <linux/slab.h>

/*
 * The PCA9541 and PCA9641 are bus master selector. They support two I2C masters
 * connected to a single slave bus.
 *
 * Before each bus transaction, a master has to acquire bus ownership. After the
 * transaction is complete, bus ownership has to be released. This fits well
 * into the I2C multiplexer framework, which provides select and release
 * functions for this purpose. For this reason, this driver is modeled as
 * single-channel I2C bus multiplexer.
 *
 * This driver assumes that the two bus masters are controlled by two different
 * hosts. If a single host controls both masters, platform code has to ensure
 * that only one of the masters is instantiated at any given time.
 */

#define PCA9541_CONTROL		0x01
#define PCA9541_ISTAT		0x02

#define PCA9541_CTL_MYBUS	BIT(0)
#define PCA9541_CTL_NMYBUS	BIT(1)
#define PCA9541_CTL_BUSON	BIT(2)
#define PCA9541_CTL_NBUSON	BIT(3)
#define PCA9541_CTL_BUSINIT	BIT(4)
#define PCA9541_CTL_TESTON	BIT(6)
#define PCA9541_CTL_NTESTON	BIT(7)

#define PCA9541_ISTAT_INTIN	BIT(0)
#define PCA9541_ISTAT_BUSINIT	BIT(1)
#define PCA9541_ISTAT_BUSOK	BIT(2)
#define PCA9541_ISTAT_BUSLOST	BIT(3)
#define PCA9541_ISTAT_MYTEST	BIT(6)
#define PCA9541_ISTAT_NMYTEST	BIT(7)

#define PCA9541_BUSON	(PCA9541_CTL_BUSON | PCA9541_CTL_NBUSON)
#define PCA9541_MYBUS	(PCA9541_CTL_MYBUS | PCA9541_CTL_NMYBUS)

#define PCA9641_ID			0x00
#define PCA9641_ID_MAGIC		0x38

#define PCA9641_CONTROL			0x01
#define PCA9641_STATUS			0x02
#define PCA9641_TIME			0x03

#define PCA9641_CTL_LOCK_REQ		BIT(0)
#define PCA9641_CTL_LOCK_GRANT		BIT(1)
#define PCA9641_CTL_BUS_CONNECT		BIT(2)
#define PCA9641_CTL_BUS_INIT		BIT(3)
#define PCA9641_CTL_SMBUS_SWRST		BIT(4)
#define PCA9641_CTL_IDLE_TIMER_DIS	BIT(5)
#define PCA9641_CTL_SMBUS_DIS		BIT(6)
#define PCA9641_CTL_PRIORITY		BIT(7)

#define PCA9641_STS_OTHER_LOCK		BIT(0)
#define PCA9641_STS_BUS_INIT_FAIL	BIT(1)
#define PCA9641_STS_BUS_HUNG		BIT(2)
#define PCA9641_STS_MBOX_EMPTY		BIT(3)
#define PCA9641_STS_MBOX_FULL		BIT(4)
#define PCA9641_STS_TEST_INT		BIT(5)
#define PCA9641_STS_SCL_IO		BIT(6)
#define PCA9641_STS_SDA_IO		BIT(7)

#define PCA9641_RES_TIME		0x03

/* arbitration timeouts, in jiffies */
#define ARB_TIMEOUT	(HZ / 8)	/* 125 ms until forcing bus ownership */
#define ARB2_TIMEOUT	(HZ / 4)	/* 250 ms until acquisition failure */

/* arbitration retry delays, in us */
#define SELECT_DELAY_SHORT	50
#define SELECT_DELAY_LONG	1000

enum chip_name {
	pca9541,
	pca9641,
};

struct chip_desc {
	int (*arbitrate)(struct i2c_client *client);
	void (*release_bus)(struct i2c_client *client);
};

struct pca9541 {
	const struct chip_desc *chip;

	struct i2c_client *client;
	unsigned long select_timeout;
	unsigned long arb_timeout;
};

static bool pca9541_mybus(int ctl)
{
	if (!(ctl & PCA9541_MYBUS))
		return true;
	return (ctl & PCA9541_MYBUS) == PCA9541_MYBUS;
}

static bool pca9541_busoff(int ctl)
{
	if (!(ctl & PCA9541_BUSON))
		return true;
	return (ctl & PCA9541_BUSON) == PCA9541_BUSON;
}

static bool pca9641_lock_grant(int ctl)
{
	return !!(ctl & PCA9641_CTL_LOCK_GRANT);
}

static bool pca9641_other_lock(int sts)
{
	return !!(sts & PCA9641_STS_OTHER_LOCK);
}

static bool pca9641_busoff(int ctl, int sts)
{
	return !pca9641_lock_grant(ctl) && !pca9641_other_lock(sts);
}

/*
 * Write to chip register. Don't use i2c_transfer()/i2c_smbus_xfer()
 * as they will try to lock the adapter a second time.
 */
static int pca9541_reg_write(struct i2c_client *client, u8 command, u8 val)
{
	struct i2c_adapter *adap = client->adapter;
	union i2c_smbus_data data = { .byte = val };

	return __i2c_smbus_xfer(adap, client->addr, client->flags,
				I2C_SMBUS_WRITE, command,
				I2C_SMBUS_BYTE_DATA, &data);
}

/*
 * Read from chip register. Don't use i2c_transfer()/i2c_smbus_xfer()
 * as they will try to lock adapter a second time.
 */
static int pca9541_reg_read(struct i2c_client *client, u8 command)
{
	struct i2c_adapter *adap = client->adapter;
	union i2c_smbus_data data;
	int ret;

	ret = __i2c_smbus_xfer(adap, client->addr, client->flags,
			       I2C_SMBUS_READ, command,
			       I2C_SMBUS_BYTE_DATA, &data);

	return ret ?: data.byte;
}

/*
 * Read from chip register of pca9641. We needed to use a separate
 * read routine for pca9641 since pca9541_read_reg uses a single
 * transaction that always fails for pca9641. Doing a 2-phase read
 * -a write with value of the offset reg to read followed by a read
 * succeeds for pca9641.
 */
static int pca9641_reg_read(struct i2c_client *client, u8 command)
{
	struct i2c_adapter *adap = client->adapter;
	union i2c_smbus_data data;
	int ret;

	ret = __i2c_smbus_xfer(adap,
			       client->addr, client->flags,
			       I2C_SMBUS_WRITE, command,
			       I2C_SMBUS_BYTE, NULL);
	if (ret < 0)
		goto out;

	ret = __i2c_smbus_xfer(adap, client->addr, client->flags,
			       I2C_SMBUS_READ, 0,
			       I2C_SMBUS_BYTE, &data);

out:
	return ret ?: data.byte;
}

/*
 * Arbitration management functions
 */

/* Release bus. Also reset NTESTON and BUSINIT if it was set. */
static void pca9541_release_bus(struct i2c_client *client)
{
	int reg;

	reg = pca9541_reg_read(client, PCA9541_CONTROL);
	if (reg >= 0 && !pca9541_busoff(reg) && pca9541_mybus(reg))
		pca9541_reg_write(client, PCA9541_CONTROL,
				  (reg & PCA9541_CTL_NBUSON) >> 1);
}

/*
 * Arbitration is defined as a two-step process. A bus master can only activate
 * the slave bus if it owns it; otherwise it has to request ownership first.
 * This multi-step process ensures that access contention is resolved
 * gracefully.
 *
 * Bus	Ownership	Other master	Action
 * state		requested access
 * ----------------------------------------------------
 * off	-		yes		wait for arbitration timeout or
 *					for other master to drop request
 * off	no		no		take ownership
 * off	yes		no		turn on bus
 * on	yes		-		done
 * on	no		-		wait for arbitration timeout or
 *					for other master to release bus
 *
 * The main contention point occurs if the slave bus is off and both masters
 * request ownership at the same time. In this case, one master will turn on
 * the slave bus, believing that it owns it. The other master will request
 * bus ownership. Result is that the bus is turned on, and master which did
 * _not_ own the slave bus before ends up owning it.
 */

/* Control commands per PCA9541 datasheet */
static const u8 pca9541_control[16] = {
	4, 0, 1, 5, 4, 4, 5, 5, 0, 0, 1, 1, 0, 4, 5, 1
};

/*
 * Channel arbitration
 *
 * Return values:
 *  <0: error
 *  0 : bus not acquired
 *  1 : bus acquired
 */
static int pca9541_arbitrate(struct i2c_client *client)
{
	struct i2c_mux_core *muxc = i2c_get_clientdata(client);
	struct pca9541 *data = i2c_mux_priv(muxc);
	int reg;

	reg = pca9541_reg_read(client, PCA9541_CONTROL);
	if (reg < 0)
		return reg;

	if (pca9541_busoff(reg)) {
		int istat;
		/*
		 * Bus is off. Request ownership or turn it on unless
		 * other master requested ownership.
		 */
		istat = pca9541_reg_read(client, PCA9541_ISTAT);
		if (!(istat & PCA9541_ISTAT_NMYTEST)
		    || time_is_before_eq_jiffies(data->arb_timeout)) {
			/*
			 * Other master did not request ownership,
			 * or arbitration timeout expired. Take the bus.
			 */
			pca9541_reg_write(client,
					  PCA9541_CONTROL,
					  pca9541_control[reg & 0x0f]
					  | PCA9541_CTL_NTESTON);
			data->select_timeout = SELECT_DELAY_SHORT;
		} else {
			/*
			 * Other master requested ownership.
			 * Set extra long timeout to give it time to acquire it.
			 */
			data->select_timeout = SELECT_DELAY_LONG * 2;
		}
	} else if (pca9541_mybus(reg)) {
		/*
		 * Bus is on, and we own it. We are done with acquisition.
		 * Reset NTESTON and BUSINIT, then return success.
		 */
		if (reg & (PCA9541_CTL_NTESTON | PCA9541_CTL_BUSINIT))
			pca9541_reg_write(client,
					  PCA9541_CONTROL,
					  reg & ~(PCA9541_CTL_NTESTON
						  | PCA9541_CTL_BUSINIT));
		return 1;
	} else {
		/*
		 * Other master owns the bus.
		 * If arbitration timeout has expired, force ownership.
		 * Otherwise request it.
		 */
		data->select_timeout = SELECT_DELAY_LONG;
		if (time_is_before_eq_jiffies(data->arb_timeout)) {
			/* Time is up, take the bus and reset it. */
			pca9541_reg_write(client,
					  PCA9541_CONTROL,
					  pca9541_control[reg & 0x0f]
					  | PCA9541_CTL_BUSINIT
					  | PCA9541_CTL_NTESTON);
		} else {
			/* Request bus ownership if needed */
			if (!(reg & PCA9541_CTL_NTESTON))
				pca9541_reg_write(client,
						  PCA9541_CONTROL,
						  reg | PCA9541_CTL_NTESTON);
		}
	}
	return 0;
}

/* Release bus. */
static void pca9641_release_bus(struct i2c_client *client)
{
	pca9541_reg_write(client, PCA9641_CONTROL, 0);
}

/*
 * Channel arbitration
 *
 * Return values:
 *  <0: error
 *  0 : bus not acquired
 *  1 : bus acquired
 */
static int pca9641_arbitrate(struct i2c_client *client)
{
	struct i2c_mux_core *muxc = i2c_get_clientdata(client);
	struct pca9541 *data = i2c_mux_priv(muxc);
	int reg_ctl, reg_sts;

	reg_ctl = pca9641_reg_read(client, PCA9641_CONTROL);
	if (reg_ctl < 0)
		return reg_ctl;
	reg_sts = pca9641_reg_read(client, PCA9641_STATUS);

	if (pca9641_busoff(reg_ctl, reg_sts)) {
		/*
		 * Bus is off. Request ownership or turn it on unless
		 * other master requested ownership.
		 */
		reg_ctl |= PCA9641_CTL_LOCK_REQ;
		pca9541_reg_write(client, PCA9641_CONTROL, reg_ctl);
		reg_ctl = pca9641_reg_read(client, PCA9641_CONTROL);

		if (pca9641_lock_grant(reg_ctl)) {
			/*
			 * Other master did not request ownership,
			 * or arbitration timeout expired. Take the bus.
			 */
			reg_ctl |= PCA9641_CTL_BUS_CONNECT |
				PCA9641_CTL_LOCK_REQ;
			pca9541_reg_write(client, PCA9641_CONTROL, reg_ctl);
			data->select_timeout = SELECT_DELAY_SHORT;

			return 1;
		}

		/*
		 * Other master requested ownership.
		 * Set extra long timeout to give it time to acquire it.
		 */
		data->select_timeout = SELECT_DELAY_LONG * 2;

		return 0;
	}

	if (pca9641_lock_grant(reg_ctl)) {
		/*
		 * Bus is on, and we own it. We are done with acquisition.
		 */
		reg_ctl |= PCA9641_CTL_BUS_CONNECT | PCA9641_CTL_LOCK_REQ;
		pca9541_reg_write(client, PCA9641_CONTROL, reg_ctl);

		return 1;
	}

	if (pca9641_other_lock(reg_sts)) {
		/*
		 * Other master owns the bus.
		 * If arbitration timeout has expired, force ownership.
		 * Otherwise request it.
		 */
		data->select_timeout = SELECT_DELAY_LONG;
		reg_ctl |= PCA9641_CTL_LOCK_REQ;
		pca9541_reg_write(client, PCA9641_CONTROL, reg_ctl);
	}

	return 0;
}

static int pca9541_select_chan(struct i2c_mux_core *muxc, u32 chan)
{
	struct pca9541 *data = i2c_mux_priv(muxc);
	struct i2c_client *client = data->client;
	int ret;
	unsigned long timeout = jiffies + ARB2_TIMEOUT;
		/* give up after this time */

	data->arb_timeout = jiffies + ARB_TIMEOUT;
		/* force bus ownership after this time */

	do {
		ret = data->chip->arbitrate(client);
		if (ret)
			return ret < 0 ? ret : 0;

		if (data->select_timeout == SELECT_DELAY_SHORT)
			udelay(data->select_timeout);
		else
			msleep(data->select_timeout / 1000);
	} while (time_is_after_eq_jiffies(timeout));

	return -ETIMEDOUT;
}

static int pca9541_release_chan(struct i2c_mux_core *muxc, u32 chan)
{
	struct pca9541 *data = i2c_mux_priv(muxc);
	struct i2c_client *client = data->client;

	data->chip->release_bus(client);
	return 0;
}

static const struct chip_desc chips[] = {
	[pca9541] = {
		.arbitrate = pca9541_arbitrate,
		.release_bus = pca9541_release_bus,
	},
	[pca9641] = {
		.arbitrate = pca9641_arbitrate,
		.release_bus = pca9641_release_bus,
	},
};

static const struct i2c_device_id pca9541_id[] = {
	{ "pca9541", pca9541 },
	{ "pca9641", pca9641 },
	{}
};

MODULE_DEVICE_TABLE(i2c, pca9541_id);

#ifdef CONFIG_OF
static const struct of_device_id pca9541_of_match[] = {
	{ .compatible = "nxp,pca9541", .data = &chips[pca9541] },
	{ .compatible = "nxp,pca9641", .data = &chips[pca9641] },
	{}
};
MODULE_DEVICE_TABLE(of, pca9541_of_match);
#endif

/*
 * I2C init/probing/exit functions
 */
static int pca9541_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	struct i2c_adapter *adap = client->adapter;
	struct pca954x_platform_data *pdata = dev_get_platdata(&client->dev);
	const struct of_device_id *match;
	const struct chip_desc *chip;
	struct i2c_mux_core *muxc;
	struct pca9541 *data;
	int force;
	int ret;

	if (!i2c_check_functionality(adap, I2C_FUNC_SMBUS_BYTE_DATA))
		return -ENODEV;

	match = of_match_device(of_match_ptr(pca9541_of_match), &client->dev);
	if (match)
		chip = of_device_get_match_data(&client->dev);
	else
		chip = &chips[id->driver_data];

	/*
	 * I2C accesses are unprotected here.
	 * We have to lock the I2C segment before releasing the bus.
	 */
	i2c_lock_bus(adap, I2C_LOCK_SEGMENT);
	chip->release_bus(client);
	i2c_unlock_bus(adap, I2C_LOCK_SEGMENT);

	/* Create mux adapter */

	force = 0;
	if (pdata)
		force = pdata->modes[0].adap_id;
	muxc = i2c_mux_alloc(adap, &client->dev, 1, sizeof(*data),
			     I2C_MUX_ARBITRATOR,
			     pca9541_select_chan, pca9541_release_chan);
	if (!muxc)
		return -ENOMEM;

	data = i2c_mux_priv(muxc);
	data->chip = chip;
	data->client = client;

	i2c_set_clientdata(client, muxc);

	ret = i2c_mux_add_adapter(muxc, force, 0, 0);
	if (ret)
		return ret;

	dev_info(&client->dev, "registered master selector for I2C %s\n",
		 client->name);

	return 0;
}

static int pca9541_remove(struct i2c_client *client)
{
	struct i2c_mux_core *muxc = i2c_get_clientdata(client);

	i2c_mux_del_adapters(muxc);
	return 0;
}

static struct i2c_driver pca9541_driver = {
	.driver = {
		   .name = "pca9541",
		   .of_match_table = of_match_ptr(pca9541_of_match),
		   },
	.probe = pca9541_probe,
	.remove = pca9541_remove,
	.id_table = pca9541_id,
};

module_i2c_driver(pca9541_driver);

MODULE_AUTHOR("Guenter Roeck <linux@roeck-us.net>");
MODULE_DESCRIPTION("PCA9541/PCA9641 I2C master selector driver");
MODULE_LICENSE("GPL v2");
