/*
 * Hardware monitoring driver for 3Y Power technology
 * YM-2651Y BR Power Supplies
 *
 * Copyright (C) 2016 Cumulus Networks, LLC
 * Author: Nikhil Dhar <ndhar@cumulusnetworks.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/err.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <linux/pmbus.h>
#include "pmbus.h"

enum chips { ym2651 };

static int ym2651_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{
	struct pmbus_driver_info *info;
	int ret;

	if (!i2c_check_functionality(client->adapter,
				     I2C_FUNC_SMBUS_BYTE_DATA |
				     I2C_FUNC_SMBUS_WORD_DATA |
				     I2C_FUNC_SMBUS_PEC))
		return -ENODEV;

	info = kzalloc(sizeof(struct pmbus_driver_info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	info->pages = 1;
	info->func[0] = PMBUS_HAVE_FAN12 | PMBUS_HAVE_STATUS_FAN12 |
			PMBUS_HAVE_TEMP | PMBUS_HAVE_STATUS_TEMP;
	/* Writes require PEC */
	client->flags = I2C_CLIENT_PEC;
	ret = pmbus_do_probe(client, id, info);
	if (ret < 0)
		kfree(info);
	return ret;
}

static int ym2651_remove(struct i2c_client *client)
{
	pmbus_do_remove(client);
	return 0;
}

static const struct i2c_device_id ym2651_id[] = {
	{"ym2651", ym2651},
	{}
};
MODULE_DEVICE_TABLE(i2c, ym2651_id);

static struct i2c_driver ym2651_driver = {
	.driver = {
		   .name = "ym2651",
		   },
	.probe = ym2651_probe,
	.remove = ym2651_remove,
	.id_table = ym2651_id,
};

module_i2c_driver(ym2651_driver);

MODULE_AUTHOR("Nikhil Dhar");
MODULE_DESCRIPTION("PMBus driver for YM-2651Y");
MODULE_LICENSE("GPL");
