/*
 * max6639.c - Support for Maxim MAX6639
 *
 * 2-Channel Temperature Monitor with Dual PWM Fan-Speed Controller
 *
 * Copyright (C) 2010, 2011 Roland Stigge <stigge@antcom.de>
 *
 * based on the initial MAX6639 support from semptian.net
 * by He Changqing <hechangqing@semptian.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/jiffies.h>
#include <linux/i2c.h>
#include <linux/hwmon.h>
#include <linux/hwmon-sysfs.h>
#include <linux/err.h>
#include <linux/mutex.h>
#include <linux/platform_data/max6639.h>

/* Addresses to scan */
static const unsigned short normal_i2c[] = { 0x2c, 0x2e, 0x2f, I2C_CLIENT_END };

/* The MAX6639 registers, valid channel numbers: 0, 1 */
#define MAX6639_REG_TEMP(ch)			(0x00 + (ch))
#define MAX6639_REG_STATUS			0x02
#define MAX6639_REG_OUTPUT_MASK			0x03
#define MAX6639_REG_GCONFIG			0x04
#define MAX6639_REG_TEMP_EXT(ch)		(0x05 + (ch))
#define MAX6639_REG_ALERT_LIMIT(ch)		(0x08 + (ch))
#define MAX6639_REG_OT_LIMIT(ch)		(0x0A + (ch))
#define MAX6639_REG_THERM_LIMIT(ch)		(0x0C + (ch))
#define MAX6639_REG_FAN_CONFIG1(ch)		(0x10 + (ch) * 4)
#define MAX6639_REG_FAN_CONFIG2a(ch)		(0x11 + (ch) * 4)
#define MAX6639_REG_FAN_CONFIG2b(ch)		(0x12 + (ch) * 4)
#define MAX6639_REG_FAN_CONFIG3(ch)		(0x13 + (ch) * 4)
#define MAX6639_REG_FAN_CNT(ch)			(0x20 + (ch))
#define MAX6639_REG_TARGET_CNT(ch)		(0x22 + (ch))
#define MAX6639_REG_FAN_PPR(ch)			(0x24 + (ch))
#define MAX6639_REG_TARGTDUTY(ch)		(0x26 + (ch))
#define MAX6639_REG_FAN_START_TEMP(ch)		(0x28 + (ch))
#define MAX6639_REG_DEVID			0x3D
#define MAX6639_REG_MANUID			0x3E
#define MAX6639_REG_DEVREV			0x3F

/* Register bits */
#define MAX6639_GCONFIG_STANDBY			0x80
#define MAX6639_GCONFIG_POR			0x40
#define MAX6639_GCONFIG_DISABLE_TIMEOUT		0x20
#define MAX6639_GCONFIG_CH2_LOCAL		0x10
#define MAX6639_GCONFIG_PWM_FREQ_HI		0x08

#define MAX6639_GCONFIG_STANDBY_OFFSET		7
#define MAX6639_GCONFIG_POR_OFFSET		6
#define MAX6639_GCONFIG_DISABLE_TIMEOUT_OFFSET	5
#define MAX6639_GCONFIG_CH2_LOCAL_OFFSET	4
#define MAX6639_GCONFIG_PWM_FREQ_HI_OFFSET	3

#define MAX6639_OUTMASK_CH1_ALERT_DISABLE       7
#define MAX6639_OUTMASK_CH2_ALERT_DISABLE       6
#define MAX6639_OUTMASK_CH1_OT_DISABLE          5
#define MAX6639_OUTMASK_CH2_OT_DISABLE          4
#define MAX6639_OUTMASK_CH1_THERM_DISABLE       3
#define MAX6639_OUTMASK_CH2_THERM_DISABLE       2
#define MAX6639_OUTMASK_FAN1_FAULT_ENABLE       1
#define MAX6639_OUTMASK_FAN2_FAULT_ENABLE       0

#define MAX6639_FAN_CONFIG1_PWM		        0x80
#define MAX6639_FAN_CONFIG1_PWM_MODE		0x80
#define MAX6639_FAN_CONFIG1_PWM_MODE_OFFSET	7
#define MAX6639_FAN_CONFIG1_DUTY		0x70
#define MAX6639_FAN_CONFIG1_DUTY_OFFSET	        4
#define MAX6639_FAN_CONFIG1_TEMP_CH		0x0c
#define MAX6639_FAN_CONFIG1_TEMP_CH_OFFSET      2
#define MAX6639_FAN_CONFIG1_RPM_RANGE	        0x03
#define MAX6639_FAN_CONFIG1_RPM_RANGE_OFFSET    0

#define MAX6639_FAN_CONFIG2a_RPM_STEPA          0xf0
#define MAX6639_FAN_CONFIG2a_RPM_STEPA_OFFSET   4
#define MAX6639_FAN_CONFIG2a_TEMP_STEPA         0x0f
#define MAX6639_FAN_CONFIG2a_TEMP_STEPA_OFFSET  0

#define MAX6639_FAN_CONFIG2b_RPM_STEPB          0xf0
#define MAX6639_FAN_CONFIG2b_RPM_STEPB_OFFSET   4
#define MAX6639_FAN_CONFIG2b_START_STEPB         0x0f
#define MAX6639_FAN_CONFIG2b_START_STEPB_OFFSET  0

#define MAX6639_FAN_CONFIG3_SPINUP_DISABLE           0x80
#define MAX6639_FAN_CONFIG3_SPINUP_DISABLE_OFFSET    7
#define MAX6639_FAN_CONFIG3_THERM_FULL_SPEED	     0x40
#define MAX6639_FAN_CONFIG3_THERM_FULL_SPEED_OFFSET  6
#define MAX6639_FAN_CONFIG3_PULSE_STR_DISABLE        0x20
#define MAX6639_FAN_CONFIG3_PULSE_STR_DISABLE_OFFSET 5
#define MAX6639_FAN_CONFIG3_PWM_FREQ                 0x3
#define MAX6639_FAN_CONFIG3_PWM_FREQ_OFFSET          0

#define MAX6639_FAN_PPR_COUNT              0xc0
#define MAX6639_FAN_PPR_COUNT_OFFSET       6
#define MAX6639_FAN_PPR_TACH_MIN           0x3f
#define MAX6639_FAN_PPR_TACH_MIN_OFFSET    0

#define MAX6639_FAN_TARGET_CNT           0xff
#define MAX6639_FAN_TARGET_CNT_OFFSET    0

#define MAX6639_FAN_CNT           0xff
#define MAX6639_FAN_CNT_OFFSET    0


static const int rpm_ranges[] = { 2000, 4000, 8000, 16000 };

#define FAN_FROM_REG(val, rpm_range)	((val) == 0 || (val) == 255 ? \
				0 : (rpm_ranges[rpm_range] * 30) / (val))
#define TEMP_LIMIT_TO_REG(val)	clamp_val((val) / 1000, 0, 255)

/*
 * Client data (each client gets its own)
 */
struct max6639_data {
	struct i2c_client *client;
	struct mutex update_lock;
	char valid;		/* !=0 if following fields are valid */
	unsigned long last_updated;	/* In jiffies */

	/* Register values sampled regularly */
	u16 temp[2];		/* Temperature, in 1/8 C, 0..255 C */
	bool temp_fault[2];	/* Detected temperature diode failure */
	u8 fan[2];		/* Register value: TACH count for fans >=30 */
	u8 status;		/* Detected channel alarms and fan failures */

	/* Register values only written to */
	u8 pwm[2];		/* Register value: Duty cycle 0..120 */
	u8 temp_therm[2];	/* THERM Temperature, 0..255 C (->_max) */
	u8 temp_alert[2];	/* ALERT Temperature, 0..255 C (->_crit) */
	u8 temp_ot[2];		/* OT Temperature, 0..255 C (->_emergency) */

	/* Register values initialized only once */
	u8 ppr;			/* Pulses per rotation 0..3 for 1..4 ppr */
	u8 rpm_range;		/* Index in above rpm_ranges table */
};

static struct max6639_data *max6639_update_device(struct device *dev)
{
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	struct max6639_data *ret = data;
	int i;
	int status_reg;

	mutex_lock(&data->update_lock);

	if (time_after(jiffies, data->last_updated + 2 * HZ) || !data->valid) {
		int res;

		dev_dbg(&client->dev, "Starting max6639 update\n");

		status_reg = i2c_smbus_read_byte_data(client,
						      MAX6639_REG_STATUS);
		if (status_reg < 0) {
			ret = ERR_PTR(status_reg);
			goto abort;
		}

		data->status = status_reg;

		for (i = 0; i < 2; i++) {
			res = i2c_smbus_read_byte_data(client,
					MAX6639_REG_FAN_CNT(i));
			if (res < 0) {
				ret = ERR_PTR(res);
				goto abort;
			}
			data->fan[i] = res;

			res = i2c_smbus_read_byte_data(client,
					MAX6639_REG_TEMP_EXT(i));
			if (res < 0) {
				ret = ERR_PTR(res);
				goto abort;
			}
			data->temp[i] = res >> 5;
			data->temp_fault[i] = res & 0x01;

			res = i2c_smbus_read_byte_data(client,
					MAX6639_REG_TEMP(i));
			if (res < 0) {
				ret = ERR_PTR(res);
				goto abort;
			}
			data->temp[i] |= res << 3;
		}

		data->last_updated = jiffies;
		data->valid = 1;
	}
abort:
	mutex_unlock(&data->update_lock);

	return ret;
}

static ssize_t show_fan_target(struct device *dev,
			       struct device_attribute *dev_attr, char *buf)
{
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	int conf, tach_count, range;
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);

	mutex_lock(&data->update_lock);
	conf = i2c_smbus_read_byte_data(client,
				MAX6639_REG_FAN_CONFIG1(attr->index));
	tach_count = i2c_smbus_read_byte_data(client,
				MAX6639_REG_TARGET_CNT(attr->index));
	mutex_unlock(&data->update_lock);
	if (conf < 0)
		return conf;
	if (tach_count < 0)
		return tach_count;

	range = (conf & MAX6639_FAN_CONFIG1_RPM_RANGE) >>
			     MAX6639_FAN_CONFIG1_RPM_RANGE_OFFSET;
	return sprintf(buf, "%d\n", FAN_FROM_REG(tach_count, range));
}

static ssize_t set_fan_target(struct device *dev,
					  struct device_attribute *dev_attr,
					  const char *buf, size_t count)
{
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	int conf, range, tach_count, target_speed;
	int res;

	res = kstrtoint(buf, 10, &target_speed);
	if (res < 0)
		return res;

	mutex_lock(&data->update_lock);
	conf = i2c_smbus_read_byte_data(client,
					MAX6639_REG_FAN_CONFIG1(attr->index));
	mutex_unlock(&data->update_lock);
	if (conf < 0)
		return conf;

	range = (conf & MAX6639_FAN_CONFIG1_RPM_RANGE) >>
			    MAX6639_FAN_CONFIG1_RPM_RANGE_OFFSET;
	tach_count = FAN_FROM_REG(target_speed, range);

	mutex_lock(&data->update_lock);
	res = i2c_smbus_write_byte_data(client,
				MAX6639_REG_TARGET_CNT(attr->index),
				tach_count);
	mutex_unlock(&data->update_lock);
	if (res < 0)
		return res;
	return count;
}

static ssize_t show_pwm_enable(struct device *dev,
			       struct device_attribute *dev_attr, char *buf)
{
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	int conf;
	int pwm_mode;
	int temp_ch;
	int mode = 0;
	/*Modes 0: Auto Control, 1: Manual PWM Control, 2: Manual RPM Control */

	mutex_lock(&data->update_lock);
	conf = i2c_smbus_read_byte_data(client,
			   MAX6639_REG_FAN_CONFIG1(attr->index));
	mutex_unlock(&data->update_lock);
	if (conf < 0)
		return conf;

	pwm_mode = (conf & MAX6639_FAN_CONFIG1_PWM_MODE) >>
			       MAX6639_FAN_CONFIG1_PWM_MODE_OFFSET;
	temp_ch = (conf & MAX6639_FAN_CONFIG1_TEMP_CH) >>
			       MAX6639_FAN_CONFIG1_TEMP_CH_OFFSET;

	if (pwm_mode == 1)
		mode = 1;
	else if (temp_ch == 0)
		mode = 2;
	return sprintf(buf, "%d\n", mode);
}

static ssize_t set_pwm_enable(struct device *dev,
			    struct device_attribute *dev_attr,
			    const char *buf, size_t count)
{
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	int mode, pwm_mode, temp_ch;
	int res;
	int val = 0;
	/*Modes 0: Auto Control, 1: Manual PWM Control, 2: Manual RPM Control */
	res = kstrtoint(buf, 10, &mode);
	if (res < 0)
		return res;

	if (mode == 0) {
		pwm_mode = 0;
		temp_ch = 3;
	} else if (mode == 1) {
		pwm_mode = 1;
		temp_ch = 3;
	} else {
		pwm_mode = 0;
		temp_ch = 0;
	}
	val |= ((pwm_mode << MAX6639_FAN_CONFIG1_PWM_MODE_OFFSET) &
		MAX6639_FAN_CONFIG1_PWM_MODE);
	val |= ((temp_ch << MAX6639_FAN_CONFIG1_TEMP_CH_OFFSET) &
		MAX6639_FAN_CONFIG1_TEMP_CH);

	mutex_lock(&data->update_lock);
	res = i2c_smbus_write_byte_data(client,
				MAX6639_REG_FAN_CONFIG1(attr->index), val);
	mutex_unlock(&data->update_lock);
	if (res < 0)
		return res;
	return count;
}

static ssize_t show_value(struct device *dev, char *buf,
			  u8 reg, u8 mask, u8 offset)
{
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	int reg_val;

	if (IS_ERR(data))
		return PTR_ERR(data);
	mutex_lock(&data->update_lock);
	reg_val = i2c_smbus_read_byte_data(client, reg);
	mutex_unlock(&data->update_lock);
	if (reg_val < 0)
		return reg_val;

	reg_val = (reg_val & mask) >> offset;
	return sprintf(buf, "%d\n", reg_val);
}

static ssize_t set_value(struct device *dev, const char *buf, size_t count,
			 u8 reg, u8 mask, u8 offset)
{
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;

	int val;
	int res;
	int reg_val;

	if (IS_ERR(data))
		return PTR_ERR(data);
	res = kstrtoint(buf, 10, &val);
	if (res)
		return res;
	mutex_lock(&data->update_lock);
	reg_val = i2c_smbus_read_byte_data(client, reg);
	mutex_unlock(&data->update_lock);
	if (reg_val < 0)
		return reg_val;

	reg_val &= ~mask;
	val = (val << offset) & mask;
	reg_val |= val;

	mutex_lock(&data->update_lock);
	res = i2c_smbus_write_byte_data(client, reg, val);
	mutex_unlock(&data->update_lock);
	if (res < 0)
		return res;
	return count;
}

#define max6639_attr(type, reg, mask, off)	       \
static ssize_t show_##type(struct device *dev, \
			 struct device_attribute *dev_attr, char *buf) \
{ \
	return  show_value(dev, buf, reg, mask, off); \
} \
static ssize_t set_##type(struct device *dev, \
			  struct device_attribute *dev_attr, \
			  const char *buf, size_t count) \
{ \
	return  set_value(dev, buf, count, reg, mask, off); \
} \
static SENSOR_DEVICE_ATTR(type, S_IWUSR | S_IRUGO, \
			  show_##type, set_##type, 0);  \

static SENSOR_DEVICE_ATTR(pwm1_enable, S_IWUSR | S_IRUGO,
				show_pwm_enable, set_pwm_enable, 0);
static SENSOR_DEVICE_ATTR(pwm2_enable, S_IWUSR | S_IRUGO,
				show_pwm_enable, set_pwm_enable, 1);
static SENSOR_DEVICE_ATTR(fan1_target, S_IWUSR | S_IRUGO,
				show_fan_target, set_fan_target, 0);
static SENSOR_DEVICE_ATTR(fan2_target, S_IWUSR | S_IRUGO,
				show_fan_target, set_fan_target, 1);

max6639_attr(standby, MAX6639_REG_GCONFIG, MAX6639_GCONFIG_STANDBY,
	     MAX6639_GCONFIG_STANDBY_OFFSET)
max6639_attr(por, MAX6639_REG_GCONFIG, MAX6639_GCONFIG_POR,
	     MAX6639_GCONFIG_POR_OFFSET)
max6639_attr(disable_smbus_timeout, MAX6639_REG_GCONFIG,
	     MAX6639_GCONFIG_DISABLE_TIMEOUT,
	     MAX6639_GCONFIG_DISABLE_TIMEOUT_OFFSET)
max6639_attr(ch2_local, MAX6639_REG_GCONFIG, MAX6639_GCONFIG_CH2_LOCAL,
	     MAX6639_GCONFIG_CH2_LOCAL_OFFSET)
max6639_attr(pwm_freq_hi, MAX6639_REG_GCONFIG, MAX6639_GCONFIG_PWM_FREQ_HI,
	     MAX6639_GCONFIG_PWM_FREQ_HI_OFFSET)

max6639_attr(ch1_alert_disable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_CH1_ALERT_DISABLE,
	     MAX6639_OUTMASK_CH1_ALERT_DISABLE)
max6639_attr(ch2_alert_disable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_CH2_ALERT_DISABLE,
	     MAX6639_OUTMASK_CH2_ALERT_DISABLE)
max6639_attr(ch1_ot_disable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_CH1_OT_DISABLE,
	     MAX6639_OUTMASK_CH1_OT_DISABLE)
max6639_attr(ch2_ot_disable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_CH2_OT_DISABLE,
	     MAX6639_OUTMASK_CH2_OT_DISABLE)
max6639_attr(ch1_therm_disable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_CH1_THERM_DISABLE,
	     MAX6639_OUTMASK_CH1_THERM_DISABLE)
max6639_attr(ch2_therm_disable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_CH2_THERM_DISABLE,
	     MAX6639_OUTMASK_CH2_THERM_DISABLE)
max6639_attr(fan1_fault_enable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_FAN1_FAULT_ENABLE,
	     MAX6639_OUTMASK_FAN1_FAULT_ENABLE)
max6639_attr(fan2_fault_enable, MAX6639_REG_OUTPUT_MASK,
	     1 << MAX6639_OUTMASK_FAN2_FAULT_ENABLE,
	     MAX6639_OUTMASK_FAN2_FAULT_ENABLE)


max6639_attr(fan1_pwm_mode, MAX6639_REG_FAN_CONFIG1(0),
	     MAX6639_FAN_CONFIG1_PWM_MODE, MAX6639_FAN_CONFIG1_PWM_MODE_OFFSET)
max6639_attr(fan2_pwm_mode, MAX6639_REG_FAN_CONFIG1(1),
	     MAX6639_FAN_CONFIG1_PWM_MODE, MAX6639_FAN_CONFIG1_PWM_MODE_OFFSET)
max6639_attr(fan1_duty, MAX6639_REG_FAN_CONFIG1(0),
	     MAX6639_FAN_CONFIG1_DUTY, MAX6639_FAN_CONFIG1_DUTY_OFFSET)
max6639_attr(fan2_duty, MAX6639_REG_FAN_CONFIG1(1),
	     MAX6639_FAN_CONFIG1_DUTY, MAX6639_FAN_CONFIG1_DUTY_OFFSET)
max6639_attr(fan1_temp_ch, MAX6639_REG_FAN_CONFIG1(0),
	     MAX6639_FAN_CONFIG1_TEMP_CH, MAX6639_FAN_CONFIG1_TEMP_CH_OFFSET)
max6639_attr(fan2_temp_ch, MAX6639_REG_FAN_CONFIG1(1),
	     MAX6639_FAN_CONFIG1_TEMP_CH, MAX6639_FAN_CONFIG1_TEMP_CH_OFFSET)
max6639_attr(fan1_rpm_range, MAX6639_REG_FAN_CONFIG1(0),
	     MAX6639_FAN_CONFIG1_RPM_RANGE,
	     MAX6639_FAN_CONFIG1_RPM_RANGE_OFFSET)
max6639_attr(fan2_rpm_range, MAX6639_REG_FAN_CONFIG1(1),
	     MAX6639_FAN_CONFIG1_RPM_RANGE,
	     MAX6639_FAN_CONFIG1_RPM_RANGE_OFFSET)

max6639_attr(fan1_rpm_step_size_a, MAX6639_REG_FAN_CONFIG2a(0),
	     MAX6639_FAN_CONFIG2a_RPM_STEPA,
	     MAX6639_FAN_CONFIG2a_RPM_STEPA_OFFSET)
max6639_attr(fan2_rpm_step_size_a, MAX6639_REG_FAN_CONFIG2a(1),
	     MAX6639_FAN_CONFIG2a_RPM_STEPA,
	     MAX6639_FAN_CONFIG2a_RPM_STEPA_OFFSET)
max6639_attr(fan1_temp_step_size, MAX6639_REG_FAN_CONFIG2a(0),
	     MAX6639_FAN_CONFIG2a_TEMP_STEPA,
	     MAX6639_FAN_CONFIG2a_RPM_STEPA_OFFSET)
max6639_attr(fan2_temp_step_size, MAX6639_REG_FAN_CONFIG2a(1),
	     MAX6639_FAN_CONFIG2a_TEMP_STEPA,
	     MAX6639_FAN_CONFIG2a_TEMP_STEPA_OFFSET)

max6639_attr(fan1_rpm_step_size_b, MAX6639_REG_FAN_CONFIG2b(0),
	     MAX6639_FAN_CONFIG2b_RPM_STEPB,
	     MAX6639_FAN_CONFIG2b_RPM_STEPB_OFFSET)
max6639_attr(fan2_rpm_step_size_b, MAX6639_REG_FAN_CONFIG2b(1),
	     MAX6639_FAN_CONFIG2b_RPM_STEPB,
	     MAX6639_FAN_CONFIG2b_RPM_STEPB_OFFSET)
max6639_attr(fan1_rpm_start_step_b, MAX6639_REG_FAN_CONFIG2b(0),
	     MAX6639_FAN_CONFIG2b_START_STEPB,
	     MAX6639_FAN_CONFIG2b_START_STEPB_OFFSET)
max6639_attr(fan2_rpm_start_step_b, MAX6639_REG_FAN_CONFIG2b(1),
	     MAX6639_FAN_CONFIG2b_RPM_STEPB,
	     MAX6639_FAN_CONFIG2b_START_STEPB_OFFSET)

max6639_attr(fan1_spinup_disable, MAX6639_REG_FAN_CONFIG3(0),
	     MAX6639_FAN_CONFIG3_SPINUP_DISABLE,
	     MAX6639_FAN_CONFIG3_SPINUP_DISABLE_OFFSET)
max6639_attr(fan2_spinup_disable, MAX6639_REG_FAN_CONFIG3(1),
	     MAX6639_FAN_CONFIG3_SPINUP_DISABLE,
	     MAX6639_FAN_CONFIG3_SPINUP_DISABLE_OFFSET)
max6639_attr(fan1_therm_full_speed_enable, MAX6639_REG_FAN_CONFIG3(0),
	     MAX6639_FAN_CONFIG3_THERM_FULL_SPEED,
	     MAX6639_FAN_CONFIG3_THERM_FULL_SPEED_OFFSET)
max6639_attr(fan2_therm_full_speed_enable, MAX6639_REG_FAN_CONFIG3(1),
	     MAX6639_FAN_CONFIG3_THERM_FULL_SPEED,
	     MAX6639_FAN_CONFIG3_THERM_FULL_SPEED_OFFSET)
max6639_attr(fan1_pulse_stretch_disable, MAX6639_REG_FAN_CONFIG3(0),
	     MAX6639_FAN_CONFIG3_PULSE_STR_DISABLE,
	     MAX6639_FAN_CONFIG3_PULSE_STR_DISABLE_OFFSET)
max6639_attr(fan2_pulse_stretch_disable, MAX6639_REG_FAN_CONFIG3(1),
	     MAX6639_FAN_CONFIG3_PULSE_STR_DISABLE,
	     MAX6639_FAN_CONFIG3_PULSE_STR_DISABLE_OFFSET)
max6639_attr(fan1_pwm_freq, MAX6639_REG_FAN_CONFIG3(0),
	     MAX6639_FAN_CONFIG3_PWM_FREQ, MAX6639_FAN_CONFIG3_PWM_FREQ_OFFSET)
max6639_attr(fan2_pwm_freq, MAX6639_REG_FAN_CONFIG3(1),
	     MAX6639_FAN_CONFIG3_PWM_FREQ, MAX6639_FAN_CONFIG3_PWM_FREQ_OFFSET)

max6639_attr(fan1_ppr_count, MAX6639_REG_FAN_PPR(0), MAX6639_FAN_PPR_COUNT,
	     MAX6639_FAN_PPR_COUNT_OFFSET)
max6639_attr(fan2_ppr_count, MAX6639_REG_FAN_PPR(1), MAX6639_FAN_PPR_COUNT,
	     MAX6639_FAN_PPR_COUNT_OFFSET)
max6639_attr(fan1_tach_min, MAX6639_REG_FAN_PPR(0), MAX6639_FAN_PPR_TACH_MIN,
	     MAX6639_FAN_PPR_TACH_MIN_OFFSET)
max6639_attr(fan2_tach_min, MAX6639_REG_FAN_PPR(1), MAX6639_FAN_PPR_TACH_MIN,
	     MAX6639_FAN_PPR_TACH_MIN_OFFSET)

max6639_attr(fan1_cnt_target, MAX6639_REG_TARGET_CNT(0), MAX6639_FAN_TARGET_CNT,
	     MAX6639_FAN_TARGET_CNT_OFFSET)
max6639_attr(fan2_cnt_target, MAX6639_REG_TARGET_CNT(1), MAX6639_FAN_TARGET_CNT,
	     MAX6639_FAN_TARGET_CNT_OFFSET)

max6639_attr(fan1_cnt, MAX6639_REG_FAN_CNT(0), MAX6639_FAN_CNT,
	     MAX6639_FAN_CNT_OFFSET)
max6639_attr(fan2_cnt, MAX6639_REG_FAN_CNT(1), MAX6639_FAN_CNT,
	     MAX6639_FAN_CNT_OFFSET)

/*Puneet end */
static ssize_t show_temp_input(struct device *dev,
			       struct device_attribute *dev_attr, char *buf)
{
	long temp;
	struct max6639_data *data = max6639_update_device(dev);
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);

	if (IS_ERR(data))
		return PTR_ERR(data);

	temp = data->temp[attr->index] * 125;
	return sprintf(buf, "%ld\n", temp);
}

static ssize_t show_temp_fault(struct device *dev,
			       struct device_attribute *dev_attr, char *buf)
{
	struct max6639_data *data = max6639_update_device(dev);
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);

	if (IS_ERR(data))
		return PTR_ERR(data);

	return sprintf(buf, "%d\n", data->temp_fault[attr->index]);
}

static ssize_t show_temp_max(struct device *dev,
			     struct device_attribute *dev_attr, char *buf)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%d\n", (data->temp_therm[attr->index] * 1000));
}

static ssize_t set_temp_max(struct device *dev,
			    struct device_attribute *dev_attr,
			    const char *buf, size_t count)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	unsigned long val;
	int res;

	res = kstrtoul(buf, 10, &val);
	if (res)
		return res;

	mutex_lock(&data->update_lock);
	data->temp_therm[attr->index] = TEMP_LIMIT_TO_REG(val);
	i2c_smbus_write_byte_data(client,
				  MAX6639_REG_THERM_LIMIT(attr->index),
				  data->temp_therm[attr->index]);
	mutex_unlock(&data->update_lock);
	return count;
}

static ssize_t show_temp_crit(struct device *dev,
			      struct device_attribute *dev_attr, char *buf)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%d\n", (data->temp_alert[attr->index] * 1000));
}

static ssize_t set_temp_crit(struct device *dev,
			     struct device_attribute *dev_attr,
			     const char *buf, size_t count)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	unsigned long val;
	int res;

	res = kstrtoul(buf, 10, &val);
	if (res)
		return res;

	mutex_lock(&data->update_lock);
	data->temp_alert[attr->index] = TEMP_LIMIT_TO_REG(val);
	i2c_smbus_write_byte_data(client,
				  MAX6639_REG_ALERT_LIMIT(attr->index),
				  data->temp_alert[attr->index]);
	mutex_unlock(&data->update_lock);
	return count;
}

static ssize_t show_temp_emergency(struct device *dev,
				   struct device_attribute *dev_attr,
				   char *buf)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%d\n", (data->temp_ot[attr->index] * 1000));
}

static ssize_t set_temp_emergency(struct device *dev,
				  struct device_attribute *dev_attr,
				  const char *buf, size_t count)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	unsigned long val;
	int res;

	res = kstrtoul(buf, 10, &val);
	if (res)
		return res;

	mutex_lock(&data->update_lock);
	data->temp_ot[attr->index] = TEMP_LIMIT_TO_REG(val);
	i2c_smbus_write_byte_data(client,
				  MAX6639_REG_OT_LIMIT(attr->index),
				  data->temp_ot[attr->index]);
	mutex_unlock(&data->update_lock);
	return count;
}

static ssize_t show_pwm(struct device *dev,
			struct device_attribute *dev_attr, char *buf)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);

	return sprintf(buf, "%d\n", data->pwm[attr->index] * 255 / 120);
}

static ssize_t set_pwm(struct device *dev,
		       struct device_attribute *dev_attr,
		       const char *buf, size_t count)
{
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	struct max6639_data *data = dev_get_drvdata(dev);
	struct i2c_client *client = data->client;
	unsigned long val;
	int res;

	res = kstrtoul(buf, 10, &val);
	if (res)
		return res;

	val = clamp_val(val, 0, 255);

	mutex_lock(&data->update_lock);
	data->pwm[attr->index] = (u8)(val * 120 / 255);
	i2c_smbus_write_byte_data(client,
				  MAX6639_REG_TARGTDUTY(attr->index),
				  data->pwm[attr->index]);
	mutex_unlock(&data->update_lock);
	return count;
}

static ssize_t show_fan_input(struct device *dev,
			      struct device_attribute *dev_attr, char *buf)
{
	struct max6639_data *data = max6639_update_device(dev);
	struct i2c_client *client = data->client;
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);
	int conf, range, input;

	if (IS_ERR(data))
		return PTR_ERR(data);
	mutex_lock(&data->update_lock);
	conf = i2c_smbus_read_byte_data(client,
					MAX6639_REG_FAN_CONFIG1(attr->index));
	input = i2c_smbus_read_byte_data(client,
					MAX6639_REG_FAN_CNT(attr->index));
	mutex_unlock(&data->update_lock);
	if (conf < 0)
		return conf;

	if (input < 0)
		return input;

	range = (conf & MAX6639_FAN_CONFIG1_RPM_RANGE) >>
		MAX6639_FAN_CONFIG1_RPM_RANGE_OFFSET;

	return sprintf(buf, "%d\n", FAN_FROM_REG(input, range));
}

static ssize_t show_alarm(struct device *dev,
			  struct device_attribute *dev_attr, char *buf)
{
	struct max6639_data *data = max6639_update_device(dev);
	struct sensor_device_attribute *attr = to_sensor_dev_attr(dev_attr);

	if (IS_ERR(data))
		return PTR_ERR(data);

	return sprintf(buf, "%d\n", !!(data->status & (1 << attr->index)));
}

static SENSOR_DEVICE_ATTR(temp1_input, S_IRUGO, show_temp_input, NULL, 0);
static SENSOR_DEVICE_ATTR(temp2_input, S_IRUGO, show_temp_input, NULL, 1);
static SENSOR_DEVICE_ATTR(temp1_fault, S_IRUGO, show_temp_fault, NULL, 0);
static SENSOR_DEVICE_ATTR(temp2_fault, S_IRUGO, show_temp_fault, NULL, 1);
static SENSOR_DEVICE_ATTR(temp1_max, S_IWUSR | S_IRUGO, show_temp_max,
		set_temp_max, 0);
static SENSOR_DEVICE_ATTR(temp2_max, S_IWUSR | S_IRUGO, show_temp_max,
		set_temp_max, 1);
static SENSOR_DEVICE_ATTR(temp1_crit, S_IWUSR | S_IRUGO, show_temp_crit,
		set_temp_crit, 0);
static SENSOR_DEVICE_ATTR(temp2_crit, S_IWUSR | S_IRUGO, show_temp_crit,
		set_temp_crit, 1);
static SENSOR_DEVICE_ATTR(temp1_emergency, S_IWUSR | S_IRUGO,
		show_temp_emergency, set_temp_emergency, 0);
static SENSOR_DEVICE_ATTR(temp2_emergency, S_IWUSR | S_IRUGO,
		show_temp_emergency, set_temp_emergency, 1);
static SENSOR_DEVICE_ATTR(pwm1, S_IWUSR | S_IRUGO, show_pwm, set_pwm, 0);
static SENSOR_DEVICE_ATTR(pwm2, S_IWUSR | S_IRUGO, show_pwm, set_pwm, 1);
static SENSOR_DEVICE_ATTR(fan1_input, S_IRUGO, show_fan_input, NULL, 0);
static SENSOR_DEVICE_ATTR(fan2_input, S_IRUGO, show_fan_input, NULL, 1);
static SENSOR_DEVICE_ATTR(fan1_fault, S_IRUGO, show_alarm, NULL, 1);
static SENSOR_DEVICE_ATTR(fan2_fault, S_IRUGO, show_alarm, NULL, 0);
static SENSOR_DEVICE_ATTR(temp1_max_alarm, S_IRUGO, show_alarm, NULL, 3);
static SENSOR_DEVICE_ATTR(temp2_max_alarm, S_IRUGO, show_alarm, NULL, 2);
static SENSOR_DEVICE_ATTR(temp1_crit_alarm, S_IRUGO, show_alarm, NULL, 7);
static SENSOR_DEVICE_ATTR(temp2_crit_alarm, S_IRUGO, show_alarm, NULL, 6);
static SENSOR_DEVICE_ATTR(temp1_emergency_alarm, S_IRUGO, show_alarm, NULL, 5);
static SENSOR_DEVICE_ATTR(temp2_emergency_alarm, S_IRUGO, show_alarm, NULL, 4);


static struct attribute *max6639_attrs[] = {
	&sensor_dev_attr_temp1_input.dev_attr.attr,
	&sensor_dev_attr_temp2_input.dev_attr.attr,
	&sensor_dev_attr_temp1_fault.dev_attr.attr,
	&sensor_dev_attr_temp2_fault.dev_attr.attr,
	&sensor_dev_attr_temp1_max.dev_attr.attr,
	&sensor_dev_attr_temp2_max.dev_attr.attr,
	&sensor_dev_attr_temp1_crit.dev_attr.attr,
	&sensor_dev_attr_temp2_crit.dev_attr.attr,
	&sensor_dev_attr_temp1_emergency.dev_attr.attr,
	&sensor_dev_attr_temp2_emergency.dev_attr.attr,
	&sensor_dev_attr_pwm1.dev_attr.attr,
	&sensor_dev_attr_pwm2.dev_attr.attr,
	&sensor_dev_attr_fan1_input.dev_attr.attr,
	&sensor_dev_attr_fan2_input.dev_attr.attr,
	&sensor_dev_attr_fan1_fault.dev_attr.attr,
	&sensor_dev_attr_fan2_fault.dev_attr.attr,
	&sensor_dev_attr_temp1_max_alarm.dev_attr.attr,
	&sensor_dev_attr_temp2_max_alarm.dev_attr.attr,
	&sensor_dev_attr_temp1_crit_alarm.dev_attr.attr,
	&sensor_dev_attr_temp2_crit_alarm.dev_attr.attr,
	&sensor_dev_attr_temp1_emergency_alarm.dev_attr.attr,
	&sensor_dev_attr_temp2_emergency_alarm.dev_attr.attr,
	&sensor_dev_attr_standby.dev_attr.attr,
	&sensor_dev_attr_por.dev_attr.attr,
	&sensor_dev_attr_disable_smbus_timeout.dev_attr.attr,
	&sensor_dev_attr_ch2_local.dev_attr.attr,
	&sensor_dev_attr_pwm_freq_hi.dev_attr.attr,
	&sensor_dev_attr_ch1_alert_disable.dev_attr.attr,
	&sensor_dev_attr_ch2_alert_disable.dev_attr.attr,
	&sensor_dev_attr_ch1_ot_disable.dev_attr.attr,
	&sensor_dev_attr_ch2_ot_disable.dev_attr.attr,
	&sensor_dev_attr_ch1_therm_disable.dev_attr.attr,
	&sensor_dev_attr_ch2_therm_disable.dev_attr.attr,
	&sensor_dev_attr_fan1_fault_enable.dev_attr.attr,
	&sensor_dev_attr_fan2_fault_enable.dev_attr.attr,
	&sensor_dev_attr_fan1_pwm_mode.dev_attr.attr,
	&sensor_dev_attr_fan2_pwm_mode.dev_attr.attr,
	&sensor_dev_attr_fan1_duty.dev_attr.attr,
	&sensor_dev_attr_fan2_duty.dev_attr.attr,
	&sensor_dev_attr_fan1_temp_ch.dev_attr.attr,
	&sensor_dev_attr_fan2_temp_ch.dev_attr.attr,
	&sensor_dev_attr_fan1_rpm_range.dev_attr.attr,
	&sensor_dev_attr_fan2_rpm_range.dev_attr.attr,
	&sensor_dev_attr_fan1_rpm_step_size_a.dev_attr.attr,
	&sensor_dev_attr_fan2_rpm_step_size_a.dev_attr.attr,
	&sensor_dev_attr_fan1_rpm_step_size_b.dev_attr.attr,
	&sensor_dev_attr_fan2_rpm_step_size_b.dev_attr.attr,
	&sensor_dev_attr_fan1_temp_step_size.dev_attr.attr,
	&sensor_dev_attr_fan2_temp_step_size.dev_attr.attr,
	&sensor_dev_attr_fan1_rpm_start_step_b.dev_attr.attr,
	&sensor_dev_attr_fan2_rpm_start_step_b.dev_attr.attr,
	&sensor_dev_attr_fan1_spinup_disable.dev_attr.attr,
	&sensor_dev_attr_fan2_spinup_disable.dev_attr.attr,
	&sensor_dev_attr_fan1_therm_full_speed_enable.dev_attr.attr,
	&sensor_dev_attr_fan2_therm_full_speed_enable.dev_attr.attr,
	&sensor_dev_attr_fan1_pulse_stretch_disable.dev_attr.attr,
	&sensor_dev_attr_fan2_pulse_stretch_disable.dev_attr.attr,
	&sensor_dev_attr_fan1_pwm_freq.dev_attr.attr,
	&sensor_dev_attr_fan2_pwm_freq.dev_attr.attr,
	&sensor_dev_attr_fan1_ppr_count.dev_attr.attr,
	&sensor_dev_attr_fan2_ppr_count.dev_attr.attr,
	&sensor_dev_attr_fan1_tach_min.dev_attr.attr,
	&sensor_dev_attr_fan2_tach_min.dev_attr.attr,
	&sensor_dev_attr_fan1_cnt_target.dev_attr.attr,
	&sensor_dev_attr_fan2_cnt_target.dev_attr.attr,
	&sensor_dev_attr_fan1_cnt.dev_attr.attr,
	&sensor_dev_attr_fan2_cnt.dev_attr.attr,
	&sensor_dev_attr_pwm1_enable.dev_attr.attr,
	&sensor_dev_attr_pwm2_enable.dev_attr.attr,
	&sensor_dev_attr_fan1_target.dev_attr.attr,
	&sensor_dev_attr_fan2_target.dev_attr.attr,
	NULL
};
ATTRIBUTE_GROUPS(max6639);

/*
 *  returns respective index in rpm_ranges table
 *  1 by default on invalid range
 */
static int rpm_range_to_reg(int range)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(rpm_ranges); i++) {
		if (rpm_ranges[i] == range)
			return i;
	}

	return 1; /* default: 4000 RPM */
}

static int max6639_init_client(struct i2c_client *client,
			       struct max6639_data *data)
{
	struct max6639_platform_data *max6639_info =
		dev_get_platdata(&client->dev);
	int i;
	int rpm_range = 1; /* default: 4000 RPM */
	int err;

	/* Reset chip to default values, see below for GCONFIG setup */
	err = i2c_smbus_write_byte_data(client, MAX6639_REG_GCONFIG,
				  MAX6639_GCONFIG_POR);
	if (err)
		goto exit;

	/* Fans pulse per revolution is 2 by default */
	if (max6639_info && max6639_info->ppr > 0 &&
			max6639_info->ppr < 5)
		data->ppr = max6639_info->ppr;
	else
		data->ppr = 2;
	data->ppr -= 1;

	if (max6639_info)
		rpm_range = rpm_range_to_reg(max6639_info->rpm_range);
	data->rpm_range = rpm_range;

	for (i = 0; i < 2; i++) {

		/* Set Fan pulse per revolution */
		err = i2c_smbus_write_byte_data(client,
				MAX6639_REG_FAN_PPR(i),
				data->ppr << 6);
		if (err)
			goto exit;

		/* Fans config PWM, RPM */
		err = i2c_smbus_write_byte_data(client,
			MAX6639_REG_FAN_CONFIG1(i),
			MAX6639_FAN_CONFIG1_PWM | rpm_range);
		if (err)
			goto exit;

		/* Fans PWM polarity high by default */
		if (max6639_info && max6639_info->pwm_polarity == 0)
			err = i2c_smbus_write_byte_data(client,
				MAX6639_REG_FAN_CONFIG2a(i), 0x00);
		else
			err = i2c_smbus_write_byte_data(client,
				MAX6639_REG_FAN_CONFIG2a(i), 0x02);
		if (err)
			goto exit;

		/*
		 * /THERM full speed enable,
		 * PWM frequency 25kHz, see also GCONFIG below
		 */
		err = i2c_smbus_write_byte_data(client,
			MAX6639_REG_FAN_CONFIG3(i),
			MAX6639_FAN_CONFIG3_THERM_FULL_SPEED | 0x03);
		if (err)
			goto exit;

		/* Max. temp. 80C/90C/100C */
		data->temp_therm[i] = 80;
		data->temp_alert[i] = 90;
		data->temp_ot[i] = 100;
		err = i2c_smbus_write_byte_data(client,
				MAX6639_REG_THERM_LIMIT(i),
				data->temp_therm[i]);
		if (err)
			goto exit;
		err = i2c_smbus_write_byte_data(client,
				MAX6639_REG_ALERT_LIMIT(i),
				data->temp_alert[i]);
		if (err)
			goto exit;
		err = i2c_smbus_write_byte_data(client,
				MAX6639_REG_OT_LIMIT(i), data->temp_ot[i]);
		if (err)
			goto exit;

		/* PWM 120/120 (i.e. 100%) */
		data->pwm[i] = 120;
		err = i2c_smbus_write_byte_data(client,
				MAX6639_REG_TARGTDUTY(i), data->pwm[i]);
		if (err)
			goto exit;
	}
	/* Start monitoring */
	err = i2c_smbus_write_byte_data(client, MAX6639_REG_GCONFIG,
		MAX6639_GCONFIG_DISABLE_TIMEOUT | MAX6639_GCONFIG_CH2_LOCAL |
		MAX6639_GCONFIG_PWM_FREQ_HI);
exit:
	return err;
}

/* Return 0 if detection is successful, -ENODEV otherwise */
static int max6639_detect(struct i2c_client *client,
			  struct i2c_board_info *info)
{
	struct i2c_adapter *adapter = client->adapter;
	int dev_id, manu_id;

	if (!i2c_check_functionality(adapter, I2C_FUNC_SMBUS_BYTE_DATA))
		return -ENODEV;

	/* Actual detection via device and manufacturer ID */
	dev_id = i2c_smbus_read_byte_data(client, MAX6639_REG_DEVID);
	manu_id = i2c_smbus_read_byte_data(client, MAX6639_REG_MANUID);
	if (dev_id != 0x58 || manu_id != 0x4D)
		return -ENODEV;

	strlcpy(info->type, "max6639", I2C_NAME_SIZE);

	return 0;
}

static int max6639_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	struct device *dev = &client->dev;
	struct max6639_data *data;
	struct device *hwmon_dev;
	int err;

	data = devm_kzalloc(dev, sizeof(struct max6639_data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;

	data->client = client;
	mutex_init(&data->update_lock);

	/* Initialize the max6639 chip */
	err = max6639_init_client(client, data);
	if (err < 0)
		return err;

	hwmon_dev = devm_hwmon_device_register_with_groups(dev, client->name,
							   data,
							   max6639_groups);
	return PTR_ERR_OR_ZERO(hwmon_dev);
}

#ifdef CONFIG_PM_SLEEP
static int max6639_suspend(struct device *dev)
{
	struct max6639_data *devdata = dev_get_drvdata(dev);
	struct i2c_client *client = devdata->client;
	int data = i2c_smbus_read_byte_data(client, MAX6639_REG_GCONFIG);
	if (data < 0)
		return data;

	return i2c_smbus_write_byte_data(client,
			MAX6639_REG_GCONFIG, data | MAX6639_GCONFIG_STANDBY);
}

static int max6639_resume(struct device *dev)
{
	struct max6639_data *devdata = dev_get_drvdata(dev);
	struct i2c_client *client = devdata->client;
	int data = i2c_smbus_read_byte_data(client, MAX6639_REG_GCONFIG);
	if (data < 0)
		return data;

	return i2c_smbus_write_byte_data(client,
			MAX6639_REG_GCONFIG, data & ~MAX6639_GCONFIG_STANDBY);
}
#endif /* CONFIG_PM_SLEEP */

static const struct i2c_device_id max6639_id[] = {
	{"max6639", 0},
	{ }
};

MODULE_DEVICE_TABLE(i2c, max6639_id);

static SIMPLE_DEV_PM_OPS(max6639_pm_ops, max6639_suspend, max6639_resume);

static struct i2c_driver max6639_driver = {
	.class = I2C_CLASS_HWMON,
	.driver = {
		   .name = "max6639",
		   .pm = &max6639_pm_ops,
		   },
	.probe = max6639_probe,
	.id_table = max6639_id,
	.detect = max6639_detect,
	.address_list = normal_i2c,
};

module_i2c_driver(max6639_driver);

MODULE_AUTHOR("Roland Stigge <stigge@antcom.de>");
MODULE_DESCRIPTION("max6639 driver");
MODULE_LICENSE("GPL");
