/*********************************************************************
 * Copyright 2016 Cumulus Networks, LLC.  All rights reserved.
 *
 * netlink.c
 *
 * netlink handler for interfaces
 */

#ifdef NETLINK

#include "dhcpd.h"

#include <errno.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>

#define REPLY_BUFFER    8192
#define SOCKET_BUFSIZE (16 * 1024 * 1024)

typedef struct nl_req_s {
	struct nlmsghdr hdr;
	struct rtgenmsg gen;
} nl_req_t;

struct netlink_info {
	OMAPI_OBJECT_PREAMBLE;
	struct timeval cur_tv;
	int state;
	int sock;
};

struct netlink_info *g_nl_info;

omapi_object_type_t *dhcp_type_netlink;

OMAPI_OBJECT_ALLOC (netlink, struct netlink_info, dhcp_type_netlink)

static isc_result_t netlink_socket_init(int);
static void netlink_timer (void *);

isc_result_t
netlink_setup (int state)
{
	isc_result_t status;
	status = omapi_object_type_register (&dhcp_type_netlink,
					     "netlink",
					     0, 0, 0, 0, 0, 0,
					     0, 0, 0, 0, 0,
					     sizeof (struct netlink_info),
					     0, RC_MISC);
	if (status != ISC_R_SUCCESS)
		log_fatal ("Can't register netlink object type: %s",
				isc_result_totext (status));
	else
		status = netlink_socket_init(state);

	return status;
}

static void
_parse_rtattr(struct rtattr *tb[], int max, struct rtattr *rta, int len)
{
	while (RTA_OK(rta, len)) {
		if ((rta->rta_type <= max) && (!tb[rta->rta_type]))
			tb[rta->rta_type] = rta;
		rta = RTA_NEXT(rta,len);
	}
}

#define parse_rtattr_nested(tb, max, rta) \
		_parse_rtattr((tb), (max), RTA_DATA(rta), RTA_PAYLOAD(rta))

static int
_parse_link(struct nlmsghdr *msg)
{
	struct ifinfomsg *ifi;
	struct rtattr *tb[IFLA_MAX+1] = {};
	char *name = NULL;
	int len;
	char *rtm;
	int call_timer = 0;

	ifi = NLMSG_DATA(msg);
	len = msg->nlmsg_len - NLMSG_LENGTH(sizeof(struct ifinfomsg));
	if (msg->nlmsg_type == RTM_NEWLINK) {
		rtm = "RTM_NEWLINK";
	} else {
		rtm = "RTM_DELLINK";
	}

	memset(tb, 0, sizeof(tb));
	_parse_rtattr(tb, IFLA_MAX, IFLA_RTA(ifi), len);

	if (!tb[IFLA_IFNAME]) {
		log_debug("netlink %s interface %d does not have a name - ignore\n",
			  rtm, ifi->ifi_index);
		return -1;
	}

	name = (char *) RTA_DATA (tb[IFLA_IFNAME]);

	if (!(ifi->ifi_flags & IFF_RUNNING)) {
		interface_handle_link_down(name);
	} else {
		call_timer = 1;
		interface_handle_link_up(name);
	}

	if (call_timer && !g_nl_info->cur_tv.tv_sec) {
		gettimeofday(&g_nl_info->cur_tv, NULL);
		g_nl_info->cur_tv.tv_sec += 3;
		add_timeout (&g_nl_info->cur_tv,
			     (void (*)(void *))netlink_timer, 0, 0, 0);
	}

	return 0;
}

void netlink_timer (void *vlp)
{
	discover_interfaces(g_nl_info->state);
	memset(&g_nl_info->cur_tv, 0, sizeof(g_nl_info->cur_tv));
}

isc_result_t nl_recv(omapi_object_t *h)
{
	int end = 0;
	int rc = ISC_R_SUCCESS;
	struct iovec iov;
	char *reply = NULL;
	int reply_len = REPLY_BUFFER;
	int len;
	struct nlmsghdr *msg = NULL;
	struct sockaddr_nl peer;
	struct msghdr rtnl_reply;
	struct netlink_info *nl_info = (struct netlink_info *)h;

	while (!end) {

		memset(&rtnl_reply, 0, sizeof(rtnl_reply));

		if (!reply) {
			if ((reply = malloc(reply_len)) == NULL) {
				log_error("Can't malloc memory for reply buf %d\n",
						reply_len);
				end++;
				continue;
			}
		}

		iov.iov_base = reply;
		iov.iov_len = reply_len;
		rtnl_reply.msg_iov = &iov;
		rtnl_reply.msg_iovlen = 1;
		rtnl_reply.msg_name = &peer;
		rtnl_reply.msg_namelen = sizeof(peer);

		len = recvmsg(nl_info->sock, &rtnl_reply, 0);
		if (len < 0) {
			if (errno == EWOULDBLOCK || errno == EAGAIN) {
				rc = ISC_R_UNEXPECTED;
				goto out;
			}
			else if ((errno == ENOMEM) || (errno == ENOBUFS)) {
				free(reply);
				reply = NULL;
				reply_len *=2;
				continue;
			}
			else {
				log_error("%s: recvmsg error (%s), continuing\n",
					  __FUNCTION__, strerror(errno));
				continue;
			}
		}

		if (len == 0) {
			log_error("%s: recvmsg EOF\n", __FUNCTION__);
			rc = ISC_R_UNEXPECTED;
			goto out;
		}

		for (msg = (struct nlmsghdr *)reply;
		     NLMSG_OK(msg, len);
		     msg = NLMSG_NEXT(msg, len)) {
			switch (msg->nlmsg_type) {
			case NLMSG_DONE:
				end++;
				break;
			case RTM_NEWLINK:
			case RTM_DELLINK:
				if (_parse_link(msg) < 0) {
					rc = ISC_R_FAILURE;
					goto out;
				}
				break;
			} // end switch
		} // end for msg
	} // end while

out:

	if (reply)
		free(reply);
	return rc;
}

int netlink_readsocket (omapi_object_t *h)
{
	struct netlink_info *nl_info = (struct netlink_info *)h;

	if (h -> type != dhcp_type_netlink)
		return -1;
	return nl_info->sock;
}

static isc_result_t netlink_socket_init(int state)
{
	struct sockaddr_nl local;
	struct netlink_info *nl_info = NULL;
	int s;
	int flags = 0;
	int sz = SOCKET_BUFSIZE;
	isc_result_t status;

	/*
	 * prepare netlink socket for kernel/userland communication
	 */
	flags = SOCK_RAW | SOCK_CLOEXEC | SOCK_NONBLOCK;
	s = socket(AF_NETLINK, flags, NETLINK_ROUTE);
	if (s < 0) {
		log_error("Can't open netlink socket(%m)\n");
		return ISC_R_FAILURE;
	}

	memset(&local, 0, sizeof(local));
	local.nl_family = AF_NETLINK;
	local.nl_pid = getpid();
	local.nl_groups = RTMGRP_LINK;

	if (bind(s, (struct sockaddr *) &local, sizeof(local)) < 0) {
		close(s);
		log_error("cannot bind netlink socket (%m)\n");
		return ISC_R_FAILURE;
	}

	setsockopt(s, SOL_SOCKET, SO_RCVBUF, &sz, sizeof (sz));

	status = netlink_allocate(&nl_info, MDL);
	if (status != ISC_R_SUCCESS)
		log_fatal("Error allocating netlink object: %s",
				isc_result_totext(status));

	g_nl_info = nl_info;
	nl_info->sock = s;
	nl_info->state = state;

	/* omapi registrations */
	status = omapi_register_io_object((omapi_object_t *)nl_info,
					  netlink_readsocket,
					  0, nl_recv, 0, 0);

	if (status != ISC_R_SUCCESS)
		log_fatal ("Can't register netlink object type: %s",
			   isc_result_totext (status));

	return status;
}

#endif /* NETLINK */
