/*
 *	"POLICE" target extension for iptables
 *	Copyright © Sebastian Claßen <sebastian.classen [at] freenet.ag>, 2007
 *	Jan Engelhardt <jengelh [at] medozas de>, 2007 - 2010
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License; either
 *	version 2 of the License, or any later version, as published by the
 *	Free Software Foundation.
 */
#include <sys/socket.h>
#include <getopt.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <arpa/inet.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netdb.h>

#include <xtables.h>
#include <linux/netfilter.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter/xt_TRICOLORPOLICE.h>

enum {
    O_SET_COLOR_MODE = 0, /* color-blind or color-aware */
    O_SET_CIR,
    O_SET_CBS,
    O_SET_PIR, /* if set and different from CIR, this is a 2-rate policer */
    O_SET_EBS,
    O_SET_CONFORM_ACTION_DSCP,
    O_SET_EXCEED_ACTION_DSCP,
    O_SET_VIOLATE_ACTION,
    O_SET_VIOLATE_ACTION_DSCP,
};

#define s struct xt_tricolor_police_tginfo
static const struct xt_option_entry tricolor_police_tg_opts[] = {
    {.name = "set-color-mode", .id = O_SET_COLOR_MODE, .type = XTTYPE_STRING,
     .flags = XTOPT_PUT, XTOPT_POINTER(s, color_mode)},
    {.name = "set-cir", .id = O_SET_CIR, .type = XTTYPE_UINT32,
     .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, cir), .min = XT_TRICOLOR_POLICE_MIN_CIR, .max = XT_TRICOLOR_POLICE_MAX_CIR},
    {.name = "set-cbs", .id = O_SET_CBS, .type = XTTYPE_UINT32,
     .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, cbs), .min = XT_TRICOLOR_POLICE_MIN_CBS, .max = XT_TRICOLOR_POLICE_MAX_CBS},
    {.name = "set-pir", .id = O_SET_PIR, .type = XTTYPE_UINT32,
     .flags = XTOPT_PUT, XTOPT_POINTER(s, pir), .min = XT_TRICOLOR_POLICE_MIN_PIR, .max = XT_TRICOLOR_POLICE_MAX_PIR},
    {.name = "set-ebs", .id = O_SET_EBS, .type = XTTYPE_UINT32,
     .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, ebs), .min = XT_TRICOLOR_POLICE_MIN_EBS, .max = XT_TRICOLOR_POLICE_MAX_EBS},
    {.name = "set-conform-action-dscp", .id = O_SET_CONFORM_ACTION_DSCP, .type = XTTYPE_UINT8,
     .flags = XTOPT_PUT, XTOPT_POINTER(s, conform_action.dscp_value)},
    {.name = "set-exceed-action-dscp", .id = O_SET_EXCEED_ACTION_DSCP, .type = XTTYPE_UINT8,
     .flags = XTOPT_PUT, XTOPT_POINTER(s, exceed_action.dscp_value)},
    {.name = "set-violate-action", .id = O_SET_VIOLATE_ACTION, .type = XTTYPE_STRING,
     .flags = XTOPT_PUT, XTOPT_POINTER(s, violate_action.type)},
    {.name = "set-violate-action-dscp", .id = O_SET_VIOLATE_ACTION_DSCP, .type = XTTYPE_UINT8,
     .flags = XTOPT_PUT, XTOPT_POINTER(s, violate_action.dscp_value), .also = O_SET_VIOLATE_ACTION},
    XTOPT_TABLEEND,
};
#undef s

static const struct xt_tricolor_police_action_info {
    unsigned char value;
    const char *name;
} xt_tricolor_police_action_names[] = {
    {XT_TRICOLOR_POLICE_ACTION_PERMIT, "accept"},
    {XT_TRICOLOR_POLICE_ACTION_DENY, "drop"},
    {},
};

static bool tricolor_police_action_print(uint8_t value)
{
    const struct xt_tricolor_police_action_info *symbol;

    for (symbol = xt_tricolor_police_action_names; symbol->name != NULL; ++symbol) {
        if (value == symbol->value) {
            printf("%s", symbol->name);
            return true;
        }
    }

    printf(" Bad action type value %d", value);
    return false;
}

static uint8_t parse_tricolor_police_action_type(const char *typestring)
{
    const struct xt_tricolor_police_action_info *symbol;
    uint8_t value = 0;

    for (symbol = xt_tricolor_police_action_names; symbol->name != NULL; ++symbol) {
        if (!strcasecmp(symbol->name, typestring)) {
            value = symbol->value;
            return value;
        }
    }

    xtables_error(PARAMETER_PROBLEM, "Unknown Police Action type `%s'", typestring);
    return value;
}

static void tricolor_police_tg_help(void)
{
    printf(
"3-COLOR POLICE target options:\n"
"  --set-color-mode  STRING setting the mode in blind or aware\n"
"  --set-cir  INT    setting committed information rate in kbits per second\n"
"  --set-cbs  INT    setting committed burst size in kbyte\n"
"  --set-pir  INT    setting peak information rate in kbits per second\n"
"  --set-ebs  INT    setting excess burst size in kbyte\n"
"  --set-conform-action-dscp INT setting dscp value if the action is accept for conforming packets\n"
"  --set-exceed-action-dscp INT setting dscp value if the action is accept for exceeding packets\n"
"  --set-violate-action  STRING setting the action (accept/drop) for violating packets\n"
"  --set-violate-action-dscp INT setting dscp value if the action is accept for violating packets\n"
"\n");
}

#define XTOPT_MKPTR(cb) \
        ((void *)((char *)(cb)->data + (cb)->entry->ptroff))

static void tricolor_police_tg_init(struct xt_entry_target *target)
{
    struct xt_tricolor_police_tginfo *info = (void *)target->data;

    info->bitmask = 0;
    info->color_mode = XT_TRICOLOR_POLICE_COLOR_MODE_AWARE;
    info->bitmask |= XT_TRICOLOR_POLICE_COLOR_MODE;
    info->cir = 0;
    info->cbs = 0;
    info->pir = 0;
    info->ebs = 0;
    info->conform_action.type = 0;
    info->exceed_action.type = 0;
    info->violate_action.type = 0;
    info->violate_action.type = XT_TRICOLOR_POLICE_ACTION_PERMIT;
    info->bitmask |= XT_TRICOLOR_POLICE_VIOLATE_ACTION;
    info->conform_action.dscp_value = 0;
    info->exceed_action.dscp_value = 0;
    info->violate_action.dscp_value = 0;
}

static void tricolor_police_tg_print(const void *ip, const struct xt_entry_target *target,
                         int numeric)
{
    const struct xt_tricolor_police_tginfo *info = (const void *)target->data;

    printf(" POLICE ");
    if (info->bitmask & XT_TRICOLOR_POLICE_COLOR_MODE) {
        printf(" color-mode:%s", (info->color_mode == XT_TRICOLOR_POLICE_COLOR_MODE_BLIND)?
               "blind":"aware");
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_CIR) {
        printf(" cir:%d", info->cir);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_CBS) {
        printf(" cbs:%d", info->cbs);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_PIR) {
        printf(" pir:%d", info->pir);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_EBS) {
        printf(" ebs:%d", info->ebs);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_VIOLATE_ACTION) {
        printf(" violate-action:");
        tricolor_police_action_print(info->violate_action.type);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_CONFORM_ACTION_DSCP) {
        printf(" conform-action-dscp:%d", info->conform_action.dscp_value);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_EXCEED_ACTION_DSCP) {
        printf(" exceed-action-dscp:%d", info->exceed_action.dscp_value);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_VIOLATE_ACTION_DSCP) {
        printf(" violate-action-dscp:%d", info->violate_action.dscp_value);
    }
}

static void tricolor_police_tg_save(const void *ip, const struct xt_entry_target *target)
{
    const struct xt_tricolor_police_tginfo *info = (const void *)target->data;

    if (info->bitmask & XT_TRICOLOR_POLICE_COLOR_MODE) {
        printf(" --set-color-mode %s",
               (info->color_mode == XT_TRICOLOR_POLICE_COLOR_MODE_BLIND)?  "blind":"aware");
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_CIR) {
        printf(" --set-cir %d", info->cir);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_CBS) {
        printf(" --set-cbs %d", info->cbs);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_PIR) {
        printf(" --set-pir %d", info->pir);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_EBS) {
        printf(" --set-ebs %d", info->ebs);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_VIOLATE_ACTION) {
        printf(" --set-violate-action ");
        tricolor_police_action_print(info->violate_action.type);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_CONFORM_ACTION_DSCP) {
        printf(" --set-conform-action-dscp %d", info->conform_action.dscp_value);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_EXCEED_ACTION_DSCP) {
        printf(" --set-exceed-action-dscp %d", info->exceed_action.dscp_value);
    }
    if (info->bitmask & XT_TRICOLOR_POLICE_VIOLATE_ACTION_DSCP) {
        printf(" --set-violate-action-dscp %d", info->violate_action.dscp_value);
    }

}

static void xtables_tricolor_police_option_parse(struct xt_option_call *cb)
{
    struct xt_tricolor_police_tginfo *info = cb->data;

    xtables_option_parse(cb);

    switch (cb->entry->id) {
    case O_SET_COLOR_MODE:
        if (strncmp(cb->arg, "blind", strlen(cb->arg)) == 0)
            info->color_mode = XT_TRICOLOR_POLICE_COLOR_MODE_BLIND;
        else if (strncmp(cb->arg, "aware", strlen(cb->arg)) == 0)
            info->color_mode = XT_TRICOLOR_POLICE_COLOR_MODE_AWARE;
        else
            xtables_error(PARAMETER_PROBLEM,
                          "unsupported mode %s", cb->arg);
        info->bitmask |= XT_TRICOLOR_POLICE_COLOR_MODE;
        break;
    case O_SET_CIR:
        info->bitmask |= XT_TRICOLOR_POLICE_CIR;
        break;
    case O_SET_CBS:
        info->bitmask |= XT_TRICOLOR_POLICE_CBS;
        break;
    case O_SET_PIR:
        info->bitmask |= XT_TRICOLOR_POLICE_PIR;
        break;
    case O_SET_EBS:
        info->bitmask |= XT_TRICOLOR_POLICE_EBS;
        break;

    case O_SET_VIOLATE_ACTION:
        info->violate_action.type = parse_tricolor_police_action_type(cb->arg);
        info->bitmask |= XT_TRICOLOR_POLICE_VIOLATE_ACTION;
        break;
    case O_SET_CONFORM_ACTION_DSCP:
        info->bitmask |= XT_TRICOLOR_POLICE_CONFORM_ACTION_DSCP;
        break;
    case O_SET_EXCEED_ACTION_DSCP:
        info->bitmask |= XT_TRICOLOR_POLICE_EXCEED_ACTION_DSCP;
        break;
    case O_SET_VIOLATE_ACTION_DSCP:
        info->bitmask |= XT_TRICOLOR_POLICE_VIOLATE_ACTION_DSCP;
        break;
    }
}

static struct xtables_target tricolor_police_tg_reg[] = {
    {
        .name          = "TRICOLORPOLICE",
        .version       = XTABLES_VERSION,
        .revision      = 1,
        .family        = NFPROTO_UNSPEC,
        .size          = XT_ALIGN(sizeof(struct xt_tricolor_police_tginfo)),
        .userspacesize = XT_ALIGN(sizeof(struct xt_tricolor_police_tginfo)),
        .help          = tricolor_police_tg_help,
        .init          = tricolor_police_tg_init,
        .print         = tricolor_police_tg_print,
        .save          = tricolor_police_tg_save,
        .x6_parse      = xtables_tricolor_police_option_parse,
        .x6_options    = tricolor_police_tg_opts,
    },
};

void _init(void)
{
    xtables_register_targets(tricolor_police_tg_reg, ARRAY_SIZE(tricolor_police_tg_reg));
}
