/*
 *	"SPAN" target extension for iptables
 *	Copyright © Sebastian Claßen <sebastian.classen [at] freenet.ag>, 2007
 *	Jan Engelhardt <jengelh [at] medozas de>, 2007 - 2010
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License; either
 *	version 2 of the License, or any later version, as published by the
 *	Free Software Foundation.
 */
#include <sys/socket.h>
#include <getopt.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <arpa/inet.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netdb.h>

#include <xtables.h>
#include <linux/netfilter.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter/xt_SPAN.h>

enum {
	O_OIF = 0,
};

#define s struct xt_span_tginfo
static const struct xt_option_entry span_tg_opts[] = {
	{.name = "dport", .id = O_OIF, .type = XTTYPE_STRING,
	 .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, dport)},
	XTOPT_TABLEEND,
};
#undef s

static void span_tg_help(void)
{
	printf(
"SPAN target options:\n"
"  --dport INT         SPAN output interface\n"
"\n");
}

#define XTOPT_MKPTR(cb) \
        ((void *)((char *)(cb)->data + (cb)->entry->ptroff))

static void span_tg_print(const void *ip, const struct xt_entry_target *target,
                         int numeric)
{
	const struct xt_span_tginfo *info = (const void *)target->data;

        if (info->dport != 0) {
            printf(" dport:%s", info->dport);
        }
        printf("\n");
}

static void span_tg_save(const void *ip, const struct xt_entry_target *target)
{
	const struct xt_span_tginfo *info = (const void *)target->data;

	if (info->dport != 0)
		printf(" --dport %s", info->dport);
	printf("\n");
}

static void xtables_span_option_parse(struct xt_option_call *cb)
{
	const struct xt_option_entry *entry = cb->entry;

	xtables_option_parse(cb);
}

static struct xtables_target span_tg_reg[] = {
	{
		.name          = "SPAN",
		.version       = XTABLES_VERSION,
		.revision      = 1,
		.family        = NFPROTO_UNSPEC,
		.size          = XT_ALIGN(sizeof(struct xt_span_tginfo)),
		.userspacesize = XT_ALIGN(sizeof(struct xt_span_tginfo)),
		.help          = span_tg_help,
		.print         = span_tg_print,
		.save          = span_tg_save,
		.x6_parse      = xtables_span_option_parse,
		.x6_options    = span_tg_opts,
	},
};

void _init(void)
{
	xtables_register_targets(span_tg_reg, ARRAY_SIZE(span_tg_reg));
}
