#include <stdio.h>
#include <xtables.h>
#include <linux/netfilter/xt_SETQOS.h>
#include <linux/pkt_sched.h>

#include "dscp_helper.c"

enum {
	O_SET_COS = 0,
	O_SET_DSCP,
	O_SET_DSCP_CLASS,
	F_SET_COS			= 1 << O_SET_COS,
	F_SET_DSCP			= 1 << O_SET_DSCP,
	F_SET_DSCP_CLASS	= 1 << O_SET_DSCP_CLASS,
};

static void
SETQOS_help(void)
{
	printf(
"setqos target options:\n"
"  --set-cos INT   Set datapath resource/queuing class value\n"
"  --set-dscp value		Set DSCP field in packet header to value\n"
"  		                This value can be in decimal (ex: 32)\n"
"               		or in hex (ex: 0x20)\n"
"  --set-dscp-class class	Set the DSCP field in packet header to the\n"
"				value represented by the DiffServ class value.\n"
"				This class may be EF,BE or any of the CSxx\n"
"				or AFxx classes.\n"
"\n"
"				--set-dscp & --set-dscp-class options are mutually exclusive !\n");
}

#define s struct xt_setqos_target_info
static const struct xt_option_entry SETQOS_opts[] = {
	{.name = "set-cos", .id = O_SET_COS, .type = XTTYPE_UINT8,
	 .flags = XTOPT_PUT, XTOPT_POINTER(s, cos), .min = XT_SETQOS_MIN_COS, . max = XT_SETQOS_MAX_COS},
	{.name = "set-dscp", .id = O_SET_DSCP, .excl = F_SET_DSCP_CLASS,
	 .type = XTTYPE_UINT8, .min = 0, .max = XT_SETQOS_DSCP_MAX,
	 .flags = XTOPT_PUT, XTOPT_POINTER(s, dscp)},
	{.name = "set-dscp-class", .id = O_SET_DSCP_CLASS, .excl = F_SET_DSCP,
	 .type = XTTYPE_STRING},
	XTOPT_TABLEEND,
};
#undef s

static void SETQOS_init(struct xt_entry_target *target)
{
	struct xt_setqos_target_info *info = (void *)target->data;

	info->cos = 0;
	info->dscp = 0;
	info->bitmask = 0;
}

static void SETQOS_parse(struct xt_option_call *cb)
{
	struct xt_setqos_target_info *info = cb->data;

	xtables_option_parse(cb);

	switch (cb->entry->id) {
	case O_SET_COS:
		info->bitmask |= XT_SETQOS_COS;
		break;
	case O_SET_DSCP:
		info->bitmask |= XT_SETQOS_DSCP;
		break;
	case O_SET_DSCP_CLASS:
		info->dscp = class_to_dscp(cb->arg);
		info->bitmask |= XT_SETQOS_DSCP;
		break;
	}
}

static void
SETQOS_print(const void *ip,
      const struct xt_entry_target *target,
      int numeric)
{
	const struct xt_setqos_target_info *info =
		(const struct xt_setqos_target_info *)target->data;

	printf(" SETQOS ");
	if (info->bitmask & XT_SETQOS_COS) {
		printf(" cos:%d", info->cos);
	}
	if (info->bitmask & XT_SETQOS_DSCP) {
		printf(" dscp:%d", info->dscp);
	}

}

static void
SETQOS_save(const void *ip, const struct xt_entry_target *target)
{
	const struct xt_setqos_target_info *info =
		(const struct xt_setqos_target_info *)target->data;

	if (info->bitmask & XT_SETQOS_COS) {
		printf(" --set-cos %d", info->cos);
	}

	if (info->bitmask & XT_SETQOS_DSCP) {
		printf(" --set-dscp %d", info->dscp);
	}

}

static struct xtables_target setqos_target = {
	.family		= NFPROTO_UNSPEC,
	.name		= "SETQOS",
	.version	= XTABLES_VERSION,
	.revision	= 1,
	.size		= XT_ALIGN(sizeof(struct xt_setqos_target_info)),
	.userspacesize	= XT_ALIGN(sizeof(struct xt_setqos_target_info)),
	.help		= SETQOS_help,
	.init		= SETQOS_init,
	.print		= SETQOS_print,
	.save		= SETQOS_save,
	.x6_parse	= SETQOS_parse,
	.x6_options	= SETQOS_opts,
};

void _init(void)
{
	xtables_register_target(&setqos_target);
}
