#include <stdio.h>
#include <xtables.h>
#include <linux/netfilter/xt_SETCLASS.h>
#include <linux/pkt_sched.h>

enum {
	O_SET_CLASS = 0,
};

static void
SETCLASS_help(void)
{
	printf(
"SETCLASS target options:\n"
"--class class    Set datapath resource/queuing class value\n");
}

static const struct xt_option_entry SETCLASS_opts[] = {
	{.name = "class", .id = O_SET_CLASS, .type = XTTYPE_STRING,
	 .flags = XTOPT_MAND},
	XTOPT_TABLEEND,
};

static int SETCLASS_string_to_priority(const char *s, unsigned int *p)
{
	unsigned int i;

	if (sscanf(s, "%d", &i) != 1)
		return 1;

	*p = i;
	return 0;
}

static void SETCLASS_parse(struct xt_option_call *cb)
{
	struct xt_setclass_target_info *clinfo = cb->data;

	xtables_option_parse(cb);
	if (SETCLASS_string_to_priority(cb->arg, &clinfo->priority))
		xtables_error(PARAMETER_PROBLEM,
			   "Bad class value \"%s\"", cb->arg);
}

static void
SETCLASS_print_class(unsigned int priority, int numeric)
{
	printf(" class:%d", priority);
}

static void
SETCLASS_print(const void *ip,
      const struct xt_entry_target *target,
      int numeric)
{
	const struct xt_setclass_target_info *clinfo =
		(const struct xt_setclass_target_info *)target->data;
	printf(" SETCLASS ");
	SETCLASS_print_class(clinfo->priority, numeric);
}

static void
SETCLASS_save(const void *ip, const struct xt_entry_target *target)
{
	const struct xt_setclass_target_info *clinfo =
		(const struct xt_setclass_target_info *)target->data;

	printf(" --class %d", clinfo->priority);
}

static struct xtables_target setclass_target = {
	.family		= NFPROTO_UNSPEC,
	.name		= "SETCLASS",
	.version	= XTABLES_VERSION,
	.size		= XT_ALIGN(sizeof(struct xt_setclass_target_info)),
	.userspacesize	= XT_ALIGN(sizeof(struct xt_setclass_target_info)),
	.help		= SETCLASS_help,
	.print		= SETCLASS_print,
	.save		= SETCLASS_save,
	.x6_parse	= SETCLASS_parse,
	.x6_options	= SETCLASS_opts,
};

void _init(void)
{
	xtables_register_target(&setclass_target);
}
