/*
 *	"POLICE" target extension for iptables
 *	Copyright © Sebastian Claßen <sebastian.classen [at] freenet.ag>, 2007
 *	Jan Engelhardt <jengelh [at] medozas de>, 2007 - 2010
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License; either
 *	version 2 of the License, or any later version, as published by the
 *	Free Software Foundation.
 */
#include <sys/socket.h>
#include <getopt.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <arpa/inet.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netdb.h>

#include <xtables.h>
#include <linux/netfilter.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter/xt_POLICE.h>

enum {
	O_SET_CLASS = 0,
	O_SET_RATE,
	O_SET_BURST,
	O_SET_MODE,
	O_SET_VIOLATE_ACTION,
};

#define s struct xt_police_tginfo
static const struct xt_option_entry police_tg_opts[] = {
	{.name = "set-class", .id = O_SET_CLASS, .type = XTTYPE_UINT8,
	 .flags = XTOPT_PUT, XTOPT_POINTER(s, class), .min = XT_POLICE_MIN_CLASS, .max = XT_POLICE_MAX_CLASS},
	{.name = "set-rate", .id = O_SET_RATE, .type = XTTYPE_UINT32,
	 .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, rate), .min = XT_POLICE_MIN_RATE, .max = XT_POLICE_MAX_RATE},
	{.name = "set-burst", .id = O_SET_BURST, .type = XTTYPE_UINT32,
	 .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, burst), .min = XT_POLICE_MIN_BURST, .max = XT_POLICE_MAX_BURST},
	{.name = "set-mode", .id = O_SET_MODE, .type = XTTYPE_STRING,
	 .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, mode)},
	{.name = "set-violate-action", .id = O_SET_VIOLATE_ACTION, .type = XTTYPE_STRING,
	 .flags = XTOPT_PUT, XTOPT_POINTER(s, violate_action.type)},
	XTOPT_TABLEEND,
};
#undef s

static const struct xt_police_action_info {
		unsigned char value;
		const char *name;
	} xt_police_action_names[] = {
		{XT_POLICE_ACTION_PERMIT, "accept"},
		{XT_POLICE_ACTION_DENY, "drop"},
		{},
};

static bool police_action_print(uint8_t value)
{
	const struct xt_police_action_info *symbol;

	for (symbol = xt_police_action_names; symbol->name != NULL; ++symbol) {
		if (value == symbol->value) {
			printf("%s", symbol->name);
			return true;
		}
	}

	printf(" Bad action type value %d", value);
	return false;
}

static uint8_t parse_police_action_type(const char *typestring)
{
	const struct xt_police_action_info *symbol;
	uint8_t value = 0;

	for (symbol = xt_police_action_names; symbol->name != NULL; ++symbol) {
		if (!strcasecmp(symbol->name, typestring)) {
			value = symbol->value;
			return value;
        }
	}

	xtables_error(PARAMETER_PROBLEM, "Unknown Police Action type `%s'", typestring);
	return value;
}

static void police_tg_help(void)
{
	printf(
"POLICE target options:\n"
"  --set-class INT    setting data path resource/queuing class\n"
"  --set-rate  INT    setting bit rate that traffic is limited to\n"
"  --set-burst INT    setting number of bytes allowed to arrive sequentially\n"
"  --set-mode  STRING setting the mode in KB (kiloBytes) or pkt (packets)\n"
"  --set-violate-action  STRING setting the action (accept/drop/dscp-accept) for violating packets\n"
"\n");
}

#define XTOPT_MKPTR(cb) \
        ((void *)((char *)(cb)->data + (cb)->entry->ptroff))

static void police_tg_init(struct xt_entry_target *target)
{
	struct xt_police_tginfo *info = (void *)target->data;

	info->class = 0;
	info->rate  = 0;
	info->burst = 0;
	info->mode = XT_POLICE_MODE_PPS;
	info->bitmask = 0;
	info->violate_action.type = 0;
}

static void police_tg_print(const void *ip, const struct xt_entry_target *target,
                         int numeric)
{
	const struct xt_police_tginfo *info = (const void *)target->data;

	printf(" POLICE ");
	if (info->bitmask & XT_POLICE_MODE) {
		if (info->mode == XT_POLICE_MODE_PPS)
			printf(" mode:%s", "pkt");
		else if (info->mode == XT_POLICE_MODE_KB)
			printf(" mode:%s", "KB");
		else if (info->mode == XT_POLICE_MODE_MB)
			printf(" mode:%s", "MB");
		else
			printf(" mode:%s", "GB");
	}
	if (info->bitmask & XT_POLICE_RATE) {
		printf(" rate:%d", info->rate);
	}
	if (info->bitmask & XT_POLICE_BURST) {
		printf(" burst:%d", info->burst);
	}
	if (info->bitmask & XT_POLICE_CLASS) {
		printf(" class:%d", info->class);
	}
	if (info->bitmask & XT_POLICE_VIOLATE_ACTION) {
		printf(" violate-action:");
		police_action_print(info->violate_action.type);
	}
}

static void police_tg_save(const void *ip, const struct xt_entry_target *target)
{
	const struct xt_police_tginfo *info = (const void *)target->data;

	if (info->bitmask & XT_POLICE_MODE) {
		if (info->mode == XT_POLICE_MODE_PPS)
			printf(" --set-mode %s", "pkt");
		else if (info->mode == XT_POLICE_MODE_KB)
			printf(" --set-mode %s", "KB");
		else if (info->mode == XT_POLICE_MODE_MB)
			printf(" --set-mode %s", "MB");
		else
			printf(" --set-mode %s", "GB");
	}
	if (info->bitmask & XT_POLICE_RATE) {
		printf(" --set-rate %d", info->rate);
	}
	if (info->bitmask & XT_POLICE_BURST) {
		printf(" --set-burst %d", info->burst);
	}
	if (info->bitmask & XT_POLICE_CLASS) {
		printf(" --set-class %d", info->class);
	}
	if (info->bitmask & XT_POLICE_VIOLATE_ACTION) {
		printf(" --set-violate-action ");
		police_action_print(info->violate_action.type);
	}
}

static void xtables_police_option_parse(struct xt_option_call *cb)
{
	struct xt_police_tginfo *info = cb->data;

	xtables_option_parse(cb);

	switch (cb->entry->id) {
	case O_SET_CLASS:
		info->bitmask |= XT_POLICE_CLASS;
		break;
	case O_SET_RATE:
		info->bitmask |= XT_POLICE_RATE;
		break;
	case O_SET_BURST:
		info->bitmask |= XT_POLICE_BURST;
		break;
	case O_SET_MODE:
		if (strncmp(cb->arg, "pkt", strlen(cb->arg)) == 0)
			info->mode = XT_POLICE_MODE_PPS;
		else if (strncmp(cb->arg, "KB", strlen(cb->arg)) == 0)
			info->mode = XT_POLICE_MODE_KB;
		else if (strncmp(cb->arg, "MB", strlen(cb->arg)) == 0)
			info->mode = XT_POLICE_MODE_MB;
		else if (strncmp(cb->arg, "GB", strlen(cb->arg)) == 0)
			info->mode = XT_POLICE_MODE_GB;
		else
			xtables_error(PARAMETER_PROBLEM,
				"unsupported mode %s", cb->arg);

		info->bitmask |= XT_POLICE_MODE;
		break;
	case O_SET_VIOLATE_ACTION:
		info->violate_action.type = parse_police_action_type(cb->arg);
		info->bitmask |=XT_POLICE_VIOLATE_ACTION;
		break;
	}
}

static struct xtables_target police_tg_reg[] = {
	{
		.name          = "POLICE",
		.version       = XTABLES_VERSION,
		.revision      = 1,
		.family        = NFPROTO_UNSPEC,
		.size          = XT_ALIGN(sizeof(struct xt_police_tginfo)),
		.userspacesize = XT_ALIGN(sizeof(struct xt_police_tginfo)),
		.help          = police_tg_help,
		.init          = police_tg_init,
		.print         = police_tg_print,
		.save          = police_tg_save,
		.x6_parse      = xtables_police_option_parse,
		.x6_options    = police_tg_opts,
	},
};

void _init(void)
{
	xtables_register_targets(police_tg_reg, ARRAY_SIZE(police_tg_reg));
}
