/*
 *	"ERSPAN" target extension for iptables
 *	Copyright © Sebastian Claßen <sebastian.classen [at] freenet.ag>, 2007
 *	Jan Engelhardt <jengelh [at] medozas de>, 2007 - 2010
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License; either
 *	version 2 of the License, or any later version, as published by the
 *	Free Software Foundation.
 */
#include <sys/socket.h>
#include <getopt.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <arpa/inet.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netdb.h>

#include <xtables.h>
#include <linux/netfilter.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter/xt_ERSPAN.h>

enum {
	O_SRC_IP = 0,
	O_DST_IP,
	O_TTL,
	O_PREC,
};

#define s struct xt_erspan_tginfo
static const struct xt_option_entry erspan_tg_opts[] = {
	{.name = "src-ip", .id = O_SRC_IP, .type = XTTYPE_HOST,
	 .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, src_ip)},
	{.name = "dst-ip", .id = O_DST_IP, .type = XTTYPE_HOST,
	 .flags = XTOPT_MAND | XTOPT_PUT, XTOPT_POINTER(s, dst_ip)},
	{.name = "ttl", .id = O_TTL, .type = XTTYPE_UINT8,
	 .flags = XTOPT_PUT, XTOPT_POINTER(s, ttl)},
	{.name = "prec", .id = O_PREC, .type = XTTYPE_UINT8,
	 .flags = XTOPT_PUT, XTOPT_POINTER(s, prec)},
	XTOPT_TABLEEND,
};
#undef s

static void erspan_tg_help(void)
{
	printf(
"ERSPAN target options:\n"
"  --src-ip IPADDR    ERSPAN tunnel source ip address\n"
"  --dst-ip IPADDR    ERSPAN tunnel destination ip address\n"
"  --ttl INT          optional ip ttl value for ERSPAN tunnel header\n"
"  --prec INT         optional ip precedence value for ERSPAN tunnel header\n"
"\n");
}

#define XTOPT_MKPTR(cb) \
        ((void *)((char *)(cb)->data + (cb)->entry->ptroff))

static void erspan_tg_print(const void *ip, const struct xt_entry_target *target,
                         int numeric)
{
	struct xt_erspan_tginfo *info = (void *)target->data;
	char *sa, *da;

	if (numeric) {
		sa = strdup(xtables_ipaddr_to_numeric(&info->src_ip.in));
		da = strdup(xtables_ipaddr_to_numeric(&info->dst_ip.in));
		printf(" ERSPAN src-ip:%s dst-ip:%s",
				sa, da);
		free(sa);
		free(da);
	} else {
		sa = strdup(xtables_ipaddr_to_anyname(&info->src_ip.in));
		da = strdup(xtables_ipaddr_to_anyname(&info->dst_ip.in));
		printf(" ERSPAN src-ip:%s dst-ip:%s",
		                sa, da);
		free(sa);
		free(da);
	}

	if (info->ttl != 0) {
		printf(" ttl:%d", info->ttl);
	}
	if (info->prec != 0) {
		printf(" prec:%d", info->prec);
	}
	printf("\n");
}

static void erspan_tg_save(const void *ip, const struct xt_entry_target *target)
{
	const struct xt_erspan_tginfo *info = (const void *)target->data;

	printf(" --src-ip %s", xtables_ipaddr_to_numeric(&info->src_ip.in));
	printf(" --dst-ip %s", xtables_ipaddr_to_numeric(&info->dst_ip.in));
	if (info->ttl != 0)
		printf(" --ttl %d", info->ttl);
	if (info->prec != 0)
		printf(" --prec %d", info->prec);
	printf("\n");
}

static void xtables_erspan_parse_v4_host(struct xt_option_call *cb)
{
	const struct xt_option_entry *entry = cb->entry;
	unsigned int eflag = 1 << cb->entry->id;
        struct addrinfo hints = {.ai_family = NFPROTO_IPV4};
        unsigned int adcount = 0;
        struct addrinfo *res, *p;
        int ret;

	if ((!(entry->flags & XTOPT_MULTI) || (entry->excl & eflag)) &&
            cb->xflags & eflag)
                xt_params->exit_err(PARAMETER_PROBLEM,
                        "%s: option \"--%s\" can only be used once.\n",
                        cb->ext_name, cb->entry->name);
        if (cb->invert && !(entry->flags & XTOPT_INVERT))
                xt_params->exit_err(PARAMETER_PROBLEM,
                        "%s: option \"--%s\" cannot be inverted.\n",
                        cb->ext_name, entry->name);
        if (entry->type != XTTYPE_NONE && optarg == NULL)
                xt_params->exit_err(PARAMETER_PROBLEM,
                        "%s: option \"--%s\" requires an argument.\n",
                        cb->ext_name, entry->name);

	cb->nvals = 1;
        ret = getaddrinfo(cb->arg, NULL, &hints, &res);
        if (ret < 0)
                xt_params->exit_err(PARAMETER_PROBLEM,
                        "getaddrinfo: %s\n", gai_strerror(ret));

        memset(&cb->val.hmask, 0xFF, sizeof(cb->val.hmask));
        cb->val.hlen = 32;

        for (p = res; p != NULL; p = p->ai_next) {
                if (adcount == 0) {
                        memset(&cb->val.haddr, 0, sizeof(cb->val.haddr));
                        memcpy(&cb->val.haddr,
                               &((struct sockaddr_in *)p->ai_addr)->sin_addr,
                               sizeof(struct in_addr));
                        ++adcount;
                        continue;
                }
                if (memcmp(&cb->val.haddr,
                    &((struct sockaddr_in *)p->ai_addr)->sin_addr,
                    sizeof(struct in_addr) != 0))
                        xt_params->exit_err(PARAMETER_PROBLEM,
                                "%s resolves to more than one address\n",
                                cb->arg);
        }

        freeaddrinfo(res);
        if (cb->entry->flags & XTOPT_PUT)
                /* Validation in xtables_option_metavalidate */
                memcpy(XTOPT_MKPTR(cb), &cb->val.haddr,
                       sizeof(cb->val.haddr));
	cb->xflags |= 1 << cb->entry->id;
}

static void xtables_erspan_option_parse(struct xt_option_call *cb)
{
	const struct xt_option_entry *entry = cb->entry;

	if (entry->type != XTTYPE_HOST)
		xtables_option_parse(cb);
	else {
		xtables_erspan_parse_v4_host(cb);
	}
}

static struct xtables_target erspan_tg_reg[] = {
	{
		.name          = "ERSPAN",
		.version       = XTABLES_VERSION,
		.revision      = 1,
		.family        = NFPROTO_IPV4,
		.size          = XT_ALIGN(sizeof(struct xt_erspan_tginfo)),
		.userspacesize = XT_ALIGN(sizeof(struct xt_erspan_tginfo)),
		.help          = erspan_tg_help,
		.print         = erspan_tg_print,
		.save          = erspan_tg_save,
		.x6_parse      = xtables_erspan_option_parse,
		.x6_options    = erspan_tg_opts,
	},
	{
		.name          = "ERSPAN",
		.version       = XTABLES_VERSION,
		.revision      = 1,
		.family        = NFPROTO_IPV6,
		.size          = XT_ALIGN(sizeof(struct xt_erspan_tginfo)),
		.userspacesize = XT_ALIGN(sizeof(struct xt_erspan_tginfo)),
		.help          = erspan_tg_help,
		.print         = erspan_tg_print,
		.save          = erspan_tg_save,
		.x6_parse      = xtables_erspan_option_parse,
		.x6_options    = erspan_tg_opts,
	},
};

void _init(void)
{
	xtables_register_targets(erspan_tg_reg, ARRAY_SIZE(erspan_tg_reg));
}
