/*
 * Zebra Layer-2 interface handling code
 * Copyright (C) 2016, 2017 Cumulus Networks, Inc.
 *
 * This file is part of FRR.
 *
 * FRR is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * FRR is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with FRR; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <zebra.h>

#include "if.h"
#include "prefix.h"
#include "table.h"
#include "memory.h"
#include "log.h"
#include "linklist.h"
#include "stream.h"
#include "hash.h"
#include "jhash.h"

#include "zebra/rib.h"
#include "zebra/rt.h"
#include "zebra/zebra_ns.h"
#include "zebra/zserv.h"
#include "zebra/debug.h"
#include "zebra/interface.h"
#include "zebra/zebra_memory.h"
#include "zebra/zebra_vrf.h"
#include "zebra/rt_netlink.h"
#include "zebra/interface.h"
#include "zebra/zebra_l2.h"
#include "zebra/zebra_vxlan.h"
#include "zebra/zebra_evpn_mh.h"
#include "zebra/zebra_evpn_arp_nd.h"

/* definitions */

/* static function declarations */

/* Private functions */
static void zebra_l2_brvlan_mac_iterate_callback(struct hash_bucket *bucket, void *ctxt)
{
        struct zebra_l2_brvlan_mac *bmac;
	struct zebra_l2_brvlan_mac_ctx *ctx;

	bmac = (struct zebra_l2_brvlan_mac *)bucket->data;
	ctx = (struct zebra_l2_brvlan_mac_ctx *)ctxt;

	ctx->func(ctx->br_if, ctx->vid, &bmac->macaddr, bmac->ifindex,
		  bmac->sticky, bmac->local_inactive, bmac->dp_static,
		  ctx->arg);
}

static unsigned int zebra_l2_brvlan_mac_hash_keymake(const void *p)
{
	const struct zebra_l2_brvlan_mac *bmac;
	const void *pnt;

	bmac = (const struct zebra_l2_brvlan_mac *)p;
	pnt = (void *)bmac->macaddr.octet;

	return jhash(pnt, ETH_ALEN, 0xa5a5a55a);
}

static bool zebra_l2_brvlan_mac_hash_cmp(const void *p1, const void *p2)
{
	const struct zebra_l2_brvlan_mac *bmac1;
	const struct zebra_l2_brvlan_mac *bmac2;

	bmac1 = (const struct zebra_l2_brvlan_mac *)p1;
	bmac2 = (const struct zebra_l2_brvlan_mac *)p2;

	if (bmac1 == NULL && bmac2 == NULL)
		return 1;

	if (bmac1 == NULL || bmac2 == NULL)
		return 0;

	return (memcmp(bmac1->macaddr.octet, bmac2->macaddr.octet, ETH_ALEN)
		== 0);
}

static void zebra_l2_brvlan_mac_free(void *p)
{
	struct zebra_l2_brvlan_mac *bmac;

	bmac = (struct zebra_l2_brvlan_mac *)p;
	XFREE(MTYPE_TMP, bmac);
}

static void *zebra_l2_brvlan_mac_alloc(void *p)
{
	struct zebra_l2_brvlan_mac *bmac;
	const struct zebra_l2_brvlan_mac *tmp_mac;

	tmp_mac = (const struct zebra_l2_brvlan_mac *)p;
	bmac = XCALLOC(MTYPE_TMP, sizeof(*bmac));
	bmac->vid = tmp_mac->vid;
	memcpy(&bmac->macaddr, &tmp_mac->macaddr, ETH_ALEN);

	return (void *)bmac;
}

static void zebra_l2_brvlan_mac_table_destroy(struct hash *mac_table)
{
	if (mac_table) {
		hash_clean(mac_table, zebra_l2_brvlan_mac_free);
		hash_free(mac_table);
	}
}

static struct hash *zebra_l2_brvlan_mac_table_create()
{
	return hash_create(zebra_l2_brvlan_mac_hash_keymake,
			   zebra_l2_brvlan_mac_hash_cmp,
			   "Zebra L2 Bridge MAC Table");
}

static void map_slaves_to_bridge(struct interface *br_if, int link)
{
	struct vrf *vrf;
	struct interface *ifp;

	RB_FOREACH (vrf, vrf_name_head, &vrfs_by_name) {
		FOR_ALL_INTERFACES (vrf, ifp) {
			struct zebra_if *zif;
			struct zebra_l2info_brslave *br_slave;

			if (ifp->ifindex == IFINDEX_INTERNAL || !ifp->info)
				continue;
			if (!IS_ZEBRA_IF_BRIDGE_SLAVE(ifp))
				continue;

			/* NOTE: This assumes 'zebra_l2info_brslave' is the
			 * first field
			 * for any L2 interface.
			 */
			zif = (struct zebra_if *)ifp->info;
			br_slave = &zif->brslave_info;

			if (link) {
				if (br_slave->bridge_ifindex == br_if->ifindex)
					br_slave->br_if = br_if;
			} else {
				if (br_slave->br_if == br_if)
					br_slave->br_if = NULL;
			}
		}
	}
}

static void zebra_l2_brvlan_print_mac_hash(struct hash_bucket *bucket,
					   void *ctxt)
{
	struct zebra_l2_brvlan_mac_ctx *ctx;
	struct vty *vty;
	struct zebra_l2_brvlan_mac *bmac;
	struct interface *ifp;
	json_object *json_obj = NULL, *json_mac = NULL;
	char mac[ETHER_ADDR_STRLEN];

	ctx = (struct zebra_l2_brvlan_mac_ctx *)ctxt;
	vty = (struct vty *)(ctx->arg);
	bmac = (struct zebra_l2_brvlan_mac *)bucket->data;
	json_obj = ctx->json;

	prefix_mac2str(&bmac->macaddr, mac, sizeof(mac));
	ifp = if_lookup_by_index_per_ns(zebra_ns_lookup(NS_DEFAULT),
					bmac->ifindex);
	if (json_obj) {
		json_mac = json_object_new_object();
		json_object_string_add(json_mac, "MAC", mac);
		json_object_int_add(json_mac, "IfIndex", bmac->ifindex);
		json_object_string_add(json_mac, "Interface",
				       ifp ? ifp->name : "-");
		json_object_object_add(json_obj, "MAC", json_mac);
	} else {
		vty_out(vty, "%-17s %-7u %s\n", mac, bmac->ifindex,
			ifp ? ifp->name : "-");
	}
}

/* Public functions */

void zebra_l2_brvlan_mac_iterate(struct interface *br_if, vlanid_t vid,
				 int (*func)(struct interface *br_if,
					     vlanid_t vid, struct ethaddr *,
					     ifindex_t, bool, bool, bool,
					     void *),
				 void *arg)
{
	struct zebra_if *br_zif;
	struct zebra_l2info_bridge *br;
	struct zebra_l2_brvlan_mac_ctx ctx;

	br_zif = (struct zebra_if *)br_if->info;
	assert(br_zif);
	br = BRIDGE_FROM_ZEBRA_IF(br_zif);
	if (!br || !br->mac_table[vid])
		return;
	memset(&ctx, 0, sizeof(ctx));
	ctx.br_if = br_if;
	ctx.vid = vid;
	ctx.func = func;
	ctx.arg = arg;
	hash_iterate(br->mac_table[vid], zebra_l2_brvlan_mac_iterate_callback, &ctx);
}

void zebra_l2_brvlan_print_macs(struct vty *vty, struct interface *br_if,
				vlanid_t vid, bool uj)
{
	struct zebra_if *br_zif;
	struct zebra_l2info_bridge *br;
	uint32_t num_macs;
	struct zebra_l2_brvlan_mac_ctx ctx;
	json_object *json_obj = NULL;

	if (uj) /* json format */
		json_obj = json_object_new_object();


	br_zif = (struct zebra_if *)br_if->info;
	assert(br_zif);
	br = BRIDGE_FROM_ZEBRA_IF(br_zif);
	if (!br) {
		return;
	}
	if (!br->mac_table[vid])
		return;
	num_macs = hashcount(br->mac_table[vid]);
	if (uj) {
		json_object_string_add(json_obj, "bridge", br_if->name);
		json_object_int_add(json_obj, "VID", vid);
		json_object_int_add(json_obj, "number of local MACS", num_macs);
		vty_out(vty, "%s\n",
			json_object_to_json_string_ext(
				json_obj, JSON_C_TO_STRING_PRETTY));
	} else {
		vty_out(vty, "bridge %s VID %u - Number of local MACs: %u\n",
			br_if->name, vid, num_macs);
		vty_out(vty, "%-17s %-7s %-30s\n", "MAC", "IfIndex",
			"Interface");
	}
	if (!num_macs) {
		vty_out(vty, "bridge %s VID %u - No local MACs\n",
			br_if->name, vid);
		return;
	}

	vty_out(vty, "bridge %s VID %u - Number of local MACs: %u\n",
		br_if->name, vid, num_macs);
	vty_out(vty, "%-17s %-7s %-30s\n", "MAC", "IfIndex", "Interface");
	memset(&ctx, 0, sizeof(ctx));
	ctx.br_if = br_if;
	ctx.vid = vid;
	ctx.arg = vty;
	ctx.json = json_obj;
	hash_iterate(br->mac_table[vid], zebra_l2_brvlan_print_mac_hash, &ctx);
	if (uj) {
		vty_out(vty, "%s\n",
			json_object_to_json_string_ext(
				json_obj, JSON_C_TO_STRING_PRETTY));
		json_object_free(json_obj);
	}
}

int zebra_l2_brvlan_mac_del(struct interface *br_if,
			    struct zebra_l2_brvlan_mac *bmac)
{
	struct zebra_if *br_zif;
	struct zebra_l2info_bridge *br;
	vlanid_t vid;
	struct zebra_l2_brvlan_mac *tmp_mac;
	char buf[ETHER_ADDR_STRLEN];
	uint32_t num_macs;

	br_zif = (struct zebra_if *)br_if->info;
	assert(br_zif);
	br = BRIDGE_FROM_ZEBRA_IF(br_zif);
	if (!br->mac_table[bmac->vid]) {
		zlog_debug("bridge %s VID %u - MAC hash table not found",
			   br_if->name, bmac->vid);
		return -1;
	}

	vid = bmac->vid;
	tmp_mac = hash_release(br->mac_table[vid], bmac);
	if (tmp_mac)
		zebra_l2_brvlan_mac_free(tmp_mac);

	num_macs = hashcount(br->mac_table[vid]);

	if (IS_ZEBRA_DEBUG_VXLAN)
		zlog_debug("bridge %s VID %u bmac %p MAC %s delete - hash# %u",
			   br_if->name, vid, bmac,
			   prefix_mac2str(&bmac->macaddr, buf, sizeof(buf)),
			   num_macs);

	if (!num_macs) {
		if (IS_ZEBRA_DEBUG_VXLAN)
			zlog_debug("bridge %s vlan %u - destroying MAC table",
				   br_if->name, vid);
		zebra_l2_brvlan_mac_table_destroy(br->mac_table[vid]);
		br->mac_table[vid] = NULL;
	}

	return 0;
}

int zebra_l2_brvlan_mac_update(struct interface *br_if,
			       struct zebra_l2_brvlan_mac *bmac,
			       ifindex_t ifidx)
{
	char buf[ETHER_ADDR_STRLEN];

	if (IS_ZEBRA_DEBUG_VXLAN)
		zlog_debug("bridge %s VID %u bmac %p MAC %s update ifidx %u",
			   br_if->name, bmac->vid, bmac,
			   prefix_mac2str(&bmac->macaddr, buf, sizeof(buf)),
			   ifidx);

	bmac->ifindex = ifidx;
	return 0;
}

struct zebra_l2_brvlan_mac *
zebra_l2_brvlan_mac_add(struct interface *br_if, vlanid_t vid,
			struct ethaddr *mac, ifindex_t ifidx, bool sticky,
			bool local_inactive, bool dp_static)
{
	struct zebra_if *br_zif;
	struct zebra_l2info_bridge *br;
	struct zebra_l2_brvlan_mac *bmac;
	struct zebra_l2_brvlan_mac tmp_mac;
	char buf[ETHER_ADDR_STRLEN];
	uint32_t num_macs;

	br_zif = (struct zebra_if *)br_if->info;
	assert(br_zif);
	br = BRIDGE_FROM_ZEBRA_IF(br_zif);
	if (!br->mac_table[vid]) {
		br->mac_table[vid] = zebra_l2_brvlan_mac_table_create();
		if (!br->mac_table[vid]) {
			zlog_err(
				"bridge %s vid %u - failed to create MAC hash table",
				br_if->name, vid);
			return NULL;
		}
		if (IS_ZEBRA_DEBUG_VXLAN)
			zlog_debug("bridge %s vid %u - MAC hash table created",
				   br_if->name, vid);
	}

	memset(&tmp_mac, 0, sizeof(tmp_mac));
	memcpy(&tmp_mac.macaddr, mac, ETH_ALEN);
	bmac = hash_get(br->mac_table[vid], (void *)&tmp_mac,
			zebra_l2_brvlan_mac_alloc);
	assert(bmac);
	bmac->br_if = br_if;
	bmac->vid = vid;
	bmac->ifindex = ifidx;

	num_macs = hashcount(br->mac_table[vid]);

	if (IS_ZEBRA_DEBUG_VXLAN)
		zlog_debug(
			"bridge %s VID %u MAC %s ifidx %u add - %p, hash# %u",
			br_if->name, vid, prefix_mac2str(mac, buf, sizeof(buf)),
			ifidx, bmac, num_macs);

	return bmac;
}

struct zebra_l2_brvlan_mac *zebra_l2_brvlan_mac_find(struct interface *br_if,
						     vlanid_t vid,
						     struct ethaddr *mac)
{
	struct zebra_if *br_zif;
	struct zebra_l2info_bridge *br;
	struct zebra_l2_brvlan_mac *bmac;
	struct zebra_l2_brvlan_mac tmp_mac;
	char buf[ETHER_ADDR_STRLEN];

	br_zif = (struct zebra_if *)br_if->info;
	assert(br_zif);
	br = BRIDGE_FROM_ZEBRA_IF(br_zif);
	if (!br->mac_table[vid])
		return NULL;

	memset(&tmp_mac, 0, sizeof(tmp_mac));
	memcpy(&tmp_mac.macaddr, mac, ETH_ALEN);
	bmac = (struct zebra_l2_brvlan_mac *)hash_lookup(br->mac_table[vid],
							 (void *)&tmp_mac);

	if (IS_ZEBRA_DEBUG_VXLAN)
		zlog_debug("bridge %s VID %u MAC %s find - %p", br_if->name,
			   vid, prefix_mac2str(mac, buf, sizeof(buf)), bmac);

	return bmac;
}

void zebra_l2_map_slave_to_bridge(struct zebra_l2info_brslave *br_slave)
{
	struct interface *br_if;

	/* TODO: Handle change of master */
	br_if = if_lookup_by_index_per_ns(zebra_ns_lookup(NS_DEFAULT),
					  br_slave->bridge_ifindex);
	if (br_if)
		br_slave->br_if = br_if;
}

void zebra_l2_unmap_slave_from_bridge(struct zebra_l2info_brslave *br_slave)
{
	br_slave->br_if = NULL;
}

/* If any of the bond members are in bypass state the bond is placed
 * in bypass state
 */
static void zebra_l2_bond_lacp_bypass_eval(struct zebra_if *bond_zif)
{
	struct listnode *node;
	struct zebra_if *bond_mbr;
	bool old_bypass = !!(bond_zif->flags & ZIF_FLAG_LACP_BYPASS);
	bool new_bypass = false;

	if (bond_zif->bond_info.mbr_zifs) {
		for (ALL_LIST_ELEMENTS_RO(bond_zif->bond_info.mbr_zifs, node,
					  bond_mbr)) {
			if (bond_mbr->flags & ZIF_FLAG_LACP_BYPASS) {
				new_bypass = true;
				break;
			}
		}
	}

	if (old_bypass == new_bypass)
		return;

	if (IS_ZEBRA_DEBUG_EVPN_MH_ES || IS_ZEBRA_DEBUG_EVENT)
		zlog_debug("bond %s lacp bypass changed to %s",
			   bond_zif->ifp->name, new_bypass ? "on" : "off");

	if (new_bypass)
		bond_zif->flags |= ZIF_FLAG_LACP_BYPASS;
	else
		bond_zif->flags &= ~ZIF_FLAG_LACP_BYPASS;

	if (bond_zif->es_info.es)
		zebra_evpn_es_bypass_update(bond_zif->es_info.es, bond_zif->ifp,
					    new_bypass);
}

/* Returns true if member was newly linked to bond */
void zebra_l2_map_slave_to_bond(struct zebra_if *zif, vrf_id_t vrf_id)
{
	struct interface *bond_if;
	struct zebra_if *bond_zif;
	struct zebra_l2info_bondslave *bond_slave = &zif->bondslave_info;

	bond_if = if_lookup_by_index_all_vrf(bond_slave->bond_ifindex);
	if (bond_if == bond_slave->bond_if)
		return;

	/* unlink the slave from the old master */
	zebra_l2_unmap_slave_from_bond(zif);

	/* If the bond is present and ready link the bond-member
	 * to it
	 */
	if (bond_if && (bond_zif = bond_if->info)) {
		if (bond_zif->bond_info.mbr_zifs) {
			if (IS_ZEBRA_DEBUG_EVPN_MH_ES ||
					IS_ZEBRA_DEBUG_EVENT)
				zlog_debug("bond mbr %s linked to %s",
					zif->ifp->name, bond_if->name);
			bond_slave->bond_if = bond_if;
			/* link the slave to the new bond master */
			listnode_add(bond_zif->bond_info.mbr_zifs, zif);
			/* inherit protodown flags from the es-bond */
			if (zebra_evpn_is_es_bond(bond_if))
				zebra_evpn_mh_update_protodown_bond_mbr(zif,
					false /*clear*/, __func__);
			zebra_l2_bond_lacp_bypass_eval(bond_zif);
		}
	} else {
		if (IS_ZEBRA_DEBUG_EVPN_MH_ES || IS_ZEBRA_DEBUG_EVENT)
			zlog_debug("bond mbr %s link to bond skipped",
				zif->ifp->name);
	}
}

void zebra_l2_unmap_slave_from_bond(struct zebra_if *zif)
{
	struct zebra_l2info_bondslave *bond_slave = &zif->bondslave_info;
	struct zebra_if *bond_zif;

	if (!bond_slave->bond_if) {
		if (IS_ZEBRA_DEBUG_EVPN_MH_ES || IS_ZEBRA_DEBUG_EVENT)
			zlog_debug("bond mbr %s unlink from bond skipped",
				zif->ifp->name);
		return;
	}

	if (IS_ZEBRA_DEBUG_EVPN_MH_ES || IS_ZEBRA_DEBUG_EVENT)
		zlog_debug("bond mbr %s un-linked from %s",
			zif->ifp->name, bond_slave->bond_if->name);

	/* unlink the slave from the bond master */
	bond_zif = bond_slave->bond_if->info;
	/* clear protodown flags */
	if (zebra_evpn_is_es_bond(bond_zif->ifp))
		zebra_evpn_mh_update_protodown_bond_mbr(zif,
			true /*clear*/, __func__);
	listnode_delete(bond_zif->bond_info.mbr_zifs, zif);
	bond_slave->bond_if = NULL;
	zebra_l2_bond_lacp_bypass_eval(bond_zif);
}

void zebra_l2if_update_bond(struct interface *ifp, bool add)
{
	struct zebra_if *zif;
	struct zebra_l2info_bond *bond;

	zif = ifp->info;
	assert(zif);
	bond = &zif->bond_info;

	if (add) {
		if (!bond->mbr_zifs) {
			if (IS_ZEBRA_DEBUG_EVPN_MH_ES || IS_ZEBRA_DEBUG_EVENT)
				zlog_debug("bond %s mbr list create",
						ifp->name);
			bond->mbr_zifs = list_new();
		}
	} else {
		struct listnode *node;
		struct listnode *nnode;
		struct zebra_if *bond_mbr;

		if (!bond->mbr_zifs)
			return;

		if (IS_ZEBRA_DEBUG_EVPN_MH_ES || IS_ZEBRA_DEBUG_EVENT)
			zlog_debug("bond %s mbr list delete",
					ifp->name);
		for (ALL_LIST_ELEMENTS(bond->mbr_zifs, node, nnode, bond_mbr))
			zebra_l2_unmap_slave_from_bond(bond_mbr);

		list_delete(&bond->mbr_zifs);
	}
}

/* Initialize the mac_table in bridge intf */
static void zebra_init_mac_table(struct zebra_l2info_bridge *br)
{
	for (int i = 0; i < VLANID_MAX; i++)
		br->mac_table[i] = NULL;
}

/*
 * Handle Bridge interface add or update. Update relevant info,
 * map slaves (if any) to the bridge.
 */
void zebra_l2_bridge_add_update(struct interface *ifp,
				struct zebra_l2info_bridge *bridge_info,
				int add)
{
	struct zebra_if *zif;
	struct zebra_l2info_bridge *br;

	zif = ifp->info;
	assert(zif);

	br = BRIDGE_FROM_ZEBRA_IF(zif);
	br->vlan_aware = bridge_info->vlan_aware;
	if (add)
		zebra_init_mac_table(br);
	/* Link all slaves to this bridge */
	map_slaves_to_bridge(ifp, 1);
}

/*
 * Handle Bridge interface delete.
 */
void zebra_l2_bridge_del(struct interface *ifp)
{
	/* Unlink all slaves to this bridge */
	map_slaves_to_bridge(ifp, 0);
}

/*
 * Update L2 info for a VLAN interface. Only relevant parameter is the
 * VLAN Id and this cannot change.
 */
void zebra_l2_vlanif_update(struct interface *ifp,
			    struct zebra_l2info_vlan *vlan_info)
{
	struct zebra_if *zif;

	zif = ifp->info;
	assert(zif);

	/* Copy over the L2 information. */
	memcpy(&zif->l2info.vl, vlan_info, sizeof(*vlan_info));
}

/*
 * Update L2 info for a VxLAN interface. This is called upon interface
 * addition as well as update. Upon add, need to invoke the VNI create
 * function. Upon update, the params of interest are the local tunnel
 * IP and VLAN mapping, but the latter is handled separately.
 */
void zebra_l2_vxlanif_add_update(struct interface *ifp,
				 struct zebra_l2info_vxlan *vxlan_info, int add)
{
	struct zebra_if *zif;
	struct in_addr old_vtep_ip;
	uint16_t chgflags = 0;

	zif = ifp->info;
	assert(zif);

	if (add) {
		memcpy(&zif->l2info.vxl, vxlan_info, sizeof(*vxlan_info));
		zebra_evpn_vl_vxl_ref(zif->l2info.vxl.access_vlan, zif);
		zebra_vxlan_if_add(ifp);
		return;
	}

	old_vtep_ip = zif->l2info.vxl.vtep_ip;

	if (!IPV4_ADDR_SAME(&old_vtep_ip, &vxlan_info->vtep_ip)) {
		chgflags |= ZEBRA_VXLIF_LOCAL_IP_CHANGE;
		zif->l2info.vxl.vtep_ip = vxlan_info->vtep_ip;
	}

	if (!IPV4_ADDR_SAME(&zif->l2info.vxl.mcast_grp,
				&vxlan_info->mcast_grp)) {
		chgflags |= ZEBRA_VXLIF_MCAST_GRP_CHANGE;
		zif->l2info.vxl.mcast_grp = vxlan_info->mcast_grp;
	}

	if (chgflags)
		zebra_vxlan_if_update(ifp, chgflags);
}

/*
 * Handle change to VLAN to VNI mapping.
 */
void zebra_l2_vxlanif_update_access_vlan(struct interface *ifp,
					 vlanid_t access_vlan)
{
	struct zebra_if *zif;
	vlanid_t old_access_vlan;

	zif = ifp->info;
	assert(zif);

	old_access_vlan = zif->l2info.vxl.access_vlan;
	if (old_access_vlan == access_vlan)
		return;

	zif->l2info.vxl.access_vlan = access_vlan;

	zebra_evpn_vl_vxl_deref(old_access_vlan, zif);
	zebra_evpn_vl_vxl_ref(zif->l2info.vxl.access_vlan, zif);
	zebra_vxlan_if_update(ifp, ZEBRA_VXLIF_VLAN_CHANGE);
}

/*
 * Handle VxLAN interface delete.
 */
void zebra_l2_vxlanif_del(struct interface *ifp)
{
	struct zebra_if *zif;

	zif = ifp->info;
	assert(zif);

	zebra_evpn_vl_vxl_deref(zif->l2info.vxl.access_vlan, zif);
	zebra_vxlan_if_del(ifp);
}

/*
 * Map or unmap interface from bridge.
 * NOTE: It is currently assumped that an interface has to be unmapped
 * from a bridge before it can be mapped to another bridge.
 */
void zebra_l2if_update_bridge_slave(struct interface *ifp,
				    ifindex_t bridge_ifindex)
{
	struct zebra_if *zif;
	ifindex_t old_bridge_ifindex;

	zif = ifp->info;
	assert(zif);

	old_bridge_ifindex = zif->brslave_info.bridge_ifindex;
	if (old_bridge_ifindex == bridge_ifindex)
		return;

	zif->brslave_info.bridge_ifindex = bridge_ifindex;

	/* Set up or remove link with master */
	if (bridge_ifindex != IFINDEX_INTERNAL) {
		zebra_l2_map_slave_to_bridge(&zif->brslave_info);
		/* In the case of VxLAN, invoke the handler for EVPN. */
		if (zif->zif_type == ZEBRA_IF_VXLAN)
			zebra_vxlan_if_update(ifp, ZEBRA_VXLIF_MASTER_CHANGE);
		else
			zebra_evpn_arp_nd_if_update(zif, true);
		if (zif->es_info.es)
			zebra_evpn_es_local_br_port_update(zif);
	} else if (old_bridge_ifindex != IFINDEX_INTERNAL) {
		/*
		 * In the case of VxLAN, invoke the handler for EVPN.
		 * Note that this should be done *prior*
		 * to unmapping the interface from the bridge.
		 */
		if (zif->zif_type == ZEBRA_IF_VXLAN)
			zebra_vxlan_if_update(ifp, ZEBRA_VXLIF_MASTER_CHANGE);
		else
			zebra_evpn_arp_nd_if_update(zif, false);
		if (zif->es_info.es)
			zebra_evpn_es_local_br_port_update(zif);
		zebra_l2_unmap_slave_from_bridge(&zif->brslave_info);
	}
}

void zebra_l2if_update_bond_slave(struct interface *ifp, ifindex_t bond_ifindex,
				  bool new_bypass)
{
	struct zebra_if *zif;
	ifindex_t old_bond_ifindex;
	bool old_bypass;
	struct zebra_l2info_bondslave *bond_mbr;

	zif = ifp->info;
	assert(zif);

	old_bypass = !!(zif->flags & ZIF_FLAG_LACP_BYPASS);
	if (old_bypass != new_bypass) {
		if (IS_ZEBRA_DEBUG_EVPN_MH_ES || IS_ZEBRA_DEBUG_EVENT)
			zlog_debug("bond-mbr %s lacp bypass changed to %s",
				   zif->ifp->name, new_bypass ? "on" : "off");

		if (new_bypass)
			zif->flags |= ZIF_FLAG_LACP_BYPASS;
		else
			zif->flags &= ~ZIF_FLAG_LACP_BYPASS;

		bond_mbr = &zif->bondslave_info;
		if (bond_mbr->bond_if) {
			struct zebra_if *bond_zif = bond_mbr->bond_if->info;

			zebra_l2_bond_lacp_bypass_eval(bond_zif);
		}
	}

	old_bond_ifindex = zif->bondslave_info.bond_ifindex;
	if (old_bond_ifindex == bond_ifindex)
		return;

	zif->bondslave_info.bond_ifindex = bond_ifindex;

	/* Set up or remove link with master */
	if (bond_ifindex != IFINDEX_INTERNAL)
		zebra_l2_map_slave_to_bond(zif, ifp->vrf_id);
	else if (old_bond_ifindex != IFINDEX_INTERNAL)
		zebra_l2_unmap_slave_from_bond(zif);
}

void zebra_vlan_bitmap_compute(struct interface *ifp,
		uint32_t vid_start, uint16_t vid_end)
{
	uint32_t vid;
	struct zebra_if *zif;

	zif = (struct zebra_if *)ifp->info;
	assert(zif);

	for (vid = vid_start; vid <= vid_end; ++vid)
		bf_set_bit(zif->vlan_bitmap, vid);
}

void zebra_vlan_mbr_re_eval(struct interface *ifp, bitfield_t old_vlan_bitmap)
{
	uint32_t vid;
	struct zebra_if *zif;

	zif = (struct zebra_if *)ifp->info;
	assert(zif);

	if (!bf_cmp(zif->vlan_bitmap, old_vlan_bitmap))
		/* no change */
		return;

	bf_for_each_set_bit(zif->vlan_bitmap, vid, IF_VLAN_BITMAP_MAX) {
		/* if not already set create new reference */
		if (!bf_test_index(old_vlan_bitmap, vid))
			zebra_evpn_vl_mbr_ref(vid, zif);

		/* also clear from the old vlan bitmap */
		bf_release_index(old_vlan_bitmap, vid);
	}

	/* any bits remaining in the old vlan bitmap are stale references */
	bf_for_each_set_bit(old_vlan_bitmap, vid, IF_VLAN_BITMAP_MAX) {
		zebra_evpn_vl_mbr_deref(vid, zif);
	}
}
