/* Copyright (C) 2014, 2015, 2017 Cumulus Networks, Inc. All rights reserved
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * https://www.gnu.org/licenses/gpl-2.0-standalone.html
 */

/* ebt_tricolorpolice
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "../include/ebtables_u.h"
#include <linux/netfilter_bridge/ebt_tricolorpolice.h>

#define _TRICOLORPOLICE_COLOR_MODE           '1'
#define _TRICOLORPOLICE_CIR                  '2'
#define _TRICOLORPOLICE_CBS                  '3'
#define _TRICOLORPOLICE_PIR                  '4'
#define _TRICOLORPOLICE_EBS                  '5'
#define _TRICOLORPOLICE_CONFORM_ACTION_DSCP  '6'
#define _TRICOLORPOLICE_EXCEED_ACTION_DSCP   '7'
#define _TRICOLORPOLICE_VIOLATE_ACTION       '8'
#define _TRICOLORPOLICE_VIOLATE_ACTION_DSCP  '9'

static struct option opts[] =
{
    { "set-color-mode", required_argument, 0, _TRICOLORPOLICE_COLOR_MODE },
    { "set-cir", required_argument, 0, _TRICOLORPOLICE_CIR },
    { "set-cbs", required_argument, 0, _TRICOLORPOLICE_CBS },
    { "set-pir", required_argument, 0, _TRICOLORPOLICE_PIR },
    { "set-ebs", required_argument, 0, _TRICOLORPOLICE_EBS },
    { "set-conform-action-dscp", required_argument, 0, _TRICOLORPOLICE_CONFORM_ACTION_DSCP },
    { "set-exceed-action-dscp", required_argument, 0, _TRICOLORPOLICE_EXCEED_ACTION_DSCP },
    { "set-violate-action", required_argument, 0, _TRICOLORPOLICE_VIOLATE_ACTION },
    { "set-violate-action-dscp", required_argument, 0, _TRICOLORPOLICE_VIOLATE_ACTION_DSCP },
    { 0 }
};

static void print_help()
{
    printf(
           "tricolorpolice option:\n"
"  --set-color-mode  STRING setting the mode in blind or aware\n"
"  --set-cir  INT    setting committed information rate in kbits per second\n"
"  --set-cbs  INT    setting committed burst size in kbyte\n"
"  --set-pir  INT    setting peak information rate in kbits per second\n"
"  --set-ebs  INT    setting excess burst size in kbyte\n"
"  --set-conform-action-dscp INT setting dscp value if the action is accept for conforming packets\n"
"  --set-exceed-action-dscp INT setting dscp value if the action is accept for exceeding packets\n"
"  --set-violate-action  STRING setting the action (accept/drop) for violating packets\n"
"  --set-violate-action-dscp INT setting dscp value if the action is accept for violating packets\n");
}

static void init(struct ebt_entry_target *target)
{
    struct ebt_tricolor_police_info *info =
        (struct ebt_tricolor_police_info *)target->data;

    info->color_mode = EBT_TRICOLOR_POLICE_COLOR_MODE_AWARE;
    info->cir = 0;
    info->cbs = 0;
    info->pir = 0;
    info->ebs = 0;
    info->conform_action.type = 0;
    info->exceed_action.type = 0;
    info->violate_action.type = 0;
    info->violate_action.type = EBT_TRICOLOR_POLICE_ACTION_PERMIT;
    info->conform_action.dscp_value = 0;
    info->exceed_action.dscp_value = 0;
    info->violate_action.dscp_value = 0;

    return;
}

#define OPT_TRICOLORPOLICE_COLOR_MODE           0x01
#define OPT_TRICOLORPOLICE_CIR                  0x02
#define OPT_TRICOLORPOLICE_CBS                  0x04
#define OPT_TRICOLORPOLICE_PIR                  0x08
#define OPT_TRICOLORPOLICE_EBS                  0x10
#define OPT_TRICOLORPOLICE_CONFORM_ACTION_DSCP  0x20
#define OPT_TRICOLORPOLICE_EXCEED_ACTION_DSCP   0x40
#define OPT_TRICOLORPOLICE_VIOLATE_ACTION       0x80
#define OPT_TRICOLORPOLICE_VIOLATE_ACTION_DSCP  0x100

static int parse(int c, char **argv, int argc,
   const struct ebt_u_entry *entry, unsigned int *flags,
   struct ebt_entry_target **target)
{
    struct ebt_tricolor_police_info *info =
        (struct ebt_tricolor_police_info *)(*target)->data;
    uint32_t mask;
    long int i;
    char *end;

    switch (c) {
    case _TRICOLORPOLICE_COLOR_MODE:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_COLOR_MODE);
        if (strncmp(optarg, "blind", strlen(optarg)) == 0)
            info->color_mode = EBT_TRICOLOR_POLICE_COLOR_MODE_BLIND;
        else if (strncmp(optarg, "aware", strlen(optarg)) == 0)
            info->color_mode = EBT_TRICOLOR_POLICE_COLOR_MODE_AWARE;
        else
            ebt_print_error2("Problem with specified mode");
        info->bitmask |= EBT_TRICOLOR_POLICE_COLOR_MODE;
        break;
    case _TRICOLORPOLICE_CIR:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_CIR);
        i = strtoul(optarg, &end, 10);
        if (i < EBT_TRICOLOR_POLICE_MIN_CIR || i > EBT_TRICOLOR_POLICE_MAX_CIR || *end != '\0')
            ebt_print_error2("Problem with specified rate");
        info->cir = i;
        info->bitmask |= EBT_TRICOLOR_POLICE_CIR;
        break;
    case _TRICOLORPOLICE_CBS:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_CBS);
        i = strtoul(optarg, &end, 10);
        if (i < EBT_TRICOLOR_POLICE_MIN_CBS || i > EBT_TRICOLOR_POLICE_MAX_CBS || *end != '\0')
            ebt_print_error2("Problem with specified rate");
        info->cbs = i;
        info->bitmask |= EBT_TRICOLOR_POLICE_CBS;
        break;
    case _TRICOLORPOLICE_PIR:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_PIR);
        i = strtoul(optarg, &end, 10);
        if (i < EBT_TRICOLOR_POLICE_MIN_PIR || i > EBT_TRICOLOR_POLICE_MAX_PIR || *end != '\0')
            ebt_print_error2("Problem with specified rate");
        info->pir = i;
        info->bitmask |= EBT_TRICOLOR_POLICE_PIR;
        break;
    case _TRICOLORPOLICE_EBS:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_EBS);
        i = strtoul(optarg, &end, 10);
        if (i < EBT_TRICOLOR_POLICE_MIN_EBS || i > EBT_TRICOLOR_POLICE_MAX_EBS || *end != '\0')
            ebt_print_error2("Problem with specified rate");
        info->ebs = i;
        info->bitmask |= EBT_TRICOLOR_POLICE_EBS;
        break;

    case _TRICOLORPOLICE_CONFORM_ACTION_DSCP:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_CONFORM_ACTION_DSCP);
        i = strtoul(optarg, &end, 10);
        info->conform_action.dscp_value = i;
        info->bitmask |= EBT_TRICOLOR_POLICE_CONFORM_ACTION_DSCP;
        break;
    case _TRICOLORPOLICE_EXCEED_ACTION_DSCP:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_EXCEED_ACTION_DSCP);
        i = strtoul(optarg, &end, 10);
        info->exceed_action.dscp_value = i;
        info->bitmask |= EBT_TRICOLOR_POLICE_EXCEED_ACTION_DSCP;
        break;
    case _TRICOLORPOLICE_VIOLATE_ACTION_DSCP:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_VIOLATE_ACTION_DSCP);
        i = strtoul(optarg, &end, 10);
        info->violate_action.dscp_value = i;
        info->bitmask |= EBT_TRICOLOR_POLICE_VIOLATE_ACTION_DSCP;
        break;
    case _TRICOLORPOLICE_VIOLATE_ACTION:
        ebt_check_option2(flags, OPT_TRICOLORPOLICE_VIOLATE_ACTION);
        if (strncmp(optarg, "accept", strlen(optarg)) == 0)
            info->violate_action.type = EBT_TRICOLOR_POLICE_ACTION_PERMIT;
        else if (strncmp(optarg, "drop", strlen(optarg)) == 0)
            info->violate_action.type = EBT_TRICOLOR_POLICE_ACTION_DENY;
        else
            ebt_print_error2("Problem with specified action");
        info->bitmask |= EBT_TRICOLOR_POLICE_VIOLATE_ACTION;
        break;

    default:
        return 0;
    }
    return 1;
}

static void final_check(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target, const char *name,
   unsigned int hookmask, unsigned int time)
{
    struct ebt_tricolor_police_info *info =
        (struct ebt_tricolor_police_info *)target->data;
    if (!(info->bitmask & EBT_TRICOLOR_POLICE_CIR) ||
        !(info->bitmask & EBT_TRICOLOR_POLICE_CBS)) {
        ebt_print_error("For tricolorpolice target, CIR and CBS settings are madatory");
    }
}

static void print(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target)
{
    struct ebt_tricolor_police_info *info =
        (struct ebt_tricolor_police_info *)target->data;

    if (info->bitmask & EBT_TRICOLOR_POLICE_COLOR_MODE)
        printf("--set-color-mode %s ", (info->color_mode == EBT_TRICOLOR_POLICE_COLOR_MODE_AWARE)?
               "aware":"blind");
    if (info->bitmask & EBT_TRICOLOR_POLICE_CIR)
        printf("--set-cir %d ", info->cir);
    if (info->bitmask & EBT_TRICOLOR_POLICE_CBS)
        printf("--set-cbs %d ", info->cbs);
    if (info->bitmask & EBT_TRICOLOR_POLICE_PIR)
        printf("--set-pir %d ", info->pir);
    if (info->bitmask & EBT_TRICOLOR_POLICE_EBS)
        printf("--set-ebs %d ", info->ebs);
    if (info->bitmask & EBT_TRICOLOR_POLICE_CONFORM_ACTION_DSCP)
        printf("--set-conform_action_dscp %d ", info->conform_action.dscp_value);
    if (info->bitmask & EBT_TRICOLOR_POLICE_EXCEED_ACTION_DSCP)
        printf("--set-exceed_action_dscp %d ", info->exceed_action.dscp_value);
    if (info->bitmask & EBT_TRICOLOR_POLICE_VIOLATE_ACTION_DSCP)
        printf("--set-violate_action_dscp %d ", info->violate_action.dscp_value);
    if (info->bitmask & EBT_TRICOLOR_POLICE_VIOLATE_ACTION)
        printf("--set-violate-action %s ", (info->violate_action.type == EBT_TRICOLOR_POLICE_ACTION_PERMIT)?
               "accept":"drop");
}

static int compare(const struct ebt_entry_target *t1,
   const struct ebt_entry_target *t2)
{
    struct ebt_tricolor_police_info *info1 =
        (struct ebt_tricolor_police_info *)t1->data;
    struct ebt_tricolor_police_info *info2 =
        (struct ebt_tricolor_police_info *)t2->data;

    return (!memcmp(info1, info2, sizeof(*info1)));
}

static struct ebt_u_target police_target =
{
    .name		= "tricolorpolice",
    .size		= sizeof(struct ebt_tricolor_police_info),
    .help		= print_help,
    .init		= init,
    .parse		= parse,
    .final_check	= final_check,
    .print		= print,
    .compare	= compare,
    .extra_ops	= opts,
};

void _init(void)
{
    ebt_register_target(&police_target);
}
