/*
 * Copyright (C) 2016, 2017 Cumulus Networks, Inc. All rights reserved
 */

/* ebt_setqos
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "../include/ebtables_u.h"
#include <linux/netfilter_bridge/ebt_setqos.h>

static const struct ds_class
{
        const char *name;
        unsigned int dscp;
} ds_classes[] =
{
        { "CS0", 0x00 },
        { "CS1", 0x08 },
        { "CS2", 0x10 },
        { "CS3", 0x18 },
        { "CS4", 0x20 },
        { "CS5", 0x28 },
        { "CS6", 0x30 },
        { "CS7", 0x38 },
        { "BE", 0x00 },
        { "AF11", 0x0a },
        { "AF12", 0x0c },
        { "AF13", 0x0e },
        { "AF21", 0x12 },
        { "AF22", 0x14 },
        { "AF23", 0x16 },
        { "AF31", 0x1a },
        { "AF32", 0x1c },
        { "AF33", 0x1e },
        { "AF41", 0x22 },
        { "AF42", 0x24 },
        { "AF43", 0x26 },
        { "EF", 0x2e }
};

#define _SETQOS_COS        '1'
#define _SETQOS_DSCP       '2'
#define _SETQOS_DSCP_CLASS '3'
static struct option opts[] =
{
	{ "set-cos", required_argument, 0, _SETQOS_COS },
	{ "set-dscp", required_argument, 0, _SETQOS_DSCP },
	{ "set-dscp-class", required_argument, 0, _SETQOS_DSCP_CLASS },
	{ 0 }
};

static void print_help()
{
	printf(
	"setqos option:\n"
	" --set-cos INT          : Set datapath resource/queuing class value\n"
	" --set-dscp value       : Set dSCP Field in packet header to value\n"
        "                          This value can be in decimal (ex: 32)\n"
        "                          or in hex (ex: 0x20)\n"
        " --set-dscp-class class : Set the DSCP field in packet header to the\n"
        "                          value represented by the DiffServ class value.\n"
        "                          This class may be EF,BE or any of the CSxx\n"
        "                          or AFxx classes.\n"
        "\n"
        "                         --set-dscp & --set-dscp-class options are mutually exclusive! \n");
}

static void init(struct ebt_entry_target *target)
{
	struct ebt_setqos_info *setqosinfo =
	   (struct ebt_setqos_info *)target->data;

	setqosinfo->cos = 0;
        setqosinfo->dscp = 0;
        setqosinfo->bitmask = 0;

	return;
}

static unsigned int
class_to_dscp(const char *name)
{
        unsigned int i;

        for (i = 0; i < ARRAY_SIZE(ds_classes); i++) {
                if (!strncasecmp(name, ds_classes[i].name,
                                        strlen(ds_classes[i].name)))
                        return ds_classes[i].dscp;
        }

        ebt_print_error2("Invalid DSCP value `%s'\n", name);
}

#define OPT_SETQOS_COS        0x01
#define OPT_SETQOS_DSCP       0x02
#define OPT_SETQOS_DSCP_CLASS 0x03
static int parse(int c, char **argv, int argc,
   const struct ebt_u_entry *entry, unsigned int *flags,
   struct ebt_entry_target **target)
{
	struct ebt_setqos_info *setqosinfo =
	   (struct ebt_setqos_info *)(*target)->data;
	uint32_t mask;
	long int i;
	char *end;

	switch (c) {
	case _SETQOS_COS:
		ebt_check_option2(flags, OPT_SETQOS_COS);
		i = strtol(optarg, &end, 16);
                if (i < EBT_SETQOS_MIN_COS || i > EBT_SETQOS_MAX_COS || *end != '\0')
                        ebt_print_error2("Problem with specified cos id");
		setqosinfo->cos = i;
		setqosinfo->bitmask |= EBT_SETQOS_COS;
		break;
        case _SETQOS_DSCP:
		ebt_check_option2(flags, OPT_SETQOS_DSCP);
		i = strtol(optarg, &end, 16);
                if (i < 0 || i > EBT_SETQOS_DSCP_MAX || *end != '\0')
                        ebt_print_error2("Problem with specified dscp id");
		setqosinfo->dscp = i;
		setqosinfo->bitmask |= EBT_SETQOS_DSCP;
                break;
        case _SETQOS_DSCP_CLASS:
		setqosinfo->dscp = class_to_dscp(optarg);
		setqosinfo->bitmask |= EBT_SETQOS_DSCP;
                break;
	default:
                ebt_print_error("Invalid setqos option\n");
		return 0;
	}
	return 1;
}

static void final_check(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target, const char *name,
   unsigned int hookmask, unsigned int time)
{
	struct ebt_setqos_info *setqosinfo =
	   (struct ebt_setqos_info *)target->data;

	if (!(setqosinfo->bitmask & EBT_SETQOS_COS)
            && !(setqosinfo->bitmask & EBT_SETQOS_DSCP)) {
		ebt_print_error("For SETQOS, cos or dscp must be specified");
	}
}

static void print(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target)
{
	struct ebt_setqos_info *setqosinfo =
	   (struct ebt_setqos_info *)target->data;

	if (setqosinfo->bitmask & EBT_SETQOS_COS)
		printf("--set-cos %d ", setqosinfo->cos);
	if (setqosinfo->bitmask & EBT_SETQOS_DSCP)
		printf("--set-dscp %d ", setqosinfo->dscp);
}

static int compare(const struct ebt_entry_target *t1,
   const struct ebt_entry_target *t2)
{
	struct ebt_setqos_info *setqosinfo1 =
	   (struct ebt_setqos_info *)t1->data;
	struct ebt_setqos_info *setqosinfo2 =
	   (struct ebt_setqos_info *)t2->data;

	return (!memcmp(setqosinfo1, setqosinfo2, sizeof(*setqosinfo1)));
}

static struct ebt_u_target setqos_target =
{
	.name		= "setqos",
	.size		= sizeof(struct ebt_setqos_info),
	.help		= print_help,
	.init		= init,
	.parse		= parse,
	.final_check	= final_check,
	.print		= print,
	.compare	= compare,
	.extra_ops	= opts,
};

void _init(void)
{
	ebt_register_target(&setqos_target);
}
