/*
 * Copyright (C) 2013, 2014, 2015, 2017 Cumulus Networks, Inc. All rights reserved
 */

/* ebt_setclass
 *
 * Authors:
 * Bart De Schuymer <bdschuym@pandora.be>
 *
 * April, 2002
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "../include/ebtables_u.h"
#include <linux/netfilter_bridge/ebt_setclass.h>

#define _SETCLASS_CLASS '1'
static struct option opts[] =
{
	{ "class", required_argument, 0, _SETCLASS_CLASS },
	{ 0 }
};

static void print_help()
{
	printf(
	"setclass option:\n"
	" --class : setting data path resource/queuing class\n");
}

static void init(struct ebt_entry_target *target)
{
	struct ebt_setclass_info *setclassinfo =
	   (struct ebt_setclass_info *)target->data;

	setclassinfo->class = 0;
	return;
}

#define OPT_SETCLASS_CLASS   0x01
static int parse(int c, char **argv, int argc,
   const struct ebt_u_entry *entry, unsigned int *flags,
   struct ebt_entry_target **target)
{
	struct ebt_setclass_info *setclassinfo =
	   (struct ebt_setclass_info *)(*target)->data;
	uint32_t mask;
	long int i;
	char *end;

	switch (c) {
	case _SETCLASS_CLASS:
		ebt_check_option2(flags, OPT_SETCLASS_CLASS);
		i = strtol(optarg, &end, 16);
                if (i < EBT_SETCLASS_MIN_CLASS || i > EBT_SETCLASS_MAX_CLASS || *end != '\0')
                        ebt_print_error2("Problem with specified class id");
		setclassinfo->class = i;
		setclassinfo->bitmask |= EBT_SETCLASS_CLASS;
		break;
	default:
		return 0;
	}
	return 1;
}

static void final_check(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target, const char *name,
   unsigned int hookmask, unsigned int time)
{
	struct ebt_setclass_info *setclassinfo =
	   (struct ebt_setclass_info *)target->data;

	if (!(setclassinfo->bitmask & EBT_SETCLASS_CLASS)) {
		ebt_print_error("For SETCLASS , the class id must be specified ");
	}
}

static void print(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target)
{
	struct ebt_setclass_info *setclassinfo =
	   (struct ebt_setclass_info *)target->data;

	if (setclassinfo->bitmask & EBT_SETCLASS_CLASS)
		printf("--class %d ", setclassinfo->class);
}

static int compare(const struct ebt_entry_target *t1,
   const struct ebt_entry_target *t2)
{
	struct ebt_setclass_info *setclassinfo1 =
	   (struct ebt_setclass_info *)t1->data;
	struct ebt_setclass_info *setclassinfo2 =
	   (struct ebt_setclass_info *)t2->data;

	return (!memcmp(setclassinfo1, setclassinfo2, sizeof(*setclassinfo1)));
}

static struct ebt_u_target setclass_target =
{
	.name		= "setclass",
	.size		= sizeof(struct ebt_setclass_info),
	.help		= print_help,
	.init		= init,
	.parse		= parse,
	.final_check	= final_check,
	.print		= print,
	.compare	= compare,
	.extra_ops	= opts,
};

void _init(void)
{
	ebt_register_target(&setclass_target);
}
