/*
 * Copyright (C) 2013, 2014, 2015, 2016, 2017 Cumulus Networks, Inc. All rights reserved
 */

/* ebt_police
 *
 * Authors:
 * Bart De Schuymer <bdschuym@pandora.be>
 *
 * April, 2002
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "../include/ebtables_u.h"
#include <linux/netfilter_bridge/ebt_police.h>

#define _POLICE_CLASS '1'
#define _POLICE_RATE  '2'
#define _POLICE_BURST '3'
#define _POLICE_MODE  '4'

static struct option opts[] =
{
	{ "set-class", required_argument, 0, _POLICE_CLASS },
	{ "set-rate", required_argument, 0, _POLICE_RATE },
	{ "set-burst", required_argument, 0, _POLICE_BURST },
	{ "set-mode", required_argument, 0, _POLICE_MODE },
	{ 0 }
};

static void print_help()
{
	printf(
	"police option:\n"
	" --set-class : setting data path resource/queuing class\n"
	" --set-rate : setting bit rate that traffic is limited to\n"
	" --set-burst : setting number of bytes allowed to arrive sequentially\n"
	" --set-mode : setting the mode in KB (kiloBytes) or pkt (packets\n");
}

static void init(struct ebt_entry_target *target)
{
	struct ebt_police_info *policeinfo =
	   (struct ebt_police_info *)target->data;

	policeinfo->class = 0;
	policeinfo->rate = 0;
	policeinfo->burst = 0;
	policeinfo->mode = EBT_POLICE_MODE_PPS;
	return;
}

#define OPT_POLICE_CLASS   0x01
#define OPT_POLICE_RATE    0x02
#define OPT_POLICE_BURST   0x04
#define OPT_POLICE_MODE    0x08
static int parse(int c, char **argv, int argc,
   const struct ebt_u_entry *entry, unsigned int *flags,
   struct ebt_entry_target **target)
{
	struct ebt_police_info *policeinfo =
	   (struct ebt_police_info *)(*target)->data;
	uint32_t mask;
	long int i;
	char *end;

	switch (c) {
	case _POLICE_CLASS:
		ebt_check_option2(flags, OPT_POLICE_CLASS);
		i = strtol(optarg, &end, 16);
		if (i < EBT_POLICE_MIN_CLASS || i > EBT_POLICE_MAX_CLASS || *end != '\0')
			ebt_print_error2("Problem with specified class id");
		policeinfo->class = i;
		policeinfo->bitmask |= EBT_POLICE_CLASS;
		break;
	case _POLICE_RATE:
		ebt_check_option2(flags, OPT_POLICE_RATE);
		i = strtoul(optarg, &end, 10);
		if (i < EBT_POLICE_MIN_RATE || i > EBT_POLICE_MAX_RATE || *end != '\0')
			ebt_print_error2("Problem with specified rate");
		policeinfo->rate = i;
		policeinfo->bitmask |= EBT_POLICE_RATE;
		break;
	case _POLICE_BURST:
		ebt_check_option2(flags, OPT_POLICE_BURST);
		i = strtoul(optarg, &end, 10);
		if (i < EBT_POLICE_MIN_BURST || i > EBT_POLICE_MAX_BURST || *end != '\0')
			ebt_print_error2("Problem with specified burst");
		policeinfo->burst = i;
		policeinfo->bitmask |= EBT_POLICE_BURST;
		break;
	case _POLICE_MODE:
		ebt_check_option2(flags, OPT_POLICE_MODE);
		if (strncmp(optarg, "pkt", strlen(optarg)) == 0)
			policeinfo->mode = EBT_POLICE_MODE_PPS;
		else if (strncmp(optarg, "KB", strlen(optarg)) == 0)
			policeinfo->mode = EBT_POLICE_MODE_KB;
		else if (strncmp(optarg, "MB", strlen(optarg)) == 0)
			policeinfo->mode = EBT_POLICE_MODE_MB;
		else if (strncmp(optarg, "GB", strlen(optarg)) == 0)
			policeinfo->mode = EBT_POLICE_MODE_GB;
		else
			ebt_print_error2("Problem with specified mode");
		policeinfo->bitmask |= EBT_POLICE_MODE;
		break;
	default:
		return 0;
	}
	return 1;
}

static void final_check(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target, const char *name,
   unsigned int hookmask, unsigned int time)
{
	struct ebt_police_info *policeinfo =
	   (struct ebt_police_info *)target->data;
	if (!(policeinfo->bitmask & EBT_POLICE_BURST) ||
		!(policeinfo->bitmask & EBT_POLICE_RATE) ||
		!(policeinfo->bitmask & EBT_POLICE_MODE)) {
		ebt_print_error("For police target, mode, rate and burst settings are madatory");
	}
}

static void print(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target)
{
	struct ebt_police_info *policeinfo =
	   (struct ebt_police_info *)target->data;

	if (policeinfo->bitmask & EBT_POLICE_MODE)
		if (policeinfo->mode == EBT_POLICE_MODE_PPS)
			printf("--set-mode %s ", "pkt");
		else if (policeinfo->mode == EBT_POLICE_MODE_KB)
			printf("--set-mode %s ", "KB");
		else if (policeinfo->mode == EBT_POLICE_MODE_MB)
			printf("--set-mode %s ", "MB");
		else
			printf("--set-mode %s ", "GB");
	if (policeinfo->bitmask & EBT_POLICE_RATE)
		printf("--set-rate %d ", policeinfo->rate);
	if (policeinfo->bitmask & EBT_POLICE_BURST)
		printf("--set-burst %d ", policeinfo->burst);
	if (policeinfo->bitmask & EBT_POLICE_CLASS)
		printf("--set-class %d ", policeinfo->class);
}

static int compare(const struct ebt_entry_target *t1,
   const struct ebt_entry_target *t2)
{
	struct ebt_police_info *policeinfo1 =
	   (struct ebt_police_info *)t1->data;
	struct ebt_police_info *policeinfo2 =
	   (struct ebt_police_info *)t2->data;

	return (!memcmp(policeinfo1, policeinfo2, sizeof(*policeinfo1)));
}

static struct ebt_u_target police_target =
{
	.name		= "police",
	.size		= sizeof(struct ebt_police_info),
	.help		= print_help,
	.init		= init,
	.parse		= parse,
	.final_check	= final_check,
	.print		= print,
	.compare	= compare,
	.extra_ops	= opts,
};

void _init(void)
{
	ebt_register_target(&police_target);
}
