/*
 * Copyright (C) 2013, 2014, 2015, 2017 Cumulus Networks, Inc. All rights reserved
 */

/* ebt_erspan
 *
 * Authors:
 * Bart De Schuymer <bdschuym@pandora.be>
 *
 * April, 2002
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "../include/ebtables_u.h"
#include <linux/netfilter_bridge/ebt_erspan.h>

#define _ERSPAN_SRC_IP '1'
#define _ERSPAN_DST_IP '2'
#define _ERSPAN_TTL '3'
#define _ERSPAN_PREC '4'
static struct option opts[] =
{
	{ "src-ip", required_argument, 0, _ERSPAN_SRC_IP },
	{ "dst-ip", required_argument, 0, _ERSPAN_DST_IP },
	{ "ttl", required_argument, 0, _ERSPAN_TTL },
	{ "prec", required_argument, 0, _ERSPAN_PREC },
	{ 0 }
};

static void print_help()
{
	printf(
	"erspan option:\n"
	" --src-ip : tunnel source ip address\n"
	" --dst-ip : tunnel destination ip address\n"
	" --ttl : ttl to put on the tunnel encapsulation\n"
	" --prec : ip precedence value to put on tunnel encapsulation\n");
}

static void init(struct ebt_entry_target *target)
{
	struct ebt_erspan_info *erspaninfo =
	   (struct ebt_erspan_info *)target->data;

	erspaninfo->src_ip = 0;
	erspaninfo->dst_ip = 0;
	erspaninfo->ttl = 0;
	erspaninfo->prec = 0;
	return;
}

#define OPT_ERSPAN_SRC_IP  0x01
#define OPT_ERSPAN_DST_IP  0x02
#define OPT_ERSPAN_TTL     0x04
#define OPT_ERSPAN_PREC    0x08
static int parse(int c, char **argv, int argc,
   const struct ebt_u_entry *entry, unsigned int *flags,
   struct ebt_entry_target **target)
{
	struct ebt_erspan_info *erspaninfo =
	   (struct ebt_erspan_info *)(*target)->data;
	uint32_t mask;
	unsigned long i;
	char *end;

	switch (c) {
	case _ERSPAN_SRC_IP:
		ebt_check_option2(flags, OPT_ERSPAN_SRC_IP);
		ebt_parse_ip_address(optarg, &erspaninfo->src_ip, &mask);
		erspaninfo->bitmask |= EBT_ERSPAN_SRC_IP;
		break;
	case _ERSPAN_DST_IP:
		ebt_check_option2(flags, OPT_ERSPAN_DST_IP);
		ebt_parse_ip_address(optarg, &erspaninfo->dst_ip, &mask);
		erspaninfo->bitmask |= EBT_ERSPAN_DST_IP;
		break;
	case _ERSPAN_TTL:
		ebt_check_option2(flags, OPT_ERSPAN_TTL);
		i = strtoul(optarg, &end, 10);
		if (*end != '\0') {
			ebt_print_error2("Problem with specified TTL");
		} else {
			erspaninfo->ttl = (unsigned char) i;
		}
		erspaninfo->bitmask |= EBT_ERSPAN_TTL;
		break;
	case _ERSPAN_PREC:
		ebt_check_option2(flags, OPT_ERSPAN_PREC);
		i = strtoul(optarg, &end, 10);
		if (*end != '\0') {
			ebt_print_error2("Problem with specified Precision");
		} else {
			erspaninfo->prec = (unsigned char) i;
		}
		erspaninfo->bitmask |= EBT_ERSPAN_PREC;
		break;
	default:
		return 0;
	}
	return 1;
}

static void final_check(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target, const char *name,
   unsigned int hookmask, unsigned int time)
{
	struct ebt_erspan_info *erspaninfo =
	   (struct ebt_erspan_info *)target->data;

	if (!(erspaninfo->bitmask & EBT_ERSPAN_SRC_IP) ||
		!(erspaninfo->bitmask & EBT_ERSPAN_DST_IP)) {
		ebt_print_error("For erspan target, source and destination fields are mandatory");
	}

}

static void print(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target)
{
	struct ebt_erspan_info *erspaninfo =
	   (struct ebt_erspan_info *)target->data;

	if (erspaninfo->bitmask & EBT_ERSPAN_SRC_IP) {
		printf("--src_ip %d.%d.%d.%d ",
			((unsigned char *)&erspaninfo->src_ip)[0],
			((unsigned char *)&erspaninfo->src_ip)[1],
			((unsigned char *)&erspaninfo->src_ip)[2],
			((unsigned char *)&erspaninfo->src_ip)[3]);
	}
	if (erspaninfo->bitmask & EBT_ERSPAN_DST_IP) {
		printf("--dst_ip  %d.%d.%d.%d ",
			((unsigned char *)&erspaninfo->dst_ip)[0],
			((unsigned char *)&erspaninfo->dst_ip)[1],
			((unsigned char *)&erspaninfo->dst_ip)[2],
			((unsigned char *)&erspaninfo->dst_ip)[3]);
	}
	if (erspaninfo->bitmask & EBT_ERSPAN_TTL)
		printf("--ttl %d ", erspaninfo->ttl);
	if (erspaninfo->bitmask & EBT_ERSPAN_PREC)
		printf("--prec %d ", erspaninfo->prec);
}

static int compare(const struct ebt_entry_target *t1,
   const struct ebt_entry_target *t2)
{
	struct ebt_erspan_info *erspaninfo1 =
	   (struct ebt_erspan_info *)t1->data;
	struct ebt_erspan_info *erspaninfo2 =
	   (struct ebt_erspan_info *)t2->data;

	return (!memcmp(erspaninfo1, erspaninfo2, sizeof(*erspaninfo1)));
}

static struct ebt_u_target erspan_target =
{
	.name		= "erspan",
	.size		= sizeof(struct ebt_erspan_info),
	.help		= print_help,
	.init		= init,
	.parse		= parse,
	.final_check	= final_check,
	.print		= print,
	.compare	= compare,
	.extra_ops	= opts,
};

void _init(void)
{
	ebt_register_target(&erspan_target);
}
