/* Copyright 2020 Cumulus Networks, Inc.  All rights reserved. */
#include "dhcpsnoop.h"
#include "dhcpsnoop6.h"
#include "dhcp.h"
#include "dhcp6.h"
#include "cJSON.h"
#include "cJSON_Utils.h"
#include <sys/stat.h>
#include <sys/mman.h> 
#include <fcntl.h>
#include <sys/inotify.h>
#include <limits.h>
#include <pthread.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h> 
#include <syslog.h>
#include <regex.h>
#include <dirent.h>
#include <signal.h>

#include <netlink/netlink.h>
#include <netlink/route/addr.h>
#include <netlink/route/link.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/route.h>
#include <netlink/route/link/bcm_knet.h>
#include <netlink/route/link/mlx_sx_netdev.h>


#define ARRAY_SIZE(x) (sizeof(x) / sizeof((x)[0]))

#define DEST_MAC0	0x00
#define DEST_MAC1	0x00
#define DEST_MAC2	0x00
#define DEST_MAC3	0x00
#define DEST_MAC4	0x00
#define DEST_MAC5	0x00

#define ETHER_TYPE	0x0800

#define DEFAULT_IF	"br0"
#define BUF_SIZ		1024
#define BUF_LEN_NOTIFY (10 * (sizeof(struct inotify_event) + NAME_MAX + 1))

const char *conf_file = "/etc/dhcpsnoop/dhcp_snoop.json";
const char *db_file = "/etc/dhcpsnoop/dhcp_table.json";
const char *db_file6 = "/etc/dhcpsnoop/dhcpv6_table.json";

const char *db_bkup_file = "/etc/dhcpsnoop/dhcp_table_bkup.json";
const char *db_bkup_file6 = "/etc/dhcpsnoop/dhcpv6_table_bkup.json";

char *aclcommand = "/usr/cumulus/bin/cl-acltool";
char *args[] = {"/usr/cumulus/bin/cl-acltool", "-i", NULL};

int tcp=0,udp=0,icmp=0,others=0,igmp=0,total=0,i,j;
struct sockaddr_in source,dest;

void ProcessPacket(unsigned char* buffer, int size, uint16_t vlan,
		struct interface_info *if_info, unsigned int rx_ifindex);
size_t receive_packet_snoop (int fd, short ev, void *arg);
void if_register_receive_snoop(struct interface_info *info);
int if_register_lpf_snoop (struct interface_info *info);
static uint8_t get_option_msgtype(unsigned char *opt);
static uint32_t get_option_lease(unsigned char *opt);
static uint32_t get_option_serverid (unsigned char *opt);
void print_configuration_table(hash_table_t *ht);
void print_conf_table_outer();
int is_vlan_aware(char *iface);
void get_port_name_from_mac_vlan_pair(unsigned char * mac, int vlan, 
		char *bridge, char *iface, int size, int vlan_aware);
int print_binding_table_to_file();
int dhcpsnoop_apply_interface_trust_config(char *iface,
	uint32_t vlan, bool vlan_aware);
int dhcpsnoop6_apply_interface_trust_config(char *iface,
	uint32_t vlan, bool vlan_aware);
int run_clacltool();
static uint32_t get_option_requested_ip (unsigned char *opt);
void dhcpsnoop_age_function(int fd, short event, void *arg);
void set_dirty_bridge_table();
void dhcpsnoop_init_syslog(int logcode);
int dhcpsnoop_inotify_init(inotify_event_bundle_t * inot);
int event_reg();
int dhcpsnoop_is_vlan_snooping_enabled(char *ifname, 
	uint32_t vlan,uint32_t ip_version);
void print_dhcp_entry (dhcp_mac_ip_t *tmp_entry);
int ipStringToNumber (const char*  pDottedQuad,uint32_t *    pIpAddr);
int dhcpsnoop_is_bridge_trad(char *bridge_name);
void process_dhcpv6_packet(struct dhcpv6_packet * dh_pkt,
        uint16_t vlan, char *ifName, unsigned int rx_ifindex,uint16_t udp_pkt_len);
/*void PrintData (unsigned char* data , int Size);*/
int print_binding_table_to_file6();
int dhcpsnoop_update_bridge(char *br_name,int vlan, 
	uint32_t rate,uint32_t ip_version);
int dhcpsnoop_bridge_change_cb(int fd, short ev, void *arg);
void dhcpsnoop_sig_handler(int signum);
void dhcpsnoop_delete_binding_entries_vlan(uint32_t vlan);
void dhcpsnoop6_delete_binding_entries_vlan(uint32_t vlan);


hash_table_t *binding_table=NULL;
hash_table_t *binding_table6=NULL;
hash_table_t *master_config_table=NULL;

struct nl_sock *sock=NULL;
struct nl_cache *cache=NULL; 

struct sock_filter code[] = {
	{ 0x28, 0, 0, 0x0000000c },
	{ 0x15, 0, 13, 0x000086dd },
	{ 0x30, 0, 0, 0x00000014 },
	{ 0x15, 0, 5, 0x00000011 },
	{ 0x28, 0, 0, 0x00000036 },
	{ 0x15, 30, 0, 0x00000043 },
	{ 0x15, 29, 0, 0x00000044 },
	{ 0x28, 0, 0, 0x00000038 },
	{ 0x15, 27, 17, 0x00000043 },
	{ 0x15, 1, 0, 0x00000084 },
	{ 0x15, 0, 26, 0x00000006 },
	{ 0x28, 0, 0, 0x00000036 },
	{ 0x15, 23, 0, 0x00000044 },
	{ 0x28, 0, 0, 0x00000038 },
	{ 0x15, 21, 22, 0x00000044 },
	{ 0x15, 0, 21, 0x00000800 },
	{ 0x30, 0, 0, 0x00000017 },
	{ 0x15, 0, 9, 0x00000011 },
	{ 0x28, 0, 0, 0x00000014 },
	{ 0x45, 17, 0, 0x00001fff },
	{ 0xb1, 0, 0, 0x0000000e },
	{ 0x48, 0, 0, 0x0000000e },
	{ 0x15, 13, 0, 0x00000043 },
	{ 0x15, 12, 0, 0x00000044 },
	{ 0x48, 0, 0, 0x00000010 },
	{ 0x15, 10, 0, 0x00000043 },
	{ 0x15, 9, 10, 0x00000044 },
	{ 0x15, 1, 0, 0x00000084 },
	{ 0x15, 0, 8, 0x00000006 },
	{ 0x28, 0, 0, 0x00000014 },
	{ 0x45, 6, 0, 0x00001fff },
	{ 0xb1, 0, 0, 0x0000000e },
	{ 0x48, 0, 0, 0x0000000e },
	{ 0x15, 2, 0, 0x00000044 },
	{ 0x48, 0, 0, 0x00000010 },
	{ 0x15, 0, 1, 0x00000044 },
	{ 0x6, 0, 0, 0x00040000 },
	{ 0x6, 0, 0, 0x00000000 },
};

struct sock_fprog bpf = {
	.len = ARRAY_SIZE(code),
	.filter = code,
};

struct sock_filter code6[] = {
	{ 0x28, 0, 0, 0x0000000c },
	{ 0x15, 0, 14, 0x000086dd },
	{ 0x30, 0, 0, 0x00000014 },
	{ 0x15, 0, 6, 0x00000011 },
	{ 0x28, 0, 0, 0x00000036 },
	{ 0x15, 22, 0, 0x00000222 },
	{ 0x15, 21, 0, 0x00000223 },
	{ 0x28, 0, 0, 0x00000038 },
	{ 0x15, 19, 0, 0x00000222 },
	{ 0x15, 18, 19, 0x00000223 },
	{ 0x15, 1, 0, 0x00000084 },
	{ 0x15, 0, 17, 0x00000006 },
	{ 0x28, 0, 0, 0x00000036 },
	{ 0x15, 14, 0, 0x00000223 },
	{ 0x28, 0, 0, 0x00000038 },
	{ 0x15, 12, 13, 0x00000223 },
	{ 0x15, 0, 12, 0x00000800 },
	{ 0x30, 0, 0, 0x00000017 },
	{ 0x15, 2, 0, 0x00000084 },
	{ 0x15, 1, 0, 0x00000006 },
	{ 0x15, 0, 8, 0x00000011 },
	{ 0x28, 0, 0, 0x00000014 },
	{ 0x45, 6, 0, 0x00001fff },
	{ 0xb1, 0, 0, 0x0000000e },
	{ 0x48, 0, 0, 0x0000000e },
	{ 0x15, 2, 0, 0x00000223 },
	{ 0x48, 0, 0, 0x00000010 },
	{ 0x15, 0, 1, 0x00000223 },
	{ 0x6, 0, 0, 0x00040000 },
	{ 0x6, 0, 0, 0x00000000 },
};

struct sock_fprog bpf6 = {
    .len = ARRAY_SIZE(code6),
    .filter = code6,
};

void parse_dhcp_table(char *mapped)
{
	cJSON *dhcp_table = NULL;
	cJSON *root = cJSON_Parse(mapped);
	cJSON *ifName_cJSON = NULL;
	cJSON *ipadd_cJSON = NULL;
	cJSON *vlan_cJSON = NULL;
	cJSON *mac_cJSON = NULL;
	cJSON *lease_cJSON = NULL;
	cJSON *rx_ifname_cJSON = NULL;
	cJSON *type_cJSON = NULL;
	cJSON *state_cJSON = NULL;
	cJSON *subitem = NULL;
	cJSON *timestamp_cJSON = NULL;

	int t;
	binding_entry_key_t tmp_key;
	dhcp_mac_ip_t *tmp_entry = NULL;
	unsigned char buf[sizeof(struct in_addr)];
	struct in_addr ip_tmp;

	dhcp_table = cJSON_GetObjectItem(root,"dhcp_table");
	if (!dhcp_table) {
		syslog(LOG_ERR,"%s: No dhcp_table structure found\n",
			__FUNCTION__);
		return;
	} 
	for(t=0; t < cJSON_GetArraySize(dhcp_table); t++) {

		subitem = NULL;
		subitem =  cJSON_GetArrayItem(dhcp_table, t);
		if (!subitem) {
			syslog(LOG_ERR,"Not able to parse subitem");
			continue;
		} 

		lease_cJSON = NULL;
		lease_cJSON = cJSON_GetObjectItem(subitem,"Lease");
		if(!lease_cJSON) {
			syslog(LOG_ERR,"Not able to parse lease");
			continue;
		}

		timestamp_cJSON = NULL;
		timestamp_cJSON = cJSON_GetObjectItem(subitem,"Timestamp");
		if(!timestamp_cJSON) {
			syslog(LOG_ERR,"Not able to parse timestamp");
			continue;
		}

		if (time(NULL) > ((long)timestamp_cJSON->valuedouble) + lease_cJSON->valueint) {
			syslog(LOG_INFO,"Older entry skiping");
			continue;
		}

		mac_cJSON = NULL;
		mac_cJSON = cJSON_GetObjectItem(subitem,"MAC");
		if(!mac_cJSON) {
			syslog(LOG_ERR,"Not able to parse mac address");
			continue;
		}

		ipadd_cJSON = NULL;
		ipadd_cJSON = cJSON_GetObjectItem(subitem,"IP");
		if (!ipadd_cJSON) {
			syslog(LOG_ERR,"Not able to parse IP address");
			continue;
		}

		vlan_cJSON = NULL;
		vlan_cJSON = cJSON_GetObjectItem(subitem,"VLAN");
		if (!vlan_cJSON) {
			syslog(LOG_ERR,"Not able to parse VLAN");
			continue;
		}

		ifName_cJSON = NULL;
		ifName_cJSON = cJSON_GetObjectItem(subitem,"Bridge");
		if (!ifName_cJSON) {
			syslog(LOG_ERR,"Not able to parse Bridge");
			continue;
		}

		tmp_key.vlan = (uint32_t) vlan_cJSON->valueint;
		ipStringToNumber(ipadd_cJSON->valuestring,&tmp_key.ip_add);

		strncpy(tmp_key.ifName,ifName_cJSON->valuestring,IFNAMSIZ-1);
		sscanf(mac_cJSON->valuestring,"%x:%x:%x:%x:%x:%x",
				&tmp_key.mac_addr[0],
				&tmp_key.mac_addr[1],
				&tmp_key.mac_addr[2],
				&tmp_key.mac_addr[3],
				&tmp_key.mac_addr[4],
				&tmp_key.mac_addr[5]);

		hash_table_find(binding_table,&tmp_key,
				sizeof(tmp_key),(void **)&tmp_entry);

		if (!tmp_entry) {
			tmp_entry = calloc(1, sizeof(dhcp_mac_ip_t));
			if (!tmp_entry) {
				syslog(LOG_ERR,"Memory allocation failed\n");
				return;
			}

			tmp_entry->key.ip_add = (uint32_t) tmp_key.ip_add;
			tmp_entry->key.vlan = tmp_key.vlan;
			strncpy(tmp_entry->key.ifName,ifName_cJSON->valuestring,IFNAMSIZ-1);
			memcpy(tmp_entry->key.mac_addr,tmp_key.mac_addr,6);

			rx_ifname_cJSON = NULL;
			rx_ifname_cJSON = cJSON_GetObjectItem(subitem,"Port");
			if (rx_ifname_cJSON) {
				strncpy(tmp_entry->rx_ifname,rx_ifname_cJSON->valuestring,
						IFNAMSIZ-1);
			} else {
				syslog(LOG_ERR,"Not able to parse rx_ifname");
			}

			lease_cJSON = NULL;
			lease_cJSON = cJSON_GetObjectItem(subitem,"Lease");
			if(lease_cJSON) {
				tmp_entry->lease = lease_cJSON->valueint;
			} else  {
				syslog(LOG_ERR,"Not able to parse lease");
			}

			state_cJSON  = NULL;
			state_cJSON = cJSON_GetObjectItem(subitem,"State_internal");
			if (state_cJSON) {
				tmp_entry->state = (uint8_t) state_cJSON->valueint;
			} else {
				syslog(LOG_ERR,"Not able to parse state");
			}

			type_cJSON = NULL;
			type_cJSON = cJSON_GetObjectItem(subitem,"Type");
			if (type_cJSON) {
				tmp_entry->type = (uint32_t) type_cJSON->valueint;
			} else {
				syslog(LOG_ERR,"Not able to parse type");
			}

			if (!hash_table_add(binding_table,&tmp_entry->key,
						sizeof(tmp_entry->key),tmp_entry)) {
				return 0;
			}			
		}
	}
	cJSON_Delete(root);
}

void parse_dhcp6_table(char *mapped)
{
	cJSON *dhcp_table = NULL;
	cJSON *root = cJSON_Parse(mapped);
	cJSON *ifName_cJSON = NULL;
	cJSON *ipadd_cJSON = NULL;
	cJSON *vlan_cJSON = NULL;
	cJSON *mac_cJSON = NULL;
	cJSON *lease_cJSON = NULL;
	cJSON *rx_ifname_cJSON = NULL;
	cJSON *type_cJSON = NULL;
	cJSON *state_cJSON = NULL;
	cJSON *subitem = NULL;
	cJSON *timestamp_cJSON = NULL;

	int t;
	binding_entry_key_6_t tmp_key;
	dhcp_mac_ip_6_t *tmp_entry = NULL;
	unsigned char buf[sizeof(struct in_addr)];
	struct in6_addr in6addr;

	dhcp_table = cJSON_GetObjectItem(root,"dhcpv6_table");
	if (!dhcp_table) {
		syslog(LOG_ERR,"%s: No dhcp_table structure found\n",
			__FUNCTION__);
		return;
	} 
	for(t=0; t < cJSON_GetArraySize(dhcp_table); t++) {

		subitem = NULL;
		subitem =  cJSON_GetArrayItem(dhcp_table, t);
		if (!subitem) {
			syslog(LOG_ERR,"Not able to parse subitem");
			continue;
		} 

		lease_cJSON = NULL;
		lease_cJSON = cJSON_GetObjectItem(subitem,"Lease");
		if(!lease_cJSON) {
			syslog(LOG_ERR,"Not able to parse lease");
			continue;
		}

		timestamp_cJSON = NULL;
		timestamp_cJSON = cJSON_GetObjectItem(subitem,"Timestamp");
		if(!timestamp_cJSON) {
			syslog(LOG_ERR,"Not able to parse timestamp");
			continue;
		}

		if (time(NULL) > ((long)timestamp_cJSON->valuedouble) + lease_cJSON->valueint) {
			syslog(LOG_INFO,"Older entry skiping");
			continue;
		}

		mac_cJSON = NULL;
		mac_cJSON = cJSON_GetObjectItem(subitem,"MAC");
		if(!mac_cJSON) {
			syslog(LOG_ERR,"Not able to parse mac address");
			continue;
		}

		ipadd_cJSON = NULL;
		ipadd_cJSON = cJSON_GetObjectItem(subitem,"IP");
		if (!ipadd_cJSON) {
			syslog(LOG_ERR,"Not able to parse IP address");
			continue;
		}

		vlan_cJSON = NULL;
		vlan_cJSON = cJSON_GetObjectItem(subitem,"VLAN");
		if (!vlan_cJSON) {
			syslog(LOG_ERR,"Not able to parse VLAN");
			continue;
		}

		ifName_cJSON = NULL;
		ifName_cJSON = cJSON_GetObjectItem(subitem,"Bridge");
		if (!ifName_cJSON) {
			syslog(LOG_ERR,"Not able to parse Bridge");
			continue;
		}

		tmp_key.vlan = (uint32_t) vlan_cJSON->valueint;
		inet_pton(AF_INET6, ipadd_cJSON->valuestring, &tmp_key.ip_add);

		strncpy(tmp_key.ifName,ifName_cJSON->valuestring,IFNAMSIZ-1);
		sscanf(mac_cJSON->valuestring,"%x:%x:%x:%x:%x:%x",
				&tmp_key.mac_addr[0],
				&tmp_key.mac_addr[1],
				&tmp_key.mac_addr[2],
				&tmp_key.mac_addr[3],
				&tmp_key.mac_addr[4],
				&tmp_key.mac_addr[5]);

		hash_table_find(binding_table6,&tmp_key,
				sizeof(tmp_key),(void **)&tmp_entry);

		if (!tmp_entry) {
			tmp_entry = calloc(1, sizeof(dhcp_mac_ip_6_t));
			if (!tmp_entry) {
				syslog(LOG_ERR,"Memory allocation failed\n");
				return;
			}

			memcpy(&tmp_entry->key.ip_add, &tmp_key.ip_add, sizeof(struct in6_addr));
			tmp_entry->key.vlan = tmp_key.vlan;

			strncpy(tmp_entry->key.ifName,ifName_cJSON->valuestring,IFNAMSIZ-1);
			memcpy(tmp_entry->key.mac_addr,tmp_key.mac_addr,6);

			rx_ifname_cJSON = NULL;
			rx_ifname_cJSON = cJSON_GetObjectItem(subitem,"Port");
			if (rx_ifname_cJSON) {
				strncpy(tmp_entry->rx_ifname,rx_ifname_cJSON->valuestring,
						IFNAMSIZ-1);
			} else {
				syslog(LOG_ERR,"Not able to parse rx_ifname");
			}

			lease_cJSON = NULL;
			lease_cJSON = cJSON_GetObjectItem(subitem,"Lease");
			if(lease_cJSON) {
				tmp_entry->lease = lease_cJSON->valueint;
			} else  {
				syslog(LOG_ERR,"Not able to parse lease");
			}

			state_cJSON  = NULL;
			state_cJSON = cJSON_GetObjectItem(subitem,"State_internal");
			if (state_cJSON) {
				tmp_entry->state = (uint8_t) state_cJSON->valueint;
			} else {
				syslog(LOG_ERR,"Not able to parse state");
			}

			type_cJSON = NULL;
			type_cJSON = cJSON_GetObjectItem(subitem,"Type");
			if (type_cJSON) {
				tmp_entry->type = (uint32_t) type_cJSON->valueint;
			} else {
				syslog(LOG_ERR,"Not able to parse type");
			}

			if (!hash_table_add(binding_table6,&tmp_entry->key,
						sizeof(tmp_entry->key),tmp_entry)) {
				return 0;
			}			
		}
	}
	cJSON_Delete(root);
}

int parse_object(cJSON *root)
{
	cJSON* vlan = NULL;
	cJSON* subitem = NULL;
	cJSON* vlan_id = NULL;
	cJSON* bridge_id = NULL;
	cJSON* trusted_interface = NULL;
	cJSON* untrusted_interface = NULL;
	cJSON* trust_subitem = NULL;
	cJSON* untrust_subitem = NULL;
	cJSON* snoop_cJSON = NULL;
	cJSON* ip_cJSON = NULL;
	cJSON* rate_cJSON = NULL;
	cJSON* vlan_cJSON = NULL;
	cJSON* bridge_cJSON = NULL;  
	cJSON* bridge = NULL;
	cJSON* br_subitem = NULL;
	cJSON* vlan_aware_cJSON = NULL;
	char del_command[FILENAME];

	int i,t;
	int vlan_aware=0;
	bridge_key_t br_key;
	bridge_t * br_entry=NULL;
	vlan_t *temp_config_entry = NULL;
	vlan_key_t temp_config_key;
	interface_t *temp_iface_entry;
	interface_key_t temp_iface_key;

	uint32_t snooping=0;
	uint32_t rate_limit=0;
	uint32_t ip_version=0;
	bridge = cJSON_GetObjectItem(root,"bridge");
	if (!bridge) {
		syslog(LOG_ERR,"No bridge structure found\n");
		return -1;
	}    
	system("rm /etc/cumulus/acl/policy.d/dhcpsnoop*");

	for(t=0; t < cJSON_GetArraySize(bridge); t++) {
		br_subitem = NULL;
		br_subitem =  cJSON_GetArrayItem(bridge, t);
		if (!br_subitem) {
			syslog(LOG_ERR,"Failed to get subitem");
			continue;
		}
		bridge_cJSON = NULL;
		bridge_cJSON = cJSON_GetObjectItem(br_subitem, "bridge_id");
		if (!bridge_cJSON) {
			syslog(LOG_ERR,"Failed to get bridgeid");
			continue;
		}
		vlan_aware = is_vlan_aware(bridge_cJSON->valuestring); 

		br_key.hash = hash_str(bridge_cJSON->valuestring);  

		hash_table_find(master_config_table, &br_key,
				sizeof(br_key),(void **)&br_entry);

		if (br_entry) {
			br_entry->dirty = false;
			strncpy(br_entry->if_info.name,bridge_cJSON->valuestring,IFNAMSIZ-1);
			br_entry->if_info.name[IFNAMSIZ-1] = '\0';
			br_entry->if_info.index = if_nametoindex(bridge_cJSON->valuestring);

			strncpy(br_entry->iface,bridge_cJSON->valuestring,IFNAMSIZ-1);
			br_entry->iface[IFNAMSIZ-1] = '\0';
			br_entry->vlan_aware = vlan_aware;
		} else {
			br_entry = NULL;
			br_entry = calloc(1, sizeof(bridge_t));
			if (!br_entry) {
				syslog(LOG_ERR,"Mem allocation failed\n");
				continue;
			}

			br_entry->dirty = false;

			strncpy(br_entry->if_info.name,bridge_cJSON->valuestring,IFNAMSIZ-1);
			br_entry->if_info.name[IFNAMSIZ-1] = '\0';
			br_entry->if_info.index = if_nametoindex(bridge_cJSON->valuestring);

			strncpy(br_entry->iface,bridge_cJSON->valuestring,IFNAMSIZ-1);
			br_entry->iface[IFNAMSIZ-1] = '\0';
			memcpy(&br_entry->br_key,&br_key,sizeof(bridge_key_t));

			br_entry->vlan_aware = vlan_aware;

			br_entry->vlan_table = hash_table_alloc(MAX_IFACE);

			if (!hash_table_add(master_config_table,&br_entry->br_key,
						sizeof(br_entry->br_key),br_entry)) {
				syslog(LOG_ERR,"Failed to add  bridge entry to hash table");
				continue;
			}            
		}
		vlan = cJSON_GetObjectItem(br_subitem, "vlan");
		if (!vlan) {
			syslog(LOG_ERR,"No VLAN structure found\n");
			return -1;
		}
		for (i = 0 ; i < cJSON_GetArraySize(vlan) ; i++)
		{
			cJSON * subitem = cJSON_GetArrayItem(vlan, i);
			temp_config_entry = NULL;

			vlan_cJSON = NULL;
			vlan_cJSON = cJSON_GetObjectItem(subitem, "vlan_id");
			if(!vlan_cJSON) {
				syslog(LOG_ERR,"No VLAN id found");
				continue;
			}
			ip_cJSON = NULL;
			ip_cJSON = cJSON_GetObjectItem(subitem, "ip_version");
			if (!ip_cJSON) {
				syslog(LOG_ERR,"ip_version not found");
				continue;
			} else {
				ip_version = ip_cJSON->valueint;
			}

			temp_config_key.vlan = vlan_cJSON->valueint;
			syslog(LOG_DEBUG,"VLAN %u\n",vlan_cJSON->valueint); 

			temp_config_key.ip_version = ip_version;
			syslog(LOG_INFO,"ip_version %d\n",ip_version);
			hash_table_find(br_entry->vlan_table, &temp_config_key,
					sizeof(temp_config_key),(void **)&temp_config_entry);

			snooping = 0;
			snoop_cJSON = NULL;
			snoop_cJSON = cJSON_GetObjectItem(subitem, "snooping");
			if (!snoop_cJSON) {
				syslog(LOG_ERR,"No snooping value found");
			} else {
				snooping = snoop_cJSON->valueint;
			}
			syslog(LOG_INFO,"Snooping %d\n",snooping);

			rate_limit = 0;
			rate_cJSON = NULL;
			rate_cJSON = cJSON_GetObjectItem(subitem, "rate_limit");
			if (!rate_cJSON) {
				syslog(LOG_ERR,"No snooping rate_limit found");
			} else {
				rate_limit = rate_cJSON->valueint;
			}
			syslog(LOG_INFO,"Rate limit %d\n",rate_limit);

			if (temp_config_entry) {
				syslog(LOG_INFO,"Existing\n");
				temp_config_entry->snooping  = snooping;
				temp_config_entry->rate_limit = rate_limit;
				temp_config_entry->dirty = false;

			} else {
				syslog(LOG_INFO,"New entry\n");
				temp_config_entry = calloc(1,sizeof(vlan_t));
				if (temp_config_entry == NULL) {
					continue;
				}	
				temp_config_entry->dirty = false;
				temp_config_entry->key.vlan = temp_config_key.vlan;        
				temp_config_entry->snooping  = snooping;
				temp_config_entry->rate_limit = rate_limit;
				temp_config_entry->key.ip_version = ip_version;
				//temp_config_entry->trusted_iface = hash_table_alloc(MAX_IFACE);
				//temp_config_entry->untrusted_iface = hash_table_alloc(MAX_IFACE);
				memcpy(&temp_config_entry->key,&temp_config_key,sizeof(vlan_key_t));
				if (!hash_table_add(br_entry->vlan_table,&temp_config_entry->key,
							sizeof(temp_config_entry->key),temp_config_entry)) {
					continue;
				}

			}
			if (!dhcpsnoop_update_bridge(br_entry->iface,temp_config_key.vlan,
						rate_limit,temp_config_key.ip_version)) {
				syslog(LOG_ERR,"Failed to update bridge");
			}

			trusted_interface = cJSON_GetObjectItem(subitem,"trusted_interface");
			if (trusted_interface) {
				for (j=0; j < cJSON_GetArraySize(trusted_interface); j++) {
					memset(&temp_iface_key,0,sizeof(temp_iface_key));
					temp_iface_entry = NULL;
					trust_subitem = cJSON_GetArrayItem(trusted_interface, j);
					if (ip_version == 4) {
						snprintf(del_command,FILENAME-1,"rm /etc/cumulus/acl/policy.d/dhcpsnoop_%s_%d.rules",
								trust_subitem->valuestring,temp_config_key.vlan);
						del_command[FILENAME-1] = '\0';
						system(del_command);
					} else if (ip_version == 6) {
						snprintf(del_command,FILENAME-1,"rm /etc/cumulus/acl/policy.d/dhcpsnoop6_%s_%d.rules",
								trust_subitem->valuestring,temp_config_key.vlan);
						del_command[FILENAME-1] = '\0';
						system(del_command);
					}
				}
			}
		}
	}
	if (run_clacltool() == -1) {
		sleep(2);
		syslog(LOG_ERR,"%s: Failed to run cl-acltool -i",__FUNCTION__);
		run_clacltool();
	}
	return 0;
}

int dhcp_snoop_parse(char *mapped) {
	int retval = 0;
	if (mapped) {
		set_dirty_bridge_table(); 
		cJSON *root = NULL;
		root = cJSON_Parse(mapped);
		if (!root) {
			syslog(LOG_ERR,"Unable to parse the configuration");
			return -1;
		}
		retval = parse_object(root);
		cJSON_Delete(root);

		delete_dirty_bridge_table();
		print_conf_table_outer();	
	} else {
		syslog(LOG_ERR,"%s: Unable to read configuration file",
			__FUNCTION__);
		return -1;
	}
	return retval;
}

int dhcp_snoop_open_config() {
	int fd;
	struct stat s;
	int status;
	size_t size;
	char * mapped=NULL;

	fd = open (conf_file, O_RDONLY);
	if (fd < 0) {
		return -1;
	}
	/* Get the size of the file. */
	status = fstat (fd, &s);
	if (status < 0) {
                close(fd);
		return -1;
	}
	size = s.st_size; 

	mapped = mmap (0, size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (mapped == MAP_FAILED) {
                close(fd);
		return -1;
	}
	if (dhcp_snoop_parse(mapped) != 0) {
		status = -1;
	}
	if (munmap(mapped, size) == -1)
	{
		syslog(LOG_ERR,"%s Unamp failed",__FUNCTION__);
		close(fd);
		exit(EXIT_FAILURE);
	}
	close(fd);
	return status;
}

int dhcpsnoop_open_dhcp_table() {

	int fd;
	struct stat s;
	int status;
	size_t size;
	char * mapped=NULL;
	char * del_command[FILENAME];

	fd = open (db_bkup_file, O_RDONLY);
	if (fd < 0) {
		return -1;
	}
	/* Get the size of the file. */
	status = fstat (fd, &s);
	if (status < 0) {
		syslog(LOG_ERR,"%s Failed to get file size",
			__FUNCTION__);
                close(fd);
		return -1;
	}
	size = s.st_size;

	mapped = mmap (0, size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (mapped == MAP_FAILED) {
		syslog(LOG_ERR,"%s Map failed",__FUNCTION__);
                close(fd);
		return -1;
	}
	parse_dhcp_table(mapped);

	if (munmap(mapped, size) == -1)
	{
		syslog(LOG_ERR,"%s Unmap failed",__FUNCTION__);
		close(fd);
		exit(EXIT_FAILURE);
	}
	close(fd);
	snprintf(del_command,FILENAME-1,
		"rm -rf %s",db_bkup_file);
	del_command[FILENAME-1] = '\0';
	system(del_command);
	print_binding_table_to_file();
	return 0;
}
/*
#define	LOG_EMERG	0	system is unusable 
#define	LOG_ALERT	1	action must be taken immediately 
#define	LOG_CRIT	2	critical conditions 
#define	LOG_ERR		3   error conditions 
#define	LOG_WARNING	4	warning conditions 
#define	LOG_NOTICE	5	normal but significant condition
#define	LOG_INFO	6	informational 
#define	LOG_DEBUG	7	debug-level messages
*/

int dhcpsnoop_log_code_parse(char *logdata, size_t len) {
	int log_code=0;
	if (strstr(logdata,"DEBUG") != NULL) {
		log_code = LOG_DEBUG;
	} else if (strstr(logdata,"INFO") != NULL) {
		log_code = LOG_INFO;
	} else if (strstr(logdata,"NOTICE") != NULL) {
		log_code = LOG_NOTICE;
	} else if (strstr(logdata,"WARNING") != NULL) {
		log_code = LOG_WARNING;
	} else if (strstr(logdata,"ERR") != NULL) {
		log_code = LOG_ERR;
	} else if (strstr(logdata,"CRIT") != NULL) {
		log_code = LOG_CRIT;
	} else if (strstr(logdata,"ALERT") != NULL) {
		log_code = LOG_ALERT;
	} else if (strstr(logdata,"EMERG") != NULL) {
		log_code = LOG_EMERG;
	} else {
		log_code = LOG_INFO;
	}
	dhcpsnoop_init_syslog(log_code);	
}

int dhcpsnoop_open_log_conf() {

	int fd;
	struct stat s;
	int status;
	size_t size;
	char * mapped=NULL;
	char * log_conf = "/etc/dhcpsnoop/syslog.conf";

	fd = open (log_conf, O_RDONLY);
	if (fd < 0) {
		return -1;
	}
	/* Get the size of the file. */
	status = fstat (fd, &s);
	if (status < 0) {
		syslog(LOG_ERR,"%s Failed to get file size",
				__FUNCTION__);
                close(fd);
		return -1;
	}
	size = s.st_size;

	mapped = mmap (0, size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (mapped == MAP_FAILED) {
		syslog(LOG_ERR,"%s Map failed",__FUNCTION__);
                close(fd);
		return -1;
	}
	dhcpsnoop_log_code_parse(mapped,size);

	if (munmap(mapped, size) == -1)
	{
		syslog(LOG_ERR,"%s Unmap failed",__FUNCTION__);
		close(fd);
		exit(EXIT_FAILURE);
	}
	close(fd);
	return 0;
}

int dhcpsnoop6_open_dhcp_table() {

	int fd;
	struct stat s;
	int status;
	size_t size;
	char * mapped=NULL;
	char * del_command[FILENAME];

	fd = open (db_bkup_file6, O_RDONLY);
	if (fd < 0) {
		return -1;
	}
	/* Get the size of the file. */
	status = fstat (fd, &s);
	if (status < 0) {
		syslog(LOG_ERR,"%s Failed to get file size",
				__FUNCTION__);
                close(fd);
		return -1;
	}
	size = s.st_size;

	mapped = mmap (0, size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (mapped == MAP_FAILED) {
		syslog(LOG_ERR,"%s Map failed",__FUNCTION__);
                close(fd);
		return -1;
	}
	parse_dhcp6_table(mapped);

	if (munmap(mapped, size) == -1)
	{
		syslog(LOG_ERR,"%s Unmap failed",__FUNCTION__);
		close(fd);
		exit(EXIT_FAILURE);
	}
	close(fd);
	snprintf(del_command,FILENAME-1,"rm -rf %s",db_bkup_file6);
	del_command[FILENAME-1] = '\0';
	system(del_command);
	print_binding_table_to_file6();
	return 0;
}

int dhcp_snoop_init() {
	binding_table = hash_table_alloc(DHCP_SNOOP_BINDING_TBL_SIZE);
	binding_table6 = hash_table_alloc(DHCP_SNOOP_BINDING_TBL_SIZE);
	master_config_table = hash_table_alloc(DHCP_SNOOP_CONFIG_TBL_SIZE);

	if (dhcp_snoop_open_config() != 0) {
		syslog(LOG_ERR,"Failed to open config file\n");
		return -1;
	}
	dhcpsnoop_open_dhcp_table();
	dhcpsnoop6_open_dhcp_table();
	return 0;
}

static void	displayInotifyEvent(struct inotify_event *i)
{
	if (i->cookie > 0)
		syslog(LOG_DEBUG,"cookie =%4d; ", i->cookie);

	if (i->mask & IN_ACCESS)        syslog(LOG_DEBUG,"IN_ACCESS ");
	if (i->mask & IN_ATTRIB)        syslog(LOG_DEBUG,"IN_ATTRIB ");
	if (i->mask & IN_CLOSE_NOWRITE) syslog(LOG_DEBUG,"IN_CLOSE_NOWRITE ");
	if (i->mask & IN_CLOSE_WRITE) {
		if (dhcp_snoop_open_config() != 0) {
			syslog(LOG_ERR,"Failed to open config file\n");
			return 1;
		}
		event_reg();
		syslog(LOG_DEBUG,"IN_CLOSE_WRITE ");       
	}
	if (i->mask & IN_CREATE)        syslog(LOG_DEBUG,"IN_CREATE ");
	if (i->mask & IN_DELETE)        syslog(LOG_DEBUG,"IN_DELETE ");
	if (i->mask & IN_DELETE_SELF)   syslog(LOG_DEBUG,"IN_DELETE_SELF ");
	if (i->mask & IN_IGNORED)       syslog(LOG_DEBUG,"IN_IGNORED ");

	if (i->mask & IN_ISDIR && i->mask & IN_CLOSE_NOWRITE)	{
		syslog(LOG_DEBUG,"Bridge update");
		sleep(2);
		if (dhcp_snoop_open_config() != 0) {
			syslog(LOG_ERR,"Failed to open config file\n");
			return 1;
		}
		event_reg();
	}
	if (i->mask & IN_MODIFY) {
		if (dhcp_snoop_open_config() != 0) {
			syslog(LOG_ERR,"Failed to open config file\n");
			return 1;
		}
		event_reg();
		syslog(LOG_DEBUG,"IN_MODIFY ");
	}
	if (i->mask & IN_MOVE_SELF)     syslog(LOG_DEBUG,"IN_MOVE_SELF ");
	if (i->mask & IN_MOVED_FROM)    syslog(LOG_DEBUG,"IN_MOVED_FROM ");
	if (i->mask & IN_MOVED_TO)      syslog(LOG_DEBUG,"IN_MOVED_TO ");
	if (i->mask & IN_OPEN)          syslog(LOG_DEBUG,"IN_OPEN ");
	if (i->mask & IN_Q_OVERFLOW)    syslog(LOG_DEBUG,"IN_Q_OVERFLOW ");
	if (i->mask & IN_UNMOUNT)       syslog(LOG_DEBUG,"IN_UNMOUNT ");

	if (i->len > 0)
		syslog(LOG_DEBUG,"        name = %s\n", i->name);
}

int check_config_change(int fd, short ev, void *arg) {
	char *p;
	ssize_t numRead;
	int j;
	char notify_buf[BUF_LEN_NOTIFY] __attribute__ ((aligned(8)));
	struct inotify_event *event;
	inotify_event_bundle_t * inot = (inotify_event_bundle_t *)arg;

	numRead = read(fd, notify_buf, BUF_LEN_NOTIFY);

	for (p = notify_buf; p < notify_buf + numRead; ) {
		event = (struct inotify_event *) p;
		displayInotifyEvent(event);
		if (event->mask & IN_DELETE_SELF || 
				event->mask & IN_MOVE_SELF || 
				event->mask & IN_IGNORED) {
			inotify_rm_watch(fd,inot->wd);
			close(fd);
			dhcpsnoop_inotify_init(inot);
			return 0;          
		}
		p += sizeof(struct inotify_event) + event->len;
	}
	return 0;
}

int dhcp_snoop_master_reg() {
	int dhcp_snoop_master_reg_wrapper(void *data, void *arg) {
		bridge_t * tmp = (bridge_t *) data;
		if (tmp->if_info.rfdesc && tmp->if_info.rfdesc6) {
			syslog(LOG_DEBUG,"Socket exists\n");
		} else {
			if_register_receive_snoop(&tmp->if_info);
		}
		return hash_table_foreach_done;
	}
	hash_table_foreach(master_config_table,dhcp_snoop_master_reg_wrapper,NULL);	
}

int event_reg() { 
	dhcp_snoop_master_reg();   
	int event_reg_wrapper(void *data, void *arg) {
		bridge_t * tmp = (bridge_t*) data;
		event_del(&tmp->if_info.br_event);
		event_set(&tmp->if_info.br_event, tmp->if_info.rfdesc, 
				EV_READ|EV_PERSIST, receive_packet_snoop, &tmp->if_info);
		event_add(&tmp->if_info.br_event, NULL);

        event_del(&tmp->if_info.br_event6);
        event_set(&tmp->if_info.br_event6, tmp->if_info.rfdesc6,
                EV_READ|EV_PERSIST, receive_packet_snoop, &tmp->if_info);
        event_add(&tmp->if_info.br_event6, NULL);
		dhcpsnoop_inotify_brchange(&tmp->br_change,tmp->if_info.name);
		return hash_table_foreach_done;
	}
	hash_table_foreach(master_config_table,event_reg_wrapper,NULL);
}

int dhcpsnoop_age_event_reg(struct event *ev, struct timeval *tv, int *timeout) {
	tv->tv_sec = MINUTE; //Minute
	tv->tv_usec = 0;
	event_set(ev, 0, EV_PERSIST, dhcpsnoop_age_function, timeout);
	evtimer_add(ev, tv);
	return 0;
}

int dhcpsnoop_inotify_brchange(inotify_event_bundle_t * inot,char *br_name) {

	char dir_name[FILENAME];

	event_del(&inot->notify_ev);
	inot->inotifyFd = inotify_init();
	if (inot->inotifyFd == -1) {
		syslog(LOG_ERR,"inotify_init_failed\n");
		return -1;
	}
	syslog(LOG_INFO,"Inside %s",__FUNCTION__);
	snprintf(dir_name,FILENAME-1,"/sys/class/net/%s/brif",br_name);
	dir_name[FILENAME-1] = '\0';

	inot->wd = inotify_add_watch(inot->inotifyFd,dir_name,IN_ALL_EVENTS);
	if (inot->wd == -1) {
		syslog(LOG_ERR,"Failed to add watch\n");
		return -1;
	}

	int flags = fcntl(inot->inotifyFd, F_GETFL, 0);
	fcntl(inot->inotifyFd, F_SETFL, flags | O_NONBLOCK);

	strncpy(inot->br_name,br_name,IFNAMSIZ-1);
	inot->br_name[IFNAMSIZ-1] = '\0';

	event_set(&inot->notify_ev,inot->inotifyFd, EV_READ|EV_PERSIST,
			dhcpsnoop_bridge_change_cb,(void *) inot);
	event_add(&inot->notify_ev,NULL);
	return 0;
}

int dhcpsnoop_inotify_init(inotify_event_bundle_t * inot) {

	event_del(&inot->notify_ev);
	inot->inotifyFd = inotify_init();
	if (inot->inotifyFd == -1) {
		syslog(LOG_ERR,"inotify_init_failed\n");
		return -1;
	}

	inot->wd = inotify_add_watch(inot->inotifyFd,conf_file,IN_ALL_EVENTS);
	if (inot->wd == -1) {
		syslog(LOG_ERR,"Failed to add watch\n");
		return -1;
	}

	int flags = fcntl(inot->inotifyFd, F_GETFL, 0);
	fcntl(inot->inotifyFd, F_SETFL, flags | O_NONBLOCK);

	event_set(&inot->notify_ev,inot->inotifyFd, EV_READ|EV_PERSIST,
		check_config_change,(void *) inot);
	event_add(&inot->notify_ev,NULL); 

	return 0;
}

int dhcpsnoop_echo_dhcp_to_cpu(int enable) {
	if (enable) {
		system("/bin/echo TRUE > /cumulus/switchd/config/dhcp_snoop/dhcp_to_cpu");	
	} else {
		system("/bin/echo FALSE > /cumulus/switchd/config/dhcp_snoop/dhcp_to_cpu");
	}
}

int dhcp_signal_register() 
{
	signal(SIGABRT,dhcpsnoop_sig_handler);
	signal(SIGSEGV,dhcpsnoop_sig_handler);
	signal(SIGTERM,dhcpsnoop_sig_handler);
	signal(SIGQUIT,dhcpsnoop_sig_handler);
	signal(SIGKILL,dhcpsnoop_sig_handler);
	signal(SIGHUP,dhcpsnoop_sig_handler);
}

int main(int argc, char *argv[]) {

	inotify_event_bundle_t inot;
	memset(&inot,0,sizeof(inotify_event_bundle_t));
	event_init();

	dhcpsnoop_init_syslog(LOG_INFO);
	dhcpsnoop_open_log_conf();

	if (dhcp_snoop_init() != 0) {
		syslog(LOG_ERR,"Failed to load configuration\n");
		return 1;
	}
	dhcpsnoop_echo_dhcp_to_cpu(1);

	if (dhcpsnoop_inotify_init(&inot) != 0) {
		syslog(LOG_ERR,"Inotify init for config file failed");
		return 1;
	}
	dhcp_signal_register();
	event_reg();
	event_dispatch();
cleanup:
	inotify_rm_watch(inot.inotifyFd,inot.wd);
	close(inot.inotifyFd);
	return 0;
}

void ProcessPacket(unsigned char* buf, int size, uint16_t vlan,
		struct interface_info *if_info,unsigned int rx_ifindex)
{
	struct ethhdr *eh = (struct ethhdr *) buf;
	uint32_t ip_proto = 0;
	if (ntohs(eh->h_proto) == ETHERTYPE_IP) {
		struct iphdr *iph = (struct iphdr *) (buf + sizeof(struct ethhdr));
		struct udphdr *udph = (struct udphdr *) (buf + sizeof(struct iphdr) + sizeof(struct ethhdr));
		ip_proto = iph->protocol;
	} else if (ntohs(eh->h_proto) == ETHERTYPE_IPv6) {
		struct ipv6hdr *iph6 = (struct ipv6hdr *) (buf + sizeof(struct ethhdr));
		struct udphdr *udph = (struct udphdr *) (buf + sizeof(struct ipv6hdr) + sizeof(struct ethhdr));
		ip_proto = iph6->nexthdr;
	} 

	switch (ip_proto)
	{		
		case 6:
			++tcp;
			break;

		case 17:
			++udp;
			print_udp_packet(buf,size,vlan,if_info,rx_ifindex);
			break;

		default: 
			++others;
			break;
	}
}
void print_binding_table() {
	syslog(LOG_DEBUG,"Port	VLAN	IP Address		MAC		TYPE	LEASE	STATE	BRIDGE_NAME\n");
	syslog(LOG_DEBUG,"=====================================================================================\n");
	int print_binding_table_wrapper(void *data,void *arg) {
		struct in_addr ip_addr;
		dhcp_mac_ip_t *tmp_entry = (dhcp_mac_ip_t *) data;
		ip_addr.s_addr = tmp_entry->key.ip_add;
		syslog(LOG_DEBUG,"%s	%d	%s	"
				"%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x"
				"	%d	%d	%d  %s\n",
				tmp_entry->rx_ifname,tmp_entry->key.vlan, 
				inet_ntoa(ip_addr),
				tmp_entry->key.mac_addr[0],
				tmp_entry->key.mac_addr[1],
				tmp_entry->key.mac_addr[2],
				tmp_entry->key.mac_addr[3],
				tmp_entry->key.mac_addr[4],
				tmp_entry->key.mac_addr[5],
				0, tmp_entry->lease,
				tmp_entry->state,
				tmp_entry->key.ifName);
		return hash_table_foreach_done;
	}
	hash_table_foreach(binding_table,print_binding_table_wrapper,NULL);
}

void print_interfaces(hash_table_t *ht) {
	int print_config_interface_wrapper(void *data, void *arg) {
		interface_t * tmp = (interface_t*) data;
		syslog(LOG_INFO,"	%s    %u    %d",tmp->iface,tmp->key.hash,tmp->dirty);
	}
	hash_table_foreach(ht,print_config_interface_wrapper,NULL);
}

void print_configuration_table(hash_table_t *ht) {
	syslog(LOG_INFO,"VLAN   RATE 	IP_Version 		SNOOPING\n");
	syslog(LOG_INFO,"=====================================================================\n");
	int print_config_table_wrapper(void *data, void *arg) {
		vlan_t *temp_config_entry = (vlan_t *) data;
		syslog(LOG_INFO,"%d    %d    %d    %d    %d\n",
				temp_config_entry->key.vlan,
				temp_config_entry->rate_limit,
				temp_config_entry->key.ip_version,
				temp_config_entry->snooping,
				temp_config_entry->dirty);
		syslog(LOG_INFO,"Trusted Interfaces\n");
		syslog(LOG_INFO,"\nUnTrusted Interfaces\n");
		return hash_table_foreach_done; 
	}
	hash_table_foreach(ht,print_config_table_wrapper,NULL);
}

void print_conf_table_outer() {
	syslog(LOG_INFO,"Bridge ID	VLAN aware\n");
	syslog(LOG_INFO,"=================================================================\n");
	int print_conf_table_outer_wrapper(void *data, void *arg) {
		bridge_t *br_entry = (bridge_t*) data;
		syslog(LOG_INFO,"%s    %s    %d    %d\n",br_entry->if_info.name,
				br_entry->iface,br_entry->vlan_aware,br_entry->dirty);
		print_configuration_table(br_entry->vlan_table);
		return hash_table_foreach_done;
	}
	hash_table_foreach(master_config_table,print_conf_table_outer_wrapper,NULL);
}

void set_dirty_interfaces(hash_table_t *ht) {
	int set_dirty_interfaces_wrapper(void *data, void *arg) {
		interface_t * tmp = (interface_t*) data;
		tmp->dirty = true;
	}
	hash_table_foreach(ht,set_dirty_interfaces_wrapper,NULL);
}
void set_dirty_vlan_table(hash_table_t *ht) {
	int set_dirty_vlan_table_wrapper(void *data, void *arg) {
		vlan_t *temp_config_entry = (vlan_t *) data;
		temp_config_entry->dirty = true;
		return hash_table_foreach_done;
	}
	hash_table_foreach(ht,set_dirty_vlan_table_wrapper,NULL);
}

void set_dirty_bridge_table() {
	int set_dirty_bridge_table_wrapper(void *data, void *arg) {
		bridge_t *br_entry = (bridge_t*) data;
		br_entry->dirty = true;
		set_dirty_vlan_table(br_entry->vlan_table);
		return hash_table_foreach_done;
	}
	hash_table_foreach(master_config_table,set_dirty_bridge_table_wrapper,NULL);
}


void delete_dirty_interfaces(hash_table_t *ht) {
	int delete_dirty_interfaces_wrapper(void *data, void *arg) {
		interface_t * tmp = (interface_t*) data;
		if (tmp->dirty == true) {
			syslog(LOG_DEBUG,"%s Inside delete",__FUNCTION__);
			hash_table_delete(ht,&tmp->key,sizeof(tmp->key),NULL);
			if (tmp) {
				free(tmp);
			}
		}
	}
	hash_table_foreach(ht,delete_dirty_interfaces_wrapper,NULL);
}


void  delete_dirty_vlan_table(hash_table_t *ht) {
	int delete_dirty_vlan_table_wrapper(void *data, void *arg) {
		vlan_t *temp_config_entry = (vlan_t *) data;
		if (temp_config_entry->dirty == true) {
			syslog(LOG_DEBUG,"%s Inside delete",__FUNCTION__);
			if (temp_config_entry->key.ip_version == 4) { 
				dhcpsnoop_delete_binding_entries_vlan(temp_config_entry->key.vlan);
			} else {
				dhcpsnoop6_delete_binding_entries_vlan(temp_config_entry->key.vlan);
			}
			hash_table_delete(ht,&temp_config_entry->key,
					sizeof(temp_config_entry->key),NULL);
			if(temp_config_entry){
				free(temp_config_entry);
			}
		}
	}
	hash_table_foreach(ht,delete_dirty_vlan_table_wrapper,NULL);
}

void delete_dirty_bridge_table() {
	int delete_dirty_bridge_table_wrapper(void *data, void *arg) {
		bridge_t *br_entry = (bridge_t*) data;
		delete_dirty_vlan_table(br_entry->vlan_table);
		if(	br_entry->dirty == true) {
			syslog(LOG_DEBUG,"%s Inside delete",__FUNCTION__);
			event_del(&br_entry->if_info.br_event);
			event_del(&br_entry->if_info.br_event6);
			close(br_entry->if_info.rfdesc);
            close(br_entry->if_info.rfdesc6);
			hash_table_delete(master_config_table,&br_entry->br_key,
					sizeof(br_entry->br_key),NULL);
			if (br_entry) {
				free(br_entry);
			}
		}
		return hash_table_foreach_done;
	}
	hash_table_foreach(master_config_table,delete_dirty_bridge_table_wrapper,NULL);
}

void process_dhcpv6_packet(struct dhcpv6_packet * dh_pkt,
		uint16_t vlan, char *ifName, unsigned int rx_ifindex, uint16_t udp_pkt_len) {

	struct in6_addr ip6_addr;
	char buf6[INET6_ADDRSTRLEN];
	char cmac[6];
	uint32_t prelife=0;
	uint32_t validlife=0;
	int ret = 0;
	binding_entry_key_6_t tmp_key;
	dhcp_mac_ip_6_t *tmp_entry = NULL;
	uint32_t temp_vlan=0;
	int is_vlan_aware = 0;

	temp_vlan = dhcpsnoop_is_bridge_trad(ifName);

	if (temp_vlan) {
		vlan = temp_vlan;
	} else {
		is_vlan_aware = 1;
	}


	if (dh_pkt->msg_type == DHCPV6_RELAY_FORW || 
			dh_pkt->msg_type == DHCPV6_RELAY_REPL ) {
		syslog(LOG_DEBUG,"Relay messages");
	} else {
		ret = dhcpsnoop6_get_ia_address(dh_pkt,D6O_IA_NA, udp_pkt_len, 
				&ip6_addr,&prelife,&validlife);	
		if (ret != -1) {
			if (inet_ntop(AF_INET6, &ip6_addr, buf6, sizeof(buf6)) != NULL) {
				syslog(LOG_DEBUG,"inet6 addr: %s\n", buf6);
				syslog(LOG_DEBUG,"Preffered life %d valid life %d",
						ntohl(prelife),ntohl(validlife));
			}
		} else {
			syslog(LOG_DEBUG,"Invalid IP address");
			return;
		}
		ret = 0;
		ret = dhcpsnoop6_get_client_identifier(dh_pkt, udp_pkt_len, cmac);
		if (ret != -1) {
			syslog(LOG_DEBUG,"Client MAC %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x",
					cmac[0],cmac[1],cmac[2],cmac[3],cmac[4],cmac[5]);
		} else {
			return;
		}

		if ( dh_pkt->msg_type == DHCPV6_REPLY) {
			memset(&tmp_key, 0, sizeof(tmp_key));
			memcpy(tmp_key.mac_addr,cmac,6);
			tmp_key.vlan = vlan;
			memcpy(&tmp_key.ip_add,&ip6_addr,sizeof(struct in6_addr));
			strncpy(tmp_key.ifName,ifName,strlen(ifName)); //Need to pass ifindex        

			hash_table_find(binding_table6, &tmp_key,
					sizeof(tmp_key),(void **)&tmp_entry);
			if(tmp_entry) {
				tmp_entry->state = DHCPV6_REPLY;
				tmp_entry->lease = (int) ntohl(validlife);	
			} else {
				tmp_entry = calloc(1, sizeof(dhcp_mac_ip_6_t));
				if (!tmp_entry) {
					syslog(LOG_ERR,"Memory allocation failed\n");
					return;
				}
				memcpy(&tmp_entry->key.ip_add,&ip6_addr,sizeof(struct in6_addr));
				memcpy(tmp_entry->key.mac_addr,cmac,6);
				tmp_entry->key.vlan = vlan;
				strncpy(tmp_entry->key.ifName,ifName,strlen(ifName));
				tmp_entry->type = DHCP_SNOOP;
				tmp_entry->state = DHCPV6_REPLY;
				tmp_entry->lease = (int) ntohl(validlife);
				get_port_name_from_mac_vlan_pair(tmp_entry->key.mac_addr,
						tmp_entry->key.vlan,tmp_entry->key.ifName,
						tmp_entry->rx_ifname,IFNAMSIZ,is_vlan_aware);
				tmp_entry->rx_ifindex = if_nametoindex(tmp_entry->rx_ifname);
				if (!hash_table_add(binding_table6,&tmp_entry->key,
							sizeof(tmp_entry->key),tmp_entry)) {
					return 0;
				}
			}
			print_binding_table_to_file6();
		} else if (dh_pkt->msg_type == DHCPV6_RELEASE) {
			memset(&tmp_key, 0, sizeof(tmp_key));
			memcpy(tmp_key.mac_addr,cmac,6);
			tmp_key.vlan = vlan;
			memcpy(&tmp_key.ip_add,&ip6_addr,sizeof(struct in6_addr));
			strncpy(tmp_key.ifName,ifName,strlen(ifName)); //Need to pass ifindex        

			hash_table_find(binding_table6, &tmp_key,
					sizeof(tmp_key),(void **)&tmp_entry);
			if(tmp_entry) {
				hash_table_delete(binding_table6, &tmp_key,
						sizeof(tmp_key),(void **)&tmp_entry);
				if(tmp_entry) {
					free(tmp_entry);
				}
				return 0;	
			}
			print_binding_table_to_file6();
		}
	}
}

void process_dhcp_packet(struct dhcp_packet * dh_pkt, 
		uint16_t vlan, char *ifName, unsigned int rx_ifindex) {

	uint32_t msg_type=0;
	binding_entry_key_t tmp_key;
	dhcp_mac_ip_t *tmp_entry = NULL;
  	uint32_t temp_vlan=0;
	int is_vlan_aware = 0;

	temp_vlan = dhcpsnoop_is_bridge_trad(ifName);

	if (temp_vlan) {
		vlan = temp_vlan; 
	} else {
		is_vlan_aware = 1;
	}

	msg_type = get_option_msgtype(dh_pkt->options);
	switch(msg_type) {
		case DHCPDISCOVER:
			syslog(LOG_DEBUG,"DHCPDISCOVER\n");
			break;
		case DHCPOFFER:
			syslog(LOG_DEBUG,"DHCPOFFER\n");
			break;
		case DHCPREQUEST:
			syslog(LOG_DEBUG,"DHCPREQUEST\n");
			break;
		case DHCPACK:
			syslog(LOG_DEBUG,"DHCPACK\n");
			break;
		default:
			syslog(LOG_DEBUG,"DEFAULT\n");
	}

	if (msg_type == DHCPACK) {
		memset(&tmp_key, 0, sizeof(tmp_key));
		memcpy(tmp_key.mac_addr,dh_pkt -> chaddr,6);
		tmp_key.vlan = vlan;
		tmp_key.ip_add = (uint32_t)dh_pkt->yiaddr.s_addr;
		strncpy(tmp_key.ifName,ifName,strlen(ifName)); //Need to pass ifindex        

		hash_table_find(binding_table, &tmp_key,
				sizeof(tmp_key),(void **)&tmp_entry);
		if(tmp_entry) {
			if (tmp_entry->state == DHCPREQUEST) {
				tmp_entry->lease = (int) ntohl(get_option_lease(dh_pkt->options));	
			}
			tmp_entry->state = DHCPACK;
		} else {
			tmp_entry = calloc(1, sizeof(dhcp_mac_ip_t));
			if (!tmp_entry) {
				syslog(LOG_ERR,"Memory allocation failed\n");
				return;
			}
			tmp_entry->key.ip_add = (uint32_t)dh_pkt->yiaddr.s_addr;
			memcpy(tmp_entry->key.mac_addr,dh_pkt -> chaddr,6);
			tmp_entry->key.vlan = vlan;
			strncpy(tmp_entry->key.ifName,ifName,strlen(ifName));
			tmp_entry->type = DHCP_SNOOP;
			tmp_entry->state = DHCPACK;
			get_port_name_from_mac_vlan_pair(tmp_entry->key.mac_addr,
					tmp_entry->key.vlan,tmp_entry->key.ifName,
					tmp_entry->rx_ifname,IFNAMSIZ,is_vlan_aware);
			tmp_entry->rx_ifindex = if_nametoindex(tmp_entry->rx_ifname);
			if (!hash_table_add(binding_table,&tmp_entry->key,
						sizeof(tmp_entry->key),tmp_entry)) {
				return 0;
			}
		}
	} else if (msg_type == DHCPOFFER) {
		memset(&tmp_key, 0, sizeof(tmp_key));
		memcpy(tmp_key.mac_addr,dh_pkt -> chaddr,6);
		tmp_key.vlan = vlan;
		tmp_key.ip_add = (uint32_t)dh_pkt->yiaddr.s_addr;
		strncpy(tmp_key.ifName,ifName,strlen(ifName)); //Need to pass ifindex        

		hash_table_find(binding_table, &tmp_key,
				sizeof(tmp_key),(void **)&tmp_entry);
		if (tmp_entry) {
			tmp_entry->state = DHCPOFFER;
			tmp_entry->lease = (int) ntohl(get_option_lease(dh_pkt->options));
		} else {
			tmp_entry = calloc(1, sizeof(dhcp_mac_ip_t));
			if (!tmp_entry) {
				syslog(LOG_ERR,"Memory allocation failed\n");
				return;
			}
			tmp_entry->key.ip_add = (uint32_t)dh_pkt->yiaddr.s_addr;
			memcpy(tmp_entry->key.mac_addr,dh_pkt -> chaddr,6);
			tmp_entry->key.vlan = vlan;
			strncpy(tmp_entry->key.ifName,ifName,strlen(ifName));
			tmp_entry->state = DHCPOFFER;
			tmp_entry->type = DHCP_SNOOP;
			tmp_entry->lease = (int) ntohl(get_option_lease(dh_pkt->options));
			get_port_name_from_mac_vlan_pair(tmp_entry->key.mac_addr,
					tmp_entry->key.vlan,tmp_entry->key.ifName,
					tmp_entry->rx_ifname,IFNAMSIZ,is_vlan_aware);
			tmp_entry->rx_ifindex = if_nametoindex(tmp_entry->rx_ifname);
			if (!hash_table_add(binding_table,&tmp_entry->key,
						sizeof(tmp_entry->key),tmp_entry)) {
				return 0;
			}
		}		
	} else if (msg_type == DHCPREQUEST) {
		memset(&tmp_key, 0, sizeof(tmp_key));
		memcpy(tmp_key.mac_addr,dh_pkt -> chaddr,6);
		tmp_key.vlan = vlan;
		strncpy(tmp_key.ifName,ifName,strlen(ifName)); //Need to pass ifindex        
		tmp_key.ip_add = (uint32_t)dh_pkt->ciaddr.s_addr;
		if (ntohl((uint32_t)dh_pkt->ciaddr.s_addr) == 0) {
			tmp_key.ip_add = get_option_requested_ip(dh_pkt->options);
			if (ntohl(get_option_requested_ip(dh_pkt->options)) == 0) {
				return 0;
			}
		}

		hash_table_find(binding_table, &tmp_key,
				sizeof(tmp_key),(void **)&tmp_entry);
		if (tmp_entry) {
			tmp_entry->state = DHCPREQUEST;
		} else { 
			tmp_entry = calloc(1, sizeof(dhcp_mac_ip_t));
			if (!tmp_entry) {
				syslog(LOG_ERR,"Memory allocation failed\n");
				return;
			}
			tmp_entry->key.ip_add = tmp_key.ip_add;
			memcpy(tmp_entry->key.mac_addr,dh_pkt -> chaddr,6);
			tmp_entry->key.vlan = vlan;
			strncpy(tmp_entry->key.ifName,ifName,strlen(ifName));
			tmp_entry->state = DHCPREQUEST;
			tmp_entry->type = DHCP_SNOOP;
			tmp_entry->lease = (int) ntohl(get_option_lease(dh_pkt->options));
			get_port_name_from_mac_vlan_pair(tmp_entry->key.mac_addr,
					tmp_entry->key.vlan,tmp_entry->key.ifName,
					tmp_entry->rx_ifname,IFNAMSIZ,is_vlan_aware);
			tmp_entry->rx_ifindex = if_nametoindex(tmp_entry->rx_ifname);
			if (!hash_table_add(binding_table,&tmp_entry->key,
						sizeof(tmp_entry->key),tmp_entry)) {
				return 0;
			}
		}
	} else if (msg_type == DHCPRELEASE) {

	} 
	print_binding_table_to_file();
} 

void print_udp_packet(unsigned char *Buffer , int Size, uint16_t vlan, 	
		struct interface_info *if_info, unsigned int rx_ifindex)
{

	unsigned short iphdrlen;
	struct ethhdr *eh = (struct ethhdr *) Buffer;
	struct udphdr *udph = NULL;
	int ipversion=0;

	if (ntohs(eh->h_proto) == ETHERTYPE_IP) {
		struct iphdr *iph = (struct iphdr *)(Buffer +  sizeof(struct ethhdr));
		iphdrlen = iph->ihl*4;
		udph = (struct udphdr*)(Buffer + iphdrlen  + sizeof(struct ethhdr));
	} else if (ntohs(eh->h_proto) == ETHERTYPE_IPv6) {
		struct ipv6hdr *iph6 = (struct ipv6hdr *)(Buffer +  sizeof(struct ethhdr));
		iphdrlen = 40;
		udph = (struct udphdr*)(Buffer + iphdrlen  + sizeof(struct ethhdr));
	} else {
		return;
	}

	int header_size =  sizeof(struct ethhdr) + iphdrlen + sizeof udph;
	unsigned char * buff_dhcp = (Buffer + sizeof(struct ethhdr) + iphdrlen + sizeof udph);

	if (ntohs(eh->h_proto) == ETHERTYPE_IP ) {
		process_dhcp_packet((struct dhcp_packet *) buff_dhcp, vlan, if_info->name,rx_ifindex); 
	} else if(ntohs(eh->h_proto) == ETHERTYPE_IPv6) {
		//PrintData(Buffer + iphdrlen  + sizeof(struct ethhdr),ntohs(udph->len));
		process_dhcpv6_packet((struct dhcpv6_packet *) buff_dhcp, vlan, if_info->name,rx_ifindex,ntohs(udph->len));	
	}   
}

void print_ip_header(unsigned char* Buffer, int Size)
{
	print_ethernet_header(Buffer , Size);

	unsigned short iphdrlen;

	struct iphdr *iph = (struct iphdr *)(Buffer  + sizeof(struct ethhdr) );
	iphdrlen =iph->ihl*4;

	memset(&source, 0, sizeof(source));
	source.sin_addr.s_addr = iph->saddr;

	memset(&dest, 0, sizeof(dest));
	dest.sin_addr.s_addr = iph->daddr;

}

void print_ethernet_header(unsigned char* Buffer, int Size)
{
	struct ethhdr *eth = (struct ethhdr *)Buffer;
	syslog(LOG_DEBUG,"Ethernet Header\n");
	syslog(LOG_DEBUG,"   |-Destination Address : %.2X-%.2X-%.2X-%.2X-%.2X-%.2X \n", eth->h_dest[0] , eth->h_dest[1] , eth->h_dest[2] , eth->h_dest[3] , eth->h_dest[4] , eth->h_dest[5] );
	syslog(LOG_DEBUG,"   |-Source Address      : %.2X-%.2X-%.2X-%.2X-%.2X-%.2X \n", eth->h_source[0] , eth->h_source[1] , eth->h_source[2] , eth->h_source[3] , eth->h_source[4] , eth->h_source[5] );
	syslog(LOG_DEBUG,"   |-Protocol            : %u \n",(unsigned short)eth->h_proto);
}

int if_register_lpf_snoop (struct interface_info *info)
{   
	int sock;
	union {
		struct sockaddr_ll ll;
		struct sockaddr common;
	} sa;
	struct ifreq ifr;

	/* Make an LPF socket. */
	if ((sock = socket(PF_PACKET, SOCK_RAW,
					htons((short)ETH_P_ALL))) < 0) {
		if (errno == ENOPROTOOPT || errno == EPROTONOSUPPORT ||
				errno == ESOCKTNOSUPPORT || errno == EPFNOSUPPORT ||
				errno == EAFNOSUPPORT || errno == EINVAL) {
			syslog(LOG_ERR,"socket: %m - make sure");
			syslog(LOG_ERR,"CONFIG_PACKET (Packet socket) %s",
					"and CONFIG_FILTER");
			syslog(LOG_ERR,"(Socket Filtering) are enabled %s",
					"in your kernel");
			syslog(LOG_ERR,"configuration!");
		}
		syslog(LOG_ERR,"Open a socket for LPF: %m");
	}

	memset (&ifr, 0, sizeof ifr); 
	strncpy (ifr.ifr_name,info->name,strlen(info->name));
	ifr.ifr_name[IFNAMSIZ-1] = '\0';

	syslog(LOG_INFO,"Listning on %s\n",ifr.ifr_name);
	if (ioctl (sock, SIOCGIFINDEX, &ifr)) {
		syslog(LOG_ERR,"Failed to get interface index: %m");
		return -1;
	}

	/* Bind to the interface name */
	memset (&sa, 0, sizeof sa);
	sa.ll.sll_family = AF_PACKET;
	sa.ll.sll_ifindex = ifr.ifr_ifindex;
	if (bind (sock, &sa.common, sizeof sa)) {
		if (errno == ENOPROTOOPT || errno == EPROTONOSUPPORT ||
				errno == ESOCKTNOSUPPORT || errno == EPFNOSUPPORT ||
				errno == EAFNOSUPPORT || errno == EINVAL) {
			syslog(LOG_ERR,"socket: %m - make sure");
			syslog(LOG_ERR,"CONFIG_PACKET (Packet socket) %s",
					"and CONFIG_FILTER");
			syslog(LOG_ERR,"(Socket Filtering) are enabled %s",
					"in your kernel");
			syslog(LOG_ERR,"configuration!");
		}
		close(sock);
		syslog(LOG_ERR,"Bind socket to interface %s: %m", info->name);
		return -1;
	}

	if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0) {
		exit(EXIT_FAILURE);
	}

	ifr.ifr_flags |= IFF_PROMISC;

	if (ioctl(sock, SIOCSIFFLAGS, &ifr) < 0) {
		exit(EXIT_FAILURE);
	}
	return sock;
}


void if_register_receive_snoop(struct interface_info *info)
{
	/* Open a LPF device and hang it on this interface... */
	info->rfdesc = if_register_lpf_snoop(info);
	info->rfdesc6 = if_register_lpf_snoop(info);

	int ret;
	if (info->rfdesc < 0)
		return;

	if (info->rfdesc6 < 0)
		return;

	int val = 1;    
	if (setsockopt(info->rfdesc, SOL_PACKET, PACKET_AUXDATA,
				&val, sizeof(val)) < 0) {
		if (errno != ENOPROTOOPT) {
			syslog(LOG_ERR,"Failed to set auxiliary packet data: %m");
		}
	}

	val = 1;   
	if (setsockopt(info->rfdesc6, SOL_PACKET, PACKET_AUXDATA,
				&val, sizeof(val)) < 0) {
		if (errno != ENOPROTOOPT) {
			syslog(LOG_ERR,"Failed to set auxiliary packet data: %m");
		}
	}

	ret = setsockopt(info->rfdesc,SOL_PACKET, 
			SO_ATTACH_FILTER, &bpf, sizeof(bpf));
	if (ret < 0) {
		return -1;
	}

	ret = setsockopt(info->rfdesc6,SOL_PACKET,
			SO_ATTACH_FILTER, &bpf6, sizeof(bpf6));
	if (ret < 0) {
		return -1;
	}

	/*
	   val = 1;
	   if (setsockopt(info->rfdesc, IPPROTO_IP, IP_PKTINFO,
	   &val, sizeof(val)) < 0) {
	   if (errno != ENOPROTOOPT) {
	   printf("Failed to set IP_PKTINFO data: %m");
	   }
	   }
	   */
}

size_t receive_packet_snoop (int fd, short ev, void *arg)
{
	int length = 0;
	int offset = 0;
	int csum_ready = 1;
	uint32_t ip_version = 4;
	unsigned char ibuf [1536];
	unsigned bufix = 0;
	unsigned paylen = 0;
	struct iovec iov = {
		.iov_base = ibuf,
		.iov_len = sizeof ibuf,
	};
	unsigned char cmsgbuf[CMSG_LEN(sizeof(struct tpacket_auxdata)) + CMSG_LEN(sizeof(struct in_pktinfo))];
	struct msghdr msg = {
		.msg_iov = &iov,
		.msg_iovlen = 1,
		.msg_control = cmsgbuf,
		.msg_controllen = sizeof(cmsgbuf),
	};
	length = recvmsg (fd, &msg, MSG_DONTWAIT);
	if (length <= 0)
		return length;

	struct cmsghdr *cmsg;
	uint32_t vlan = 0;
	unsigned int ifindex=0;
	for (cmsg = CMSG_FIRSTHDR(&msg); cmsg; cmsg = CMSG_NXTHDR(&msg, cmsg)) {
		if (cmsg->cmsg_level == SOL_PACKET &&
				cmsg->cmsg_type == PACKET_AUXDATA) {
			struct tpacket_auxdata *aux = (void *)CMSG_DATA(cmsg);
			if (aux->tp_vlan_tci & 0x0fff)
				vlan = aux->tp_vlan_tci;
		}
		if (cmsg->cmsg_type == IP_PKTINFO) {
			struct in_pktinfo *pkt_info = (void *)CMSG_DATA(cmsg);
			ifindex = pkt_info->ipi_ifindex;
		}
	}

	struct interface_info *if_info = (struct interface_info *) arg;

	struct ethhdr *eh = (struct ethhdr *) msg.msg_iov->iov_base;
	if (ntohs(eh->h_proto) == ETHERTYPE_IP) {
		ip_version = 4;
	} else if (ntohs(eh->h_proto) == ETHERTYPE_IPv6) { 
		ip_version = 6;
	}
	if (dhcpsnoop_is_vlan_snooping_enabled(if_info->name, vlan, ip_version)) { 
		ProcessPacket(msg.msg_iov->iov_base,msg.msg_iov->iov_len,vlan, 
				(struct interface_info *) arg, ifindex); 
	}
	return paylen;
}

static uint8_t get_option_msgtype(unsigned char *opt)
{
	uint32_t var = 0;
	opt = opt + 4;
	while (*opt != DHO_DHCP_MESSAGE_TYPE) {
		if (*opt == DHO_END) return var;
		opt += opt[1] + 2;
	}
	memcpy(&var, opt+2, sizeof(uint8_t));
	return var;
}

static uint32_t get_option_serverid (unsigned char *opt)
{
	uint32_t var = 0;
	opt = opt + 4;
	while (*opt != DHO_DHCP_SERVER_IDENTIFIER) {
		if (*opt == DHO_END) return var;
		opt += opt[1] + 2;
	}
	memcpy(&var, opt+2, sizeof(uint32_t));
	return var;
}

static uint32_t get_option_requested_ip (unsigned char *opt)
{
	uint32_t var = 0;
	opt = opt + 4;
	while (*opt != DHO_DHCP_REQUESTED_ADDRESS) {
		if (*opt == DHO_END) return var;
		opt += opt[1] + 2;
	}
	memcpy(&var, opt+2, sizeof(uint32_t));
	return var;
}

static uint32_t get_option_lease(unsigned char *opt)
{
	uint32_t var = 0;
	opt = opt + 4;
	while (*opt != DHO_DHCP_LEASE_TIME) {
		if (*opt == DHO_END) return var;
		opt += opt[1] + 2;
	}
	memcpy(&var, opt+2, sizeof(uint32_t));
	return var;
}

int is_vlan_aware(char *iface) {
	FILE *fp;
	char op[IFNAMSIZ];
	char filter_file[FILENAME];
	int vlan_aware=0;
	/*check if bridge dir is loaded*/
	struct stat sb;
	int iter=0;

wait_pc:
	snprintf(filter_file,FILENAME-1,"/sys/class/net/%s",iface);
	filter_file[FILENAME-1] = '\0';
	if (stat(filter_file, &sb) == 0 && S_ISDIR(sb.st_mode)) {
		syslog(LOG_INFO,"Directory present");
	} else {
		syslog(LOG_INFO,"Directory Not present, waiting for some time");
		iter++;
		if (iter < 10) {
			sleep(30);
			goto wait_pc;
		} else {
			syslog(LOG_ERR,"Failed to load bridge dir");
			exit(0);
		}
	}
	
	snprintf(filter_file,FILENAME-1,"/sys/class/net/%s/bridge/vlan_filtering",iface);
	filter_file[FILENAME-1] = '\0';

	fp = fopen(filter_file,"r");
	if (fp == NULL) {
		syslog(LOG_ERR,"Unable to open %s",filter_file);
		return 0;
	}
	fgets(op,IFNAMSIZ-1,fp);
	vlan_aware = atoi(op);
	
	syslog(LOG_INFO,"VLAN aware %d",vlan_aware);
	fclose(fp);
	return vlan_aware;
}

void get_port_name_from_mac_vlan_pair(unsigned char * mac, 
    int vlan, char *bridge, char *iface, int size, int vlan_aware) {
	FILE *fp = NULL;
	char command[128];
	if (vlan_aware) {
		snprintf(command,127,"/sbin/bridge fdb show %s ""| grep \"%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x.*vlan.*%d.*master.*%s\" | awk '{print $3}'",
			bridge,mac[0],mac[1],mac[2],mac[3],mac[4],mac[5],vlan,bridge);
	} else {
		snprintf(command,127,"/sbin/bridge fdb show %s | grep \"%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x.*master.*%s\" | awk '{print $3}'",
            bridge,mac[0],mac[1],mac[2],mac[3],mac[4],mac[5],bridge);
	}
	fp = popen(command,"r");
	if (fp) {
		fscanf(fp,"%s",iface);
	} else {
		strncpy(iface,"NA",IFNAMSIZ-1);
		iface[IFNAMSIZ-1] = '\0';
	}

        close(fp);
}

void dhcpsnoop_state(char *out_str, uint8_t state_int) {
	char *state;
	switch(state_int) {
		case DHCPDISCOVER:
			state = "DISCOVER";
			break;
		case DHCPOFFER:
			state = "OFFER";
			break;
		case DHCPREQUEST:
			state = "REQUEST";
			break;
		case DHCPDECLINE:
			state = "DECLINE";
			break;
		case DHCPACK:
			state = "ACK";
			break;
		case DHCPNAK:
			state = "NACK";
			break;
		case DHCPRELEASE:
			state = "RELEASE";
			break;
		case DHCPINFORM:
			state = "INFORM";
			break;
		case DHCPFORCERENEW:
			state = "FORCERENEW";
			break;
		case DHCPLEASEQUERY:
			state = "LEASEQUERY";
			break;
		case DHCPLEASEUNASSIGNED:
			state = "LEASEUNASSIGNED";
			break;
		case DHCPLEASEUNKNOWN:
			state = "LEASEUNKNOWN";
			break;
		case DHCPLEASEACTIVE:
			state = "LEASEACTIVE";
			break;
		default:
			state = "INV";
	}
	snprintf(out_str,FILENAME-1,"%s",state);
	out_str[FILENAME-1] = '\0';
}


void dhcpsnoop_state6(char *out_str, uint8_t state_int) {
	char *state;
	switch(state_int) {
		case DHCPV6_SOLICIT:
			state = "SOLICIT";
			break;
		case DHCPV6_ADVERTISE:
			state = "ADVERTISE";
			break;
		case DHCPV6_REQUEST:
			state = "REQUEST";
			break;
		case DHCPV6_CONFIRM:
			state = "CONFIRM";
			break;
		case DHCPV6_RENEW:
			state = "RENEW";
			break;
		case DHCPV6_REBIND:
			state = "REBIND";
			break;
		case DHCPV6_REPLY:
			state = "REPLY";
			break;
		case DHCPV6_RELEASE:
			state = "RELEASE";
			break;
		case DHCPV6_DECLINE:
			state = "DECLINE";
			break;
		case DHCPV6_RECONFIGURE:
			state = "RECONFIGURE";
			break;
		case DHCPV6_INFORMATION_REQUEST:
			state = "INFORMATION_REQUEST";
			break;
		case DHCPV6_RELAY_FORW:
			state = "RELAY_FORW";
			break;
		case DHCPV6_RELAY_REPL:
			state = "RELAY_REPL";
			break;
		case DHCPV6_LEASEQUERY:
			state = "LEASEQUERY";
			break;
		case DHCPV6_LEASEQUERY_REPLY:
			state = "LEASEQUERY_REPLY";
			break;
		case DHCPV6_LEASEQUERY_DONE:
			state = "LEASEQUERY_DONE";
			break;
		case DHCPV6_LEASEQUERY_DATA:
			state = "LEASEQUERY_DATA";
			break;
		case DHCPV6_RECONFIGURE_REQUEST:
			state = "RECONFIGURE_REQUEST";
			break;
		case DHCPV6_RECONFIGURE_REPLY:
			state = "RECONFIGURE_REPLY";
			break;
		case DHCPV6_DHCPV4_QUERY:
			state = "DHCPV4_QUERY";
			break;
		case DHCPV6_DHCPV4_RESPONSE:
			state = "DHCPV4_RESPONSE";
			break;
		default:
			state = "INV";
	}
	snprintf(out_str,FILENAME-1,"%s",state);
	out_str[FILENAME-1] = '\0';
}

int print_binding_table_to_file() {
	char *out=NULL;
	cJSON *root, *dhcp_table, *entry;
	FILE *fp=NULL;
	fp = fopen(db_file, "w");
	char dhcp_state[FILENAME];
	long buf_size = 0;

	if (fp==NULL) {
		syslog(LOG_ERR,"Error opening file\n");
		return -1;
	}
	/* create root node and array */
	root = cJSON_CreateObject();
	dhcp_table = cJSON_CreateArray();

	cJSON_AddItemToObject(root, "dhcp_table", dhcp_table);

	int print_binding_table_to_file_wrapper(void *data,void *arg) {
		struct in_addr ip_addr;
		char mac_str[20];
		dhcp_mac_ip_t *tmp_entry = (dhcp_mac_ip_t *) data;
		ip_addr.s_addr = tmp_entry->key.ip_add;
		snprintf(mac_str,18, "%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x",
				tmp_entry->key.mac_addr[0],
				tmp_entry->key.mac_addr[1],
				tmp_entry->key.mac_addr[2],
				tmp_entry->key.mac_addr[3],
				tmp_entry->key.mac_addr[4],
				tmp_entry->key.mac_addr[5]);
		mac_str[18] = '\0';
		dhcpsnoop_state(dhcp_state,tmp_entry->state);
		cJSON_AddItemToArray(dhcp_table, entry = cJSON_CreateObject()); 
		cJSON_AddItemToObject(entry, "Port", cJSON_CreateString(tmp_entry->rx_ifname));
		cJSON_AddItemToObject(entry, "VLAN", cJSON_CreateNumber(tmp_entry->key.vlan));
		cJSON_AddItemToObject(entry, "IP", cJSON_CreateString(inet_ntoa(ip_addr)));
		cJSON_AddItemToObject(entry, "MAC", cJSON_CreateString(mac_str));
		cJSON_AddItemToObject(entry, "Type", cJSON_CreateNumber(0));
		cJSON_AddItemToObject(entry, "Lease", cJSON_CreateNumber(tmp_entry->lease));
		cJSON_AddItemToObject(entry, "State", cJSON_CreateString(dhcp_state));
		cJSON_AddItemToObject(entry, "State_internal", cJSON_CreateNumber(tmp_entry->state));
		cJSON_AddItemToObject(entry, "Bridge", cJSON_CreateString(tmp_entry->key.ifName));
		cJSON_AddItemToObject(entry, "Timestamp", cJSON_CreateNumber(time(NULL)));

		return hash_table_foreach_done;
	}
	hash_table_foreach(binding_table,print_binding_table_to_file_wrapper,NULL);    

	buf_size = ONERECSIZE*cJSON_GetArraySize(dhcp_table);

	if (buf_size == 0) {
		syslog(LOG_DEBUG,"Buffer size is %d",buf_size);
		buf_size = ONERECSIZE;
	}
	out = (char*)calloc(buf_size,sizeof(char));

	if (out) {
		if (cJSON_PrintPreallocated(root,out,buf_size,1)) {
			fputs(out,fp);
			fflush(fp);
		}
		free(out);
	} else {
		syslog(LOG_INFO,"%s Failed to allocate memory",__FUNCTION__);
	}

	fclose(fp);
	cJSON_Delete(root);
	return 0;
}

int print_binding_table_to_file6() {
	char *out=NULL;
	cJSON *root, *dhcp_table, *entry;
	FILE *fp=NULL;
	fp = fopen(db_file6, "w");
	char dhcp_state[FILENAME];
	char buf6[INET6_ADDRSTRLEN];
	long buf_size=0;

	if (fp==NULL) {
		syslog(LOG_ERR,"Error opening file\n");
		return -1;
	}
	/* create root node and array */
	root = cJSON_CreateObject();
	dhcp_table = cJSON_CreateArray();

	cJSON_AddItemToObject(root, "dhcpv6_table", dhcp_table);

	int print_binding_table_to_file6_wrapper(void *data,void *arg) {
		struct in_addr ip_addr;
		char mac_str[20];
		dhcp_mac_ip_6_t *tmp_entry = (dhcp_mac_ip_6_t *) data;
		inet_ntop(AF_INET6, &tmp_entry->key.ip_add, buf6, sizeof(buf6));
		snprintf(mac_str,18, "%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x",
				tmp_entry->key.mac_addr[0],
				tmp_entry->key.mac_addr[1],
				tmp_entry->key.mac_addr[2],
				tmp_entry->key.mac_addr[3],
				tmp_entry->key.mac_addr[4],
				tmp_entry->key.mac_addr[5]);
		mac_str[18] = '\0';
		dhcpsnoop_state6(dhcp_state,tmp_entry->state);
		cJSON_AddItemToArray(dhcp_table, entry = cJSON_CreateObject());
		cJSON_AddItemToObject(entry, "Port", cJSON_CreateString(tmp_entry->rx_ifname));
		cJSON_AddItemToObject(entry, "VLAN", cJSON_CreateNumber(tmp_entry->key.vlan));
		cJSON_AddItemToObject(entry, "IP", cJSON_CreateString(buf6));
		cJSON_AddItemToObject(entry, "MAC", cJSON_CreateString(mac_str));
		cJSON_AddItemToObject(entry, "Type", cJSON_CreateNumber(0));
		cJSON_AddItemToObject(entry, "Lease", cJSON_CreateNumber(tmp_entry->lease));
		cJSON_AddItemToObject(entry, "State", cJSON_CreateString(dhcp_state));
		cJSON_AddItemToObject(entry, "State_internal", cJSON_CreateNumber(tmp_entry->state));
		cJSON_AddItemToObject(entry, "Bridge", cJSON_CreateString(tmp_entry->key.ifName));
		cJSON_AddItemToObject(entry, "Timestamp", cJSON_CreateNumber(time(NULL)));

		return hash_table_foreach_done;
	}
	hash_table_foreach(binding_table6,print_binding_table_to_file6_wrapper,NULL);

	buf_size = ONERECSIZE*cJSON_GetArraySize(dhcp_table);
	if (buf_size == 0) {
		syslog(LOG_DEBUG,"Buffer size is %d",buf_size);
		buf_size = ONERECSIZE;
	}
	out	= (char*)calloc(buf_size,sizeof(char));
	if (out) {
		if (cJSON_PrintPreallocated(root,out,buf_size,1)) {
			fputs(out,fp);
			fflush(fp);
		}
		free(out);
	} else {
		syslog(LOG_INFO,"%s Failed to allocate memory",__FUNCTION__);
	}

	fclose(fp);
	cJSON_Delete(root);
	return 0;
}

int dhcpsnoop_ebtables(char *iface, uint32_t vlan) {
	FILE *fp = NULL;
	char trustfile[FILENAME];
	snprintf(trustfile,FILENAME-1,"/etc/cumulus/acl/policy.d/dhcpsnoop_ebtables_%s_%d.rules",iface,vlan);
	fp = fopen(trustfile, "w");
	if (fp==NULL) {
		syslog(LOG_ERR,"Error opening file\n");
		return -1;
	}
	fprintf(fp, "# Generated by DHCP snoop EBTABLES IPTABLESRules File: Do not edit.\n");
	fprintf(fp, "[ebtables]\n");
	fprintf(fp,"-A INPUT -i %s -p 802_1Q --vlan-id %d -j mark --mark-set %d\n",iface,vlan,vlan+100);
	fprintf(fp,"\n");

	fclose(fp);
	return 0;
}

int dhcpsnoop_apply_interface_trust_config(char *iface, uint32_t vlan, bool vlan_aware) {
	FILE *fp = NULL;
	char trustfile[FILENAME];
	snprintf(trustfile,FILENAME-1,"/etc/cumulus/acl/policy.d/dhcpsnoop_%s_%d.rules",iface,vlan);
	fp = fopen(trustfile, "w");
	if (fp==NULL) {
		syslog(LOG_ERR,"Error opening file\n");
		return -1;
	}
	fprintf(fp, "# Generated by DHCP snoop IPTABLES Rules File: Do not edit.\n");
	fprintf(fp,"[iptables]\n");
	if (vlan_aware) {
		fprintf(fp,"-A FORWARD -i %s -m mark --mark %d -p udp --sport %d -j DROP\n",
				iface,vlan+100,DHCP_SPORT);
	} else if (!vlan_aware) {
		fprintf(fp,"-A FORWARD -i %s -p udp --sport %d -j DROP\n",
				iface,DHCP_SPORT);
	}
	fclose(fp);
	return 0;
}

int dhcpsnoop6_apply_interface_trust_config(char *iface, uint32_t vlan, bool vlan_aware) {
	FILE *fp = NULL;
	char trustfile[FILENAME];
	snprintf(trustfile,FILENAME-1,"/etc/cumulus/acl/policy.d/dhcpsnoop6_%s_%d.rules",iface,vlan);
	fp = fopen(trustfile, "w");
	if (fp==NULL) {
		syslog(LOG_ERR,"Error opening file\n");
		return -1;
	}
	fprintf(fp, "# Generated by DHCP snoop IP6TABLES Rules File: Do not edit.\n");
	fprintf(fp,"[ip6tables]\n");
	if (vlan_aware) {
		fprintf(fp,"-A FORWARD -i %s -m mark --mark %d -p udp --sport %d -j DROP\n",
				iface,vlan+100,DHCPV6_SPORT);
	} else if(!vlan_aware) {
		fprintf(fp,"-A FORWARD -i %s -p udp --sport %d -j DROP\n",
				iface,DHCPV6_SPORT);
	}
	fclose(fp);
	return 0;
}

int dhcpsnoop_police_acl(char *iface, uint32_t vlan, uint32_t rate) {
	FILE *fp = NULL;
	char policefile[FILENAME];
	snprintf(policefile,FILENAME-1,"/etc/cumulus/acl/policy.d/dhcpsnoop_police_%s_%d.rules",iface,vlan);
	fp = fopen(policefile, "w");
	if (fp==NULL) {
		syslog(LOG_ERR,"Error opening file\n");
		return -1;
	}
	fprintf(fp, "# Generated by DHCP snoop IPTABLES Rules File: Do not edit.\n");
	fprintf(fp,"[iptables]\n");
	fprintf(fp,"-A INPUT,FORWARD -i %s -m mark --mark %d -p udp --sport %d"
			" -j POLICE --set-mode pkt --set-rate %d --set-burst %d\n",
			iface,vlan+100,DHCP_SPORT,rate,rate);
	fprintf(fp,"-A INPUT,FORWARD -i %s -m mark --mark %d -p udp --sport %d"
			" -j POLICE --set-mode pkt --set-rate %d --set-burst %d\n",
			iface,vlan+100,DHCP_CPORT,rate,rate);

	fclose(fp);
	return 0;
}

int dhcpsnoop6_police_acl(char *iface, uint32_t vlan, uint32_t rate) {
	FILE *fp = NULL;
	char policefile[FILENAME];
	snprintf(policefile,FILENAME-1,"/etc/cumulus/acl/policy.d/dhcpsnoop6_police_%s_%d.rules",iface,vlan);
	fp = fopen(policefile, "w");
	if (fp==NULL) {
		syslog(LOG_ERR,"Error opening file\n");
		return -1;
	}
	fprintf(fp, "# Generated by DHCP snoop IPTABLES Rules File: Do not edit.\n");

	fprintf(fp,"[ip6tables]\n");
	fprintf(fp,"-A INPUT,FORWARD -i %s -m mark --mark %d -p udp --sport %d"
			" -j POLICE --set-mode pkt --set-rate %d --set-burst %d\n",
			iface,vlan+100,DHCPV6_SPORT,rate,rate);
	fprintf(fp,"-A INPUT,FORWARD -i %s -m mark --mark %d -p udp --sport %d"
			" -j POLICE --set-mode pkt --set-rate %d --set-burst %d\n",
			iface,vlan+100,DHCPV6_CPORT,rate,rate);

	fclose(fp);
	return 0;
}

int run_clacltool() {
	pid_t pid,childpid;
	int status;
	pid = fork();
	if (pid == 0) {
		execv(aclcommand,args);
	} else { 
		childpid = waitpid(pid,&status,NULL);
		if(WIFEXITED(status)) {
			return 0;
		} else {
			return -1;
		}
	}
	return 0;
}

void dhcpsnoop_age_function(int fd, short event, void *arg)
{
	int dhcpsnoop_age_function_wrapper(void *data,void *arg) {
		dhcp_mac_ip_t *tmp_entry = (dhcp_mac_ip_t *) data;
		dhcp_mac_ip_t *del_entry = NULL;
		if ((tmp_entry->lease - 60) > 0) { 
			tmp_entry->lease = tmp_entry->lease - 60;
		} else { 
			hash_table_delete(binding_table,&tmp_entry->key,sizeof(tmp_entry->key),del_entry);
			if(del_entry)
				free(del_entry);
		}
		return hash_table_foreach_done;
	}
	hash_table_foreach(binding_table,dhcpsnoop_age_function_wrapper,NULL);

	int dhcpsnoop_age_function6_wrapper(void *data,void *arg) {
		dhcp_mac_ip_6_t *tmp_entry6 = (dhcp_mac_ip_6_t *) data;
		dhcp_mac_ip_6_t *del_entry6 = NULL;
		if ((tmp_entry6->lease - 60) > 0) {
			tmp_entry6->lease = tmp_entry6->lease - 60;
		} else {
			hash_table_delete(binding_table6,&tmp_entry6->key,sizeof(tmp_entry6->key),del_entry6);
			if(del_entry6)
				free(del_entry6);
		}
		return hash_table_foreach_done;
	}
	hash_table_foreach(binding_table6,dhcpsnoop_age_function6_wrapper,NULL);

	print_binding_table_to_file();	
	print_binding_table_to_file6();

	int *timeout = (int *) arg;
	syslog(LOG_DEBUG,"Timeout value %d",*timeout);
	*timeout = *timeout - MINUTE;
	if (*timeout <= 0) {
		syslog(LOG_INFO,"Graceful Periodic restart");
		dhcpsnoop_sig_handler(99);
		exit(0);
	}
}

int dhcpsnoop_is_vlan_snooping_enabled(char *ifname,
	uint32_t vlan, uint32_t ip_version) 
{
	bridge_t *br_entry = NULL;
	bridge_key_t br_key;
	vlan_key_t vlan_key;
	vlan_key.vlan = vlan;
	vlan_key.ip_version = ip_version;
	vlan_t * vlan_entry = NULL;

	br_key.hash = hash_str(ifname);	

	hash_table_find(master_config_table, &br_key,
			sizeof(br_key),(void **)&br_entry);

	if (br_entry) {
		if (br_entry->vlan_aware) {		
			hash_table_find(br_entry->vlan_table,&vlan_key,
					sizeof(vlan_key),(void **)&vlan_entry);
			if (vlan_entry) {
				if (vlan_entry->snooping) {
					return 1;
				} else {
					return 0;
				}
			} else {
				return 0;
			}
		} else { 
			return 1;
		}
	}
}

void dhcpsnoop_init_syslog(int logcode) {
	setlogmask (LOG_UPTO (logcode));
	openlog ("dhcpsnoop", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_USER);
}

void print_dhcp_entry (dhcp_mac_ip_t *tmp_entry) {
	syslog(LOG_INFO,"Port  VLAN    IP Address      MAC     TYPE    LEASE   STATE   BRIDGE_NAME\n");
	syslog(LOG_INFO,"=====================================================================================\n");
	struct in_addr ip_addr;
	ip_addr.s_addr =  tmp_entry->key.ip_add;
	syslog(LOG_INFO,"%s    %d  %s  "
			"%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x"
			"   %d  %d  %d  %s\n",
			tmp_entry->rx_ifname,tmp_entry->key.vlan,
			inet_ntoa(ip_addr),
			tmp_entry->key.mac_addr[0],
			tmp_entry->key.mac_addr[1],
			tmp_entry->key.mac_addr[2],
			tmp_entry->key.mac_addr[3],
			tmp_entry->key.mac_addr[4],
			tmp_entry->key.mac_addr[5],
			0, tmp_entry->lease,
			tmp_entry->state,
			tmp_entry->key.ifName);	
}

int ipStringToNumber(const char*  pDottedQuad,uint32_t * pIpAddr) {
	unsigned int            byte3;
	unsigned int            byte2;
	unsigned int            byte1;
	unsigned int            byte0;
	char              dummyString[2];

	if (sscanf (pDottedQuad, "%u.%u.%u.%u%1s",
				&byte0, &byte1, &byte2, &byte3, dummyString) == 4)
	{
		if (    (byte3 < 256)
				&& (byte2 < 256)
				&& (byte1 < 256)
				&& (byte0 < 256)
		   )
		{
			*pIpAddr  =   (byte3 << 24)
				+ (byte2 << 16)
				+ (byte1 << 8)
				+  byte0;

			return 1;
		}
	}
	return 0;
}
int dhcpsnoop_bridge_change_cb(int fd, short ev, void *arg) {
	int return_value;
	int i;
	DIR *dp=NULL;
	struct dirent *ep;
	char dir_name[FILENAME];
	char ifname[IFNAMSIZ];
	int count=0;
	bridge_t *br_entry = NULL;
	bridge_key_t br_key;

	char *p;
	ssize_t numRead;
	int j;
	char notify_buf[BUF_LEN_NOTIFY] __attribute__ ((aligned(8)));
	struct inotify_event *event;
	inotify_event_bundle_t * inot = (inotify_event_bundle_t *)arg;

	numRead = read(fd, notify_buf, BUF_LEN_NOTIFY);

	syslog(LOG_INFO,"Inside %s",__FUNCTION__);

	for (p = notify_buf; p < notify_buf + numRead; ) {
		event = (struct inotify_event *) p;
		displayInotifyEvent(event);    
		if (event->mask & IN_DELETE_SELF ||
				event->mask & IN_MOVE_SELF ||
				event->mask & IN_IGNORED) {
			inotify_rm_watch(fd,inot->wd);
			close(fd);
			dhcpsnoop_inotify_brchange(inot,inot->br_name);
			return 0;
		}
		p += sizeof(struct inotify_event) + event->len;
	}
	return 0;
}

int dhcpsnoop_update_bridge(char *br_name, int vlan, 
		uint32_t rate, uint32_t ip_version) {
	regex_t regex;
	int return_value;
	int i;
	DIR *dp=NULL;
	struct dirent *ep;
	char dir_name[FILENAME];
	char ifname[IFNAMSIZ];
	bridge_t *br_entry = NULL;
	bridge_key_t br_key;

	br_key.hash = hash_str(br_name);

	hash_table_find(master_config_table, &br_key,
			sizeof(br_key),(void **)&br_entry);
	if (!br_entry) {
		syslog(LOG_ERR,"%s Failed to get dridge structure",__FUNCTION__);
		return 0;
	}
	snprintf(dir_name,FILENAME-1,"/sys/class/net/%s",br_name);
	dp = opendir (dir_name);

	if (dp == NULL) {
		syslog(LOG_ERR,"Couldn't open the directory %s",dir_name);
	} else {
		br_entry->if_count = 0;
		return_value = regcomp(&regex,"lower",0);
		while (ep = readdir (dp)) {
			return_value = regexec(&regex, ep->d_name, 0, NULL, 0);
			if (return_value == 0) {
				br_entry->if_count++;
				if (sscanf(ep->d_name,"lower_%s",ifname) != EOF) {
					syslog(LOG_INFO,"Interface %s",ifname);
					if (br_entry->vlan_aware) {
						if (dhcpsnoop_ebtables(ifname,vlan) < 0) {
							syslog(LOG_ERR,
							"Failed to update ebtable configuration\n");
						}
					}
					if (ip_version == 4 ) {
						if (dhcpsnoop_apply_interface_trust_config(ifname,
							vlan,br_entry->vlan_aware) < 0 ) {
							syslog(LOG_ERR,"Failed to update trust configuration\n");
						}
					} else if (ip_version == 6) {
						if (dhcpsnoop6_apply_interface_trust_config(ifname,
							vlan,br_entry->vlan_aware) < 0 ) {
							syslog(LOG_ERR,"Failed to update trust configuration\n");
						}
					}
					if (rate != DEFAULT_RATE && ip_version == 4) {
						if (dhcpsnoop_police_acl(ifname,vlan,rate)) {
							syslog(LOG_ERR,"Failed to update rate "
									"limit configuration\n");
						}
					}
					if (rate != DEFAULT_RATE && ip_version == 6) {
						if (dhcpsnoop6_police_acl(ifname,vlan,rate)) {
							syslog(LOG_ERR,"Failed to update rate "
									"limit configuration\n");
						}
					}
				}
			} else if (return_value == REG_NOMATCH) {
				continue;
			}
		}
		(void) closedir (dp);
	}
	regfree(&regex);
	return 1;
}

int dhcpsnoop_is_bridge_trad(char *bridge_name)
{
	bridge_key_t br_key;
	bridge_t *br_entry = NULL;
	uint32_t ret_vlan=0;
	memset(&br_key,0,sizeof(bridge_key_t));

	br_key.hash = hash_str(bridge_name);
	hash_table_find(master_config_table, &br_key,
                sizeof(br_key),(void **)&br_entry);	

	if (br_entry) {
		if (br_entry->vlan_aware) {
			return 0;
		} else {
			if (hash_table_count(br_entry->vlan_table) == 1) {
				int get_vlan_id_wrapper(void *data,void *arg) {
					vlan_t *vlan_entry = (vlan_t *) data;
					ret_vlan = vlan_entry->key.vlan; 
				}
				hash_table_foreach(br_entry->vlan_table,
					get_vlan_id_wrapper,NULL);
				return ret_vlan;	
			} else {
				syslog(LOG_ERR,"Configration error," 
					"More than one vlan specified for"
					" a traditional bridge");
				return 0;
			}
		}
	} 
	syslog(LOG_INFO,"%s Bridge not found in configuration",bridge_name);
	return 0;
}

/*void PrintData (unsigned char* data , int Size)
{
	fprintf(logfile,"Length %d\n",Size);
	fprintf(logfile,"====================================================\n");
	for(i=0 ; i < Size ; i++)
	{
		if( i!=0 && i%16==0)   //if one line of hex printing is complete...
		{
			fprintf(logfile,"         ");
			for(j=i-16 ; j<i ; j++)
			{
				if(data[j]>=32 && data[j]<=128)
					fprintf(logfile,"%c",(unsigned char)data[j]); //if its a number or alphabet

				else fprintf(logfile,"."); //otherwise print a dot
			}
			fprintf(logfile,"\n");
		} 

		if(i%16==0) fprintf(logfile,"   ");
			fprintf(logfile," %02X",(unsigned int)data[i]);

		if( i==Size-1)  //print the last spaces
		{
			for(j=0;j<15-i%16;j++) fprintf(logfile,"   "); //extra spaces

			fprintf(logfile,"         ");

			for(j=i-i%16 ; j<=i ; j++)
			{
				if(data[j]>=32 && data[j]<=128) fprintf(logfile,"%c",(unsigned char)data[j]);
				else fprintf(logfile,".");
			}
			fprintf(logfile,"\n");
		}
	}
	fprintf(logfile,"====================================================\n");
}*/

int dhcpsnoop_cleanup_config() {
	set_dirty_bridge_table();
	delete_dirty_bridge_table();
	return 0;	
}
void dhcpsnoop_cleanup_table() {

    int dhcpsnoop_cleanup_table_wrapper(void *data,void *arg) {
        dhcp_mac_ip_t *tmp_entry = (dhcp_mac_ip_t *) data;
        dhcp_mac_ip_t *del_entry = NULL;
        if (tmp_entry) {
            hash_table_delete(binding_table,
				&tmp_entry->key,sizeof(tmp_entry->key),del_entry);
            if(del_entry)
                free(del_entry);
        }
        return hash_table_foreach_done;
    }
    hash_table_foreach(binding_table,dhcpsnoop_cleanup_table_wrapper,NULL);	
	
	print_binding_table_to_file();
}

void dhcpsnoop_cleanup_table6() {

    int dhcpsnoop_cleanup_table6_wrapper(void *data,void *arg) {
        dhcp_mac_ip_6_t *tmp_entry = (dhcp_mac_ip_6_t *) data;
        dhcp_mac_ip_6_t *del_entry = NULL;
        if (tmp_entry) {
            hash_table_delete(binding_table6,
				&tmp_entry->key,sizeof(tmp_entry->key),del_entry);
            if(del_entry)
                free(del_entry);
        }
        return hash_table_foreach_done;  
    }
    hash_table_foreach(binding_table6,dhcpsnoop_cleanup_table6_wrapper,NULL); 
    
    print_binding_table_to_file6();
}

void dhcpsnoop_delete_binding_entries_vlan(uint32_t vlan) {
	int dhcpsnoop_delete_binding_entries_vlan_wrapper(void *data,void *arg) {
		dhcp_mac_ip_t *tmp_entry = (dhcp_mac_ip_t *) data;
		dhcp_mac_ip_t *del_entry = NULL;
		syslog(LOG_DEBUG,"Inside %s vlan %d",__FUNCTION__,vlan);
		if (tmp_entry->key.vlan == vlan) {
			syslog(LOG_DEBUG,"Inside %s vlan %d vlan %d",
				__FUNCTION__,vlan,tmp_entry->key.vlan);
			hash_table_delete(binding_table,
				&tmp_entry->key,sizeof(tmp_entry->key),del_entry);
			free(tmp_entry);
		}
	}
	hash_table_foreach(binding_table,
		dhcpsnoop_delete_binding_entries_vlan_wrapper,NULL);
	print_binding_table_to_file();
}

void dhcpsnoop6_delete_binding_entries_vlan(uint32_t vlan) {
    int dhcpsnoop6_delete_binding_entries_vlan_wrapper(void *data,void *arg) {
        dhcp_mac_ip_6_t *tmp_entry = (dhcp_mac_ip_6_t *) data;
        dhcp_mac_ip_6_t *del_entry = NULL;
		syslog(LOG_DEBUG,"Inside %s vlan %d",__FUNCTION__,vlan);
        if (tmp_entry->key.vlan == vlan) {
			syslog(LOG_DEBUG,"Inside %s vlan %d vlan %d",
				__FUNCTION__,vlan,tmp_entry->key.vlan);
            hash_table_delete(binding_table6,
                &tmp_entry->key,sizeof(tmp_entry->key),del_entry);
			free(tmp_entry);
        }
    }
    hash_table_foreach(binding_table6,
		dhcpsnoop6_delete_binding_entries_vlan_wrapper,NULL);
	print_binding_table_to_file6();
}

void dhcpsnoop_move_dhcp_files() {
	char *move_command[FILENAME];

	snprintf(move_command,FILENAME-1,"mv %s %s",db_file,db_bkup_file);
	move_command[FILENAME-1] = '\0';
	system(move_command);

	snprintf(move_command,FILENAME-1,"mv %s %s",db_file6,db_bkup_file6);
	move_command[FILENAME-1] = '\0';
	system(move_command);
}

void dhcpsnoop_sig_handler(int signum) {

	syslog(LOG_INFO,"Signal received %d",signum);
	system("rm /etc/cumulus/acl/policy.d/dhcpsnoop*");

	dhcpsnoop_move_dhcp_files();
	run_clacltool();
	exit(0);
}
