#include "cumulus.h"
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include "log.h"
#include "database.h"
#include "database_test.h"

#if DATABASE_TEST

typedef enum {
    UINT32_KEY_TYPE = 0,
    UINT64_KEY_TYPE,
    EXTERNAL_KEY_TYPE,
    MAX_KEY_TYPE
} key_type_t;

database_key_info_t  database_key_info_array[MAX_KEY_TYPE];

typedef struct mydata_ {
    database_key_t  key;
    int             flag;
} mydata_t;

/* uses embedded uint32_t key value */
static int _test_data_in_array(int                  NENTRIES,
                               database_type_t      db_type)
{
    database_key_info_t *key_info = &(database_key_info_array[UINT32_KEY_TYPE]);
    database_t *my_db;
    mydata_t   *rvdatap = NULL;
    int   find_key_min     = 43;
    int   find_key_max     = 45;
    int   delete_key_value = 44;
    char *key_buf;
    int   count;
    bool  success;
    int   rv = 0;

    mydata_t *myarray = calloc(NENTRIES, sizeof(mydata_t));

    int del_cb(void *data, void *cbarg) {
        mydata_t* this = (mydata_t*)data;
        if (this->flag == 0) {
            return database_foreach_delete;
        } else {
            key_buf = key_info->database_key_to_string(&(this->key));
            DLOG("keeping key %s\n", key_buf);
            free(key_buf);
            key_buf = NULL;
            return database_foreach_done;
        }
    };

    int read_cb(void *data, void *cbarg) {
	*(int*)cbarg = *(int*)cbarg + 1;
	return database_foreach_done;
    };

    DLOG("---- database test_data_in_array(): %d entries, DB type %d\n", NENTRIES, db_type);

    assert(NENTRIES > find_key_max);

    my_db = database_create(db_type, "test database", NENTRIES, database_key_info_array, MAX_KEY_TYPE);

    DLOG("after the alloc\n");

    for (int idx = 0; idx < NENTRIES; idx++) {
        key_info->database_key_populate((void*)&idx,
                                        key_info->key_type,
                                        &(myarray[idx].key));
	myarray[idx].flag = 0;
	success = database_entry_add(my_db,
				     &(myarray[idx].key),
				     (void*)&myarray[idx]);
	if (!success) {
            ERRLOG("DB entry add failed for key %d\n", idx);
            rv = -1;
	}
    }
    DLOG("after the add\n");

    count = 0;
    database_foreach(my_db, read_cb, &count);
    DLOG("%d entries in the database\n", count);

    database_key_t key;
    for (uint32_t key_value = find_key_min;
	 key_value < find_key_max;
	 key_value++) {
        key_info->database_key_populate((void*)&key_value,
                                        key_info->key_type,
                                        &key);
	success = database_entry_get(my_db, &key, (void*)&rvdatap);
        if (success) {
            if (rvdatap) {
                key_buf = key_info->database_key_to_string(&key);
                DLOG("setting flag to 1 for entry with key %s\n", key_buf);
                free(key_buf);
                key_buf = NULL;
                rvdatap->flag = 1;
            }
	} else {
	    ERRLOG("database_entry_get returned an error\n");
	    rv = -1;
	}
    }
    database_foreach(my_db, del_cb, NULL);

    DLOG("after the delete\n");

    key_info->database_key_populate((void*)&delete_key_value,
                                    key_info->key_type,
                                    &key);
    key_buf = key_info->database_key_to_string(&key);
    if (database_entry_delete(my_db, &key, NULL)) {
        DLOG("entry with key %s has been deleted\n", key_buf);
    } else {
        ERRLOG("entry with key %s is NOT IN THE DATABASE!!!\n", key_buf);
        rv = -1;
    }
    free(key_buf);
    key_buf = NULL;

    for (uint32_t key_value = find_key_min;
	 key_value < find_key_max;
	 key_value++) {
        key_info->database_key_populate((void*)&key_value,
                                        key_info->key_type,
                                        &key);
        key_buf = key_info->database_key_to_string(&key);
        DLOG("about to look up key %s\n", key_buf);
        success = database_entry_get(my_db, &key, (void*)&rvdatap);
        if (success) {
            if (rvdatap) {
                DLOG("%d is still there: key = %s : flag = %d\n",
                     key_value, key_buf, rvdatap->flag);
            } else {
                if (key_value == delete_key_value) {
                    DLOG("deleted key %s not found\n", key_buf);
                } else {
                    ERRLOG("entry with key %s is NOT IN THE DATABASE!!!\n", key_buf);
                    rv = -1;
                }
            }
        } else {
            ERRLOG("database_entry_get returned an error\n");
            rv = -1;
        }
        free(key_buf);
        key_buf = NULL;
    }

    DLOG("before database count\n");
    count = 0;
    database_foreach(my_db, read_cb, &count);
    DLOG("%d entries in the database\n", count);

    database_free(my_db, NULL);

    DLOG("after the free\n");
    free(myarray);

    return rv;
}

static int _test_data_malloc(int NENTRIES,
                             database_type_t db_type)
{
    database_key_info_t *key_info = &(database_key_info_array[UINT64_KEY_TYPE]);
    database_t     *my_db;
    mydata_t       *mydata;
    database_key_t *key;
    uint64_t    delete_key_min   = 24;
    uint64_t    delete_key_max   = 34;
    uint64_t    find_deleted_key = 27;
    uint64_t    find_key_value   = 10;
    char *key_buf;
    int   count;
    bool  success;
    int   rv = 0;

    int read_cb(void *data, void *cbarg) {
	mydata_t* tmp = data;
	if (tmp->flag != 0) {
            key_buf = key_info->database_key_to_string(&(tmp->key));
            DLOG("non-zero flag at entry with key %s\n", key_buf);
            free(key_buf);
            key_buf = NULL;
	}
	*(int*)cbarg = *(int*)cbarg + 1;
	return database_foreach_done;
    };

    void free_cb(void *data) {
	free(data);
    };

    DLOG("---- database test_data_malloc()\n");

    assert(NENTRIES > delete_key_max);

    /* create collisions!!!! */
    my_db = database_create(db_type, "test database", NENTRIES/4, database_key_info_array, MAX_KEY_TYPE);

    DLOG("after the alloc\n");

    for (uint64_t entry_value = 0; entry_value < NENTRIES; entry_value++) {
        mydata = malloc(sizeof(mydata_t));
        key_info->database_key_populate((void*)&entry_value,
                                        key_info->key_type,
                                        &(mydata->key));
	mydata->flag = 1;
	success = database_entry_add(my_db, &(mydata->key), mydata);
	if (!success) {
	    ERRLOG("database_entry_add returned an error\n");
	    rv = -1;
	}
    }
    DLOG("after the add\n");

    count = 0;
    database_foreach(my_db, read_cb, &count);
    DLOG("%d entries in the database\n", count);

    for (uint64_t key_value = delete_key_min; key_value < delete_key_max; key_value++) {
        key = key_info->database_key_create((void*)&key_value,
                                            key_info->key_type);
        key_buf = key_info->database_key_to_string(key);
        if (!database_entry_delete(my_db, key, (void*)&mydata)) {
            ERRLOG("COULD NOT FIND entry with key %s to delete\n", key_buf);
            rv = -1;
        } else {
            DLOG("deleted entry with key %s\n", key_buf);
            free(mydata);
        }
        free(key_buf);
        key_buf = NULL;
	free(key);
    }
    count = 0;
    database_foreach(my_db, read_cb, &count);

    DLOG("after the delete\n");

    key = key_info->database_key_create((void*)&find_key_value,
                                        key_info->key_type);
    key_buf = key_info->database_key_to_string(key);
    success = database_entry_get(my_db, key, (void*)&mydata);
    if (success) {
        if (mydata) {
            char *my_key_buf = key_info->database_key_to_string(&(mydata->key));
            DLOG("entry with key %s is still there: key = %s, flag = %d\n",
                 key_buf, my_key_buf, mydata->flag);
            free(my_key_buf);
        } else {
            ERRLOG("entry with key %s is NOT IN THE DATABASE!!!\n", key_buf);
            rv = -1;
        }
    } else {
        ERRLOG("database_entry_get returned an error\n");
        rv = -1;
    }
    free(key_buf);
    key_buf = NULL;
    free(key);

    key = key_info->database_key_create((void*)&find_deleted_key,
                                        key_info->key_type);
    key_buf = key_info->database_key_to_string(key);
    success = database_entry_get(my_db, key, (void*)&mydata);
    if (success) {
        if (mydata) {
            char *my_key_buf = key_info->database_key_to_string(&(mydata->key));
            ERRLOG("entry with deleted key %s is STILL THERE: key = %s : flag = %d\n",
                   key_buf, my_key_buf, mydata->flag);
            rv = -1;
            free(my_key_buf);
        } else {
            DLOG("entry with deleted key %s not found\n", key_buf);
        }
    } else {
        ERRLOG("database_entry_get returned an error\n");
    }
    free(key_buf);
    key_buf = NULL;
    free(key);

    count = 0;
    database_foreach(my_db, read_cb, &count);
    DLOG("%d entries in the table\n", count);

    database_free(my_db, free_cb);

    DLOG("after the free\n");

    return rv;
}

static int _test_key(database_type_t db_type)
{
    database_key_info_t *key_info = &(database_key_info_array[EXTERNAL_KEY_TYPE]);
    database_t *my_db;
    int   num_entries = 8;
    char *str_array[] =
        {"string1",
         "string2",
         "string3",
         "string4",
         "string5",
         "string6",
         "string7",
         "string8"};
    mydata_t obj_array[8];
    mydata_t *rvdatap = NULL;
    bool success;
    int  rv;

    my_db = database_create(db_type, "test database", num_entries, database_key_info_array, MAX_KEY_TYPE);
    if (!my_db) {
        ERRLOG("%s: database_create failed\n", __FUNCTION__);
        goto error;
    }
    DLOG("after the create\n");

    for (int idx = 0; idx < num_entries; ++idx) {
        database_key_t *key     = &(obj_array[idx].key);
        mydata_t       *my_data = &(obj_array[idx]);
        key_info->database_key_populate((void*)str_array[idx],
                                        key_info->key_type,
                                        key);
	char *key_buf = key_info->database_key_to_string(key);
	success = database_entry_add(my_db, key, my_data);
        if (!success) {
            ERRLOG("database_entry_add returned an error for %s\n", key_buf);
            rv = -1;
	    goto error;
        } else {
	    DLOG("added entry with key %s\n", key_buf);
	}
	free(key_buf);
	key_buf = NULL;
    }
    DLOG("after the add\n");

    for (int idx = 0; idx < num_entries; ++idx) {
        database_key_t *key = &(obj_array[idx].key);
        success = database_entry_get(my_db, key, (void*)&rvdatap);
	char *key_buf = key_info->database_key_to_string(key);
        if (success) {
            if (rvdatap) {
                DLOG("setting flag to 1 for entry with key %s\n", key_buf);
                rvdatap->flag = 1;
            }
        } else {
            ERRLOG("database_entry_get returned an error for key %s\n", key_buf);
            rv = -1;
            free(key_buf);
            key_buf = NULL;
	    goto error;
        }
        free(key_buf);
        key_buf = NULL;
    }

    return 0;

 error:
    ERRLOG("%s: failed with key type %d\n", __FUNCTION__, key_info->key_type);
    return rv;

}

static int _uint32_key_populate(void           *key_value_p,
                                int             key_type,
                                database_key_t *key)
{
    if (!key) {
        ERRLOG("%s: key is NULL\n", __FUNCTION__);
        goto error;
    }
    /* The entire database_key_t object is used as
       the database key: it is composed of the
       key type and the unique identifier stored
       in the uint32_value field */
    memset(key, 0, sizeof(database_key_t));
    key->type         = key_type;
    key->uint32_value = *((uint32_t*)key_value_p);
    return 0;

 error:
    return -1;
}

static database_key_t* _uint32_key_create(void* key_value, int key_type)
{
    database_key_t *key;
    int database_key_size = sizeof(database_key_t);

    key = CALLOC(1, database_key_size);
    _uint32_key_populate(key_value, key_type, key);

    return key;
}

static char* _uint32_key_to_string(database_key_t *key)
{
    int   key_str_size;
    char *key_str;

    if (!key) {
        ERRLOG("%s: key is NULL\n", __FUNCTION__);
        goto error;
    }

    /* keep the string contents consistent with the calculated size */
    key_str_size = 64 + sizeof(key->type) + sizeof(key->uint32_value);
    key_str = CALLOC(key_str_size, sizeof(char));
    snprintf(key_str, key_str_size, "<type %d, value %d>", key->type, key->uint32_value);

    return key_str;

 error:
    return NULL;
}

static void * _uint32_key_ptr(database_key_t *key)
{
    /* the database_key_t object is used as the key */
    return (void*)key;
}

static int _uint64_key_populate(void           *key_value_p,
                                int             key_type,
                                database_key_t *key)
{
    if (!key) {
        ERRLOG("%s: key is NULL\n", __FUNCTION__);
        goto error;
    }
    /* The entire database_key_t object is used as
       the database key: it is composed of the
       key type and the unique identifier stored
       in the uint64_value field */
    memset(key, 0, sizeof(database_key_t));
    key->type         = key_type;
    key->uint64_value = *((uint64_t*)key_value_p);
    return 0;

 error:
    return -1;
}

static database_key_t* _uint64_key_create(void* key_value, int key_type)
{
    database_key_t *key;
    int database_key_size = sizeof(database_key_t);

    key = CALLOC(1, database_key_size);
    _uint64_key_populate(key_value, key_type, key);

    return key;
}

static char* _uint64_key_to_string(database_key_t *key)
{
    int   key_str_size;
    char *key_str;

    if (!key) {
        ERRLOG("%s: key is NULL\n", __FUNCTION__);
	goto error;
    }

    /* keep the string contents consistent with the calculated size */
    key_str_size = 64 + sizeof(key->type) + sizeof(key->uint64_value);
    key_str = CALLOC(key_str_size, sizeof(char));
    snprintf(key_str, key_str_size, "<type %d, value %"PRIu64">", key->type, key->uint64_value);

    return key_str;

 error:
    return NULL;
}

static void * _uint64_key_ptr(database_key_t *key)
{
    /* the database_key_t object is used as the key */
    return (void*)key;
}

static int _external_key_populate(void           *key_value_p,
                                  int             key_type,
                                  database_key_t *key)
{
    int key_obj_size;

    key_obj_size = sizeof(database_key_t);

    if (!key) {
        ERRLOG("%s: key is NULL\n", __FUNCTION__);
        goto error;
    }
    /* The external object is used as
       the database key */
    memset(key, 0, key_obj_size);
    DLOG("%s: key obj size: %d\n", __FUNCTION__, key_obj_size);
    key->type      = key_type;
    key->value_ptr = key_value_p;
    return 0;

 error:
    return -1;
}

static database_key_t* _external_key_create(void* key_value, int key_type)
{
    database_key_t *key;
    int key_obj_size;

    key_obj_size = sizeof(database_key_t);
    key = CALLOC(1, key_obj_size);
    _external_key_populate(key_value, key_type, key);

    return key;
}

static char* _external_key_to_string(database_key_t *key)
{
    int   key_str_size;
    char *value_str;
    char *key_str;

    if (!key) {
        ERRLOG("%s: key is NULL\n", __FUNCTION__);
	goto error;
    }

    /* keep the string contents consistent with the calculated size */
    value_str = (char*)(key->value_ptr);
    key_str_size = 64 + sizeof(key->type) + strlen(value_str);
    key_str = CALLOC(key_str_size, sizeof(char));
    snprintf(key_str, key_str_size, "<type %d, value %s>",
	     key->type, value_str);

    return key_str;

 error:
    return NULL;
}

static void * _external_key_ptr(database_key_t *key)
{
    /* the external object is used as the key */
    return key->value_ptr;
}

int _key_type_create(void)
{
    database_key_info_t *key_info;

    /* the entire database_key_t object is used as the key,
       populating the uint32 value embededed in the object */
    key_info = &(database_key_info_array[UINT32_KEY_TYPE]);
    key_info->key_size               = sizeof(database_key_t);
    key_info->key_type               = UINT32_KEY_TYPE;
    key_info->database_key_create    = _uint32_key_create;
    key_info->database_key_populate  = _uint32_key_populate;
    key_info->database_key_to_string = _uint32_key_to_string;
    key_info->database_key_ptr       = _uint32_key_ptr;

    /* the entire database_key_t object is used as the key,
       populating the uint64 value embededed in the object */
    key_info = &(database_key_info_array[UINT64_KEY_TYPE]);
    key_info->key_size               = sizeof(database_key_t);
    key_info->key_type               = UINT64_KEY_TYPE;
    key_info->database_key_create    = _uint64_key_create;
    key_info->database_key_populate  = _uint64_key_populate;
    key_info->database_key_to_string = _uint64_key_to_string;
    key_info->database_key_ptr       = _uint64_key_ptr;

    /* the external object as the key */
    key_info = &(database_key_info_array[EXTERNAL_KEY_TYPE]);
    key_info->key_size               = 64;
    key_info->key_type               = EXTERNAL_KEY_TYPE;
    key_info->database_key_create    = _external_key_create;
    key_info->database_key_populate  = _external_key_populate;
    key_info->database_key_to_string = _external_key_to_string;
    key_info->database_key_ptr       = _external_key_ptr;
    return 0;
}

int database_test(void)
{
    int rv;

    rv = _key_type_create();
    if (rv) {
        goto error;
    }

    rv = _test_data_in_array(64, DATABASE_HASHTABLE);
    if (rv) {
        goto error;
    }

    rv = _test_data_malloc(64, DATABASE_HASHTABLE);
    if (rv) {
        goto error;
    }

    rv = _test_key(DATABASE_HASHTABLE);
    if (rv) {
        goto error;
    }

    rv = _test_key(DATABASE_HASHTABLE);
    if (rv) {
        goto error;
    }


    LOG("%s: passed\n", __FUNCTION__);
    return 0;

 error:
    ERRLOG("%s: failed\n", __FUNCTION__);
    return rv;
}

#else

int database_test(void) {
    return 0;
}

#endif /* DATABASE_TEST */
