/* Copyright 2019-2020 Cumulus Networks, Inc.  All rights reserved. */

/*
 * A database object and methods encapsulate storage mechanism implementation decisions.
 */

#ifndef _DATABASE_H_
#define _DATABASE_H_

/* maintained as a separate file to export the callback return values
   to specific database implementation types */
#include "database_rv.h"

typedef enum {
    DATABASE_NONE = 0,
    DATABASE_HASHTABLE,
} database_type_t;

/*
 * The key used to access a database is either
 * -- a database_key_t object, with a key type and embedded unique identifier, or
 * -- the value stored at the database_key_t.value_ptr address.
 *
 * A database_key_t object type associates a key object with
 * a database_key_info_t object.
 *
 * The caller is responsible for creating the database_key_info_t
 * object to support the desired key characteristics.
 */
typedef struct database_key_ {
    uint16_t type;
    uint16_t external_flag;
    int      external_value_size;
    union {
        uint32_t uint32_value;
        uint64_t uint64_value;
        void    *value_ptr;
        uint8_t val[255];
    };
} database_key_t;

/* The database_key_info_t object fields include
 * -- key_type
 * -- key_size
 *    -- embedded value: the database_key_t size, or
 *    -- external value:  the size of the object located at database_key_t.value_ptr
 * -- database_key_create: allocates the key object and populates the fields
 * -- database_key_populate: populates an existing key object
 * -- database_key_to_string: creates a string which describes the key
 * -- database_key_ptr: returns the address of the key
 *    -- embedded value: the address of the database_key_t object
 *    -- external value: database_key_t.value_ptr
 */
typedef struct database_key_info_ {
    int key_type;
    int key_size;
    database_key_t*(*database_key_create)(void *key_value, int key_type);
    int(*database_key_populate)(void *key_value, int key_type, database_key_t *key);
    char *(*database_key_to_string)(database_key_t *key);
    void*(*database_key_ptr)(database_key_t *key);
} database_key_info_t;

typedef struct hash_table_ hash_table_t;
typedef struct database_   database_t;

typedef struct database_ {
    char                *name;
    database_type_t      type;
    uint64_t             valid;
    database_key_info_t *key_info_array;
    int                  num_key_types;
    union {
        hash_table_t *ht;
    } container;
    bool(*entry_add)(database_t *db, database_key_t *key, void *entry);
    bool(*entry_get)(database_t *db, database_key_t *key, void **entry);
    bool(*entry_delete)(database_t *db, database_key_t *key, void **entry);
    int(*container_count)(database_t *db);
    void(*container_foreach)(database_t *db, int (*foreach_cb)(void *data, void *cbarg), void *cbarg);
    void(*container_free)(database_t *db, void (*free_cb)(void* data));
} database_t;

database_t* database_create(database_type_t type,
                            char *name,
                            int size,
                            database_key_info_t *key_info_array,
                            int num_key_types);
bool        database_entry_get(database_t *db, database_key_t* key, void** entry);
bool        database_entry_add(database_t *db, database_key_t *key, void *entry);
bool        database_entry_delete(database_t *db, database_key_t *key, void **entry);
bool        database_count(database_t *db, int *num_items_ptr);
bool        database_foreach(database_t *db, int (*foreach_cb)(void *data, void *cbarg), void *cbarg);
bool        database_free(database_t *db, void (*free_cb)(void* data));

#endif /* _DATABASE_H_ */
