/* Copyright 2019-2020 Cumulus Networks, Inc.  All rights reserved. */
#include "cumulus.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>

#include "log.h"
#include "hashtable.h"
#include "database.h"

#define DATABASE_VALID_FLAG   0xcc1155aa99330022

typedef enum { ENTRY_NONE = 0,
               ENTRY_OPTIONAL,
               ENTRY_REQUIRED } entry_flag_t;

static bool _ht_add(database_t *db, database_key_t *key, void *entry)
{
    int   key_size = db->key_info_array[key->type].key_size?
                     db->key_info_array[key->type].key_size:
                     key->external_value_size;
    void *key_ptr  = db->key_info_array[key->type].database_key_ptr(key);

    return hash_table_add(db->container.ht, key_ptr, key_size, entry);
}

static bool _ht_find(database_t *db, database_key_t *key, void **entry)
{
    int   key_size = db->key_info_array[key->type].key_size?
                     db->key_info_array[key->type].key_size:
                     key->external_value_size;
    void *key_ptr  = db->key_info_array[key->type].database_key_ptr(key);

    return hash_table_find(db->container.ht, key_ptr, key_size, entry);
}

static bool _ht_delete(database_t *db, database_key_t *key, void **entry)
{
    int   key_size = db->key_info_array[key->type].key_size?
                     db->key_info_array[key->type].key_size:
                     key->external_value_size;
    void *key_ptr  = db->key_info_array[key->type].database_key_ptr(key);

    return hash_table_delete(db->container.ht, key_ptr, key_size, entry);
}

static int _ht_count(database_t *db)
{
    return hash_table_count(db->container.ht);
}

static void _ht_foreach(database_t *db, int (*foreach_cb)(void *data, void *cbarg), void *cbarg)
{
    hash_table_foreach(db->container.ht, foreach_cb, cbarg);
}

static void _ht_free(database_t *db, void (*free_cb)(void* data))
{
    hash_table_free(db->container.ht, free_cb);
}

database_t* database_create(database_type_t      type,
                            char                *name,
                            int                  size,
                            database_key_info_t *key_info_array,
                            int                  num_key_types)
{
    database_t *db = CALLOC(1, sizeof(database_t));

    if (type == DATABASE_HASHTABLE) {
        db->container.ht = hash_table_alloc(size);
        if (db->container.ht == NULL) {
            ERRLOG("%s: could not allocate a hash table\n", __FUNCTION__);
            free(db);
            goto error;
	}
	db->entry_add         = _ht_add;
	db->entry_get         = _ht_find;
	db->entry_delete      = _ht_delete;
	db->container_count   = _ht_count;
	db->container_foreach = _ht_foreach;
	db->container_free    = _ht_free;
    } else {
        ERRLOG("%s: database type %d not supported\n", __FUNCTION__, type);
        free(db);
        goto error;
    }
    db->name = CALLOC(strlen(name) + 1, sizeof(char));
    strcpy(db->name, name);
    db->type  = type;
    db->valid = DATABASE_VALID_FLAG;
    db->key_info_array = key_info_array;
    db->num_key_types  = num_key_types;

    /* no errors */
    return db;

 error:
    return NULL;
}

static bool _sanity_check_op(database_t     *db,
                             database_key_t *key,
                             void           *entry_ptr,
                             entry_flag_t    entry_flag,
                             const char     *op_name)
{
    database_key_info_t *this_key_info;
    char *key_buf;
    char *name;

    if (!db) {
        ERRLOG("%s: db is NULL\n", __FUNCTION__);
        goto error;
    }

    if (!db->valid) {
        ERRLOG("%s: db is not initialized\n", __FUNCTION__);
        goto error;
    }

    if (entry_flag == ENTRY_REQUIRED) {
        if (!entry_ptr) {
            ERRLOG("%s: entry is NULL\n", __FUNCTION__);
            goto error;
	}

	if (key == NULL) {
            ERRLOG("%s: key is NULL\n", __FUNCTION__);
            goto error;
	}

	if (key->type >= db->num_key_types) {
            ERRLOG("%s: key type %d is out of bounds: %d key types supported\n",
                   __FUNCTION__, key->type, db->num_key_types);
            goto error;
	}
    }

    /* no errors */
    return TRUE;

 error:
    if (db) {
        if (key->type < db->num_key_types) {
            this_key_info = &(db->key_info_array[key->type]);
            key_buf = this_key_info->database_key_to_string(key);
        } else {
            key_buf = "<invalid>";
        }
        name = ((db && db->valid) ? db->name : "<no db name>");
        if (entry_flag) {
            ERRLOG("%s failed for %s %s: db %p, key %s, entry pointer %p\n",
                   __FUNCTION__,
                   name,
                   op_name,
                   db,
                   key_buf,
                   entry_ptr);
            if (key->type < db->num_key_types) {
                free(key_buf);
            }
        } else {
            ERRLOG("%s failed for %s %s: db %p\n",
                   __FUNCTION__, name, op_name, db);
        }
    }
    return FALSE;
}

bool database_entry_get(database_t *db, database_key_t* key, void** entry)
{
    bool success;
    entry_flag_t entry_flag = ENTRY_REQUIRED;

    if (!(_sanity_check_op(db, key, entry, entry_flag, __FUNCTION__))) {
        goto error;
    }

    success = db->entry_get(db, key, entry);
    if (!success) {
        *entry = NULL;
    }

    /* no errors */
    return TRUE;

 error:
    *entry = NULL;
    return FALSE;
}

bool database_entry_add(database_t *db, database_key_t *key, void *entry)
{
    bool         success;
    entry_flag_t entry_flag = ENTRY_REQUIRED;

    if (!(_sanity_check_op(db, key, entry, entry_flag, __FUNCTION__))) {
        goto error;
    }

    success = db->entry_add(db, key, entry);
    if (!success) {
	goto error;
    }

    /* no errors */
    return TRUE;

 error:
    return FALSE;
}

bool database_entry_delete(database_t *db, database_key_t *key, void **entry)
{
    bool success;
    entry_flag_t entry_flag = ENTRY_OPTIONAL;

    if (!(_sanity_check_op(db, key, entry, entry_flag, __FUNCTION__))) {
        goto error;
    }

    success = db->entry_delete(db, key, entry);
    if (!success) {
        goto error;
    }

    /* no errors */
    return TRUE;

 error:
    return FALSE;
}

bool database_count(database_t *db, int *num_items_ptr)
{
    entry_flag_t  entry_flag = ENTRY_NONE;

    if (!(_sanity_check_op(db, NULL, NULL, entry_flag, __FUNCTION__))) {
        goto error;
    }

    if (num_items_ptr == NULL) {
        ERRLOG("%s: num_items pointer is NULL\n", __FUNCTION__);
        goto error;
    }
    *num_items_ptr = db->container_count(db);

    /* no errors */
    return TRUE;

 error:
    return FALSE;
}

bool database_foreach(database_t *db, int (*foreach_cb)(void *data, void *cbarg), void *cbarg)
{
    entry_flag_t  entry_flag = ENTRY_NONE;

    if (!(_sanity_check_op(db, NULL, NULL, entry_flag, __FUNCTION__))) {
        goto error;
    }

    db->container_foreach(db, foreach_cb, cbarg);

    /* no errors */
    return TRUE;

 error:
    return FALSE;
}

bool database_free(database_t *db, void (*free_cb)(void* data))
{
    entry_flag_t  entry_flag = ENTRY_NONE;

    if (!(_sanity_check_op(db, NULL, NULL, entry_flag, __FUNCTION__))) {
        goto error;
    }

    free(db->name);
    db->container_free(db, free_cb);

    /* free the database object */
    free(db);

    /* no errors */
    return TRUE;

 error:
    return FALSE;
}
