#!/usr/bin/python
# -*- coding: utf-8 -*-

from __future__ import print_function
try:
    import argparse
    import os
    import random, time
    import re
    import shutil
    import subprocess
    import sys
    import termios, fcntl
except ImportError as e:
    raise ImportError (str(e) + "- required module not found")

def wait_key():
    ''' Wait for a key press on the console and return it. '''
    result = None
    if os.name == 'nt':
        import msvcrt
        result = msvcrt.getch()
    else:
        import termios
        fd = sys.stdin.fileno()
        oldterm = termios.tcgetattr(fd)
        newattr = termios.tcgetattr(fd)
        newattr[3] = newattr[3] & ~termios.ICANON & ~termios.ECHO
        termios.tcsetattr(fd, termios.TCSANOW, newattr)
        try:
            result = sys.stdin.read(1)
        except IOError:
            pass
        finally:
            termios.tcsetattr(fd, termios.TCSAFLUSH, oldterm)
    return result


def ParseCmdLine():
    """
    Parse the command line options using the standard argparse library.
    """
    # The default values for the arguments
    parser = argparse.ArgumentParser(description="Stress Test filesystem.")
    parser.add_argument("-a", "--autocount", action='store_true', dest='autocount',
                        help="Automatically calculate file count. default: False")
    parser.add_argument("-c", "--count", action='store', dest='filecount', default=100000, type=int,
                        help="Number of files to create. default: 100000")
    parser.add_argument("-s", "--size", action='store', dest='filesize', default=1024, type=int,
                        help="Maximum file size. default: 1024")
    parser.add_argument("-t", "--threshold", action='store', dest='threshold', default=85, type=int,
                        help="Maximum disk usage. default: 85 percent")
    parser.add_argument("--readtest", action='store_true', dest='readtest',
                        help="Perform read tests. default: False")
    args = parser.parse_args()
    return(args)


args = ParseCmdLine()

filecount = args.filecount
filesize = args.filesize
threshold = args.threshold

# automatic calculate proper stress
if args.autocount:
    print("Automatically calculating file count")
    cmd = subprocess.check_output("btrfs fi usage -b /", shell=True)
    output = cmd.split('\n')
    disksize = output[1].split()[2]
    diskused = output[2].split()[2]
    diskfree = output[6].split()[2]
    # if the target disk space to fill is less than the free space,
    # calculate the number of files to create to hit the intended threshold.
    if (disksize * (threshold/100) < diskfree):
        filecount = int(round(int(diskfree) / filesize))

print("Beginning to create {} files with sizes between 50-100% of {} Bytes".format(filecount, filesize))
flush = "sudo su -c 'sync ; echo 3 > /proc/sys/vm/drop_caches'"

randfile = open("/dev/urandom", "r")

# Getting the current work dir and create test dir
currentdir = os.getcwd()
testpath = os.path.join(currentdir, "test_fs_bench")

starttime = time.time()
print("\nSetting up work directory: {}".format(testpath))
if os.path.exists(testpath):
    print("Removing old test folder.".format(testpath))
    shutil.rmtree(testpath)
print("\nCreate test folder: {}".format(testpath))
os.mkdir(testpath)
print(time.time() - starttime)
os.system(flush)

print("\nCreating {} files:".format(str(filecount)))
starttime = time.time()
count = 0
stop = False
while count < filecount and not stop:
    count += 1
    filename = os.path.join(testpath, unicode(count))
    #print("Creating File: {}".format(filename))
    rand = randfile.read(int(filesize * 0.5 + filesize * random.random()))
    outfile = open(filename, "w")
    outfile.write(rand)
    # print("Count: "+str(i), end='\r')
    # Check that the disk space is not exhausted
    if not (count % 100):
        cmd = subprocess.check_output("btrfs fi usage -b /", shell=True)
        output = cmd.split('\n')
        d = output[1].split()[2]
        a = output[2].split()[2]
        sys.stdout.write("\x1b[2K\r")
        sys.stdout.write("\rCreated {} files, Device Size: {} Bytes, Allocated: {} Bytes".format(str(count), d, a))
        sys.stdout.flush()
        # check if we are at the disk utilization threshold, and wait for user input.
        if int(a) >= (int(d) * threshold / 100):
            print("\nFilesystem is currently at {}% utilization.".format( (int(a)*100)/int(d) ))
            print("Press 'c' to continue creating files, or any other key to stop.")
            while not stop:
                key_press = wait_key()
                if "c" == key_press:
                    print("Continue creating files...")
                    break
                else:
                    print("Stopping file creation, and moving to next test.")
                    stop = True
                    break

print(time.time() - starttime)
os.system(flush)

raw_input("Press Enter to continue to rewrite test.")

# compile a list of files created
filelist = [os.path.join(testpath, f) for f in os.listdir(testpath) if os.path.isfile(os.path.join(testpath, f))]

if count != len(filelist):
    print("File count is {}, and filelist is {}. Something may break.".format(count, len(filelist)))

print("\nRewrite files:")
starttime = time.time()
for i in filelist:
    #print("Rewriting File: {}".format(i))
    rand = randfile.read(int(filesize * 0.5 + filesize * random.random()))
    outfile = open(filelist[int(random.random() * count)], "w")
    outfile.write(rand)
print(time.time() - starttime)
os.system(flush)

if args.readtest:
    raw_input("Press Enter to continue to read tests.")

    print("\nRead All Linearly:")
    starttime = time.time()
    for i in filelist:
        testfile = os.path.join(testpath, i)
        infile = open(testpath + "/" + unicode(i), "r")
        outfile.write(infile.read());
    print(time.time() - starttime)
    os.system(flush)

    print("\nRead 10% Randomly:")
    starttime = time.time()
    outfile = open("/dev/null", "w")
    for i in int(count / 10):
        infile = open(testpath + "/" + unicode(int(random.random() * count)), "r")
        outfile.write(infile.read());
    print(time.time() - starttime)
    os.system(flush)

raw_input("Press Enter to continue to delete test files.")

print("\ndelete all files:")
starttime = time.time()
os.system("rm -rf test")
print(time.time() - starttime)
os.system(flush)

