#!/usr/bin/env python
# Copyright 2012-2020 Cumulus Networks, Inc
#

"""
Compare kernel vxlan info with hw vxlan info
"""

from cStringIO import StringIO
from ipaddr import IPv6Address, IPNetwork, IPv4Network, IPv6Network, AddressValueError
from pprint import pformat, pprint
from subprocess import Popen, PIPE, check_output, CalledProcessError
import argparse
import json
import os
import re
import sys
import cumulus.platforms
from nlmanager.kernel_nlhelpers import *
from nlmanager.nlpacket import Route, Neighbor, Link
from nlmanager.nlmanager import NetlinkManager
import yaml
from cumulus.mlx import mlx_open_connection
from cumulus.mlx import mlx_close_connection
from cumulus.mlx import mlx_get_uc_route
from cumulus.mlx import mlx_get_port_ids
from cumulus.mlx import mlx_get_neighbor
from cumulus.mlx import mlx_get_hw_bondInfo
import cumulus.porttab

class Asic:

    def __init__(self):
        #
        # The various tables from kernel/HW are collected into dictionaries
        # with these mappings:
        #
        #PI dictionary for comparing with kernel objects
        self.asicVxlanDict = {}
        #Lag interface mapping
        self.hwBonds = {}
        self.asicVxlanDict = {}
        #Enable the vxlan sfs node
        self.vxlanFuseNode = "/cumulus/switchd/debug/vxlan/info"
        self.switchd_debug_enabled = self.get_switchd_debug()
        if not self.switchd_debug_enabled:
            self.enable_switchd_debug()

        #hw dump enable
        self.hwDumpCtrl = "/cumulus/switchd/ctrl/hw/enable"

        self.hwBridgeDumpCtrl = "/cumulus/switchd/ctrl/hw/bridge"
        self.hwBridgeDumpLog = "/cumulus/switchd/run/hw/bridge/all"

        self.hwLagDumpCtrl = "/cumulus/switchd/ctrl/hw/lag"
        self.hwLagDumpLog = "/cumulus/switchd/run/hw/lag/all"

        self.hwTunnelDumpCtrl = "/cumulus/switchd/ctrl/hw/tunnel"
        self.hwTunnelDumpLog = "/cumulus/switchd/run/hw/tunnel/all"

        self.hwRouterDumpCtrl = "/cumulus/switchd/ctrl/hw/router"
        self.hwRouterDumpLog = "/cumulus/switchd/run/hw/router/all"

        #Helper Libraries
        self.portInfoToLinuxDict = {}
        self.tunnelMapHelper = {}
        self.hwBridgePortMapHelper = {}
        self.hwBridgeFdbHelperDict = {}
        self.hwRouteTableHelperDict = {}
        self.hwHostTableHelperDict = {}
        self.l3EgressIfHelperDict = {}

        self.enable_hw_dump()
        self.buildSdkPortToLinuxDict()
        self.buildHwBondsDict()

    def get_switchd_debug(self):
        return os.path.isfile(self.vxlanFuseNode)

    def enable_switchd_debug(self):
        try:
            os.system('echo 1 > /cumulus/switchd/ctrl/debug')
        except:
            raise Exception("Switchd Fuse Node not enabled")

    def disable_switchd_debug(self):
        try:
            os.system('echo 1 > /cumulus/switchd/ctrl/debug')
        except:
            raise Exception("Switchd Fuse Node not disabled")

    def _insert_ip_addr_mask(self, ipaddr):
        """
        Inserts mask for host ips
        """
        if ipaddr.count(".") > 0:
            ipaddr += "/32"
        else:
            ipaddr += "/128"
        return ipaddr

    @property
    def get_sdk_to_linux(self):
        sdk_to_linux = {}
        pt = cumulus.porttab.porttab(use_hw=False)
        for p in pt.get_linux_ports():
            sdk_to_linux[pt.linux2mlx(p)] = p
        return sdk_to_linux

    @property
    def get_linux_to_sdk(self):
        linux_to_sdk = {}
        sdk_to_linux = get_sdk_to_linux()
        for sdkport in sdk_to_linux.keys():
            linux_to_sdk[sdk_to_linux.get(sdkport)] = sdkport
        return linux_to_sdk[0]

    def enable_hw_dump(self):
        try:
            os.system("echo TRUE > "+self.hwDumpCtrl)
            os.system("echo TRUE > "+self.hwBridgeDumpCtrl)
            os.system("echo TRUE > "+self.hwLagDumpCtrl)
            os.system("echo TRUE > "+self.hwTunnelDumpCtrl)
            os.system("echo TRUE > "+self.hwRouterDumpCtrl)
        except:
            raise Exception("Failed to enable hw dump")

    def buildSdkPortToLinuxDict(self):
        mlx_open_connection()
        swidList=[0]
        portIdDict = mlx_get_port_ids(swidList)
        self.portInfoToLinuxDict = self.get_sdk_to_linux
        mlx_close_connection()

    def buildHwBondsDict(self):
        self.hwBonds = mlx_get_hw_bondInfo()

    def parseTunnelInfo(self):
        """
        Parses the Tunnel Info from mlx sfs node
        """
        try:
            with open(self.hwTunnelDumpLog) as sfs:
                tunnelParser = yaml.safe_load(sfs)
            sfs.close()
        except:
            raise Exception("Failed to parse the sfs file %s" \
                            % (self.hwTunnelDumpLog))

        rawData = {}
        rawData = tunnelParser.get("hw-tunnels", {})
        if not rawData:
            print "No hw tunnels are configured"
            return

        tunnelPortKeyL = rawData.keys()
        if not tunnelPortKeyL:
            print "No hw tunnel port keys are configured"
            return
        tunnelPortKey = tunnelPortKeyL[0]
        sip = rawData[tunnelPortKey]["hw-tunnel-attribute-entry"]\
            ["hw-tunnel-nve-attribute-entry"]\
            ["hw-tunnel-nve-encap-attribute-entry"]["underlay-sip"]

        tunnelMapDict = rawData[tunnelPortKey]["hw-tunnel-map-entries"]
        if not tunnelMapDict:
            print "No tunnel Map Entries Found"
            return

        for keys, values in tunnelMapDict.iteritems():
            vni = tunnelMapDict[keys]["hw-tunnel-map-entry"]["vni"]
            bridgeId = tunnelMapDict[keys]["hw-tunnel-map-entry"]["bridge-id"]

            if bridgeId not in self.tunnelMapHelper.keys():
                self.tunnelMapHelper[bridgeId] = values
                self.tunnelMapHelper[bridgeId]["sip"] = sip
                self.asicVxlanDict[vni] = {"bridgeId"      : bridgeId,
                                           "localIp"       : sip,
                                           "localPorts"    : [],
                                           "rawLocalPorts" : [],
                                           "localMacs"     : [],
                                           "rawLocalMacs"  : [],
                                           "remoteMacs"    : [],
                                           "rawRemoteMacs" : []}

    def parseBridgeInfo(self):
        """
        Parses the Bridge info from the mlx sfs node to derive the
        bridge local ports and local/remote macs
        """
        try:
            with open(self.hwBridgeDumpLog) as sfs:
                bridgeParser = yaml.safe_load(sfs)
            sfs.close()
        except:
            raise Exception("Failed to parse the sfs file %s" \
                            % (self.hwBridgeDumpLog))
        rawData = {}
        rawData = bridgeParser.get("hw-bridges", {})
        if not rawData:
            print "No hw bridge programmed"
            return

        for keys, values in rawData.iteritems():
            bridgeId = int(keys.split()[1])
            self.hwBridgePortMapHelper[bridgeId] = values

        rawData = {}
        rawData = bridgeParser.get("hw-fdb", {})
        if not rawData:
            print "No hw fdb entries programmed"
            return

        bridgeFdbData = rawData.get("hw-bridge-fdb-unicast-multicast-entries", {})
        if not bridgeFdbData:
            print "No hw-bridge-fdb unicast and multicast entries are configured"
            return

        for keys, values in bridgeFdbData.iteritems():
            bridgeId = int(keys.split()[1])
            self.hwBridgeFdbHelperDict[bridgeId] = \
                bridgeFdbData[keys]["hw-bridge-fdb-unicast " +str(bridgeId)]

    def buildRouterNeighDict(self):
        """
        Builds the router/neigh dictionary
        """
        mlx_open_connection()
        routeTable = mlx_get_uc_route()
        for p, routeEntry in routeTable.iteritems():
            netAddr = p.split(' ')[1]
            self.hwRouteTableHelperDict[netAddr] = routeEntry

        neighTable = mlx_get_neighbor()
        for p, neighEntry in neighTable.iteritems():
            neighEntry = p.split()
            rif = int(neighEntry[1])
            nh = neighEntry[0]
            self.hwHostTableHelperDict[(nh, rif)] = neighEntry
        mlx_close_connection()

        with open(self.hwRouterDumpLog) as sfs:
            l3EgressIfParser = yaml.safe_load(sfs)
        sfs.close()
        rawData = {}
        rawData = l3EgressIfParser["hw-router-interfaces"]

        for keys, values in rawData.iteritems():
            rif = int(keys.split()[1])
            hwlogPortDict = rawData[keys]\
                        ["hw-router-interface-param"]
            neighEntryDict = rawData[keys]\
                        ["hw-router-interface-neighbor-entry ipv4 "+str(rif)]\
                        ["ipv4-neighbors"]
            attrDict = rawData[keys]["hw-router-interface-attributes"]
            mtu = attrDict["mtu"]
            if not neighEntryDict:
                destMac = None
                continue
            else:
                destMac = neighEntryDict\
            ["hw-router-neighbor-entry"]["hw-router-neighbor-data"]["mac-address"]
            #TODO Support other interface types in egress lif
            if hwlogPortDict["interface-type"] == "vport":
                hwLogbasePortDict = hwlogPortDict["hw-port-vport-base"]
                for keys in hwLogbasePortDict.keys():
                    if keys == "vlan":
                        dstVlan = hwLogbasePortDict["vlan"]
                    else:
                        if hwLogbasePortDict[keys]["port-type"] == "lag":
                            lagId = hwLogbasePortDict[keys]["lag-id"]
                            if lagId not in self.hwBonds.keys():
                                print ("lagId %d is not configured in hw" % (lagId))
                                dstIntf = "unknown"
                            else:
                                dstIntf = self.hwBonds[lagId]["linuxIntfName"]
                        elif hwLogbasePortDict[keys]["port-type"] == "network":
                            logId = int(keys.split()[1], 16)
                            dstIntf = self.portInfoToLinuxDict.get(logId, None)
                dstVlan = None if dstVlan in range(3000,4000) else dstVlan
                self.l3EgressIfHelperDict[rif] = (destMac, dstIntf, dstVlan, mtu)

    def buildLocalPortInfo(self):
        """
        Builds the local ports info in the form
        (intfType, (slavePorts), vlan) for comparison with kernel objects
        """
        if not self.tunnelMapHelper:
            print "No tunnel Map Entries to derive the localPorts"
            return
        for hwBridges in self.tunnelMapHelper:
            if hwBridges not in self.hwBridgePortMapHelper.keys():
                raise Exception("The given bridge Id %d is not programmed" \
                        % (hwBridges))
            if self.hwBridgePortMapHelper[hwBridges] is None:
                continue
            vportDict = self.hwBridgePortMapHelper[hwBridges].get("hw-bridge-vports", {})
            vni = self.tunnelMapHelper[hwBridges]["hw-tunnel-map-entry"]["vni"]
            if vni not in self.asicVxlanDict.keys():
                raise Exception("The given vni %d is not configured \
                        in tunnel-map %d" % (vni))
            if not vportDict:
                print "No local ports found for the vni %d" % (vni)
                continue
            for keys, values in vportDict.iteritems():
                localIfTuple = ()
                basePortDict = vportDict[keys]["hw-port-vport-base"]
                basePortLogIdVlan = basePortDict.keys()
                for values in basePortLogIdVlan:
                    if values == "vlan":
                        vlan = basePortDict["vlan"]
                        vlan = None if vlan in range(3000,4000) else vlan
                    else:
                        basePortLogId = values

                logId = int(basePortLogId.split()[1], 16)
                portType = "trunk" \
                    if basePortDict[basePortLogId]["port-type"] == "lag" \
                    else "modport"
                if portType == "trunk":
                    lagId = basePortDict[basePortLogId]["lag-id"]
                    if lagId not in self.hwBonds.keys():
                        print "No hw bonds configured for lagId %d" % (lagId)
                        slavePorts = ()
                    else :
                        slavePorts = self.hwBonds[lagId]["slaveLinuxIfs"]
                else:
                    logId = int(basePortLogId.split()[1], 16)
                    slavePorts = self.portInfoToLinuxDict.get(logId, None)
                localIfTuple = (portType, slavePorts, vlan)
                self.asicVxlanDict[vni]["localPorts"].append(localIfTuple)
                self.asicVxlanDict[vni]["rawLocalPorts"].append((portType, logId, vlan, slavePorts))

    def buildLocalAndRemoteMacs(self):
        """
        Builds the local/remote macs
        """
        if not self.tunnelMapHelper:
            print "No tunnel Map Entries to derive local and remote macs"
            return
        for hwBridge in self.tunnelMapHelper:
            if hwBridge not in self.hwBridgeFdbHelperDict.keys():
                raise Exception("hwBridge fdb info not found")
            vni = self.tunnelMapHelper[hwBridge]["hw-tunnel-map-entry"]["vni"]
            if vni not in self.asicVxlanDict.keys():
                raise Exception("vni %d not configured in hw" % (vni))
            fdbAgeableEntries = self.hwBridgeFdbHelperDict[hwBridge]\
                        ["hw-bridge-fdb-unicast-ageable-entries"]
            fdbStaticEntries = self.hwBridgeFdbHelperDict[hwBridge]\
                        ["hw-bridge-fdb-unicast-static-entries"]

            fdbAgeableMacs = fdbAgeableEntries\
                        ["hw-bridge-fdb-unicast-entry "+str(hwBridge)]["mac-address"]
            fdbStaticMacs = fdbStaticEntries\
                        ["hw-bridge-fdb-unicast-entry "+str(hwBridge)]["mac-address"]
            if not fdbAgeableMacs and not fdbStaticMacs:
                print "No fdb Ageable/Static entries" \
                    "programmed to derive the local/remote macs"
                continue
            remoteMacs = []
            localMacs = []
            dbgLocalMacs = []
            sip = self.tunnelMapHelper[hwBridge]["sip"]
            if fdbAgeableMacs and fdbStaticMacs:
                fdbMacs = dict(fdbAgeableMacs.items() + fdbStaticMacs.items())
            else:
                fdbMacs = fdbAgeableMacs if fdbAgeableMacs else fdbStaticMacs
            for keys, values in fdbMacs.iteritems():
                macAddress = fdbMacs[keys]["mac-address"]
                dstType = fdbMacs[keys]["fdb-uc-mac-dest-type"]
                entryType = fdbMacs[keys]["fdb-uc-mac-entry-type"]
                dstTypeDict = fdbMacs[keys] \
                    ["hw-logical-dst-logical-properties"]
                hwLogPortId = dstTypeDict.keys()[0]
                if dstType == "next-hop":
                    dstIpInfo = fdbMacs[keys]["hw-logical-dst-logical-properties"]\
                            ["nexthop-entry"]["nh-type-encap"]
                    dstIp = dstIpInfo.split()[1]
                    macDstIpTuple = (hwBridge, macAddress, entryType, dstType, \
                                    sip, dstIp)
                    remoteMacs.append(macDstIpTuple)

                elif dstType == "logical-port":
                    if entryType == "static":
                        continue
                    hwLogId = int(hwLogPortId.split()[1], 16)
                    if dstTypeDict[hwLogPortId]["port-type"] == "vport-lag":
                        lagId = dstTypeDict[hwLogPortId]["phy-id/lag-id"]
                        if lagId not in self.hwBonds.keys():
                            print "lag Id %d is not configured in hw" % \
                                    (lagId)
                            dstIntf = "unknown"
                        else :
                            dstIntf = self.hwBonds[lagId]["linuxIntfName"]
                    elif dstTypeDict[hwLogPortId]["port-type"] == "vport":
                        #TODO
                        dstIntf = self.portInfoToLinuxDict.get(hwLogId, "unknown")
                        lagId = -1
                    else:
                        print "Unknown interface not supported by cc %s" % \
                             (dstTypeDict[hwLogPortId]["port-type"])
                        dstIntf = "unknown"
                        lagId = -1
                    macIfTuple = (macAddress, dstIntf)
                    localMacs.append(macIfTuple)
                    dbgLocalMacs.append((macAddress, lagId, dstIntf))
                else :
                    raise Exception("unknown dstType not supported by cc % s"
                            % (dstType))
            self.asicVxlanDict[vni]["localMacs"] = localMacs
            self.asicVxlanDict[vni]["rawLocalMacs"] = dbgLocalMacs
            self.asicVxlanDict[vni]["remoteMacs"] = remoteMacs
        self.resolveDstIpNext()

    def resolveDstIpNext(self):
        if not self.asicVxlanDict:
            return
        if not self.hwRouteTableHelperDict:
            print "No hw route entries programmed"
            return
        for vni, vniInfo in self.asicVxlanDict.iteritems():
            remoteMacs = self.asicVxlanDict[vni]["remoteMacs"]
            self.asicVxlanDict[vni]["remoteMacs"] = []
            for hwBridge, mac, entryType, dstType, sip, dstIp in remoteMacs:
                if not dstIp:
                    continue
                if dstIp.count("/") == 0:
                    dstIp = self._insert_ip_addr_mask(dstIp)
                if dstIp not in self.hwRouteTableHelperDict.keys():
                    print "dstIp %s not programmed in hw" % (dstIp)
                    self.asicVxlanDict[vni]["remoteMacs"].append((mac, sip, \
                    dstIp, None, None, None, None))
                    self.asicVxlanDict[vni]["rawRemoteMacs"]\
                    .append((hwBridge, mac, entryType, dstType, \
                        sip, dstIp, None, None, None, None))
                    continue
                nextHopList = self.hwRouteTableHelperDict[dstIp]\
                        .get("next_hop_rifs", [])
                for nh in nextHopList:
                    if nh not in self.l3EgressIfHelperDict.keys():
                        print "rif %d is not configured" % (nh)
                        self.asicVxlanDict[vni]["remoteMacs"]\
                        .append((mac, dstIp, None, None, None, None))
                        self.asicVxlanDict[vni]["rawRemoteMacs"]\
                        .append((hwBridge, mac, entryType, dstType, \
                        sip, dstIp,\
                         None, None, None, None))
                        continue
                    (dstMac, dstIf, dstVlan, mtu) = self.l3EgressIfHelperDict[nh]
                    self.asicVxlanDict[vni]["remoteMacs"]\
                        .append((mac, sip, dstIp, dstMac, dstIf, dstVlan, mtu))
                    self.asicVxlanDict[vni]["rawRemoteMacs"]\
                        .append((hwBridge, mac, entryType, dstType,\
                         sip, dstIp, dstMac, dstIf, dstVlan, mtu))

    def get_chipname(self):
        pass

    def prefetchHwInfo(self):
        self.buildSdkPortToLinuxDict()
        self.parseTunnelInfo()
        self.parseBridgeInfo()
        self.buildRouterNeighDict()
        self.buildLocalPortInfo()
        self.buildLocalAndRemoteMacs()

    def dumpVxlans(self):
        self.prefetchHwInfo()
        return self.asicVxlanDict
