#!/usr/bin/env python
# Copyright 2012-2020 Cumulus Networks, Inc
#

"""
Compare kernel vxlan info with hw vxlan info
"""

from cStringIO import StringIO
from ipaddr import IPv6Address, IPNetwork, IPv4Network, IPv6Network, AddressValueError
from pprint import pformat, pprint
from subprocess import Popen, PIPE, check_output, CalledProcessError
import argparse
import json
import os
import re
import sys
import cumulus.platforms
from nlmanager.kernel_nlhelpers import *
from nlmanager.nlpacket import Route, Neighbor, Link
from nlmanager.nlmanager import NetlinkManager

class KernelVxlan(object):

    def __init__(self):
        print "Kernel Object Init"
        self.kernelVxlanDict = {}
        #nlmanager lib
        self.kernel = Kernel()
        self.kernIfaces = self.kernel.ifaces
        #Helper Libraries for RouteLookup
        self.ifIndexToIfName = {}
        self.routeToNextHopDict = {}
        self.neighborDict = {}
        self.MasterToSlaveLinks = {}
        self.vlantoVniMap = {} # vlan to vni map for vlan-aware case 
        self.bridgeToVniMap = {} #bridge to vni map for vlan-unaware case
        self.buildRouteDict()
        self.buildNeighDict()
        #Kernel vxlanInfo lib
        """
        kernelVxlanDict[vxlanIntf] = {"vni"         : vniId,
                                      "vlan"        : vlan,
                                      "igmpSnoop"   : igmpSnoop,
                                      "arpSuppress" : arpSuppress,
                                      "localPorts"  : [(portType, (port1, port2,..), vlan), ]
                                      "localMacs"   : [(mac, port), ..]
                                      "remoteMacs"  : [(remoteMac, dstIp, dMac, intf, vlan), ..]
        """

    def _format_mac_addr(self, macAddrStr):
        mac = re.sub("[.:-]" ,"", macAddrStr)
        assert(len(mac) == 12)
        mac = ":".join([mac[i:i+2] for i in range(0, 12,2)])
        return mac.lower()

    def _insert_ip_addr_mask(self, ipaddr):
        if ipaddr.count(".") > 0:
            ipaddr += "/32"
        else:
            ipaddr += "/128"
        return ipaddr

    def _expand_kern_bond(self, dev):
        # returns a list of bond member names
        bondMembers = []
        bondPath = '/sys/class/net/%s' % dev + '/bonding/'
        if os.path.isdir(bondPath):
            try:
                for line in open(bondPath + 'slaves'):
                    bondMembers.extend(line.strip().split())
            except IOError:
                pass
        if not bondMembers:
            return
        bondMembers.sort()
        return tuple(bondMembers)

    def isSubIfName(self, subInterfaceName):
        intfObj = self.kernIfaces.get(subInterfaceName, None)
        if intfObj.kind == "vlan":
            return True
        return False

    def getSubIfsVlanMap(self, subInterfaceName):
        """
        Returns the parent interface and vlan tuple for the given 
        vlan subinterface
        """
        intfObj = self.kernIfaces.get(subInterfaceName, None)
        linkInfo = intfObj.ifobject.attributes.get(Link.IFLA_LINKINFO, {})
        data = linkInfo.value.get(Link.IFLA_INFO_DATA, {})
        vlanId = data.get(Link.IFLA_VLAN_ID)
        masterIfindex = intfObj.ifobject.get_attribute_value(Link.IFLA_LINK)
        masterIfName = self.ifIndexToIfName[masterIfindex]
        return (masterIfName, vlanId)

    def isVlanFilteringBridge(self, bridge):
        bridgeObj = self.kernIfaces.get(bridge)
        linkInfo = bridgeObj.ifobject.attributes.get(Link.IFLA_LINKINFO, {})
        if linkInfo and Link.IFLA_INFO_DATA in linkInfo.value:
            vlanAware = linkInfo.value.get(Link.IFLA_INFO_DATA)\
                    [Link.IFLA_BR_VLAN_FILTERING]
        return vlanAware

    def buildRouteDict(self):
        """
        Builds the route to next hop dictionary
        """
        routes = self.kernel.routes
        for route, routeObjList in routes.iteritems():
            nextHopList = []
            for routeObj in routeObjList:
                nhs = routeObj.get_nhs
                nexthopList = nextHopList+nhs 
            self.routeToNextHopDict[route] = nexthopList

    def buildNeighDict(self):
        """
        Builds the neigh dictionary 
        """
        neighs = self.kernel.neighbors
        for neigh, neighObjList in neighs.iteritems():
            for neighObj in neighObjList:
                neighIfindexTuple = (neigh, neighObj.iface)
                self.neighborDict[neighIfindexTuple] = (neighObj.mac, neighObj.vlan)

    def buildVxlanParamsDict(self):
        """
        Builds the vxlan param dictionary
        """
        if self.kernIfaces is None:
            raise Exception("No kernel interfaces found")

        for intf, intfObj in self.kernIfaces.iteritems():
            self.ifIndexToIfName[intfObj.ifindex] = intf
            isVxlan = False
            vlan = None
            vni = None
            vlanAware = None

            masterObj = intfObj.bridge
            master = masterObj.ifindex if masterObj is not None else None

            if intfObj and intfObj.kind == "vxlan":
                vni = self.kernel.get_vxlan_vni(intfObj)
                if masterObj:
                    vlanAware = self.isVlanFilteringBridge(masterObj.ifname)

                    if vlanAware:
                        vlan = self.kernel.get_pvid(intf)
                        self.vlantoVniMap[vlan] = vni
                    else:
                        self.bridgeToVniMap[master] = vni 
                #LocalIp 
                ifObject = intfObj.ifobject
                linkInfo = ifObject.get_attribute_value(Link.IFLA_LINKINFO, {})
                linkData = linkInfo.get(Link.IFLA_INFO_DATA, {})
                localIp = linkData.get(Link.IFLA_VXLAN_LOCAL)
                mtu = intfObj.mtu
                if localIp:
                    localIp = str(localIp)
                    
                # TODO Pending Igmp Snooping
                igmpSnoop = 0
                # TODO Pending Arp Suppression
                arpSupress = 0
                self.kernelVxlanDict[intf] = {"master"    :  master,
                                              "vlan-aware" : vlanAware,
                                              "vni"       :  vni,
                                               "vlan"      :  vlan,
                                               "localIp"   : localIp,
                                               "mtu"      :  mtu,
                                               "igmpSnoop" :  igmpSnoop,
                                               "arpSupress" : arpSupress,
                                               "localPorts" : [],
                                               "localMacs"  : [],
                                               "remoteMacs" : [] }
                isVxlan = True

            if not isVxlan and master is not None:
                if master not in self.MasterToSlaveLinks.keys():
                    self.MasterToSlaveLinks[master] = []
                self.MasterToSlaveLinks[master].append(intf)

    def buildLocalPortDict(self):
        """
        Builds the local port dictionary 
        """
        if not self.kernelVxlanDict:
            print "No vxlan links configured"
            return  

        for vxlanIf in self.kernelVxlanDict.keys():
            bridgeId = self.kernelVxlanDict[vxlanIf]["master"]
            if bridgeId is None:
                continue
            vlanfilter = self.kernelVxlanDict[vxlanIf]["vlan-aware"]
            vlanId = self.kernelVxlanDict[vxlanIf]["vlan"]
            localIfs = self.MasterToSlaveLinks.get(bridgeId, [])
            if not localIfs:
                continue
            for iface in localIfs:
                if vlanfilter:
                    masteriface = iface        
                    vlan = vlanId if vlanId in self.kernel.get_l2_vlans(iface) \
                            else None
                else:
                    (masteriface, vlan) = self.getSubIfsVlanMap(iface)\
                        if self.isSubIfName(iface) else (iface, vlanId)
                if self.kernIfaces[masteriface].kind == "bond":
                    slavePorts = self._expand_kern_bond(masteriface)
                    localIfType = "trunk"
                else:
                    localIfType = "modport"
                    slavePorts = (masteriface)
                localIfTuple = (localIfType, slavePorts, vlan)
                self.kernelVxlanDict[vxlanIf]["localPorts"].append(localIfTuple)

    def buildLocalRemoteMacDict(self):
        """
        Builds the localmac and remote mac dictionary
        """
        bridgeFdbEntries = self.kernel.fdb_entries
        for fdbEntryObj in bridgeFdbEntries:
            mac = self._format_mac_addr(fdbEntryObj.\
                    get_attribute_value(Neighbor.NDA_LLADDR))
            bif    =  self.ifIndexToIfName.get(fdbEntryObj.ifindex, None)
            vlan   =  fdbEntryObj.get_attribute_value(Neighbor.NDA_VLAN)
            dst    =  fdbEntryObj.get_attribute_value(Neighbor.NDA_DST)
            if dst is not None:
                dst = str(dst)  
            masterIndex =  fdbEntryObj.get_attribute_value(Neighbor.NDA_MASTER)
            master =  self.ifIndexToIfName.get(masterIndex, None)
            state  =  fdbEntryObj.state
            if master is None and dst is not None:
                if mac != "00:00:00:00:00:00":
                    if bif in self.kernelVxlanDict.keys():
                        self.kernelVxlanDict[bif]["remoteMacs"].append((mac, dst))
                    else:
                        raise \
                    Exception("VxlanInfo not found for the given fdb entry")
            else:
                if bif not in self.kernelVxlanDict.keys() and \
                (state == Neighbor.NUD_REACHABLE or state == Neighbor.NUD_NOARP):
                    (kBif, kvlan)  = self.getSubIfsVlanMap(bif) \
                                if self.isSubIfName(bif) else (bif, vlan)
    
                    if self.isVlanFilteringBridge(master):
                        if kvlan not in self.vlantoVniMap.keys():
                            continue 
                        vxlanIntfObj = self.kernel.\
                        get_vni_vxlan(self.vlantoVniMap[kvlan])
                    else:
                        if masterIndex not in self.bridgeToVniMap.keys():
                            continue
                        vxlanIntfObj = self.kernel.\
                        get_vni_vxlan(self.bridgeToVniMap[masterIndex])

                    if vxlanIntfObj:
                        vxlanIntf = vxlanIntfObj.ifname
                        macIntfTuple = (mac, kBif)
                        if vxlanIntf in self.kernelVxlanDict.keys():
                            self.kernelVxlanDict[vxlanIntf]["localMacs"]\
                            .append(macIntfTuple)
        self.resolveRemoteIp()

    def resolveRemoteIp(self):
        """
        Resolves the ip address for the remote vteps and returns 
        (destMac, egressIf, vlan) tuple
        """
        for vni, vniInfo in self.kernelVxlanDict.iteritems():
            remoteMacs = self.kernelVxlanDict[vni]["remoteMacs"]
            self.kernelVxlanDict[vni]["remoteMacs"] = []
            sip = self.kernelVxlanDict[vni]["localIp"]
            mtu = self.kernelVxlanDict[vni]["mtu"]
            for remoteMac, dstIp in remoteMacs:
                if dstIp is None:
                    self.kernelVxlanDict[vni]["remoteMacs"]\
                    .append((remoteMac, sip, None, None, None, None, mtu))
                    continue
                if dstIp.count("/") == 0:
                    dstIp = self._insert_ip_addr_mask(dstIp)
                if dstIp not in self.routeToNextHopDict.keys():
                    print ("No route found for dstip %s" % (dstIp))
                    self.kernelVxlanDict[vni]["remoteMacs"]\
                        .append((remoteMac, sip, dstIp, None, None, None, mtu))
                    continue
            
                nextHopList = self.routeToNextHopDict[dstIp]
                if not nextHopList:
                    self.kernelVxlanDict[vni]["remoteMacs"]\
                        .append((remoteMac, sip, dstIp, None, None, None, mtu))
                    continue

                for nhIp, ifindex in nextHopList:
                    (dstMac, vlan) = self.neighborDict.\
                        get((str(nhIp), int(ifindex)),(None, None))
                    remoteIfTuple = (remoteMac, sip, dstIp, self._format_mac_addr(dstMac),
                        self.ifIndexToIfName[int(ifindex)], vlan, mtu)
                    self.kernelVxlanDict[vni]["remoteMacs"].append(remoteIfTuple)
                    
    def dumpVxlans(self):
        print "Collect Data for Kernel"
        self.buildVxlanParamsDict()
        self.buildLocalPortDict()
        self.buildLocalRemoteMacDict()
        return self.kernelVxlanDict
