#!/usr/bin/env python
# Copyright 2017-2020 Cumulus Networks, Inc.
#
# mroute-check lib

import os
import subprocess

class VrfEntry:
    def __init__(self, vrf, table_id):
        self.vrf = vrf
        self.table_id = table_id
        if table_id > 1000:
            self.vrf_id = table_id - 1000
        else:
            self.vrf_id = 0

    def __str__(self):
        return 'vrf: %s table: %d\n' % (self.vrf, self.table_id)

class VrfDb:
    def __init__(self):
        # key: vrf-dev-name info:VrfEntry
        self.vrfs = {}

        # add the default vrf
        self.vrfs['default'] = VrfEntry('default', 0)
        # gather the rest
        self._parse_vrf_list()

    def __str__(self):
        disp_str = 'VRFs:\n'
        for key in sorted(self.vrf):
            disp_str += '%s\n' % self.vrfs[key]
        return disp_str

    @staticmethod
    def _get_cmd_output(cmdList, dispStr=None):
        try:
            with open(os.devnull, 'w') as fnull:
                dispStr = subprocess.check_output(cmdList, stderr=fnull)
        except subprocess.CalledProcessError:
            pass
        return dispStr

    def _parse_vrf_list(self):
        cmd = '/usr/bin/vrf list'
        disp_str = self._get_cmd_output(cmd.split(), '')

        # sample format -
        #VRF              Table
        #---------------- -----
        #mars              1001
        #pluto             1003
        #venus             1002

        lines = disp_str.strip().splitlines()
        if len(lines) < 3:
            return

        try:
            vrf_index = lines[0].split().index('Name')
            table_id_index = lines[0].split().index('Table')
            max_index = max(vrf_index, table_id_index)
        except:
            return

        for line in lines[2:]:
            words = line.split()
            if len(words) != (max_index + 1):
                continue

            vrf = words[vrf_index]
            table_id = int(words[table_id_index])
            self.vrfs[vrf] = VrfEntry(vrf, table_id)

    def vrf_id2name(self, vrf_id):
        for key, info in self.vrfs.items():
            if info.vrf_id == vrf_id:
                return key
        return str(vrf_id)
