#!/usr/bin/python
# Copyright 2018 Cumulus Networks, Inc.
#           
# mroute-check asic backend helper

import os
import sys
import subprocess
import argparse
import cumulus.platforms

BRIDGE_VLANS = '/Support/bridge.vlan'
BRIDGE_LINKS = '/Support/bridge.link'
PORTMAP = '/Support/portmap'
MCASTSHOW = '/Support/mcast.show'
VXLAN_INFO = '/cumulus/switchd/debug/vxlan/info'



class mcast_checker(object):
    def __init__(self, prefix=None, debug=False):
        self.prefix = prefix
        self.debug = debug

        # check and enable switchd debug for sfs node to appear
        if not self.prefix:
            self.switchd_debug_enabled = self.get_switchd_debug(self)
            if not self.switchd_debug_enabled:
                self.enable_switchd_debug(self)

        self.bonds = {}
        self.portmap = {}
        self.vnigrps = {}
        self.intfs = []
        self.mcastgrps = {}
        self.get_portmap()
        self.get_mcast_groups()
        self.get_vlan_vni_group()
        self.get_bond_vlans()


    def __del__(self):
        # restore switchd debug
        if not self.prefix:
            if not self.switchd_debug_enabled:
                self.disable_switchd_debug(self)

    @staticmethod
    def get_switchd_debug(self):
        return os.path.isfile(VXLAN_INFO)

    @staticmethod
    def enable_switchd_debug(self):
        os.system('echo 1 > /cumulus/switchd/ctrl/debug')
        return

    @staticmethod
    def disable_switchd_debug(self):
        os.system('echo 1 > /cumulus/switchd/ctrl/debug')
        return

        
    #This function will extract all the interfaces in the
    #bridge and the associated vlans
    def get_bond_vlans(self):
        line_number = 0
        #Extract the output of "bridge -c vlan show" command 
        #output
        if self.prefix:
            path = self.prefix + BRIDGE_VLANS
            with open(path, 'r') as f:
                lines = f.readlines()
        else:
            output = subprocess.check_output('bridge -c vlan show'.split())
            #output = subprocess.check_output('bridge link show'.split())
            lines = output.splitlines()

        intf = ''

        vids_list = []
        intf_name = '' 
        ports = ''
        #Parse the output of "bridge -c vlan show" command
        #Sample output will be as follows
        #swp2          1 PVID Egress Untagged
        #            400
        #            401
        #            420-425
        #            500
        #            520
        # So following code will extract interface and associate valns
        # vlan can be in three format, one with PVID, second just with
        # vlan and thrid with vlan range (for eg: in the above output
        # 420-425).
        # After extracting interface and vlan, check wheter interface
        # is real(physical) or not. If this not real interface (for eg:
        # vxlan interfaces) then skip and continue.
        # Also skip peerlink interfaces (this is done by checking the 
        # 'peerlink on' flag).
        # Once interface is cofirmed to be real, then get the
        # mapping for the interface by checking the portmap.
        # if there is no mapping then it could be bond interface, for
        # bond interface get the associate slaves from 
        # /sys/class/net/ + intf_name + /bonding/slaves
        for l in lines:
            if 'Start' in l or 'Complete' in l or 'ids' in l:
                continue

            vids = ''
            a = l.split()
            if not a:
                if intf_name:
                    if ze:
                        self.intfs.append(ze)
                        self.bonds.update({intf_name: {'xe': ze, 'vlans': [vid for vid in vids_list]}})
		    else:
		       for p in str(ports).split():
                           self.intfs.append(self.portmap.get(p))
                           self.bonds.update({p: {'xe': self.portmap.get(p), 'vlans': [vid for vid in vids_list]}})

                while len(vids_list) > 0:
                    vids_list.pop()
                intf_name = ''
                ports = ''
                continue

            if intf_name == '' and len(a) >= 1:
		intf_name = a[0]
	    	out = ""
                if self.prefix:
                    dev_file = self.prefix + "/sys/class/net/" + intf_name
                else:
                    dev_file = "/sys/class/net/" + intf_name
                if not os.path.isdir(dev_file):
	            intf_name = ''
                    continue

                if self.prefix:
                    bridge_link = self.prefix + BRIDGE_LINKS
	            p1 = subprocess.Popen(['grep', a[0], bridge_link], stdout=subprocess.PIPE)
	            p2 = subprocess.Popen(['grep', 'peerlink on'], stdin=p1.stdout, stdout=subprocess.PIPE)
	            out = p2.communicate()[0]
                else:
                    p1 = subprocess.Popen(['bridge', '-d', 'link', 'show', 'dev', a[0]], stdout=subprocess.PIPE)
                    p2 = subprocess.Popen(['grep', 'peerlink on'], stdin=p1.stdout, stdout=subprocess.PIPE)
                    out = p2.communicate()[0]

            	if out:
                    intf_name = ''
                    continue
                else:
	            ze = ''
                    ze = self.portmap.get(intf_name)
                    if not ze:
                        if self.prefix:
                            file_name = self.prefix + "/sys/class/net/" + a[0] + "/bonding/slaves"
                        else:
		            file_name = "/sys/class/net/" + a[0] + "/bonding/slaves"

		        if os.path.isfile(file_name):
		            ports = subprocess.check_output(["cat", file_name])
                        else:
 			    intf_name = ''
			    continue
                    if len(a) > 1:
                        vids = a[1]
            elif intf_name and len(a) >= 1:
	         vids = a[0]

            if vids:
	        if "-" in vids:
		    vidsl = vids.split("-")
		    vidstart = int(vidsl[0])
		    vidend = int(vidsl[1])
	        else: 
		    vidstart = int(vids)
		    vidend = vidstart

                while vidstart <= vidend:
		      vids_list.append(str(vidstart))
                      vidstart = vidstart + 1
         

        if self.debug:
            print self.bonds
            print self.intfs

    #This function will extract the mapping between linux and 
    #bcm interfaces 
    def get_portmap(self):
        if self.prefix:
            path = self.prefix + PORTMAP
            with open(path, 'r') as f:
                lines = f.readlines()
        else:
            output = subprocess.check_output('/usr/lib/cumulus/portmap'.split())
            lines = output.splitlines()

        for l in lines:
            if 'Start' in l or 'linux_intf' in l or 'Complete' in l:
                continue

            a = l.split()
            self.portmap.update({a[0]: a[1]})

        if self.debug:
            print self.portmap

    #This function finds the VNIs in the system and 
    # associated mcast and broad cast groups.
    # 1. Find the SVI interface (in /cumulus/switchd/debug/vxlan/info)
    # Sample output will be as follows
    # vpn: 0x7004; vni: 1004; has_svi: 1
    #   flags: 0x9a; eg_vlan: 0; bc_grp: 0xc000008; mc_grp: 0xc000007
    # 2. Get the associated VNI
    # 3. Skip the default SVI.
    # 4. Extract the associated bradcast and multicast groups
    # 5. mcastgrps[] contains output of bcm multicast show
    #   Sample output will be as follows
    # Group 0xc000007 (VXLAN)
    #        port xe4, encap id 400011
    #        port xe5, encap id 400011
    #        port xe6, encap id 400025
    #        port xe6, encap id 400026
    # Group 0xc000008 (VXLAN)
    #        port xe4, encap id 400021
    #        port xe5, encap id 400021
    #        port xe6, encap id 400035
    #        port xe6, encap id 400036
    # 6. Now merge so that for a given VNI we have list interfaces 
    #    and the number of interface count.
    #    Sample merged output will be as follows
    # {'1004': {'mc-freq': {'xe4': 1, 'xe5': 1, 'xe6': 2}, 'bc-freq': {'xe4': 1, 'xe5': 1, 'xe6': 2}, 'mc': ['xe4', 'xe5', 'xe6', 'xe6'], 'bc': ['xe4', 'xe5', 'xe6', 'xe6']}
    # mc-->Multicast group, bc--> broadcast group, 
    # mc-freq-->frequency of interfaces in mc, bc-freq--> frequency of interfaces in bc
    def get_vlan_vni_group(self):
        if self.prefix:
            path = self.prefix + VXLAN_INFO
        else:
            path = VXLAN_INFO

        p1 = subprocess.Popen(['grep', '-A1', 'has_svi', path], stdout=subprocess.PIPE)
        out = p1.communicate()[0]
        lines = out.splitlines()

        vni = ''
        for l in lines:
            if vni == '777215':
                vni = ''
                continue

            if 'vni' in l:
                vni = l.split()[3][0:-1]
                if vni == '16777215':
                    vni = ''
                    continue
            elif 'grp' in l:
                a = l.split()
                bc = a[5][:-1]
                mc = a[7]
		if vni: 
                    self.vnigrps.update({vni: {'bc': self.mcastgrps[bc],
                                               'bc-freq':{x:self.mcastgrps[bc].count(x) for x in self.mcastgrps[bc]},
                                               'mc': self.mcastgrps[mc],
                                               'mc-freq':{x:self.mcastgrps[mc].count(x) for x in self.mcastgrps[mc]}}})
        if self.debug:
            print self.vnigrps

    #This function parses the output of bcm multicast show 
    #and stores it in a dictionary format where key is the
    #group id and associated data is list of interfaces.
    #   Sample output of 'bcm multicast show' will be as follows
    # Group 0xc000007 (VXLAN)
    #        port xe4, encap id 400011
    #        port xe5, encap id 400011
    #        port xe6, encap id 400025
    #        port xe6, encap id 400026
    # Group 0xc000008 (VXLAN)
    #        port xe4, encap id 400021
    #        port xe5, encap id 400021
    #        port xe6, encap id 400035
    #        port xe6, encap id 400036
    # corresponding dictionary will be as follows
    #{'0xc000007': ['xe4', 'xe5', 'xe6', 'xe6'], '0xc000009': ['xe4', 'xe5', 'xe6', 'xe6']}
    def get_mcast_groups(self):
        if self.prefix:
            path = self.prefix + MCASTSHOW
            with open(path, 'r') as f:
                lines = f.readlines()
        else:
            output = subprocess.check_output('/usr/lib/cumulus/bcmcmd multicast show'.split())
            lines = output.splitlines()

        group = ''
        for l in lines:
            if 'Start' in l or 'Complete' in l:
                continue

            if 'Group' in l:
                group = l.split()[1]
                self.mcastgrps.update({group: []})
            else:
                port = l.split()[1][:-1]
                self.mcastgrps[group].append(port)

        if self.debug:
            print self.mcastgrps

    #This function will check for following errors
    # For each of the VNI, for the associated mcast and 
    # broadcast groups check 
    # (a) whether any interface is missing
    # (b) check we have duplicate entries for an interface.
    def check(self):
        err = 0
        for b in self.bonds.items():
            for v in b[1]['vlans']:
                if v == '1':
                    continue

                if v in self.vnigrps:
			if b[1]['xe'] not in self.vnigrps[v]['bc']:
			    print '{}:{} port not in vni {} bc group'.format(b[0], self.portmap.get(b[0]), v)
			    err += 1
			else:
			    if self.vnigrps[v]['bc-freq'][b[1]['xe']] > 1:
				print 'duplicate entries for port {}:{} in vni {} bc group'.format(b[0],self.portmap.get(b[0]), v)
				err += 1

			if b[1]['xe'] not in self.vnigrps[v]['mc']:
			    print '{}:{} port not in vni {} mc group'.format(b[0], self.portmap.get(b[0]), v)
			    err += 1
			else:
			    if self.vnigrps[v]['mc-freq'][b[1]['xe']] > 1:
				print 'duplicate entries for port {}:{} in vni {} mc group'.format(b[0],self.portmap.get(b[0]), v)
				err += 1

        return err


def setup_arg_parser():
    parser = argparse.ArgumentParser()
    #parser.add_argument('path', help='path of cl-supports or files')
    parser.add_argument('--s', help='parse cl-supports')
    parser.add_argument('--d', help='enable debug', action="store_true")
    return parser


def main(args):
    parser = setup_arg_parser()
    ns = parser.parse_args(args)

    platform_object = cumulus.platforms.probe()
    if platform_object.switch.chip.sw_base != 'bcm':
        print "This script is currently supported on Broadcom platforms only\n"
        sys.exit(-1)

    err = 0
    if ns.s:
        supports = os.listdir(ns.s)
        for supp in supports:
            if 'cl_supp' not in supp or 'accsw' not in supp:
                continue

            print 'checking {}...'.format(supp)
            mc = mcast_checker(supp, ns.d)
            err += mc.check()
    else:
        mc = mcast_checker(debug=ns.d)
        err += mc.check()

    print 'Found {} errors'.format(err)

if __name__ == '__main__':
    sys.exit(main(sys.argv[1:]))

