#!/bin/bash

#==============================================================================#
#=  Global variables                                                           #
#=
#=====
VERSION="2.1"
#=============
UTIL_TITLE="Nvidia Mellanox SSD identification and Analysis tool  ${VERSION}"
DEPENDECIES=("smartctl" "cut" "egrep" "grep" "/bin/bash" "awk")
TRUE="1"
FALSE="0"
SWITCH_SN=""
SWITCH_PN=""
MST_DEV=""
SSD_DEV="/dev/sda"
SSD_FW_VERSION=""
SSD_MODEL=""
SSD_SIZE=""
SSD_HEALTH=""
SSD_AVG_ERASE_COUNT=""
SSD_POWER_ON_HOURS=""
SSD_PE_CYCLE=""

FULL_MODE='no'
LOG_DIR='/tmp/'
SCRIPT=$(realpath $0)
SCRIPT_NAME=$(basename $SCRIPT)
SCRIPTPATH=$(dirname $SCRIPT)
COMMON_PATH="$SCRIPTPATH/common"
iSMART_PATH="${COMMON_PATH}/iSMART_64"
vtView_PATH="${COMMON_PATH}/vtViewCmd_64b_1.0.2437"
OS_TYPE=""
OS_FILE="${COMMON_PATH}/platform"
HOSTNAME=$(hostname)
DECODE_SYSEEPROM_OUTPUT=""
MLXLINK_OUTPUT=""
SWITCH_SN=""
TODAY=$(date +%d_%m_%Y)


M89_PE_CYCLE=3000
M109_PE_CYCLE=3000
M145_PE_CYCLE=20000
STORFLY_PE_CYCLE=20000
MLC_WARN_THRESHOLD=2500
MLC_ERR_THRESHOLD=6000
iSLC_WARN_THRESHOLD=16000
iSLC_ERR_THRESHOLD=21000

# SSD Models
M54="InnoDisk Corp. - mSATA 3ME"
M54_FW="S130710D"
M89="InnoDisk Corp. - mSATA 3ME"
M89_FW="S140714"
WEAROUT_RATE="NA"
M109="M.2 (S42) 3ME3"
M109_FW_OLD1="S15A19"
# bug fix for issue is in S16425M but it still has less bug fixes then S19903M
M109_FW_OLD2="S16425M"
M109_FW_OLD3="S19903M"
M109_FW_NEW="S20728"
M145="M.2 (S42) 3IE3"
M145_FW_OLD1="S16425i"
M145_FW_OLD2="S19903Mi"
M145_FW_NEW="S20728i"
M207="StorFly VSF302XC016G-MLX"
M207_FW_OLD="0115-000"
M207_FW_NEW="1210-000"


MODE="log"
REMOTE_SERVER_IP="0"
REMOTE_USER="0"
REMOTE_PASSWORD="0"
TRANSFER_METHOD="scp"
REMOTE_PATH="/tmp/"
LOG_REMOTE="no"

function print_delimiter {
  echo "=================================================="
}

function print_line {
  echo "$1"
}

function get_virtium_ssd_nand_endurance {

    # Virtium StorFly
    # get Attribute 168 in output of /vtViewCmd_64b_1.0.2437 -m /dev/sda
    # SMART attributes
    # ID
    # 168                 NAND_Endurance          0      20000   100   100         0

    local nand_endurance=$(echo "${vtViewcmd_output_m}" | egrep -i "NAND_Endurance" | awk '{print $4}' )

    eval $1='$nand_endurance'
}

function get_virtium_ssd_average_erase_count {

    # Virtium StorFly
    # get Attribute 167 in output of /vtViewCmd_64b_1.0.2437 -m /dev/sda
    # SMART attributes
    # ID
    # 167          Average_Erase_Count          0        103   100   100         0
    local average_erase_count=$(echo "${vtViewcmd_output_m}" | egrep -i "Average_Erase_Count" | awk '{print $4}' )
    eval $1='$average_erase_count'
}

function  get_virtium_ssd_life {
    [ $1 ] || print_line "Error: ${FUNCNAME[0]} received an expected number of variables"

    # Virtium StorFly
    # get Attribute 168 in output of /vtViewCmd_64b_1.0.2437 -m /dev/sda
    # SMART attributes
    # ID
    # 167          Average_Erase_Count          0        103   100   100         0
    # 168                 NAND_Endurance          0      20000   100   100         0

    local virtium_life=$(echo "${vtViewcmd_output_r}" | egrep -i "^Life%:" | awk '{print $2}' )
    eval $1='$virtium_life'
}

function get_innodisk_ssd_pe_cycle {
    [ $1 ] || print_line "Wrong usage - ${FUNCNAME[0]}"
    local pe_cycle=$(echo "${iSMART_64_output}" | egrep -i "PE Cycle" | awk '{print $4}' )
    eval $1='$pe_cycle'
}

function get_innodisk_ssd_average_erase_count {
    [ $1 ] || print_line print_line "Wrong usage - ${FUNCNAME[0]}"
    local average_erase_count=$(echo "${iSMART_64_output}" |  egrep -i "^AVG. Erase" | cut -d ":" -f 2 | sed 's/^[ \t]*//' )
    eval $1='$average_erase_count'
}

function get_innodisk_ssd_power_on_hours {
    [ $1 ] || print_line print_line "Wrong usage - ${FUNCNAME[0]}"
    local power_on_hours=$(echo "${iSMART_64_output}" | egrep -i "^Power ON Hours" | cut -d ":" -f 2 | sed 's/^[ \t]*//' )
    eval $1='$power_on_hours'
}

function get_innodisk_ssd_health {
    [ $1 ] || print_line print_line "Wrong usage - ${FUNCNAME[0]}"
    local health=$(echo "${iSMART_64_output}" | egrep ^Health | awk '{print $3}' )
    eval $1='$health'
}


function analysis {

if [[ $M207 == $SSD_MODEL ]]; then
    if [[  $SSD_AVG_ERASE_COUNT -ge $iSLC_WARN_THRESHOLD ]] && [[ $SSD_AVG_ERASE_COUNT -le $iSLC_ERR_THRESHOLD ]]; then
      print_line
      print_line "Health is 20% or less - please contact Nvidia Mellanox Technical Support for further review, provide the script output"
      print_line "Upgrade of the SSD FW (if required) is still needed before contacting Nvidia Mellanox Technical Support"
      print_line
   fi
   if [[  $SSD_AVG_ERASE_COUNT -gt $iSLC_ERR_THRESHOLD ]]; then
      print_line
      print_line "Health is 20% or less - please contact Nvidia Mellanox Technical Support for further review, provide the script output"
      print_line
   fi
   if [[ $SSD_SIZE == "15.8GB" ]]; then

    if [[ $SSD_FW_VERSION == $M207_FW_OLD  ]]; then
      print_line "Recommendation: Upgrade SSD FW to version $M207_FW_NEW"
    else
     if [[ $SSD_FW_VERSION == $M207_FW_NEW ]]; then
      print_line "Recommendation: No Actions required - SSD FW is already $M207_FW_NEW"
     else
      print_line "Recommendation: No Actions required - SSD FW version is $SSD_FW_VERSION"
     fi
    fi
  else
    print_line "Reccommendation: No Actions required - There are no known issues with the SSD Model, SIZE = $SSD_SIZE"
  fi
fi

if [[ $M109 == $SSD_MODEL ]]; then
  if [[  $SSD_AVG_ERASE_COUNT -ge $MLC_WARN_THRESHOLD ]] && [[  $SSD_AVG_ERASE_COUNT -le $MLC_ERR_THRESHOLD ]]; then
     print_line
     print_line "Health is 20% or less - please contact Nvidia Mellanox Technical Support for further review, provide the script output"
     print_line "Upgrade of the SSD FW (if required) is still needed before contacting Nvidia Mellanox Technical Support"
     print_line
  fi
  if [[ $SSD_AVG_ERASE_COUNT -gt $MLC_ERR_THRESHOLD ]]; then
    print_line
    print_line "Health is 20% or less - please contact Nvidia Mellanox Technical Support for further review, provide the script output"
    print_line
  fi
  if [[ $SSD_SIZE == "16.0GB" ]]; then
    if [[ $SSD_FW_VERSION == $M109_FW_OLD1  ]] || [[ $SSD_FW_VERSION == $M109_FW_OLD2 ]] || [[ $SSD_FW_VERSION == $M109_FW_OLD3 ]]; then
      print_line "Recommendation: Upgrade SSD FW to version $M109_FW_NEW"
    else
     if [[ $SSD_FW_VERSION == $M109_FW_NEW ]]; then
       print_line "Recommendation: No Actions required - SSD FW version is already $M109_FW_NEW"
      else
       print_line "Recommendation: No Actions required - SSD FW version is $SSD_FW_VERSION"
     fi
    fi
  else
    print_line "Recommendation: No Actions required - There are no known issues with the SSD Model, SIZE = $SSD_SIZE"
  fi
fi

if [[ $M145 == $SSD_MODEL ]]; then
  if [[  $SSD_AVG_ERASE_COUNT -ge $iSLC_WARN_THRESHOLD ]] && [[  $SSD_AVG_ERASE_COUNT -le $iSLC_ERR_THRESHOLD ]]; then
    print_line
    print_line "Health is 20% or less - please contact Nvidia Mellanox Technical Support for further review, provide the script output"
    print_line "Upgrade of the SSD FW is still needed before contacting Nvidia Mellanox Technical Support"
    print_line
  fi
  if [[  $SSD_AVG_ERASE_COUNT -gt $iSLC_ERR_THRESHOLD ]]; then
    print_line
    print_line "Health is 20% or less - please contact Nvidia Mellanox Technical Support for further review, provide the script output"
    print_line
  fi
  if [[ $SSD_SIZE == "16.0GB" ]]; then
    if [[ $SSD_FW_VERSION == $M145_FW_OLD1  ]] || [[ $SSD_FW_VERSION == $M145_FW_OLD2 ]]; then
        print_line "Recommendation: Upgrade SSD FW to version $M145_FW_NEW"
    else
      if [[ $SSD_FW_VERSION == $M145_FW_NEW ]] ; then
        print_line "Recommendation: No Actions required - SSD FW version is already $M145_FW_NEW"
      else
        print_line "Recommendation: No Actions required - There are no known issues with the SSD Model, SIZE = $SSD_SIZE"
      fi
    fi
    else
      print_line "Recommendation: No Actions required - SSD FW version is $SSD_FW_VERSION , SIZE = $SSD_SIZE"
   fi
fi

if [[ $M89 == $SSD_MODEL ]]; then
  # parsing based on iSMART V5.3.15 output
  if [[ $SSD_SIZE == "32.0GB" ]] || [[ $SSD_SIZE == "16.0GB" ]]; then
    #AVG_ERASE_COUNT=$(echo "${iSMART_64_output}" | egrep -i "^AVG. Erase" | cut -d ":" -f 2 | sed 's/^[ \t]*//' )
    #POWER_ON_HOURS=$(echo "${iSMART_64_output}" | egrep -i "^Power ON Hours" | cut -d ":" -f 2 | sed 's/^[ \t]*//')
    # using awk as cumulus doesn't have bc installed by default

    WEAROUT_RATE=$(awk -v v1="$SSD_AVG_ERASE_COUNT" -v v2="$SSD_POWER_ON_HOURS" 'BEGIN { print  ( v1 / v2 ) }' | awk '{printf "%.3f" ,$1}' )
    echo "$WEAROUT_RATE $SSD_AVG_ERASE_COUNT $SSD_POWER_ON_HOURS" |  awk '{

    if ($1 >= 0.086 || $2 >= 3000)
          print "Recommendation: Contact Mellanox support - Health Code =  "$1" "$2" "$3
      else
          print "Recommendation: No Actions required for now - Health Code =  "$1" "$2" "$3
    }'
  else
    print_line "Recommendation: No Actions required, SSD Size = $SSD_SIZE"
 fi
fi

if [[ $SSD_MODEL != $M89 ]] && [[ $M207 != $SSD_MODEL ]] && [[ $M109 != $SSD_MODEL ]] && [[ $M145 != $SSD_MODEL ]]; then
  print_line "Recommendation: No Actions required - There are no known issues with the SSD Model, SIZE = $SSD_SIZE"
  if [[  $SSD_AVG_ERASE_COUNT -ge $iSLC_WARN_THRESHOLD ]]; then
    print_line
    print_line "Health is 20% or less - please contact Nvidia Mellanox Technical Support for further review, provide the script output"
    print_line
  fi
fi

}

function print_line_to_file {
 echo "$1" >> $2
}

function general_data_log {
  log_filename=$(echo "$LOG_DIR/${HOSTNAME}_${SWITCH_PN}_${TODAY}_ssd_health.log")
  rm -f $log_filename
  touch $log_filename

  if test -f "$log_filename"; then
    print_line "Writing to $log_filename"
    print_line_to_file "SSD Information" $log_filename
    date >> $log_filename
    print_delimiter >> $log_filename
    print_line_to_file "Switch details:"  $log_filename
    print_line_to_file "Hostname: $HOSTNAME" $log_filename
    print_line_to_file "Switch SN: $SWITCH_SN"   $log_filename
    print_line_to_file "Switch PN: $SWITCH_PN" $log_filename
    print_line_to_file "SSD Model: $SSD_MODEL"   $log_filename
    print_line_to_file "SSD Size: $SSD_SIZE" $log_filename
    print_line_to_file "SSD FW Version: $SSD_FW_VERSION" $log_filename
    if [[ $SSD_PE_CYCLE == "3000"  ]]; then
      print_line_to_file "SSD Flash Technology: MLC" $log_filename
    elif [[ $SSD_PE_CYCLE == "20000"  ]]; then
          print_line_to_file "SSD Flash Technology: iSLC" $log_filename
    fi

    print_line_to_file "SSD Avg Erase Count: $SSD_AVG_ERASE_COUNT" $log_filename
    print_delimiter >> $log_filename
    analysis >> $log_filename
    if [[ $FULL_MODE == 'yes' ]]; then
      print_line_to_file "SSD Health: $SSD_HEALTH%"  $log_filename
      print_line_to_file "SSD AVG Erase Count / PE Cycle: $CALCULATED_DEVICE_HEALTH_USED" $log_filename
      print_line_to_file "Output of 'smartctl -i $SSD_DEV':" $log_filename
      print_line_to_file "" $log_filename
      print_line_to_file "$SMARTCTL_OUTPUT"  $log_filename
    fi
  else
    print_line "Can't create log file $log_filename, Exiting..."
    exit 1;
  fi

}

function general_data_to_stdout {
  date
  print_delimiter
  print_line "Switch details:"
  print_line "Hostname: $HOSTNAME"
  print_line "Switch SN: $SWITCH_SN"
  print_line "Switch PN: $SWITCH_PN"
  print_line "SSD Model: $SSD_MODEL"
  print_line "SSD Size: $SSD_SIZE"
  print_line "SSD FW Version: $SSD_FW_VERSION"
  if [[ $SSD_PE_CYCLE == "3000"  ]]; then
    print_line "SSD Flash Technology: MLC"
  elif [[ $SSD_PE_CYCLE == "20000"  ]]; then
        print_line "SSD Flash Technology: iSLC"
  fi
  print_line "SSD Avg Erase Count: $SSD_AVG_ERASE_COUNT"
  print_delimiter
  analysis
  print_delimiter
  if [[ $FULL_MODE == 'yes' ]]; then
   print_line "SSD Health: $SSD_HEALTH%"
   print_line "SSD AVG Erase Count / PE Cycle: $CALCULATED_DEVICE_HEALTH_USED"
   print_delimiter
   print_line "Output of 'smartctl -i $SSD_DEV':"
   print_line ""
   print_line "$SMARTCTL_OUTPUT"
  fi
}

function Virtium {

  vtViewcmd_output_m=$($vtView_PATH -m $SSD_DEV)
  vtViewcmd_output_r=$($vtView_PATH -r $SSD_DEV)
  get_virtium_ssd_average_erase_count SSD_AVG_ERASE_COUNT
  get_virtium_ssd_nand_endurance SSD_PE_CYCLE
  get_virtium_ssd_life SSD_HEALTH

  if [[ $MODE == "manual" ]]; then
    general_data_to_stdout
    if [[ $FULL_MODE == 'yes' ]]; then
      print_delimiter
      print_line "Output of 'vtViewcmd_64b_1.0.2437 -m $SSD_DEV':"
      print_line ""
      print_line "$vtViewcmd_output_m"
      print_delimiter
      print_line "Output of 'vtViewcmd_64b_1.0.2437 -r $SSD_DEV':"
      print_line ""
      print_line "$vtViewcmd_output_r"
      print_delimiter
    fi
  fi

  if [[ $MODE == "log" ]]; then

    general_data_log
    if [[ $FULL_MODE == 'yes' ]]; then
      print_line_to_file "Output of 'vtViewcmd_64b_1.0.2437 -m $SSD_DEV':"  $log_filename
      print_line_to_file "" $log_filename
      print_line_to_file "$vtViewcmd_output_m"   $log_filename
      print_line_to_file "Output of 'vtViewcmd_64b_1.0.2437 -r $SSD_DEV':"  $log_filename
      print_line_to_file "" $log_filename
      print_line_to_file "$vtViewcmd_output_r"   $log_filename
      print_delimiter >> $log_filename
    fi
    print_line "SSD Health Data was collected to file $log_filename"

  fi

}

function InnoDisk {
  iSMART_64_output=$(${iSMART_PATH} -d $SSD_DEV)
  # Health          : 70.47
  get_innodisk_ssd_health SSD_HEALTH
  get_innodisk_ssd_pe_cycle SSD_PE_CYCLE
  get_innodisk_ssd_average_erase_count SSD_AVG_ERASE_COUNT
  get_innodisk_ssd_power_on_hours SSD_POWER_ON_HOURS
  CALCULATED_DEVICE_HEALTH_USED=$(awk -v v1=$SSD_PE_CYCLE -v v2=$SSD_AVG_ERASE_COUNT 'BEGIN { print ( v2 / v1 ) }'  )
  #AVG_ERASE_COUNT_THRESHOLD=$(awk -v v1="$SSD_PE_CYCLE" 'BEGIN { print ( v1 * 0.83 )  }' | awk '{printf "%d" ,$1}')


  if [[ $MODE == "manual" ]]; then
    general_data_to_stdout
    if [[ $FULL_MODE == 'yes' ]]; then
       print_line "Output of 'iSMART_64 -d $SSD_DEV :"
       print_line "$iSMART_64_output"
    fi
  fi

  if [[ $MODE == "log" ]]; then
    general_data_log
    print_delimiter >> $log_filename
    if [[ $FULL_MODE == 'yes' ]]; then
      print_line_to_file "Output of 'iSMART_64 -d $SSD_DEV :"  $log_filename
      print_line_to_file "$iSMART_64_output"  $log_filename
      print_delimiter >> $log_filename
    fi
    print_line "SSD Health Data was collected to file $log_filename"
  fi

}

function other {

  if [[ $MODE == "manual" ]]; then
    general_data_to_stdout
  fi

  if [[ $MODE == "log" ]]; then
    general_data_log
    print_delimiter >> $log_filename
  fi
  print_delimiter
  #print_line "Recommendation: No Actions required - There are no known issues with the SSD Model"

}

function transfer {
echo "Sending the file $log_filename to:"
echo "${TRANSFER_METHOD}: $REMOTE_USER@$REMOTE_SERVER_IP:/$REMOTE_PATH/"
local log_file=$(basename $log_filename)
local ftpscript="/tmp/ftpcmd.sh"
if [[ -f $log_filename ]]; then
  if [[   "$REMOTE_SERVER_IP" != "0"  ]] && [[  "$REMOTE_USER" != "0" ]] && [[ "$REMOTE_PASSWORD" != "0" ]]; then
    if [[ $TRANSFER_METHOD == "scp" ]]; then
       sshpass -p $REMOTE_PASSWORD scp -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no  $log_filename ${REMOTE_USER}@${REMOTE_SERVER_IP}:/${REMOTE_PATH}
    fi
    if [[ $TRANSFER_METHOD == "ftp" ]]; then
       cp $log_filename /tmp/
       rm -f $ftpscript
       touch $ftpscript
       echo '#!/bin/bash' >> $ftpscript
       echo "ftp -n ${REMOTE_SERVER_IP} << EOF " >> $ftpscript
       echo "quote USER  ${REMOTE_USER}" >> $ftpscript
       echo "quote PASS ${REMOTE_PASSWORD}" >> $ftpscript
       echo "binary" >> $ftpscript
       #echo "lcd /shared/" >> $ftpscript
       echo "cd ${REMOTE_PATH}" >> $ftpscript
       echo "put ${log_file}" >> $ftpscript
       echo "quit" >> $ftpscript
       echo "EOF" >> $ftpscript
       chmod +x $ftpscript
       ${ftpscript}
    fi
  fi
fi

}
#==============================================================================#
#=  This function check if given argument is valid and return boolean result.  #
#=
function check_tool_dependencies() {
   	for i in "${!DEPENDECIES[@]}"
	do
		if [ ! -x "$(command -v ${DEPENDECIES[$i]})" ]; then
			print_line "Error: This tool require the following utils to be installed ${DEPENDECIES[$i]}"
      exit 0;
		fi
	done
}

function help {
     local me=$(basename "$0")
     echo
     echo -e "$UTIL_TITLE"
     echo
     echo -e "Usage:"
     echo -e "\t $me [OPTIONS]"
     echo -e "Commands:"
     echo -e "\t--version|-V\t\t print $me version"
     echo -e "\t--mode, \t\t output mode - [log|manual] to log file or to stdout"
     echo -e "\t--log_remote\t\t send log file to remote server - applicable with mode log"
     echo -e "\t--transfer_method\t\t protocol to use with log_remote - currently only scp and ftp are supported"
     echo -e "\t--remote_server_ip\t\t remote server ip to send log file - used with log_remote"
     echo -e "\t--remote_user\t\t remote server username to send log file - used with log_remote"
     echo -e "\t--remote_password\t\t remote server password to send log file - used with log_remote"
     echo -e "\t--remote_path\t\t remote directory on remote_server to send log file - used with log_remote"
     echo -e "\t--full\t\t gather more debug data from SSD for internal Mellanox review"
     echo -e "\t-h, --help\t\t show this usage."
     echo
     echo -e "Example:"
     echo -e "\tssd_health.sh --mode log "
     echo -e "\tssd_health.sh --mode manual "
     echo -e "\tssd_health.sh --mode log --full --log_remote --transfer_method scp --remote_server_ip 10.228.1.1 --remote_user youruser --remote_password yourpassword --remote_path /var/log/ "
     echo -e "\tssd_health.sh --mode log --full --log_remote --transfer_method scp --remote_server_ip 10.228.1.1 --remote_user youruser --remote_password yourpassword --remote_path /var/log/ "
     echo
     exit 0;
}

function check_usage() {
  local argument_count=$#

  if [ $# -eq 0 ]; then
    echo -e "\tError: false usage given."
    help
  fi

  while [[ $# -gt 0 ]]
  do
  key="$1"

  case $key in
    --version|-V)
        echo "$SCRIPT_NAME $VERSION"
        shift # passed argument
        exit 0
        ;;
    --mode)
        MODE=$2
        shift # passed argument
        shift # passed value
        ;;
    --full)
        FULL_MODE='yes'
        shift # passed argument
        ;;
    --log_remote)
        LOG_REMOTE="yes"
        shift # passed argument
        ;;
    --transfer_method)
        TRANSFER_METHOD="$2"
        shift # passed argument
        shift # passed value
        ;;
    --remote_server_ip)
        REMOTE_SERVER_IP="$2"
        shift # passed argument
        shift # passed value
        ;;
    --remote_user)
        REMOTE_USER="$2"
        shift # passed argument
        shift # passed value
        ;;
    --remote_password)
        REMOTE_PASSWORD="$2"
        shift # passed argument
        shift # passed value
        ;;
    --remote_path)
        REMOTE_PATH="$2"
        shift # passed argument
        shift # passed value
        ;;
    --user)
      SWITCH_USER=$2
      shift #passed argument
      shift #passed value
      ;;
    --password)
      SWITCH_PASSWORD=$2
      shift #passed argument
      shift #passed value
      ;;
    --help|-h)
        help ;
        exit 0
        ;;
      *)
        echo -e "\t\tError: false usage given." ; echo "$1"
        help
        exit 1
        ;;

  esac
  done
}

# Main

check_tool_dependencies
check_usage "$@"

if [[ -f $OS_FILE ]] ; then
  OS_TYPE=$(grep NOS $OS_FILE | cut -d "=" -f 2)
fi

#gathering smartctl

SMARTCTL_OUTPUT=$(smartctl -i $SSD_DEV)
# based on output Device Model:     InnoDisk Corp. - mSATA 3ME
SSD_MODEL=$(echo "$SMARTCTL_OUTPUT"  | grep "Device Model:" | cut -d ":" -f 2 | sed 's/^[ \t]*//' )
# based on output  - User Capacity:    16,013,942,784 bytes [16.0 GB]
SSD_SIZE=$(echo "$SMARTCTL_OUTPUT"  | grep "User Capacity" | awk '{print $5$6}' |  sed 's/\[//g'  | sed 's/\]//g' )
#SSD_SIZE=$(echo "$SMARTCTL_OUTPUT"  | grep -Po "User Capacity:.+bytes \[\K[^ ]+ GB" | awk '{print $1$2}')
# based on output Firmware Version: S140714
SSD_FW_VERSION=$(echo "$SMARTCTL_OUTPUT" | grep "Firmware Version:" | awk '{print $3}' )

if [[ $OS_TYPE == "CUMULUS" ]]; then
 DECODE_SYSEEPROM_OUTPUT=$(sudo decode-syseeprom)
 SWITCH_SN=$(echo "$DECODE_SYSEEPROM_OUTPUT" | grep "Serial Number" | awk '{print $5}')
 SWITCH_PN=$(echo "$DECODE_SYSEEPROM_OUTPUT" | grep "Part Number" | awk '{print $5}')
fi
  if [[ $OS_TYPE == "ONYX" ]]; then
    if [[ -d /shared/ ]]; then
      LOG_DIR="/shared"
    fi
    MST_DEV=$(mst status | egrep -i 'mt52000_pciconf0|mt52100_pciconf0|mt54000_pciconf0|mt53000_pciconf0'| awk '{print $1}' )
    if [[ -e $MST_DEV ]]; then
      MLXLINK_OUTPUT=$(mlxlink -d $MST_DEV --show_device | sed -r "s/[[:cntrl:]]\[[0-9]{1,3}m//g")
      SWITCH_SN=$(echo "$MLXLINK_OUTPUT" | grep "Serial Number" | awk '{print $4}')
      SWITCH_PN=$(echo "$MLXLINK_OUTPUT" | grep "Part Number" | awk '{print $4}')
    fi
fi

if [[ $M89 == $SSD_MODEL ]] || [[ $M109 == $SSD_MODEL ]] || [[ $M145 == $SSD_MODEL ]] || [[ $SSD_MODEL =~ "mSATA 3" ]]; then
  InnoDisk
else
   if [[ $M207 == $SSD_MODEL ]] || [[ $SSD_MODEL =~ "StorFly" ]]; then
      Virtium
   else
      other
   fi
fi

if [[ $LOG_REMOTE == "yes" ]] && [[ $MODE == "log" ]] && [[ $TRANSFER_METHOD == "scp" ]] ; then
   if [[  $(command -v sshpass) ]]; then
    transfer
   else
    print_line "Can't send to external server - sshpass is required for scp"
    exit 2;
   fi
fi

if [[ $LOG_REMOTE == "yes" ]] && [[ $MODE == "log" ]] && [[ $TRANSFER_METHOD == "ftp" ]] ; then
   if [[  $(command -v ftp) ]]; then
    transfer
   else
    print_line "Can't send to external server - ftp is required for ftp"
    exit 2;
   fi
fi
exit 0;
